# Class: unbound::remote
#
# Configure remote control of the unbound daemon process
#
# === Parameters:
#
# [*enable*]
#   (optional) The option is used to enable remote control, default is false.
#   If turned off, the server does not listen for  control.
#
# [*interface*]
#   (optional) Give  IPv4  or IPv6 addresses to listen on for control commands.
#   By default localhost (127.0.0.1 and ::1) is listened.
#
# [*port*]
#   (optional) The port number to listen on for control commands, default is
#   8953.  If you change this port number,  and  permissions have been dropped,
#   a reload is not sufficient to open the port again, you must then restart.
#
# [*server_key_file*]
#   (optional) Path to the server private key, by default unbound_server.key.
#   This file is generated by the unbound-control-setup utility.  This file is
#   used by the unbound server, but not by unbound-control.
#
# [*server_cert_file*]
#   (optional) Path to the server self signed certificate, by default
#   unbound_server.pem.  This file is generated by the unbound-control-setup
#   utility.  This file is used by the unbound server, and also by
#   unbound-control.
#
# [*control_key_file*]
#   (optional) Path to the control client private key, by default
#   unbound_control.key.  This file is generated by the unbound-control-setup
#   utility.  This file is used by unbound-control.
#
# [*control_cert_file*]
#   (optional) Path to the control client certificate, by default
#   unbound_control.pem.  This certificate has to be  signed  with  the server
#   certificate. This file is generated by the unbound-control-setup utility.
#   This file is used by unbound-control.
#
# [*group*]
#   (optional) Name of the group for unbound files and directory
#
# [*confdir*]
#   (optional) Name of the directory where configuration files are stored
#
# [*config_file*]
#   (optional) Name of the unbound config file
#
class unbound::remote (
  Boolean $enable                     = $unbound::control_enable,
  Array $interface                    = ['::1', '127.0.0.1'],
  Integer $port                       = 8953,
  Boolean $control_use_cert           = true,
  String $server_key_file             = "${unbound::confdir}/unbound_server.key",
  String $server_cert_file            = "${unbound::confdir}/unbound_server.pem",
  String $control_key_file            = "${$unbound::confdir}/unbound_control.key",
  String $control_cert_file           = "${$unbound::confdir}/unbound_control.pem",
  $group                              = $unbound::group,
  $confdir                            = $unbound::confdir,
  $config_file                        = $unbound::config_file,
  $control_setup_path                 = $unbound::control_setup_path,
) {
  concat::fragment { 'unbound-remote':
    order   => '10',
    target  => $config_file,
    content => template('unbound/remote.erb'),
  }

  unless $control_setup_path.empty {
    exec { 'unbound-control-setup':
      command => "${control_setup_path} -d ${confdir}",
      creates => $server_key_file,
    }

    file { [$server_key_file, $server_cert_file, $control_key_file, $control_cert_file]:
      owner   => 'root',
      group   => $group,
      mode    => '0640',
      require => Exec['unbound-control-setup'],
    }
  }
}
