// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


/*
  chartdialogs.cc
  This file contains code for the dialog boxes in the StarPlot Chart menu.
*/

#include "starplot.h"
#include <gtk/gtk.h>
using std::string;

/* automagnitude(): Tries to estimate a good dim magnitude cutoff for a given 
 *  chart radius. */
double automagnitude(double radius)
{
  if (radius <= 10) return 25;
  else if (radius <   12.5) return 25 - (radius -   10) * 4.4;
  else if (radius <   15) return 14   - (radius - 12.5) * 1.0;
  else if (radius <   20) return 11.5 - (radius -   15) * 0.8;
  else if (radius <   30) return  7.5 - (radius -   20) * 0.26;
  else if (radius <   40) return  4.9 - (radius -   30) * 0.14;
  else if (radius <   50) return  3.5 - (radius -   40) * 0.10;
  else if (radius <   75) return  2.5 - (radius -   50) * 0.068;
  else if (radius <  150) return  0.8 - (radius -   75) * 0.0173;
  else if (radius <  200) return -0.5 - (radius -  150) * 0.002;
  else if (radius <  600) return -0.6 - (radius -  200) * 0.005;
  else if (radius < 1000) return -2.6 - (radius -  600) * 0.0025;
  else if (radius < 2000) return -3.6 - (radius - 1000) * 0.001;
  else if (radius < 4000) return -4.6 - (radius - 2000) * 0.00045;
  else if (radius <10000) return -5.5 - (radius - 4000) * 0.000133;
  else if (radius <12500) return -6.3;
  else return -6.4;
}

/* my_gtk_position_dialog(): Sets up a box containing entry boxes for
 *  RA and Declination */
void my_gtk_position_dialog( GtkWidget **entrybox, GtkWidget *insertionbox,
			     SolidAngle posn, bool celestial_coords )
{
  GtkWidget *table, *labels[8];
  string labeltext[8], boxentries[6];

  labeltext[5] = DEGREE_UTF8;
  labeltext[6] = "'";
  labeltext[7] = "\"";

  if (celestial_coords) {
    labeltext[1] =
      /* TRANSLATORS: This is the abbreviation for
         "hours of right ascension". */
      _("h");
    labeltext[2] =
      /* TRANSLATORS: This is the abbreviation for
         "minutes of right ascension". */
      _("m");
    labeltext[3] =
      /* TRANSLATORS: This is the abbreviation for
         "seconds of right ascension". */
      _("s");
    labeltext[0] = _("Right Ascension:"); labeltext[4] = _("Declination:");
  }
  else {
    labeltext[1] = DEGREE_UTF8; labeltext[2] = "'"; labeltext[3] = "\"";
    labeltext[0] = _("Galactic Longitude:");
    labeltext[4] = _("Galactic Latitude:");
  }

  StringList rastrings=starstrings::ra_to_strs(posn.getPhi(),celestial_coords);
  StringList decstrings = starstrings::dec_to_strs(posn.getTheta());
  for (unsigned int i = 0; i < 3; i++) boxentries[i] = rastrings[i];
  for (unsigned int i = 0; i < 3; i++) boxentries[i + 3] = decstrings[i];

  for (int i = 0; i < 6; i++) {
    starstrings::stripspace(boxentries[i]);
    entrybox[i] = gtk_entry_new_with_max_length (10);
    gtk_entry_set_text (GTK_ENTRY (entrybox[i]), boxentries[i].c_str());
    gtk_widget_set_usize (entrybox[i], 40, 20);
    gtk_widget_show (entrybox[i]);
  }
  for (int i = 0; i < 8; i++) {
    labels[i] = gtk_label_new (labeltext[i].c_str());
    gtk_misc_set_alignment (GTK_MISC (labels[i]), 0.0F, 0.0F);
    gtk_widget_show (labels[i]);
  }

  table = gtk_table_new (2, 7, false);
  gtk_widget_show (table);
  for (int i = 0; i < 2; i++) {
    gtk_table_attach (GTK_TABLE (table), labels[i*4], 0, 1, i, i + 1,
		      (GtkAttachOptions)GTK_FILL, (GtkAttachOptions)GTK_FILL,
		      5, 5);
    for (int j = 0; j < 3; j++) {
      gtk_table_attach (GTK_TABLE (table), entrybox[i*3 + j],
			j*2 + 1, j*2 + 2, i, i + 1,
			(GtkAttachOptions)0, (GtkAttachOptions)0, 5, 5);
      gtk_table_attach (GTK_TABLE (table), labels[i*4 + 1 + j],
			j*2 + 2, j*2 + 3, i, i + 1,
			(GtkAttachOptions)0, (GtkAttachOptions)0, 5, 5);
    }
  }
  gtk_box_pack_start (GTK_BOX (insertionbox), table, false, false, 0);

  return;
}


/* Handler for the Chart->Define Chart menu item */

/* The top-level Chart Define window pointer */
GtkWidget * definedialog;

/* Callback functions */

/* chart_define_callback(): Set the chart parameters to what was specified
 * in the dialog box and update the chart. */
void chart_define_callback(GtkWidget *widget, gpointer chart_define_data)
{
  GtkWidget **data = (GtkWidget **)chart_define_data;
  double oldchartradius = globals::chart_rules.ChartRadius;
  double newchartradius = starmath::atof(
		          gtk_entry_get_text (GTK_ENTRY (data[8])))
		  / distance_conversion[globals::chart_rules.ChartUnits[1]];
  
  if (newchartradius <= 0) {
    my_gtk_error (GTK_WINDOW (definedialog),
	  starstrings::ssprintf(
_("You have entered a negative or\n\
zero value (%s %s) for the chart radius!\n\
Ignoring the change in this parameter."),
	    gtk_entry_get_text (GTK_ENTRY (data[8])),
	    distance_name[globals::chart_rules.ChartUnits[1]]).c_str());
    newchartradius = oldchartradius;
  }

  globals::chart_rules.ChartLocation =
    SolidAngle( starstrings::strs_to_ra(
			    gtk_entry_get_text (GTK_ENTRY (data[0])),
			    gtk_entry_get_text (GTK_ENTRY (data[1])),
			    gtk_entry_get_text (GTK_ENTRY (data[2])),
			    globals::chart_rules.CelestialCoords),
		starstrings::strs_to_dec(
			     gtk_entry_get_text (GTK_ENTRY (data[3])),
			     gtk_entry_get_text (GTK_ENTRY (data[4])),
			     gtk_entry_get_text (GTK_ENTRY (data[5]))
			   )
		).toCartesian(starmath::atof(
				gtk_entry_get_text(GTK_ENTRY(data[6])))
		  / distance_conversion[globals::chart_rules.ChartUnits[1]]);
  
  globals::chart_rules.ChartRadius = newchartradius;

  /* Even if the magnitude autoset feature is on, the user may have set
   *  the dim magnitude limit manually.  We should discard the user's
   *  changes only if the new chart radius differs from the old radius.
   *  In this test, we compare the ratio of new and old radii to 1.0 in order
   *  to get around floating point imprecision. */

  if (globals::chart_rules.ChartAutosetDimmestMagnitude
      && (oldchartradius == 0.0
	  || fabs(newchartradius / oldchartradius - 1.0) > 0.01))
    globals::chart_rules.ChartDimmestMagnitude =
      automagnitude(globals::chart_rules.ChartRadius);

  redraw_all(LOCATION_CHANGE);
  return;
}

/* chart_define_defaults(): Restore the values handled by this dialog box
 * to their default settings and update the chart. */
void chart_define_defaults(GtkWidget *widget, gpointer dummy /* not used */)
{
  double oldchartradius = globals::chart_rules.ChartRadius;
  double newchartradius = defaults::chart_rules.ChartRadius;
  RESET_GLOBAL(chart_rules.ChartLocation);
  RESET_GLOBAL(chart_rules.ChartRadius);

  /* Even if the magnitude autoset feature is on, the user may have set
   *  the dim magnitude limit manually.  We should discard the user's
   *  changes only if the new chart radius differs from the old radius.
   *  In this test, we compare the ratio of new and old radii to 1.0 in order
   *  to get around floating point imprecision. */

  if (globals::chart_rules.ChartAutosetDimmestMagnitude
      && (oldchartradius == 0.0
	  || fabs(newchartradius / oldchartradius - 1.0) > 0.01))
    globals::chart_rules.ChartDimmestMagnitude =
      automagnitude(globals::chart_rules.ChartRadius);

  redraw_all(LOCATION_CHANGE);
  return;
}

/* chart_define_search(): Look up the star in the star name entry box, and if
 *  found, insert its coordinates into all the other entry boxes.  Note: this
 *  function does not do anything else, it is still up to the user to hit
 *  OK to set these coordinates! */
void chart_define_search(GtkWidget *widget, gpointer chart_define_data)
{
  GtkWidget **data = (GtkWidget **)chart_define_data;
  string starname = gtk_entry_get_text (GTK_ENTRY (data[7]));
  StarArray searchresults;

  if (starstrings::isempty(starname)) return;

  StringList ra, dec;

  if (starname == string("[") + _("Chart Center") + string("]")) {
    SolidAngle s = globals::chart_rules.ChartLocation.toSpherical();
    //if (globals::chart_rules.CelestialCoords == GALACTIC)
      //s = s.toGalactic();

    ra = starstrings::ra_to_strs(s.getPhi(),
				 globals::chart_rules.CelestialCoords);
    dec = starstrings::dec_to_strs(s.getTheta());

    gtk_entry_set_text (GTK_ENTRY (data[6]), starstrings::ftoa(
	globals::chart_rules.ChartLocation.magnitude()
	* distance_conversion[globals::chart_rules.ChartUnits[1]], 8).c_str());
  }
  else {

  // Try several different kinds of searches, order of most to least specific
  searchresults.Search(starname, globals::chart_rules.ChartFileNames,
		       globals::chart_rules,
		       /* not case-sensitive */ false,
		       /* exact match        */ true,
		       /* exit on match      */ true);

  // If the search string was quoted, an exact match was forced, so skip
  // the remaining redundant search attempts
  unsigned size = starname.size();
  if (size > 2 && starname[0] == '"' && starname[size - 1] == '"')
    /* nothing */;
  else {

  if (! searchresults.size())
    searchresults.Search(starname, globals::chart_rules.ChartFileNames,
			 globals::chart_rules,
			 /* case-sensitive  */ true,
			 /* substring match */ false,
			 /* exit on match   */ true);
  if (! searchresults.size())
    searchresults.Search(starname, globals::chart_rules.ChartFileNames,
			 globals::chart_rules,
			 /* not case-sensitive */ false,
			 /* substring match    */ false,
			 /* exit on match      */ true);
  } // end "else"

  // if we don't find anything, leave a message in the star name entry
  //  box and return
  if (! searchresults.size()) {
    my_gtk_error (GTK_WINDOW (definedialog),
	starstrings::ssprintf(_("Sorry, star \"%s\" not found."),
	  starname.c_str()).c_str());
    return;
  }

  // otherwise, put the name which matched into the star name entry box,
  //  and put the star coordinates into the coordinate entry boxes.

  StringList infolist = searchresults[0].GetInfo(globals::chart_rules,
						 false /*no dms punctuation*/,
						 SUBFIELD_DELIMITER);
  // where is the star name which matches the search string in the
  //  list of star characteristics?
  unsigned int nameplace = starmath::atoi(infolist[0]);
  unsigned int infolistplace = nameplace ? nameplace + 9 : 1;
  gtk_entry_set_text (GTK_ENTRY (data[7]), infolist[infolistplace].c_str());

  ra = StringList(infolist[2], SUBFIELD_DELIMITER);
  dec = StringList(infolist[3], SUBFIELD_DELIMITER);

  if (ra[0] == string(_("N/A")))
    ra = dec = StringList("00,00,00", ',');

  gtk_entry_set_text (GTK_ENTRY (data[6]), starstrings::ftoa(
      searchresults[0].GetStarDistance()
      * distance_conversion[globals::chart_rules.ChartUnits[1]], 8).c_str());
  } // end "else"

  for (unsigned int i = 0; i < 3; i++) {
    gtk_entry_set_text (GTK_ENTRY (data[i]), ra[i].c_str());
    gtk_entry_set_text (GTK_ENTRY (data[i + 3]), dec[i].c_str());
  }

  return;
}

/* my_gtk_star_coordinates(): Create a dialog that permits entering a
 * star name and clicking "Search" to enter coordinates, or entering
 * coordinates by hand.  Dialog is packed into an existing vbox passed
 * in as an argument.
 *
 * entryboxes must be an array of or pointer to at least 8 GtkWidget *'s
 * and is assumed to have the following layout:
 *
 * entryboxes[0] through [5] - controlled by my_gtk_position_dialog()
 *                             gives the two angular coordinates
 * entryboxes[6]             - distance of coordinate from Sun
 * entryboxes[7]             - entry box for star name
 *
 * */
void my_gtk_star_coordinates(GtkWidget ** entryboxes, GtkWidget * main_vbox)
{
  GtkWidget * lookup_frame, * lookup_hbox, * search;
  GtkWidget * coord_frame, * coord_vbox;
  GtkWidget * distance_hbox, * distance_label;
  string distance_text;

  // First element: a frame containing an hbox containing the star search
  //  entry box
  lookup_frame = gtk_frame_new (
      _("Enter a star name and press \"Search\""));
  gtk_box_pack_start (GTK_BOX (main_vbox), lookup_frame, false, false, 0);
  gtk_widget_show (lookup_frame);

  lookup_hbox = gtk_hbox_new (false, 5);
  gtk_container_add (GTK_CONTAINER (lookup_frame), lookup_hbox);
  gtk_container_set_border_width (GTK_CONTAINER (lookup_hbox), 5);
  gtk_widget_show (lookup_hbox);
  
  entryboxes[7] = gtk_entry_new();
  gtk_box_pack_start (GTK_BOX (lookup_hbox), entryboxes[7], true, true, 5);
  gtk_widget_show (entryboxes[7]);

  search = gtk_button_new_with_mnemonic (_("_Search"));
  gtk_box_pack_start (GTK_BOX (lookup_hbox), search, false, false, 5);
  gtk_widget_set_usize (search, defaults::program_button_width,
			defaults::program_button_height);
  gtk_widget_show (search);

  // Second element: a frame containing a vbox containing the coordinate
  //  entry boxes
  coord_frame = gtk_frame_new (
      _("Or, enter coordinates manually"));
  gtk_box_pack_start (GTK_BOX (main_vbox), coord_frame, false, false, 0);
  gtk_widget_show (coord_frame);

  coord_vbox = gtk_vbox_new (false, 5);
  gtk_container_add (GTK_CONTAINER (coord_frame), coord_vbox);
  gtk_container_set_border_width (GTK_CONTAINER (coord_vbox), 5);
  gtk_widget_show (coord_vbox);

  // fills in entryboxes[0] through [5]
  my_gtk_position_dialog (entryboxes, coord_vbox,
			  globals::chart_rules.ChartLocation.toSpherical(),
			  globals::chart_rules.CelestialCoords);

  distance_hbox = gtk_hbox_new (false, 5);
  gtk_box_pack_start (GTK_BOX (coord_vbox), distance_hbox, false, false, 5);
  gtk_widget_show (distance_hbox);

  distance_label = gtk_label_new ((string(
      _("Distance of position from Sun")) + " ("
      + distance_name[globals::chart_rules.ChartUnits[1]] + "):").c_str());
  gtk_box_pack_start (GTK_BOX (distance_hbox), distance_label, false, false, 5);
  gtk_widget_show (distance_label);

  distance_text =
    starstrings::ftoa(globals::chart_rules.ChartLocation.magnitude()
	* distance_conversion[globals::chart_rules.ChartUnits[1]], 8);
  entryboxes[6] = gtk_entry_new_with_max_length (14);
  gtk_box_pack_start (GTK_BOX(distance_hbox), entryboxes[6], false, false, 5);
  gtk_entry_set_text (GTK_ENTRY (entryboxes[6]), distance_text.c_str());
  gtk_widget_set_usize (entryboxes[6], 80, 20);
  gtk_widget_show (entryboxes[6]);

  // Connect the buttons to signals
  g_signal_connect (G_OBJECT (search), "clicked",
		      G_CALLBACK (chart_define_search),
		      entryboxes);
  g_signal_connect (G_OBJECT (entryboxes[7]), "activate",
		      G_CALLBACK (chart_define_search),
		      entryboxes);
}

/* This function has lots of GTK gui boilerplate because it is the most 
 *  complicated of the dialog boxes.  Sorry about that. */
void chart_define()
{
  GtkWidget *main_vbox, *query;
  GtkWidget *chartrad_hbox, *chartrad_label, *chartrad_entry;
  GtkWidget *OK, *defaults, *cancel;
  static GtkWidget *chart_define_data[9];
  string chartrad_text;

  // The window
  definedialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_resizable (GTK_WINDOW (definedialog), false);
  gtk_window_set_title (GTK_WINDOW (definedialog), _("StarPlot: Define Chart"));
  gtk_window_set_modal (GTK_WINDOW (definedialog), true);
  g_signal_connect (G_OBJECT (definedialog), "destroy",
		      G_CALLBACK (gtk_widget_destroy), NULL);

  // Pack it vertically
  main_vbox = gtk_vbox_new (false, 5);
  gtk_container_set_border_width (GTK_CONTAINER (main_vbox), 10);
  gtk_container_add (GTK_CONTAINER (definedialog), main_vbox);
  gtk_widget_show (main_vbox);

  // Query text
  query = gtk_label_new (
      _("Where should the chart origin be located?"));
  gtk_box_pack_start (GTK_BOX (main_vbox), query, false, false, 0);
  gtk_misc_set_alignment (GTK_MISC (query), 0.0F, 0.0F);
  gtk_widget_show (query);

  // Drop in the two elements of the star search frame
  // (fills in chart_define_data[0] through [7])
  my_gtk_star_coordinates(chart_define_data, main_vbox);

  // Third element: an hbox containing the Chart Radius entry box
  chartrad_hbox = gtk_hbox_new (false, 5);
  gtk_box_pack_start (GTK_BOX (main_vbox), chartrad_hbox, false, false, 10);
  gtk_widget_show (chartrad_hbox);

  chartrad_label = gtk_label_new ((string(
      _("Radius of chart")) + " ("
      + distance_name[globals::chart_rules.ChartUnits[1]] + "):").c_str());
  gtk_box_pack_start (GTK_BOX(chartrad_hbox), chartrad_label, false, false, 5);
  gtk_misc_set_alignment (GTK_MISC (chartrad_label), 0.0F, 0.0F);
  gtk_widget_show (chartrad_label);

  chartrad_text = starstrings::ftoa(globals::chart_rules.ChartRadius
      * distance_conversion[globals::chart_rules.ChartUnits[1]], 8);
  chartrad_entry = gtk_entry_new_with_max_length (14);
  gtk_box_pack_start(GTK_BOX(chartrad_hbox), chartrad_entry, false, false, 10);
  gtk_entry_set_text (GTK_ENTRY (chartrad_entry), chartrad_text.c_str());
  gtk_widget_set_usize (chartrad_entry, 80, 20);
  gtk_widget_show (chartrad_entry);
  chart_define_data[8] = chartrad_entry;

  // Last element: the OK/Defaults/Cancel buttons
  my_gtk_button_bar(&OK, &defaults, &cancel, main_vbox);

  // Connect the buttons to signals
  g_signal_connect (G_OBJECT (OK), "clicked", 
		      G_CALLBACK (chart_define_callback),
		      chart_define_data);
  g_signal_connect (G_OBJECT (defaults), "clicked", 
		      G_CALLBACK (chart_define_defaults), NULL);
  my_gtk_buttons_connect_destroy(OK, defaults, cancel, definedialog);

  // Finally, set the window policy and show it
  gtk_window_set_focus (GTK_WINDOW (definedialog), OK);
  gtk_widget_show (definedialog);

  return;
}


/* Handler for the Chart->Orientation menu item */

/* Callback functions */

/* chart_orient_callback(): Set the chart parameters to what was specified
 * in the dialog box and update the chart. */
void chart_orient_callback(GtkWidget *widget, gpointer chart_orient_data)
{
  GtkWidget **data = (GtkWidget **)chart_orient_data;

  globals::chart_rules.ChartOrientation =
    SolidAngle( starstrings::strs_to_ra(
			    gtk_entry_get_text (GTK_ENTRY (data[0])),
			    gtk_entry_get_text (GTK_ENTRY (data[1])),
			    gtk_entry_get_text (GTK_ENTRY (data[2])),
			    globals::chart_rules.CelestialCoords),
		starstrings::strs_to_dec(
			     gtk_entry_get_text (GTK_ENTRY (data[3])),
			     gtk_entry_get_text (GTK_ENTRY (data[4])),
			     gtk_entry_get_text (GTK_ENTRY (data[5]))
			   )
		); // <sarcasm>what is this, LISP?</sarcasm>

  redraw_all(ORIENTATION_CHANGE);
  return;
}

/* chart_orient_defaults(): Restore the values handled by this dialog box
 * to their default settings and update the chart. */
void chart_orient_defaults(GtkWidget *widget, gpointer data)
{
  RESET_GLOBAL(chart_rules.ChartOrientation);
  redraw_all(ORIENTATION_CHANGE);
  return;
}

void chart_orient()
{
  GtkWidget *orientdialog, *vbox, *entrybox[6], *explanation;
  GtkWidget *OK, *defaults, *cancel;
  static GtkWidget *chart_orient_data[6];

  // The window
  orientdialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_resizable (GTK_WINDOW (orientdialog), false);
  gtk_window_set_title (GTK_WINDOW (orientdialog), _("StarPlot: Orientation"));
  gtk_window_set_modal (GTK_WINDOW (orientdialog), true);
  g_signal_connect (G_OBJECT (orientdialog), "destroy",
		      G_CALLBACK (gtk_widget_destroy), NULL);

  // Pack it vertically
  vbox = gtk_vbox_new (false, 10);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 10);
  gtk_container_add (GTK_CONTAINER (orientdialog), vbox);
  gtk_widget_show (vbox);

  explanation = 
    gtk_label_new (_("Set the orientation of the front of the chart:"));
  gtk_misc_set_alignment (GTK_MISC (explanation), 0.0F, 0.0F);
  gtk_box_pack_start (GTK_BOX (vbox), explanation, false, false, 0);
  gtk_widget_show (explanation);

  // The entry boxes
  my_gtk_position_dialog (entrybox, vbox, globals::chart_rules.ChartOrientation,
		          globals::chart_rules.CelestialCoords);
  for (int i = 0; i < 6; i++)
    chart_orient_data[i] = entrybox[i];

  // set up the buttons
  my_gtk_button_bar (&OK, &defaults, &cancel, vbox);
  g_signal_connect (G_OBJECT (OK), "clicked", 
		      G_CALLBACK (chart_orient_callback),
		      chart_orient_data);
  g_signal_connect (G_OBJECT (defaults), "clicked", 
		      G_CALLBACK (chart_orient_defaults), NULL);
  my_gtk_buttons_connect_destroy (OK, defaults, cancel, orientdialog);

  gtk_window_set_focus (GTK_WINDOW (orientdialog), OK);
  gtk_widget_show (orientdialog);

  return;
}


/* Handler for the Chart->Star Filter menu item */

/* Callback functions */

/* chart_filter_callback(): Set the chart parameters to what was specified
 * in the dialog box and update the chart. */
void chart_filter_callback(GtkWidget *widget, gpointer chart_filter_data)
{
  double origdimmag = globals::chart_rules.ChartDimmestMagnitude;
  bool origautoset = globals::chart_rules.ChartAutosetDimmestMagnitude;
  GtkWidget **data = (GtkWidget **)chart_filter_data;

  for (int i = 0; i < 10; i++)
    globals::chart_rules.StarClasses[i] = GTK_TOGGLE_BUTTON (data[i])->active;

  globals::chart_rules.ChartDimmestMagnitude =
    starmath::atof(gtk_entry_get_text (GTK_ENTRY (data[11])));
  globals::chart_rules.ChartBrightestMagnitude =
    starmath::atof(gtk_entry_get_text (GTK_ENTRY (data[10])));
  globals::chart_rules.ChartAutosetDimmestMagnitude = 
    GTK_TOGGLE_BUTTON (data[12])->active;
  globals::chart_rules.ChartHideCompanions =
    GTK_TOGGLE_BUTTON (data[13])->active;

  /* If the user has just turned on the magnitude autoset feature, we should
   *  autoset the dim magnitude limit, UNLESS the user has also just manually
   *  edited the dim magnitude limit. */
  if (!origautoset && globals::chart_rules.ChartAutosetDimmestMagnitude
      && fabs(origdimmag - globals::chart_rules.ChartDimmestMagnitude) < 0.1)
    globals::chart_rules.ChartDimmestMagnitude =
      automagnitude(globals::chart_rules.ChartRadius);

  redraw_all(FILTER_CHANGE);
  return;
}

/* chart_filter_defaults(): Restore the values handled by this dialog box
 * to their default settings and update the chart. */
void chart_filter_defaults(GtkWidget *widget, gpointer data)
{
  for (int i = 0; i < 10; i++)
    RESET_GLOBAL(chart_rules.StarClasses[i]);
  RESET_GLOBAL(chart_rules.ChartBrightestMagnitude);
  RESET_GLOBAL(chart_rules.ChartDimmestMagnitude);
  RESET_GLOBAL(chart_rules.ChartHideCompanions);
  RESET_GLOBAL(chart_rules.ChartAutosetDimmestMagnitude);

  if (globals::chart_rules.ChartAutosetDimmestMagnitude)
    globals::chart_rules.ChartDimmestMagnitude =
      automagnitude(globals::chart_rules.ChartRadius);

  redraw_all(FILTER_CHANGE);
  return;
}

/* chart_filter(): This function creates the dialog box for turning stars on
 *  or off by spectral class and magnitude. */

void chart_filter()
{ 
  GtkWidget *filterdialog, *mainbox, *classframe, *magframe;
  GtkWidget *classtable, *classboxes[10], *clabel;
  GtkWidget *magtable, *magentrybox[2], *maglabel[2], *magautoset;
  GtkWidget *hidecomps;
  GtkWidget *OK, *defaults, *cancel;
  static GtkWidget *chart_filter_data[14];
  string maglimits[2];

  // The window
  filterdialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_resizable (GTK_WINDOW (filterdialog), false);
  gtk_window_set_title (GTK_WINDOW (filterdialog), _("StarPlot: Star Filter"));
  gtk_window_set_modal (GTK_WINDOW (filterdialog), true);
  g_signal_connect (G_OBJECT (filterdialog), "destroy",
		      G_CALLBACK (gtk_widget_destroy), NULL);

  // Pack it vertically
  mainbox = gtk_vbox_new (false, 10);
  gtk_container_set_border_width (GTK_CONTAINER (mainbox), 10);
  gtk_container_add (GTK_CONTAINER (filterdialog), mainbox);
  gtk_widget_show (mainbox);

  // Frames in the window
  classframe = gtk_frame_new (
      _("By Spectral Class"));
  magframe = gtk_frame_new (_("By Magnitude"));
  gtk_box_pack_start (GTK_BOX (mainbox), classframe, true, false, 0);
  gtk_box_pack_start (GTK_BOX (mainbox), magframe, true, false, 0);
  gtk_widget_show (classframe);
  gtk_widget_show (magframe);

  // Create the spectral class checkboxes
  classtable = gtk_table_new (5, 4, false);
  gtk_container_add (GTK_CONTAINER (classframe), classtable);
  gtk_widget_show (classtable);

  clabel = gtk_label_new (_("Indicate which stellar classes to include:"));
  gtk_widget_show (clabel);
  gtk_misc_set_alignment (GTK_MISC (clabel), 0.0F, 0.0F);
  gtk_table_attach (GTK_TABLE (classtable), clabel, 0, 4, 0, 1,
		    (GtkAttachOptions)(GTK_FILL | GTK_EXPAND),
		    (GtkAttachOptions)(GTK_FILL | GTK_EXPAND), 5, 5);

  classboxes[0] = 
    gtk_check_button_new_with_mnemonic (
	/* TRANSLATORS: "Wolf-Rayet" is a proper name for a type of star. */
	_("Class _O / Wolf-Rayet"));
  classboxes[1] = gtk_check_button_new_with_mnemonic (_("Class _B"));
  classboxes[2] = gtk_check_button_new_with_mnemonic (_("Class _A"));
  classboxes[3] = gtk_check_button_new_with_mnemonic (_("Class _F"));
  classboxes[4] = gtk_check_button_new_with_mnemonic (_("Class _G"));
  classboxes[5] = gtk_check_button_new_with_mnemonic (_("Class _K"));
  classboxes[6] = gtk_check_button_new_with_mnemonic (_("Class _M"));
  classboxes[7] = gtk_check_button_new_with_mnemonic (_("_Dwarf stars"));
  classboxes[9] =
    gtk_check_button_new_with_mnemonic (_("Show _Non-Stellar Objects"));
  classboxes[8] = 
    gtk_check_button_new_with_mnemonic (_("Show _Unclassified Objects"));

  for (int i = 0; i < 10; i++) {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (classboxes[i]),
				  globals::chart_rules.StarClasses[i]);
    gtk_widget_show (classboxes[i]);
    chart_filter_data[i] = classboxes[i];
  }

  for (int i = 0; i < 4; i++) {
    gtk_table_attach_defaults (GTK_TABLE (classtable), classboxes[2*i],
			       i, i + 1, 1, 2);
    gtk_table_attach_defaults (GTK_TABLE (classtable), classboxes[2*i + 1],
			       i, i + 1, 2, 3);
  }
  gtk_table_attach_defaults (GTK_TABLE (classtable), classboxes[9], 0,2,3,4);
  gtk_table_attach_defaults (GTK_TABLE (classtable), classboxes[8], 0,2,4,5);

  // Create the magnitude text boxes
  maglimits[0]=starstrings::ftoa(globals::chart_rules.ChartBrightestMagnitude);
  maglimits[1]=starstrings::ftoa(globals::chart_rules.ChartDimmestMagnitude);
  maglabel[0] = gtk_label_new (_("Smallest (brightest) allowed magnitude: "));
  maglabel[1] = gtk_label_new (_("Largest (dimmest) allowed magnitude: "));

  magtable = gtk_table_new (3, 2, false);
  gtk_container_add (GTK_CONTAINER (magframe), magtable);
  gtk_widget_show (magtable);

  for (int i = 0; i < 2; i++) {
    gtk_widget_show (maglabel[i]);
    gtk_misc_set_alignment (GTK_MISC (maglabel[i]), 0.0F, 0.0F);

    magentrybox[i] = gtk_entry_new_with_max_length (6);
    gtk_entry_set_text (GTK_ENTRY (magentrybox[i]), maglimits[i].c_str());
    gtk_widget_set_usize (magentrybox[i], 50, 20);
    gtk_widget_show (magentrybox[i]);

    gtk_table_attach (GTK_TABLE (magtable), maglabel[i], 0, 1, i, i + 1,
		      (GtkAttachOptions)GTK_FILL, (GtkAttachOptions)GTK_FILL,
		      5, 5);
    gtk_table_attach (GTK_TABLE (magtable), magentrybox[i], 1, 2, i, i + 1,
		      (GtkAttachOptions)0, (GtkAttachOptions)0, 0, 5);

    chart_filter_data[10 + i] = magentrybox[i];
  }

  magautoset = gtk_check_button_new_with_mnemonic
    (_("Automatically set dim magnitude limit\nbased on _chart radius"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (magautoset),
    globals::chart_rules.ChartAutosetDimmestMagnitude);
  gtk_table_attach (GTK_TABLE (magtable), magautoset, 0, 2, 2, 3,
		    (GtkAttachOptions)0, (GtkAttachOptions)0, 0, 5);
  gtk_widget_show (magautoset);
  chart_filter_data[12] = magautoset;

  hidecomps = gtk_check_button_new_with_mnemonic
    (_("_Hide nearby companion stars"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (hidecomps),
    globals::chart_rules.ChartHideCompanions);
  gtk_box_pack_start (GTK_BOX (mainbox), hidecomps, true, false, 0);
  gtk_widget_show (hidecomps);
  chart_filter_data[13] = hidecomps;

  // set up the buttons
  my_gtk_button_bar (&OK, &defaults, &cancel, mainbox);
  g_signal_connect (G_OBJECT (OK), "clicked", 
		      G_CALLBACK (chart_filter_callback),
		      chart_filter_data);
  g_signal_connect (G_OBJECT (defaults), "clicked", 
		      G_CALLBACK (chart_filter_defaults), NULL);
  my_gtk_buttons_connect_destroy (OK, defaults, cancel, filterdialog);

  gtk_window_set_focus (GTK_WINDOW (filterdialog), OK);
  gtk_widget_show (filterdialog);

  return;
}
