#' Password Storage
#'
#' Wrapper that implements best practices for storing passwords based on scrypt with
#' a random salt.
#'
#' The \link{password_store} function returns an ASCII encoded string which contains
#' the result of a memory-hard, CPU-intensive hash function along with the automatically
#' generated salt and other parameters required to verify the password. Use
#' \link{password_verify} to verify a password from this string.
#'
#' @export
#' @rdname password
#' @name Password storage
#' @aliases password
#' @param password a string of length one with a password
#' @param hash a hash string of length one generated by \code{password_store}
#' @useDynLib sodium R_password_hash
#' @references \url{https://doc.libsodium.org/password_hashing/}
#' @examples # Example password
#' password <- "I like cookies"
#'
#' # Hash is what you store in the database
#' hash <- password_store(password)
#'
#' # To verify the password when the user logs in
#' stopifnot(password_verify(hash, password))
password_store <- function(password){
  .Call(R_password_hash, password)
}

#' @export
#' @rdname password
#' @useDynLib sodium R_password_verify
password_verify <- function(hash, password){
  .Call(R_password_verify, hash, password)
}
