% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/iidCox.R
\name{iidCox}
\alias{iidCox}
\alias{iidCox.coxph}
\alias{iidCox.cph}
\alias{iidCox.phreg}
\alias{iidCox.CauseSpecificCox}
\title{Extract iid decomposition from a Cox model}
\usage{
iidCox(
  object,
  newdata,
  baseline.iid,
  tau.hazard,
  tau.max,
  store.iid,
  keep.times,
  return.object
)

\method{iidCox}{coxph}(
  object,
  newdata = NULL,
  baseline.iid = TRUE,
  tau.hazard = NULL,
  tau.max = NULL,
  store.iid = "full",
  keep.times = TRUE,
  return.object = TRUE
)

\method{iidCox}{cph}(
  object,
  newdata = NULL,
  baseline.iid = TRUE,
  tau.hazard = NULL,
  tau.max = NULL,
  store.iid = "full",
  keep.times = TRUE,
  return.object = TRUE
)

\method{iidCox}{phreg}(
  object,
  newdata = NULL,
  baseline.iid = TRUE,
  tau.hazard = NULL,
  tau.max = NULL,
  store.iid = "full",
  keep.times = TRUE,
  return.object = TRUE
)

\method{iidCox}{CauseSpecificCox}(
  object,
  newdata = NULL,
  baseline.iid = TRUE,
  tau.hazard = NULL,
  tau.max = NULL,
  store.iid = "full",
  keep.times = TRUE,
  return.object = TRUE
)
}
\arguments{
\item{object}{object The fitted Cox regression model object either
obtained with \code{coxph} (survival package) or \code{cph}
(rms package).}

\item{newdata}{[data.frame] Optional new data at which to do iid decomposition}

\item{baseline.iid}{[logical] Should the influence function for the baseline hazard be computed.}

\item{tau.hazard}{[numeric vector] the vector of times at which the i.i.d decomposition of the baseline hazard will be computed}

\item{tau.max}{[numeric] latest time at which the i.i.d decomposition of the baseline hazard will be computed. Alternative to \code{tau.hazard}.}

\item{store.iid}{[character] the method used to compute the influence function and the standard error.
Can be \code{"full"} or \code{"minimal"}. See the details section.}

\item{keep.times}{[logical] If \code{TRUE} add the evaluation times to the output.}

\item{return.object}{[logical] If \code{TRUE} return the object where the iid decomposition has been added.
Otherwise return a list (see the return section)}
}
\value{
For Cox models, it returns the object with an additional iid slot (i.e. \code{object$iid}).
It is a list containing:
\itemize{
 \item IFbeta: Influence function for the regression coefficient.
 \item IFhazard: Time differential of the influence function of the hazard.
 \item IFcumhazard: Influence function of the cumulative hazard.
 \item calcIFhazard: Elements used to compute the influence function at a given time.
 \item time: Times at which the influence function has been evaluated.
 \item etime1.min: Time of first event (i.e. jump) in each strata.
 \item etime.max: Last observation time (i.e. jump or censoring) in each strata.
 \item indexObs: Index of the observation in the original dataset.}

For Cause-Specific Cox models,
it returns the object with an additional iid slot for each model (e.g. \code{object$models[[1]]iid}).
}
\description{
Compute the influence function for each observation used to estimate the model
}
\details{
This function implements the first three formula (no number,10,11) of the subsection
"Empirical estimates" in Ozenne et al. (2017).
If there is no event in a strata, the influence function for the baseline hazard is set to 0. 

\bold{Argument store.iid}:
If \code{n} denotes the sample size, \code{J} the number of jump times, and \code{p} the number of coefficients:
\itemize{
\item \code{store.iid="full"} exports the influence function for the coefficients and the baseline hazard at each event time.
\item \code{store.iid="minimal"} exports the influence function for the coefficients. For the
baseline hazard it only computes the quantities necessary to compute the influence function in order to save memory. }

More details can be found in appendix B of Ozenne et al. (2017).
}
\examples{
library(survival)
library(data.table)
library(prodlim)
set.seed(10)
d <- sampleData(100, outcome = "survival")[,.(eventtime,event,X1,X6)]
setkey(d, eventtime)

m.cox <- coxph(Surv(eventtime, event) ~ X1+X6, data = d, y = TRUE, x = TRUE)
system.time(IF.cox <- iidCox(m.cox))

IF.cox.all <- iidCox(m.cox, tau.hazard = sort(unique(c(7,d$eventtime))))
IF.cox.beta <- iidCox(m.cox, baseline.iid = FALSE)

}
\references{
Brice Ozenne, Anne Lyngholm Sorensen, Thomas Scheike, Christian Torp-Pedersen and Thomas Alexander Gerds.
riskRegression: Predicting the Risk of an Event using Cox Regression Models.
The R Journal (2017) 9:2, pages 440-460.
}
