/*
 * Copyright (c) 2022-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SME2)

#include <algorithm>
#include <cstddef>

namespace arm_conv {
namespace depthwise {

void sme2_fp32bf16fp32_planar_3x3_s1_4rows_dot_za_impl(
  const float *inptr,
  size_t ld_in_row,
  size_t ld_in_col,
  size_t ld_in_vl,
  unsigned int pad_top,
  unsigned int valid_input_rows,
  unsigned int pad_left,
  unsigned int valid_input_cols,
  const float *weights,
  const float *bias,
  float **outptrs,
  const size_t *outlds,
  const size_t *outvllds,
  unsigned int output_cols,
  unsigned int start_channel,
  unsigned int valid_channels,
  float act_min,
  float act_max
)
{
  struct Args
  {
    const float *inptr;
    size_t ld_in_vl;
    long unsigned int pad_top, pad_bottom, pad_left;
    const float *weights;
    const float *bias;
    long unsigned int input_cols, output_cols;
    float **outptrs;
    const size_t *ld_out_cols;
    const size_t *ld_out_vls;
    long unsigned int current_channel, n_channels;
    float clamp_min, clamp_max;
  };

  Args args = { inptr, ld_in_vl, pad_top, 6u - std::min(6u, pad_top + valid_input_rows), pad_left, weights, bias, valid_input_cols, output_cols, outptrs, outlds, outvllds, start_channel, valid_channels, act_min, act_max };

  __asm__ __volatile__(
    "ldr x7, [%x[args], %[offsetof_Args_pad_bottom]]\n"
    "mov x20, #0x6\n"
    ".inst 0xd503477f  // SMSTART ZA\n"
    "ldr x17, [%x[args], %[offsetof_Args_pad_top]]\n"
    "ptrue p2.b\n"
    "ldr x16, [%x[args], %[offsetof_Args_n_channels]]\n"
    "ld1rw { z23.s }, p2/Z, [%x[args], %[offsetof_Args_clamp_min]]\n"
    "sub x20, x20, x7\n"
    "ldr x15, [%x[args], %[offsetof_Args_current_channel]]\n"
    "ld1rw { z13.s }, p2/Z, [%x[args], %[offsetof_Args_clamp_max]]\n"
    "whilelt p1.s, XZR, x16\n"
    "whilelt p9.s, XZR, x20\n"
    "whilelt p8.s, XZR, x17\n"
    "eor p8.b, p2/Z, p8.b, p9.b\n"
    "1:"  // Channel loop
    "ldr x20, [%x[args], %[offsetof_Args_bias]]\n"
    "fmov z4.s, #0x0\n"
    "cbz x20, 2f\n"
    "ld1w { z4.s }, p1/Z, [x20, x15, LSL #2]\n"
    "2:"  // Load bias: Done
    "ldr x25, [%x[args], %[offsetof_Args_weights]]\n"
    "fmov z1.s, #0x0\n"
    "fmov z12.s, #0x0\n"
    "mov x24, #0x6\n"
    "ldr x14, [%x[args], %[offsetof_Args_input_cols]]\n"
    "fmov z15.s, #0x0\n"
    "add x20, x17, x7\n"
    "lsl x23, %x[ld_in_row], #0x2\n"
    "ldr x13, [%x[args], %[offsetof_Args_inptr]]\n"
    "mov z5.d, z4.d\n"
    "mov x8, #0x0\n"
    "sub x24, x24, x20\n"
    "mov x22, x25\n"
    "incb x25\n"
    "ldr x11, [%x[args], %[offsetof_Args_output_cols]]\n"
    "ld1w { z31.s }, p2/Z, [x22]\n"
    "incb x22, ALL, MUL #3\n"
    "sub x20, x14, #0x1\n"
    "ld1w { z21.s }, p2/Z, [x22]\n"
    "incb x22, ALL, MUL #3\n"
    "orr x21, x20, %x[ld_in_col], LSL #18\n"
    "ld1w { z10.s }, p2/Z, [x22]\n"
    "mov x20, x25\n"
    "incb x25\n"
    ".inst 0x648aabe1  // bfcvtnt z1.h, p2/M, z31.s\n"
    ".inst 0x658aabe2  // bfcvt z2.h, p2/M, z31.s\n"
    "ld1w { z8.s }, p2/Z, [x20]\n"
    "incb x20, ALL, MUL #3\n"
    ".inst 0x658aaaa3  // bfcvt z3.h, p2/M, z21.s\n"
    "orr x21, x16, x21, LSL #20\n"
    "madd x23, x23, x17, x13\n"
    ".inst 0x658aa94e  // bfcvt z14.h, p2/M, z10.s\n"
    "lsl x21, x21, #0x2\n"
    ".inst 0x658aa907  // bfcvt z7.h, p2/M, z8.s\n"
    ".inst 0x648aa90c  // bfcvtnt z12.h, p2/M, z8.s\n"
    ".inst 0x648aaaa2  // bfcvtnt z2.h, p2/M, z21.s\n"
    "ld1w { z29.s }, p2/Z, [x20]\n"
    "incb x20, ALL, MUL #3\n"
    ".inst 0x648aa943  // bfcvtnt z3.h, p2/M, z10.s\n"
    "ld1w { z20.s }, p2/Z, [x20]\n"
    "mov x20, x25\n"
    "ld1w { z17.s }, p2/Z, [x20]\n"
    "incb x20, ALL, MUL #3\n"
    ".inst 0x648aaba7  // bfcvtnt z7.h, p2/M, z29.s\n"
    ".inst 0x658aaba0  // bfcvt z0.h, p2/M, z29.s\n"
    "ld1w { z25.s }, p2/Z, [x20]\n"
    "incb x20, ALL, MUL #3\n"
    ".inst 0x658aaa86  // bfcvt z6.h, p2/M, z20.s\n"
    ".inst 0x658aaa29  // bfcvt z9.h, p2/M, z17.s\n"
    ".inst 0x648aaa2f  // bfcvtnt z15.h, p2/M, z17.s\n"
    ".inst 0x658aab28  // bfcvt z8.h, p2/M, z25.s\n"
    ".inst 0x648aaa80  // bfcvtnt z0.h, p2/M, z20.s\n"
    "ld1w { z10.s }, p2/Z, [x20]\n"
    ".inst 0x648aab29  // bfcvtnt z9.h, p2/M, z25.s\n"
    ".inst 0x648aa948  // bfcvtnt z8.h, p2/M, z10.s\n"
    ".inst 0x658aa94a  // bfcvt z10.h, p2/M, z10.s\n"
    "3:"  // Issue prefetches
    "subs x24, x24, #0x1\n"
    ".inst 0xf8b54afc  // rprfm pldstrm, x21, [x23]\n"
    "add x23, x23, %x[ld_in_col], LSL #2\n"
    "bgt 3b\n"
    "ldr x22, [%x[args], %[offsetof_Args_outptrs]]\n"
    "lsl x21, %x[ld_in_row], #0x2\n"
    ".inst 0xc0040880  // mova za.d[x8, #0], { z4.d-z5.d }\n"
    "mov x10, #0x2\n"
    "ldr x20, [%x[args], %[offsetof_Args_ld_out_cols]]\n"
    "msub x13, x17, x21, x13\n"
    ".inst 0xc0040881  // mova za.d[x8, #1], { z4.d-z5.d }\n"
    "ldr x21, [%x[args], %[offsetof_Args_pad_left]]\n"
    ".inst 0xc0040882  // mova za.d[x8, #2], { z4.d-z5.d }\n"
    "ldp x9, x28, [x22], #0x10\n"
    ".inst 0xc0040883  // mova za.d[x8, #3], { z4.d-z5.d }\n"
    "ldp x27, x26, [x20], #0x10\n"
    ".inst 0xc0040884  // mova za.d[x8, #4], { z4.d-z5.d }\n"
    ".inst 0xc0040885  // mova za.d[x8, #5], { z4.d-z5.d }\n"
    "ldp x25, x24, [x22], #0x10\n"
    "ldp x23, x22, [x20], #0x10\n"
    "cbz x21, 5f\n"
    "cmp x21, x10\n"
    "csel x20, x21, x10, LT\n"
    "sub x21, x21, x20\n"
    "sub x10, x10, x20\n"
    "cbz x21, 5f\n"
    ".inst 0xc006081c  // mova { z28.d-z29.d }, za.d[x8, #0]\n"
    "sub x11, x11, x21\n"
    ".inst 0xc006083e  // mova { z30.d-z31.d }, za.d[x8, #1]\n"
    ".inst 0xc1adcafc  // fclamp { z28.s-z31.s }, z23.s, z13.s\n"
    "4:"  // Left padding
    "subs x21, x21, #0x1\n"
    "st1w { z28.s }, p1, [x9]\n"
    "add x9, x9, x27, LSL #2\n"
    "st1w { z30.s }, p1, [x28]\n"
    "add x28, x28, x26, LSL #2\n"
    "st1w { z29.s }, p1, [x25]\n"
    "add x25, x25, x23, LSL #2\n"
    "st1w { z31.s }, p1, [x24]\n"
    "add x24, x24, x22, LSL #2\n"
    "bgt 4b\n"
    "5:"  // Left padding: End
    "adds XZR, x17, x7\n"
    "bne 10f\n"
    "cbz x10, 8f\n"
    "cmp x10, #0x1\n"
    "sub x14, x14, x10\n"
    "beq 7f\n"
    "6:"  // Unpadded: 2 priming loads
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "ld1w { z21.s }, p1/Z, [x13]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z19.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z24.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z17.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x658aaabb  // bfcvt z27.h, p2/M, z21.s\n"
    "ld1w { z22.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x658aab1c  // bfcvt z28.h, p2/M, z24.s\n"
    "ld1w { z16.s }, p1/Z, [x20]\n"
    ".inst 0x648aaa7b  // bfcvtnt z27.h, p2/M, z19.s\n"
    ".inst 0x658aaadd  // bfcvt z29.h, p2/M, z22.s\n"
    ".inst 0x648aaa3c  // bfcvtnt z28.h, p2/M, z17.s\n"
    ".inst 0x648aaa1d  // bfcvtnt z29.h, p2/M, z16.s\n"
    ".inst 0xc1221370  // bfdot za.s[x8, 0], { z27.h-z28.h }, z2.h\n"
    ".inst 0xc1211371  // bfdot za.s[x8, 1], { z27.h-z28.h }, z1.h\n"
    ".inst 0xc12e1390  // bfdot za.s[x8, 0], { z28.h-z29.h }, z14.h\n"
    ".inst 0xc1231391  // bfdot za.s[x8, 1], { z28.h-z29.h }, z3.h\n"
    "7:"  // Unpadded: 1 priming loads
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "ld1w { z24.s }, p1/Z, [x13]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z30.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z18.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x658aab13  // bfcvt z19.h, p2/M, z24.s\n"
    "ld1w { z16.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x658aaa94  // bfcvt z20.h, p2/M, z20.s\n"
    "ld1w { z17.s }, p1/Z, [x20]\n"
    ".inst 0x648aabd3  // bfcvtnt z19.h, p2/M, z30.s\n"
    ".inst 0x658aaa15  // bfcvt z21.h, p2/M, z16.s\n"
    ".inst 0x648aaa54  // bfcvtnt z20.h, p2/M, z18.s\n"
    ".inst 0x648aaa35  // bfcvtnt z21.h, p2/M, z17.s\n"
    ".inst 0xc1271270  // bfdot za.s[x8, 0], { z19.h-z20.h }, z7.h\n"
    ".inst 0xc12c1271  // bfdot za.s[x8, 1], { z19.h-z20.h }, z12.h\n"
    ".inst 0xc1221272  // bfdot za.s[x8, 2], { z19.h-z20.h }, z2.h\n"
    ".inst 0xc1211273  // bfdot za.s[x8, 3], { z19.h-z20.h }, z1.h\n"
    ".inst 0xc1261290  // bfdot za.s[x8, 0], { z20.h-z21.h }, z6.h\n"
    ".inst 0xc1201291  // bfdot za.s[x8, 1], { z20.h-z21.h }, z0.h\n"
    ".inst 0xc12e1292  // bfdot za.s[x8, 2], { z20.h-z21.h }, z14.h\n"
    ".inst 0xc1231293  // bfdot za.s[x8, 3], { z20.h-z21.h }, z3.h\n"
    "8:"  // Unpadded: 0 priming loads
    "cbz x14, 16f\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "ld1w { z17.s }, p1/Z, [x13]\n"
    "sub x14, x14, #0x1\n"
    "ld1w { z16.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "sub x11, x11, #0x1\n"
    "ld1w { z21.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "cmp x14, x11\n"
    "ld1w { z27.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x658aaa3c  // bfcvt z28.h, p2/M, z17.s\n"
    "csel x21, x14, x11, LT\n"
    "ld1w { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    ".inst 0x658aaabd  // bfcvt z29.h, p2/M, z21.s\n"
    "ld1w { z17.s }, p1/Z, [x20]\n"
    "sub x11, x11, x21\n"
    ".inst 0x658aaa9e  // bfcvt z30.h, p2/M, z20.s\n"
    ".inst 0x648aaa1c  // bfcvtnt z28.h, p2/M, z16.s\n"
    ".inst 0x648aab7d  // bfcvtnt z29.h, p2/M, z27.s\n"
    ".inst 0x648aaa3e  // bfcvtnt z30.h, p2/M, z17.s\n"
    "cbz x21, 15f\n"
    "9:"  // Unpadded: Main loop
    ".inst 0xc1291390  // bfdot za.s[x8, 0], { z28.h-z29.h }, z9.h\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "ld1w { z19.s }, p1/Z, [x13]\n"
    "subs x21, x21, #0x1\n"
    ".inst 0xc12f1391  // bfdot za.s[x8, 1], { z28.h-z29.h }, z15.h\n"
    "ld1w { z18.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z17.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1271392  // bfdot za.s[x8, 2], { z28.h-z29.h }, z7.h\n"
    ".inst 0xc12c1393  // bfdot za.s[x8, 3], { z28.h-z29.h }, z12.h\n"
    "ld1w { z21.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc12a13b0  // bfdot za.s[x8, 0], { z29.h-z30.h }, z10.h\n"
    "ld1w { z16.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc12813b1  // bfdot za.s[x8, 1], { z29.h-z30.h }, z8.h\n"
    "ld1w { z20.s }, p1/Z, [x20]\n"
    ".inst 0xc1221394  // bfdot za.s[x8, 4], { z28.h-z29.h }, z2.h\n"
    ".inst 0xc1211395  // bfdot za.s[x8, 5], { z28.h-z29.h }, z1.h\n"
    ".inst 0x658aaa7c  // bfcvt z28.h, p2/M, z19.s\n"
    ".inst 0xc12613b2  // bfdot za.s[x8, 2], { z29.h-z30.h }, z6.h\n"
    ".inst 0xc12013b3  // bfdot za.s[x8, 3], { z29.h-z30.h }, z0.h\n"
    ".inst 0xc12e13b4  // bfdot za.s[x8, 4], { z29.h-z30.h }, z14.h\n"
    ".inst 0xc12313b5  // bfdot za.s[x8, 5], { z29.h-z30.h }, z3.h\n"
    ".inst 0x658aaa3d  // bfcvt z29.h, p2/M, z17.s\n"
    ".inst 0x658aaa1e  // bfcvt z30.h, p2/M, z16.s\n"
    ".inst 0xc0060810  // mova { z16.d-z17.d }, za.d[x8, #0]\n"
    ".inst 0x648aaa5c  // bfcvtnt z28.h, p2/M, z18.s\n"
    ".inst 0xc0060832  // mova { z18.d-z19.d }, za.d[x8, #1]\n"
    "add x8, x8, #0x2\n"
    ".inst 0xc0040884  // mova za.d[x8, #4], { z4.d-z5.d }\n"
    ".inst 0xc0040885  // mova za.d[x8, #5], { z4.d-z5.d }\n"
    ".inst 0x648aaabd  // bfcvtnt z29.h, p2/M, z21.s\n"
    ".inst 0x648aaa9e  // bfcvtnt z30.h, p2/M, z20.s\n"
    ".inst 0xc1adcaf0  // fclamp { z16.s-z19.s }, z23.s, z13.s\n"
    "st1w { z16.s }, p1, [x9]\n"
    "add x9, x9, x27, LSL #2\n"
    "st1w { z18.s }, p1, [x28]\n"
    "add x28, x28, x26, LSL #2\n"
    "st1w { z17.s }, p1, [x25]\n"
    "add x25, x25, x23, LSL #2\n"
    "st1w { z19.s }, p1, [x24]\n"
    "add x24, x24, x22, LSL #2\n"
    "bgt 9b\n"
    "b 15f\n"
    "10:"  // Padded
    "cbz x10, 13f\n"
    "cmp x10, #0x1\n"
    "sub x14, x14, x10\n"
    "beq 12f\n"
    "11:"  // Padded: 2 priming loads
    "mov x12, #0x0\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z17.s }, p0/Z, [x13]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x658aaa33  // bfcvt z19.h, p2/M, z17.s\n"
    "ld1w { z17.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    ".inst 0x648aaa13  // bfcvtnt z19.h, p2/M, z16.s\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x658aaa34  // bfcvt z20.h, p2/M, z17.s\n"
    "ld1w { z31.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x648aaa14  // bfcvtnt z20.h, p2/M, z16.s\n"
    ".inst 0x658aabf5  // bfcvt z21.h, p2/M, z31.s\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0xc1221270  // bfdot za.s[x8, 0], { z19.h-z20.h }, z2.h\n"
    ".inst 0xc1211271  // bfdot za.s[x8, 1], { z19.h-z20.h }, z1.h\n"
    ".inst 0x648aaa15  // bfcvtnt z21.h, p2/M, z16.s\n"
    ".inst 0xc12e1290  // bfdot za.s[x8, 0], { z20.h-z21.h }, z14.h\n"
    ".inst 0xc1231291  // bfdot za.s[x8, 1], { z20.h-z21.h }, z3.h\n"
    "12:"  // Padded: 1 priming loads
    "mov x12, #0x0\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z17.s }, p0/Z, [x13]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x658aaa34  // bfcvt z20.h, p2/M, z17.s\n"
    "ld1w { z17.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    ".inst 0x648aaa14  // bfcvtnt z20.h, p2/M, z16.s\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x658aaa35  // bfcvt z21.h, p2/M, z17.s\n"
    "ld1w { z25.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x648aaa15  // bfcvtnt z21.h, p2/M, z16.s\n"
    ".inst 0x658aab36  // bfcvt z22.h, p2/M, z25.s\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0xc1271290  // bfdot za.s[x8, 0], { z20.h-z21.h }, z7.h\n"
    ".inst 0xc12c1291  // bfdot za.s[x8, 1], { z20.h-z21.h }, z12.h\n"
    ".inst 0x648aaa16  // bfcvtnt z22.h, p2/M, z16.s\n"
    ".inst 0xc1221292  // bfdot za.s[x8, 2], { z20.h-z21.h }, z2.h\n"
    ".inst 0xc1211293  // bfdot za.s[x8, 3], { z20.h-z21.h }, z1.h\n"
    ".inst 0xc12612b0  // bfdot za.s[x8, 0], { z21.h-z22.h }, z6.h\n"
    ".inst 0xc12012b1  // bfdot za.s[x8, 1], { z21.h-z22.h }, z0.h\n"
    ".inst 0xc12e12b2  // bfdot za.s[x8, 2], { z21.h-z22.h }, z14.h\n"
    ".inst 0xc12312b3  // bfdot za.s[x8, 3], { z21.h-z22.h }, z3.h\n"
    "13:"  // Padded: 0 priming loads
    "cbz x14, 16f\n"
    "mov x12, #0x0\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "sub x14, x14, #0x1\n"
    "sub x11, x11, #0x1\n"
    "cmp x14, x11\n"
    "ld1w { z16.s }, p0/Z, [x13]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "csel x21, x14, x11, LT\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "sub x11, x11, x21\n"
    "ld1w { z17.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x658aaa1c  // bfcvt z28.h, p2/M, z16.s\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    ".inst 0x648aaa3c  // bfcvtnt z28.h, p2/M, z17.s\n"
    "ld1w { z17.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x658aaa1d  // bfcvt z29.h, p2/M, z16.s\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x648aaa3d  // bfcvtnt z29.h, p2/M, z17.s\n"
    ".inst 0x658aaa1e  // bfcvt z30.h, p2/M, z16.s\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0x648aaa1e  // bfcvtnt z30.h, p2/M, z16.s\n"
    "cbz x21, 15f\n"
    "14:"  // Padded: Main loop
    "mov x12, #0x0\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1291390  // bfdot za.s[x8, 0], { z28.h-z29.h }, z9.h\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc12f1391  // bfdot za.s[x8, 1], { z28.h-z29.h }, z15.h\n"
    "subs x21, x21, #0x1\n"
    ".inst 0xc1271392  // bfdot za.s[x8, 2], { z28.h-z29.h }, z7.h\n"
    ".inst 0xc12c1393  // bfdot za.s[x8, 3], { z28.h-z29.h }, z12.h\n"
    "ld1w { z17.s }, p0/Z, [x13]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc12a13b0  // bfdot za.s[x8, 0], { z29.h-z30.h }, z10.h\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    ".inst 0xc12813b1  // bfdot za.s[x8, 1], { z29.h-z30.h }, z8.h\n"
    ".inst 0xc1221394  // bfdot za.s[x8, 4], { z28.h-z29.h }, z2.h\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1211395  // bfdot za.s[x8, 5], { z28.h-z29.h }, z1.h\n"
    ".inst 0xc12613b2  // bfdot za.s[x8, 2], { z29.h-z30.h }, z6.h\n"
    ".inst 0x658aaa3c  // bfcvt z28.h, p2/M, z17.s\n"
    ".inst 0xc12013b3  // bfdot za.s[x8, 3], { z29.h-z30.h }, z0.h\n"
    "ld1w { z17.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc12e13b4  // bfdot za.s[x8, 4], { z29.h-z30.h }, z14.h\n"
    "mov x12, #0x4\n"
    ".inst 0xc12313b5  // bfdot za.s[x8, 5], { z29.h-z30.h }, z3.h\n"
    ".inst 0xc0060818  // mova { z24.d-z25.d }, za.d[x8, #0]\n"
    ".inst 0x648aaa1c  // bfcvtnt z28.h, p2/M, z16.s\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc006083a  // mova { z26.d-z27.d }, za.d[x8, #1]\n"
    ".inst 0x658aaa3d  // bfcvt z29.h, p2/M, z17.s\n"
    "add x8, x8, #0x2\n"
    ".inst 0xc0040884  // mova za.d[x8, #4], { z4.d-z5.d }\n"
    "ld1w { z31.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc0040885  // mova za.d[x8, #5], { z4.d-z5.d }\n"
    ".inst 0xc1adcaf8  // fclamp { z24.s-z27.s }, z23.s, z13.s\n"
    ".inst 0x648aaa1d  // bfcvtnt z29.h, p2/M, z16.s\n"
    ".inst 0x658aabfe  // bfcvt z30.h, p2/M, z31.s\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    "st1w { z24.s }, p1, [x9]\n"
    "add x9, x9, x27, LSL #2\n"
    "st1w { z26.s }, p1, [x28]\n"
    "add x28, x28, x26, LSL #2\n"
    "st1w { z25.s }, p1, [x25]\n"
    "add x25, x25, x23, LSL #2\n"
    ".inst 0x648aaa1e  // bfcvtnt z30.h, p2/M, z16.s\n"
    "st1w { z27.s }, p1, [x24]\n"
    "add x24, x24, x22, LSL #2\n"
    "bgt 14b\n"
    "15:"  // Main loop tail
    ".inst 0xc1291390  // bfdot za.s[x8, 0], { z28.h-z29.h }, z9.h\n"
    ".inst 0xc12f1391  // bfdot za.s[x8, 1], { z28.h-z29.h }, z15.h\n"
    ".inst 0xc1271392  // bfdot za.s[x8, 2], { z28.h-z29.h }, z7.h\n"
    ".inst 0xc12c1393  // bfdot za.s[x8, 3], { z28.h-z29.h }, z12.h\n"
    ".inst 0xc12a13b0  // bfdot za.s[x8, 0], { z29.h-z30.h }, z10.h\n"
    ".inst 0xc12813b1  // bfdot za.s[x8, 1], { z29.h-z30.h }, z8.h\n"
    ".inst 0xc1221394  // bfdot za.s[x8, 4], { z28.h-z29.h }, z2.h\n"
    ".inst 0xc1211395  // bfdot za.s[x8, 5], { z28.h-z29.h }, z1.h\n"
    ".inst 0xc0060818  // mova { z24.d-z25.d }, za.d[x8, #0]\n"
    ".inst 0xc006083a  // mova { z26.d-z27.d }, za.d[x8, #1]\n"
    ".inst 0xc12613b2  // bfdot za.s[x8, 2], { z29.h-z30.h }, z6.h\n"
    ".inst 0xc12013b3  // bfdot za.s[x8, 3], { z29.h-z30.h }, z0.h\n"
    ".inst 0xc12e13b4  // bfdot za.s[x8, 4], { z29.h-z30.h }, z14.h\n"
    ".inst 0xc12313b5  // bfdot za.s[x8, 5], { z29.h-z30.h }, z3.h\n"
    "add x8, x8, #0x2\n"
    ".inst 0xc1adcaf8  // fclamp { z24.s-z27.s }, z23.s, z13.s\n"
    ".inst 0xc0040884  // mova za.d[x8, #4], { z4.d-z5.d }\n"
    ".inst 0xc0040885  // mova za.d[x8, #5], { z4.d-z5.d }\n"
    "st1w { z24.s }, p1, [x9]\n"
    "add x9, x9, x27, LSL #2\n"
    "st1w { z26.s }, p1, [x28]\n"
    "add x28, x28, x26, LSL #2\n"
    "st1w { z25.s }, p1, [x25]\n"
    "add x25, x25, x23, LSL #2\n"
    "st1w { z27.s }, p1, [x24]\n"
    "add x24, x24, x22, LSL #2\n"
    "16:"  // Main loop skip tail
    "cbz x11, 18f\n"
    "17:"  // Right padding loop
    ".inst 0xc006081c  // mova { z28.d-z29.d }, za.d[x8, #0]\n"
    "subs x11, x11, #0x1\n"
    ".inst 0xc006083e  // mova { z30.d-z31.d }, za.d[x8, #1]\n"
    "add x8, x8, #0x2\n"
    ".inst 0xc0040884  // mova za.d[x8, #4], { z4.d-z5.d }\n"
    ".inst 0xc0040885  // mova za.d[x8, #5], { z4.d-z5.d }\n"
    ".inst 0xc1adcafc  // fclamp { z28.s-z31.s }, z23.s, z13.s\n"
    "st1w { z28.s }, p1, [x9]\n"
    "add x9, x9, x27, LSL #2\n"
    "st1w { z30.s }, p1, [x28]\n"
    "add x28, x28, x26, LSL #2\n"
    "st1w { z29.s }, p1, [x25]\n"
    "add x25, x25, x23, LSL #2\n"
    "st1w { z31.s }, p1, [x24]\n"
    "add x24, x24, x22, LSL #2\n"
    "bgt 17b\n"
    "18:"  // End
    "ldr x20, [%x[args], %[offsetof_Args_weights]]\n"
    "incw x15\n"
    "whilelt p1.s, x15, x16\n"
    "incb x20, ALL, MUL #9\n"
    "str x20, [%x[args], %[offsetof_Args_weights]]\n"
    "ldr x21, [%x[args], %[offsetof_Args_ld_in_vl]]\n"
    "ldr x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "add x20, x20, x21, LSL #2\n"
    "str x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "ldr x25, [%x[args], %[offsetof_Args_outptrs]]\n"
    "ldr x24, [%x[args], %[offsetof_Args_ld_out_vls]]\n"
    "ldp x23, x22, [x25, #0x0]\n"
    "ldp x21, x20, [x24, #0x0]\n"
    "add x23, x23, x21, LSL #2\n"
    "add x22, x22, x20, LSL #2\n"
    "stp x23, x22, [x25, #0x0]\n"
    "ldp x23, x22, [x25, #0x10]\n"
    "ldp x21, x20, [x24, #0x10]\n"
    "add x23, x23, x21, LSL #2\n"
    "add x22, x22, x20, LSL #2\n"
    "stp x23, x22, [x25, #0x10]\n"
    "b.any 1b\n"
    ".inst 0xd503467f  // SMSTOP\n"
    :
    : [args] "r" (&args), [ld_in_col] "r" (ld_in_col), [ld_in_row] "r" (ld_in_row), [offsetof_Args_bias] "I" (offsetof(Args, bias)), [offsetof_Args_clamp_max] "I" (offsetof(Args, clamp_max)), [offsetof_Args_clamp_min] "I" (offsetof(Args, clamp_min)), [offsetof_Args_current_channel] "I" (offsetof(Args, current_channel)), [offsetof_Args_inptr] "I" (offsetof(Args, inptr)), [offsetof_Args_input_cols] "I" (offsetof(Args, input_cols)), [offsetof_Args_ld_in_vl] "I" (offsetof(Args, ld_in_vl)), [offsetof_Args_ld_out_cols] "I" (offsetof(Args, ld_out_cols)), [offsetof_Args_ld_out_vls] "I" (offsetof(Args, ld_out_vls)), [offsetof_Args_n_channels] "I" (offsetof(Args, n_channels)), [offsetof_Args_outptrs] "I" (offsetof(Args, outptrs)), [offsetof_Args_output_cols] "I" (offsetof(Args, output_cols)), [offsetof_Args_pad_bottom] "I" (offsetof(Args, pad_bottom)), [offsetof_Args_pad_left] "I" (offsetof(Args, pad_left)), [offsetof_Args_pad_top] "I" (offsetof(Args, pad_top)), [offsetof_Args_weights] "I" (offsetof(Args, weights))
    : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
  );
}

}  // namespace depthwise
}  // namespace arm_conv

#endif  // defined(ARM_COMPUTE_ENABLE_SME2)
