// $Id$
// $Source$

//@HEADER
// ************************************************************************
//
//            NOX: An Object-Oriented Nonlinear Solver Package
//                 Copyright (2002) Sandia Corporation
//
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact Roger Pawlowski (rppawlo@sandia.gov) or
// Eric Phipps (etphipp@sandia.gov), Sandia National Laboratories.
// ************************************************************************
//  CVS Information
//  $Source$
//  $Author$
//  $Date$
//  $Revision$
// ************************************************************************
//@HEADER

#include "NOX_StatusTest_Stagnation.H" // class definition
#include "NOX_Common.H"
#include "NOX_Solver_Generic.H"
#include "NOX_Abstract_Group.H"

NOX::StatusTest::Stagnation::Stagnation(int maxSteps_, double tolerance_) :
  maxSteps(maxSteps_),
  numSteps(0),
  lastIteration(-1),
  tolerance(tolerance_),
  convRate(1.0),
  status(NOX::StatusTest::Unevaluated)
{

}

NOX::StatusTest::Stagnation::~Stagnation()
{
}

NOX::StatusTest::StatusType
NOX::StatusTest::Stagnation::
checkStatus(const Solver::Generic& problem,
        NOX::StatusTest::CheckType /* checkType */)
{
  status = Unconverged;

  // This test should ignore the checkType!  This test must be run
  // each iteration because it triggers after a set number of
  // iterations.

  // First time through we don't do anything
  int niters = problem.getNumIterations();
  if (niters == 0) {
    lastIteration = 0;
    numSteps = 0;
    return Unconverged;
  }

  // Make sure we have not already counted the last nonlinear iteration.
  // This protects against multiple calls to checkStatus() in between
  // nonlinear iterations.
  bool isCounted = false;
  if (niters == lastIteration) {
    isCounted = true;
  }
  else
    lastIteration = niters;

  // Compute the convergence rate and set counter appropriately
  if (!isCounted) {

    convRate = problem.getSolutionGroup().getNormF() /
               problem.getPreviousSolutionGroup().getNormF();

    if (convRate >= tolerance)
      numSteps ++;
    else
      numSteps = 0;

  }

  if (numSteps >= maxSteps)
    status = Failed;

  return status;
}

NOX::StatusTest::StatusType NOX::StatusTest::Stagnation::getStatus() const
{
  return status;
}

std::ostream& NOX::StatusTest::Stagnation::print(std::ostream& stream, int indent) const
{
  for (int j = 0; j < indent; j ++)
    stream << ' ';
  stream << status;
  stream << "Stagnation Count = " << numSteps << " < " << maxSteps << "\n";

  for (int j = 0; j < indent; j ++)
    stream << ' ';
  stream << "             (convergence rate = " << convRate << ")";
  stream << std::endl;
 return stream;
}


int NOX::StatusTest::Stagnation::getMaxNumSteps() const
{
  return maxSteps;
}

int NOX::StatusTest::Stagnation::getCurrentNumSteps() const
{
  return numSteps;
}

double NOX::StatusTest::Stagnation::getTolerance() const
{
  return tolerance;
}

double NOX::StatusTest::Stagnation::getConvRate() const
{
  return convRate;
}
