/*
  Optimizer
  Entry C++ file
*/

/*
 *  Copyright (C) 1997,1998  Jesper Pedersen <jews@imada.ou.dk>
 *  This code is released under GNU GPL version 2 or later
 */

#include <stdio.h>
#include <iostream.h>
#include <string.h>
#include "Global.h"
#include "Misc.h"
#include "Holder.h"
#include "Entry.h"
#include "optimizer.tab.h"

//
//	Method name : Entry
//
//	Description : Constructor to Entry
//	Input : The information for the entry
//	Output : ---
//
Entry::Entry(YYSTYPE *curEntry) {
  unsigned int length;
  this->flag = curEntry->flag;
  this->size = curEntry->size;
  if (this->flag & OPCODE) {
    this->opcode = curEntry->opcode;
  } else {
    this->opcode = EMPTY;
  }
  if (this->flag & LABEL) {
    length = strlen(curEntry->label) + 1;
    this->label = new char[length];
    if (!this->label) alloc_error();
    strcpy(this->label,curEntry->label);
  } else {
    length = 1;
    this->label = new char[length];
    if (!this->label) alloc_error();
    strcpy(this->label,"");
  }

  if (this->flag & SRC) {
    length = strlen(curEntry->src) + 1;
    this->src = new char[length];
    if (!this->src) alloc_error();
    strcpy(this->src,curEntry->src);
  } else {
    length = 1;
    this->src = new char[length];
    if (!this->src) alloc_error();
    strcpy(this->src,"");
  }

  if (this->flag & DEST) {
    length = strlen(curEntry->dest) + 1;
    this->dest = new char[length];
    if (!this->dest) alloc_error();
    strcpy(this->dest,curEntry->dest);
  } else {
    length = 1;
    this->dest = new char[length];
    if (!this->dest) alloc_error();
    strcpy(this->dest,"");
  }

  this->next = (PEntry) NULL;

  this->setopcodestr();
}

//
//	Method name : ~Entry
//
//	Description : Destructor to Entry
//	Input : ---
//	Output : ---
//
Entry::~Entry(void) {
  delete[] this->label;
  delete[] this->opcodestr;
  delete[] this->src;
  delete[] this->dest;
}

//
//	Method name : getFlag
//
//	Description : Returns the flag for the entry
//	Input : ---
//	Output : Flag value
//
unsigned int Entry::getFlag(void) {
  return this->flag;
}

//
//	Method name : setFlag
//
//	Description : Sets the flag for the entry
//	Input : Flag value
//	Output : ---
//
void Entry::setFlag(unsigned int f) {
  this->flag = f;
}

//
//	Method name : getopcode
//
//	Description : Returns the opcode
//	Input : ---
//	Output : opcode
//
unsigned int Entry::getopcode(void) {
  return this->opcode;
}

//
//	Method name : getlabel
//
//	Description : Returns the label string
//	Input : ---
//	Output : Label string
//
char *Entry::getlabel(void){
  return this->label;
}

//
//	Method name : getsize
//
//	Description : Returns the "size" of the instruction
//	Input : ---
//	Output : The Size
//
unsigned int Entry::getsize(void) {
  return this->size;
}

//
//	Method name : getopcodestr
//
//	Description : Returns the opcodestr string
//	Input : ---
//	Output : Opcodestr string
//
char *Entry::getopcodestr(void){
  return this->opcodestr;
}

//
//	Method name : setsrc
//
//	Description : Sets the src string
//	Input : ---
//	Output : ---
//
void Entry::setsrc(char *s) {
  unsigned int length;
  delete[] this->src;
  length = strlen(s) + 1;
  this->src = new char[length];
  if (!this->src) alloc_error();
  strcpy(this->src,s);
}

//
//	Method name : getsrc
//
//	Description : Returns the src string
//	Input : ---
//	Output : Src string
//
char *Entry::getsrc(void){
  return this->src;
}

//
//	Method name : setdest
//
//	Description : Sets the dest string
//	Input : ---
//	Output : ---
//
void Entry::setdest(char *s) {
  unsigned int length;
  delete[] this->dest;
  length = strlen(s) + 1;
  this->dest = new char[length];
  if (!this->dest) alloc_error();
  strcpy(this->dest,s);
}

//
//	Method name : getdest
//
//	Description : Returns the dest string
//	Input : ---
//	Output : Dest string
//
char *Entry::getdest(void){
  return this->dest;
}


//
//	Method name : setopcodestr
//
//	Description : Generates the opcode string
//	Input : ---
//	Output : ---
//
void Entry::setopcodestr(void){
  char *tmpstr = new char[MaxSize];
  unsigned int length;
  
  if (!tmpstr) alloc_error();

  switch (this->opcode) {
      case EMPTY        :  {strcpy(tmpstr,""); break;}
      case _AAA         :  {strcpy(tmpstr,"aaa"); break;}
      case _AAD         :  {strcpy(tmpstr,"aad"); break;}
      case _AAM         :  {strcpy(tmpstr,"aam"); break;}
      case _AAS         :  {strcpy(tmpstr,"aas"); break;}
      case _ADC         :  {strcpy(tmpstr,"adc"); break;}
      case _ADD         :  {strcpy(tmpstr,"add"); break;}
      case _AND         :  {strcpy(tmpstr,"and"); break;}
      case _ARPL        :  {strcpy(tmpstr,"arpl"); break;}
      case _BOUND       :  {strcpy(tmpstr,"bound"); break;}
      case _BSF         :  {strcpy(tmpstr,"bsf"); break;}
      case _BSR         :  {strcpy(tmpstr,"bsr"); break;}
      case _BSWAP       :  {strcpy(tmpstr,"bswap"); break;}
      case _BT          :  {strcpy(tmpstr,"bt"); break;}
      case _BTC         :  {strcpy(tmpstr,"btc"); break;}
      case _BTR         :  {strcpy(tmpstr,"btr"); break;}
      case _BTS         :  {strcpy(tmpstr,"bts"); break;}
      case _CALL        :  {strcpy(tmpstr,"call"); break;}
      case _CBW         :  {strcpy(tmpstr,"cbw"); break;}
      case _CDQ         :  {strcpy(tmpstr,"cdq"); break;}
      case _CLC         :  {strcpy(tmpstr,"clc"); break;}
      case _CLD         :  {strcpy(tmpstr,"cld"); break;}
      case _CLI         :  {strcpy(tmpstr,"cli"); break;}
      case _CLTS        :  {strcpy(tmpstr,"clts"); break;}
      case _CMC         :  {strcpy(tmpstr,"cmc"); break;}
      case _CMP         :  {strcpy(tmpstr,"cmp"); break;}
      case _CMPSB       :  {strcpy(tmpstr,"cmpsb"); break;}
      case _CMPSD       :  {strcpy(tmpstr,"cmpsd"); break;}
      case _CMPSW       :  {strcpy(tmpstr,"cmpsw"); break;}
      case _CMPXCHG     :  {strcpy(tmpstr,"cmpxchg"); break;}
      case _CMPXCHG8B   :  {strcpy(tmpstr,"cmpxchg8b"); break;}
      case _CPUID       :  {strcpy(tmpstr,"cpuid"); break;}
      case _CWD         :  {strcpy(tmpstr,"cwd"); break;}
      case _CWDE        :  {strcpy(tmpstr,"cwde"); break;}
      case _DAA         :  {strcpy(tmpstr,"daa"); break;}
      case _DAS         :  {strcpy(tmpstr,"das"); break;}
      case _DEC         :  {strcpy(tmpstr,"dec"); break;}
      case _DIV         :  {strcpy(tmpstr,"div"); break;}
      case _EMMS        :  {strcpy(tmpstr,"emms"); break;}
      case _ENTER       :  {strcpy(tmpstr,"enter"); break;}
      case _ESC         :  {strcpy(tmpstr,"esc"); break;}
      case _F2XM1       :  {strcpy(tmpstr,"f2xm1"); break;}
      case _FABS        :  {strcpy(tmpstr,"fabs"); break;}
      case _FADD        :  {strcpy(tmpstr,"fadd"); break;}
      case _FADDP       :  {strcpy(tmpstr,"faddp"); break;}
      case _FBLD        :  {strcpy(tmpstr,"fbld"); break;}
      case _FBSTP       :  {strcpy(tmpstr,"fbstp"); break;}
      case _FCHS        :  {strcpy(tmpstr,"fchs"); break;}
      case _FCLEX       :  {strcpy(tmpstr,"fclex"); break;}
      case _FCMOVB      :  {strcpy(tmpstr,"fcmovb"); break;}
      case _FCMOVBE     :  {strcpy(tmpstr,"fcmovbe"); break;}
      case _FCMOVE      :  {strcpy(tmpstr,"fcmove"); break;}
      case _FCMOVNB     :  {strcpy(tmpstr,"fcmovnb"); break;}
      case _FCMOVNBE    :  {strcpy(tmpstr,"fcmovnbe"); break;}
      case _FCMOVNE     :  {strcpy(tmpstr,"fcmovne"); break;}
      case _FCMOVNU     :  {strcpy(tmpstr,"fcmovnu"); break;}
      case _FCMOVU      :  {strcpy(tmpstr,"fcmovu"); break;}
      case _FCOM        :  {strcpy(tmpstr,"fcom"); break;}
      case _FCOMI       :  {strcpy(tmpstr,"fcomi"); break;}
      case _FCOMIP      :  {strcpy(tmpstr,"fcomip"); break;}
      case _FCOMP       :  {strcpy(tmpstr,"fcomp"); break;}
      case _FCOMPP      :  {strcpy(tmpstr,"fcompp"); break;}
      case _FCOS        :  {strcpy(tmpstr,"fcos"); break;} 
      case _FDECSTP     :  {strcpy(tmpstr,"fdecstp"); break;}  
      case _FDIV        :  {strcpy(tmpstr,"fdiv"); break;} 
      case _FDIVP       :  {strcpy(tmpstr,"fdivp"); break;}
      case _FDIVR       :  {strcpy(tmpstr,"fdivr"); break;}
      case _FDIVRP      :  {strcpy(tmpstr,"fdivrp"); break;}
      case _FFREE       :  {strcpy(tmpstr,"ffree"); break;}
      case _FFREEP      :  {strcpy(tmpstr,"ffreep"); break;}   // Not supported
      case _FIADD       :  {strcpy(tmpstr,"fiadd"); break;}
      case _FICOM       :  {strcpy(tmpstr,"ficom"); break;}
      case _FICOMP      :  {strcpy(tmpstr,"ficomp"); break;} 
      case _FIDIV       :  {strcpy(tmpstr,"fidiv"); break;} 
      case _FIDIVR      :  {strcpy(tmpstr,"fidivr"); break;}
      case _FILD        :  {strcpy(tmpstr,"fild"); break;}  
      case _FIMUL       :  {strcpy(tmpstr,"fimul"); break;}
      case _FINCSTP     :  {strcpy(tmpstr,"fincstp"); break;} 
      case _FINIT       :  {strcpy(tmpstr,"finit"); break;}
      case _FIST        :  {strcpy(tmpstr,"fist"); break;} 
      case _FISTP       :  {strcpy(tmpstr,"fistp"); break;}
      case _FISUB       :  {strcpy(tmpstr,"fisub"); break;}
      case _FISUBR      :  {strcpy(tmpstr,"fisubr"); break;}  
      case _FLD         :  {strcpy(tmpstr,"fld"); break;} 
      case _FLD1        :  {strcpy(tmpstr,"fld1"); break;}
      case _FLDCW       :  {strcpy(tmpstr,"fldcw"); break;}
      case _FLDENV      :  {strcpy(tmpstr,"fldenv"); break;} 
      case _FLDL2E      :  {strcpy(tmpstr,"fldl2e"); break;} 
      case _FLDL2T      :  {strcpy(tmpstr,"fldl2t"); break;}
      case _FLDLG2      :  {strcpy(tmpstr,"fldlg2"); break;}
      case _FLDLN2      :  {strcpy(tmpstr,"fldln2"); break;}
      case _FLDPI       :  {strcpy(tmpstr,"fldpi"); break;} 
      case _FLDZ        :  {strcpy(tmpstr,"fldz"); break;}
      case _FMUL        :  {strcpy(tmpstr,"fmul"); break;} 
      case _FMULP       :  {strcpy(tmpstr,"fmulp"); break;} 
      case _FNCLEX      :  {strcpy(tmpstr,"fnclex"); break;} 
      case _FNINIT      :  {strcpy(tmpstr,"fninit"); break;} 
      case _FNOP        :  {strcpy(tmpstr,"fnop"); break;} 
      case _FNSAVE      :  {strcpy(tmpstr,"fnsave"); break;}
      case _FNSTCW      :  {strcpy(tmpstr,"fnstcw"); break;}
      case _FNSTENV     :  {strcpy(tmpstr,"fnstenv"); break;} 
      case _FNSTSW      :  {strcpy(tmpstr,"fnstsw"); break;} 
      case _FPATAN      :  {strcpy(tmpstr,"fpatan"); break;}
      case _FPREM       :  {strcpy(tmpstr,"fprem"); break;} 
      case _FPREM1      :  {strcpy(tmpstr,"fprem1"); break;} 
      case _FPTAN       :  {strcpy(tmpstr,"fptan"); break;}
      case _FRNDINT     :  {strcpy(tmpstr,"frndint"); break;}
      case _FRSTOR      :  {strcpy(tmpstr,"frstor"); break;} 
      case _FSAVE       :  {strcpy(tmpstr,"fsave"); break;} 
      case _FSCALE      :  {strcpy(tmpstr,"fscale"); break;} 
      case _FSIN        :  {strcpy(tmpstr,"fsin"); break;}
      case _FSINCOS     :  {strcpy(tmpstr,"fsincos"); break;} 
      case _FSQRT       :  {strcpy(tmpstr,"fsqrt"); break;} 
      case _FST         :  {strcpy(tmpstr,"fst"); break;}
      case _FSTCW       :  {strcpy(tmpstr,"fstcw"); break;} 
      case _FSTENV      :  {strcpy(tmpstr,"fstenv"); break;}
      case _FSTP        :  {strcpy(tmpstr,"fstp"); break;}
      case _FSTSW       :  {strcpy(tmpstr,"fstsw"); break;}
      case _FSUB        :  {strcpy(tmpstr,"fsub"); break;} 
      case _FSUBP       :  {strcpy(tmpstr,"fsubp"); break;}
      case _FSUBR       :  {strcpy(tmpstr,"fsubr"); break;} 
      case _FSUBRP      :  {strcpy(tmpstr,"fsubrp"); break;} 
      case _FTST        :  {strcpy(tmpstr,"ftst"); break;} 
      case _FUCOM       :  {strcpy(tmpstr,"fucom"); break;} 
      case _FUCOMI      :  {strcpy(tmpstr,"fucomi"); break;} 
      case _FUCOMIP     :  {strcpy(tmpstr,"fucomip"); break;} 
      case _FUCOMP      :  {strcpy(tmpstr,"fucomp"); break;} 
      case _FUCOMPP     :  {strcpy(tmpstr,"fucompp"); break;}
      case _FWAIT       :  {strcpy(tmpstr,"fwait"); break;} 
      case _FXAM        :  {strcpy(tmpstr,"fxam"); break;} 
      case _FXCH        :  {strcpy(tmpstr,"fxch"); break;}
      case _FXTRACT     :  {strcpy(tmpstr,"fxtract"); break;} 
      case _FYL2X       :  {strcpy(tmpstr,"fyl2x"); break;}
      case _FYL2XP1     :  {strcpy(tmpstr,"fyl2xp1"); break;}  
      case _HLT         :  {strcpy(tmpstr,"hlt"); break;}
      case _IDIV        :  {strcpy(tmpstr,"idiv"); break;}
      case _IMUL        :  {strcpy(tmpstr,"imul"); break;}
      case _IN          :  {strcpy(tmpstr,"in"); break;}
      case _INC         :  {strcpy(tmpstr,"inc"); break;}
      case _INSB        :  {strcpy(tmpstr,"insb"); break;}
      case _INSD        :  {strcpy(tmpstr,"insd"); break;}
      case _INSW        :  {strcpy(tmpstr,"insw"); break;}
      case _INT         :  {strcpy(tmpstr,"int"); break;}
      case _INTO        :  {strcpy(tmpstr,"into"); break;}
      case _INVD        :  {strcpy(tmpstr,"intd"); break;}
      case _INVLPG      :  {strcpy(tmpstr,"intlpg"); break;}
      case _IRET        :  {strcpy(tmpstr,"iret"); break;}
      case _IRETD       :  {strcpy(tmpstr,"iretd"); break;}
      case _JCXZ        :  {strcpy(tmpstr,"jcxz"); break;}
      case _JECXZ       :  {strcpy(tmpstr,"jecxz"); break;}
      case _JMP         :  {strcpy(tmpstr,"jmp"); break;}
      case _LAHF        :  {strcpy(tmpstr,"lahf"); break;}
      case _LAR         :  {strcpy(tmpstr,"lar"); break;}
      case _LDS         :  {strcpy(tmpstr,"lds"); break;}
      case _LEA         :  {strcpy(tmpstr,"lea"); break;}
      case _LEAVE       :  {strcpy(tmpstr,"leave"); break;}
      case _LES         :  {strcpy(tmpstr,"les"); break;}
      case _LFS         :  {strcpy(tmpstr,"lfs"); break;}
      case _LGDT        :  {strcpy(tmpstr,"lgdt"); break;}
      case _LGS         :  {strcpy(tmpstr,"lgs"); break;}
      case _LIDT        :  {strcpy(tmpstr,"lidt"); break;}
      case _LLDT        :  {strcpy(tmpstr,"lldt"); break;}
      case _LMSW        :  {strcpy(tmpstr,"lmsw"); break;}
      case _LOCK        :  {strcpy(tmpstr,"lock"); break;}
      case _LODSB       :  {strcpy(tmpstr,"lodsb"); break;}
      case _LODSD       :  {strcpy(tmpstr,"lodsd"); break;}
      case _LODSW       :  {strcpy(tmpstr,"lodsw"); break;}
      case _LOOP        :  {strcpy(tmpstr,"loop"); break;}
      case _LOOPE       :  {strcpy(tmpstr,"loope"); break;}
      case _LOOPNE      :  {strcpy(tmpstr,"loopne"); break;}
      case _LOOPNZ      :  {strcpy(tmpstr,"loopnz"); break;}
      case _LOOPZ       :  {strcpy(tmpstr,"loopz"); break;}
      case _LSL         :  {strcpy(tmpstr,"lsl"); break;}
      case _LSS         :  {strcpy(tmpstr,"lss"); break;}
      case _LTR         :  {strcpy(tmpstr,"ltr"); break;}
      case _MOV         :  {strcpy(tmpstr,"mov"); break;}
      case _MOVD        :  {strcpy(tmpstr,"movd"); break;}
      case _MOVQ        :  {strcpy(tmpstr,"movq"); break;}
      case _MOVS        :  {if (this->size == BYTE) {
                              strcpy(tmpstr,"movsb"); 
                            } else if (this->size == WORD) {
                              strcpy(tmpstr,"movsw");
                            } else {
                              strcpy(tmpstr,"movsd");
                            }
                            break;}
      case _MOVSX_B     :  {strcpy(tmpstr,"movsx"); break;}
      case _MOVZX_B     :  {strcpy(tmpstr,"movzx"); break;}
      case _MOVSX_W     :  {strcpy(tmpstr,"movsx"); break;}
      case _MOVZX_W     :  {strcpy(tmpstr,"movzx"); break;}
      case _MUL         :  {strcpy(tmpstr,"mul"); break;}
      case _NEG         :  {strcpy(tmpstr,"neg"); break;}
      case _NOP         :  {strcpy(tmpstr,"nop"); break;}
      case _NOT         :  {strcpy(tmpstr,"not"); break;}
      case _OR          :  {strcpy(tmpstr,"or"); break;}
      case _OUT         :  {strcpy(tmpstr,"out"); break;}
      case _OUTSB       :  {strcpy(tmpstr,"outsb"); break;}
      case _OUTSD       :  {strcpy(tmpstr,"outsd"); break;}
      case _OUTSW       :  {strcpy(tmpstr,"outsw"); break;}
      case _PACKSSDW    :  {strcpy(tmpstr,"packssdw"); break;}
      case _PACKSSWB    :  {strcpy(tmpstr,"packsswb"); break;}
      case _PACKUSWB    :  {strcpy(tmpstr,"packuswb"); break;}
      case _PADDB       :  {strcpy(tmpstr,"paddb"); break;}
      case _PADDD       :  {strcpy(tmpstr,"paddd"); break;}
      case _PADDSB      :  {strcpy(tmpstr,"paddsb"); break;}
      case _PADDSW      :  {strcpy(tmpstr,"paddsw"); break;}
      case _PADDUSB     :  {strcpy(tmpstr,"paddusb"); break;}
      case _PADDUSW     :  {strcpy(tmpstr,"paddusw"); break;}
      case _PADDW       :  {strcpy(tmpstr,"paddw"); break;}
      case _PAND        :  {strcpy(tmpstr,"pand"); break;}
      case _PANDN       :  {strcpy(tmpstr,"pandn"); break;}
      case _PCMPEQB     :  {strcpy(tmpstr,"pcmpeqb"); break;}
      case _PCMPEQD     :  {strcpy(tmpstr,"pcmpeqd"); break;}
      case _PCMPEQW     :  {strcpy(tmpstr,"pcmpeqw"); break;}
      case _PCMPGTB     :  {strcpy(tmpstr,"pcmpgtb"); break;}
      case _PCMPGTD     :  {strcpy(tmpstr,"pcmpgtd"); break;}
      case _PCMPGTW     :  {strcpy(tmpstr,"pcmpgtw"); break;}
      case _PMADDWD     :  {strcpy(tmpstr,"pmaddwd"); break;}
      case _PMULHW      :  {strcpy(tmpstr,"pmulhw"); break;}
      case _PMULLW      :  {strcpy(tmpstr,"pmullw"); break;}
      case _POP         :  {strcpy(tmpstr,"pop"); break;}
      case _POPA        :  {strcpy(tmpstr,"popa"); break;}
      case _POPAD       :  {strcpy(tmpstr,"popad"); break;}
      case _POPF        :  {strcpy(tmpstr,"popf"); break;}
      case _POPFD       :  {strcpy(tmpstr,"popfd"); break;}
      case _POR         :  {strcpy(tmpstr,"por"); break;}
      case _PSHIMW      :  {strcpy(tmpstr,"pshimw"); break;}
      case _PSHIMD      :  {strcpy(tmpstr,"pshimd"); break;}
      case _PSHIMQ      :  {strcpy(tmpstr,"pshimq"); break;}
      case _PSLLD       :  {strcpy(tmpstr,"pslld"); break;}
      case _PSLLQ       :  {strcpy(tmpstr,"psllq"); break;}
      case _PSLLW       :  {strcpy(tmpstr,"psllw"); break;}
      case _PSRAD       :  {strcpy(tmpstr,"psrad"); break;}
      case _PSRAW       :  {strcpy(tmpstr,"psraw"); break;}
      case _PSRLD       :  {strcpy(tmpstr,"psrld"); break;}
      case _PSRLQ       :  {strcpy(tmpstr,"psrlq"); break;}
      case _PSRLW       :  {strcpy(tmpstr,"psrlw"); break;}
      case _PSUBB       :  {strcpy(tmpstr,"psubb"); break;}
      case _PSUBD       :  {strcpy(tmpstr,"psubd"); break;}
      case _PSUBSB      :  {strcpy(tmpstr,"psubsb"); break;}
      case _PSUBSW      :  {strcpy(tmpstr,"psubsw"); break;}
      case _PSUBUSB     :  {strcpy(tmpstr,"psubusb"); break;}
      case _PSUBUSW     :  {strcpy(tmpstr,"psubisw"); break;}
      case _PSUBW       :  {strcpy(tmpstr,"psubw"); break;}
      case _PUNPCKHBW   :  {strcpy(tmpstr,"punpckhbw"); break;}
      case _PUNPCKHDQ   :  {strcpy(tmpstr,"punpckhdq"); break;}
      case _PUNPCKHWD   :  {strcpy(tmpstr,"punpckhwd"); break;}
      case _PUNPCKLBW   :  {strcpy(tmpstr,"punpcklbw"); break;}
      case _PUNPCKLDQ   :  {strcpy(tmpstr,"punpckldq"); break;}
      case _PUNPCKLWD   :  {strcpy(tmpstr,"punpcklwd"); break;}
      case _PUSH        :  {strcpy(tmpstr,"push"); break;}
      case _PUSHA       :  {strcpy(tmpstr,"pusha"); break;}
      case _PUSHAD      :  {strcpy(tmpstr,"pushad"); break;}
      case _PUSHF       :  {strcpy(tmpstr,"pushf"); break;}
      case _PUSHFD      :  {strcpy(tmpstr,"pushfd"); break;}
      case _PXOR        :  {strcpy(tmpstr,"pxor"); break;}
      case _RCL         :  {strcpy(tmpstr,"rcl"); break;}
      case _RCR         :  {strcpy(tmpstr,"rcr"); break;}
      case _RDMSR       :  {strcpy(tmpstr,"rdmsr"); break;}
      case _RDTSC       :  {strcpy(tmpstr,"rdtsc"); break;}
      case _REP         :  {strcpy(tmpstr,"rep"); break;}
      case _REPE        :  {strcpy(tmpstr,"repe"); break;}
      case _REPZ        :  {strcpy(tmpstr,"repz"); break;}
      case _REPNE       :  {strcpy(tmpstr,"repne"); break;}
      case _REPNZ       :  {strcpy(tmpstr,"repnz"); break;}
      case _RET         :  {strcpy(tmpstr,"ret"); break;}
      case _RETF        :  {strcpy(tmpstr,"retf"); break;}
      case _ROL         :  {strcpy(tmpstr,"rol"); break;}
      case _ROR         :  {strcpy(tmpstr,"ror"); break;}
      case _RSM         :  {strcpy(tmpstr,"rsm"); break;}
      case _SAHF        :  {strcpy(tmpstr,"sahf"); break;}
      case _SAL         :  {strcpy(tmpstr,"sal"); break;}
      case _SALC        :  {strcpy(tmpstr,"salc"); break;}
      case _SAR         :  {strcpy(tmpstr,"sar"); break;}
      case _SBB         :  {strcpy(tmpstr,"sbb"); break;}
      case _SCASB       :  {strcpy(tmpstr,"scasb"); break;}
      case _SCASD       :  {strcpy(tmpstr,"scasd"); break;}
      case _SCASW       :  {strcpy(tmpstr,"scasw"); break;}
      case _SGDT        :  {strcpy(tmpstr,"sgdt"); break;}
      case _SHL         :  {strcpy(tmpstr,"shl"); break;}
      case _SHLD        :  {strcpy(tmpstr,"shld"); break;}
      case _SHR         :  {strcpy(tmpstr,"shr"); break;}
      case _SHRD        :  {strcpy(tmpstr,"shrd"); break;}
      case _SIDT        :  {strcpy(tmpstr,"sidt"); break;}
      case _SLDT        :  {strcpy(tmpstr,"sldt"); break;}
      case _SMSW        :  {strcpy(tmpstr,"smsw"); break;}
      case _STC         :  {strcpy(tmpstr,"stc"); break;}
      case _STD         :  {strcpy(tmpstr,"std"); break;}
      case _STI         :  {strcpy(tmpstr,"sti"); break;}
      case _STOSB       :  {strcpy(tmpstr,"stosb"); break;}
      case _STOSD       :  {strcpy(tmpstr,"stosd"); break;}
      case _STOSW       :  {strcpy(tmpstr,"stosw"); break;}
      case _STR         :  {strcpy(tmpstr,"str"); break;}
      case _SUB         :  {strcpy(tmpstr,"sub"); break;}
      case _TEST        :  {strcpy(tmpstr,"test"); break;}
      case _VERR        :  {strcpy(tmpstr,"verr"); break;}
      case _VERW        :  {strcpy(tmpstr,"verw"); break;}
      case _WAIT        :  {strcpy(tmpstr,"wait"); break;}
      case _WBINVD      :  {strcpy(tmpstr,"wbinvd"); break;}
      case _WRMSR       :  {strcpy(tmpstr,"wrmsr"); break;}
      case _XADD        :  {strcpy(tmpstr,"xadd"); break;}
      case _XCHG        :  {strcpy(tmpstr,"xchg"); break;}
      case _XLAT        :  {strcpy(tmpstr,"xlat"); break;}
      case _XOR         :  {strcpy(tmpstr,"xor"); break;}
    
      case _JO          :  {strcpy(tmpstr,"jo"); break;}
      case _JNO         :  {strcpy(tmpstr,"jno"); break;}
      case _JC          :  {strcpy(tmpstr,"jc"); break;}
      case _JB          :  {strcpy(tmpstr,"jb"); break;}
      case _JNAE        :  {strcpy(tmpstr,"jnae"); break;}
      case _JNC         :  {strcpy(tmpstr,"jnc"); break;}
      case _JAE         :  {strcpy(tmpstr,"jae"); break;}
      case _JNB         :  {strcpy(tmpstr,"jnb"); break;}
      case _JE          :  {strcpy(tmpstr,"je"); break;}
      case _JZ          :  {strcpy(tmpstr,"jz"); break;}
      case _JNE         :  {strcpy(tmpstr,"jne"); break;}
      case _JNZ         :  {strcpy(tmpstr,"jnz"); break;}
      case _JBE         :  {strcpy(tmpstr,"jbe"); break;}
      case _JNA         :  {strcpy(tmpstr,"jna"); break;}
      case _JA          :  {strcpy(tmpstr,"ja"); break;}
      case _JNBE        :  {strcpy(tmpstr,"jnbe"); break;}
      case _JS          :  {strcpy(tmpstr,"js"); break;}
      case _JNS         :  {strcpy(tmpstr,"jns"); break;}
      case _JP          :  {strcpy(tmpstr,"jp"); break;}
      case _JPE         :  {strcpy(tmpstr,"jpe"); break;}
      case _JNP         :  {strcpy(tmpstr,"jnp"); break;}
      case _JPO         :  {strcpy(tmpstr,"jpo"); break;}
      case _JL          :  {strcpy(tmpstr,"jl"); break;}
      case _JNGE        :  {strcpy(tmpstr,"jnge"); break;}
      case _JGE         :  {strcpy(tmpstr,"jge"); break;}
      case _JNL         :  {strcpy(tmpstr,"jnl"); break;}
      case _JLE         :  {strcpy(tmpstr,"jle"); break;}
      case _JNG         :  {strcpy(tmpstr,"jng"); break;}
      case _JG          :  {strcpy(tmpstr,"jg"); break;}
      case _JNLE        :  {strcpy(tmpstr,"jnle"); break;}
    
      case _SETO        :  {strcpy(tmpstr,"seto"); break;}
      case _SETNO       :  {strcpy(tmpstr,"setno"); break;}
      case _SETB        :  {strcpy(tmpstr,"setb"); break;}
      case _SETNAE      :  {strcpy(tmpstr,"setnae"); break;}
      case _SETAE       :  {strcpy(tmpstr,"setae"); break;}
      case _SETNB       :  {strcpy(tmpstr,"setnb"); break;}
      case _SETE        :  {strcpy(tmpstr,"sete"); break;}
      case _SETZ        :  {strcpy(tmpstr,"setz"); break;}
      case _SETNE       :  {strcpy(tmpstr,"setne"); break;}
      case _SETNZ       :  {strcpy(tmpstr,"setnz"); break;}
      case _SETBE       :  {strcpy(tmpstr,"setbe"); break;}
      case _SETNA       :  {strcpy(tmpstr,"setna"); break;}
      case _SETA        :  {strcpy(tmpstr,"seta"); break;}
      case _SETNBE      :  {strcpy(tmpstr,"setnbe"); break;}
      case _SETS        :  {strcpy(tmpstr,"sets"); break;}
      case _SETNS       :  {strcpy(tmpstr,"setns"); break;}
      case _SETP        :  {strcpy(tmpstr,"setp"); break;}
      case _SETPE       :  {strcpy(tmpstr,"setpe"); break;}
      case _SETNP       :  {strcpy(tmpstr,"setnp"); break;}
      case _SETPO       :  {strcpy(tmpstr,"setpo"); break;}
      case _SETL        :  {strcpy(tmpstr,"setl"); break;}
      case _SETNGE      :  {strcpy(tmpstr,"setnge"); break;}
      case _SETGE       :  {strcpy(tmpstr,"setge"); break;}
      case _SETNL       :  {strcpy(tmpstr,"setnl"); break;}
      case _SETLE       :  {strcpy(tmpstr,"setle"); break;}
      case _SETNG       :  {strcpy(tmpstr,"setng"); break;}
      case _SETG        :  {strcpy(tmpstr,"setg"); break;}
      case _SETNLE      :  {strcpy(tmpstr,"setnle"); break;}
    
      case _CMOVO       :  {strcpy(tmpstr,"cmovo"); break;}
      case _CMOVNO      :  {strcpy(tmpstr,"cmovno"); break;}
      case _CMOVC       :  {strcpy(tmpstr,"cmovc"); break;}
      case _CMOVB       :  {strcpy(tmpstr,"cmovb"); break;}
      case _CMOVNAE     :  {strcpy(tmpstr,"cmovnae"); break;}
      case _CMOVNC      :  {strcpy(tmpstr,"cmovnc"); break;}
      case _CMOVAE      :  {strcpy(tmpstr,"cmovae"); break;}
      case _CMOVNB      :  {strcpy(tmpstr,"cmovnb"); break;}
      case _CMOVE       :  {strcpy(tmpstr,"cmove"); break;}
      case _CMOVZ       :  {strcpy(tmpstr,"cmovz"); break;}
      case _CMOVNE      :  {strcpy(tmpstr,"cmovne"); break;}
      case _CMOVNZ      :  {strcpy(tmpstr,"cmovnz"); break;}
      case _CMOVBE      :  {strcpy(tmpstr,"cmovbe"); break;}
      case _CMOVNA      :  {strcpy(tmpstr,"cmovna"); break;}
      case _CMOVA       :  {strcpy(tmpstr,"cmova"); break;}
      case _CMOVNBE     :  {strcpy(tmpstr,"cmovnbe"); break;}
      case _CMOVS       :  {strcpy(tmpstr,"cmovs"); break;}
      case _CMOVNS      :  {strcpy(tmpstr,"cmovns"); break;}
      case _CMOVP       :  {strcpy(tmpstr,"cmovp"); break;}
      case _CMOVPE      :  {strcpy(tmpstr,"cmovpe"); break;}
      case _CMOVNP      :  {strcpy(tmpstr,"cmovnp"); break;}
      case _CMOVPO      :  {strcpy(tmpstr,"cmovpo"); break;}
      case _CMOVL       :  {strcpy(tmpstr,"cmovl"); break;}
      case _CMOVNGE     :  {strcpy(tmpstr,"cmovnge"); break;}
      case _CMOVGE      :  {strcpy(tmpstr,"cmovge"); break;}
      case _CMOVNL      :  {strcpy(tmpstr,"cmovnl"); break;}
      case _CMOVLE      :  {strcpy(tmpstr,"cmovle"); break;}
      case _CMOVNG      :  {strcpy(tmpstr,"cmovng"); break;}
      case _CMOVG       :  {strcpy(tmpstr,"cmovg"); break;}
      case _CMOVNLE     :  {strcpy(tmpstr,"cmovnle"); break;}
    
      case _DB          :  {strcpy(tmpstr,""); break;}
      case _DD          :  {strcpy(tmpstr,""); break;}
      case _DQ          :  {strcpy(tmpstr,""); break;}
      case _DT          :  {strcpy(tmpstr,""); break;}
      case _DW          :  {strcpy(tmpstr,""); break;}
      case _EQU         :  {strcpy(tmpstr,""); break;}
      case _RESB        :  {strcpy(tmpstr,""); break;}
      case _RESD        :  {strcpy(tmpstr,""); break;}
      case _RESQ        :  {strcpy(tmpstr,""); break;}
      case _REST        :  {strcpy(tmpstr,""); break;}
      case _RESW        :  {strcpy(tmpstr,""); break;}
    
      case _IBTS        :  {strcpy(tmpstr,""); break;}
      case _ICEBP       :  {strcpy(tmpstr,""); break;}
      case _INCBIN      :  {strcpy(tmpstr,""); break;}
      case _INT1        :  {strcpy(tmpstr,""); break;}
      case _INT01       :  {strcpy(tmpstr,""); break;}
      case _INT3        :  {strcpy(tmpstr,""); break;}
      case _IRETW       :  {strcpy(tmpstr,""); break;}
      case _LOADALL     :  {strcpy(tmpstr,""); break;}
      case _LOADALL286  :  {strcpy(tmpstr,""); break;}
      case _POPAW       :  {strcpy(tmpstr,""); break;}
      case _POPFW       :  {strcpy(tmpstr,""); break;}
      case _PUSHAW      :  {strcpy(tmpstr,""); break;}
      case _PUSHFW      :  {strcpy(tmpstr,""); break;}
      case _RDPMC       :  {strcpy(tmpstr,""); break;}
      case _RETN        :  {strcpy(tmpstr,""); break;}
      case _SMI         :  {strcpy(tmpstr,""); break;}
      case _UMOV        :  {strcpy(tmpstr,""); break;}
      case _XBTS        :  {strcpy(tmpstr,""); break;}
  
      default           :  {cout << "Bad object initialization." << endl; 
                            this->print();
                            exit(1);}
  }

  length = strlen(tmpstr) + 1;
  this->opcodestr = new char[length];
  if (!this->opcodestr) alloc_error();
  strcpy(this->opcodestr,tmpstr);

  delete[] tmpstr;
}


//
//	Method name : getindirect
//
//	Description : Returns a string that contains the indirect mem. ref.
//	Input : The parent, SRC or DEST
//	Output : String containing the indirect mem. ref.
//
char *Entry::getindirect(PHolder parent, unsigned int where){
  char *section = new char[MaxSize];
  char *disp = new char[MaxSize];
  char *base = new char[MaxSize];
  char *index = new char[MaxSize];
  char *scale = new char[MaxSize];
  char *tmpstr = new char[MaxSize];
  char *rettmpstr = new char[MaxSize];
  char *tmpdisp = new char[MaxSize];
  char *retstr;
  int i,j,com,numindex;
  PIdEntry tmp1,tmp2,tmp3;
  PGloblEntry tmp4;
  bool paran = false;
  bool switchstr = false;

  if (!section) alloc_error();
  if (!disp) alloc_error();
  if (!base) alloc_error();
  if (!index) alloc_error();
  if (!scale) alloc_error();
  if (!tmpstr) alloc_error();
  if (!rettmpstr) alloc_error();
  if (!tmpdisp) alloc_error();

  strcpy(section,"");
  strcpy(disp,"");
  strcpy(base,"");
  strcpy(index,"");
  strcpy(scale,"");
  strcpy(rettmpstr,"");
  strcpy(tmpdisp,"");

  i = 0; j = 0; com = 0;

  if (where == SRC) {
    while (this->src[i] != '\0') {
      if (this->src[i] == ',') {
	com++;
      }
      i++;
    }
    i = 0;
    if (strstr(this->src,":") != NULL) {
      while (this->src[i] != ':') {
	section[j] = this->src[i];
	i++; j++;
      }
      section[j] = '\0';
      strcat(section,":");
      j = 0;
      i++;
    }
    if (strstr(this->src,"(") != NULL) {
      paran = true;
    } 
    if (paran == true) {
      while (this->src[i] != '(') {
	disp[j] = this->src[i];
	i++; j++;
      }
      disp[j] = '\0';
      j = 0;
      i++;
      if (com == 0) {
	while (this->src[i] != ')') {
	  base[j] = this->src[i];
	  i++; j++;
	}
	base[j] = '\0';
      } else if (com == 1) {
	while (this->src[i] != ',') {
	  base[j] = this->src[i];
	  i++; j++;
	}
	base[j] = '\0';
	j = 0;
	i++;
	if (strcmp(base,"")) {
	  while (this->src[i] != ')') {
	    index[j] = this->src[i];
	    i++; j++;
	  }
	  index[j] = '\0';
	}
      } else if (com == 2) {
	while (this->src[i] != ',') {
	  base[j] = this->src[i];
	  i++; j++;
	}
	base[j] = '\0';
	i++; j = 0;
	while (this->src[i] != ',') {
	  index[j] = this->src[i];
	  i++; j++;
	}
	index[j] = '\0';
	i++; j = 0;
	while (this->src[i] != ')') {
	  scale[j] = this->src[i];
	  i++; j++;
	}
	scale[j] = '\0';
	i++; j = 0;
      } else {
	cout << "Fatal error !! (Contact author) \n"; 
      }
    } else {
      while (this->src[i] != '\0') {
	disp[j] = this->src[i];
	i++; j++;
      }
      disp[j] = '\0';
    }
  }

  if (where == DEST) {
    while (this->dest[i] != '\0') {
      if (this->dest[i] == ',') {
	com++;
      }
      i++;
    }
    i = 0;
    if (strstr(this->dest,":") != NULL) {
      while (this->dest[i] != ':') {
	section[j] = this->dest[i];
	i++; j++;
      }
      section[j] = '\0';
      strcat(section,":");
      j = 0;
      i++;
    }
    if (strstr(this->dest,"(") != NULL) {
      paran = true;
    } 
    if (paran == true) {
      while (this->dest[i] != '(') {
	disp[j] = this->dest[i];
	i++; j++;
      }
      disp[j] = '\0';
      j = 0;
      i++;
      if (com == 0) {
	while (this->dest[i] != ')') {
	  base[j] = this->dest[i];
	  i++; j++;
	}
	base[j] = '\0';
      } else if (com == 1) {
	while (this->dest[i] != ',') {
	  base[j] = this->dest[i];
	  i++; j++;
	}
	base[j] = '\0';
	j = 0;
	i++;
	if (strcmp(base,"")) {
	  while (this->dest[i] != ')') {
	    index[j] = this->dest[i];
	    i++; j++;
	  }
	  index[j] = '\0';
	}
      } else if (com == 2) {
	while (this->dest[i] != ',') {
	  base[j] = this->dest[i];
	  i++; j++;
	}
	base[j] = '\0';
	i++; j = 0;
	while (this->dest[i] != ',') {
	  index[j] = this->dest[i];
	  i++; j++;
	}
	index[j] = '\0';
	i++; j = 0;
	while (this->dest[i] != ')') {
	  scale[j] = this->dest[i];
	  i++; j++;
	}
	scale[j] = '\0';
	i++; j = 0;
      } else {
	cout << "Fatal error !! (Contact author) \n"; 
      }
    } else {
      while (this->dest[i] != '\0') {
	disp[j] = this->dest[i];
	i++; j++;
      }
      disp[j] = '\0';
    }
  }

  if (strcmp(section,"")) {
    strcpy(rettmpstr,section);
  }
  if (paran == true) {
    strcat(rettmpstr,"[");
    if (strcmp(base,"")) {
      strcat(rettmpstr,base);
    }
    if (strcmp(index,"")) {
      if ((index[0] != '-')&&(strcmp(base,""))) {
	strcat(rettmpstr,"+");
      }
      strcat(rettmpstr,index);
    }
    if (strcmp(scale,"")) {
      strcat(rettmpstr,"*");
      strcat(rettmpstr,scale);
    }
    if (strcmp(disp,"")) {
      if (disp[0] == '*') {
	strcpy(tmpstr,disp);
      	strdel(&tmpstr,"*","");
	strcpy(disp,tmpstr);
      }
      if (disp[0] != '-') {
	strcat(rettmpstr,"+");
      }
      switchstr = false;
      strcpy(tmpdisp,"");
      if (chknum(disp) == false) {
	numindex = findnum(disp);
	if (numindex != -1) {
	  strcpy(tmpdisp,disp);
	  switchstr = true;
	  strcpy(tmpstr,disp);
	  strdel(&tmpstr,numindex);
	  strcpy(disp,tmpstr);
	}
      }
      if (disp[0] == '$') {
	strcpy(tmpstr,disp);
      	strdel(&tmpstr,"$","");
	strcpy(disp,tmpstr);
      }
      tmp1 = parent->findfunc(disp);
      tmp2 = parent->findvar(disp);
      tmp3 = parent->findstr(disp);
      tmp4 = parent->findglobl(disp);
      if ((chknum(disp) == false)&&(isReg(disp) == NOREG)&&(tmp1 == NULL)&&
          (strncmp(disp,".L",2))&&(tmp2 == NULL)&&(tmp3 == NULL)&&(tmp4 == NULL)) {
	parent->insertid(disp,"",EXTLIST);
      }
      if (switchstr == true) {
	strcpy(disp,tmpdisp);
      }
      strcat(rettmpstr,disp);
    }
    strcat(rettmpstr,"]");
  } else {
    strcat(rettmpstr,disp);
  }

  i = strlen(rettmpstr) + 1;
  retstr = new char[i];

  if (!retstr) alloc_error();

  strcpy(retstr,rettmpstr);

  delete[] section;
  delete[] base;
  delete[] index;
  delete[] scale;
  delete[] disp;
  delete[] tmpstr;
  delete[] rettmpstr;
  delete[] tmpdisp;

  return retstr;
}

//
//	Method name : print
//
//	Description : Prints the entry
//	Input : ---
//	Output : ---
//
void Entry::print(void){
  cout << "Flag      : ";
  if (this->flag == EMPTY) {
    cout << "EMPTY ";
  } 
  if (this->flag & LABEL) {
    cout << "LABEL ";
  } 
  if (this->flag & OPCODE) {
    cout << "OPCODE ";
  } 
  if (this->flag & SRC) {
    cout << "SRC ";
  } 
  if (this->flag & DEST) {
    cout << "DEST ";
  }
  cout << "\n"; 
  cout << "Size      : ";
  if (this->size == NOSIZE) {
    cout << "NOSIZE";
  } else 
  if (this->size == BYTE) {
    cout << "BYTE";
  } else
  if (this->size == WORD) {
    cout << "WORD";
  } else
  if (this->size == LONG) {
    cout << "LONG";
  } else
  if (this->size == SHORT) {
    cout << "SHORT";
  } else
  if (this->size == QUAD) {
    cout << "QUAD";
  } else
  if (this->size == TERA) {
    cout << "TERA";
  }
  cout << "\n"; 
  cout << "Label     : " << this->label << "\n";
  cout << "Opcode    : " << this->opcodestr << "\n";
  cout << "Src       : " << this->src << "\n";
  cout << "Dest      : " << this->dest << "\n";
  cout << "\n";
}
