/*
** Copyright (c) Massachusetts Institute of Technology 1994-1998.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. Works derived from this code are not distributed for
**        commercial gain without explicit permission from MIT 
**        (for permission contact lclint-request@sds.lcs.mit.edu).
*/
/*
** sortSet.c
**
** based on set_template.c
**
** where T has T_equal (or change this) and T_unparse
*/

# include "lclintMacros.nf"
# include "llbasic.h"

sortSet sortSet_new ()
{
  sortSet s = (sortSet) dmalloc (sizeof (*s));
  
  s->entries = 0;
  s->nspace = sortSetBASESIZE;
  s->elements = (sort *) dmalloc (sizeof (*s->elements) * sortSetBASESIZE);
  
  return (s);
}

static /*@notnull@*/ sortSet
sortSet_predict (int size)
{
  sortSet s = (sortSet) dmalloc (sizeof (*s));
  
  s->entries = 0;

  if (size > 0)
    {
      s->nspace = size;
      s->elements = (sort *) dmalloc (sizeof (*s->elements) * size);
    }
  else
    {
      s->nspace = 0;
      s->elements = NULL;
    }
  
  return (s);
}

static void
sortSet_grow (/*@notnull@*/ sortSet s)
{
  int i;
  sort *newelements; 

  s->nspace = sortSetBASESIZE;
  newelements = (sort *) dmalloc (sizeof (*newelements) * (s->entries + s->nspace));

  if (newelements == (sort *) 0)
    {
      llfatalerror (cstring_makeLiteral ("sortSet_grow: out of memory!"));
    }

  for (i = 0; i < s->entries; i++)
    {
      newelements[i] = s->elements[i];
    }

  sfree (s->elements); 
  s->elements = newelements;
}

/*
** Ensures: if *e \in *s
**          then unchanged (*s) & result = false
**          else *s' = insert (*s, *e) & result = true
** Modifies: *s
*/

bool
sortSet_insert (sortSet s, sort el)
{
  llassert (sortSet_isDefined (s));

  if (sortSet_member (s, el))
    {
      return FALSE;
    }
  else
    {
      if (s->nspace <= 0)
	sortSet_grow (s);
      s->nspace--;
      s->elements[s->entries] = el;
      s->entries++;
      return TRUE;
    }
}

sort
sortSet_choose (sortSet s)
{
  llassert (sortSet_isDefined (s) && s->entries > 0);
  return (s->elements[0]);
}

bool
sortSet_member (sortSet s, sort el)
{
  if (sortSet_isDefined (s))
    {
      int i;

      for (i = 0; i < s->entries; i++)
	{
	  if (sort_equal (&el, &(s->elements[i])))
	    {
	      return TRUE;
	    }
	}
    }

  return FALSE;
}

/*@only@*/ cstring
sortSet_unparse (sortSet s)
{
  return (message ("{ %q }", sortSet_unparseClean (s)));
}

/*@only@*/ cstring
sortSet_unparseClean (sortSet s)
{
  cstring st = cstring_undefined;

  if (sortSet_isDefined (s))
    {
      int i;

      for (i = 0; i < s->entries; i++)
	{
	  if (i == 0)
	    {
	      st = message ("%q%s", st, sort_unparseName (s->elements[i]));
	    }
	  else
	    {
	      st = message ("%q, %s", st, sort_unparseName (s->elements[i]));
	    }
	}
    }

  return st;
}

/*@only@*/ cstring
sortSet_unparseOr (sortSet s)
{
  cstring st = cstring_undefined;

  if (sortSet_isDefined (s))
    {
      int i;
      int last = s->entries - 1;
      
      for (i = 0; i < s->entries; i++)
	{
	  if (i == 0)
	    {
	      st = cstring_concatFree (st, sort_unparse (s->elements[i]));
	    }
	  else
	    {
	      if (i == last)
		{
		  /* was sort_unparse ??? */
		  st = message ("%q or %q", st, sort_unparse (s->elements[i]));
		}
	      else
		{
		  st = message ("%q, %q", st, sort_unparse (s->elements[i]));
		}
	    }
	}
    }
  
  return st;
}

void
sortSet_free (sortSet s)
{
  if (sortSet_isDefined (s))
    {
      sfree (s->elements); 
      sfree (s);
    }
}

/*@only@*/ sortSet
sortSet_copy (sortSet s)
{
  sortSet t = sortSet_predict (sortSet_size (s));
  int i;

  if (sortSet_isDefined (s))
    {
      for (i = 0; i < sortSet_size (s); i++)
	{
	  (void) sortSet_insert (t, s->elements[i]); 
	}
    }

  return t;
}



