/*
** Copyright (c) Massachusetts Institute of Technology 1994-1998.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. Works derived from this code are not distributed for
**        commercial gain without explicit permission from MIT 
**        (for permission contact lclint-request@sds.lcs.mit.edu).
*/
/*
** exprNodeList.c
**
** based on list_template.c
**
** where T has T_equal (or change this) and T_unparse
*/

# include "lclintMacros.nf"
# include "basic.h"

/*@only@*/ exprNodeList
exprNodeList_new ()
{
  exprNodeList s = (exprNodeList) dmalloc (sizeof (*s));
  
  s->nelements = 0;
  s->nspace = exprNodeListBASESIZE; 
  s->elements = (exprNode *)
    dmalloc (sizeof (*s->elements) * exprNodeListBASESIZE);
  s->current = 0;

    return (s);
}

static void
exprNodeList_grow (exprNodeList s)
{
  int i;
  exprNode *newelements; 
  int numnew;

  if (s->nelements < exprNodeListBASESIZE)
    {
      numnew = exprNodeListBASESIZE;
    }
  else
    {
      numnew = s->nelements;
    }

  s->nspace = numnew + s->nspace; 

  newelements = (exprNode *) dmalloc (sizeof (*newelements) * (s->nelements + numnew));

  if (newelements == (exprNode *) 0)
    {
      llfatalerror (cstring_makeLiteral ("exprNodeList_grow: out of memory!"));
    }

  for (i = 0; i < s->nelements; i++)
    {
      newelements[i] = s->elements[i];
    }
  
  sfree (s->elements);
  s->elements = newelements;
}

void exprNodeList_addh (exprNodeList s, /*@only@*/ exprNode el)
{
  llassert (exprNodeListBASESIZE > 0);

  if (s->nspace <= 0)
    exprNodeList_grow (s);
  
  s->nspace--;
  s->elements[s->nelements] = el;
  s->nelements++;
}

void exprNodeList_reset (exprNodeList s)
{
  s->current = 0;
}

void exprNodeList_advance (exprNodeList s)
{
  s->current++;
  llassert (s->current <= s->nelements);
}

/*@observer@*/ exprNode exprNodeList_head (exprNodeList s)
{
  llassert (s->nelements > 0);
  return (s->elements[0]);
}

/*@observer@*/ exprNode exprNodeList_current (exprNodeList s)
{
  llassert (s->current >= 0 && s->current < s->nelements);
  return (s->elements[s->current]);
}

exprNode exprNodeList_getN (exprNodeList s, int n)
{
  llassert (n >= 0 && n < s->nelements);
  return (s->elements[n]);
}

/*@only@*/ exprNodeList exprNodeList_singleton (/*@only@*/ exprNode e)
{
  exprNodeList s = (exprNodeList) dmalloc (sizeof (*s));
  
  s->nelements = 1;
  s->nspace = exprNodeListBASESIZE - 1; 
  s->elements = (exprNode *) dmalloc (sizeof (*s->elements) * exprNodeListBASESIZE);
  s->elements[0] = e;
  s->current = 0;

  return (s);
}

exprNodeList exprNodeList_push (/*@returned@*/ exprNodeList args, /*@only@*/ exprNode e)
{
  exprNodeList_addh (args, e);
  return (args);
}

/*@exposed@*/ exprNode
exprNodeList_nth (exprNodeList args, int n)
{
  if (n >= exprNodeList_size (args) || n < 0)
    {
      llcontbug (message ("exprNodeList_nth: out of range: %q arg %d\n", 
			  exprNodeList_unparse (args), n));
      return exprNode_undefined;
    }

  return args->elements[n]; 
}

/*@only@*/ cstring
exprNodeList_unparse (exprNodeList s)
{
   int i;
   cstring st = cstring_undefined;

      for (i = 0; i < s->nelements; i++)
     {
       if (i == 0)
	 {
	   st = cstring_copy (exprNode_unparse (s->elements[i]));
	 }
       else
	 st = message ("%q, %s", st, exprNode_unparse (s->elements[i]));
     }
   
   return st;
}

void
exprNodeList_free (exprNodeList s)
{
  int i;

  for (i = 0; i < s->nelements; i++)
    {
      exprNode_free (s->elements[i]); 
    }
  
  sfree (s->elements); 
  sfree (s);
}

void
exprNodeList_freeShallow (/*@only@*/ exprNodeList s)
{
  int i;

  for (i = 0; i < s->nelements; i++)
    {
      exprNode_freeShallow (s->elements[i]); 
    }
  
  sfree (s->elements); 
  sfree (s);
}

