package biss.awt.kernel;

import biss.FileLib;
import biss.Queue;
import biss.awt.MutableIndexClrModel;
import java.awt.Graphics;
import java.awt.image.ColorModel;
import java.awt.image.ImageConsumer;
import java.awt.image.ImageObserver;
import java.awt.image.ImageProducer;
import java.awt.image.IndexColorModel;
import java.io.File;
import java.util.Hashtable;

/**
 * kernel implementation of java.awt.Image
 * In difference to FontMetrics or Graphics, this class really needs to
 * be an extension of its java.awt base. It has to settle on a state model
 * of images as well as on a format for their internal representation.
 * Images are by no means simple objects, they may have an external
 * representation (file o URL), an internal representation (e.g. a
 * MemoryImageSource), a mechanism to convert the first into the second one
 * (the ImageProducer). It is recommended that an image may act as an
 * producer of itself once it has been successfully initialized (it has its
 * internal representation set).
 * And it even might be a completely different thing: an offscreen drawing
 * canvas.
 * All of this is accompanied by significant memory and speed requirements
 * (both within the Java process and the XServer)
 *
 * This implementation lets the producer decide what might be the best
 * internal representation (since it knows best of the most efficient/
 * conserving way to transform the external info into Java data). But
 * this puts the burden of getting the RGB data to render on the Graphics
 * object / the native lib. Within the current native layer, just
 * MutableColorIndex based and plain RGB (int[]) pixel data can be
 * rendered. An alternative would be to use the producer protocol of
 * Images to get the RGBs which have to be turned into pixel values. But
 * there are operations (scaling images, for instance) which requires the
 * whole image being accessible randomly, not sequential. This could be
 * done just be building an Java RGB array out of the - whatever it is -
 * internal model. Since this might occur quite frequently (e.g. displaying
 * scaled images), it seems to be too costly both in terms of speed and space.
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
public class Image
  extends java.awt.Image
  implements ImageConsumer, ImageProducer
{
	NativeLib Lib;
	int Width;
	int Height;
	ImageProducer Source;
	int Background = -1;
	ColorModel Model;
	Object Pixels;
	Hashtable Props;
	ImageConsumer Consumer;
	int State = UNINITIALIZED;
	final static int UNINITIALIZED = 0;
	final static int INITIALIZING = 2;
	final static int INITIALIZED = 4;
	static ImageProdThread ProdThread;
	static Queue ProdQueue = new Queue( 30);
	ImageObserver Obs;
	final static int QUEUED = 1;
	final static int PARTLYINIT = 3;
	long PDataPix;
	long PDataImg;
	int ModelType = MT_NONE;
	final static int MT_NONE = 0;
	final static int MT_INDEX = 1;
	final static int MT_MUTABLE_INDEX = 2;

Image ( NativeLib lib, ImageProducer source ) {
	Lib = lib;

	Source = source;
}

Image ( NativeLib lib, int width, int height ) {
	Lib = lib;

	Width = width;
	Height = height;
	Lib.initBlankImage( this, width, height);
	State = INITIALIZED;
}

public void addConsumer ( ImageConsumer ic ) {
	Consumer = ic;
	startProduction();
	Consumer = null;
}

public void finalize () {
	flush();
}

public void flush() {
	Lib.releaseImage( this);
}

public java.awt.Graphics getGraphics() {
	return (new biss.awt.kernel.Graphics( Lib, this));
}

public int getHeight ( ImageObserver observer ) {
	if ( State >= INITIALIZING )
		return Height;

	if ( State == UNINITIALIZED ) {
		queueProduction( observer);
	}

	return -1;
}

public Object getProperty ( String name, ImageObserver observer ){
	return null;
}

public ImageProducer getSource() {
	return Source;
}

public int getWidth ( ImageObserver observer ) {
	if ( State >= INITIALIZING )
		return Width;

	if ( State == UNINITIALIZED ) {
		queueProduction( observer);
	}

	return -1;
}

public void imageComplete ( int status ) {
	int flags = 0;

	State = INITIALIZED;

	if ( status == SINGLEFRAMEDONE )		flags |= ImageObserver.SOMEBITS;
	if ( status == STATICIMAGEDONE )    flags |= ImageObserver.ALLBITS;
	if ( status == IMAGEERROR )         flags |= ImageObserver.ERROR;
	if ( status == IMAGEABORTED )       flags |= ImageObserver.ABORT;

	if ( Obs != null ) {
		if ( !Obs.imageUpdate( this, flags, 0,0, Width, Height) )
			Obs = null;
	}
}

public boolean isConsumer ( ImageConsumer ic ) {
	return (ic == Consumer);
}

synchronized void queueProduction ( ImageObserver obs ) {

	if ( State == UNINITIALIZED) {
		State = QUEUED;

		Obs = obs;
		ProdQueue.append( this);

		if ( ProdThread == null )
			ProdThread = new ImageProdThread();
	}
	else if ( State == INITIALIZED ) {
		obs.imageUpdate( this, ImageObserver.ALLBITS, 0,0, Width, Height);
	}
	else {
		Obs = obs;
	}
}

public void removeConsumer ( ImageConsumer ic ) {
	if ( Consumer == ic )
		Consumer = null;
}

public void requestTopDownLeftRightResend(ImageConsumer ic) {
}

public void setColorModel ( ColorModel model ) {
	Model = model;

	// these are the ColorModels which currently can be rendered by the native layer
	if ( Model instanceof IndexColorModel )
		ModelType = MT_INDEX;
	else if ( Model instanceof MutableIndexClrModel )
		ModelType = MT_MUTABLE_INDEX;
	else
		ModelType = MT_NONE;
}

public void setDimensions ( int width, int height ) {
	Width = width;
	Height = height;

	if ( Obs != null ) {
		if ( !Obs.imageUpdate( this, ImageObserver.WIDTH | ImageObserver.HEIGHT,
		                       0,0, width, height) )
			Obs = null;
	}
}

public void setHints(int hintflags) {
}

public void setPixels ( int x, int y, int w, int h,
                 ColorModel model, byte pixels[], int off, int scansize) {
	setColorModel( model);

	if ( (pixels.length >= Width*Height) && (off == 0) && (scansize == Width) ) {
		Pixels = pixels;
		State = PARTLYINIT;
	}
}

public void setPixels(int x, int y, int w, int h,
               ColorModel model, int pixels[], int off, int scansize) {
}

public void setProperties ( Hashtable props ) {
	Props = props;

	if ( Obs != null ) {
		if ( !Obs.imageUpdate( this, ImageObserver.PROPERTIES, 0,0, Width, Height) )
			Obs = null;
	}
}

void startProduction () {
	try {
		if ( Consumer != null ) {
			Consumer.setDimensions( Width, Height);
		}
		if ( Consumer != null ) {
			Consumer.setProperties( Props);
		}
		if ( Consumer != null ) {
			Consumer.setColorModel( Model);
		}
		if ( Consumer != null ) {
			Consumer.setHints( ImageConsumer.TOPDOWNLEFTRIGHT | ImageConsumer.COMPLETESCANLINES |
			                   ImageConsumer.SINGLEPASS | ImageConsumer.SINGLEFRAME);
		}
		if ( Consumer != null ) {
			if ( Pixels instanceof byte[] ) {
				Consumer.setPixels( 0, 0, Width, Height, Model, ((byte[]) Pixels), 0, Width);
			} else {
				Consumer.setPixels( 0, 0, Width, Height, Model, ((int[]) Pixels), 0, Width);
			}
		}
		if ( Consumer != null ) {
			Consumer.imageComplete( ImageConsumer.STATICIMAGEDONE);
		}
	}
	catch (Exception e) {
		if ( Consumer != null) {
			Consumer.imageComplete( ImageConsumer.IMAGEERROR);
		}
	}
}

public void startProduction(ImageConsumer ic) {
	addConsumer( ic);
}
}

class ImageProdThread
  extends Thread
{

ImageProdThread () {
	setPriority( Thread.MIN_PRIORITY);
	start();
}

public void run () {
	Image img;

	while ( !Image.ProdQueue.empty() ) {
		img = (Image) Image.ProdQueue.getNext();
		img.Source.startProduction( img);
	}

	Image.ProdThread = null;
}
}
