package biss.awt;

import biss.ObserverSocket;
import biss.Queue;
import biss.StringLib;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Point;
import java.util.Enumeration;
import java.util.Observable;
import java.util.Vector;

/**
 * replaces java.awt.TextArea
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
public class TextArea
  extends ScrollablePane
  implements TextAreaPeer
{
	TextAreaCanvas Cv;
	int FindMode = 1;
	String FindPattern = null;
/**
 * triggered by contents modification
 * parameter: -
 */
	public ObserverSocket OsModified = new ObserverSocket( this);
/**
 * triggered by LineBuffer modification (e.g. used for colorizing
 * source code)
 * parameter: changed LineBuffer
 */
	public ObserverSocket OsFormatLine = new ObserverSocket( this);
	PromptViewer Prompter = null;

public TextArea() {
	this( null, -1, 1, Awt.EntryBackClr, Awt.EntryForeClr);
}

public TextArea( Font fnt) {
	this( fnt, -1, 1, Color.white, Color.black);
}

public TextArea( Font fnt, int xBorder) {
	this( fnt, xBorder, 1, Color.white, Color.black);
}

public TextArea( Font fnt, int xBorder, double lineSpacing, Color cb, Color cf) {
	Cv = new TextAreaCanvas( Horz, Vert, this, xBorder, lineSpacing, cb, cf, fnt);
	setCanvas( Cv);
	buildMenu();
}

public TextArea(String text) {
	this();
	setContents( text);
}

public TextArea( String text, int rows, int cols) {
	//java.awt compatible ct
	this( text);
}

public TextArea( int rows, int cols) {
	//java.awt compatible ct
	this();
}

public TextArea( java.awt.TextArea area){
	this();
	setPeerTarget( area);
	addNotify();

	setContents( area.getText() );
}

public void addContents( String s){
	Cv.setContents( s, true);
}

public void appendText ( String s ) {
	addContents( s);
}

Menu buildMenu(){
	Menu m = new Menu();
	m.addItem( "C~ut");
	m.addItem( "C~opy");
	m.addItem( "P~aste");
	m.addSeparator();
	m.addItem( "~Find...", "FindString");
	m.addItem( "Find ~next", "FindNext");
	m.addItem( "Find/~Replace", "FindReplace");
	m.addSeparator();
	m.addItem( "~Select all", "SelectAll");
	m.addItem( "~Goto...", "GotoLine");

	Cv.setMenu( m);
	Cv.OsCommand.addObserver( this); 

	return m;
}

public void clearSelection () {
	Cv.updateSel( true);
}

public void colorizeLine ( LineBuffer lb, int startIdx, int endIdx, Color clr ) {
	Cv.colorizeLine( lb, startIdx, endIdx, clr);
}

public void cursorDown ( boolean extendSel ) {
	Cv.cursorDown( extendSel );
}

public void cursorEnd ( boolean extendSel ) {
	Cv.cursorEnd( extendSel );
}

public void cursorHome ( boolean extendSel ) {
	Cv.cursorHome( extendSel );
}

public void cursorUp ( boolean extendSel ) {
	Cv.cursorUp( extendSel );
}

public void findNext(){
	String s = Cv.getSelection();

	if ( s != null)
		FindPattern = s;

	if ( FindMode == 2)
		Cv.findStringId( FindPattern );
	else
		Cv.findString( FindPattern);
}

public void findString( String pattern){
	Cv.findString( pattern);
}

public void findStringId( String pattern){
	FindMode = 2;
	FindPattern = pattern;
	Cv.findStringId( pattern);
}

public String getContents() {
	return Cv.getContents();
}

public Point getCursorIdx () {
	return Cv.CursorIdx;
}

public char[] getLineContents () {
	return getLineContents( Cv.CursorIdx.y);
}

public char[] getLineContents ( int idx ) {
	if ( idx >= 0 && idx < Cv.Lines.size() ) {
		LineBuffer lb = (LineBuffer) Cv.Lines.elementAt( idx);
		return lb.Text;
	}
	else
		return null;
}

public Vector getLines() {
	return Cv.Lines;
}

public int getNumberOfLines () {
	return Cv.Lines.size();
}

public String getSelection() {
	return Cv.getSelection();
}

public int getSelectionEnd( ) {
	//java.awt.peer.TextAreaPeer interface
	return Cv.idxOfPos( Cv.selEnd() );
}

public int getSelectionStart( ) {
	//java.awt.peer.TextAreaPeer interface
	return Cv.idxOfPos( Cv.selStart() );
}

public String getStrippedContents() {
	return Cv.getStrippedContents();
}

public String getText( ) {
	//java.awt.peer.TextAreaPeer interface
	return getContents();
}

public boolean hasSelection( ) {
	return Cv.hasSelection();
}

public void insertText ( String txt, int pos) {
	//java.awt.peer.TextArea interface
	Point p = Cv.posOf( pos);
	setCursorIdx( p.x, p.y);
	Cv.insert( txt);
}

public boolean isModified() {
	return Cv.Modified;
}

public boolean isReadOnly(){
	return ! Cv.isEnabled();
}

public Dimension minimumSize ( int rows, int cols) {
	//java.awt.peer.TextArea interface
	return new Dimension( Awt.SysFontWidth*cols, Awt.SysFontHeight*rows);
}

public void openPrompter( String label, String txt, String defCont, int fm){
	if ( Prompter == null) {
		TopWindow tw = Cv.getTopWindow();
		FindMode = fm;
		Prompter  = new PromptViewer( label, txt, defCont, tw);
		Prompter.OsClose.addObserver( this);
		Prompter.openCentered();
	}
}

public Dimension preferredSize(){
	return new Dimension( 15*Awt.SysFontWidth, 7*Awt.SysFontHeight);
}

public Dimension preferredSize ( int rows, int cols) {
	//java.awt.peer.TextArea interface
	return new Dimension( Awt.SysFontWidth*cols, Awt.SysFontHeight*rows);
}

public void replaceSelectionWith( String s) {
	Cv.replaceSelectionWith( s);
}

public void replaceText( String txt, int start, int end ) {
	//java.awt.peer.TextAreaPeer interface
	select( start, end);
	Cv.replaceSelectionWith( txt);
}

public void select( int selStart, int selEnd ) {
	//java.awt.peer.TextAreaPeer interface
	Point ps = Cv.posOf( selStart);
	Point pe = Cv.posOf( selEnd);
	setSelection( ps.x, ps.y, pe.x, pe.y);
}

public void selectAll () {
	Cv.selectAll();
}

public void selectLine () {
	Cv.selectLine( Cv.CursorIdx.y);
}

public void selectLine( int lineY){
	Cv.selectLine( lineY);
}

public synchronized void setBackground( Color cc){
	Cv.setBackground( cc );
	Cv.Cursor.setBackground( cc);
}

public void setContents( String s){
	Cv.setContents( s, false);
}

public void setCursorClr( Color cc){
	Cv.setCursorClr( cc );
}

public void setCursorIdx( int x, int y){
	Cv.setCursorIdx( x, y);
}

public void setCursorWidth( int pels){
	Cv.setCursorWidth( pels );
}

public void setEditable( boolean editable ) {
	//java.awt.peer.TextAreaPeer interface
	Cv.enable( editable);
}

public synchronized void setFont( Font fnt){
	Cv.setFont( fnt );
}

public synchronized void setForeground( Color cc){
	Cv.setForeground( cc );
	Cv.Cursor.setForeground( cc);
}

public void setModified( boolean state){
	Cv.setModified( state);
}

public void setReadOnly( boolean state){
	if ( state)  Cv.disable();
	else         Cv.enable();
}

public void setSelection( int startX, int startY,
                   int endX, int endY) {
	Cv.setSelection( startX, startY, endX, endY);
}

public void setTabWidth( int numChars){
	Cv.setTabWidth( numChars );
}

public void setText( String l ) {
	//java.awt.peer.TextAreaPeer interface
	setContents( l);
}

public void update ( Observable o, Object arg) {

	if ( o == Cv.OsCommand ) {
		if ( arg.equals( "Cut")) {
			ClipBoard.set( Cv.getSelection() );
			Cv.deleteSelection();
		}
		else if ( arg.equals( "Copy"))
			ClipBoard.set( Cv.getSelection() );
		else if ( arg.equals( "Paste"))
			Cv.replaceSelectionWith( ClipBoard.get() );
		else if ( arg.equals( "SelectAll"))
			Cv.selectAll();
		else if ( arg.equals( "FindString"))
			openPrompter( "Find", "Lookup for string:", Cv.getSelection(), 1 );
		else if ( arg.equals( "FindNext"))
			findNext();
		else if ( arg.equals( "GotoLine"))
			openPrompter( "Position", "Goto line:", null, 0);
		else if ( arg.equals( "FindReplace"))
			new TextReplacer( this, Cv.getTopWindow(), Cv.getSelection() );
	}
	else if ( o instanceof TextReplacer){
		TextReplacer tr = (TextReplacer)o;
		switch( tr.action() ){
		case TextReplacer.FIND:
			findString( tr.getFindText() );
			break;
		case TextReplacer.REPLACE:
			if ( Cv.hasSelection() )
				Cv.replaceSelectionWith( tr.getReplaceText() );
			break;
		}
	}
	else if ( (Prompter != null) && (o == Prompter.OsClose) ){
		if ( ! Prompter.WasCanceled) {
			switch( FindMode) {
			case 0:
				try { Cv.selectLine (Integer.parseInt( Prompter.getContents() )); }
				catch ( Exception e) {}
				break;
			case 1:
				FindPattern = Prompter.getContents();
				Cv.findString( FindPattern);
				break;
			}
		}
		FindMode = 1;
		Prompter = null;
	}

	super.update( o, arg);
}

public void updateContents( String s){
	Cv.updateContents( s);
}
}

class TextAreaCanvas
  extends ScrollableCanvas
{
	TextCursor Cursor;
	Point CursorIdx = new Point(0, 0);
	int Descent;
	FontMetrics Fm;
	final static char LBr = '\n';
	Point LineSelRange = new Point(0, 0);
	double LineSpacing = 1;
	TextArea Master;
	int Mode = 0;
	boolean Modified = false;
	Point SelEnd = new Point(0, 0);
	Point SelStart = new Point(0, 0);
	int TabWidth;
	int TryCursorX;
	static FontMetrics DefFm;
/**
 * This state is used to enable special treatment of re-gaining the focus
 * when the TextArea had a selection. In this case, the cursor should NOT
 * be positioned on the first Mouse1 click (i.e. the selection should be
 * remembered if re-selecting the TextArea from another window).
 * '0' : TextArea has no focus (set by lostFocus())
 * '1' : TextArea (with selection) just reentered (set by gotFocus())
 * '2' : all other cases (set by gotFocus(), mouse1Down())
 */
	int FocusState = 0;

static {
	DefFm = Awt.getFontMetrics( Awt.FixedFont);
}

public TextAreaCanvas( HScrollBar hs, VScrollBar vs, TextArea m, int xBorder,
                double ls, Color cb, Color cf, Font fnt) {
	super( hs, vs);

	Master	= m;
	Lines 	= new Vector( 50);
	XBorder	= xBorder > -1 ? xBorder : 3;
	TabWidth	= Awt.TabWidth;

	LineSpacing = ls > 0 ? ls : LineSpacing;
	EventHandler.CursorType = TopWindow.TEXT_CURSOR;

	Cursor = new TextCursor( cb);
	setForeground( cf);
	setBackground( cb);

	setFont( fnt);
	reset();
}

int addContents( String s) {

	LineBuffer lb;
	Vector vi = buildLines( s);
	int    sz = vi.size();

	for ( int i=0; i<sz; i++){
		lb = (LineBuffer)vi.elementAt( i);
		if ( lb != null)
			Lines.addElement( lb);
	}

	if ( Lines.size() < IdxTop)
		IdxTop = Lines.size();

	updateVScroll();
	redrawLines( IdxTop, -1);

	return vi.size();
}

void addLine( LineBuffer lb) {
	lb.setTabWidth( TabWidth);
	Lines.addElement( lb);
	updateVScroll();
}

boolean backSpace() {
	int idx;
	LineBuffer lbl, lb;

	if ( hasSelection() ){
		deleteSelection();
		return false;
	}	

	lb = cursorLine();
	if ( CursorIdx.x > 0){
		CursorIdx.x--;
		lb.remove( CursorIdx.x, CursorIdx.x + 1);
		updateCursor( lb);
		redrawLines( CursorIdx.y, CursorIdx.y);
	}
	else if ( CursorIdx.y > 0) {
		if ( CursorIdx.y == IdxTop)
			lineUp();
		CursorIdx.y--;
		lbl = cursorLine();
		CursorIdx.x = lbl.Len;
		updateCursor( lbl);
		lbl.append( lb.Text, 0, lb.Len);
		removeLine( lb);
		redrawLines( CursorIdx.y, -1);
	}

	TryCursorX = CursorIdx.x;
	updateSel( false);

	checkModified();
	return true;
}

void  blankCursor () {
	LineBuffer lb;

	if ( ResGraph == null)
		return;

	Cursor.clearAtOffs( ResGraph, XOffset, YOffset);

	ResGraph.setColor( getForeground());
	lb = (LineBuffer)Lines.elementAt( CursorIdx.y);
	if ( lb.Len > CursorIdx.x){
		if ( lb.Text[CursorIdx.x] != '\t')
			ResGraph.drawChars( lb.Text, CursorIdx.x, 1,
		                    XOffset + Cursor.XPos,
		                    YOffset + Cursor.YPos - Descent - 1); 
	}		
}

Vector buildLines( String s){
	Vector v = new Vector();
	int    li = 0;
	char   ca[] = s.toCharArray();

	for ( int i = 0; i < ca.length; i++){
		if ( ca[i] == LBr){
			LineBuffer lb;
			if ( i > 0 && ca[i-1] == '\r' )  // eliminate DOS CRs
				lb = new LineBuffer( Fm, ca, li, i-1, TabWidth);
			else
				lb = new LineBuffer( Fm, ca, li, i, TabWidth);

			v.addElement( lb);
			li = i+1;
		}
	}

	if ( li < ca.length)
		v.addElement( new LineBuffer( Fm, ca, li, ca.length, TabWidth));
	else
		v.addElement( null);

	return v;
}

public void charInput( char c, int mod) {
	LineBuffer lb;

	if ( hasSelection() )
		replaceSelectionWith( c);
	else {	
		lb= cursorLine();		
		lb.insert( c, CursorIdx.x);
		CursorIdx.x++;        
		updateSel( true);
		updateCursor( lb);
		redrawLine( CursorIdx.y);
		TryCursorX = CursorIdx.x;
		checkModified();
	}	
}

void checkModified(){
	if ( ! Modified){
		Modified = true;
		Master.OsModified.notifyObservers();
	}
}

synchronized void colorizeLine( LineBuffer lb, int si, int ei, Color clr){
	int yPos = lineYPos( lb);
	int xPos;

	if ( yPos == 0)
		return;

	ResGraph.setColor( clr);
	xPos = lb.posFor( si) + XOffset;

	lb.draw( si, ei-si, ResGraph, xPos, yPos - 1, XOffset);
}

void cursorDown( boolean extSel){
	LineBuffer lb;

	if ( CursorIdx.y < Lines.size() - 1) {

		blankCursor();
		pushFormat( cursorLine() );

		CursorIdx.y++;
		lb = cursorLine();
		CursorIdx.x = Math.min( lb.Len, TryCursorX);
		updateCursor( lb);

		if ( extSel) {
			SelEnd.x = CursorIdx.x;
			SelEnd.y = CursorIdx.y;
			redrawLines( CursorIdx.y - 1, CursorIdx.y);
		}
		else{
			updateSel( true);				
			drawCursor();
		}
	}	
}

void cursorEnd( boolean extSel) {
	LineBuffer lb = cursorLine();

	if ( CursorIdx.x < lb.Len) {

		if ( extSel) {
			SelEnd.x = CursorIdx.x = lb.Len;
			updateCursor( lb);
			redrawLine( CursorIdx.y);
		}	
		else {					
			blankCursor();
			CursorIdx.x = lb.Len;
			updateCursor( lb);
			updateSel( true);
			drawCursor();
		}
	}
	TryCursorX = CursorIdx.x;
}

void cursorHome( boolean extSel) {
	if ( CursorIdx.x != 0) {

		positionHorizontal( 0, true);
		if ( extSel) {
			SelEnd.x = CursorIdx.x = Cursor.XPos = 0;
			redrawLine( CursorIdx.y);
		}	
		else {					
			blankCursor();
			CursorIdx.x = 0;
			Cursor.XPos = 0;
			updateSel( true);
			drawCursor();
		}
	}
	TryCursorX = CursorIdx.x;
}

void cursorLeft( boolean extSel) {
	LineBuffer lb;

	blankCursor();

	if ( CursorIdx.x > 0){
		CursorIdx.x--;
		updateCursor( null);
	}
	else if ( CursorIdx.y > 0) {
		CursorIdx.y--;
		lb = (LineBuffer)Lines.elementAt( CursorIdx.y);
		CursorIdx.x = lb.Len;
		updateCursor( lb);
	}	

	if ( extSel) {
		SelEnd.x = CursorIdx.x;
		SelEnd.y = CursorIdx.y;
		redrawLines( CursorIdx.y, CursorIdx.y + 1);
	}
	else{
		updateSel( true);
		drawCursor();
	}

	TryCursorX = CursorIdx.x;
}

LineBuffer cursorLine() {
	if ( (Lines == null) || (Lines.size() == 0) )
		return null;
	return (LineBuffer) Lines.elementAt( CursorIdx.y);
}

void cursorRight( boolean extSel) {
	LineBuffer  lb = (LineBuffer) Lines.elementAt( CursorIdx.y);

	blankCursor();

	if ( CursorIdx.x < lb.Len){
		CursorIdx.x++;
		updateCursor( lb);
	}
	else if ( CursorIdx.y < Lines.size()- 1) {
		CursorIdx.x = 0;
		CursorIdx.y++;
		updateCursor( null);
	}	

	if ( extSel) {
		SelEnd.x = CursorIdx.x;
		SelEnd.y = CursorIdx.y;
		redrawLines( CursorIdx.y - 1, CursorIdx.y);
	}
	else{
		updateSel( true);
		drawCursor();
	}

	TryCursorX = CursorIdx.x;	
}

void cursorUp( boolean extSel){
	LineBuffer lb;

	if ( CursorIdx.y > 0) {

		blankCursor();
		pushFormat( cursorLine() );

		CursorIdx.y--;
		lb = cursorLine();
		CursorIdx.x = Math.min( lb.Len, TryCursorX);
		updateCursor( lb);

		if ( extSel) {
			SelEnd.x = CursorIdx.x;
			SelEnd.y = CursorIdx.y;
			redrawLines( CursorIdx.y, CursorIdx.y + 1);
		}
		else{
			updateSel( true);				
			drawCursor();
		}	
	}	
}

boolean del() {    	
	if ( hasSelection() ){
		deleteSelection();
		return false;
	}	
	if ( CursorIdx.x == cursorLine().Len){
		if ( CursorIdx.y == Lines.size() - 1)
			return false;
		CursorIdx.x = 0;
		CursorIdx.y++;
	}
	else
		CursorIdx.x++;

	return backSpace();	
}

void deleteSelection() {
	replaceSelectionWith( null);
}

void drawCursor () {
	if ( ResGraph != null)
		Cursor.drawAtOffs( ResGraph, XOffset, YOffset);
}

void findString ( String pattern) {
	LineBuffer lb;
	String     lStr;
	int        lIdx, sIdx, mIdx;

	if ( (pattern == null) || isEmpty() )
		return;

	for ( lIdx = CursorIdx.y; lIdx < Lines.size(); lIdx++){
		lb = (LineBuffer)(Lines.elementAt( lIdx));
		if ( lIdx == CursorIdx.y)
			sIdx = CursorIdx.x;
		else
			sIdx = 0;
		lStr = String.valueOf( lb.Text, sIdx, lb.Len - sIdx);
		mIdx = lStr.indexOf( pattern);
		if ( mIdx > -1){
			setSelection( sIdx + mIdx, lIdx,
			              sIdx + mIdx + pattern.length(), lIdx);	
			return;
		}
	}

}

void findStringId ( String pattern) {
	LineBuffer lb;
	String     lStr;
	int        lIdx, sIdx, mIdx;

	if ( (pattern == null) || isEmpty() )
		return;

	for ( lIdx = CursorIdx.y; lIdx < Lines.size(); lIdx++){
		lb = (LineBuffer)(Lines.elementAt( lIdx));
		if ( lIdx == CursorIdx.y)
			sIdx = CursorIdx.x;
		else
			sIdx = 0;
		lStr = String.valueOf( lb.Text, sIdx, lb.Len - sIdx);
		mIdx = StringLib.indexOfId( lStr, pattern);
		if ( mIdx > -1){
			setSelection( sIdx + mIdx, lIdx,
			              sIdx + mIdx + pattern.length(), lIdx);	
			return;
		}
	}

}

String getContents() {
	StringBuffer buf = new StringBuffer();
	LineBuffer   lb;

	for ( Enumeration e = Lines.elements(); true;){
		lb = (LineBuffer)e.nextElement();
		buf.append( lb.Text, 0, lb.Len);

		if ( e.hasMoreElements() )
			buf.append( LBr);
		else
			break;
	}

	return buf.toString();
}

String getSelection() {
	StringBuffer buf;
	LineBuffer	 lb;
	int		 sl, sy, len, idx;
	Point	 range;

	if ( ! hasSelection() )
		return null;

	buf = new StringBuffer();
	sy  = selStart().y;
	sl  = selLines();

	for ( idx = sy; idx <= sy + sl; idx++) {
		lb = (LineBuffer)Lines.elementAt( idx);
		range = lineSelRange( idx, lb);
		buf.append( lb.Text, range.x, range.y - range.x);
		if ( idx < sy + sl)
			buf.append( LBr);
	}

	return buf.toString();
}

String getStrippedContents() {
	StringBuffer buf = new StringBuffer();
	boolean      skipHead = true;
	int          nLBr = 0;
	LineBuffer   lb;

	for ( Enumeration e = Lines.elements(); e.hasMoreElements();){

		lb = (LineBuffer)e.nextElement();

		if ( !lb.isBlankLine() ) {
			while ( nLBr-- > 0 )
				buf.append( LBr);

			buf.append( lb.Text, 0, lb.Len);
			nLBr = 1;
			skipHead = false;
		}
		else {
			if ( skipHead )
				continue;
			else
				nLBr++;
		}
	}

	return buf.toString();
}

public void gotFocus(){
	setCursorClr( Awt.CursorClr);
//	drawCursor();

	if ( hasSelection() ) {
		if ( FocusState == 0) // may be before or after mouse1Down -> check value
			FocusState = 1;     // just re-entered from outside, skip Cursor positioning
		// redrawLines( SelStart.y, SelEnd.y);  // just if we changed colors
	}
	else
		FocusState = 2;
}

boolean hasSelection() {
	return (! SelStart.equals( SelEnd));
}

int idxOfPos( Point p) {
	if ( Lines == null)
		return 0;

	int i, s = Lines.size();
	int idx = 0;

	for ( i=0; i<s; i++){
		LineBuffer lb = (LineBuffer)Lines.elementAt( i);
		if ( i == p.y)
			break;
		idx += lb.Len;
	}
	return idx + p.x;    
}

int insert ( String s){
	LineBuffer lbo, lb, lbi, nlb;
	Vector     vi;
	int        ol, idx, al = 0;
	int        y0 = CursorIdx.y;
	boolean    ll = true;
	int        nx = CursorIdx.x;
	int        ny = CursorIdx.y;

	if ( s == null)
		return 0;

	nlb = lbo = cursorLine();
	vi = buildLines( s);

	for ( idx = vi.size() - 1; idx >= 0; idx--){
		lbi = (LineBuffer)vi.elementAt( idx);
		if ( ll){
			if ( lbi == null){
				lb = new LineBuffer( Fm, lbo.Text, CursorIdx.x, lbo.Len, TabWidth);
				lbo.setLen( CursorIdx.x);
				Lines.insertElementAt( lb, y0 + 1);
				nlb = lb;
				nx = 0;
				ny++;
			}
			else{
				lbo.insert( lbi.Text, 0, lbi.Len, CursorIdx.x);
				nlb = lbo;
				nx = CursorIdx.x + lbi.Len;
			}
			ll = false;
		}
		else if ( idx > 0){
			Lines.insertElementAt( lbi, y0 + 1);
			al++;
			ny++;
		}
		else {
			ol = lbo.Len;
			lbo.insert( lbi.Text, 0, lbi.Len, CursorIdx.x);
			if ( ol > CursorIdx.x){
				ol = CursorIdx.x + lbi.Len;
				lb = new LineBuffer( Fm, lbo.Text, ol, lbo.Len, TabWidth);
				lbo.setLen( ol);
				Lines.insertElementAt( lb, y0 + al + 1);
				ny++;
			}
		}
	}	

	CursorIdx.x = nx;
	CursorIdx.y = ny;
	updateCursor( nlb);

	updateVScroll();
	checkModified();

	redrawLines( IdxTop, -1);

	return 1;
}

boolean isEmpty(){
	if ( (Lines == null) || (Lines.size() == 0) ||
	     ((Lines.size() == 1) &&
	 (((LineBuffer)Lines.elementAt( 0)).Len == 0 )))
		return true;

	return false;
}

Point lineSelRange( int lIdx, LineBuffer lb) {
	Point sp, ep;

	// no selection at all
	if ( SelStart.equals( SelEnd))
		return null;

	sp = selStart();
	ep = sp.equals( SelStart) ? SelEnd : SelStart;


	// line not within selection
	if ( (lIdx < sp.y) || (lIdx > ep.y))
		return null;

	// line completely within selection range
	if ( (lIdx > sp.y) && (lIdx < ep.y)){
		LineSelRange.x = 0;
		LineSelRange.y = lb.Len;
	}
	// first line partly selected
	else if ( (lIdx == sp.y) && (lIdx != ep.y) ){
		LineSelRange.x = sp.x;
		LineSelRange.y = lb.Len;
	}
	// last line partly selected
	else if ( (lIdx == ep.y) && (lIdx != sp.y) ){
		LineSelRange.x = 0;
		LineSelRange.y = ep.x;
	}
	// single line select
	else {
		LineSelRange.x = sp.x;
		LineSelRange.y = ep.x;
	}

	return LineSelRange;
}

int lineYPos( LineBuffer lb){
	int idx = Lines.indexOf( lb);

	if ( (idx < IdxTop) || ( idx > IdxTop + Vert.PageInc))
		return 0;

	return ((idx-IdxTop+1) * LineHeight + FBorder.Ext) - Descent;
}

public void lostFocus(){
	if ( (ResGraph != null) ) {
		setCursorClr( Awt.InactiveClr);     // why doing more to get less (loose cursor info)
//		blankCursor();

//		if ( hasSelection() )             // just required if we change colors
//			redrawLines( SelStart.y, SelEnd.y);
	}
	
	FocusState = 0;
}

int maxLineLen() {
	int maxLen = 0;
	for ( Enumeration e = Lines.elements(); e.hasMoreElements(); )
		maxLen = Math.max( maxLen, ((LineBuffer)e.nextElement()).Len);
	return maxLen;
}

public void mouse1Down( Event evt) {

	if ( hasSelection() && (FocusState < 2) ){ // check if we just re-entered the window
		FocusState = 2;                          // don't position the cursor in this case
		return;
	}

	LineBuffer lb = cursorLine();

	if ( lb != null) {
		updateCursorPos(lb);
		blankCursor();
		setCursorPos( evt.x, evt.y);
		updateSel( true);
		drawCursor();
		pushFormat( lb);
	}
	else
		reset();
}

public void mouse1Drag( Event evt) {
	int sIdx, eIdx, px, py;

	px = CursorIdx.x;
	py = CursorIdx.y;

	if ( evt.x < 0)
		scrollHorizontal( -10, true);
	else if ( evt.x > Width)
		scrollHorizontal( 10, true);

	if ( evt.y < 0)
		cursorUp( true);
	else if ( (evt.y > Height) && ( ! atEnd() ))
		cursorDown( true);
	else {
		setCursorPos( evt.x, evt.y);
		SelEnd.x = CursorIdx.x;
		SelEnd.y = CursorIdx.y;

		if (py != CursorIdx.y) {
			sIdx = Math.min( py, CursorIdx.y);
			eIdx = Math.max( py, CursorIdx.y);
			redrawLines( sIdx, eIdx);
		}
		else if ( px != CursorIdx.x)
			redrawLines( py, py);
	}
}

void newLine() {
	int	       lts = 0;
	LineBuffer lbn, lbp;

	lbp = cursorLine();
	lbn = new LineBuffer( Fm, lbp.Text, CursorIdx.x, lbp.Len, TabWidth);
	lbp.Len = CursorIdx.x;
	lts = lbp.leadingTabs();

	for ( int i = 0; i < lts; i++)
		lbn.insert( '\t', 0);

	CursorIdx.x = lts;
	CursorIdx.y++;
	TryCursorX = lts;

	Lines.insertElementAt( lbn, CursorIdx.y);

	updateSel( true);
	updateCursor( lbn);
	updateVScroll();
	redrawLines( CursorIdx.y - 1, -1);

	checkModified();
}

Point posOf( int cIdx) {
	if ( Lines == null)
		return new Point( 0, 0);

	int i, s = Lines.size();

	for ( i=0; i<s; i++){
		LineBuffer lb = (LineBuffer)Lines.elementAt( i);
		if ( cIdx-lb.Len <=0)
			break;
		cIdx -= lb.Len;
		cIdx--;            // '\n' ! (has to be consistent with getContents() length)
	}
	return new Point( cIdx, i);    
}

void  pushFormat( LineBuffer lb){
	Master.OsFormatLine.notifyObservers( lb);
}

void redrawLine( int idx) {
	int        yo = (idx-IdxTop) * LineHeight + YOffset;
	int	     	 bl = yo + LineHeight - Descent - 1;
	LineBuffer lb = (LineBuffer)Lines.elementAt( idx);
	Color      cf = getForeground();
	Color      cb = getBackground();

	Point sp = lineSelRange( idx, lb);

	if ( sp == null) {
		ResGraph.setColor( cb);
		ResGraph.fillRect( 0, yo, Width, LineHeight);
		ResGraph.setColor( cf);
		lb.draw( 0, lb.Len, ResGraph, XOffset , bl, XOffset);
	}
	else {
		int offs = XOffset;
		int dx   = 0;

		if ( sp.x > 0) {
			ResGraph.setColor( cb);
			dx = lb.charsWidth( 0, sp.x, offs - XOffset);
			ResGraph.fillRect( 0, yo, offs + dx, LineHeight);
			ResGraph.setColor( cf);
			lb.draw( 0, sp.x, ResGraph, offs, bl, XOffset);
			offs += dx;
		}

		ResGraph.setColor( Awt.SelBackClr);
		dx = lb.charsWidth( sp.x, sp.y - sp.x, offs - XOffset);
		ResGraph.fillRect( offs, yo, dx, LineHeight);
		ResGraph.setColor( Awt.SelForeClr);
		lb.draw( sp.x, sp.y - sp.x, ResGraph, offs, bl, XOffset);
		offs += dx;

		ResGraph.setColor( cb);
		ResGraph.fillRect( offs, yo, Width - offs, LineHeight);
		if ( sp.y < lb.Len) {
			ResGraph.setColor( cf);
			lb.draw( sp.y, lb.Len - sp.y, ResGraph, offs, bl, XOffset);
		}
	}

	if ( EventHandler.HasFocus && (idx == CursorIdx.y) )
		drawCursor();

	pushFormat( lb);
}

void removeLine( LineBuffer lb){
	Lines.removeElement( lb);
	updateVScroll();
}

boolean replaceSelectionWith( String s) {

	if ( !hasSelection() ) {
		insert( s);
		return false;
	}

	LineBuffer lb, llb = null;
	Vector dv = new Vector( 10);
	Point pt, ps = selStart();
	int   sl = selLines();

	for ( int idx = ps.y; idx <= ps.y + sl; idx++) {
		lb = (LineBuffer)Lines.elementAt( idx);
		pt = lineSelRange( idx, lb);

		if ( pt != null) {
			if ( (pt.x == 0) && ( pt.y == lb.Len) && ( idx != ps.y))
				dv.addElement( lb);
			else{
				llb = lb;
				lb.remove( pt.x, pt.y);
			}	
		}	
	}

	for ( Enumeration e = dv.elements(); e.hasMoreElements(); )
		removeLine( (LineBuffer)e.nextElement() );

	CursorIdx.x = ps.x;
	CursorIdx.y = ps.y;
	lb = cursorLine();

	if ( (llb != null) && (! lb.equals( llb)) ){
		lb.append( llb.Text, 0, llb.Len);
		removeLine( llb);
	}	
	updateCursor( lb );
	updateSel( false);

	updateVScroll();
	redrawLines( ps.y, -1);
	TryCursorX = CursorIdx.x;

	checkModified();

	if ( s != null)
		insert( s);

	return true;
}

boolean replaceSelectionWith( char c) {
	return replaceSelectionWith( String.valueOf( c));
}

void reset() {    
	LineHeight  = (int)(LineSpacing * Fm.getHeight() + 1);
	Cursor.XPos = 0;
	Cursor.YPos = LineHeight;
	CursorIdx.x = 0;
	CursorIdx.y = 0;
	TryCursorX  = 0;

	updateSel( true);
	addLine( new LineBuffer(Fm) );

	positionVertical( 0, true);
	positionHorizontal( 0, true);

	drawCursor();
}

Point selEnd(){
	Point p = selStart();
	if ( p == SelStart)
		return SelEnd;
	return SelStart;
}

int selLines() {
	return Math.abs( SelStart.y - SelEnd.y);
}

Point selStart(){		
	if ( SelStart.y < SelEnd.y)
		return SelStart;
	if ( SelStart.y == SelEnd.y)
		return (SelStart.x < SelEnd.x) ? SelStart : SelEnd;

	return SelEnd;
}

void selectAll(){
	LineBuffer lb;

	if ( ! isEmpty() ){
		lb = (LineBuffer)Lines.lastElement();
		setSelection( 0, 0, lb.Len, Lines.size() - 1);
	}
}

void selectLine( int lineY) {
	setSelection( 0, lineY, -1, lineY);
}

int setContents( String s, boolean add) {

	Modified = false;

	if ( s == null || s.length() == 0 ){
		Lines.removeAllElements();
		reset();
		if ( ResGraph != null)
			blank( ResGraph);
		return 0;
	}

	if ( ! add) {
		reset();
		Lines.removeAllElements();
	}	

	return addContents( s);
}

void setCursorClr( Color cc) {
	Cursor.setForeground( cc);
	
//	if ( (ResGraph != null) && EventHandler.HasFocus)
	if ( ResGraph != null )
		Cursor.drawAtOffs( ResGraph, XOffset, YOffset);
}

boolean setCursorIdx ( int x, int y){
	LineBuffer lb;

	if ( (isEmpty()) || ( ResGraph == null ) )
		return false;

	y = Math.min( y, Lines.size() - 1);
	lb = (LineBuffer)Lines.elementAt( y);
	x = Math.min( x, lb.Len);

	blankCursor();
	TryCursorX = CursorIdx.x = x;
	CursorIdx.y = y;
	updateCursor( lb);
	updateSel( true);
	drawCursor();

	return true;
}

LineBuffer setCursorPos( int x, int y) {
	LineBuffer lb;
	int        idx, dx, lPos = LineHeight;

	for ( idx=IdxTop; idx < Lines.size() - 1 ; idx++) {
		if ( y < lPos)
			break;
		lPos += LineHeight;
	}

	CursorIdx.y = idx;
	Cursor.YPos = lPos;
	lb = cursorLine();

	lPos = XOffset ;
	for( idx=0; idx < lb.Len; idx++) {
		dx = lb.charsWidth( idx, 1, lPos - XOffset);
		if ( lPos + dx > x)
			break;
		lPos += dx;
	}
	TryCursorX = CursorIdx.x = idx;
	Cursor.XPos = lPos - XOffset;

	return lb;
}

void setCursorWidth( int w) {
	if ( (ResGraph != null) && EventHandler.HasFocus)
		Cursor.clearAtOffs( ResGraph, XOffset, YOffset );
	Cursor.setWidth( w);
}

public synchronized void setFont( Font fnt) {
	Font uf = ( fnt != null) ? fnt : Awt.FixedFont;

	super.setFont( uf);

	if ( uf.equals( Awt.FixedFont))
		Fm = DefFm;
	else
		Fm = Awt.getFontMetrics( uf);

	Cursor.setHeight( Fm.getHeight());

	for ( Enumeration e = Lines.elements(); e.hasMoreElements(); )
		((LineBuffer)e.nextElement()).setMetrics( Fm);

	LineHeight = (int)(LineSpacing * Fm.getHeight() + 1 );
	LineWidth  = 200 * Fm.charWidth( 'x');
	Descent = Fm.getMaxDecent();

	Horz.setLineIncrement(Fm.charWidth( 'x'));
	Horz.setMinMax( 0, LineWidth);

	Vert.setStepPels( LineHeight);

	if ( ResGraph != null )
		redraw();
}

void setModified( boolean state){
	if ( state)  checkModified();
	else         Modified = false;
}

void setSelection( int startX, int startY, int endX, int endY) {

	endY = Math.min( endY, Lines.size() - 1);
	if ( endX == -1)
		endX = ((LineBuffer)Lines.elementAt( endY)).Len;

	blankCursor();
	showArea( startY, endY);
	showRange( startX, startY, endX, endY);

	CursorIdx.x = endX;
	CursorIdx.y = endY;
	updateSel( true);

	SelStart.x	= startX;
	SelStart.y	= startY;
	SelEnd.x	= endX;
	SelEnd.y	= endY;

	updateCursor( null);

	redrawLines( SelStart.y, SelEnd.y);
}

void setTabWidth( int numChars) {
	TabWidth = numChars;

	for ( Enumeration e = Lines.elements(); e.hasMoreElements(); )
		((LineBuffer)e.nextElement()).setTabWidth( numChars);

	if ( ResGraph != null )
		redraw();
}

void setValues( int xBorder, double lineSpacing, Font fnt, Color cb, Color cf){
	XBorder	= (xBorder > -1) ? xBorder : 3;
	LineSpacing	= (lineSpacing > 0) ? lineSpacing : 1;

	setForeground( cf);
	setBackground( cb);
	Cursor.setBackground( cb);

	setFont( fnt);
}

void showRange( int startX, int startY,
                int endX, int endY){
	LineBuffer lb = (LineBuffer)Lines.elementAt( endY);
	int        maxX, minX;

	maxX = lb.charsWidth( 0, endX, 0);
	minX = lb.charsWidth( 0, startX, 0);

	if ( maxX + XOffset >= Width){
		positionHorizontal( maxX - Width + 20, true);
	}
	else if ( minX + XOffset < 0){
		positionHorizontal( minX, true);
	}
}

void updateContents ( String s) {
	Lines.removeAllElements();
	addContents( s);
}

void updateCursor( LineBuffer lb){

	if ( lb == null)
		lb = (LineBuffer)Lines.elementAt( CursorIdx.y);

	Cursor.XPos = lb.charsWidth( 0, CursorIdx.x, 0);
	showArea( CursorIdx.y, CursorIdx.y);
	Cursor.YPos = (CursorIdx.y + 1 - IdxTop) * LineHeight;

	if ( Cursor.XPos + XOffset  >= Width)
		positionHorizontal( Cursor.XPos - Width + 20, true);
	else if ( Cursor.XPos + XOffset < 0)
		positionHorizontal( Cursor.XPos, true);

	pushFormat( lb);
}

void updateCursorPos( LineBuffer lb){
	Cursor.XPos = lb.charsWidth( 0, CursorIdx.x, 0);
	Cursor.YPos = (CursorIdx.y + 1 - IdxTop) * LineHeight;
}

void updateScrollImage( int sIdx) {
	int			 bl = LineHeight - Descent - 1;
	Graphics g = SoftGraph;

	g.setFont( getFont() );
	g.setColor( getBackground() );
	g.fillRect( 0, 0, SoftSize.width, SoftSize.height);
	g.setColor( getForeground() );

	for ( int i=0; i<2; i++) {
		try {
			LineBuffer lb = (LineBuffer)Lines.elementAt( sIdx + i);
			lb.draw( 0, lb.Len, g, XOffset , bl, XOffset);
		}
		catch ( Throwable t) {}
		bl += LineHeight;
	}
}

void updateSel( boolean refresh) {
	int ul = -1;
	int us = -1;

	if ( refresh && hasSelection()) {
		us = selStart().y;
		ul = selLines();
	}

	SelStart.x = SelEnd.x = CursorIdx.x;
	SelStart.y = SelEnd.y = CursorIdx.y;

	if ( us != -1)
		redrawLines( us, us + ul);
}

public boolean vKeyInput( int k, int mod) {

	if ( mod == 0){
		switch ( k) {
		case Key.Tab:
			charInput( '\t', 0);
			return true;
		case Key.Escape:
			updateSel( true);
			return true;
		case Key.F12:
			selectAll();
			return true;
		case Key.Backspace:
			backSpace();
			return true;
		case Key.Newline:
			newLine();
			return true;
		case Key.Delete:
			del();
			return true;
		case Key.PageUp:
			Vert.pageDown();
			return true;
		case Key.PageDown:
			Vert.pageUp();
			return true;
		}
	}

	if ( mod < 2){
		boolean sd = (mod == 1);
		switch( k){
		case Key.CursorUp:
			cursorUp( sd);
			return true;
		case Key.CursorDown:
			cursorDown( sd);
			return true;
		case Key.CursorLeft:
			cursorLeft( sd);
			return true;
		case Key.CursorRight:
			cursorRight( sd);
			return true;
		case Key.Home:
			cursorHome( sd);
			return true;
		case Key.End:
			cursorEnd( sd);
			return true;
		}
	}

	return false;
}

void vPosChange (){
	Cursor.YPos = (CursorIdx.y + 1 - IdxTop) * LineHeight;
}
}
