;* --------------------------------------------------------------------*/
;*    Copyright (c) 1992-1998 by Manuel Serrano. All rights reserved.  */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \   /  '                               */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome. Send them to                                          */
;*        Manuel Serrano -- Manuel.Serrano@unice.fr                    */
;*-------------------------------------------------------------------- */
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/comptime/Module/module.scm           */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Fri May 31 10:29:03 1996                          */
;*    Last change :  Sat Feb 28 10:25:08 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The compilation of a Module clause                               */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module module_module

   (include "Ast/unit.sch"
	    "Engine/pass.sch")
   
   (import  engine_param
	    engine_pass
	    tools_error
	    module_main
	    module_statexp
	    module_impuse
	    module_include
	    module_with
	    module_type
	    module_foreign
	    module_eval
	    module_load
	    module_pragma
	    module_checksum
	    module_option)
   
   (export  (class ccomp::object
	       ;; the compiler identifier
	       (id::symbol read-only)
	       ;; the compilation method when compiling the module
	       ;; which contains this clause
	       (producer::procedure read-only (default (lambda (c) '())))
	       ;; what to do when importing a module which contains
	       ;; this clause
	       (consumer::procedure read-only (default (lambda (m c) '()))) 
	       ;; what to do after the overall module compilation
	       (finalizer::procedure read-only (default (lambda () 'void)))
	       ;; we do so module checksumming in order to ensure
	       ;; modules coherence at initialization time
	       (checksummer::procedure read-only (default (lambda (m c) c))))
	   
	    (produce-module!                    <module-clause>)
	    (produce-module-clause!             <clause>)
	    (consume-module!                    name::symbol <module-clause>)
	    (consume-module-clause!             name::symbol <clause>)
	    (checksum-module                    <clause>)
	    
	    (module-initialization-id::symbol ::symbol)

	    *main*             
	    *module*
	    *module-clause*
	    *module-checksum*))

;*---------------------------------------------------------------------*/
;*    Informations about the module being compiled                     */
;*---------------------------------------------------------------------*/
(define *module*               #f)    ;; module identifer
(define *module-clause*        #f)    ;; module clause (for pretty-printing)
(define *main*                 #f)    ;; the main entry point identifier
(define *module-checksum*      #f)    ;; the checksum of the current module

;*---------------------------------------------------------------------*/
;*    produce-module! ...                                              */
;*---------------------------------------------------------------------*/
(define (produce-module! mclause)
   (pass-prelude "Module")
   (match-case mclause
      ((module (and (? symbol?) ?name) . ?clauses)
       (let ((clauses (if (symbol? *main*)
			  (cons `(main ,*main*) clauses)
			  clauses)))
	  (if (not (legal-module-name? name))
	      (user-error "Parse error"
			  "Illegal module name"
			  mclause)
	      (begin
		 (set! *module* name)
		 (set! *module-clause* mclause)
		 (install-clauses-compiler!)
		 (for-each produce-module-clause! clauses)
		 (set! *module-checksum* (checksum-module mclause))
		 (pass-postlude (finalize-clause-compilations))))))
      (else
       (user-error "Parse error"
		   "Illegal module declaration"
		   mclause))))

;*---------------------------------------------------------------------*/
;*    finalize-clause-compilations ...                                 */
;*---------------------------------------------------------------------*/
(define (finalize-clause-compilations)
   (let loop ((cc    *clause-compilers*)
	      (units '()))
      (if (null? cc)
	  units
	  (let* ((finalizer (ccomp-finalizer (car cc)))
		 (finalres  (finalizer)))
	     (loop (cdr cc)
		   (if (pair? finalres)
		       (append finalres units)
		       units))))))

;*---------------------------------------------------------------------*/
;*    produce-module-clause! ...                                       */
;*---------------------------------------------------------------------*/
(define (produce-module-clause! clause)
   (match-case clause
      (((and (? symbol?) ?id) . ?-)
       ((find-clause-producer id clause) clause))
      (else
       (user-error "Parse error"
		   "Illegal module clause"
		   clause
		   '()))))
	     
;*---------------------------------------------------------------------*/
;*    *clause-compilers* ...                                           */
;*---------------------------------------------------------------------*/
(define *clause-compilers* '())

;*---------------------------------------------------------------------*/
;*    install-clauses-compiler ...                                     */
;*---------------------------------------------------------------------*/
(define (install-clauses-compiler!)
   ;; the order of the compiler is important. Don't change it
   (set! *clause-compilers*
	 (list (make-eval-compiler)
	       (make-main-compiler)
	       (make-load-compiler)
	       (make-use-compiler)
	       (make-import-compiler)
	       (make-from-compiler)
	       (make-static-compiler)
	       (make-export-compiler)
	       (make-include-compiler)
	       (make-with-compiler)
	       (make-foreign-compiler)
	       (make-extern-compiler)
	       (make-type-compiler)
	       (make-pragma-compiler)
	       (make-option-compiler)
	       )))

;*---------------------------------------------------------------------*/
;*    find-clause-producer ...                                         */
;*---------------------------------------------------------------------*/
(define (find-clause-producer keyword clause)
   (define (unknown-clause-producer values)
      (user-error "Parse error"
		  "Unknown module clause"
		  clause
		  '()))
   (let loop ((cc *clause-compilers*))
      (cond
	 ((null? cc)
	  unknown-clause-producer)
	 ((eq? (ccomp-id (car cc)) keyword)
	  (ccomp-producer (car cc)))
	 (else
	  (loop (cdr cc))))))

;*---------------------------------------------------------------------*/
;*    module-initialization-id ...                                     */
;*---------------------------------------------------------------------*/
(define (module-initialization-id id)
   'module-initialization)

;*---------------------------------------------------------------------*/
;*    consume-module! ...                                              */
;*---------------------------------------------------------------------*/
(define (consume-module! pname mclause)
   (match-case mclause
      ((module (and (? symbol?) ?name) . ?clauses)
       (cond
	((not (legal-module-name? name))
	 (user-error "Parse error" "Illegal module name" mclause '()))
	((not (eq? pname name))
	 (user-error "Module declaration"
		     (string-append "conflict in module's name: "
				    (symbol->string name) " vs "
				    (symbol->string pname))
		     mclause
		     '()))
	(else
	 (apply append (map (lambda (c) (consume-module-clause! name c))
			    clauses)))))
      (else
       (user-error "Parse error"
		   "Illegal module declaration"
		   mclause
		   '()))))

;*---------------------------------------------------------------------*/
;*    consume-module-clause! ...                                       */
;*---------------------------------------------------------------------*/
(define (consume-module-clause! module clause)
   (match-case clause
      (((and (? symbol?) ?id) . ?values)
       ((find-clause-consumer id clause) module clause))
      (else
       (user-error "Parse error"
		   "Illegal module clause"
		   clause
		   '()))))
	     
;*---------------------------------------------------------------------*/
;*    find-clause-consumer ...                                         */
;*---------------------------------------------------------------------*/
(define (find-clause-consumer keyword clause)
   (define (unknown-clause-consumer module values)
      (user-error "Parse error"
		  "Unknown module clause"
		  clause
		  '()))
   (let loop ((cc *clause-compilers*))
      (cond
	 ((null? cc)
	  unknown-clause-consumer)
	 ((eq? (ccomp-id (car cc)) keyword)
	  (ccomp-consumer (car cc)))
	 (else
	  (loop (cdr cc))))))

;*---------------------------------------------------------------------*/
;*    legal-module-name? ...                                           */
;*---------------------------------------------------------------------*/
(define (legal-module-name? name)
   (not (memq name '(eval foreign))))

;*---------------------------------------------------------------------*/
;*    checksum-module ...                                              */
;*---------------------------------------------------------------------*/
(define (checksum-module mclause)
   (if *unsafe-version*
       0
       (match-case mclause
	  ((module (and (? symbol?) ?name) . ?clauses)
	   (cond
	      ((not (legal-module-name? name))
	       (user-error "Parse error" "Illegal module name" mclause '()))
	      (else
	       (let loop ((clauses  clauses)
			  (checksum (initial-checksum name)))
		  (if (null? clauses)
		      checksum
		      (loop (cdr clauses)
			    (checksum-module-clause (car clauses)
						    checksum)))))))
	  (else
	   (user-error "Parse error"
		       "Illegal module declaration"
		       mclause
		       '())))))

;*---------------------------------------------------------------------*/
;*    checksum-module-clause ...                                       */
;*---------------------------------------------------------------------*/
(define (checksum-module-clause clause checksum::long)
   (match-case clause
      (((and (? symbol?) ?id) . ?-)
       ((find-clause-checksummer id clause) clause checksum))
      (else
       (user-error "Parse error"
		   "Illegal module clause"
		   clause
		   '()))))
	     
;*---------------------------------------------------------------------*/
;*    find-clause-checksummer ...                                      */
;*---------------------------------------------------------------------*/
(define (find-clause-checksummer keyword clause)
   (define (unknown-clause-checksummer module values)
      (user-error "Parse error"
		  "Unknown module clause"
		  clause
		  '()))
   (let loop ((cc *clause-compilers*))
      (cond
	 ((null? cc)
	  unknown-clause-checksummer)
	 ((eq? (ccomp-id (car cc)) keyword)
	  (ccomp-checksummer (car cc)))
	 (else
	  (loop (cdr cc))))))

