;* --------------------------------------------------------------------*/
;*    Copyright (c) 1992-1998 by Manuel Serrano. All rights reserved.  */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \   /  '                               */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome. Send them to                                          */
;*        Manuel Serrano -- Manuel.Serrano@unice.fr                    */
;*-------------------------------------------------------------------- */
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/comptime/Engine/compiler.scm         */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Fri May 31 08:22:54 1996                          */
;*    Last change :  Sat Feb 28 18:43:37 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The compiler driver                                              */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module engine_compiler
   (include "Ast/unit.sch"
	    "Engine/pass.sch")
   (import  tools_error
	    engine_pass
	    engine_signals
	    engine_param
	    engine_engine
	    read_src
	    write_expanded
	    write_ast
	    read_access
	    heap_restore
	    heap_make
	    ast_env
	    type_type
	    ast_var
	    ast_node
	    ast_build
	    ast_unit
	    user_user
	    type_env
	    type_cache
	    module_module
	    module_include
	    expand_syntax-case
	    expand_eps
	    expand_install
	    init_main
	    trace_walk
	    inline_walk
	    effect_walk
	    callcc_walk
	    fail_walk
	    globalize_walk
	    cfa_walk
	    cfa_tvector
	    integrate_walk
	    coerce_walk
	    reduce_walk
	    cnst_walk
	    cgen_walk
	    hgen_walk
	    cc_indent
	    cc_cc
	    cc_ld)
   (export  (compiler)))

;*---------------------------------------------------------------------*/
;*    compiler ...                                                     */
;*---------------------------------------------------------------------*/
(define (compiler)
   
   ;; we catch signals
   (install-compiler-signals!)
   
   ;; we read the source file
   (let ((src (read-src)))
      ;; if src is false, we launch the interpreter because it means
      ;; that the reader has found a #!... expression instead of a
      ;; module clause
      (cond
	 ((not src)
	  (set! *interpreter* #t)
	  (exit-bigloo (engine)))
	 ((not (pair? src))
	  (user-error "Parse error" "Illegal source file" src)))
      
      ;; now (and only now) we can say hello
      (hello-world)
      
      ;; we check now if we have parsed all argument
      (if (not (null? *rest-args*))
          (warning "Don't know what to do with arguments: " *rest-args*))
      
      ;; we read access file
      (read-access-file)
      
      ;; we create (or restore) the compilation environment
      (if *lib-mode*
	  (begin
	     (initialize-Genv!)
	     (initialize-Tenv!))
	  (restore-heap))

      ;; we initialized the type caching system
      (install-type-cache!)
      
      ;; when the vector->tvcector optimization is enable we have to
      ;; patch the types of vector-set! familly function.
      (patch-vector-set!)
      
      ;; we compile the module clause which leads to the
      ;; complete source code.
      (let* ((module   (car src))
	     (src-code (cdr src))
	     (units    (produce-module! module)))
	 (unit-sexp*-add! (get-toplevel-unit)
			  (if (null? src-code) '(#unspecified) src-code))
	 ;; we check error occured while building the ast
	 (pass-postlude #unspecified)
	 
	 ;; we check if all types are defined
	 (check-types)
	 
	 ;; C header generation
	 (stop-on-pass 'hgen hgen-walk)
					    
	 ;; we perfom user pass
	 (user-walk units)
	 (stop-on-pass 'user (lambda () (write-expanded units)))

	 ;; when compiling in hygien mode, we perform the r5rs expansion
	 (if *hygien?*
	     (syntax-expand-units units))
	 (stop-on-pass 'syntax (lambda () (write-expanded units)))
	 
	 ;; we install macros ...
	 (install-initial-expander)
	 ;; ... and we macro expand
	 (expand-units units)
	 (stop-on-pass 'expand (lambda () (write-expanded units)))
	 
	 ;; ok, now we build the ast
	 (let ((ast (build-ast units)))
	    (stop-on-pass 'ast (lambda () (write-ast ast)))
	    
	    ;; we make a heap on `mkheap' mode
	    (stop-on-pass 'make-heap (lambda () (make-heap)))
	    
	    ;; when the compiler is invoked in -g3 mode, we install
	    ;; traces before the inlining
	    (if (and (integer? *compiler-debug*) (>fx *compiler-debug* 2))
		(set! ast (trace-walk! ast)))
	    (stop-on-pass 'trace (lambda () (write-ast ast)))
	    
	    ;; when we are compiling with call/cc we have to
	    ;; put all written local variables in cells
	    (if *call/cc?*
		(set! ast (callcc-walk! ast)))
	    (stop-on-pass 'callcc (lambda () (write-ast ast)))
	    
	    ;; the effect property computation
	    (if *optim-unroll-loop?*
		(begin
		   (set! ast (effect-walk! ast))
		   (stop-on-pass 'effect (lambda () (write-ast ast)))))
	    
	    ;; we perform the inlining pass
	    (set! ast (inline-walk! ast 'all))
	    (stop-on-pass 'inline (lambda () (write-ast ast)))
	    
	    ;; we introduce traces in `small debug mode'
	    (if (and (integer? *compiler-debug*)
		     (>fx *compiler-debug* 0) (<=fx *compiler-debug* 2))
		(set! ast (trace-walk! ast)))
	    
	    ;; we replace `failure' invokation by `error/location' when
	    ;; invoked in debug mode (to be performed after the coercion stage)
	    (if (and (integer? *compiler-debug*) (>fx *compiler-debug* 0))
		(set! ast (fail-walk! ast)))
	    (stop-on-pass 'fail  (lambda () (write-ast ast)))
	    
	    ;; the globalization stage
	    (set! ast (globalize-walk! ast 'globalization))
	    (stop-on-pass 'globalize (lambda () (write-ast ast)))

	    ;; we perform a type recovery pass (using predicate-of pragma)
;* 	    (set! ast (recovery-walk! ast))                            */
;* 	    (stop-on-pass 'recovery (lambda () (write-ast ast)))       */
	    
	    ;; the control flow analysis
	    (set! ast (cfa-walk! ast))
	    (stop-on-pass 'cfa (lambda () (write-ast ast)))

	    ;; now we have done the cfa, type election has been performed
	    ;; we change the default type from *_* to *obj*.
	    (set-default-type! *obj*)
	    
	    ;; the integration pass
	    (set! ast (integrate-walk! ast))
	    (stop-on-pass 'integrate (lambda () (write-ast ast)))

	    ;; we introduce type coercion and checking
	    (set! ast (coerce-walk! ast))
	    (stop-on-pass 'coerce (lambda () (write-ast ast)))

	    ;; we re-run the effect computations (for coercion and
	    ;; type checks)
	    (if (>=fx *optim* 2)
		(begin
		   (set! ast (effect-walk! ast))
		   (stop-on-pass 'effect (lambda () (write-ast ast)))))
	    
	    ;; the reduction optimizations
	    (if (>=fx *optim* 1)
		(set! ast (reduce-walk! ast)))
	    (stop-on-pass 'reduce (lambda () (write-ast ast)))
	 
	    ;; the constant computation
	    (set! ast (cnst-walk! ast)) 
	    (stop-on-pass 'cnst (lambda () (write-ast ast)))
	    
	    ;; we re-perform the inlining pass in high optimization mode
	    ;; in order to inline all type checkers.
	    (if (>=fx *optim* 3)
		(set! ast (inline-walk! ast 'reducer)))
	    (stop-on-pass 'inline+ (lambda () (write-ast ast)))
	    
	    ;; the C generation
	    (let ((c-prefix (cgen-walk ast)))
	       (stop-on-pass 'cgen (lambda () 'done))
	       (stop-on-pass 'distrib (lambda () 'done))

	       (if (string? c-prefix)
		   (begin
		      ;; we indent the resulting C file (if wanted)
		      (if (or (eq? *pass* 'cindent) *c-debug*)
			  (indent c-prefix))
		      (stop-on-pass 'cindent (lambda () 'done))
		      
		      ;; we invoke now the C compiler
		      (cc c-prefix (not (eq? *pass* 'cc)))
		      (stop-on-pass 'cc (lambda () 'done))
		      
		      ;; and the linker
		      (ld c-prefix #t)))
	       
	       0)))))

