/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		main.cpp

	Contains:	Simple Rhapsody main function to drive TimeShare.

	$NoKeywords: $
	
	$Log: main.cpp,v $
	Revision 1.2  1999/02/19 23:15:47  ds
	Created
	

*/

#include <stdio.h>
#include <stdlib.h>

#ifndef __MW_
#if __MacOSX__
#include <mach/mach.h>
#include <mach/cthreads.h>
#else
#include <pthread.h>
#endif
#include <sys/time.h>
#include <sys/resource.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>
#include <sys/wait.h>
#include "ev.h"
#endif

#include "RTPServer.h"
#include "RTSPPrefs.h"
#include "Socket.h"
#include "OS.h"
#include "MyAssert.h"

#include "QTRTPFile.h"
#include "SocketUtils.h"
#include "StreamDictionary.h"

#include "PrefsSource.h"
#include "FilePrefsSource.h"
#if __MacOSX__
#include "NetInfoPrefsSource.h"
#endif

void sigcatcher(int sig, int /*sinfo*/, struct sigcontext* /*sctxt*/);

void sigcatcher(int sig, int /*sinfo*/, struct sigcontext* /*sctxt*/)
{
#if DEBUG
	printf("Signal %d caught\n", sig);
#endif
}

//This function runs the server
void RunServer(RTSPPrefs* inPrefs, UInt16 inPortOverride);


int main(int argc, char * argv[]) 
{
	extern char* optarg;
	
	// on write, don't send signal for SIGPIPE, just set errno to EPIPE
	// and return -1
	//signal is a deprecated and potentially dangerous function
	//(void) ::signal(SIGPIPE, SIG_IGN);
	struct sigaction act;
	
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	act.sa_handler = (void(*)(int))&sigcatcher;
	
	// old code
//	act.sa_mask = 0;
//	act.sa_flags = 0;
//	act.sa_handler = (void(*)(...))&sigcatcher;
	(void)::sigaction(SIGPIPE, &act, NULL);

	//grow our pool of file descriptors to the max!
	struct rlimit rl;
               
	rl.rlim_cur = Socket::kMaxNumSockets;
	rl.rlim_max = Socket::kMaxNumSockets;

	setrlimit (RLIMIT_NOFILE, &rl);
	
	//First thing to do is to read command-line arguments and initialize the pipe
	//file descriptor. This is used for returning errors back to whatever is starting
	//the process in the event of a fatal error very early in the startup process
	int thePipe = -1; 
	int ch;
	UInt16 thePort = 0; //port can be set on the command line
	
	bool dontFork = false;
	bool logTimestamps = false;
	bool appendRandomOffsets = true;
	bool keyframesonly = false;
	bool nobframes = false;
	char* theConfigFilePath = NULL;
#ifndef __MacOSX__
	static char* sDefaultConfigFilePath = "/etc/QTSS.conf";
	//Other platforms don't have the option of using NetInfo for preferences,
	//so by default the server will grab prefs from a config file
	theConfigFilePath = sDefaultConfigFilePath;
#endif
	
	while ((ch = getopt(argc,argv, "vdlrZbkp:c:")) != EOF)
	{
		switch(ch)
		{
			case 'v':
				printf("%s/%s Built on: %s\n", RTSPPrefs::GetServerName().Ptr,
						RTSPPrefs::GetServerVersion().Ptr, RTSPPrefs::GetServerBuildDate().Ptr);
				printf("-d: Run in the foreground\n");
				printf("-l: For each RTP stream, write a log of all timestamps & sequence numbers\n");
				printf("-r: Don't append any random offsets to timestamps & sequence numbers\n");
				printf("-k: Send key frames only\n");
				printf("-b: Always drop b-frames\n");
				printf("-p XXX: Specify the default RTSP listening port of the server\n");
				printf("-c /myconfigpath.conf: Specify a config file\n");
				exit(0);	
			case 'Z':
				//this option is passed in by server control lib when it
				//starts the server. Immediately following is a path to the piped file descriptor
				thePipe = open(argv[2], O_WRONLY);
				break;
			case 'd':
				dontFork = true;
				break;
			case 'l':
				logTimestamps = true;
				break;
			case 'r':
				appendRandomOffsets = false;
				break;
			case 'p':
				thePort = ::atoi(optarg);
				break;
			case 'k':
				keyframesonly = true;
				break;
			case 'b':
				nobframes = true;
				break;
			case 'c':
				theConfigFilePath = optarg;
				break;
			default:
				break;
		}
	}
	
	//Unless the command line option is set, fork & daemonize the process at this point
	if (!dontFork)
	{
		if (daemon(0,0) != 0)
		{
#if DEBUG
			printf("Failed to daemonize process. Error = %d\n", cthread_errno());
#endif
			exit(-1);
		}
	}

	//Construct a Prefs Source object to get server preferences
	PrefsSource* thePrefsSource = NULL;
	if (theConfigFilePath != NULL)
		thePrefsSource = new FilePrefsSource(theConfigFilePath);
#if __MacOSX__
	else
		thePrefsSource = new NetInfoPrefsSource();
#endif
	Assert(thePrefsSource != NULL);//we must have one in order to function
	
	//check if we should do auto restart. If so, we should fork the process at this point,
	//have the child run the server. Parent will just wait for the child to die.
	char autoStartSetting[256] = "";
	(void)thePrefsSource->GetValue( "auto_restart", autoStartSetting);
	
	int statloc = 0;
	if ((!dontFork) && (::strcmp(autoStartSetting, "enabled") == 0))
	{
		//loop until the server exits normally. If the server doesn't exit
		//normally, then restart it.
		while (true)
		{
			pid_t processID = fork();
			Assert(processID >= 0);
			if (processID > 0)
			{
				::wait(&statloc);
				
#if DEBUG
				::printf("Child Process %d exited with status %d\n", processID, statloc);
#endif
				if (statloc == 0)
					return 0;
			}
			else if (processID == 0)
				break;
			else
				exit(-1);
			//eek. If you auto-restart too fast, you might start the new one before the OS has
			//cleaned up from the old one, resulting in startup errors when you create the new
			//one. Waiting for a second seems to work
#ifdef __MacOSX__
			thread_switch(THREAD_NULL, SWITCH_OPTION_WAIT, 1000);
#else
			sleep(1);
#endif
		}
	}
	
	//we have to do this again for the child process, because sigaction states
	//do not span multiple processes.
	(void)::sigaction(SIGPIPE, &act, NULL);

				
	//cause prefs to initialize
	//RTSPPrefs* prefs = new ('Prfs') RTSPPrefs(thePort);
	RTSPPrefs prefs(thePrefsSource, logTimestamps, appendRandomOffsets, keyframesonly, nobframes);
				
	//This function starts, runs, and shuts down the server
	RunServer(&prefs, thePort);
}

void RunServer(RTSPPrefs* inPrefs, UInt16 inPortOverride)
{
	//Mark when we are done starting up. If auto-restart is enabled, we want to make sure
	//to always exit with a status of 0 if we encountered a problem WHILE STARTING UP. This
	//will prevent infinite-auto-restart-loop type problems
	bool doneStartingUp = false;
	
	//Initialize utility classes
	OS::Initialize();
	SocketUtils::Initialize();
	StreamDictionary::Initialize();
	Socket::Initialize();

	const UInt32 kNumTaskThreads = 1;
	TaskThreadPool::AddThreads(kNumTaskThreads);
#if DEBUG
	printf("Number of task threads: %lu\n",kNumTaskThreads);
#endif
	IdleTask::Initialize();
	TimeoutTask::Initialize();
	QTRTPFile::Initialize();
#if !__MacOSX__
	::startevents();//initialize the select() implementation of the event queue
#endif
	
	//start the server
	RTPServer* server = new ('RTPs') RTPServer();
	server->Initialize(inPrefs, inPortOverride);

	//Make sure to do this stuff last. Because these are all the threads that
	//do work in the server, this ensures that no work can go on while the server
	//is in the process of staring up
	if (RTPServerInterface::GetServerState() != RTPServerInterface::kFatalErrorState)
		Socket::StartThread();

	if (RTPServerInterface::GetServerState() != RTPServer::kFatalErrorState)
	{
		doneStartingUp = true;
		printf("TimeShare done starting up\n");
		OS::SetMemoryError(QTSS_MemAllocFailed);
	}
	
	//just wait until someone stops the server or a fatal error occurs.
	UInt32 theServerState = RTPServerInterface::GetServerState();
	while ((theServerState != RTPServerInterface::kShuttingDownState) &&
			(theServerState != RTPServerInterface::kFatalErrorState))
	{
#ifdef __MacOSX__
		thread_switch(THREAD_NULL, SWITCH_OPTION_WAIT, 1000);
#else
		sleep(1);
#endif
		theServerState = RTPServerInterface::GetServerState();
	}
		
	//first off, make sure that the server can't do any work
	TaskThreadPool::RemoveThreads();
	
	//now that the server is definitely stopped, it is safe to initate
	//the shutdown process
	delete server;
	
	//ok, we're ready to exit. If we're quitting because of some fatal error
	//while running the server, make sure to let the parent process know by
	//exiting with a nonzero status. Otherwise, exit with a 0 status
	if ((doneStartingUp) && (theServerState == RTPServerInterface::kFatalErrorState))
		exit (-1);//signals parent process to restart server
}


#if __MacOSX__
void terminate() {}  
#endif



