/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		Task.cpp

	Contains:	implements Task class
					
	$Log: Task.cpp,v $
	Revision 1.2  1999/02/19 23:13:22  ds
	Created
	

*/

#include "Task.h"
#include "OS.h"
#include "atomic.h"

unsigned int	Task::sThreadPicker = 0;

Task::Task()
: 	fEvents(0), fTimerHeapElem(this), fTaskQueueElem(this)
{
#if DEBUG
	fInRunCount = 0;
#endif
	
}

Task::EventFlags Task::GetEvents()
{
	//Mask off every event currently in the mask except for the alive bit, of course,
	//which should remain unaffected and unreported by this call.
	EventFlags events = fEvents & kAliveOff;
	(void)atomic_sub(&fEvents, events);
	return events;
}

void Task::Signal(EventFlags events)
{
	//Fancy no mutex implementation. We atomically mask the new events into
	//the event mask. Because atomic_or returns the old state of the mask,
	//we only schedule this task once.
	events |= kAlive;
	EventFlags oldEvents = atomic_or(&fEvents, events);
	if ((!(oldEvents & kAlive)) && (TaskThreadPool::sNumTaskThreads > 0))
	{
		//find a thread to put this task on
		unsigned int theThread = atomic_add(&sThreadPicker, 1);
		theThread %= TaskThreadPool::sNumTaskThreads;
		TaskThreadPool::sTaskThreadArray[theThread]->fTaskQueue.EnQueue(&fTaskQueueElem);
	}
}

#pragma mark __TaskThread__


void TaskThread::Entry()
{
	Task* theTask = NULL;
	
	while (true) 
	{
		theTask = this->WaitForTask();
		Assert(theTask != NULL);
		
		bool doneProcessingEvent = false;
		
		while (!doneProcessingEvent)
		{
			//If a task holds locks when it returns from its Run function,
			//that would be catastrophic and certainly lead to a deadlock
#if DEBUG
			Assert(this->GetNumLocksHeld() == 0);
			Assert(theTask->fInRunCount == 0);
			theTask->fInRunCount++;
#endif
			SInt64 theTimeout = theTask->Run();
#if DEBUG
			Assert(this->GetNumLocksHeld() == 0);
			theTask->fInRunCount--;
			Assert(theTask->fInRunCount == 0);
#endif			
			if (theTimeout < 0)
			{
				delete theTask;
				theTask = NULL;
				doneProcessingEvent = true;
			}
			else if (theTimeout == 0)
			{
				//We want to make sure that 100% definitely the task's Run function WILL
				//be invoked when another thread calls Signal. We also want to make sure
				//that if an event sneaks in right as the task is returning from Run()
				//(via Signal) that the Run function will be invoked again.
				doneProcessingEvent = compare_and_store(Task::kAlive, 0, &theTask->fEvents);
				if (doneProcessingEvent)
					theTask = NULL;	
			}
			else
			{
				//note that if we get here, we don't reset theTask, so it will get passed into
				//WaitForTask
				theTask->fTimerHeapElem.SetValue(OS::Milliseconds() + theTimeout);
				fHeap.Insert(&theTask->fTimerHeapElem);
				(void)atomic_or(&theTask->fEvents, Task::kIdleEvent);
				doneProcessingEvent = true;
			}
		}
	}
}

Task* TaskThread::WaitForTask()
{
	while (true)
	{
		SInt64 theCurrentTime = OS::Milliseconds();
		
		if ((fHeap.PeekMin() != NULL) && (fHeap.PeekMin()->GetValue() <= theCurrentTime))
			return (Task*)fHeap.ExtractMin()->GetEnclosingObject();
	
		//if there is an element waiting for a timeout, figure out how long we should wait.
		SInt32 theTimeout = 0;
		if (fHeap.PeekMin() != NULL)
			theTimeout = fHeap.PeekMin()->GetValue() - theCurrentTime;
		Assert(theTimeout >= 0);
		
		//wait...
		OSQueueElem* theElem = fTaskQueue.DeQueueBlocking(this, theTimeout);
		if (theElem != NULL)
			return (Task*)theElem->GetEnclosingObject();
	}	
}

TaskThread** TaskThreadPool::sTaskThreadArray = NULL;
UInt32		 TaskThreadPool::sNumTaskThreads = 0;

bool TaskThreadPool::AddThreads(UInt32 numToAdd)
{
	Assert(sTaskThreadArray == NULL);
	sTaskThreadArray = new TaskThread*[numToAdd];
		
	for (UInt32 x = 0; x < numToAdd; x++)
	{
		sTaskThreadArray[x] = new ('tskT') TaskThread();
		sTaskThreadArray[x]->Start();
	}
	sNumTaskThreads = numToAdd;
	return true;
}

void TaskThreadPool::RemoveThreads()
{
	//Tell all the threads to stop
	for (UInt32 x = 0; x < sNumTaskThreads; x++)
		sTaskThreadArray[x]->SendStopRequest();

	//Because any (or all) threads may be blocked on the queue, cycle through
	//all the threads, signalling each one
	for (UInt32 y = 0; y < sNumTaskThreads; y++)
		sTaskThreadArray[y]->fTaskQueue.GetCond()->Signal();
	
	//Ok, now wait for the selected threads to terminate, deleting them and removing
	//them from the queue.
	for (UInt32 z = 0; z < sNumTaskThreads; z++)
		delete sTaskThreadArray[z];
	
	sNumTaskThreads = 0;
}
