/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		Dictionary.cpp

	Contains:	
					
	$Log: Dictionary.cpp,v $

	Created: Tue, Mar 2, 1999 @ 4:23 PM
*/


#include "Dictionary.h"

#include <string.h>
#include <stdio.h>

#include "RTPServerInterface.h"
#include "MyAssert.h"
#include "OS.h"


//UInt32 MinUInt32(UInt32 a, UInt32 b);
inline UInt32 MinUInt32(UInt32 a, UInt32 b) 	{ if (a < b) return a; return b; }
inline UInt32 MaxUInt32(UInt32 a, UInt32 b) 	{ if (a > b) return a; return b; }


static DictValueIDManager sDefaultDictValueIDManager;

//Helper Classes

//internal storage representation
class DictValueElement
{

	public:
		DictValueElement() : fValueSignature(0),fValueLen(0), fValueBuffer(0)
{}
		~DictValueElement() {delete[] fValueBuffer;}
		
		void SetValue(const void*  inValueBuffer, UInt32 inBufferLen);
		void GetValue(void* ioValueBuffer, UInt32 inBufferLen);
		void ClearValue();
		
		inline UInt32 GetLength(){return fValueLen;}
		inline FourCharCode GetSignature(){return fValueSignature;}
		
		FourCharCode fValueSignature;
		UInt32 fValueLen;
		UInt8* fValueBuffer;	//not really UInt8*, just so I can do "fValueBuffer = new UInt8[1234]"

};

typedef DictValueElement* CValueElementPtr;


Dictionary::Dictionary(DictValueIDManager* inDictValueIDManager)  :
	fValueElementArraySize(0),
	fValueElementArray(NULL),
	fDictValueIDManager(inDictValueIDManager)
{
	if (NULL == inDictValueIDManager)
		fDictValueIDManager = &sDefaultDictValueIDManager;
}


Dictionary::~Dictionary()
{
	if (fValueElementArray != NULL)
	{
		for (UInt32 i=0;i<fValueElementArraySize; i++)
			delete fValueElementArray[i];
			
		//::free(fValueElementArray);
		delete[] fValueElementArray;
	}
}

void Dictionary::DisposeValue(UInt32 inValueID)
{
	Assert(NULL != fDictValueIDManager);
	Assert(fDictValueIDManager->ValidValueID(inValueID));
	
	if (fDictValueIDManager->ValidValueID(inValueID) && 
		fValueElementArray[inValueID] != NULL)
	{
		fValueElementArray[inValueID]->ClearValue();
	}
}


void Dictionary::SetValue(UInt32 inValueID, const void* inValueBuffer, UInt32 inBufferLen)
{
	Assert(NULL != fDictValueIDManager);
	Assert(fDictValueIDManager->ValidValueID(inValueID));
	
	if (fDictValueIDManager->ValidValueID(inValueID))
	{
		if (inValueID >= fValueElementArraySize) 
			this->GrowValueElementArray(inValueID);
		

		Assert(NULL != fValueElementArray);

		//if there's no element already there, then instantiate a new one
		if (fValueElementArray[inValueID] == NULL)
            fValueElementArray[inValueID] = new ('dict') DictValueElement;
			
		Assert(fValueElementArray[inValueID] != NULL);
		
		fValueElementArray[inValueID]->SetValue(inValueBuffer, inBufferLen);
	}
}


UInt32 Dictionary::LookupIDBySignature(FourCharCode inValueSignature)
{
	Assert(NULL != fDictValueIDManager);
	
	UInt32 arrayIndex = fDictValueIDManager->LookupIDBySignature (inValueSignature);
	
	return arrayIndex;
}


void Dictionary::GetValue(UInt32 inValueID, void* ioValueBuffer, UInt32 inBufferLen, UInt32* ioValueLen)
{
		Assert(fDictValueIDManager != NULL);
		
		if (inValueID < fValueElementArraySize &&		//is available in this individual dictionary -and-
			 fDictValueIDManager->ValidValueID(inValueID) &&	// available in the any dictionaries using the same fDictValueIDManager 
			 fValueElementArray[inValueID] != NULL)
		{
			fValueElementArray[inValueID]->GetValue(ioValueBuffer, inBufferLen);
			
			*ioValueLen = fValueElementArray[inValueID]->GetLength();
		}
		else
			*ioValueLen = 0;
}

void Dictionary::GrowValueElementArray(UInt32 inNewID)
{
	//Very expensive function - optimize this later if it's a problem...
	//maybe realloc?

	int newArraySize = inNewID+8;	//chunks-o-eight
	
	DictValueElement** newValuesArray = (DictValueElement**) new('dcvb')UInt8[sizeof(DictValueElement*) * newArraySize];
	Assert(newValuesArray != NULL);
	
	//zero all the entries out...
	::memset(newValuesArray, 0, newArraySize * sizeof(DictValueElement*));
	 
	Assert(newValuesArray != NULL);
	
	if (fValueElementArray != NULL)
	{
		::memcpy(newValuesArray, fValueElementArray, fValueElementArraySize * sizeof(DictValueElement*) );
		delete[] fValueElementArray;
	}
	
	fValueElementArraySize = newArraySize;
	fValueElementArray = newValuesArray;
	
	Assert(NULL != fValueElementArray);
}



UInt32 Dictionary::AddValue(FourCharCode inValueSignature, const void* inValueBuffer, UInt32 inBufferLen)
{
	// This call must be made at server initialization time

	Assert( RTPServerInterface::kStartingUpState == RTPServerInterface::GetServerState() );
	if ( RTPServerInterface::kStartingUpState != RTPServerInterface::GetServerState() )
		return 0;
	

	UInt32 arrayIndex = this->LookupIDBySignature(inValueSignature);
	
	if (arrayIndex != 0) //update an existing value
	{
		this->SetValue(arrayIndex, inValueBuffer, inBufferLen);
	}
	else //otherwise it must be new...
	{
		arrayIndex = this->GenerateIDForSignature(inValueSignature);
		
		if (arrayIndex >= fValueElementArraySize)
			this->GrowValueElementArray(arrayIndex);
		
		Assert(fDictValueIDManager != NULL);
		Assert(fValueElementArray != NULL);
		Assert(fValueElementArray[arrayIndex] != NULL);
		
		fValueElementArray[arrayIndex]->fValueSignature = inValueSignature;
		this->SetValue(arrayIndex, inValueBuffer, inBufferLen);
	}

	return arrayIndex;
}

UInt32 Dictionary::GenerateIDForSignature(FourCharCode inValueSignature)
{
	Assert(NULL != fDictValueIDManager);

	return fDictValueIDManager->GenerateIDForSignature(inValueSignature);
}


UInt32 DictValueIDManager::GenerateIDForSignature(FourCharCode inValueSignature)
{
	// This call must only be made at server initialization time

	Assert( RTPServerInterface::kStartingUpState == RTPServerInterface::GetServerState() );
	if ( RTPServerInterface::kStartingUpState != RTPServerInterface::GetServerState() )
		return 0;
	
	UInt32 arrayIndex = 1;

	for (arrayIndex=1;arrayIndex<fTagsArraySize; arrayIndex++)
	{
		if (fTagsArray[arrayIndex] == inValueSignature)
			return arrayIndex;
	}
	
	if (fTagsArraySize <= fNextAvailableID)
	{
		fTagsArraySize += 64;	//arbitrary increment value
		FourCharCode* tagsArray;
		//fTagsArray = (FourCharCode*)realloc(fTagsArray, sizeof(FourCharCode)*fTagsArraySize);
		tagsArray = new FourCharCode[fTagsArraySize];
		delete[] fTagsArray;
		fTagsArray = tagsArray;
	}
	
	Assert(NULL != fTagsArray);
	arrayIndex = fNextAvailableID;
	fTagsArray[arrayIndex] = inValueSignature;
	
	++fNextAvailableID;
	
	return arrayIndex;
}


FourCharCode DictValueIDManager::LookupSignatureByID(UInt32 inValueID)
{
	if (this->ValidValueID(inValueID))
		return fTagsArray[inValueID];
	
	return 0;
}


UInt32 DictValueIDManager::LookupIDBySignature(FourCharCode inValueSignature)
{
	UInt32 arrayIndex = 0;

	for (arrayIndex=1;arrayIndex<fTagsArraySize; arrayIndex++)
	{
		if (fTagsArray[arrayIndex] == inValueSignature)
			return arrayIndex;
	}
	
	return 0;
}


void DictValueElement::SetValue(const void* inValueBuffer, UInt32 inBufferLen)
{
	//probably ought to optimize so it doesn't delete/new so often

	if ( inBufferLen > fValueLen ) 
	{
		delete[] fValueBuffer;
		
		fValueBuffer = new ('dcvb') UInt8[inBufferLen];
	}
	
	fValueLen = inBufferLen;

	::memcpy( fValueBuffer, inValueBuffer, inBufferLen);

}


void DictValueElement::GetValue(void* ioValueBuffer, UInt32 inBufferLen)
{
	Assert(inBufferLen >= fValueLen);

	if (inBufferLen >= fValueLen)
	{
		::memcpy( ioValueBuffer, fValueBuffer, MinUInt32(inBufferLen, fValueLen) );
	}
}


void DictValueElement::ClearValue()
{
	delete[] fValueBuffer;
	fValueBuffer = NULL;
	fValueLen = 0;
}
