/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
// $Id: QTHintTrack.cpp,v 1.16 1999/06/15 23:18:17 serenyi Exp $
//
// QTHintTrack:
//   The central point of control for a track in a QTFile.


// -------------------------------------
// Includes
//
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "QTFile.h"

#include "QTAtom.h"
#include "QTAtom_hinf.h"
#include "QTAtom_tref.h"

#include "QTHintTrack.h"
#include "OSMutex.h"



// -------------------------------------
// Macros
//
#define DEBUG_PRINT(s) if(fDebug) printf s
#define DEEP_DEBUG_PRINT(s) if(fDeepDebug) printf s



// -------------------------------------
// Class state cookie
//
QTHintTrack_HintTrackControlBlock::QTHintTrack_HintTrackControlBlock(QTFile_FileControlBlock * FCB)
	: fFCB(FCB),
	
	  fCachedSampleNumber(0),
	  fCachedSample(NULL),
	  fCachedSampleSize(0), fCachedSampleLength(0),

	  fCachedHintTrackSampleNumber(0), fCachedHintTrackSampleOffset(0),
	  fCachedHintTrackSample(NULL),
	  fCachedHintTrackSampleLength(0)
{
	fstscSTCB = NEW('scCB', QTAtom_stsc_SampleTableControlBlock, ());
	fsttsSTCB = NEW('tsCB', QTAtom_stts_SampleTableControlBlock, ());
}

QTHintTrack_HintTrackControlBlock::~QTHintTrack_HintTrackControlBlock(void)
{
	if( fstscSTCB != NULL )
		delete fstscSTCB;
	if( fsttsSTCB != NULL )
		delete fsttsSTCB;

	if( fCachedSample != NULL )
		delete []fCachedSample;
	if( fCachedHintTrackSample != NULL )
		delete []fCachedHintTrackSample;
}



// -------------------------------------
// Constructors and destructors
//
QTHintTrack::QTHintTrack(QTFile * File, QTFile::AtomTOCEntry * Atom, bool Debug, bool DeepDebug)
	: QTTrack(File, Atom, Debug, DeepDebug),
	  fHintInfoAtom(NULL), fHintTrackReferenceAtom(NULL),
	  fTrackRefs(NULL),
	  fMaxPacketSize(65536),
	  fRTPTimescale(0),
	  fFirstRTPTimestamp(0),
	  fTimestampRandomOffset(0),
	  fSequenceNumberRandomOffset(0),
	  fHintTrackInitialized(false)
{
}

QTHintTrack::~QTHintTrack(void)
{
	//
	// Free our variables
	if( fHintInfoAtom != NULL )
		delete fHintInfoAtom;
	if( fHintTrackReferenceAtom != NULL )
		delete fHintTrackReferenceAtom;
		
	if( fTrackRefs != NULL )
		delete[] fTrackRefs;
}



// -------------------------------------
// Initialization functions
//
QTTrack::ErrorCode QTHintTrack::Initialize(void)
{
	// General vars
	char		*SampleDescription, *pSampleDescription;
	UInt32		SampleDescriptionLength;

	QTFile::AtomTOCEntry	*hinfTOCEntry;

	//
	// Don't initialize more than once.
	if( IsHintTrackInitialized() )
		return errNoError;

	//
	// Initialize the QTTrack class.
	if( QTTrack::Initialize() != errNoError )
		return errInvalidQuickTimeFile;


	//
	// Get the sample description table for this track and verify that it is an
	// RTP track.
	if( !fSampleDescriptionAtom->FindSampleDescription('rtp ', &SampleDescription, &SampleDescriptionLength) )
		return errInvalidQuickTimeFile;

	memcpy(&fMaxPacketSize, SampleDescription + 20, 4);
	fMaxPacketSize = ntohl(fMaxPacketSize);
	
	for( pSampleDescription = (SampleDescription + 24);
		 pSampleDescription < (SampleDescription + SampleDescriptionLength);
	) {
		// General vars
		UInt32		EntryLength, DataType;
		
		
		//
		// Get the entry length and data type of this entry.
		memcpy(&EntryLength, pSampleDescription + 0, 4);
		EntryLength = ntohl(EntryLength);
		memcpy(&DataType, pSampleDescription + 4, 4);
		DataType = ntohl(DataType);
		
		//
		// Process this data type.
		switch( DataType ) {
			case 'tims':	// RTP timescale
				memcpy(&fRTPTimescale, pSampleDescription + 8, 4);
				fRTPTimescale = ntohl(fRTPTimescale);
			break;
			
			case 'tsro':	// Timestamp random offset
				memcpy(&fTimestampRandomOffset, pSampleDescription + 8, 4);
				fTimestampRandomOffset = ntohl(fTimestampRandomOffset);
			break;
			
			case 'snro':	// Sequence number random offset
				memcpy(&fSequenceNumberRandomOffset, pSampleDescription + 8, 2);
				fSequenceNumberRandomOffset = ntohs(fSequenceNumberRandomOffset);
			break;
		}
		
		//
		// Next entry..
		pSampleDescription += EntryLength;
	}
	
	

	//
	// Load in the hint info atom for this track.
	if( fFile->FindTOCEntry(":udta:hinf", &hinfTOCEntry, &fTOCEntry) ) {
		fHintInfoAtom = NEW('hinf', QTAtom_hinf, (fFile, hinfTOCEntry, fDebug, fDeepDebug));
		if( fHintInfoAtom == NULL )
			return errInternalError;
		if( !fHintInfoAtom->Initialize() )
			return errInvalidQuickTimeFile;
	}

	//
	// Load in the hint track reference atom for this track.
	if( !fFile->FindTOCEntry(":tref:hint", &hinfTOCEntry, &fTOCEntry) )
		return errInvalidQuickTimeFile;
	
	fHintTrackReferenceAtom = NEW('tref', QTAtom_tref, (fFile, hinfTOCEntry, fDebug, fDeepDebug));
	if( fHintTrackReferenceAtom == NULL )
		return errInternalError;
	if( !fHintTrackReferenceAtom->Initialize() )
		return errInvalidQuickTimeFile;
	

	//
	// Allocate space for our track reference table.
	fTrackRefs = NEW_ARRAY('HTtr', QTTrack *, fHintTrackReferenceAtom->GetNumReferences());
	if( fTrackRefs == NULL )
		return errInternalError;

	//
	// Locate all of the tracks that we use, but don't initialize them until we
	// actually try to access them.
	for( UInt32 CurRef = 0; CurRef < fHintTrackReferenceAtom->GetNumReferences(); CurRef++ ) {
		// General vars
		UInt32		TrackID;
		
		
		//
		// Get the reference and make sure it's not empty.
		if( !fHintTrackReferenceAtom->TrackReferenceToTrackID(CurRef, &TrackID) )
			break;
		if( TrackID == 0 )
			continue;
		
		//
		// Store away a reference to this track.
		if( !fFile->FindTrack(TrackID, &fTrackRefs[CurRef]) )
			return errInvalidQuickTimeFile;
	}


	//
	// Calculate the first RTP timestamp for this track.
	if( GetFirstEditMovieTime() > 0 ) {
		UInt64 TrackTime = GetFirstEditMovieTime();
		TrackTime *= fRTPTimescale;
		if( fFile->GetTimeScale() > 0.0 )
			TrackTime /= (UInt64)fFile->GetTimeScale();
		fFirstRTPTimestamp = (UInt32)(TrackTime & 0xffffffff);
	} else {
		fFirstRTPTimestamp = 0;
	}

	//
	// This track has been successfully initialiazed.
	fHintTrackInitialized = true;
	return errNoError;
}
	


// -------------------------------------
// Accessors.
//
QTTrack::ErrorCode QTHintTrack::GetSDPFileLength(int * Length)
{
	// General vars
	QTFile::AtomTOCEntry	*sdpTOCEntry;


	//
	// Locate the 'sdp ' atom for this track.
	if( !fFile->FindTOCEntry(":udta:hnti:sdp ", &sdpTOCEntry, &fTOCEntry) )
		return errInvalidQuickTimeFile;
	
	//
	// Return the length.
	*Length = sdpTOCEntry->AtomDataLength;
	return errNoError;
}

char * QTHintTrack::GetSDPFile(int * Length)
{
	// General vars
	QTFile::AtomTOCEntry	*sdpTOCEntry;
	QTAtom					*sdpAtom;
	
	char		*sdpBuffer;


	//
	// Locate and read in the 'sdp ' atom for this track.
	if( !fFile->FindTOCEntry(":udta:hnti:sdp ", &sdpTOCEntry, &fTOCEntry) )
		return NULL;
	
	sdpAtom = NEW('QTAt', QTAtom, (fFile, sdpTOCEntry, fDebug, fDeepDebug));
	if( sdpAtom == NULL )
		return NULL;
	if( !sdpAtom->Initialize() )
		return NULL;
	
	//
	// Allocate a buffer to hold the SDP file.
	*Length = sdpTOCEntry->AtomDataLength;
	//sdpBuffer = (char *)malloc(*Length);
	sdpBuffer = NEW_ARRAY('sdpB', char, *Length);
	if( sdpBuffer != NULL )
		sdpAtom->ReadBytes(0, sdpBuffer, *Length);
	
	//
	// Free the atom and return the buffer.
	delete sdpAtom;
	return sdpBuffer;
}


			
// -------------------------------------
// Sample functions
//
bool QTHintTrack::GetSamplePtr(UInt32 SampleNumber, char ** Buffer, UInt32 * Length, QTHintTrack_HintTrackControlBlock * HTCB)
{
	// General vars
	UInt32		NewSampleLength;
	
	
	//
	// Use the default HTCB if we weren't given one.
	if( HTCB == NULL )
		HTCB = &fDefaultHTCB;
		
		
	//
	// See if this sample is in our cache, returning it out of the cache if it
	// is, fetching and caching it if it is not.
	if( SampleNumber == HTCB->fCachedSampleNumber ) {
		*Buffer = HTCB->fCachedSample;
		*Length = HTCB->fCachedSampleLength;
		return true;
	}
	

	//
	// Get the length of the new sample.
	if( !GetSampleInfo(SampleNumber, &NewSampleLength, NULL /* Offset */, NULL /* SDidx */, HTCB->fstscSTCB) )
		return false;
	
	//
	// Create a new (bigger) cache buffer if the sample wouldn't fit in the
	// old one.
	if( (HTCB->fCachedSample == NULL) || (HTCB->fCachedSampleSize < NewSampleLength) ) {
		//
		// Free the old cache entry if we had one.
		if( HTCB->fCachedSample != NULL ) {
			HTCB->fCachedSampleNumber = 0;
			HTCB->fCachedSampleSize = 0;
			delete[] HTCB->fCachedSample;
		}
		
		//
		// Create a new cache entry.
		HTCB->fCachedSampleLength = HTCB->fCachedSampleSize = NewSampleLength;
		HTCB->fCachedSample = NEW_ARRAY('HTcs', char, HTCB->fCachedSampleSize);
		if( HTCB->fCachedSample == NULL )
			return false;
	}
	

	//
	// Read in the new sample.
	HTCB->fCachedSampleLength = NewSampleLength;
	if( !GetSample(SampleNumber, HTCB->fCachedSample, &HTCB->fCachedSampleLength, HTCB->fFCB, HTCB->fstscSTCB) )
		return false;
	
	//
	// Return the cached sample.
	HTCB->fCachedSampleNumber = SampleNumber;
	*Buffer = HTCB->fCachedSample;
	*Length = HTCB->fCachedSampleLength;
	return true;
}



// -------------------------------------
// Packet functions
//
QTTrack::ErrorCode QTHintTrack::GetNumPackets(UInt32 SampleNumber, UInt16 * NumPackets, QTHintTrack_HintTrackControlBlock * HTCB)
{
	// General vars
	char		*Buf;
	UInt32		BufLen;

	UInt16		EntryCount;
	
	
	//
	// Read this sample and figure out how many packets are in it.
	if( !GetSamplePtr(SampleNumber, &Buf, &BufLen, HTCB) )
		return errInvalidQuickTimeFile;
	
	memcpy(&EntryCount, (char *)Buf + 0, 2);
	*NumPackets = ntohs(EntryCount);

	return errNoError;
}

QTTrack::ErrorCode QTHintTrack::GetPacket(UInt32 SampleNumber, UInt16 PacketNumber, char * Buffer, UInt32 * Length, Float64 * TransmitTime, bool dropBFrames, UInt32 SSRC, QTHintTrack_HintTrackControlBlock * HTCB)
{
	// Temporary vars
	UInt16		tempInt16;
	UInt32		tempInt32;

	UInt16		curEntry;

	// General vars
	UInt32		MediaTime;
	
	char		*Buf;
	UInt32		BufLen;
	
	char		*pBuf, *pBufEnd;
	char		*pDataTableStart;

	UInt16		EntryCount;
	
	UInt16		RTPHeaderBits, RTPSequenceNumber;
	UInt32		RTPTimestamp;

	SInt32		RelativePacketTransmissionTime;
	UInt16		Flags, DataEntryCount;
	UInt32		TLVSize;

	char		*pPktBuf;
	UInt32		PktSize;
	
	
	DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - Building packet #%u in sample %lu.\n", PacketNumber, SampleNumber));

	//
	// Use the default HTCB if we weren't given one.
	if( HTCB == NULL )
		HTCB = &fDefaultHTCB;
		
		
	//
	// Get the RTP timestamp for this sample.
	if( !GetSampleMediaTime(SampleNumber, &MediaTime, HTCB->fsttsSTCB) )
		return errInvalidQuickTimeFile;

	if( fRTPTimescale == GetTimeScale() )
		RTPTimestamp = MediaTime;
	else
		RTPTimestamp = (UInt32)(MediaTime * ((Float64)fRTPTimescale * GetTimeScaleRecip()) );

	RTPTimestamp += fFirstRTPTimestamp;

	//
	// Add the first edit's media time.
	MediaTime += GetFirstEditMediaTime();

	//
	// Read this sample and generate the n'th packet.
	if( !GetSamplePtr(SampleNumber, &Buf, &BufLen, HTCB) )
		return errInvalidQuickTimeFile;
	
	pBuf = Buf;
	pBufEnd = (char *)Buf + BufLen;

	memcpy(&EntryCount, (char *)Buf + 0, 2);
	EntryCount = ntohs(EntryCount);
	if( (PacketNumber-1) > EntryCount )
		return errInvalidQuickTimeFile;
	pBuf += 4;

	//
	// Loop through the sample until we find the packet that we want.
	for( UInt16 curPacket = 0; curPacket != PacketNumber; curPacket++ ) {
		//
		// Read in this packet's header.
		memcpy(&tempInt32, pBuf, 4);
		RelativePacketTransmissionTime = ntohl(tempInt32);
		*TransmitTime =   ( MediaTime * fMediaHeaderAtom->GetTimeScaleRecip() )
						+ ( RelativePacketTransmissionTime * fMediaHeaderAtom->GetTimeScaleRecip() );

		memcpy(&RTPHeaderBits, pBuf + 4, 2);

		memcpy(&RTPSequenceNumber, pBuf + 6, 2);
		RTPSequenceNumber = ntohs(RTPSequenceNumber);

		memcpy(&Flags, pBuf + 8, 2);
		Flags = ntohs(Flags);
		
		if ((Flags & kBFrameBitMask) && (dropBFrames))
			return errIsBFrame;
		
		
		memcpy(&DataEntryCount, pBuf + 10, 2);
		DataEntryCount = ntohs(DataEntryCount);
		
		if( Flags & 0x4 ) { // Extra Information TLV is present
			memcpy(&TLVSize, pBuf + 12, 4);
			TLVSize = ntohl(TLVSize);
		} else {
			TLVSize = 0;
		}
		
		//
		// Adjust (and check) pBuf)
		pBuf += (4 + 2 + 2 + 2 + 2) + TLVSize;
		if( pBuf > pBufEnd )
			return errInvalidQuickTimeFile;
		
		//
		// Skip over the data table entries if this is *not* the packet that
		// we want.
		if( (curPacket + 1) != PacketNumber ) {
			pBuf += DataEntryCount * 16;
			if( pBuf > pBufEnd )
				return errInvalidQuickTimeFile;
		}
	}
	
	DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ..RTPTimestamp=%lu; RTPSequenceNumber=%u; TransmitTime=%.2f\n", RTPTimestamp, RTPSequenceNumber, *TransmitTime));

	//
	// We found this packet and the start of our data table.
	pDataTableStart = pBuf;
	
	//
	// Our first pass through the data table is done to compute the size of the
	// RTP packet that we will be generating and to validate the contents of
	// the data table.
	PktSize = 12; // 12 = size of RTP header
	pBuf = pDataTableStart;
	DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ..Calculating packet length.\n"));
	for( curEntry = 0; curEntry < DataEntryCount; curEntry++ ) {
		//
		// Get the size out of this entry.
		switch( *pBuf ) {
			case 0x00:	// No-Op Data Mode
				DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ....No-Op entry found\n"));
			break;
			
			case 0x01:	// Immediate Data Mode
				DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ....Immediate entry found (size=%u)\n", (UInt16)*(pBuf + 1)));
				PktSize += *(pBuf + 1);
			break;
			
			case 0x02:	// Sample Mode
				memcpy(&tempInt16, pBuf + 2, 2);
				tempInt16 = ntohs(tempInt16);

				DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ....Sample entry found (size=%u)\n", tempInt16));
				PktSize += tempInt16;
			break;
			
			case 0x03:	// Sample Description Data Mode
				memcpy(&tempInt16, pBuf + 2, 2);
				tempInt16 = ntohs(tempInt16);

				DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ....Sample Description entry found (size=%u)\n", tempInt16));
				PktSize += tempInt16;
			break;
			
			default:
				printf("Found unknown RTP data table type!\n");
				QTASSERT(0);
			break;
		}
		
		//
		// Move to the next entry.
		pBuf += 16;
		if( pBuf > pBufEnd )
			return errInvalidQuickTimeFile;
	}
	DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ..Packet length is %lu bytes.\n", PktSize));

	//
	// Allocate space for this packet.
	if( *Length < PktSize )
		return errParamError;
	*Length = PktSize;
	pPktBuf = Buffer;

	//
	// Add in the RTP header.
	tempInt16 = RTPHeaderBits | ntohs(0x8000) /* v2 RTP header */;
	memcpy(pPktBuf, (char *)&tempInt16, 2);
	pPktBuf += 2;

	tempInt16 = htons(RTPSequenceNumber);
	memcpy(pPktBuf, (char *)&tempInt16, 2);
	pPktBuf += 2;

	tempInt32 = htonl(RTPTimestamp);
	memcpy(pPktBuf, (char *)&tempInt32, 4);
	pPktBuf += 4;

	tempInt32 = htonl(SSRC);
	memcpy(pPktBuf, (char *)&tempInt32, 4);
	pPktBuf += 4;

	//
	// Now we go through the data table again, but this time we build the
	// packet.
	pBuf = pDataTableStart;
	DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ..Building packet.\n"));
	for( curEntry = 0; curEntry < DataEntryCount; curEntry++ ) {
		//
		// Get the size out of this entry.
		switch( *pBuf ) {
			case 0x00:	// No-Op Data Mode
				DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ....No-Op entry found\n"));
			break;
			
			case 0x01:	// Immediate Data Mode
				DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ....Immediate entry found (size=%u)\n", (UInt16)*(pBuf + 1)));
				
				//
				// Copy the data straight into the packet.
				memcpy(pPktBuf, pBuf + 2, *(pBuf + 1));
				pPktBuf += *(pBuf + 1);
			break;
			
			case 0x02:	// Sample Mode
			{
				// General vars
				SInt8		TrackRefIndex;
				UInt16		ReadLength;
				UInt32		MediaSampleNumber;
				UInt32		ReadOffset;
				UInt16		BytesPerCompressionBlock, SamplesPerCompressionBlock;
				
				UInt32		SampleDescriptionIndex;
				UInt64		DataOffset;
				QTTrack		*Track;
				
				
				//
				// Get the information about this sample
				TrackRefIndex = (SInt8)*(pBuf + 1);
				
				memcpy(&ReadLength, pBuf + 2, 2);
				ReadLength = ntohs(ReadLength);

				memcpy(&MediaSampleNumber, pBuf + 4, 4);
				MediaSampleNumber = ntohl(MediaSampleNumber);

				memcpy(&ReadOffset, pBuf + 8, 4);
				ReadOffset = ntohl(ReadOffset);
				
				memcpy(&BytesPerCompressionBlock, pBuf + 12, 2);
				BytesPerCompressionBlock = ntohs(BytesPerCompressionBlock);
				if( BytesPerCompressionBlock == 0 )
					BytesPerCompressionBlock = 1;

				memcpy(&SamplesPerCompressionBlock, pBuf + 14, 2);
				SamplesPerCompressionBlock = ntohs(SamplesPerCompressionBlock);
				if( SamplesPerCompressionBlock == 0 )
					SamplesPerCompressionBlock = 1;

				DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ....Sample entry found (sample#=%lu; offset=%lu; length=%u; BPCB=%u; SPCB=%u)\n", MediaSampleNumber, ReadOffset, ReadLength, BytesPerCompressionBlock, SamplesPerCompressionBlock));

				//
				// Figure out which track we're going to get the data from.
				if( TrackRefIndex == -1 ) {
					//
					// We're getting data out of the hint track..
					Track = this;
					
					//
					// ..this might be the Sample Description that is stored in the first
					// few samples of this track.  See if we have it cached and use that
					// if so.  If we do not have this block of data cached, then continue
					// on; we'll cache it later.
					if(    (MediaSampleNumber == HTCB->fCachedHintTrackSampleNumber)
						&& (ReadOffset == HTCB->fCachedHintTrackSampleOffset)
						&& (ReadLength == HTCB->fCachedHintTrackSampleLength)
					) {
						memcpy(pPktBuf, HTCB->fCachedHintTrackSample, HTCB->fCachedHintTrackSampleLength);
						pPktBuf += HTCB->fCachedHintTrackSampleLength;
						break;
					}
					
					//
					// ..or this might be in our currently-cached data sample.
					if(    (MediaSampleNumber == HTCB->fCachedSampleNumber)
						&& ((ReadOffset + ReadLength) <= HTCB->fCachedSampleLength)
					) {
						memcpy(pPktBuf, HTCB->fCachedSample + ReadOffset, ReadLength);
						pPktBuf += ReadLength;
						break;
					}
				} else {
					//
					// We're getting data out of a media track..
					Track = fTrackRefs[TrackRefIndex];
					
					//
					// Initialize this track if we haven't done so yet.
					if( !Track->IsInitialized() )
					{
						OSMutexLocker theLocker(fFile->GetMutex());
						if( Track->Initialize() != QTTrack::errNoError )
							return errInvalidQuickTimeFile;
					}
				}
					
				//
				// Get the information about this sample and compute an offset.  If the BPCB
				// and SPCB are 1, then we use the standard sample routines to get the location
				// of this sample, otherwise we have to compute it ourselves.
				if( (BytesPerCompressionBlock == 1) && (SamplesPerCompressionBlock == 1) ) {
					//
					// Fetch the location of this sample.
					// NOTE: We cannot use the STCB here because we might not be pointing at the
					// hint track.  Even if it was the hint track that we were looking at, the
					// inline hint track read code above should have made it impossible to end up
					// here.
					if( !Track->GetSampleInfo(MediaSampleNumber, NULL /* Length */, &DataOffset, &SampleDescriptionIndex) )
						return errInvalidQuickTimeFile;
				} else {
					// General vars
					UInt32		ChunkNumber, ChunkOffset, SampleOffsetInChunk;


					//
					// Adjust the MediaSampleNumber to account for the ReadOffset if
					// necessary.
					if( ReadOffset >= BytesPerCompressionBlock ) {
						UInt32		CompressionBlocksToSkip = ReadOffset / BytesPerCompressionBlock;
						MediaSampleNumber += CompressionBlocksToSkip * SamplesPerCompressionBlock;
						ReadOffset -= CompressionBlocksToSkip * BytesPerCompressionBlock;

						DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ......Used BPCB/SPCB to adjust sample number (MediaSampleNumber=%lu; ReadOffset=%lu).\n", MediaSampleNumber, ReadOffset));
					}

					//
					// Get the chunk that this sample is in and then find the chunk's offset.
					// NOTE: We cannot use the STCB here because we might not be pointing at the
					// hint track.  Even if it was the hint track that we were looking at, the
					// inline hint track read code above should have made it impossible to end up
					// here.
					if( !Track->SampleNumberToChunkNumber(MediaSampleNumber, &ChunkNumber, &SampleDescriptionIndex, &SampleOffsetInChunk) )
						return errInvalidQuickTimeFile;
					if( !Track->ChunkOffset(ChunkNumber, &ChunkOffset) )
						return errInvalidQuickTimeFile;

					//
					// Adjust the data offset to account for the BPCB and SPCB values.
					DataOffset = ChunkOffset;
					DataOffset += (UInt64)(SampleOffsetInChunk * ((Float64)BytesPerCompressionBlock / (Float64)SamplesPerCompressionBlock));
				}
				
				//
				// Adjust the offset by the amount requested in this hint entry.
				DataOffset += ReadOffset;

				//
				// Read the requested data into our packet.
			#if HAVE_64BIT_PRINTF
				#if HAVE_64BIT_PRINTF_AS_LL
					DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ......Reading %u bytes at offset %llu.\n", ReadLength, DataOffset));
				#else
					DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ......Reading %u bytes at offset %qu.\n", ReadLength, DataOffset));
				#endif
			#else
				DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ......Reading %u bytes at offset %lu.\n", ReadLength, (UInt32)DataOffset));
			#endif
                if( !Track->Read(SampleDescriptionIndex, DataOffset, pPktBuf, ReadLength, HTCB->fFCB) )
                    return errInvalidQuickTimeFile;
				
				//
				// If this chunk of data came out of our hint track; then we should cache
				// it, just in case we want it later.
				if( (TrackRefIndex == -1) && (MediaSampleNumber < SampleNumber) ) {
					//
					// Free the old cache entry if we had one.
					if( HTCB->fCachedHintTrackSample != NULL ) {
						HTCB->fCachedHintTrackSampleNumber = 0;
						HTCB->fCachedHintTrackSampleOffset = 0;
						delete[] HTCB->fCachedHintTrackSample;
					}
					
					//
					// Create a new cache entry.
					HTCB->fCachedHintTrackSampleLength = ReadLength;
					HTCB->fCachedHintTrackSample = NEW_ARRAY('HTcs', char, HTCB->fCachedHintTrackSampleLength);
					if( HTCB->fCachedHintTrackSample != NULL ) {
						memcpy(HTCB->fCachedHintTrackSample, pPktBuf, ReadLength);
						HTCB->fCachedHintTrackSampleNumber = MediaSampleNumber;
						HTCB->fCachedHintTrackSampleOffset = ReadOffset;
					}
				}
				
				pPktBuf += ReadLength;
			}
			break;
			
			case 0x03:	// Sample Description Data Mode
				memcpy(&tempInt16, pBuf + 2, 2);
				tempInt16 = ntohs(tempInt16);

				DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ....Sample Description entry found (size=%u)\n", tempInt16));
				QTASSERT(0);
			break;
			
			default:
				printf("Found unknown RTP data table type!\n");
				QTASSERT(0);
			break;
		}
		
		//
		// Move to the next entry.
		pBuf += 16;
		if( pBuf > pBufEnd )
			return errInvalidQuickTimeFile;
	}
	DEEP_DEBUG_PRINT(("QTHintTrack::GetPacket - ..Packet length is %lu bytes.\n", PktSize));
	
	//
	// The packet has been generated.
	return errNoError;
}



// -------------------------------------
// Debugging functions
//
void QTHintTrack::DumpTrack(void)
{
	//
	// Dump the QTTrack class.
	QTTrack::DumpTrack();

	//
	// Dump the sub-atoms of this track.
	if( fHintInfoAtom != NULL )
		fHintInfoAtom->DumpAtom();
}
