/*
 * X-Mame video specifics code
 *
 */
#ifdef x11
#define __X11_WINDOW_C_

/*
 * Include files.
 */

#include "xmame.h"
#include "x11.h"
#include "driver.h"
#include "sound.h"              /* for rearming the timer */
#include <X11/cursorfont.h>

static unsigned long black_xpixel;
static void x11_window_update_8_to_8bpp (void);
static void x11_window_update_8_to_16bpp (void);
static void x11_window_update_8_to_24bpp (void);
static void x11_window_update_8_to_32bpp (void);
static void x11_window_update_8_to_8bpp_direct (void);
static void x11_window_update_16_to_16bpp (void);
static void x11_window_update_16_to_24bpp (void);
static void x11_window_update_16_to_32bpp (void);
static void (*x11_window_update_display_func) (void) = NULL;

/* hmm we need these to do the clean up correctly, or we could just 
   trust unix & X to clean up after us but lett's keep things clean */
#ifdef USE_MITSHM
static int mit_shm_attached = 0;
#endif
static int private_cmap_allocated = 0;
static int use_rw_palette = 0;
static int warn_low_on_colors = 1;
static XImage *image = NULL;
static GC gc;
static char xpixel_allocated[256];
static int orig_widthscale, orig_heightscale;
static int image_width;
enum { X11_NORMAL, X11_MITSHM, X11_XIL };
static int x11_window_update_method = X11_NORMAL;


/*
 * Create a display screen, or window, large enough to accomodate a bitmap
 * of the given dimensions.
 */

#ifdef USE_MITSHM
/* following routine traps missused MIT-SHM if not available */
int test_mit_shm (Display * display, XErrorEvent * error)
{
   char msg[256];
   unsigned char ret = error->error_code;

   XGetErrorText (display, ret, msg, 256);
   /* if MIT-SHM request failed, note and continue */
   if (ret == BadAccess)
   {
      use_mit_shm = 0;
      return 0;
   }
   /* else unspected error code: notify and exit */
   fprintf (stderr_file, "Unspected X Error %d: %s\n", ret, msg);
   exit (1);
}
#endif

/*
 * This function creates an invisible cursor.
 *
 * I got the idea and code fragment from in the Apple II+ Emulator
 * version 0.06 for Linux by Aaron Culliney
 * <chernabog@baldmountain.bbn.com>
 *
 * I also found a post from Steve Lamont <spl@szechuan.ucsd.edu> on
 * xforms@bob.usuf2.usuhs.mil.  His comments read:
 *
 * Lifted from unclutter
 * Mark M Martin. cetia 1991 mmm@cetia.fr
 * Version 4 changes by Charles Hannum <mycroft@ai.mit.edu>
 *
 * So I guess this code has been around the block a few times.
 */

static Cursor create_invisible_cursor (Display * display, Window win)
{
   Pixmap cursormask;
   XGCValues xgc;
   XColor dummycolour;
   Cursor cursor;
   GC gc;

   cursormask = XCreatePixmap (display, win, 1, 1, 1 /*depth */ );
   xgc.function = GXclear;
   gc = XCreateGC (display, cursormask, GCFunction, &xgc);
   XFillRectangle (display, cursormask, gc, 0, 0, 1, 1);
   dummycolour.pixel = 0;
   dummycolour.red = 0;
   dummycolour.flags = 04;
   cursor = XCreatePixmapCursor (display, cursormask, cursormask,
                                 &dummycolour, &dummycolour, 0, 0);
   XFreeGC (display, gc);
   XFreePixmap (display, cursormask);
   return cursor;
}

/* This name doesn't really cover this function, since it also sets up mouse
   and keyboard. This is done over here, since on most display targets the
   mouse and keyboard can't be setup before the display has. */
int x11_window_create_display (void)
{
   XVisualInfo visualinfo;
   XSetWindowAttributes winattr;
   XGCValues xgcv;
   int myscreen;
   XEvent event;
   XSizeHints hints;
   XWMHints wm_hints;
   int image_height;
   int i;
   int event_mask;
   int window_width, window_height;
   int xpixels_to_allocate = 0;
   
   /* set all the default values */
   window = 0;
   image  = NULL;
   xpixel = NULL;
   x_palette_dirty = FALSE;
#ifdef USE_MITSHM
   mit_shm_attached = 0;
#endif
   private_cmap_allocated = 0;
   use_rw_palette = 0;
   warn_low_on_colors = 1;

   memset (xpixel_allocated, FALSE, sizeof (char) * 256);

   window_width     = widthscale  * visual_width;
   window_height    = heightscale * visual_height;
   image_width      = widthscale  * visual_width;
   image_height     = heightscale * visual_height;
   orig_widthscale  = widthscale;
   orig_heightscale = heightscale;
   screen           = DefaultScreenOfDisplay (display);
   myscreen         = DefaultScreen (display);
   
   /* try to find a 8bit pseudocolor visual, since that's best performance
      wise, otherwise just go with the default */
   if(!video_16bit &&
      XMatchVisualInfo (display, myscreen, 8, PseudoColor, &visualinfo))
   {
      xvisual = visualinfo.visual;
      depth   = 8;
      
      /*
       *  If we're running on a display that supports true colour and
       *  bitplanes at the same time, and the default visual is
       *  true colour, we can't inherit the colourmap and hence the
       *  XCreateWindow() call will fail. In this case we need to
       *  create our own private colourmap.
       *
       *  /Elias (elias-m@algonet.se)
       */

      if (DefaultVisual (display, myscreen)->class != PseudoColor ||
          DefaultDepth (display, myscreen) != 8)
      {
         use_private_cmap = TRUE;
      }
      fprintf(stderr_file, "Using PseudoColor Visual with a depth of 8bpp. Good!\n");
   }
   else
   {
      xvisual = DefaultVisual(display, myscreen);
      depth   = DefaultDepth (display, myscreen);
      fprintf(stderr_file, "Using a Visual with a depth of %dbpp.\n", 
         depth);
   }
   
   /* check the available extensions if compiled in */
#ifdef USE_XIL
   if (use_xil)
   {
      init_xil ();
   }

   /*
    *  If the XIL initialization worked, then use_xil will still be set.
    */
   if (use_xil)
   {
      image_width  = visual_width;
      image_height = visual_height;
      widthscale   = 1;
      heightscale  = 1;
      x11_window_update_method = X11_XIL;
   }
   else
#endif
#ifdef USE_MITSHM
   if (use_mit_shm)             /* look for available Mitshm extensions */
   {
      /* get XExtensions to be if mit shared memory is available */
      if (XQueryExtension (display, "MIT-SHM", &i, &i, &i))
      {
         x11_window_update_method = X11_MITSHM;
      }
      else
      {
         fprintf (stderr_file, "X-Server Doesn't support MIT-SHM extension\n");
         use_mit_shm = 0;
      }
   }
#endif

   /* create / asign a colormap */
   
   if (!use_private_cmap)
   {
      colormap = DefaultColormapOfScreen (screen);
      black_xpixel = BlackPixelOfScreen (screen);
   }
   else
   {
      colormap = XCreateColormap (display, RootWindowOfScreen (screen), xvisual, AllocNone);
      private_cmap_allocated = 1;
      black_xpixel = 0;         /* no way to tell which xpixel will be black,
                                   but we won't need it anyway since color allocation shouldn't
                                   fail on a privatecmap */
      fprintf (stderr_file, "Using private color map\n");
   }

   /*  Placement hints etc. */

#ifdef USE_XIL
   /*
    *  XIL allows us to rescale the window on the fly,
    *  so in this case, we don't prevent the user from
    *  resizing.
    */
   if (use_xil)
   {
      hints.flags = PSize;
   }
   else
#endif
      hints.flags = PSize | PMinSize | PMaxSize;

   hints.min_width  = hints.max_width  = hints.base_width  = window_width;
   hints.min_height = hints.max_height = hints.base_height = window_height;
   hints.x = hints.y = 0;
   hints.win_gravity = NorthWestGravity;
   
   i = XWMGeometry(display, myscreen, geometry, NULL, 0, &hints, &hints.x,
      &hints.y, &i, &i, &hints.win_gravity);
   if ((i&XValue) && (i&YValue))
      hints.flags |= PPosition | PWinGravity;
   
   /* Create and setup the window. No buttons, no fancy stuff. */
   
   winattr.background_pixel  = black_xpixel;
   winattr.border_pixel      = WhitePixelOfScreen (screen);
   winattr.bit_gravity       = ForgetGravity;
   winattr.win_gravity       = hints.win_gravity;
   winattr.backing_store     = NotUseful;
   winattr.override_redirect = False;
   winattr.save_under        = False;
   winattr.event_mask        = 0;
   winattr.do_not_propagate_mask = 0;
   winattr.colormap          = colormap;
   winattr.cursor            = None;
   if (root_window_id == 0)
   {
      root_window_id = RootWindowOfScreen (screen);
   }
   window = XCreateWindow (display, root_window_id, hints.x, hints.y,
                           window_width, window_height,
                           0, depth,
                           InputOutput, xvisual,
                           (CWBorderPixel | CWBackPixel | CWBitGravity |
                            CWWinGravity | CWBackingStore |
                            CWOverrideRedirect | CWSaveUnder | CWEventMask |
                            CWDontPropagate | CWColormap | CWCursor),
                           &winattr);
   if (!window)
   {
      fprintf (stderr_file, "OSD ERROR: failed in XCreateWindow().\n");
      return OSD_NOT_OK;
   }
   
   gc = XCreateGC (display, window, 0, &xgcv);

   wm_hints.input = TRUE;
   wm_hints.flags = InputHint;

   XSetWMHints (display, window, &wm_hints);
   XSetWMNormalHints (display, window, &hints);
   XStoreName (display, window, title);

   /* Select event mask */
   
   event_mask = FocusChangeMask | ExposureMask |
      EnterWindowMask | LeaveWindowMask |
      KeyPressMask | KeyReleaseMask;
   if (use_mouse)
   {
      event_mask |= ButtonPressMask | ButtonReleaseMask;
   }
#ifdef USE_XIL
   if (use_xil)
   {
      event_mask |= StructureNotifyMask;
   }
#endif
   XSelectInput (display, window, event_mask);

   XMapRaised (display, window);
   XClearWindow (display, window);
   XWindowEvent (display, window, ExposureMask, &event);
   
   /* create and setup the image */

   switch (x11_window_update_method)
   {
      case X11_XIL:
#ifdef USE_XIL
         /*
          *  XIL takes priority over MITSHM
          */
         setup_xil_images (image_width, image_height);
#endif
         break;
      case X11_MITSHM:
#ifdef USE_MITSHM
         /* Create a MITSHM image. */
         fprintf (stderr_file, "MIT-SHM Extension Available. trying to use... ");
         XSetErrorHandler (test_mit_shm);

         image = XShmCreateImage (display,
                                  xvisual,
                                  depth,
                                  ZPixmap,
                                  NULL,
                                  &shm_info,
                                  image_width,
                                  image_height);
         if (image)
         {
            shm_info.shmid = shmget (IPC_PRIVATE,
                                     image->bytes_per_line * image->height,
                                     (IPC_CREAT | 0777));
            if (shm_info.shmid < 0)
            {
               fprintf (stderr_file, "\nError: failed to create MITSHM block.\n");
               return OSD_NOT_OK;
            }

            /* And allocate the bitmap buffer. */
            /* new pen color code force double buffering in every cases */
            image->data = shm_info.shmaddr =
               (char *) shmat (shm_info.shmid, 0, 0);

            scaled_buffer_ptr = (unsigned char *) image->data;
            if (!scaled_buffer_ptr)
            {
               fprintf (stderr_file, "\nError: failed to allocate MITSHM bitmap buffer.\n");
               return OSD_NOT_OK;
            }

            shm_info.readOnly = FALSE;

            /* Attach the MITSHM block. this will cause an exception if */
            /* MIT-SHM is not available. so trap it and process         */
            if (!XShmAttach (display, &shm_info))
            {
               fprintf (stderr_file, "\nError: failed to attach MITSHM block.\n");
               return OSD_NOT_OK;
            }
            XSync (display, False);  /* be sure to get request processed */
            sleep (2);          /* enought time to notify error if any */
#ifdef USE_TIMER
            /* agh!! sleep() disables timer alarm. have to re-arm */
            if (play_sound)
               start_timer ();
#endif
            XSetErrorHandler (None);  /* Restore error handler to default */
            /* Mark segment as deletable after we attach.  When all processes
               detach from the segment (progam exits), it will be deleted. 
               This way it won't be left in memory if we crash or something.
               Grr, have todo this after X attaches too since slowlaris doesn't
               like it otherwise */
            shmctl(shm_info.shmid, IPC_RMID, NULL);
            
            /* if use_mit_shm is still set we've succeeded */
            if (use_mit_shm)
            {
               fprintf (stderr_file, "Success.\nUsing Shared Memory Features to speed up\n");
               mit_shm_attached = 1;
               break;
            }
            /* else we have failed clean up before retrying without MITSHM */
            shmdt ((char *) scaled_buffer_ptr);
            scaled_buffer_ptr = NULL;
            XDestroyImage (image);
            image = NULL;
         }
         fprintf (stderr_file, "Failed\nReverting to normal XPutImage() mode\n");
         x11_window_update_method = X11_NORMAL;
#endif
      case X11_NORMAL:
         scaled_buffer_ptr = malloc (4 * image_width * image_height);
         if (!scaled_buffer_ptr)
         {
            fprintf (stderr_file, "Error: failed to allocate bitmap buffer.\n");
            return OSD_NOT_OK;
         }
         image = XCreateImage (display,
                               xvisual,
                               depth,
                               ZPixmap,
                               0,
                               (char *) scaled_buffer_ptr,
                               image_width, image_height,
                               32, /* image_width always is a multiple of 8 */
                               0);

         if (!image)
         {
            fprintf (stderr_file, "OSD ERROR: could not create image.\n");
            return OSD_NOT_OK;
         }
         break;
      default:
         fprintf (stderr_file, "Error unknown X11 update method, this shouldn't happen\n");
         return OSD_NOT_OK;
   }
   
   /* verify the number of bits per pixel and choose the correct update method */
#ifdef USE_XIL
   if (use_xil)
      /* XIL uses 8 bit visuals and does any conversion it self */
      depth = 8;
   else
#endif
      depth = image->bits_per_pixel;
   fprintf(stderr_file, "Actual bits per pixel = %d... ", depth);
   if (video_16bit)
   {
      /* the 16bpp palette handling only does true color */
      if (xvisual->class != TrueColor)
      {
         fprintf(stderr_file, "\nX11-window: Warning: 16bpp modes not supported on non TrueColor visuals\n");
         return OSD_NOT_OK;
      }
      
      switch(depth)
      {
         case 16:
            x11_window_update_display_func = x11_window_update_16_to_16bpp;
            break;
         case 24:
            x11_window_update_display_func = x11_window_update_16_to_24bpp;
            xpixels_to_allocate = 32768;
            break;
         case 32:
            x11_window_update_display_func = x11_window_update_16_to_32bpp;
            xpixels_to_allocate = 32768;
            break;
      }
   }
   else
   {
      switch(depth)
      {
         case 8:
            x11_window_update_display_func = x11_window_update_8_to_8bpp;
            break;
         case 16:
            x11_window_update_display_func = x11_window_update_8_to_16bpp;
            break;
         case 24:
            x11_window_update_display_func = x11_window_update_8_to_24bpp;
            break;
         case 32:
            x11_window_update_display_func = x11_window_update_8_to_32bpp;
            break;
      }
      xpixels_to_allocate = 256;
   }
   
   if (x11_window_update_display_func == NULL)
   {
      fprintf(stderr_file, "Error: Unsupported\n");
      return OSD_NOT_OK;
   }
   fprintf(stderr_file, "Ok\n");
   
   if (xpixels_to_allocate &&
      (xpixel = calloc(xpixels_to_allocate, sizeof(unsigned long))) == NULL)
   {
      fprintf(stderr_file,
         "X11: Error: Couldn't allocate memory for the palette\n");
      return OSD_NOT_OK;
   }
   
   /* mouse pointer stuff */
   if (x11_grab_mouse)
      if (XGrabPointer (display, window, True, 0, GrabModeAsync,
                        GrabModeAsync, window, None, CurrentTime))
         x11_grab_mouse = FALSE;

   normal_cursor = XCreateFontCursor (display, XC_trek);
   invisible_cursor = create_invisible_cursor (display, window);

   if (x11_grab_mouse || !show_cursor)
      XDefineCursor (display, window, invisible_cursor);
   else
      XDefineCursor (display, window, normal_cursor);

   return OSD_OK;
}

/*
 * Shut down the display, also called by the core to clean up if any error
 * happens when creating the display.
 */
void x11_window_close_display (void)
{
   /* FIXME: free cursors */
   int i;

   widthscale  = orig_widthscale;
   heightscale = orig_heightscale;

   /* better free any allocated colors before freeing the colormap */
   if (use_rw_palette)
   {
      XFreeColors (display, colormap, xpixel, totalcolors, 0);
   }
   else
   {
      for (i = 0; i < 256; i++)
      {
         if (xpixel_allocated[i])
            XFreeColors (display, colormap, &xpixel[i], 1, 0);
      }
   }

   if (xpixel)
      free(xpixel);
   
   /* This is only allocated/done if we succeeded to get a window */
   if (window)
   {
      if (x11_grab_mouse)
         XUngrabPointer (display, CurrentTime);

#ifdef USE_MITSHM
      if (use_mit_shm)
      {
         if (mit_shm_attached)
            XShmDetach (display, &shm_info);
         if (scaled_buffer_ptr)
            shmdt (scaled_buffer_ptr);
         scaled_buffer_ptr = NULL;
      }
#endif
      if (image)
      {
         XDestroyImage (image);
         scaled_buffer_ptr = NULL;
      }
      if (scaled_buffer_ptr)
         free (scaled_buffer_ptr);
   
      XDestroyWindow (display, window);
   }

   if (private_cmap_allocated)
      XFreeColormap (display, colormap);

   XSync (display, False);      /* send all events to sync; */
}

/*
 * Set the screen colors using the given palette.
 *
 */
void x11_window_alloc_palette (void)
{
   int i;
   
   /* set xpixel to black */
   for (i = 0; i < 256; i++)
      xpixel[i] = black_xpixel;

   /* allocate color cells */
   if (XAllocColorCells (display, colormap, 0, 0, 0, xpixel, totalcolors))
   {
      use_rw_palette = 1;
      fprintf (stderr_file, "Using r/w palette entries to speed up, good\n");
      if (depth == 8)
      {
         int i;

         for (i = 0; i < totalcolors; i++)
            if (xpixel[i] != i) break;
         
         if (i == totalcolors)
         {
            x11_window_update_display_func = x11_window_update_8_to_8bpp_direct;
            fprintf (stderr_file, "Using direct copy to speed up, good\n");
         }
      }
   }
}

void x11_window_modify_pen (int pen, unsigned char red, unsigned char green, unsigned char blue)
{
   XColor color;

   /* Translate VGA 0-63 values of new color to X 0-65536 values. */
   color.flags = (DoRed | DoGreen | DoBlue);
   color.red = (int) red << 8;
   color.green = (int) green << 8;
   color.blue = (int) blue << 8;
   color.pixel = xpixel[pen];

   if (use_rw_palette)
   {
      XStoreColor (display, colormap, &color);
   }
   else
   {
      /* free previously allocated color */
      if (xpixel_allocated[pen])
      {
         XFreeColors (display, colormap, &xpixel[pen], 1, 0);
         xpixel_allocated[pen] = FALSE;
      }

      /* allocate new color and assign it to pen index */
      if (XAllocColor (display, colormap, &color))
      {
         if (xpixel[pen] != color.pixel)
            x_palette_dirty = TRUE;
         xpixel[pen] = color.pixel;
         xpixel_allocated[pen] = TRUE;
      }
      else
      {
         if (warn_low_on_colors)
         {
            warn_low_on_colors = 0;
            fprintf (stderr_file,
                     "Couldn't allocate all colors, some parts of the emulation may be black\n"
                     "Try running xmame with the -privatecmap option\n");
         }
         /* If color allocation failed, use black to ensure the
            pen is not left set to an invalid color */
         xpixel[pen] = black_xpixel;
      }
   }
}

/* invoked by main tree code to update bitmap into screen */
void x11_window_update_display (void)
{
   int old_use_dirty = use_dirty;
   
   if (x_palette_dirty)
   {
      use_dirty = 0;
      x_palette_dirty = FALSE;
   }

   (*x11_window_update_display_func) ();

#ifdef USE_XIL
   if (use_xil)
      refresh_xil_screen ();
#endif

   use_dirty = old_use_dirty;

   if (keyboard_pressed (KEYCODE_LALT) &&
       keyboard_pressed_memory (KEYCODE_PGDN))
   {
      if (x11_grab_mouse)
      {
         XUngrabPointer (display, CurrentTime);
         if (show_cursor)
            XDefineCursor (display, window, normal_cursor);
         x11_grab_mouse = FALSE;
      }
      else
      {
         if (!XGrabPointer (display, window, True, 0, GrabModeAsync,
                            GrabModeAsync, window, None, CurrentTime))
         {
            if (show_cursor)
               XDefineCursor (display, window, invisible_cursor);
            x11_grab_mouse = TRUE;
         }
      }
   }

   /* some games "flickers" with XFlush, so command line option is provided */
   if (use_xsync)
      XSync (display, False);   /* be sure to get request processed */
   else
      XFlush (display);         /* flush buffer to server */
}

void x11_window_refresh_screen (void)
{
   switch (x11_window_update_method)
   {
      case X11_XIL:
#ifdef USE_XIL
         refresh_xil_screen ();
#endif
         break;
      case X11_MITSHM:
#ifdef USE_MITSHM
         XShmPutImage (display, window, gc, image, 0, 0, 0, 0, image->width,
                       image->height, FALSE);
#endif
         break;
      case X11_NORMAL:
         XPutImage (display, window, gc, image, 0, 0, 0, 0, image->width,
                    image->height);
         break;
   }
}

INLINE void x11_window_put_image (int x, int y, int width, int height)
{
   switch (x11_window_update_method)
   {
      case X11_XIL:
         /* xil doesn't need a put_image */
         break;
      case X11_MITSHM:
#ifdef USE_MITSHM
         XShmPutImage (display, window, gc, image, x, y, x, y, width, height,
                       FALSE);
#endif
         break;
      case X11_NORMAL:
         XPutImage (display, window, gc, image, x, y, x, y, width, height);
         break;
   }
}

#define DEST_WIDTH image_width
#define DEST scaled_buffer_ptr
#define SRC_PIXEL unsigned char
#define PUT_IMAGE(X, Y, WIDTH, HEIGHT) x11_window_put_image(X, Y, WIDTH, HEIGHT);

static void x11_window_update_8_to_8bpp_direct (void)
{
#define DEST_PIXEL unsigned char
#include "blit.h"
#undef DEST_PIXEL
}

#define INDIRECT xpixel

static void x11_window_update_8_to_8bpp (void)
{
#define DEST_PIXEL unsigned char
#include "blit.h"
#undef DEST_PIXEL
}

static void x11_window_update_8_to_16bpp (void)
{
#define DEST_PIXEL unsigned short
#include "blit.h"
#undef DEST_PIXEL
}

#define DEST_PIXEL unsigned int

static void x11_window_update_8_to_24bpp (void)
{
#define PACK_BITS
#include "blit.h"
#undef PACK_BITS
}

static void x11_window_update_8_to_32bpp (void)
{
#include "blit.h"
}

#undef  DEST_PIXEL
#undef  SRC_PIXEL
#undef  INDIRECT
#define SRC_PIXEL unsigned short

static void x11_window_update_16_to_16bpp (void)
{
#define DEST_PIXEL unsigned short
#include "blit.h"
#undef DEST_PIXEL
}

#define INDIRECT xpixel
#define DEST_PIXEL unsigned int

static void x11_window_update_16_to_24bpp (void)
{
#define PACK_BITS
#include "blit.h"
#undef PACK_BITS
}

static void x11_window_update_16_to_32bpp (void)
{
#include "blit.h"
}

#undef  DEST_PIXEL

#endif /* ifdef x11 */
