/*
 * X-mame config-file and commandline parsing
 * We don't use stderr_file resp stdout_file in here since we don't know if 
 * it's valid yet.
 */

#define __CONFIG_C_
#ifdef MAME_NET
#include "network.h"
#endif /* MAME_NET */
#include "xmame.h"
#include "sound.h"
#include "driver.h"

/* some prototypes */
int parse_command_line (int argc, char *argv[], int get_gamename);
int parse_xmamerc_file();
int check_and_create_dir(char *);
int set_default_options (void);
int verify_options(void);

/* from ... */
extern char *cheatfile;
extern char *history_filename;
extern char *mameinfo_filename;

/* some local vars */
static int showconfig = 0;
static int showusage  = 0;
static int list       = 0;
static int ident      = 0;
static int showclones = 1;
static char *gamename = "";
static float beam_f, flicker_f;
#ifndef MESS
static char *defaultgamename;
#endif

/* option-struct */
enum option_type { set_bool, clear_bool, string, integer, set_int,
   seperator, floater, file, inputlog, use_function, special_rompath };

typedef int(*config_function)(char *arg);

typedef struct {
   char *name;
   int  type;
   void *dest;
   int  default_int; /* abused for the value to set with set_int, and for write flag with file */
   void *default_ptr;
} option;

static int config_scale(char *arg)
{
   widthscale_f  = atof(arg);
   heightscale_f = atof(arg);
   if (widthscale_f == 0.0)
      return OSD_NOT_OK;
   return OSD_OK;
}

static option opts[] = {
   { "### Input Related: ###", seperator, NULL,			0,		NULL},
   { "joytype",		integer,	&joytype,		0,		NULL},
   { "paddevname",	string,		&towns_pad_dev,		0,		"/dev/pad00"},
#ifdef X11_JOYNAME
   { "x11joyname",	string,		&x11joyname,		0,		X11_JOYNAME},
#else
   { "x11joyname",	string,		&x11joyname,		0,		""},
#endif
   { "analogstick",	set_bool,	&analogstick,		FALSE,		NULL},
   { "noanalogstick",	clear_bool,	&analogstick,		FALSE,		NULL},
   { "mouse",		set_bool,	&use_mouse,		TRUE,		NULL},
   { "nomouse",		clear_bool,	&use_mouse,		TRUE,		NULL},
   { "analogmouse",	set_bool,	&analogmouse,		FALSE,		NULL},
   { "noanalogmouse",	clear_bool,	&analogmouse,		FALSE,		NULL},
   { "grabmouse",	set_bool,	&x11_grab_mouse,	FALSE,		NULL},
   { "nograbmouse",	clear_bool,	&x11_grab_mouse,	FALSE,		NULL},
   { "winkeys",		set_bool,	&use_winkeys,		FALSE,		NULL},
   { "nowinkeys",	clear_bool,	&use_winkeys,		FALSE,		NULL},
   { "mapkey",		use_function,	sysdep_mapkey,		0,		NULL},
   { "### Network Related: ###", seperator, NULL,		0,		NULL},
   { "master",		integer,	&players,		0,		NULL},
   { "slave",		string,		&mastername,		0,		NULL},
   { "netmapkey",	set_bool,	&netkeymap,		FALSE,		NULL},
   { "### Video Related: ###", seperator, NULL,			0,		NULL},
   { "16bitvideo",	set_bool,       &video_16bit,	   	TRUE,   	NULL},
   { "no16bitvideo",	clear_bool,     &video_16bit,	   	TRUE,   	NULL},
   { "heightscale",	floater,	&heightscale_f,		1,		NULL},
   { "widthscale",	floater,	&widthscale_f,		1,		NULL},
   { "scale",		use_function,	config_scale,		0,		NULL},
   { "vectorres",	string,		&vector_res,		0,		NULL},
   { "autodouble",	set_bool,	&use_auto_double,	TRUE,		NULL},
   { "noautodouble",	clear_bool,	&use_auto_double,	TRUE,		NULL},
   { "keepaspect",	set_bool,	&use_aspect_ratio,	TRUE,		NULL},
   { "nokeepaspect",	clear_bool,	&use_aspect_ratio,	TRUE,		NULL},
   { "displayaspectratio", floater,	&display_aspect_ratio,	0,		NULL},
   { "geometry",	string,		&geometry,		0,		"640x480"},
   { "cursor",		set_bool,       &show_cursor,   	TRUE,   	NULL},
   { "nocursor",	clear_bool,     &show_cursor,   	TRUE,   	NULL},
   { "mitshm",		set_bool,	&use_mit_shm,		TRUE,		NULL},
   { "nomitshm",	clear_bool,	&use_mit_shm,		TRUE,		NULL},
   { "xsync",		set_bool,	&use_xsync,		TRUE,		NULL},
   { "noxsync",		clear_bool,	&use_xsync,		TRUE,		NULL},
   { "privatecmap",	set_bool,	&use_private_cmap, 	FALSE,		NULL},
   { "noprivatecmap",	clear_bool,	&use_private_cmap, 	FALSE,		NULL},
   { "xil",		set_bool,	&use_xil,		TRUE,		NULL},
   { "noxil",		clear_bool,	&use_xil,		TRUE,		NULL},
   { "mtxil",		set_bool,	&use_mt_xil,		FALSE,		NULL},
   { "nomtxil",		clear_bool,	&use_mt_xil,		FALSE,		NULL},
   { "x11-mode",	integer,	&x11_video_mode,	0,		NULL},
   { "root_window_id",	integer,	&root_window_id,	0,		NULL},
   { "tweak",		set_bool,	&use_tweak,		FALSE,		NULL},
   { "notweak",		clear_bool,	&use_tweak,		FALSE,		NULL},
   { "planar",		set_bool,	&use_planar,		TRUE,		NULL},
   { "noplanar",	clear_bool,	&use_planar,		TRUE,		NULL},
   { "linear",		set_bool,	&use_linear,		FALSE,		NULL},
   { "nolinear",	clear_bool,	&use_linear,		FALSE,		NULL},
   { "disablemode",	use_function,	mode_disable,		0,		NULL},
   { "dirty",		set_bool,	&use_dirty,		TRUE,		NULL},
   { "nodirty",		clear_bool,	&use_dirty,		TRUE,		NULL},
   { "scanlines",	set_bool,	&use_scanlines,		FALSE,		NULL},
   { "noscanlines",	clear_bool,	&use_scanlines,		FALSE,		NULL},
   { "artwork",		set_bool,	&options.use_artwork,	TRUE,		NULL},
   { "noartwork",	clear_bool,	&options.use_artwork,	TRUE,		NULL},
   { "frameskipper",	integer,	&frameskipper,		0,		NULL},
   { "throttle",	set_bool,	&throttle,		TRUE,		NULL},
   { "nothrottle",	clear_bool,	&throttle,		TRUE,		NULL},
   { "sleepidle",	set_bool,	&sleep_idle,		FALSE,		NULL},
   { "nosleepidle",	clear_bool,	&sleep_idle,		FALSE,		NULL},
   { "autoframeskip",	set_bool,	&autoframeskip,		TRUE,		NULL},
   { "noautoframeskip",	clear_bool,	&autoframeskip,		TRUE,		NULL},
   { "maxautoframeskip",integer,        &max_autoframeskip,     7,              NULL},    
   { "frameskip",	integer,	&frameskip,		0,		NULL},
   { "brightness",	integer,	&brightness,		100,		NULL},
   { "gamma-correction",floater,	&gamma_correction, 	1,		NULL},
   { "norotate",	set_bool,	&options.norotate,	FALSE,		NULL},
   { "ror",		set_bool,	&options.ror,		FALSE,		NULL},
   { "rol",		set_bool,	&options.rol,		FALSE,		NULL},
   { "flipx",		set_bool,	&options.flipx,		FALSE,		NULL},
   { "flipy",		set_bool,	&options.flipy,		FALSE,		NULL},
   { "### Vector Games: ###", seperator, NULL,			0,		NULL},
   { "beam",		floater,	&beam_f,		1,		NULL},
   { "flicker",		floater,	&flicker_f,		0,		NULL},
   { "antialias",	set_bool,	&options.antialias,	TRUE,		NULL},
   { "noantialias",	clear_bool,	&options.antialias,	TRUE,		NULL},
   { "translucency",	set_bool,	&options.translucency,	TRUE,		NULL},
   { "notranslucency",	clear_bool,	&options.translucency,	TRUE,		NULL},
   { "### GLmame: ###", seperator,	NULL,			0,		NULL},
   { "dblbuffer",	set_bool,	&doublebuffer,		TRUE,		NULL},
   { "nodblbuffer",	clear_bool,	&doublebuffer,		TRUE,		NULL},
   { "cabview",		set_bool,	&cabview,		TRUE,		NULL},
   { "fullview",	clear_bool,	&cabview,		TRUE,		NULL},
   { "cabinet",		string,		&cabname,		0,		"glmame"},
   { "### Sound Related: ###", seperator, NULL,			0,		NULL},
   { "sound",		set_bool,	&play_sound,		FALSE,		NULL},
   { "nosound",		clear_bool,	&play_sound,		FALSE,		NULL},
   { "samples",		set_bool,	&options.use_samples,	TRUE,		NULL},
   { "nosamples",	clear_bool,	&options.use_samples,	TRUE,		NULL},
   { "audiodevice",	string,		&audiodevice,		0,		"/dev/dsp"},
   { "samplefreq",	integer,	&options.samplerate, 	AUDIO_SAMPLE_FREQ, NULL},
   { "timerfreq",	integer,	&audio_timer_freq, 	AUDIO_TIMER_FREQ, NULL},
   { "8bit",		set_bool,	&sound_8bit,		FALSE,		NULL},
   { "16bit",		clear_bool,	&sound_8bit,		FALSE,		NULL},
   { "stereo",		set_bool,	&sound_stereo,		TRUE,		NULL},
   { "mono",		clear_bool,	&sound_stereo,		TRUE,		NULL},
   { "volume",		integer,	&attenuation,		-3,		NULL},
   { "fakesound",	set_bool,	&fake_sound,		FALSE,		NULL},
   { "fragsize",	integer,	&frag_size,		512,		NULL},
   { "numfrags",	integer,	&num_frags,		5,		NULL},
   { "### General: ###",seperator,	NULL,			0,		NULL},
#ifndef MESS   
   { "defaultgame",	string,		&defaultgamename, 	0,		"pacman"},
#endif
   { "rompath",		special_rompath,rompath,		0,		XMAMEROOT},
   { "spooldir",	string,		&spooldir,		0,		XMAMEROOT},
   { "screenshotdir",	string,		&screenshot_dir,	0,		"."},
   { "cheatfile",	string,		&cheatfile,		0,		XMAMEROOT"/cheat.dat"},
   { "historyfile",	string,		&history_filename, 	0,		XMAMEROOT"/history.dat"},
   { "mameinfofile",	string,		&mameinfo_filename, 	0,		XMAMEROOT"/mameinfo.dat"},
   { "record",		inputlog,	&options.record, 	1,		NULL},
   { "playback",	inputlog,	&options.playback,	0,		NULL},
   { "stdout-file",	file,		&stdout_file,		1,		NULL},
   { "stderr-file",	file,		&stderr_file,		1,		NULL},
   { "log",		file,		&options.errorlog,	1,		NULL},
   { "cheat",		set_bool,	&options.cheat,		FALSE,		NULL},
   { "nocheat",		clear_bool,	&options.cheat,		FALSE,		NULL},
   { "debug",		set_bool,	&options.mame_debug,	FALSE,		NULL},
   { "ident",		set_int,	&ident,			IDENT_IDENT,	NULL},
   { "isknown",		set_int,	&ident,			IDENT_ISKNOWN,	NULL},
   { "list",		set_int,	&list,			LIST_LIST,	NULL},
   { "listfull",	set_int,	&list,			LIST_FULL,	NULL},
   { "listgames",	set_int,	&list,			LIST_GAMES,	NULL},
   { "listclones",	set_int,	&list,			LIST_CLONES,	NULL},
   { "listdetails",	set_int,	&list,			LIST_DETAILS,	NULL},
   { "listinfo",        set_int,        &list,  	        LIST_INFO,      NULL},
   { "listroms",	set_int,	&list,			LIST_ROMS,	NULL},
   { "listsamples",	set_int,	&list,			LIST_SAMPLES,	NULL},
   { "listsamdir",	set_int,	&list,			LIST_SAMDIR,	NULL},
   { "listcrc",		set_int,	&list,			LIST_CRC,	NULL},
   { "listdupcrc",	set_int,	&list,			LIST_DUPCRC,	NULL},
   { "listcolors",	set_int,	&list,			LIST_COLORS,	NULL},
   { "lmr",		set_int,	&list,			LIST_LMR,	NULL},
   { "wrongorientation",set_int,	&list,			LIST_WRONGORIENTATION, NULL},
   { "listwrongmerge",	set_int,	&list,			LIST_WRONGMERGE,NULL},
   { "wrongfps",	set_int,	&list,			LIST_WRONGFPS,	NULL},
   { "verifyroms",	set_int,	&list,			VERIFY_ROMS,	NULL},
   { "verifysamples",	set_int,	&list,			VERIFY_SAMPLES,	NULL},
   { "noclones",	clear_bool,	&showclones,		TRUE,		NULL},
/* these are bools, but use set_int so they don't show up in -showconfig */
   { "showconfig",	set_int,	&showconfig,		TRUE,		NULL},
   { "h",		set_int,	&showusage,		TRUE,		NULL},
   { "?",		set_int,	&showusage,		TRUE,		NULL},
   { "help",		set_int,	&showusage,		TRUE,		NULL},
   { "-help",		set_int,	&showusage,		TRUE,		NULL},
   { NULL,		0,		NULL,			0,		NULL}
};

static INP_HEADER inp_header = { "", "", "" };

/*
 * get configuration from configfile and env.
 */
int get_config (int argc, char *argv[])
{
   char *pt;
   
   if (set_default_options()              != OSD_OK) return OSD_NOT_OK;
   
   if (check_and_create_dir((char *)NULL) != OSD_OK) return OSD_NOT_OK;
   if (check_and_create_dir("cfg")        != OSD_OK) return OSD_NOT_OK;
   if (check_and_create_dir("mem")        != OSD_OK) return OSD_NOT_OK;
   if (check_and_create_dir("sta")        != OSD_OK) return OSD_NOT_OK;
   if (check_and_create_dir("rc")         != OSD_OK) return OSD_NOT_OK;

   /* only calling parse_command_line() to get gamename for
      parse_xmamerc_file() */
   if (parse_command_line(argc, argv, 1)  != OSD_OK) return OSD_NOT_OK;
   if (parse_xmamerc_file()               != OSD_OK) return OSD_NOT_OK;

   /* get environment variables. This overrides xmamerc options */
   if ( (pt=getenv("ROMPATH"))  ) strncpy(rompath, pt, MAXPATHL-1);

   if (parse_command_line(argc, argv, 0)  != OSD_OK) return OSD_NOT_OK;
   
   init_rom_path();

   return verify_options();
}

/* 
 * check and if nescesarry create ${HOME}/.xmame (or .xmess).
 */
int check_and_create_dir(char *dir)
{
   char name[MAXPATHL];
   struct stat stat_buffer;

   /* check for existence of ${HOME}/.xmame/<dir> / ${HOME}/.xmess/<dir> */
   if (!dir) 
     sprintf(name, "%s/.%s", home_dir, NAME);
   else 
     sprintf(name, "%s/.%s/%s", home_dir, NAME, dir);

   if (stat(name, &stat_buffer))
   {
      /* error check if it doesn't exist or something else is wrong */
      if (errno == ENOENT)
      {
         /* doesn't exist letts create it ;) */
#ifdef BSD43
	 if (mkdir(name, 0775))
#else
         if (mkdir(name, S_IRWXU|S_IRWXG|S_IROTH|S_IXOTH))
#endif
         {
            fprintf(stderr, "Error creating dir %s", name);
            perror(" ");
            return OSD_NOT_OK;
         }
      }
      else
      {
         /* something else went wrong yell about it */
         fprintf(stderr, "Error opening %s", name);
         perror(" ");
         return OSD_NOT_OK;
      }
   }
   else
   {
      /* file exists check it's a dir otherwise yell about it */
#ifdef BSD43
      if(!(S_IFDIR & stat_buffer.st_mode))
#else
      if(!S_ISDIR(stat_buffer.st_mode))
#endif
      {
         fprintf(stderr,"Error %s exists but isn't a dir\n", name);
         return OSD_NOT_OK;
      }
   }
   return OSD_OK;
}

/* 
 * show help and exit
 */
void show_usage(void) 
{
char message[]="%s\n"
#ifdef MESS
  "Usage: xmess <system> [game] [options]\n"
#else
  "Usage: xmame [game] [options]\n"
#endif 
  "Options:\n"
  "\n"
  "*** Input ***\n"
  " -joytype <type>          Select type of joysticksupport to use:\n"
  "                          0 No joystick\n"
  "                          1 i386 style joystick driver (if compiled in)\n"
  "                          2 Fm Town Pad support (if compiled in)\n"
  "                          3 X11 input extension joystick (if compiled in)\n"
  "                          4 new i386 linux 1.x.x joystick driver(if compiled in)\n"
  " -paddevname <name>       Name of pad device (defaults to /dev/pad00)\n"
#ifdef x11
  " -x11joyname <name>       Name of X-based joystick device (if compiled in)\n"
#endif
  " -[no]analogstick         Use Joystick as analog for analog controls\n"
  " -[no]mouse               Disable/enable mouse (if supported)\n"
  " -[no]analogmouse         Use mouse to simulate analog controls\n"
#ifdef x11
  " -[no]grabmouse           Disable/enable mousegrabbing (also alt + pagedown)\n"
  " -[no]winkeys             Disable/enable mapping of windowskeys under X\n"
  " -mapkey <Xcode>,<Mcode>  Set a specific key mapping, see xmamerc.dist\n"
#endif
  "\n"
#ifdef MAME_NET
 "*** Network ***\n"
  " -master <no of players>  Enable master mode. Set number of players\n"
  " -slave  <hostname>       Enable slave mode. Set master hostname\n"
  " -netmapkey               All players use the player 1 keys.\n"
  "                          To use with * real* multiplayer games\n"
  "\n"
#endif /* MAME_NET */
  "*** Video ***\n"
  " -[no]16bitvideo          Use 16 bit video if available\n"
  " -heightscale<scale> (*)  Set Y-Scale aspect ratio\n"
  " -widthscale <scale> (*)  Set X-Scale aspect ratio\n"
  " -scale      <scale> (*)  Set X-Y Scale to the same aspect ratio\n"
  " (*)                      For vector games scale may have value's like 1.5\n"
  "                          and even 0.5. For scaling of regular games this will\n"
  "                          be rounded to an integer.\n"
  " -vectorres <XresxYres>   Scale vectorgames to XresxYres, keeping their aspect\n"
  "                          ratio. This overrides the scale options.\n"
  " -[no]autodouble          Disable/enable automatic scale doubling for 1:2\n"
  "                          pixel aspect ratio games\n"
  " -[no]keepaspect          Try to keep the aspect ratio of a game when selecting\n"
  "                          the best videomode\n"
  " -displayaspectratio <f>  Set the display aspect ratio of your monitor. This\n"
  "                          is used for -keepaspect The default = 1.33 (4/3).\n"
  "                          Use 0.75 (3/4) for a portrait monitor.\n"
  " -geometry <geometry>     Specify the size and location of the window/screen\n"
#ifdef x11
  " -[no]cursor              Disable/enable showing the cursor\n"
  " -[no]mitshm              Disable/enable MIT Shared Mem (if supported)\n"
  " -[no]xsync               Select XFlush/Xsync screen refresh method\n"
  " -[no]privatecmap         Disable/enable use of private color maps\n"
#ifdef USE_XIL
  " -[no]xil                 Disable/enable XIL for scaling (default if available)\n"
  " -[no]mtxil               Disable/enable running XIL threaded\n"
#endif
  " -x11-mode <mode>         Select x11 video mode: (if compiled in)\n"
  "                          0 Normal window  (hotkey left-alt + insert)\n"
  "                          1 Fullscreen DGA (hotkey left-alt + home)\n"
  " -root_window_id <id>     Create the xmame-window in an alternate\n"
  "                          root-window, only usefull for frontends!\n"
#endif /* ifdef x11 */
#ifdef svgalib
  " -[no]tweak               Disable/enable svgalib tweaked video modes\n"
  " -[no]planar              Disable/enable use of planar (modeX) modes (slow)\n"
  " -[no]linear              Disable/enable use of linear framebuffer (fast)\n"
  " -disablemode <XresxYres> Don't use mode Xres x Yres this can be used to\n"
  "                          disable specific modes which don't work on your\n"
  "                          system. This option may be used more then once\n"
  " -disablemode <XresxYresxDepth> The same as above but only disable mode for\n"
  "                          a specific colordepth. Accepted depths: 256, 65536\n"
#endif
  " -[no]dirty               Disable/enable use of dirty rectangles\n"
  " -[no]scanlines           Disable/enable displaying simulated scanlines\n"
  " -[no]artwork             Use/don't use artwork if available\n"
  " -[no]throttle            Disable/enable throttle\n"
  " -[no]sleepidle           Disable/enable sleep during idle\n"   
  " -[no]autoframeskip       Disable/enable autoframeskip\n"
  " -frameskipper <#>        Select which autoframeskip and throttle routines\n"
  "                          to use. Available choices are:\n"
  "                          0   Dos frameskip code\n"
  "                          1   Enhanced frameskip code by William A. Barath\n"
  " -maxautoframeskip <#>    Set highest frameskip for autoframeskip\n"
  " -frameskip  <#frames>    Skip <#frames> in video refresh\n"
  " -brightness <brightness> Set the brightness (0-100%%)\n"
  " -gamma-correction<gamma> Set the gamma-correction (0.5-2.0)\n"
  " -norotate                Disable rotation\n"
  " -ror                     Rotate display 90 degrees rigth\n"
  " -rol                     Rotate display 90 degrees left\n"
  " -flipx                   Flip X axis\n"
  " -flipy                   Flip Y axis\n"
  "\n"
  "*** Vector Games ***\n"
  " -beam                    Set the beam size for vector games\n"
  " -flicker                 Set the flicker for vector games\n"
  " -[no]antialias           Disable/enable antialiasing\n"
  " -[no]translucency        Disable/enable translucency\n"
  "\n"
#ifdef xgl
  "*** GLmame ***\n"
  " -[no]dblbuffer           Disable/enable double buffering\n"
  " -cabview                 Start in cabinet view mode\n"
  " -fullview                Start in fullscreen view mode\n"
  " -cabinet <cabname>       Use the cabinet model <cabname>\n"
  "\n"
#endif /* ifdef xgl */
  "*** Sound ***\n"
  " -[no]sound               Disable/enable sound (if available)\n"
  " -[no]samples             Use/don't use samples if available\n"
  " -audiodevice <device>    Use an alternative audiodevice instead of /dev/dsp\n"
  " -samplefreq <samplefreq> Set the playback sample-frequency/rate\n"
  " -timerfreq <timerfreq>   Set the timer frequency (if timer based audio)\n"
  " -8bit                    Mix down to 8bit audio\n"
  " -16bit                   Play sound in 16 bit (default)\n"
  " -mono                    Output mono sound\n"
  " -stereo                  Output stereo sound (default)\n"
  " -volume <db>             Set volume to <db> db, (-24 (soft) - 0(loud) )\n"
  " -fakesound               Generate sound even when sound is disabled, this is\n"
  "                          needed for some games which won't run without sound\n"
  " -fragsize <size> (*)     Set the size of the soundbuffers in bytes,\n"
  "                          Default=512, lower this if your sound seems lagged,\n"
  "                          raise it if your sound clicks. Note this defines the\n"
  "                          buffersize used with samplefreq=22050, the buffersize\n"
  "                          is automaticly scaled for other samplefrequencies\n"
  " -numfrags <num> (*)      Nummer of the above buffers to allocate, Default 5\n"
  "                          The default should be okay for everyone.\n"
  " (*)                      These options only applies to linux/freebsd, under\n"
  "                          other os's/archs these options only control xmame's\n"
  "                          own buffer, see doc/README.unix for details\n"
  "\n"
  "*** General ***\n"
  " -rompath     <path>      Add <path> to the rom search path\n"
  " -spooldir    <dir>       Store high scores in <dir>\n"
  " -screenshotdir <dir>     Store screenshots in <dir>\n"
  " -cheatfile   <filename>  Use <filename> as cheat database\n"
  " -historyfile <filename>  Use <filename> as history database\n"
  " -mameinfofile <filename> Use <filename> as mameinfo database\n"
  " -record      <filename>  Record keypresses into <filename>\n"
  " -playback    <filename>  Playback keypresses from <filename>\n"
  " -stderr-file <filename>  Redirect stderr to <filename>\n"
  " -stdout-file <filename>  Redirect stdout to <filename>\n"
  " -log         <filename>  Log debug info to <filename>\n"
  " -[no]cheat               Disable/enable cheat mode if game supports it\n"
  " -ident       <filename>  Identify unknown romdump <filename>, or unknown\n"
  "                          romdumps in dir/zip <filename>.\n"
  " -isknown     <filename>  Check if <filename> or files in <filename> are\n"
  "                          known\n"
  " -list          [regexp]  List supported games matching regexp, or all\n"
  " -listfull      [regexp]  List supported games with full description\n"
  " -listgames     [regexp]  List supported games with manufacturer and year\n"
  " -listclones    [regexp]  List clones of games matching regexp, or all\n"
  " -listdetails   [regexp]  Detailed list of supported games\n"
  " -listinfo                List all available info on drivers\n"
  " -listroms      [regexp]  List used ROMS\n"
  " -listsamples   [regexp]  List used audio samples\n"
  " -listsamdir    [regexp]  List dir where samples are taken from\n"
  " -listcrc       [regexp]  List used ROMS with crc\n" 
  " -listdupcrc    [regexp]  List ROMS with identical crc\n" 
  " -verifyroms    [regexp]  Verify ROMS for games matching regexp\n"
  " -verifysamples [regexp]  Verify audio samples for games matching regexp\n"
  " [regexp]                 Optional, May contain * and ? wildcards\n"
  " -noclones                Don't show bootlegs/clones in list commands\n"
  " -showconfig              Display Running parameters in rc style\n"
  " -help | -h | -?          Show this help\n"
  "\n"
  "Files: \n"
  " %s Global configuration file\n"
  " ${HOME}/.%s/%src   User global configuration file\n"
  " ${HOME}/.%s/<game>rc  Game-dependent user configuration file\n"
  "\n"
  "Environment variables:\n"
  "\n"
  " ROMPATH                  Rom search path\n"
  "\n"
#ifdef MESS
  "M.E.S.S.  -  Multi-Emulator Super System\n"
  "Copyright (C) 1998  by the MESS team\n"
#else
  "Mame is an Original Idea of Nicola Salmoria and Mirko Buffoni\n"
#endif
  "%s port maintained by Hans de Goede\n"
  "";
  fprintf(stdout_file,message,title,XMAMEROOT"/"NAME"rc",NAME,NAME,NAME,NAME);
}

/*
 * show running parameters in xmamerc / xmessrc style format ( using stdout_file )
 */
void show_config(void)
{
   int i;
   char blank[20];
   
   fprintf(stdout_file,"##### %s Running parameters: #####\n",title);
   fprintf(stdout_file,"#\n");
   
   for(i=0;opts[i].name;i++)
   {
      memset(blank, ' ', 20);
      switch(opts[i].type)
      {
         case set_bool:
            blank[19 - strlen(opts[i].name)] = 0;
            fprintf(stdout_file, "%s%s%d\n", opts[i].name, blank,
               (*(int *)opts[i].dest)? 1:0);
            break;
         case string:
            if (*(char **)opts[i].dest==NULL) break;
            blank[19 - strlen(opts[i].name)] = 0;
            fprintf(stdout_file, "%s%s%s\n", opts[i].name, blank,
               *(char **)opts[i].dest);
            break;
         case integer:
            blank[19 - strlen(opts[i].name)] = 0;
            fprintf(stdout_file, "%s%s%d\n", opts[i].name, blank,
               *(int *)opts[i].dest);
            break;
         case floater:
            blank[19 - strlen(opts[i].name)] = 0;
            fprintf(stdout_file, "%s%s%f\n", opts[i].name, blank,
               *(float *)opts[i].dest);
            break;
         case seperator:
            fprintf(stdout_file, "#\n%s\n#\n", opts[i].name);
            break;
         case special_rompath:
            blank[19 - strlen(opts[i].name)] = 0;
            fprintf(stdout_file, "%s%s%s\n", opts[i].name, blank,
               (char *)opts[i].dest);
            break;
      }
   }
}

/*
 * check for existence of ${HOME}/.xmame/xmamerc / ${HOME}/.xmess/xmessrc
 * if found parse it, else try XMAMEROOT/xmamerc / XMAMEROOT/xmessrc
 */
int parse_xmamerc_file()
{
	FILE            *file=(FILE *) NULL;
	int             lineno;
	char            buffer[2048];

	if (gamename[0])
	{
	    sprintf(buffer, "%s/.%s/rc/%src", home_dir, NAME, gamename);
	    if ((file = fopen(buffer,"r")) == NULL)
		fprintf(stderr,"File %s not found.\n",buffer);
        }
	if (file == NULL)
	{
	    /* No game specific config file found - trying xmamerc file */ 
	    sprintf(buffer, "%s/.%s/%src", home_dir, NAME, NAME);
	    file = fopen(buffer,"r");
	}
	if (file == NULL)
	{
	    fprintf(stderr,"File %s not found.\n",buffer);
	    /* No user config found - trying global file */
	    strcpy(buffer, XMAMEROOT"/"NAME"rc" );
	    if ( (file=fopen(buffer,"r")) == NULL)
	    {
		fprintf(stderr,"File %s not found.\n",buffer);
		return (OSD_OK);
	    }
	}
	fprintf(stderr, "Reading config options from %s...\n", buffer);
	lineno=0;
	while(fgets( buffer,2047,file) ) {
	    char *p;
	    char *(q[5]);
	    int i,j;
	    lineno++;
	    /* skip comments */
	    if ( ( p=strchr(buffer,'#') ) ) *p='\0';
	    /* scan for words */
	    for(i=0;i<5;i++) q[i]=(char *)NULL;
	    for(i=0,j=0,p=buffer; *p; p++ ) {
		if ( isspace(*p) ) { *p='\0'; j=0; }
		else               { if(!j++) q[i++]=p; }
	    }   /* end of stripping words ( for i=0 )*/
	    /* test for wrong number of args */
	    if ( i==0 ) continue; /* empty line */
	    if ( i!=2 ) { 
		    fprintf(stderr,"Warning for line %d: wrong number of parameters \n",lineno);
                    continue;
	    }
	    
	    for (i=0; opts[i].name; i++)
	    {
	       if (strcasecmp(q[0], opts[i].name) == 0) break;
	    }
	    
	    if (opts[i].name == NULL) /* unkown option */
	    {
	       fprintf(stderr,"Warning for line %d: unknown command %s\n",lineno,q[0]);
               continue;
	    }
	    
	    switch(opts[i].type)
	    {
	        case set_bool:
	        case integer:
	           *(int *)opts[i].dest = atoi(q[1]);
	           break;
	        case string:
	           *(char **)opts[i].dest = (char *)malloc(1+strlen(q[1]));
                   if(*(char **)opts[i].dest==NULL)
                   {
                        fprintf(stderr,"Malloc error: line %d\n",lineno);
                        return OSD_NOT_OK;
                   }
                   strcpy(*(char **)opts[i].dest, q[1]);
	           break;
	        case floater:
	           *(float *)opts[i].dest = atof(q[1]);
	           break;
	        case use_function:
	           if((*(config_function)opts[i].dest)(q[1]) != OSD_OK)
	              fprintf(stderr, "Warning for line %d: invalid argument %s\n",lineno,q[1]);
	           break;
	        case special_rompath:
	           strncpy((char *)opts[i].dest, q[1], MAXPATHL - 1);
	           break;
	        default:
	           fprintf(stderr, "Warning for line %d: unknown command %s\n",lineno,q[0]);
		   break;
	    }
	} /* while */
	fclose(file);
	return OSD_OK;
}

/*
 * Set all options to there defaults
 */
int set_default_options (void)
{
   int i;
   struct passwd   *pw;
   
   for(i=0;opts[i].name;i++)
   {
      switch(opts[i].type)
      {
         case set_bool:
         case clear_bool:
         case integer:
            *(int *)opts[i].dest = opts[i].default_int;
            break;
         case string:
            *(char **)opts[i].dest = (char *)opts[i].default_ptr;
            break;
         case floater:
            *(float *)opts[i].dest = opts[i].default_int;
            break;
         case file:
            *(FILE **)opts[i].dest = (FILE *)opts[i].default_ptr;
            break;
         case inputlog:
            *(void **)opts[i].dest = opts[i].default_ptr;
            break;
         case special_rompath:
            strncpy((char *)opts[i].dest, (char *)opts[i].default_ptr, MAXPATHL - 1);
            break;
      }
   }
   /* initialise non option var's (mostly video subsystem */
   bitmap		= NULL;
   snapshot_no		= 0;
   display_aspect_ratio = 4.0 / 3.0;
   options.no_fm	= TRUE;
   options.use_emulated_ym3812 = TRUE;
#ifdef MAME_NET
   netstate 		= 0;
#endif

   /* locate user's home directory */
   pw=getpwuid(getuid());
   if(!pw)
   { 
      fprintf(stderr, "Who are you? Not found in passwd database!!\n");
      return OSD_NOT_OK;
   }
   home_dir = malloc(strlen(pw->pw_dir)+1);
   if (!home_dir)
   {
      fprintf(stderr, "Error allocating memory\n");
      return OSD_NOT_OK;
   }
   strcpy(home_dir, pw->pw_dir);
   
   return OSD_OK;
}

int verify_options(void)
{
   int power, j;
   unsigned char lsb_test[2]={0,1};
   
   if (stderr_file == NULL) stderr_file = stderr;
   if (stdout_file == NULL) stdout_file = stdout;
   
   /* Lett's see of the endians of this arch is correct otherwise
      YELL about it and bail out. */
#ifdef LSB_FIRST
   if(*((unsigned short*)lsb_test) != 0x0100)
#else	
   if(*((unsigned short*)lsb_test) != 0x0001)
#endif
   {
      fprintf(stderr, "Error: Compiled byte ordering doesn't match machine byte ordering\n"
         "Are you sure you choose the right arch?\n"
#ifdef LSB_FIRST
         "Compiled for lsb-first try removing -DLSB_FIRST from DEFS.$(ARCH)\nin makefile.unix\n");
#else
         "Compiled for msb-first try adding -DLSB_FIRST to DEFS.$(ARCH)\nin makefile.unix\n");
#endif
      return OSD_NOT_OK;
   }
   
   if (showusage)
   {
      show_usage();
      return OSD_OK;
   }
   
   /* handle frontend options */
   if (ident)
   {
      if (gamename[0] == 0)
      {
         fprintf(stderr, "-ident / -isknow requires a game- or filename as second argument\n");
         return OSD_NOT_OK;
      }
      return frontend_ident(ident, gamename);
   }
   
   /* listclones is a special case since the strwildcmp */
   /* also has to be done on clone_of. */
   if (list == LIST_CLONES)
   {
      return frontend_list_clones(gamename);
   }
   
   if (list)
   {
      return frontend_list(list, gamename, showclones);
   }
   
   if (inp_header.name[0])
   {
      if(gamename[0])
      {
         if(strcmp(gamename, inp_header.name))
         {
            fprintf(stderr, "Error: Input file is for a different game as specified\n");
            return OSD_NOT_OK;
         }
         else
            fprintf(stderr, "Hint: with new .inp files you don't have to specify a game anymore\n");
      }
      gamename = inp_header.name;
   }

#ifndef MESS
   if (gamename[0] == 0) gamename = defaultgamename;
#endif

   /* do we have a drivers for this? */
   for (j = 0; drivers[j]; j++)
      if (strcasecmp(gamename,drivers[j]->name) == 0) break;

   if (drivers[j] == 0)
   {
      fprintf(stderr,"\"%s\" not supported\n", gamename);
      return OSD_NOT_OK;
   }
   else
   {
      game_index = j;
   }
   
   widthscale  = widthscale_f;
   heightscale = heightscale_f;
   if (widthscale  < 1) widthscale  = 1;
   if (heightscale < 1) heightscale = 1;
   if (widthscale_f  < 0.1) widthscale_f  = 1;
   if (heightscale_f < 0.1) heightscale_f = 1;
   
   if (frameskipper < 0 || frameskipper >= FRAMESKIP_DRIVER_COUNT)
      frameskipper = 0;
      
   if (max_autoframeskip < 0)
      max_autoframeskip = 0;
   else if (max_autoframeskip > (FRAMESKIP_LEVELS-1))
      max_autoframeskip = FRAMESKIP_LEVELS-1;
   
   for (power=0; (1 << power) < frag_size; power++) {}
   if ((1 << power)!=frag_size)
   {
      fprintf(stderr, "illegal fragsize (%d)\n", frag_size);
      fprintf(stderr, "fragsize should be a power of 2 (eg 512,1024,2048,etc)\n");
      return (OSD_NOT_OK);
   }
   if (options.samplerate < 8000 || options.samplerate > 48000) {
      fprintf (stderr,"illegal audio sample frequention (%d)\n", options.samplerate);
      return (OSD_NOT_OK);
   }
   
   options.beam = (int)(beam_f * 0x00010000);
   if (options.beam < 0x00010000)
      options.beam = 0x00010000;
   else if (options.beam > 0x00100000)
      options.beam = 0x00100000;
   options.flicker = (int)(flicker_f * 2.55);
   if      (options.flicker < 0)
      options.flicker = 0;
   else if (options.flicker > 255)
      options.flicker = 255;

   if      (brightness < 0)
      brightness = 0;
   else if (brightness > 100)
      brightness = 100;
   if      (gamma_correction < 0.5)
      gamma_correction = 0.5;
   else if (gamma_correction > 2.0)
      gamma_correction = 2.0;
   if      (attenuation < -32)
      attenuation = -32;
   else if (attenuation > 0)
      attenuation = 0;
   
#ifdef MAME_NET
   if (players) netstate = MASTER;
   if (players > 4)
   {
      fprintf(stderr, "4 players max.\n");
      return(OSD_NOT_OK);
   }
   if (mastername)
   {
      if(netstate==MASTER)
      {
         fprintf(stderr, "Error: can't be Slave and Master\n");
         return(OSD_NOT_OK);
      }
      netstate = SLAVE;
   }
#endif

   if (options.record)
   {
      memset(&inp_header, '\0', sizeof(INP_HEADER));
      strcpy(inp_header.name, drivers[game_index]->name);
      /* MAME32 stores the MAME version numbers at bytes 9 - 11
       * MAME DOS and xmame keeps this information in a string, the
       * Windows code defines them in the Makefile.
      inp_header.version[0] = 0;
      inp_header.version[1] = VERSION;
      inp_header.version[2] = BETA_VERSION;
      */
      osd_fwrite(options.record, &inp_header, sizeof(INP_HEADER));
   }

   if (showconfig)
   {
      show_config();
      return OSD_OK;
   }

   return 1234;
}

int parse_command_line (int argc, char *argv[], int get_gamename)
{
   int i, j, got_gamename	= 0;
#ifdef MESS
   int num_roms		= 0;
   int num_floppies	= 0;
#endif

   /* parse argument invocation. */
   for (i = 1;i < argc;i++)
   {
      if (argv[i][0]=='-')
      {
         for(j=0;opts[j].name;j++)
         {
            if (strcasecmp(opts[j].name, argv[i]+1) == 0) break;
         }
         
         if(opts[j].name==NULL) /* unknown option */
         {
            fprintf(stderr,"Unknown option %s. Try %s -help\n", argv[i], argv[0]);
            return OSD_NOT_OK;
         }
         
         switch(opts[j].type)
         {
            case set_bool:
               *(int *)opts[j].dest = TRUE;
               break;
            case clear_bool:
               *(int *)opts[j].dest = FALSE;
               break;
            case set_int:
               *(int *)opts[j].dest = opts[j].default_int;
               break;
            case integer: /* these all require an argument */
            case string:
            case floater:
            case file:
            case inputlog:
            case use_function:
            case special_rompath:
               i++;
               if (i==argc)
               {
                  fprintf (stderr, "Error: -%s requires an extra argument.\n", opts[j].name);
                  return OSD_NOT_OK;
               }
               if (!get_gamename)
               {
                  switch(opts[j].type)
                  {
                     case integer:
                        *(int *)opts[j].dest = atoi(argv[i]);
                        break;
                     case string:
                        *(char **)opts[j].dest = argv[i];
                        break;
                     case floater:
                        *(float *)opts[j].dest = atof(argv[i]);
                        break;
                     case file:
                        *(FILE **)opts[j].dest = fopen(argv[i],
                           (opts[j].default_int)? "w":"r");
                        if (*(FILE **)opts[j].dest == NULL)
                        {
                           fprintf(stderr, "Error: couldn't open %s\n", argv[i]);
                           return OSD_NOT_OK;
                        }
                        break;
                     case inputlog:
                        *(void **)opts[j].dest = osd_fopen(NULL, argv[i],
                           OSD_FILETYPE_INPUTLOG, opts[j].default_int);
                        if (*(void **)opts[j].dest == NULL)
                        {
                           fprintf(stderr, "Error: couldn't open %s\n", argv[i]);
                           return OSD_NOT_OK;
                        }
                        if (opts[j].default_int == 0)
                        {
                           /* we're opening an existing file,
                              read the playback header */
                           osd_fread(*(void **)opts[j].dest, &inp_header,
                              sizeof(INP_HEADER));
                           if (!isalnum(inp_header.name[0]))
                           {
                              /* old .inp file - no header */
                              osd_fseek(options.playback, 0, SEEK_SET); 
                              inp_header.name[0] = 0;
                           }
                        }
                        break;
                     case use_function:
                        if ( (*(config_function)opts[j].dest)(argv[i]) != OSD_OK)
                        {
                           fprintf(stderr, "Error: invalid argument for %s: %s\n",
                              argv[i-1], argv[i]);
                           return OSD_NOT_OK;
                        }
                        break;
                     case special_rompath:
                        strncat((char *)opts[j].dest, ":",
                           (MAXPATHL-1) - strlen((char *)opts[j].dest));
                        strncat((char *)opts[j].dest, argv[i],
                           (MAXPATHL-1) - strlen((char *)opts[j].dest));
                        break;
                  }
               }
               break;
         }
      }
      else
      {  /* if arrives here and not for the first time, 
            means syntax error */
         if (!got_gamename) /* notice: for MESS game means system */
         {
            gamename     = argv[i];
            got_gamename = 1;
         }
         else
#ifdef MESS
         {
            /* Is it a floppy or a rom? */
            if ((strstr(argv[i],".dsk")!=NULL) || (strstr(argv[i],".DSK")!=NULL))
            {
               if (num_floppies >= MAX_FLOPPY)
               {
                  fprintf(stderr, "Too many floppy names specified!\n");
                  return OSD_NOT_OK;
               }
               strcpy(options.floppy_name[num_floppies++],argv[i]);
            }
            else
            {
               if (num_roms >= MAX_ROM)
               {
                  fprintf(stderr, "Too many image names specified!\n");
                  return OSD_NOT_OK;
               }
               strcpy(options.rom_name[num_roms++],argv[i]);
            }
         }
#else
         {
            fprintf(stderr,"Duplicate gamename: %s\n",argv[i]);
            return OSD_NOT_OK;
         }
#endif
      }
   }

#ifdef MESS
   if (!got_gamename)
      showusage = TRUE;
   else if (!num_floppies && !num_roms && !get_gamename)
   {
      fprintf(stderr, "Warning:\n"
         "You didn't supply any rom/disk images on the commandline\n"
         "Most emulations of xmess will fail due to this, if xmess fails try:\n"
         "xmess <systemname> <rom/diskimage(s)> [options]\n"
         "Or try xmess -help / read README.unix for more info\n");
   }
#endif

   return OSD_OK;
}
