/*
Konami Twin16
The main processors are a pair of 68000 CPUs
Sounds are generated by a Z80, a Yamaha 2151 and 3012, a Konami custom IC and a UPD7759C

GX687	Dark Adventure / Devil World
GX785	Vulcan Venture / Gradius II
GX903	Cuebrick
GX870	Final Round

to do:
	hook up sprites
	scrolling
	finish mapping DIPs

Notes:
	Konami Twin16 version of Final Round exists, but the version supported by this driver isn't it.

	Final Round doesn't pass its GFX ROMs test.
	Cuebrick and Vulcan Venture seem to run fine
	Devils World locks up in what looks like a race condition

68000 Memory Map for Konami Twin System
	0x000000..0x03ffff	ROM
	0x040000..0x043fff	shared RAM
	0x060000..0x063fff	working RAM
	0x080000..0x080fff	CPUA: palette RAM
	0x080000..0x09ffff	CPUB: extra tile data (Dark Adventure/Devil's World only)
	0x0a0000..0x0a0001	IRQ control, gfx ROM bank select
	0x0a0010..0xa00011	watchdog

	0x0c0000..0x0c0001	bits0,1 -> screenflip
	0x0c0002..0x0c0003	scroll
	0x0c0004..0x0c0005
	0x0c0006..0x0c0007	scroll
	0x0c0008..0x0c0009	scroll
	0x0c000a..0x0c000b	scroll
	0x0c000c..0x0c000d
	0x0c000e..0x0c000f

	0x100000..0x103fff	CPUA: videoram fixed character-mapped foreground (character data in ROM)
	0x120000..0x123fff	CPUA: videoram fixed character-mapped background (character data in RAM)
	0x140000..0x143fff	spriteram
	0x400000..0x403fff	CPUB: videoram (scrolling?)
	0x480000..0x483fff	CPUB: videoram (scrolling?)
	0x500000..0x53ffff	tile gfx RAM
	0x600000..0x77ffff	sprite ROM
	0x780000..0x79ffff	sprite gfx RAM?
*/

#include "driver.h"
#include "vidhrdw/generic.h"
#include "cpu/m68000/m68000.h"
#include "cpu/z80/z80.h"

/*extern void tmnt_paletteram_w(int offset,int data); */
static void twin16_paletteram_w(int offset,int data)
{
	int oldword = READ_WORD(&paletteram[offset]);
	int newword = COMBINE_WORD(oldword,data);
	WRITE_WORD(&paletteram[offset],newword);

	offset /= 4;
	{
		int palette = ((READ_WORD(&paletteram[offset * 4]) & 0x00ff) << 8)
				+ (READ_WORD(&paletteram[offset * 4 + 2]) & 0x00ff);
		int r = palette & 31;
		int g = (palette >> 5) & 31;
		int b = (palette >> 10) & 31;

		r = (r << 3) + (r >> 2);
		g = (g << 3) + (g >> 2);
		b = (b << 3) + (b >> 2);

		palette_change_color (offset,r,g,b);
	}
}

static UINT8 twin16_soundlatch;
static UINT16 twin16_vreg[9];

static int CPUA_IRQ_enable, CPUB_IRQ_enable;
static int twin16_gfx_bank;
static UINT16 twin16_sound_command;

static UINT32 *sprite_rom;
static unsigned char *sprite_gfx_ram;
static unsigned char *tile_gfx_ram;

static unsigned char *twin16_CPUA_fg_videoram;
static unsigned char *twin16_CPUA_bg_videoram;
static unsigned char *twin16_CPUB_fg_videoram;
static unsigned char *twin16_CPUB_bg_videoram;
static unsigned char *twin16_spriteram;

static unsigned char *twin16_battery_backed_ram;
static int konami_twin_gfx_type;

#define CREDITS "Phil Stroffolino"

enum {
	MEM_CPU_SOUND = 0,
	MEM_SOUND_SAMPLES1,
	MEM_SOUND_SAMPLES2,
	MEM_GFX_CHARS,
	MEM_GFX_SPRITES,
	MEM_CPU_A,
	MEM_CPU_B,
	MEM_GFX_TILES
};

/******************************************************************************************/

int twin16_sres_r(int offset){
	return twin16_soundlatch;
}

void twin16_sres_w(int offset,int data){
	/* bit 1 resets the UPD7795C sound chip */
	if ((data & 0x02) == 0){
		UPD7759_reset_w(0,(data & 0x02) >> 1);
	}
	twin16_soundlatch = data;
}

static struct YM2151interface ym2151_interface = {
	1,			/* 1 chip */
	7159160/2,	/* 3.58 MHZ ? */
	{ YM3012_VOL(50,MIXER_PAN_LEFT,50,MIXER_PAN_RIGHT) },
	{ 0 }
};

static void volume_callback(int v)
{
	K007232_set_volume(0,0,(v >> 4) * 0x11,0);
	K007232_set_volume(0,1,0,(v & 0x0f) * 0x11);
}

static struct K007232_interface k007232_interface =
{
	1,		/* number of chips */
	{MEM_SOUND_SAMPLES1,MEM_SOUND_SAMPLES1}, /* memory regions */
	{ K007232_VOL(20,MIXER_PAN_CENTER,20,MIXER_PAN_CENTER) },	/* volume */
	{ volume_callback }	/* external port callback */
};

static struct UPD7759_interface upd7759_interface =
{
	1,		/* number of chips */
	UPD7759_STANDARD_CLOCK,
	{ 60 }, /* volume */
	{ MEM_SOUND_SAMPLES2 }, /* memory region */
	UPD7759_STANDALONE_MODE, /* chip mode */
	{0}
};

void twin16_vreg_w( int offset, int data ){
	COMBINE_WORD_MEM( &twin16_vreg[offset/2], data );
}

static int battery_backed_ram_r( int offset ){
	return READ_WORD(&twin16_battery_backed_ram[offset]);
}
static void battery_backed_ram_w( int offset, int data ){
	COMBINE_WORD_MEM(&twin16_battery_backed_ram[offset],data);
}

static int sprite_gfx_ram_r( int offset ){
	return READ_WORD(&sprite_gfx_ram[offset]);
}
static void sprite_gfx_ram_w( int offset, int data ){
	COMBINE_WORD_MEM(&sprite_gfx_ram[offset],data);
}

static int tile_gfx_ram_r( int offset ){
	return READ_WORD(&tile_gfx_ram[offset]);
}
static void tile_gfx_ram_w( int offset, int data ){
	COMBINE_WORD_MEM(&tile_gfx_ram[offset],data);
}

static int vh_start( void ){
	return 0;
}

static void vh_stop( void ){
}

static void draw_background( struct osd_bitmap *bitmap, const UINT16 *source ){
/*	const struct rectangle *clip = &Machine->drv->visible_area; */
	const UINT16 *gfx_base = (UINT16 *)tile_gfx_ram;
	unsigned short *pal_base = Machine->pens;
	int i;
	for( i=0; i<64*64; i++ ){
		int sx = (i%64)*8;
		int sy = (i/64)*8;

		if( konami_twin_gfx_type ) sy = 31*8 - sy;

		if( sx>=0 && sy>=0 && sx+8<=320 && sy+8<=256 ){
			int code = source[i];
			const UINT16 *pal_data = pal_base + 16*((code>>13)+0x20);
			const UINT16 *gfx_data = gfx_base + (code&0x7ff)*16;
			UINT16 data;
			int y;
			for( y=0; y<8; y++ ){
				UINT8 *dest = bitmap->line[sy+y]+sx;

				data = *gfx_data++;
				*dest++ = pal_data[(data>>4*3)&0xf];
				*dest++ = pal_data[(data>>4*2)&0xf];
				*dest++ = pal_data[(data>>4*1)&0xf];
				*dest++ = pal_data[(data>>4*0)&0xf];

				data = *gfx_data++;
				*dest++ = pal_data[(data>>4*3)&0xf];
				*dest++ = pal_data[(data>>4*2)&0xf];
				*dest++ = pal_data[(data>>4*1)&0xf];
				*dest++ = pal_data[(data>>4*0)&0xf];
			}
		}
	}
}

static void draw_text( struct osd_bitmap *bitmap ){
	const struct rectangle *clip = &Machine->drv->visible_area;
	const UINT16 *source = (UINT16 *)twin16_CPUA_fg_videoram;
	int i;


	for( i=0; i<64*64; i++ ){
		int code = source[i];

		int sx = (i%64)*8;
		int sy = (i/64)*8;

		if( konami_twin_gfx_type ) sy = 31*8 - sy;

		drawgfx( bitmap, Machine->gfx[0],
			code&0x1ff,(code>>9)&0xf,
			0,0,
			sx,sy,
			clip,TRANSPARENCY_PEN,0);
	}
}

/* (sprite viewer)
static void draw_sprites( struct osd_bitmap *bitmap ){
	static int width = 16;
	static int delta = 0;

	if( keyboard_pressed( KEYCODE_K ) ){
		while( keyboard_pressed( KEYCODE_K ) ){}
		if( width<256 ) width*=2;
	}
	if( keyboard_pressed( KEYCODE_J ) ){
		while( keyboard_pressed( KEYCODE_J ) ){}
		if( width>16 ) width/=2;
	}
	if( keyboard_pressed( KEYCODE_N ) ){
		while( keyboard_pressed( KEYCODE_N ) ){}
		delta -= 256*256;
	}
	if( keyboard_pressed( KEYCODE_M ) ){
		while( keyboard_pressed( KEYCODE_M ) ){}
		delta += 256*256;
	}

	if( keyboard_pressed( KEYCODE_SPACE ) ){
		int x0,sx,sy;
		int count = delta;

		for( x0=0; x0<256; x0+=width ){
			for( sy=0; sy<256; sy++ ){
				for( sx=x0; sx<x0+width; sx++ ){
					int pen = (sprite_rom[count/8]>>(7*4-(count%8)*4))&0xf;
					bitmap->line[sy][sx] = pen;
					count++;
				}
			}
		}
	}
}
*/

static void vh_screenrefresh( struct osd_bitmap *bitmap, int fullrefresh ){
/*	if( keyboard_pressed( KEYCODE_S ) ){
		FILE *f;

		f = fopen( "cpub_fg_vram", "wb" );
		fwrite( twin16_CPUB_fg_videoram, 1, 0x4000, f );
		fclose( f );

		f = fopen( "cpub_bg_vram", "wb" );
		fwrite( twin16_CPUB_bg_videoram, 1, 0x4000, f );
		fclose( f );

		f = fopen( "spriteram", "wb" );
		fwrite( spriteram, 1, 0x4000, f );
		fclose( f );

		while( keyboard_pressed( KEYCODE_S ) ){}
	}

	if( keyboard_pressed( KEYCODE_W ) ){
		while( keyboard_pressed( KEYCODE_W ) ){}
		fg_color = (fg_color+1)&0x1f;
	}
	if( keyboard_pressed( KEYCODE_Q ) ){
		while( keyboard_pressed( KEYCODE_Q ) ){}
		bg_color = (bg_color+1)&0x3f;
	}
*/

	palette_recalc();

	if( CPUA_IRQ_enable&0x20 )
		draw_background( bitmap, (UINT16 *)twin16_CPUA_bg_videoram );
	else
		draw_background( bitmap, (UINT16 *)twin16_CPUB_bg_videoram );

	draw_text( bitmap );

	twin16_vreg[8] = CPUA_IRQ_enable;
	{
		int i,n;
		for( i=0; i<9; i++ ){
			for( n=0; n<4; n++ ){
				drawgfx( bitmap, Machine->uifont,
					"0123456789abcdef"[(twin16_vreg[i]>>(12-4*n))&0xf],
					0,
					0,0,
					n*8+8,i*8,
					0,TRANSPARENCY_NONE,0);
			}
		}
	}
}

void fround_screenrefresh( struct osd_bitmap *bitmap, int fullrefresh ){
	palette_recalc();
	fillbitmap( bitmap, 0, 0 );
	draw_text( bitmap );
}

static struct GfxLayout alpha_layout = {
	8,8,
	0x200,
	4,
	{ 0,1,2,3 },
	{ 0*4,1*4,2*4,3*4,4*4,5*4,6*4,7*4 },
	{ 0*32,1*32,2*32,3*32,4*32,5*32,6*32,7*32 },
	8*32
};


static void gfx_decode( void ){
	int i;
	const UINT16 *lo_word = (UINT16 *)Machine->memory_region[MEM_GFX_SPRITES];
	const UINT16 *hi_word = lo_word + 0x080000;
	sprite_rom = malloc( 0x80000*4 );
	for( i=0; i<0x80000; i++ ){
		sprite_rom[i] = ((UINT32)hi_word[i]<<16) | (UINT32)lo_word[i];
	}
}

static void cuebrick_decode( void ){
	gfx_decode();
	konami_twin_gfx_type = 0;
}

static void konami_twin16_decode( void ){
	gfx_decode();
	konami_twin_gfx_type = 1;
}

static struct GfxDecodeInfo gfxdecodeinfo[] = {
	{ MEM_GFX_CHARS, 0x00000, &alpha_layout, 0, 16 },
	{ -1 }
};


static int unknown_port_r( int offset ){ /* probably vblank */
	static int which;
	which = 1-which;
	return which;
}

/******************************************************************************************/

static int CPUA_interrupt( void ){
	return CPUA_IRQ_enable? MC68000_IRQ_5:MC68000_INT_NONE;
}
static int CPUB_interrupt( void ){
	return CPUB_IRQ_enable? MC68000_IRQ_5:MC68000_INT_NONE;
}

static void sound_command_w( int offset, int data ){
	twin16_sound_command = COMBINE_WORD( twin16_sound_command, data );
	soundlatch_w( 0, twin16_sound_command&0xff );
}

static void twin16_interrupt_A_w( int offset, int data ){
	/*
		7	6	5	4	3	2	1	0
			?							unknown (reset trigger?)
				X						IRQ5 enable (CPUA)
					X					trigger IRQ6 on CPUB
						X				trigger IRQ on sound CPU
		--------------------------------------------------------
							X			gfx bank select
								X		IRQ5 enable (CPUB)
									X	trigger IRQ6 on CPUA
	*/
	static int old_data;
	data = COMBINE_WORD( old_data, data );
	if( data!=old_data ){
		if( errorlog ) fprintf( errorlog, "CPUA_IRQ_control:0x%04x; pc=0x%02x\n", data, cpu_get_pc() );
		CPUA_IRQ_enable = data&0x20;

		if( (old_data&0x08)==0 && (data&0x08) ){
			cpu_cause_interrupt( 0, 0xff );
			if( errorlog ) fprintf( errorlog, "trigger IRQ on sound CPU\n" );
		}

		if( (old_data&0x10)==0 && (data&0x10) ){
			cpu_cause_interrupt( 2, MC68000_IRQ_6 );
			if( errorlog ) fprintf( errorlog, "trigger IRQ6 on CPUB\n" );
		}
		old_data = data;
	}
}

static void twin16_interrupt_B_w( int offset, int data ){
	/*
		7	6	5	4	3	2	1	0
							X			gfx bank select
								X		CPUB IRQ5 enable
									X	trigger IRQ6 on CPUA
	*/
	static int old_data;
	data = COMBINE_WORD( old_data, data );
	if( data!=old_data ){
		if( errorlog ) fprintf( errorlog, "CPUB_IRQ_control:0x%04x; pc=0x%02x\n", data, cpu_get_pc() );
		CPUB_IRQ_enable = data&0x02;

		twin16_gfx_bank = data&0x04;

		if( (old_data&0x01)==0 && (data&0x1) ){
			cpu_cause_interrupt( 1, MC68000_IRQ_6 );
			if( errorlog ) fprintf( errorlog, "trigger IRQ6 on CPUA\n" );
		}
		old_data = data;
	}
}

static void fround_interrupt_w( int offset, int data ){
	static int old_data;
	data = COMBINE_WORD( old_data, data );
	if( data!=old_data ){
		CPUA_IRQ_enable = data&0x20;
		if( (old_data&0x08)==0 && (data&0x08) ) cpu_cause_interrupt( 0, 0xff );
		old_data = data;
	}
}

/******************************************************************************************/

static int tile_rom_r( int offset ){
	return ((UINT16 *)Machine->memory_region[MEM_GFX_TILES])[offset/2];
}

static int sprite_rom1_r( int offset ){
	UINT32 data = sprite_rom[offset/4];
	return ((offset/2)%2)?(data&0xffff):(data>>16);
}
static int sprite_rom2_r( int offset ){
	const UINT32 *source = sprite_rom+0x40000;
	if( twin16_gfx_bank ) source += 0x20000;
	{
		UINT32 data = source[offset/4];
		return ((offset/2)%2)?(data&0xffff):(data>>16);
	}
}

static int twin16_input_r (int offset){
	switch( offset ){
		case 0x00: return readinputport(0); /* coin */
		case 0x02: return readinputport(1); /* p1 */
		case 0x04: return readinputport(2); /* p2 */
		case 0x06: return readinputport(3); /* p3 (Devils World) */

		case 0x10: return readinputport(5); /* DSW1 */
		case 0x12: return readinputport(4); /* DSW2 */
		case 0x18: return readinputport(6); /* DSW3 */
	}
	if (errorlog) fprintf(errorlog,"CPU #0 PC %06x: warning - read unmapped memory address %06x\n",cpu_get_pc(),0x0a0000+offset);
	return 0;
}

/******************************************************************************************/

static struct MemoryReadAddress readmem_sound[] = {
	{ 0x0000, 0x7fff, MRA_ROM },
	{ 0x8000, 0x8fff, MRA_RAM },
	{ 0x9000, 0x9000, twin16_sres_r },
	{ 0xa000, 0xa000, soundlatch_r },
	{ 0xb000, 0xb00d, K007232_read_port_0_r },
	{ 0xc001, 0xc001, YM2151_status_port_0_r },
	{ 0xf000, 0xf000, UPD7759_busy_r },
	{ -1 }
};

static struct MemoryWriteAddress writemem_sound[] = {
	{ 0x0000, 0x7fff, MWA_ROM },
	{ 0x8000, 0x8fff, MWA_RAM },
	{ 0x9000, 0x9000, twin16_sres_w },
	{ 0xb000, 0xb00d, K007232_write_port_0_w  },
	{ 0xc000, 0xc000, YM2151_register_port_0_w },
	{ 0xc001, 0xc001, YM2151_data_port_0_w },
	{ 0xd000, 0xd000, UPD7759_message_w },
	{ 0xe000, 0xe000, UPD7759_start_w },
	{ -1 }
};

static struct MemoryReadAddress readmem[] = {
	{ 0x000000, 0x03ffff, MRA_ROM },
	{ 0x040000, 0x043fff, MRA_BANK1 }, /* shared */
	{ 0x060000, 0x063fff, MRA_BANK2 },
	{ 0x080000, 0x081fff, paletteram_word_r, &paletteram },
	{ 0x0a0000, 0x0a001b, twin16_input_r },/*twin16_input_r }, */
	{ 0x0b0000, 0x0b3fff, battery_backed_ram_r, &twin16_battery_backed_ram }, /* cuebrick only */
	{ 0x0c000e, 0x0c000f, unknown_port_r }, /* vblank? */
	{ 0x100000, 0x103fff, MRA_BANK3, &twin16_CPUA_fg_videoram }, /* text layer */
	{ 0x120000, 0x123fff, MRA_BANK4, &twin16_CPUA_bg_videoram },
	{ 0x140000, 0x143fff, MRA_BANK5, &twin16_spriteram },
	{ -1 }
};

static struct MemoryWriteAddress writemem[] = {
	{ 0x000000, 0x03ffff, MWA_ROM },
	{ 0x040000, 0x043fff, MWA_BANK1 }, /* shared */
	{ 0x060000, 0x063fff, MWA_BANK2 },
	{ 0x080000, 0x081fff, twin16_paletteram_w },
	{ 0x0a0000, 0x0a0001, twin16_interrupt_A_w },
	{ 0x0a0008, 0x0a0009, sound_command_w },
	{ 0x0a0010, 0x0a0011, MWA_NOP }, /* watchdog */
	{ 0x0b0000, 0x0b3fff, battery_backed_ram_w },
	{ 0x0c0000, 0x0c000f, twin16_vreg_w },
	{ 0x100000, 0x103fff, MWA_BANK3 },
	{ 0x120000, 0x123fff, MWA_BANK4 },
	{ 0x140000, 0x143fff, MWA_BANK5 },
	{ -1 }
};

static struct MemoryReadAddress readmem_sub[] = {
	{ 0x000000, 0x03ffff, MRA_ROM },
	{ 0x040000, 0x043fff, MRA_BANK1 }, /* shared */
	{ 0x060000, 0x063fff, MRA_BANK8 },
	{ 0x080000, 0x09ffff, tile_rom_r }, /* dark adv only */
	{ 0x400000, 0x403fff, MRA_BANK6, &twin16_CPUB_fg_videoram },
	{ 0x480000, 0x483fff, MRA_BANK7, &twin16_CPUB_bg_videoram },
	{ 0x500000, 0x53ffff, tile_gfx_ram_r, &tile_gfx_ram }, /* tile ram */
	{ 0x600000, 0x6fffff, sprite_rom1_r },
	{ 0x700000, 0x77ffff, sprite_rom2_r },
	{ 0x780000, 0x79ffff, sprite_gfx_ram_r, &sprite_gfx_ram },
	{ -1 }
};

static struct MemoryWriteAddress writemem_sub[] = {
	{ 0x000000, 0x03ffff, MWA_ROM },
	{ 0x040000, 0x043fff, MWA_BANK1 }, /* shared */
	{ 0x060000, 0x063fff, MWA_BANK8 },
	{ 0x0a0000, 0x0a0001, twin16_interrupt_B_w },
	{ 0x400000, 0x403fff, MWA_BANK6 }, /* videoram? */
	{ 0x480000, 0x483fff, MWA_BANK7 }, /* videoram */
	{ 0x500000, 0x53ffff, tile_gfx_ram_w }, /* tile ram */
	{ 0x780000, 0x79ffff, sprite_gfx_ram_w },
	{ 0x810000, 0x810001, MWA_NOP },
	{ -1 }
};

/******************************************************************************************/

static struct MemoryReadAddress fround_readmem[] = {
	{ 0x000000, 0x03ffff, MRA_ROM },
	{ 0x040000, 0x043fff, MRA_BANK1 },
	{ 0x060000, 0x063fff, MRA_BANK2 }, /* shared */
	{ 0x080000, 0x080fff, paletteram_word_r, &paletteram },
	{ 0x0a0000, 0x0a001b, twin16_input_r },
	{ 0x0c000e, 0x0c000f, unknown_port_r }, /* vblank? */
	{ 0x100000, 0x103fff, MRA_BANK3, &twin16_CPUA_fg_videoram },
	{ 0x120000, 0x123fff, MRA_BANK4, &twin16_CPUA_bg_videoram },
	{ 0x140000, 0x143fff, MRA_BANK5, &twin16_spriteram },
	{ 0x500000, 0x6fffff, sprite_rom1_r },
	{ -1 }
};

static struct MemoryWriteAddress fround_writemem[] = {
	{ 0x000000, 0x03ffff, MWA_ROM },
	{ 0x040000, 0x043fff, MWA_BANK1 }, /* shared */
	{ 0x060000, 0x063fff, MWA_BANK2 }, /* shared */
	{ 0x080000, 0x080fff, twin16_paletteram_w },
	{ 0x0a0000, 0x0a0001, fround_interrupt_w },
	{ 0x0a0008, 0x0a0009, sound_command_w },
	{ 0x0a0010, 0x0a0011, MWA_NOP }, /* watchdog */
	{ 0x0c0006, 0x0c0007, MWA_NOP }, /* scroll */
	{ 0x0c0008, 0x0c0009, MWA_NOP }, /* scroll */
	{ 0x0c000a, 0x0c000b, MWA_NOP }, /* scroll */
	{ 0x0c000c, 0x0c000d, MWA_NOP }, /* scroll */
/*	{ 0x0e0000, 0x0e0001, MWA_NOP }, */
	{ 0x100000, 0x103fff, MWA_BANK3 }, /* fg */
	{ 0x120000, 0x123fff, MWA_BANK4 }, /* bg */
	{ 0x140000, 0x143fff, MWA_BANK5 }, /* spriteram */
	{ -1 }
};

/******************************************************************************************/

static struct MachineDriver machine_driver =
{
	{
		{
			CPU_Z80 | CPU_AUDIO_CPU,
			4000000, /* ? */
			MEM_CPU_SOUND,
			readmem_sound,writemem_sound,0,0,
			ignore_interrupt,1
		},
		{
			CPU_M68000,
			10000000, /* ? */
			MEM_CPU_A,
			readmem,writemem,0,0,
			CPUA_interrupt,1
		},
		{
			CPU_M68000,
			10000000, /* ? */
			MEM_CPU_B,
			readmem_sub,writemem_sub,0,0,
			CPUB_interrupt,1
		}
	},
	60, DEFAULT_REAL_60HZ_VBLANK_DURATION,
	100, /* CPU slices */
	0, /* init machine */

	/* video hardware */
	320, 256, { 0, 319, 0, 255 },
	gfxdecodeinfo,
	2048,2048, /* ? */
	0,

	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE,
	0,
	vh_start,
	vh_stop,
	vh_screenrefresh,

	/* sound hardware */
	0,0,0,0,
	{
		{
			SOUND_YM2151,
			&ym2151_interface
		},
		{
			SOUND_K007232,
			&k007232_interface,
		},
		{
			SOUND_UPD7759,
			&upd7759_interface
		}
	}
};

static struct MachineDriver fround_machine_driver =
{
	{
		{
			CPU_Z80 | CPU_AUDIO_CPU,
			4000000, /* ? */
			MEM_CPU_SOUND,
			readmem_sound,writemem_sound,0,0,
			ignore_interrupt,1
		},
		{
			CPU_M68000,
			10000000, /* ? */
			MEM_CPU_A,
			fround_readmem,fround_writemem,0,0,
			CPUA_interrupt,1
		},
	},
	60, DEFAULT_REAL_60HZ_VBLANK_DURATION,
	100, /* CPU slices */
	0, /* init machine */

	/* video hardware */
	320, 256, { 0, 319, 0, 255 },
	gfxdecodeinfo,
	2048,2048, /* ? */
	0,

	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE,
	0,
	vh_start,
	vh_stop,
	fround_screenrefresh,

	/* sound hardware */
	0,0,0,0,
	{
		{
			SOUND_YM2151,
			&ym2151_interface
		},
		{
			SOUND_K007232,
			&k007232_interface,
		},
		{
			SOUND_UPD7759,
			&upd7759_interface
		}
	}
};

ROM_START( devilw_rom )
	ROM_REGION( 0x10000 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "dw-m03.rom",	0x00000,  0x8000, 0x7201983c )
	ROM_REGION(0x20000) /* samples */
	ROM_LOAD(	"dw-ic5a.rom",	0x00000, 0x20000, 0xd4992dfb )
	ROM_REGION(0x20000) /* samples */
	ROM_LOAD(	"dw-ic7c.rom",	0x00000, 0x20000, 0xe5947501 )

	ROM_REGION_DISPOSE( 0x4000 )
	ROM_LOAD( "dw-m14.rom",	0x00000,  0x4000, 0xd7338557 ) /* characters */

	ROM_REGION( 0x200000 ) /* gfx data */
	ROM_LOAD_WIDE_SWAP(	"dw-10p.rom",	0x000000, 0x80000, 0x66cb3923 )
	ROM_LOAD_WIDE_SWAP(	"dw-10r.rom",	0x080000, 0x80000, 0xa1c7d0db )
	ROM_LOAD_WIDE_SWAP(	"dw-10l.rom",	0x100000, 0x80000, 0xeec8c5b2 )
	ROM_LOAD_WIDE_SWAP(	"dw-10m.rom",	0x180000, 0x80000, 0x746cf48b )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU A) */
	ROM_LOAD_EVEN(	"dw-t05.rom",	0x00000, 0x10000, 0x8ab7dc61 )
	ROM_LOAD_ODD(	"dw-t04.rom",	0x00000, 0x10000, 0xc69924da )
	ROM_LOAD_EVEN(	"dw-t09.rom",   0x20000, 0x10000, 0xfae97de0 )
	ROM_LOAD_ODD(	"dw-t08.rom",   0x20000, 0x10000, 0x8c898d67 )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU B) */
	ROM_LOAD_EVEN(	"dw-r07.rom",	0x00000, 0x10000, 0x53110c0b )
	ROM_LOAD_ODD(	"dw-r06.rom",	0x00000, 0x10000, 0x9c53a0c5 )
	ROM_LOAD_EVEN(	"dw-r13.rom",	0x20000, 0x10000, 0x36ae6014 )
	ROM_LOAD_ODD(	"dw-r12.rom",	0x20000, 0x10000, 0x6d012167 )

	ROM_REGION( 0x20000 ) /* tile data; mapped at 0x80000 on CPUB */
	ROM_LOAD_EVEN(	"dw-m11.rom",	0x00000, 0x10000, 0x399deee8 )
	ROM_LOAD_ODD(	"dw-m10.rom",	0x00000, 0x10000, 0x117c91ee )
ROM_END

ROM_START( darkadv_rom ) /* this set is incomplete! */
	ROM_REGION( 0x10000 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "n03.10a",	0x00000,  0x8000, 0xa24c682f )
	ROM_REGION(0x20000) /* samples */
	ROM_LOAD(	"dw-ic5a.rom",	0x00000, 0x20000, 0 )
	ROM_REGION(0x20000) /* samples */
	ROM_LOAD(	"dw-ic7c.rom",	0x00000, 0x20000, 0 )

	ROM_REGION_DISPOSE( 0x4000 )
	ROM_LOAD( "n14.3f",	0x0000,  0x4000, 0xc76ac6d2 ) /* characters */

	ROM_REGION( 0x200000 ) /* gfx data */
	ROM_LOAD_WIDE_SWAP(	"dw-10p.rom",	0x000000, 0x80000, 0x66cb3923 )
	ROM_LOAD_WIDE_SWAP(	"dw-10r.rom",	0x080000, 0x80000, 0xa1c7d0db )
	ROM_LOAD_WIDE_SWAP(	"dw-10l.rom",	0x100000, 0x80000, 0xeec8c5b2 )
	ROM_LOAD_WIDE_SWAP(	"dw-10m.rom",	0x180000, 0x80000, 0x746cf48b )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU A) */
	ROM_LOAD_EVEN(	"n05.6n",	0x00000, 0x10000, 0xa9195b0b )
	ROM_LOAD_ODD(	"n04.4n",	0x00000, 0x10000, 0x65b55105 )
	ROM_LOAD_EVEN(	"n09",		0x20000, 0x10000, 0 ) /* missing! */
	ROM_LOAD_ODD(   "n08.4r",   0x20000, 0x10000, 0xa9603196 )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU B) */
	ROM_LOAD_EVEN(	"n07.10n",	0x00000, 0x10000, 0x6154322a )
	ROM_LOAD_ODD(	"n06.8n",	0x00000, 0x10000, 0x37a72e8b )
	ROM_LOAD_EVEN(	"n13.10s",	0x20000, 0x10000, 0xf1c252af )
	ROM_LOAD_ODD(	"n12.8s",	0x20000, 0x10000, 0xda221944 )

	ROM_REGION( 0x20000 ) /* tile data; mapped at 0x80000 on CPUB */
	ROM_LOAD_EVEN(	"dw-m11.rom",	0x00000, 0x10000, 0x399deee8 )
	ROM_LOAD_ODD(	"dw-m10.rom",	0x00000, 0x10000, 0x117c91ee ) /* tiles */
ROM_END

ROM_START( cuebrick_rom )
	ROM_REGION( 0x10000 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "903-d03.10a",	0x00000,  0x4000, 0x0a404411 )

	ROM_REGION( 0x20000 ) /* missing? */
	ROM_REGION( 0x20000 ) /* missing? */

	ROM_REGION_DISPOSE( 0x4000 )
	ROM_LOAD(	"903-e14.d8",	0x0000, 0x4000, 0xddbebbd5 ) /* characters */

	ROM_REGION( 0x200000 ) /* gfx data */

	ROM_REGION( 0x40000 ) /* 68000 code (CPU A) */
	ROM_LOAD_EVEN(	"903-e05.6n",	0x00000, 0x10000, 0x8b556220 )
	ROM_LOAD_ODD(	"903-e04.4n",	0x00000, 0x10000, 0xbf9c7927 )
	ROM_LOAD_EVEN(	"903-e09.6r",	0x20000, 0x10000, 0x2a77554d )
	ROM_LOAD_ODD(	"903-e08.4r",	0x20000, 0x10000, 0xc0a430c1 )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU B) */
	ROM_LOAD_EVEN(	"903-d07.10n",	0x00000, 0x10000, 0xfc0edce7 )
	ROM_LOAD_ODD(	"903-d06.8n",	0x00000, 0x10000, 0xb2cef6fe )
	ROM_LOAD_EVEN(	"903-e13.10s",	0x20000, 0x10000, 0x4fb5fb80 )
	ROM_LOAD_ODD(	"903-e12.8s",	0x20000, 0x10000, 0x883e3097 )

	ROM_REGION( 0x20000 ) /* tile data; mapped at 0x80000 on CPUB */
	ROM_LOAD_EVEN(	"903-e11.10r",	0x00000, 0x10000, 0x5c41faf8 )
	ROM_LOAD_ODD(	"903-e10.8r",	0x00000, 0x10000, 0x417576d4 )
ROM_END

ROM_START( vulcan_rom )
	ROM_REGION( 0x10000 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "gradius2.g03",	0x00000,  0x8000, 0x67a3b50d )
	ROM_REGION( 0x20000 )
	ROM_LOAD(	"gradius2.f01",	0x00000, 0x20000, 0xa0d8d69e )
	ROM_REGION( 0x20000 )
	ROM_LOAD(	"gradius2.f02",	0x00000, 0x20000, 0xc39f5ca4 )

	ROM_REGION_DISPOSE( 0x4000 )
	ROM_LOAD(	"vulcan.h14",	0x0000, 0x4000, 0x02f4b16f ) /* characters */

	ROM_REGION( 0x200000 ) /* gfx data */
	ROM_LOAD_WIDE(	"gradius2.f17",	0x000000, 0x80000, 0x8fbec1a4 )
	ROM_LOAD_WIDE(	"gradius2.f18",	0x080000, 0x80000, 0x50d61e38 )
	ROM_LOAD_WIDE(	"gradius2.f15",	0x100000, 0x80000, 0xaf96aef3 )
	ROM_LOAD_WIDE(	"gradius2.f16",	0x180000, 0x80000, 0xb858df1f )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU A) */
	ROM_LOAD_EVEN(	"vulcan.w05", 0x00000, 0x10000, 0x6e0e99cd )
	ROM_LOAD_ODD(	"vulcan.w04", 0x00000, 0x10000, 0x23ec74ca )
	ROM_LOAD_EVEN(	"vulcan.w09", 0x20000, 0x10000, 0x377e4f28 )
	ROM_LOAD_ODD(	"vulcan.w08", 0x20000, 0x10000, 0x813d41ea )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU B) */
	ROM_LOAD_EVEN(	"gradius2.p07", 0x00000, 0x10000, 0x686d549d )
	ROM_LOAD_ODD(	"gradius2.p06", 0x00000, 0x10000, 0x70c94bee )
	ROM_LOAD_EVEN(	"gradius2.p13", 0x20000, 0x10000, 0x478fdb0a )
	ROM_LOAD_ODD(	"gradius2.p12", 0x20000, 0x10000, 0x38ea402a )

	ROM_REGION( 0x20000 ) /* tile data; mapped at 0x80000 on CPUB */
ROM_END

ROM_START( gradius2_rom )
	ROM_REGION( 0x10000 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "gradius2.g03",	0x00000,  0x8000, 0x67a3b50d )
	ROM_REGION( 0x20000 )
	ROM_LOAD(	"gradius2.f01",	0x00000, 0x20000, 0xa0d8d69e )
	ROM_REGION( 0x20000 )
	ROM_LOAD(	"gradius2.f02",	0x00000, 0x20000, 0xc39f5ca4 )

	ROM_REGION_DISPOSE( 0x4000 )
	ROM_LOAD(	"gradius2.g14",	0x0000, 0x4000, 0x9dcdad9d ) /* characters */

	ROM_REGION( 0x200000 ) /* gfx data */
	ROM_LOAD_WIDE(	"gradius2.f17",	0x000000, 0x80000, 0x8fbec1a4 )
	ROM_LOAD_WIDE(	"gradius2.f18",	0x080000, 0x80000, 0x50d61e38 )
	ROM_LOAD_WIDE(	"gradius2.f15",	0x100000, 0x80000, 0xaf96aef3 )
	ROM_LOAD_WIDE(	"gradius2.f16",	0x180000, 0x80000, 0xb858df1f )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU A) */
	ROM_LOAD_EVEN(	"gradius2.p05", 0x00000, 0x10000, 0x4db0e736 )
	ROM_LOAD_ODD(	"gradius2.p04", 0x00000, 0x10000, 0x765b99e6 )
	ROM_LOAD_EVEN(	"gradius2.j09", 0x20000, 0x10000, 0x6d96a7e3 )
	ROM_LOAD_ODD(	"gradius2.j08", 0x20000, 0x10000, 0x2b1c9108 )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU B) */
	ROM_LOAD_EVEN(	"gradius2.p07", 0x00000, 0x10000, 0x686d549d )
	ROM_LOAD_ODD(	"gradius2.p06", 0x00000, 0x10000, 0x70c94bee )
	ROM_LOAD_EVEN(	"gradius2.p13", 0x20000, 0x10000, 0x478fdb0a )
	ROM_LOAD_ODD(	"gradius2.p12", 0x20000, 0x10000, 0x38ea402a )

	ROM_REGION( 0x20000 ) /* tile data; mapped at 0x80000 on CPUB */
ROM_END

ROM_START( fround_rom )
	ROM_REGION( 0x10000 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "frf03.bin",	0x00000,  0x8000, 0xa645c727 )
	ROM_REGION( 0x20000 )
	ROM_LOAD(	"frc01.bin",	0x00000, 0x20000, 0x6af96546 )
	ROM_REGION( 0x20000 )
	ROM_LOAD(	"frc02.bin",	0x00000, 0x20000, 0x54e12c6d )

	ROM_REGION_DISPOSE( 0x4000 )
	ROM_LOAD(	"frp17.bin",	0x0000, 0x4000, 0xc9b46615 ) /* characters */

	ROM_REGION( 0x200000 ) /* gfx data */
	ROM_LOAD_WIDE(	"frc17.bin",	0x000000, 0x80000, 0x2bc99ff8 )
	ROM_LOAD_WIDE(	"frc18.bin",	0x080000, 0x80000, 0x07927fe8 )
	ROM_LOAD_WIDE(	"frc15.bin",	0x100000, 0x80000, 0x8c9281df )
	ROM_LOAD_WIDE(	"frc16.bin",	0x180000, 0x80000, 0x41df6a1b )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU A) */
	ROM_LOAD_EVEN(	"frl21.bin", 0x00000, 0x20000, 0xe21a3a19 )
	ROM_LOAD_ODD(	"frl20.bin", 0x00000, 0x20000, 0x0ce9786f )
ROM_END

#define KONAMI_TWIN_COINAGE PORT_START \
	PORT_DIPNAME( 0x0f, 0x0f, DEF_STR( Coin_A ) ) \
	PORT_DIPSETTING(    0x00, "freeplay" ) \
	PORT_DIPSETTING(    0x02, DEF_STR( 4C_1C ) ) \
	PORT_DIPSETTING(    0x05, DEF_STR( 3C_1C ) ) \
	PORT_DIPSETTING(    0x08, DEF_STR( 2C_1C ) ) \
	PORT_DIPSETTING(    0x04, DEF_STR( 3C_2C ) ) \
	PORT_DIPSETTING(    0x01, DEF_STR( 4C_3C ) ) \
	PORT_DIPSETTING(    0x0f, DEF_STR( 1C_1C ) ) \
	PORT_DIPSETTING(    0x03, DEF_STR( 3C_4C ) ) \
	PORT_DIPSETTING(    0x07, DEF_STR( 2C_3C ) ) \
	PORT_DIPSETTING(    0x06, DEF_STR( 2C_5C ) ) \
	PORT_DIPSETTING(    0x0e, DEF_STR( 1C_2C ) ) \
	PORT_DIPSETTING(    0x0d, DEF_STR( 1C_3C ) ) \
	PORT_DIPSETTING(    0x0c, DEF_STR( 1C_4C ) ) \
	PORT_DIPSETTING(    0x0b, DEF_STR( 1C_5C ) ) \
	PORT_DIPSETTING(    0x0a, DEF_STR( 1C_6C ) ) \
	PORT_DIPSETTING(    0x09, DEF_STR( 1C_7C ) ) \
	PORT_DIPNAME( 0xf0, 0xf0, DEF_STR( Coin_B ) ) \
	PORT_DIPSETTING(    0x20, DEF_STR( 4C_1C ) ) \
	PORT_DIPSETTING(    0x50, DEF_STR( 3C_1C ) ) \
	PORT_DIPSETTING(    0x80, DEF_STR( 2C_1C ) ) \
	PORT_DIPSETTING(    0x40, DEF_STR( 3C_2C ) ) \
	PORT_DIPSETTING(    0x10, DEF_STR( 4C_3C ) ) \
	PORT_DIPSETTING(    0xf0, DEF_STR( 1C_1C ) ) \
	PORT_DIPSETTING(    0x30, DEF_STR( 3C_4C ) ) \
	PORT_DIPSETTING(    0x70, DEF_STR( 2C_3C ) ) \
	PORT_DIPSETTING(    0x60, DEF_STR( 2C_5C ) ) \
	PORT_DIPSETTING(    0xe0, DEF_STR( 1C_2C ) ) \
	PORT_DIPSETTING(    0xd0, DEF_STR( 1C_3C ) ) \
	PORT_DIPSETTING(    0xc0, DEF_STR( 1C_4C ) ) \
	PORT_DIPSETTING(    0xb0, DEF_STR( 1C_5C ) ) \
	PORT_DIPSETTING(    0xa0, DEF_STR( 1C_6C ) ) \
	PORT_DIPSETTING(    0x90, DEF_STR( 1C_7C ) )

INPUT_PORTS_START( twin16_input_ports ) /* generic; currently used by fround and cuebrick */
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) /* advances through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x08, 0x08, "Reserved" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

INPUT_PORTS_START( gradius2_input_ports )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) /* advances through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0xff, 0xff, IPT_UNUSED )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "4" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x18, 0x18, "Bonus" )
	PORT_DIPSETTING(    0x00, "70K" )
	PORT_DIPSETTING(    0x08, "20K" )
	PORT_DIPSETTING(    0x10, "30K/80K" )
	PORT_DIPSETTING(    0x18, "20K/70K" )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, "Easy" )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "Controls" )
	PORT_DIPSETTING(    0x02, "Single" )
	PORT_DIPSETTING(    0x00, "Dual" )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x08, 0x08, "Reserved" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

INPUT_PORTS_START( devilw_input_ports )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON3 ) /* "map" button (also advances through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 ) /* also used in-game */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 ) /* also used in-game */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START3 ) /* also used in-game */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, "Easy" )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "Controls" )
	PORT_DIPSETTING(    0x02, "Single" )
	PORT_DIPSETTING(    0x00, "Dual" )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x08, 0x08, "Reserved" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

struct GameDriver devilw_driver =  {
	__FILE__,
	0,
	"devilw",
	"Devil's World",
	"1987",
	"Konami",
	CREDITS,
	GAME_NOT_WORKING,
	&machine_driver,
	0,
	devilw_rom,
	konami_twin16_decode,
	0,
	0,
	0,	/* sound_prom */

	devilw_input_ports,

	0, 0, 0, /* colors, palette, colortable */
	ORIENTATION_DEFAULT,
	0,0
};

struct GameDriver darkadv_driver =  {
	__FILE__,
	&devilw_driver,
	"darkadv",
	"Dark Adventure",
	"1987",
	"Konami",
	CREDITS,
	GAME_NOT_WORKING,
	&machine_driver,
	0,
	darkadv_rom,
	konami_twin16_decode,
	0,
	0,
	0,	/* sound_prom */

	devilw_input_ports,

	0, 0, 0, /* colors, palette, colortable */
	ORIENTATION_DEFAULT,
	0,0
};

struct GameDriver vulcan_driver =  {
	__FILE__,
	0,
	"vulcan",
	"Vulcan Venture",
	"1988",
	"Konami",
	CREDITS,
	GAME_NOT_WORKING,
	&machine_driver,
	0,
	vulcan_rom,
	konami_twin16_decode,
	0,
	0,
	0,	/* sound_prom */

	gradius2_input_ports,

	0, 0, 0, /* colors, palette, colortable */
	ORIENTATION_DEFAULT,
	0,0
};

struct GameDriver gradius2_driver =  {
	__FILE__,
	&vulcan_driver,
	"gradius2",
	"Gradius II - Gofer no Yabou (Japan)",
	"1988",
	"Konami",
	CREDITS,
	GAME_NOT_WORKING,
	&machine_driver,
	0,
	gradius2_rom,
	konami_twin16_decode,
	0,
	0,
	0,	/* sound_prom */

	gradius2_input_ports,

	0, 0, 0, /* colors, palette, colortable */
	ORIENTATION_DEFAULT,
	0,0
};

struct GameDriver cuebrick_driver =  {
	__FILE__,
	0,
	"cuebrick",
	"Cuebrick",
	"1989",
	"Konami",
	CREDITS,
	GAME_NOT_WORKING,
	&machine_driver,
	0,
	cuebrick_rom,
	cuebrick_decode,
	0,
	0,
	0,	/* sound_prom */

	twin16_input_ports,

	0, 0, 0, /* colors, palette, colortable */
	ORIENTATION_DEFAULT,
	0,0
};

struct GameDriver fround_driver =  {
	__FILE__,
	0,
	"fround",
	"Final Round",
	"1988",
	"Konami",
	CREDITS,
	GAME_NOT_WORKING,
	&fround_machine_driver,
	0,
	fround_rom,
	cuebrick_decode, 0,
	0,
	0,	/* sound_prom */

	twin16_input_ports,

	0, 0, 0, /* colors, palette, colortable */
	ORIENTATION_DEFAULT,
	0,0
};
