package biss;

import java.util.Enumeration;
import java.util.NoSuchElementException;

/**
 * Queue container which is implemented with a ring buffer (therefor more
 * efficient than Vectors used for the same purpose). Follows the java.lang.Stack
 * api. 
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
public class Queue
{
	Object[] Buf;
	int Capacity;
	int Read;
	int Write;
	boolean IsEmpty;
	int Removed;

public Queue () {
	this( 20);
}

public Queue ( int capacity) {
	Capacity = capacity;
	Buf = new Object[Capacity];
	Read =  0;
	Write = 0;
	IsEmpty = true;
	Removed = 0;
}

final public synchronized void append ( Object o ) {
	if ( (Write == Read) && !IsEmpty )
		ensureCapacity( 2*Capacity);

	Buf[Write] = o;
	// this is faster than:  Write = ++Write % Capacity;
	Write++;  if ( Write == Capacity ) Write = 0;
	IsEmpty = false;
}

final public synchronized void appendNotify ( Object o ) {
	append( o);
	notifyAll();
}

final public synchronized boolean appendNotifyUnique ( Object o ) {
	if ( !this.contains( o) ){
		appendNotify( o);
		return true;
	}
	else
		return false;
}

final public synchronized boolean appendUnique ( Object o ) {
	if ( !this.contains( o ) ){
		append( o);
		return true;
	}
	else
		return false;
}

final public synchronized boolean contains ( Object o ) {
	int i;

	if ( IsEmpty ) return false;

	if ( Write > Read ) {
		for ( i=Read; i<Write; i++ )
			if ( Buf[i].equals( o) ) return true;
	}
	else {
		for ( i=Read; i<Capacity; i++ )
			if ( Buf[i].equals( o) ) return true;
		for ( i=0; i<Write; i++ )
			if ( Buf[i].equals( o) ) return true;
	}

	return false;
}

public Enumeration elements() {
	return new QueueEnumerator( this);
}

public boolean empty () {
	return IsEmpty;
}

public synchronized void ensureCapacity ( int newCapacity ) {
	if ( newCapacity <= Capacity )  // large enough (don't shrink)
		return;
	newCapacity = Math.max( newCapacity, 2*Capacity);

	Object[] newBuf = new Object[newCapacity];

	if ( IsEmpty ) {                // nothing to copy at all
	}
	else if ( Write > Read ) {      // simple forward (null) expansion
		/*
		  old :   ---R>>>W----
		  new :   ---R>>>W----------
		 */
		System.arraycopy( Buf, Read, newBuf, Read, (Write-Read+1));
	}
	else {
		/* R=const resizing (essential for enumeration scheme)
		   old :   >>>W---R>>>>
		   new :   -------R>>>>>>>W--
		 */
		int n = Capacity - Read;
		System.arraycopy( Buf, Read, newBuf, Read, n);
		int m = Read + n;
		System.arraycopy( Buf, 0, newBuf, m, Write);
		Write += m;
		if ( Write == newCapacity ) Write = 0;
	}

	Capacity = newCapacity;
	Buf = newBuf;
	IsEmpty = false;
}

final public synchronized Object getNext () {
	if ( IsEmpty )
		throw new EmptyQueueException();
	else {
		Object o = Buf[Read];
		Buf[Read] = null;         // enable GC recycling

		// this is faster than: Read = ++Read % Capacity;
		Read++; if ( Read == Capacity ) Read = 0;

		if ( Read == Write ){
			IsEmpty = true;
			Removed = 0;           // reorganize
		}
		else
			Removed++;

		return o;
	}
}

public int size () {
	if ( Read == Write )
		return (IsEmpty) ? 0 : Capacity;
	else if ( Read < Write )
		return Write - Read;
	else
		return (Capacity-Read) + Write;
}
}

class QueueEnumerator
  implements Enumeration
{
	Queue Q;
	int Idx;
	int N;
	boolean Ready;

QueueEnumerator ( Queue q ) {
	Q = q;
	Idx = Q.Read;
	N = Q.Removed;
	Ready = Q.IsEmpty;
}

public boolean hasMoreElements() {
	if ( Q.IsEmpty ) return false;

	if ( N < Q.Removed ) {       // check if Q.Read has passed Idx
		Idx = Q.Read;
		N = Q.Removed;
		Ready = false;
	}

	return !Ready;
}

public Object nextElement() {
	Object ret;
	synchronized ( Q ) {
		if ( Q.IsEmpty )
			throw new NoSuchElementException("QueueEnumerator");
		if ( N < Q.Removed ) {     // check if Q.Read has passed Idx
			Idx = Q.Read;
			N = Q.Removed;
			Ready = false;
		}
		if ( Ready )
			throw new NoSuchElementException("QueueEnumerator");

		ret = Q.Buf[Idx];
		Idx++;  if (Idx == Q.Capacity) Idx = 0;
		N++;
		Ready = (Idx == Q.Write);
		return ret;
	}
}
}
