/*
 * Copyright (c) 1994  Software Research Associates, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Software Research Associates not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  Software Research
 * Associates makes no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 *
 * Author:  Makoto Ishisone, Software Research Associates, Inc., Japan
 */
#include <stdio.h>
#include <stdlib.h>
#include "im.h"
#include "resrcs.h"
#include "FontMgr.h"

#ifndef XNSeparatorofNestedList
#define XNSeparatorofNestedList "separatorofNesttedList"
#endif

#define PAD4(n)	((((n) + 3) / 4) * 4)

/*
 * List of supported input styles.
 */

typedef struct {
  XIMStyle xim_style ;		/* X11R5 spec. */
  int conversion_style ;	/* kinput2 spec. */
} InputStyle ;

static InputStyle styles[] = {
    { XIMPreeditPosition|XIMStatusArea, IMSTYLE_OVER_THE_SPOT },
    { XIMPreeditPosition|XIMStatusNothing, IMSTYLE_OVER_THE_SPOT },
    { XIMPreeditArea|XIMStatusArea, IMSTYLE_OFF_THE_SPOT },
    { XIMPreeditNothing|XIMStatusNothing, IMSTYLE_SEPARATE },
    { 0 },
};

enum {
  NEST_NONE = 0, NEST_PREEDIT, NEST_STATUS,
} ;

#define CHECK_ICATTR_SIZE(validsize, code) \
	if (len != validsize) { badSizeError(icp, code); return -1; }

#undef OP_C
#undef OP_S
#undef OP_G

#define OP_C	1	/* Create */
#define OP_S	2	/* SetValues */
#define OP_G	4	/* GetValues */

typedef struct {
    char *name;		/* attribute name */
    int type;		/* type of attribute value */
    int valid_ops;	/* valid operations for this attribute */
    int (*set_proc) _Pt_((IMIM *, char *, int));
    int (*get_proc) _Pt_((IMIM *, unsigned int, int));
} IMAttribute ;

typedef struct {
    char *name;		/* attribute name */
    int type;		/* type of attribute value */
    int valid_ops;	/* valid operations for this attribute */
    int (*set_proc) _Pt_((IMIC *, char *, int, int, int, int));
    int (*get_proc) _Pt_((IMIC *, unsigned int, int, int, char *, int));
} ICAttribute ;


/*
 * IM attributes
 */

static int getQueryInputStyle _Pt_((IMIM *imp, unsigned int id, int offset));

static IMAttribute imAttributes[] = {
    { XNQueryInputStyle, TYPE_XIM_STYLES, OP_G,
	  NULL, getQueryInputStyle },
};

static int numImAttributes = XtNumber(imAttributes);


/*
 * IC attributes
 */
static int setInputStyle( IMIC *, char *, int, int, int, int ) ;
static int setClientWindow( IMIC *, char *, int, int, int, int ) ;
static int setFocusWindow( IMIC *, char *, int, int, int, int ) ;
static int setPreeditAttributes( IMIC *, char *, int, int, int, int ) ;
static int setStatusAttributes( IMIC *, char *, int, int, int, int ) ;
static int setArea( IMIC *, char *, int, int, int, int ) ;
static int setAreaNeeded( IMIC *, char *, int, int, int, int ) ;
static int setForeground( IMIC *, char *, int, int, int, int ) ;
static int setBackground( IMIC *, char *, int, int, int, int ) ;
static int setColormap( IMIC *, char *, int, int, int, int ) ;
static int setBgPixmap( IMIC *, char *, int, int, int, int ) ;
static int setLineSpace( IMIC *, char *, int, int, int, int ) ;
static int setCursor( IMIC *, char *, int, int, int, int ) ;
static int setSpotLocation( IMIC *, char *, int, int, int, int ) ;
static int setStdColormap( IMIC *, char *, int, int, int, int ) ;
static int setFontSet( IMIC *, char *, int, int, int, int ) ;

static int getPreeditAttributes
( IMIC *, unsigned int, int, int, char *, int ) ;
static int getStatusAttributes
( IMIC *, unsigned int, int, int, char *, int ) ;
static int getInputStyle( IMIC *, unsigned int, int, int, char *, int ) ;
static int getClientWindow( IMIC *, unsigned int, int, int, char *, int ) ;
static int getFocusWindow( IMIC *, unsigned int, int, int, char *, int ) ;
static int getArea( IMIC *, unsigned int, int, int, char *, int ) ;
static int getAreaNeeded( IMIC *, unsigned int, int, int, char *, int ) ;
static int getSpotLocation( IMIC *, unsigned int, int, int, char *, int ) ;
static int getColormap( IMIC *, unsigned int, int, int, char *, int ) ;
static int getStdColormap( IMIC *, unsigned int, int, int, char *, int ) ;
static int getForeground( IMIC *, unsigned int, int, int, char *, int ) ;
static int getBackground( IMIC *, unsigned int, int, int, char *, int ) ;
static int getBgPixmap( IMIC *, unsigned int, int, int, char *, int ) ;
static int getFontSet( IMIC *, unsigned int, int, int, char *, int ) ;
static int getLineSpace( IMIC *, unsigned int, int, int, char *, int ) ;
static int getCursor( IMIC *, unsigned int, int, int, char *, int ) ;

static ICAttribute icAttributes[] = {
  { XNInputStyle, TYPE_CARD32, OP_C|OP_G,
    setInputStyle, getInputStyle },
  { XNClientWindow, TYPE_WINDOW, OP_C|OP_G,
    setClientWindow, getClientWindow },
  { XNFocusWindow, TYPE_WINDOW, OP_C|OP_S|OP_G,
    setFocusWindow, getFocusWindow },
  { XNPreeditAttributes, TYPE_NESTED_LIST, OP_C|OP_S|OP_G,
    setPreeditAttributes, getPreeditAttributes },
  { XNStatusAttributes, TYPE_NESTED_LIST, OP_C|OP_S|OP_G,
    setStatusAttributes, getStatusAttributes },
  { XNArea, TYPE_XRECTANGLE, OP_C|OP_S|OP_G,
    setArea, getArea },
  { XNAreaNeeded, TYPE_XRECTANGLE, OP_C|OP_S|OP_G,
    setAreaNeeded, getAreaNeeded },
  { XNSpotLocation, TYPE_XPOINT, OP_C|OP_S|OP_G,
    setSpotLocation, getSpotLocation },
  { XNColormap, TYPE_CARD32, OP_C|OP_S|OP_G,
    setColormap, getColormap },
  { XNStdColormap, TYPE_CARD32, OP_C|OP_S|OP_G,
    setStdColormap, getStdColormap },
  { XNForeground, TYPE_CARD32, OP_C|OP_S|OP_G,
    setForeground, getForeground },
  { XNBackground, TYPE_CARD32, OP_C|OP_S|OP_G,
    setBackground, getBackground },
  { XNBackgroundPixmap, TYPE_CARD32, OP_C|OP_S|OP_G,
    setBgPixmap, getBgPixmap },
  { XNFontSet, TYPE_XFONTSET, OP_C|OP_S|OP_G,
    setFontSet, getFontSet },
  { XNLineSpace, TYPE_CARD16, OP_C|OP_S|OP_G,	  /* should be TYPE_INT16 */
    setLineSpace, getLineSpace },
  { XNCursor, TYPE_CARD32, OP_C|OP_S|OP_G,
    setCursor, getCursor },
  { XNSeparatorofNestedList, TYPE_SEPARATOR, OP_G,
    NULL, NULL },
} ;

static int numIcAttributes = XtNumber( icAttributes ) ;

static unsigned int getC16( char *data, int order ) ;
static int getI16(char *data, int order ) ;
static unsigned long getC32( char *data, int order ) ;
static int validateClientWindow( IMIC *icp ) ;
static int validateFocusWindow( IMIC *icp ) ;
static void badSizeError( IMIC *icp, int code ) ;
static void unnestedError( IMIC *icp ) ;
static IMPSAttributes *getPSPtr( IMIC *icp, int type ) ;
static int getIMValues( IMIM *imp, char *data, int len, int offset ) ;
static int getICValues
( IMIC *icp, char *data, int len, int nest, int offset, int *sepp ) ;
static int setICValues
( IMIC *icp, char *data, int len, int major, int op ) ;
static int getPSAttributes
( IMIC *icp, unsigned int id, int nest, int offset, char *data, int len ) ;
static void changeFonts( IMIC *icp ) ;
static void fillCommonDefault( IMIC *icp, unsigned long mask ) ;
static int getNaturalLineSpace( IMIC *icp ) ;
static void fillPSDefault( IMIC *icp, int type, unsigned long mask ) ;
static int validateCommonAttr( IMIC *icp, int checkonly ) ;
static int validatePSAttr( IMIC *icp, int type, int checkonly ) ;
static void changeConversionAttributes( IMIC *icp ) ;
static void computeAreaNeeded( IMIC *icp ) ;
static void computeAreaForQuery( IMIC *icp ) ;

/*
 * Functions reading out numbers from byte buffer
 */

static unsigned int getC16( char *data, int byte_order )
{
  unsigned char *p = (unsigned char *)data ;
  unsigned int x ;

  x = ( byte_order == ORDER_BIG )?
    ( ( p[ 0 ] << 8 ) | p[ 1 ] ) : ( p[ 0 ] | p[ 1 ] << 8 ) ;
  return x ;
}

static int getI16( char *data, int byte_order )
{
  unsigned char *p = (unsigned char *)data ;
  long l ;
  
  l = ( byte_order == ORDER_BIG )?
    ( ( p[ 0 ] << 8 ) | p[ 1 ] ) : ( p[ 0 ] | p[ 1 ] << 8 ) ;
  return ( l < 32768 )? ( int )l : ( int )( l - 65536L ) ;
}

static unsigned long getC32
( char *data, int order )
{
  unsigned char *p = ( unsigned char * )data ;
  unsigned long x ;

  /* ǥˤäƵưۤʤ롣*/
  if( order == ORDER_BIG ){
    x = ( p[ 0 ] << 24 ) | ( p[ 1 ] << 16 ) | ( p[ 2 ] << 8 ) | p[ 3 ] ;
  } else {
    x = p[ 0 ] | ( p[ 1 ] << 8 ) | ( p[ 2 ] << 16 ) | ( p[ 3 ] << 24 ) ;
  }
  return x ;
}


/*
 * Functions that check the validity of resources.
 */

static int validateClientWindow( IMIC *icp )
{
  return IMValidateWindow
    ( XtDisplay( icp->im->connection->protocol_widget ),
      icp->common_attr.client, &icp->client_profile ) ;
}

static int validateFocusWindow( IMIC *icp )
{
  IMCommonAttributes *ap = &icp->common_attr ;

  /*
   * This function assumes that the client window has already
   * been validated.
   */
  if( ( ap->set_mask & ATTR_MASK_CLIENT ) && ap->focus == ap->client ){
    icp->focus_profile = icp->client_profile ;
    return True ;
  } else {
    return IMValidateWindow
      ( XtDisplay( icp->im->connection->protocol_widget ),
	icp->common_attr.focus, &icp->focus_profile ) ;
  }
}


/*
 * Functions submit errors
 */

static void badSizeError( IMIC *icp, int code )
{
  IMSendError
    ( icp->im->connection, code, icp->im->id, icp->id,
      "invalid size of attribute value" ) ;
}

static void unnestedError( IMIC *icp )
{
  IMSendError
    ( icp->im->connection, IMBadSomething, icp->im->id, icp->id,
      "either preedit or status specification required" ) ;
}


/*
 * Functions getting IM attributes
 */

static IMPSAttributes *getPSPtr( IMIC *icp, int type )
{
  switch( type ){
  case NEST_PREEDIT :
    return &icp->preedit_attr ;
  case NEST_STATUS :
    return &icp->status_attr ;
  default :
    return NULL ;
  }
}

		/* request offset */
static int getIMValues( IMIM *imp, char *data, int len, int offset )
{
  unsigned int id ;		/* attribute ID */
  IMAttribute *attrp ;
  IMConnection *conn = imp->connection ;
  int byte_order = conn->byte_order ;

  while( len >= 2 ){
    id = getC16( data, byte_order ) ;
    data += 2 ;
    len -= 2 ;

    if( id > numImAttributes ){
      IMCancelRequest( conn, offset ) ;
      IMSendError
	( conn, IMBadSomething, imp->id, 0,
	  "invalid IM attribute ID" ) ;
      return -1 ;
    }

    attrp = &imAttributes[ id ] ;
    /* ڥ졼 GetValue Ǥ礦 */
    if( !( attrp->valid_ops & OP_G ) ){
      /* GetValue Ǥʤ°Ф GetValue ԤäΤǥ顼*/
      IMCancelRequest( conn, offset ) ;
      IMSendError
	( conn, IMBadSomething, imp->id, 0,
	  "invalid operation (IMGetValues) for this attribute" ) ;
	    return -1 ;
    }
    if( ( *attrp->get_proc )( imp, id, offset ) < 0 )
      return -1 ;
  }
  return 0 ;
}

/* ARGSUSED */
static int getQueryInputStyle( IMIM *imp, unsigned int id, int offset )
{
  IMConnection *conn = imp->connection ;
  unsigned int num_styles, num_bytes ;
  InputStyle *stp ;

  for( num_styles = 0, stp = styles ; stp->xim_style != 0 ; stp++ ){
    num_styles++ ;
  }
  num_bytes = num_styles * 4 + 4 ;
  IMPutC16( conn, id ) ;
  IMPutC16( conn, num_bytes ) ;
  IMPutC16( conn, num_styles ) ;
  IMPutC16( conn, 0 ) ;
  for( stp = styles ; stp->xim_style != 0 ; stp++ ){
    IMPutC32( conn, stp->xim_style ) ;
  }
  return 0 ;
}


/*
 * Functions setting IC attributes
 */

static int setICValues
( IMIC *icp, char *data, int len, int nest, int op )
{
  IMConnection *conn = icp->im->connection ;
  unsigned int imid = icp->im->id ;
  unsigned int icid = icp->id ;
  unsigned int id ;
  unsigned int value_len ;
  unsigned int attr_len ;
  char *value ;
  ICAttribute *attrp ;
  int byte_order = icp->im->connection->byte_order ;

  while( len > 0 ){
    if( len < 4 ){
      IMSendError
	( conn, IMBadSomething, imid, icid, "Bad attribute data" ) ;
      return -1 ;
    }
    id        = getC16( data, byte_order ) ;
    value_len = getC16( data + 2, byte_order ) ;
    attr_len  = PAD4( 4 + value_len ) ;

    if( attr_len > len ){
      IMSendError
	( conn, IMBadSomething, imid, icid, "Bad attribute length" ) ;
      return -1 ;
    }
    value = data + 4 ;

    if( id > numIcAttributes ){
      IMSendError
	( conn, IMBadSomething, imid, icid, "invalid IC attribute ID" ) ;
      return -1 ;
    }
    attrp = &icAttributes[ id ] ;
#ifdef DEBUG
    printf( "[IMID: %d, ICID: %d] IMSetICValues -- \"%s\"\n",
	    imid, icid, attrp->name ) ;
    fflush( stdout ) ;
#endif
    /* °Ф SetValues 뤳Ȥϲǽ */
    if( !( attrp->valid_ops & op ) ){
      IMSendError
	( conn, IMBadSomething, imid, icid,
	  "invalid operation for this attribute" ) ;
      return -1 ;
    }

    /*
     * Call attribute set procedure.
     */
    if( ( *attrp->set_proc )
	( icp, value, ( int )value_len, byte_order, nest, op ) < 0 ){
      /*
       * Error has occured.  The set procedure has already sent
       * appropriate error message, so just return here.
       */
      return -1 ;
    }
    data += attr_len ;
    len  -= attr_len ;
  }
  return 0 ;
}

/* ARGSUSED */
static int setInputStyle
( IMIC *icp, char *value, int len, int order, int nest, int op )
{
  CHECK_ICATTR_SIZE( 4, IMBadStyle ) ;

  /*
   * InputStyle must be set with CreateIC.
   */
  if( op != OP_C ){
    IMSendError
      ( icp->im->connection, IMBadStyle,
	icp->im->id, icp->id,
	"InputStyle cannot be changed by SetICValues" ) ;
    return -1 ;
  }
  
  icp->common_attr.input_style  = ( XIMStyle )getC32( value, order ) ;
  icp->common_attr.set_mask    |= ATTR_MASK_INPUT_STYLE ;
  icp->common_attr.change_mask |= ATTR_MASK_INPUT_STYLE ;
  return 0 ;
}

/* ARGSUSED */
static int setClientWindow
( IMIC *icp, char *value, int len, int order, int nest, int op )
{
  CHECK_ICATTR_SIZE( 4, IMBadClientWindow ) ;

  /*
   * ClientWindow cannot be changed.
   */
  if( icp->common_attr.set_mask & ATTR_MASK_CLIENT ){
#ifdef DEBUG
    printf( "[IMID: %d, IMIC: %d] ClientWindow already set\n" ) ;
    fflush( stdout ) ;
#endif
    IMSendError
      ( icp->im->connection, IMBadClientWindow,
	icp->im->id, icp->id, "ClientWindow already set" ) ;
    return -1 ;
  }

  icp->common_attr.client = (Window)getC32( value, order ) ;

  icp->common_attr.set_mask    |= ATTR_MASK_CLIENT ;
  icp->common_attr.change_mask |= ATTR_MASK_CLIENT ;
  return 0 ;
}

/* ARGSUSED */
static int setFocusWindow
( IMIC *icp, char *value, int len, int order, int nest, int op )
{
  Window focus ;

  CHECK_ICATTR_SIZE( 4, IMBadFocusWindow ) ;

  focus = ( Window )getC32( value, order ) ;

  if( !( icp->common_attr.set_mask & ATTR_MASK_FOCUS ) ||
      focus != icp->common_attr.focus ){
    icp->common_attr.change_mask |= ATTR_MASK_FOCUS ;
  }
  icp->common_attr.focus = focus ;
  icp->common_attr.set_mask |= ATTR_MASK_FOCUS ;
  return 0 ;
}

/* ARGSUSED */
static int setPreeditAttributes
( IMIC *icp, char *value, int len, int order, int nest, int op )
{
  return setICValues( icp, value, len, NEST_PREEDIT, op ) ;
}

/* ARGSUSED */
static int setStatusAttributes
( IMIC *icp, char *value, int len, int order, int nest, int op )
{
  return setICValues( icp, value, len, NEST_STATUS, op ) ;
}

/* ARGSUSED */
static int setArea
( IMIC *icp, char *value, int len, int order, int nest, int op )
{
  IMPSAttributes *ap ;
  XRectangle area ;

  CHECK_ICATTR_SIZE( 8, IMBadArea ) ;

  if( ( ap = getPSPtr( icp, nest ) ) == NULL ){
    unnestedError( icp ) ;
    return -1 ;
  }

  area.x =      getI16( value + 0, order ) ;
  area.y =      getI16( value + 2, order ) ;
  area.width  = getC16( value + 4, order ) ;
  area.height = getC16( value + 6, order ) ;

  if( !( ap->set_mask & ATTR_MASK_AREA ) ||
      area.x != ap->area.x ||
      area.y != ap->area.y ||
      area.width != ap->area.width ||
      area.height != ap->area.height ){
    ap->change_mask |= ATTR_MASK_AREA ;
  }

#ifdef DEBUG
  printf( "[setArea] (x,y,width,height) = (%d,%d,%d,%d)\n",
	  area.x, area.y, area.width, area.height ) ;
#endif
  ap->area.x      = area.x ;
  ap->area.y      = area.y ;
  ap->area.width  = area.width ;
  ap->area.height = area.height ;
  ap->set_mask   |= ATTR_MASK_AREA ;
  return 0 ;
}

/* ARGSUSED */
static int setAreaNeeded
( IMIC *icp, char *value, int len, int order, int nest, int op )
{
  IMPSAttributes *ap ;
  XRectangle area ;

  CHECK_ICATTR_SIZE( 8, IMBadArea ) ;

  if( ( ap = getPSPtr( icp, nest ) ) == NULL ){
    unnestedError( icp ) ;
    return -1 ;
  }
  area.width  = getC16( value + 4, order ) ;
  area.height = getC16( value + 6, order ) ;

  if( !( ap->set_mask & ATTR_MASK_AREA_NEEDED ) ||
      area.width != ap->area_needed.width ||
      area.height != ap->area_needed.height ){
    ap->change_mask |= ATTR_MASK_AREA_NEEDED ;
  }

#ifdef DEBUG
  printf( "[setAreaNeeded] (width,height) = (%d,%d)\n",
	  area.width, area.height ) ;
#endif
  ap->area_needed.width  = area.width ;
  ap->area_needed.height = area.height ;
  ap->set_mask          |= ATTR_MASK_AREA_NEEDED ;

  return 0 ;
}

/* ARGSUSED */
static int setForeground
( IMIC *icp, char *value, int len, int order, int nest, int op )
{
  IMPSAttributes *ap ;
  Pixel fore ;

  CHECK_ICATTR_SIZE( 4, IMBadForeground ) ;

  if( ( ap = getPSPtr( icp, nest ) ) == NULL ){
    unnestedError( icp ) ;
    return -1 ;
  }

  fore = getC32( value, order ) ;

  if( !( ap->set_mask & ATTR_MASK_FOREGROUND ) ||
      fore != ap->foreground ){
    ap->change_mask |= ATTR_MASK_FOREGROUND ;
  }
  ap->foreground = fore ;
  ap->set_mask |= ATTR_MASK_FOREGROUND ;
  return 0 ;
}

/* ARGSUSED */
static int setBackground
( IMIC *icp, char *value, int len, int order, int nest, int op )
{
  IMPSAttributes *ap ;
  Pixel back ;

  CHECK_ICATTR_SIZE( 4, IMBadBackground ) ;

  if( ( ap = getPSPtr( icp, nest ) ) == NULL ){
    unnestedError( icp ) ;
    return -1 ;
  }

  back = getC32( value, order ) ;

  if( !( ap->set_mask & ATTR_MASK_BACKGROUND ) ||
      back != ap->background ){
    ap->change_mask |= ATTR_MASK_BACKGROUND ;
  }
  ap->background = back ;
  ap->set_mask |= ATTR_MASK_BACKGROUND ;
  return 0 ;
}

/* ARGSUSED */
static int setColormap
( IMIC *icp, char *value, int len, int order, int nest, int op )
{
  IMPSAttributes *ap ;
  Colormap cmap ;

    CHECK_ICATTR_SIZE(4, IMBadColormap);

    if ((ap = getPSPtr(icp, nest)) == NULL) {
	unnestedError(icp);
	return -1;
    }

    cmap = getC32(value, order);

    if (!(ap->set_mask & ATTR_MASK_COLORMAP) || cmap != ap->colormap) {
	ap->change_mask |= ATTR_MASK_COLORMAP;
    }
    ap->colormap = cmap;
    ap->set_mask |= ATTR_MASK_COLORMAP;
    return 0;
}

/* ARGSUSED */
static int
setBgPixmap(icp, value, len, order, nest, op)
IMIC *icp;
char *value;
int len;
int order;
int nest;
int op;
{
    IMPSAttributes *ap;
    Pixmap pixmap;

    CHECK_ICATTR_SIZE(4, IMBadPixmap);

    if ((ap = getPSPtr(icp, nest)) == NULL) {
	unnestedError(icp);
	return -1;
    }

    pixmap = getC32(value, order);

    if (!(ap->set_mask & ATTR_MASK_BG_PIXMAP) || pixmap != ap->bg_pixmap) {
	ap->change_mask |= ATTR_MASK_BG_PIXMAP;
    }

    ap->bg_pixmap = pixmap;
    ap->set_mask |= ATTR_MASK_BG_PIXMAP;

    return 0;
}

/* ARGSUSED */
static int
setLineSpace(icp, value, len, order, nest, op)
IMIC *icp;
char *value;
int len;
int order;
int nest;
int op;
{
    IMPSAttributes *ap;
    int line_space;

    CHECK_ICATTR_SIZE(2, IMBadSomething);

    if ((ap = getPSPtr(icp, nest)) == NULL) {
	unnestedError(icp);
	return -1;
    }

    line_space = getI16(value, order);	/* ??? linespacing is 'int' */

    if (!(ap->set_mask & ATTR_MASK_LINESPACE) ||
	line_space != ap->line_space) {
	ap->change_mask |= ATTR_MASK_LINESPACE;
    }
    ap->line_space = line_space;
    ap->set_mask |= ATTR_MASK_LINESPACE;
    return 0;
}

/* ARGSUSED */
static int
setCursor(icp, value, len, order, nest, op)
IMIC *icp;
char *value;
int len;
int order;
int nest;
int op;
{
    IMPSAttributes *ap;
    Cursor cursor;

    CHECK_ICATTR_SIZE(4, IMBadCursor);

    if ((ap = getPSPtr(icp, nest)) == NULL) {
	unnestedError(icp);
	return -1;
    }

    cursor = getC32(value, order);

    if (!(ap->set_mask & ATTR_MASK_CURSOR) || cursor != ap->cursor) {
	ap->change_mask |= ATTR_MASK_CURSOR;
    }
    ap->cursor = cursor;
    ap->set_mask |= ATTR_MASK_CURSOR;
    return 0;
}

/* ARGSUSED */
static int setSpotLocation
( IMIC *icp, char *value, int len, int order, int nest, int op )
{
  IMPSAttributes *ap ;
  XPoint spot ;

  CHECK_ICATTR_SIZE( 4, IMBadSpotLocation ) ;

  if( nest == NEST_STATUS ){
    IMSendError( icp->im->connection, IMBadSpotLocation,
		 icp->im->id, icp->id,
		 "spot location isn't a status attribute" ) ;
    return -1 ;
  }

  ap = &icp->preedit_attr ;

  spot.x = getI16( value, order ) ;
  spot.y = getI16( value + 2, order ) ;

  if( !( ap->set_mask & ATTR_MASK_SPOT_LOCATION ) ||
      spot.x != ap->spot_location.x || spot.y != ap->spot_location.y ){
    ap->change_mask |= ATTR_MASK_SPOT_LOCATION ;
  }
  ap->spot_location.x = spot.x ;
  ap->spot_location.y = spot.y ;
  ap->set_mask |= ATTR_MASK_SPOT_LOCATION ;
  return 0 ;
}

/* ARGSUSED */
static int setStdColormap
( IMIC *icp, char *value, int len, int order, int nest, int op )
{
  IMPSAttributes *ap;
  Atom colormap_name;
  XStandardColormap *stdcolormaps;
  Widget w = icp->im->connection->protocol_widget;
  Display *dpy = XtDisplay(w);
  int ncolormaps;
  Window root;
  int status;

  CHECK_ICATTR_SIZE(4, IMBadAtom);

  if ((ap = getPSPtr(icp, nest)) == NULL) {
    unnestedError(icp);
    return -1;
  }
  
  colormap_name = getC32(value, order);
  if (icp->common_attr.set_mask & ATTR_MASK_CLIENT) {
    root = icp->client_profile.root;
  } else if (icp->common_attr.set_mask & ATTR_MASK_FOCUS) {
    root = icp->focus_profile.root;
  } else {
    /*
     * Client has not specified client window yet.
     * Reading standard colormap property should been deffered
     * until the window is set, but for now...
     */
    root = RootWindowOfScreen(XtScreen(w));
  }
  
  status = XGetRGBColormaps(dpy, root,
			    &stdcolormaps, &ncolormaps, colormap_name);
  /*XAEUnset(h) ; 顼̵뤷Ƥߤ*/
  if (!status || ncolormaps < 0) {
    IMSendError(icp->im->connection, IMBadName, icp->im->id, icp->id,
		"invalid standard colormap name");
    return -1;
  }
  
  if (!(ap->set_mask & ATTR_MASK_STD_COLORMAP) ||
      colormap_name != ap->std_colormap) {
    ap->change_mask |= ATTR_MASK_STD_COLORMAP;
  }
  ap->std_colormap = colormap_name;
  ap->colormap = stdcolormaps[0].colormap;
  
  ap->set_mask |= ATTR_MASK_STD_COLORMAP | ATTR_MASK_COLORMAP;
  XFree((char *)stdcolormaps);
  return 0;
}

/* ARGSUSED */
static int setFontSet
( IMIC *icp, char *value, int len, int order, int nest, int op )
{
  IMPSAttributes *ap ;
  unsigned int name_list_len ;
  char *name_list ;
  
  if( len < 2 ){
    badSizeError( icp, IMBadName ) ;
    return -1;
  }
  name_list_len = getC16( value, order ) ;
  if( 2 + name_list_len > len ){
    badSizeError( icp, IMBadName ) ;
    return -1 ;
  }
  
  if( ( ap = getPSPtr( icp, nest ) ) == NULL ){
    unnestedError( icp ) ;
    return -1 ;
  }
  
  name_list = malloc( name_list_len + 1 ) ;
  memcpy( name_list, value + 2, name_list_len ) ;
  name_list[ name_list_len ] = '\0' ;
  
#if defined(DEBUG)
  fprintf( stderr, "Set FontSet : %s\n", name_list ) ;
#endif
  if( ap->set_mask & ATTR_MASK_FONT_SET ){
    if( !strcmp( name_list, ap->font_set ) ){
      free( name_list ) ;
    } else {
      ap->change_mask |= ATTR_MASK_FONT_SET ;
      if( ap->font_set != IMDefaultFontSet( icp->im ) ){
	free( ap->font_set ) ;
      }
      ap->font_set = name_list ;
    }
  } else {
    ap->font_set = name_list ;
    ap->set_mask |= ATTR_MASK_FONT_SET ;
    ap->change_mask |= ATTR_MASK_FONT_SET ;
  }
  return 0 ;
}


/*
 * Functions getting IC attributes
 */

static int getICValues
( IMIC *icp, char *data, int len, int nest, int offset, int *sepp ) 
{
  unsigned int id;		/* attribute ID */
  ICAttribute *attrp;
  IMConnection *conn = icp->im->connection;
  int byte_order = conn->byte_order;
  char *org_data = data;
  int r;
  
  while (len >= 2) {
    id = getC16(data, byte_order);
    data += 2;
    len -= 2;
    
    if (id > numIcAttributes) {
      /* invalid attribute ID */
      IMCancelRequest( conn, offset ) ;
      IMSendError
	( conn, IMBadSomething, icp->im->id, icp->id,
	  "invalid IC attribute ID" ) ;
      return -1;
    }
    
    attrp = &icAttributes[ id ] ;
    if( attrp->type == TYPE_SEPARATOR ){
      /* nested list separator */
      *sepp = 1 ;
      return data - org_data ;
    }
    
    if (!(attrp->valid_ops & OP_G)) {
      IMCancelRequest( conn, offset ) ;
      IMSendError(conn, IMBadSomething, icp->im->id, icp->id,
		  "invalid operation (ICGetValues) for this attribute");
      return -1 ;
    }
    
    r = (*attrp->get_proc)(icp, id, nest, offset, data, len);
    /*
     * The return value of get_proc is usually 0, indicating success.
     * If it is less than 0, there are some errors.
     * If it is greater than 0,
     */
    if( r < 0 )
      return -1 ;
    
    data += r ;		/* r is extra offset */
    len -= r ;
  }
  *sepp = 0 ;
  return data - org_data ;
}

static int getPSAttributes
( IMIC *icp, unsigned int id, int nest, int offset, char *data, int len )
{
  IMConnection *conn = icp->im->connection;
  unsigned int length;
  int length_offset;
  int attr_offset;
  int nested_separator;
  int r;
  
  IMPutC16( conn, id ) ;
  
  length_offset = IMWritePos( conn ) ;
  IMPutC16( conn, 0 ) ;		/* dummy -- overwritten afterwards */
  
  attr_offset = IMWritePos( conn ) ;
  
  r = getICValues
    ( icp, data, len, nest, offset, &nested_separator ) ;
  if( r < 0 )
    return -1 ;
  
  /*
   * Nested list is written on the output buffer.
   * Calculate the length of the list.
   */
  length = IMWritePos( conn ) - attr_offset ;
  
  /* rewrite attribute length field */
  IMRewriteC16( conn, length_offset, length ) ;
  IMPutPad( conn ) ;
  
  return r ;
}

/* ARGSUSED */
static int getPreeditAttributes
( IMIC *icp, unsigned int id, int nest, int offset,
  char *data, int len )
{
  return getPSAttributes
    ( icp, id, NEST_PREEDIT, offset, data, len ) ;
}

/* ARGSUSED */
static int getStatusAttributes
( IMIC *icp, unsigned int id, int nest,		/* unused */
  int offset, char *data, int len )
{
  return getPSAttributes
    ( icp, id, NEST_STATUS, offset, data, len ) ;
}

/* ARGSUSED */
static int getInputStyle
( IMIC *icp, unsigned int id, int nest, int offset, char *data, int len )
{
  IMConnection *conn = icp->im->connection;

  /*
   * Input style must have been specified, (and validated)
   * at IC creation.  No need for checking.
   */
  IMPutC16(conn, id);		/* attribute ID */
  IMPutC16(conn, 4);		/* value length */
  IMPutC32(conn, icp->common_attr.input_style);
  return 0;
}

/* ARGSUSED */
static int getClientWindow
( IMIC *icp, unsigned int id, int nest,
  int offset, char *data, int len )
{
  IMConnection *conn = icp->im->connection;

  if (icp->common_attr.set_mask & ATTR_MASK_CLIENT) {
    IMPutC16(conn, id);		/* attribute ID */
    IMPutC16(conn, 4);		/* value length */
    IMPutC32(conn, icp->common_attr.client);
    return 0;
  } else {
    /* no default is available */
    /* ̿Υ󥻥Ȼפ*/
    IMCancelRequest( conn, offset ) ;
    IMSendError(conn, IMBadClientWindow, icp->im->id, icp->id,
		"client window not specified yet");
    return -1;
    }
}

/* ARGSUSED */
static int getFocusWindow
( IMIC *icp, unsigned int id, int nest, int offset,
  char *data, int len )
{
  IMConnection *conn = icp->im->connection ;

  if (!(icp->common_attr.set_mask & ATTR_MASK_FOCUS)) {
    /* fill default value */
    fillCommonDefault(icp, (unsigned long)ATTR_MASK_FOCUS);
  }
  
  if (icp->common_attr.set_mask & ATTR_MASK_FOCUS) {
    IMPutC16(conn, id);		/* attribute ID */
    IMPutC16(conn, 4);		/* value length */
    IMPutC32(conn, icp->common_attr.focus);
    return 0;
  } else {
    /*
     * Couldn't get the default value.  That is, neither
     * focus window nor client window is specified yet.
     */
    IMCancelRequest( conn, offset ) ;
    IMSendError(conn, IMBadFocusWindow, icp->im->id, icp->id,
		"neither of client/focus window not specified yet");
    return -1 ;
  }
}

/* ARGSUSED */
static int getArea
( IMIC *icp, unsigned int id, int nest,
  int offset, char *data, int len )
{
  IMConnection *conn = icp->im->connection;
  IMPSAttributes *ap;
  
  if ((ap = getPSPtr(icp, nest)) == NULL) {
    /* ̿ʬΥ󥻥Ȼפ*/
    IMCancelRequest( conn, offset ) ;
    unnestedError(icp);
    return -1;
  }
  
  if( !( ap->set_mask & ATTR_MASK_AREA ) ){
    fillPSDefault( icp, nest, (unsigned long)ATTR_MASK_AREA ) ;
  }
  
#ifdef DEBUG
  printf( "[getArea] (x,y,width,height) = (%d,%d,%d,%d)\n",
	  ap->area.x, ap->area.y, ap->area.width, ap->area.height ) ;
#endif
  IMPutC16( conn, id ) ;	/* attribute ID */
  IMPutC16( conn, 8 ) ;		/* value length */
  IMPutI16( conn, ap->area.x ) ;
  IMPutI16( conn, ap->area.y ) ;
  IMPutC16( conn, ap->area.width ) ;
  IMPutC16( conn, ap->area.height ) ;
  return 0;
}

/* ARGSUSED */
static int getAreaNeeded
( IMIC *icp, unsigned int id, int nest,
  int offset, char *data, int len )
{
  IMConnection *conn = icp->im->connection;
  IMPSAttributes *ap;
  
  if ((ap = getPSPtr(icp, nest)) == NULL) {
    IMCancelRequest( conn, offset ) ;
    unnestedError(icp);
    return -1;
  }
  
  if( !( ap->set_mask & ATTR_MASK_AREA_NEEDED ) ){
    fillPSDefault
      ( icp, nest, ( unsigned long )ATTR_MASK_AREA_NEEDED ) ;	/* ??? */
  }
#ifdef DEBUG
  printf( "[getAreaNeeded] (x,y,width,height) = (%d,%d,%d,%d)\n",
	  ap->area_needed.x, ap->area_needed.y,
	  ap->area_needed.width, ap->area_needed.height ) ;
#endif
  
  IMPutC16( conn, id ) ;	/* attribute ID */
  IMPutC16( conn, 8 ) ;		/* value length */
  IMPutI16( conn, ap->area_needed.x ) ;
  IMPutI16( conn, ap->area_needed.y ) ;
  IMPutC16( conn, ap->area_needed.width ) ;
  IMPutC16( conn, ap->area_needed.height ) ;
  return 0 ;
}

/* ARGSUSED */
static int getSpotLocation
( IMIC *icp, unsigned int id, int nest, int offset, char *data, int len )
{
  IMConnection *conn = icp->im->connection ;
  IMPSAttributes *ap = &icp->preedit_attr ;

  if( nest == NEST_STATUS ){
    IMCancelRequest( conn, offset ) ;
    IMSendError
      ( conn, IMBadSomething, icp->im->id, icp->id,
       "spot location isn't a status attribute" ) ;
    return -1 ;
  }

  if( !( ap->set_mask & ATTR_MASK_SPOT_LOCATION ) ){
    fillPSDefault
      ( icp, NEST_PREEDIT, ( unsigned long )ATTR_MASK_SPOT_LOCATION ) ;
  }
  IMPutC16( conn, id ) ;		/* attribute ID */
  IMPutC16( conn, 4 ) ;			/* value length */
  IMPutI16( conn, ap->spot_location.x ) ;
  IMPutI16( conn, ap->spot_location.y ) ;
  return 0 ;
}

/* ARGSUSED */
static int getColormap
( IMIC *icp, unsigned int id, int nest, int offset, char *data, int len )
{
  IMConnection *conn = icp->im->connection ;
  IMPSAttributes *ap ;

  if( ( ap = getPSPtr( icp, nest ) ) == NULL ){
    IMCancelRequest( conn, offset ) ;
    unnestedError( icp ) ;
    return -1 ;
  }

  if( !( ap->set_mask & ATTR_MASK_COLORMAP ) ){
    fillPSDefault( icp, nest, ( unsigned long )ATTR_MASK_COLORMAP ) ;
  }

  IMPutC16( conn, id ) ;		/* attribute ID */
  IMPutC16( conn, 4 ) ;			/* value length */
  IMPutC32( conn, ap->colormap ) ;
  return 0 ;
}

/* ARGSUSED */
static int getStdColormap
( IMIC *icp, unsigned int id, int nest, int offset, char *data, int len )
{
  IMConnection *conn = icp->im->connection;
  IMPSAttributes *ap;
  Atom colormap_name;
  
  if ((ap = getPSPtr(icp, nest)) == NULL) {
    IMCancelRequest( conn, offset ) ;
    unnestedError(icp);
    return -1;
  }
  
  if (ap->set_mask & ATTR_MASK_STD_COLORMAP) {
    colormap_name = ap->std_colormap;
  } else {
    /* what to do? */
    colormap_name = None;
  }
  
  IMPutC16(conn, id);		/* attribute ID */
  IMPutC16(conn, 4);		/* value length */
  IMPutC32(conn, colormap_name);
  return 0;
}

/* ARGSUSED */
static int getForeground
( IMIC *icp, unsigned int id, int nest, int offset, char *data, int len )
{
  IMConnection *conn = icp->im->connection;
  IMPSAttributes *ap;

  if ((ap = getPSPtr(icp, nest)) == NULL) {
    IMCancelRequest( conn, offset ) ;
    unnestedError(icp);
    return -1;
  }
  
  if (!(ap->set_mask & ATTR_MASK_FOREGROUND)) {
    fillPSDefault(icp, nest, (unsigned long)ATTR_MASK_FOREGROUND);
  }
  
  IMPutC16(conn, id);		/* attribute ID */
  IMPutC16(conn, 4);		/* value length */
  IMPutC32(conn, ap->foreground);
  return 0;
}

/* ARGSUSED */
static int getBackground
( IMIC *icp, unsigned int id, int nest, 
  int offset, char *data, int len )
{
  IMConnection *conn = icp->im->connection;
  IMPSAttributes *ap;
  
  if ((ap = getPSPtr(icp, nest)) == NULL) {
    IMCancelRequest( conn, offset ) ;
    unnestedError( icp ) ;
    return -1;
  }
  
  if (!(ap->set_mask & ATTR_MASK_BACKGROUND)) {
    fillPSDefault(icp, nest, (unsigned long)ATTR_MASK_BACKGROUND);
  }
  
  IMPutC16(conn, id);		/* attribute ID */
  IMPutC16(conn, 4);		/* value length */
  IMPutC32(conn, ap->background);
  return 0;
}

/* ARGSUSED */
static int getBgPixmap
( IMIC *icp, unsigned int id, int nest,
  int offset, char *data, int len )
{
  IMConnection *conn = icp->im->connection ;
  IMPSAttributes *ap;
  
  if ((ap = getPSPtr(icp, nest)) == NULL) {
    IMCancelRequest( conn, offset ) ;
    unnestedError( icp ) ;
    return -1;
  }
  
  if (!(ap->set_mask & ATTR_MASK_BG_PIXMAP)) {
    fillPSDefault(icp, nest, (unsigned long)ATTR_MASK_BG_PIXMAP);
  }
  
  IMPutC16(conn, id);		/* attribute ID */
  IMPutC16(conn, 4);		/* value length */
  IMPutC32(conn, ap->bg_pixmap);
  return 0;
}

/* ARGSUSED */
static int getFontSet
( IMIC *icp, unsigned int id, int nest, int offset, char *data, int len )
{
  IMConnection *conn = icp->im->connection ;
  IMPSAttributes *ap ;
  int name_len ;

  if( ( ap = getPSPtr( icp, nest ) ) == NULL ){
    IMCancelRequest( conn, offset ) ;
    unnestedError( icp ) ;
    return -1 ;
  }
  
  if( !( ap->set_mask & ATTR_MASK_FONT_SET ) ){
    fillPSDefault( icp, nest, ( unsigned long )ATTR_MASK_FONT_SET ) ;
  }

  name_len = strlen( ap->font_set ) ;

  IMPutC16( conn, id ) ;		/* attribute ID */
  
  IMPutC16( conn, ( unsigned int )name_len ) ;	/* value length */
  IMPutString( conn, ap->font_set, name_len ) ;
  IMPutPad( conn ) ;
  return 0 ;
}

/* ARGSUSED */
static int getLineSpace
( IMIC *icp, unsigned int id, int nest, int offset,
  char *data, int len )
{
  IMConnection *conn = icp->im->connection;
  IMPSAttributes *ap;

  if ((ap = getPSPtr(icp, nest)) == NULL) {
    IMCancelRequest( conn, offset ) ;
    unnestedError( icp ) ;
    return -1;
  }
  
  if (!(ap->set_mask & ATTR_MASK_LINESPACE)) {
    fillPSDefault(icp, nest, (unsigned long)ATTR_MASK_LINESPACE);
  }
  
  IMPutC16(conn, id);		/* attribute ID */
  IMPutC16(conn, 4);	/* value length */
  IMPutC32(conn, (unsigned long)ap->line_space);
  return 0;
}

/* ARGSUSED */
static int getCursor
( IMIC *icp, unsigned int id, int nest, int offset, 
  char *data, int len )
{
  IMConnection *conn = icp->im->connection;
  IMPSAttributes *ap;
  
  if ((ap = getPSPtr(icp, nest)) == NULL) {
    IMCancelRequest( conn, offset ) ;
    unnestedError( icp ) ;
    return -1 ;
  }
  
  if (!(ap->set_mask & ATTR_MASK_CURSOR)) {
    fillPSDefault(icp, nest, (unsigned long)ATTR_MASK_CURSOR);
  }
  
  IMPutC16(conn, id);		/* attribute ID */
  IMPutC16(conn, 4);		/* value length */
  IMPutC32(conn, ap->cursor);
  return 0;
}

static void changeFonts( IMIC *icp )
{
  /* եȤݤ롣*/
  fontMgr_PrepareFontByName
    ( XtDisplay( icp->im->connection->protocol_widget ),
      icp->fontset, icp->preedit_attr.font_set ) ;
  return ;
}


/*
 * Functions computing default attribute values
 */

static void fillCommonDefault( IMIC *icp, unsigned long mask )
{
  IMCommonAttributes *ap = &icp->common_attr ;

  /*
   * Don't bother with the attributes which have been set.
   */
  mask &= ~ap->set_mask ;

  /*
   * The only attribute that have default value is FocusWindow.
   */
  if( mask & ATTR_MASK_FOCUS ){
    /* if ClientWindow is not set... no way */
    if( mask & ATTR_MASK_CLIENT ){
      ap->focus = ap->client ;
      ap->set_mask |= ATTR_MASK_FOCUS ;
      icp->focus_profile = icp->client_profile ;
    }
  }
}

static int getNaturalLineSpace( IMIC *icp )
{
  int max_ascent = 0, max_descent = 0 ;
  int i ;

  changeFonts( icp ) ;

  for( i = 0 ; i < NUMBER_OF_CHARSET ; i++ ){
    if( icp->fontset[ i ] == NULL )
      continue ;
    if( max_ascent < icp->fontset[ i ]->font->ascent )
      max_ascent = icp->fontset[ i ]->font->ascent ;
    if( max_descent < icp->fontset[ i ]->font->descent )
      max_descent = icp->fontset[ i ]->font->descent ;
  }
  if( max_ascent + max_descent < MIN_LINE_SPACING ){
    return MIN_LINE_SPACING ;
  } else {
    return max_ascent + max_descent ;
  }
}

static void fillPSDefault( IMIC *icp, int type, unsigned long mask )
{
  IMPSAttributes *ap ;
  IMConnection *conn = icp->im->connection ;
  Widget pw = conn->protocol_widget ;
  int preedit ;
#if 0
  char *typename = (type == NEST_PREEDIT) ? "preedit" : "status" ;
#endif

  preedit = ( type == NEST_PREEDIT ) ;
  ap = ( ( preedit )? &icp->preedit_attr : &icp->status_attr ) ;

  /*
   * Don't bother with the attributes which have been set.
   * But area_needed needs to be computed each time (to get
   * correct X and Y coordinates).
   */
  mask &= ( ( ~ap->set_mask ) | ATTR_MASK_AREA_NEEDED ) ;
  
  if( mask & ATTR_MASK_AREA ){
    computeAreaForQuery( icp ) ;
    ap->set_mask |= ATTR_MASK_AREA ;
  }
  if( mask & ATTR_MASK_FOREGROUND ){
    ap->foreground = IMDefaultForeground( pw ) ;
    ap->set_mask |= ATTR_MASK_FOREGROUND ;
  }
  if( mask & ATTR_MASK_BACKGROUND ){
    ap->background = IMDefaultBackground( pw ) ;
    ap->set_mask |= ATTR_MASK_BACKGROUND;
  }
  if( mask & ATTR_MASK_COLORMAP ){
    ap->colormap = pw->core.colormap ;
    ap->set_mask |= ATTR_MASK_COLORMAP ;
  }
  if( mask & ATTR_MASK_STD_COLORMAP ){
    /* you can't fill default. what to do? */
  }
  if( mask & ATTR_MASK_BG_PIXMAP ){
    ap->bg_pixmap = None ;
    ap->set_mask |= ATTR_MASK_BG_PIXMAP ;
  }
  if( mask & ATTR_MASK_LINESPACE ){
    if( !( ap->set_mask & ATTR_MASK_FONT_SET ) ){
      fillPSDefault
	( icp, type, ( unsigned long )ATTR_MASK_FONT_SET ) ;
    }
    ap->line_space = getNaturalLineSpace( icp ) ;
    ap->set_mask |= ATTR_MASK_LINESPACE ;
  }
  if( mask & ATTR_MASK_CURSOR ){
    ap->cursor = None ;
    ap->set_mask |= ATTR_MASK_CURSOR ;
  }
  if( mask & ATTR_MASK_AREA_NEEDED ){
    computeAreaNeeded( icp ) ;
  }
  if( mask & ATTR_MASK_FONT_SET ){
    ap->font_set = IMDefaultFontSet( icp->im ) ;
    ap->set_mask |= ATTR_MASK_FONT_SET ;
  }
  if (mask & ATTR_MASK_SPOT_LOCATION) {
    ap->spot_location.x = ap->spot_location.y = 0 ;
    ap->set_mask |= ATTR_MASK_SPOT_LOCATION ;
  }
  return ;
}

/*
 * Function validating attribute values
 */

static int validateCommonAttr
( IMIC *icp, int checkonly )
{
  IMCommonAttributes *ap = &icp->common_attr;
  IMConnection *conn = icp->im->connection;
  unsigned long mask = ap->change_mask;
  int ret = 0;
  
  mask &= ap->set_mask;

#define SENDERROR(code, msg) \
  if (!checkonly && ret == 0) { \
	  IMSendError(conn, code, icp->im->id, icp->id, msg); ret = -1; \
  }

  if( mask & ATTR_MASK_INPUT_STYLE ){
    XIMStyle xstyle = icp->common_attr.input_style ;
    InputStyle *isp = styles ;

    while( isp->xim_style != 0 ){
      if( isp->xim_style == xstyle )
	break ;
      isp++ ;
    }
    if( isp->xim_style == 0 ){
      SENDERROR( IMBadStyle, "unsupported input style" ) ;
    } else {
      icp->style = isp->conversion_style ;
    }
  }
  if( mask & ATTR_MASK_CLIENT ){
    if( !validateClientWindow( icp ) ){
      SENDERROR( IMBadClientWindow, "invalid client window ID" ) ;
    }
  }
  if( mask & ATTR_MASK_FOCUS ){
    if( !validateFocusWindow( icp ) ){
      SENDERROR( IMBadFocusWindow, "invalid focus window ID" ) ;
    }
  }

  return ret ;
#undef SENDERROR
}

static int validatePSAttr
( IMIC *icp, int type, int checkonly )
{
  /*
   * Check validity of preedit/status attribute values.
   * 'type' is either NEST_PREEDIT or NEST_STATUS, indicating
   * whether preedit or status attributes are to be checked.
   * 'mask' is the attribute mask to be checked.
   * If all the attributes are valid, this function return 0.
   * Otherwise it issues an error message for the first invalid
   * value detected, and returns -1.
   */
  IMPSAttributes *ap ;
  IMConnection *conn = icp->im->connection ;
  unsigned long mask ;
  int preedit ;
  int ret = 0 ;
  
  preedit = ( type == NEST_PREEDIT ) ;
  ap = ( ( preedit )? &icp->preedit_attr : &icp->status_attr ) ;
  
  /* do not check unset attributes */
  mask = ( ap->change_mask & ap->set_mask ) ;
  
#define SENDERROR(code, msg) \
  if (!checkonly && ret == 0) { \
     IMSendError(conn, code, icp->im->id, icp->id, msg); ret = -1; \
  }
    
  if( mask & ATTR_MASK_AREA ){
    if( ap->area.width == 0 || ap->area.height == 0 ){
      ap->set_mask &= ~ATTR_MASK_AREA ;
      SENDERROR( IMBadArea, "invalid area width/height" ) ;
    }
  }
  if( mask & ATTR_MASK_LINESPACE ){
    if( ap->line_space < MIN_LINE_SPACING ){
      ap->set_mask &= ~ATTR_MASK_LINESPACE ;
    }
  }
  if( mask & ATTR_MASK_AREA_NEEDED ){
  }
  return ret ;
#undef SENDERROR
}


/*
 * Functions to extract necessary attributes and make conversion argument
 */

static void changeConversionAttributes( IMIC *icp )
{
  struct ConvAttrsMesg camsg ;

  camsg.mask = IMMakeConvAttributes( icp, &( camsg.value ) ) ;

  /* ֤ࡣľ Widget ϤƤɤΤˡġ*/
  /*   ֤äȡȤƤäǤ */ 
  /* ֤Ĥʡ(ˤϴʤä)*/
  XtVaSetValues( icp->conversion, XtNconversionAttribute, &camsg, NULL ) ;
  return ;
}

static void computeAreaNeeded( IMIC *icp )
{
  IMPSAttributes *pattr = &icp->preedit_attr ;
  IMPSAttributes *sattr = &icp->status_attr ;
  IMWindowProfile *cpr = &icp->client_profile ;
  int width, height ;
  int max_width, max_height ;
  int default_status_width ;
  int font_height ;

  if( icp->style == IMSTYLE_SEPARATE )
    return ;

#ifdef DEBUG
  printf( "[Compute Area Needed]\n" ) ;
#endif
  /*
   * Get the current dimension of the client window.
   */
  ( void )validateClientWindow( icp ) ;

  /*
   * Compute the dimensions of the status region.
   */
  fillPSDefault
    ( icp, NEST_STATUS, (unsigned long)ATTR_MASK_LINESPACE ) ;
  font_height = sattr->line_space + 2 ;
  max_width = max_height = 0 ;
  if( sattr->set_mask & ATTR_MASK_AREA_NEEDED ){
    max_width  = sattr->area_needed.width ;
    max_height = sattr->area_needed.height ;
  }
  
  default_status_width = IMStatusWidth
    ( icp->im->connection->protocol_widget ) ;
  if( default_status_width > 0 ){
    width = default_status_width ;
  } else {
    width = cpr->width / 5 ;		/* wild guess */

    if( icp->style != IMSTYLE_OFF_THE_SPOT ){
      /* ΤˤƲ˥եȤι⤵ˤ 40 ʸС80ʸ
       * ʸɽǤ뤳ȤˤʤȻפäƤɤȻפ*/
      if( width < font_height * 40 ){
	width = font_height * 40 ;	/* another wild guess */
      }
    } else {
      if( width < font_height * 3 ){
	width = font_height * 3 ;	/* another wild guess */
      }
    }
  }
  height = font_height ;
  /* ¤¤β᤹ġ*/
  if( width < max_width && max_width > 0 )
    width = max_width ;
  if( max_height > 0 && height > max_height )
    height = max_height ;
  if( width < MIN_AREA_WIDTH )
    width = MIN_AREA_WIDTH ;
  if( height < MIN_AREA_HEIGHT )
    height = MIN_AREA_HEIGHT ;
  
  sattr->area_needed.x = 0 ;
  sattr->area_needed.y = cpr->height - height ;
  sattr->area_needed.width  = width ;
  sattr->area_needed.height = height ;

  /*
   * Compute the dimensions of the pre-edit region.
   */
  if( icp->style != IMSTYLE_OFF_THE_SPOT )
    return ;

  fillPSDefault
    (icp, NEST_PREEDIT, (unsigned long)ATTR_MASK_LINESPACE) ;
  font_height = pattr->line_space + 2 ;
  max_width = max_height = 0 ;
  if( pattr->set_mask & ATTR_MASK_AREA_NEEDED ){
    max_width = pattr->area_needed.width ;
    max_height = pattr->area_needed.height ;
  }
  
  width = cpr->width - sattr->area_needed.width ;
  height = font_height ;

  if( max_width > 0 && width > max_width )
    width = max_width ;
  if( max_height > 0 && height > max_height )
    height = max_height ;
  if( width < MIN_AREA_WIDTH )
    width = MIN_AREA_WIDTH ;
  if( height < MIN_AREA_HEIGHT )
    height = MIN_AREA_HEIGHT ;
  
  pattr->area_needed.x = sattr->area_needed.width ;
  pattr->area_needed.y = cpr->height - height ;
  pattr->area_needed.width = width ;
  pattr->area_needed.height = height ;
}

static void computeAreaForQuery( IMIC *icp )
{
  IMPSAttributes *pattr = &icp->preedit_attr ;
  IMPSAttributes *sattr = &icp->status_attr ;

  if( icp->style == IMSTYLE_SEPARATE )
    return ;

  computeAreaNeeded( icp ) ;

  if( !( pattr->set_mask & ATTR_MASK_AREA ) ){
    if( icp->style == IMSTYLE_OVER_THE_SPOT ){
      IMWindowProfile *fpr = &icp->focus_profile ;
      
      pattr->area.x = 0 ;
      pattr->area.y = 0 ;
      pattr->area.width = fpr->width ;
      pattr->area.height = fpr->height ;
    } else {	/* IMSTYLE_OFF_THE_SPOT */
      pattr->area = pattr->area_needed ;
    }
  }
  if( !( sattr->set_mask & ATTR_MASK_AREA ) ){
    sattr->area = sattr->area_needed ;
  }
}


/*
 * Public functions
 */

/*- IMPutIMAttrList: write list of supported IM attributes to output buffer -*/
void IMPutIMAttrList( IMIM *imp )
{
  IMConnection *conn = imp->connection ;
  IMAttribute *iap ;
  int offset, list_start, list_end ;
  int n ;

  offset = IMWritePos(conn) ;
  IMPutC16(conn, 0) ;		/* dummy. overwritten afterwards */

  list_start = IMWritePos(conn);
  for (n = 0, iap = imAttributes; n < numImAttributes; n++, iap++) {
    int length;
    
    IMPutC16(conn, (unsigned int)n);
    IMPutC16(conn, (unsigned int)iap->type);
    length = strlen(iap->name);
    IMPutC16(conn, (unsigned int)length);
    IMPutString(conn, iap->name, length);
    IMPutPad(conn);
  }
  list_end = IMWritePos(conn);
  IMRewriteC16(conn, offset, (unsigned int)(list_end - list_start));
}

/*- IMPutICAttrList: write list of supported IC attributes to output buffer -*/
void IMPutICAttrList( IMIM *imp )
{
  IMConnection *conn = imp->connection;
  ICAttribute *iap;
  int offset, list_start, list_end;
  int n;
  
  offset = IMWritePos(conn);
  IMPutC16(conn, 0);		/* dummy. overwritten afterwards */
  IMPutC16(conn, 0);		/* unused */
  
  list_start = IMWritePos(conn);
  for (n = 0, iap = icAttributes; n < numIcAttributes; n++, iap++) {
    int length;
    
    IMPutC16(conn, (unsigned int)n);
    IMPutC16(conn, (unsigned int)iap->type);
    length = strlen(iap->name);
    IMPutC16(conn, (unsigned int)length);
    IMPutString(conn, iap->name, length);
    IMPutPad(conn);
  }
  list_end = IMWritePos(conn);
  IMRewriteC16(conn, offset, (unsigned int)(list_end - list_start));
}

/* ARGSUSED */
int IMSetIMValues( IMIM *imp, char *data, int len, int major )
{
  /* not supported yet */

  IMSendError(imp->connection, IMBadProtocol, imp->id, 0,
	      "this protocol is not supported yet") ;
  return -1 ;
}

int IMGetIMValues( IMIM *imp, char *data, int len, int offset )
{
  IMConnection *conn = imp->connection;
  int pos, list_start, list_end;

  pos = IMWritePos(conn);
  IMPutC16(conn, 0);		/* length of the list. to be overwritten. */

  list_start = IMWritePos(conn);
  
  if( getIMValues(imp, data, len, offset ) < 0 )
    return -1 ;
  
  list_end = IMWritePos(conn) ;
  IMRewriteC16(conn, pos, (unsigned int)(list_end - list_start)) ;
  return 0 ;
}

int IMSetICValues( IMIC *icp, char *data, int len, int major )
{
  int r1, r2, r3 ;

  /* clear change mask */
  icp->common_attr.change_mask  = 0 ;
  icp->preedit_attr.change_mask = 0 ;
  icp->status_attr.change_mask  = 0 ;

  /* read the specified data and set attributes */
  r1 = setICValues
    ( icp, data, len, 0, 
      ( ( major == XIM_CREATE_IC ) ? OP_C : OP_S ) ) ;

  /* validate attributes */
  r2 = IMValidateICAttributes( icp, ( r1 < 0 ) ) ;

  /*
   * If the operation is CREATE_IC, input style attribute must be set.
   */
  if( major == XIM_CREATE_IC &&
      !( icp->common_attr.set_mask & ATTR_MASK_INPUT_STYLE ) &&
      r1 == 0 && r2 == 0 ){
    IMSendError
      ( icp->im->connection, IMBadSomething, icp->im->id, icp->id,
	"inputStyle resource must be set" ) ;
    r3 = -1 ;
  } else {
    r3 = 0 ;
  }
  /* if conversion is taking place... */
  if( icp->state & IC_CONVERTING ){
    changeConversionAttributes( icp ) ;
  }
  return ( ( r1 < 0 || r2 < 0 || r3 < 0 ) ? -1 : 0 ) ;
}

int IMGetICValues( IMIC *icp, char *data, int len, int offset )
{
  int nested_separator;
  int r;
  IMConnection *conn = icp->im->connection;
  int pos, list_start, list_end;

  pos = IMWritePos(conn);
  IMPutC16(conn, 0);		/* dummy. overwritten afterwards */
  IMPutC16(conn, 0);		/* unused */
  
  list_start = IMWritePos(conn);
  
  r = getICValues(icp, data, len, NEST_NONE, offset, &nested_separator);
  if (r < 0) return -1;
  
  list_end = IMWritePos(conn);
  IMRewriteC16(conn, pos, (unsigned int)(list_end - list_start));
  
  if (nested_separator) {
    /*
     * There must have been some unbalanced NestedListSeparator.
     */
    /* Ĥ̿Υ󥻥ȿ*/
    IMCancelRequest( icp->im->connection, offset ) ;
    IMSendError(icp->im->connection, IMBadSomething, icp->im->id, icp->id,
		"corrupted nested list") ;
    return -1 ;
  }
  return 0 ;
}

void IMFillDefault
( IMIC *icp, unsigned long common_mask, unsigned long preedit_mask, 
  unsigned long status_mask )
{
  if( common_mask != 0 )
    fillCommonDefault( icp, common_mask ) ;
  if( preedit_mask != 0 )
    fillPSDefault( icp, NEST_PREEDIT, preedit_mask ) ;
  if( status_mask != 0 )
    fillPSDefault( icp, NEST_STATUS, status_mask ) ;
  return ;
}

int IMValidateWindow
( Display *disp, Window win, IMWindowProfile *profilep )
{
  Window root ;
  int x, y ;
  unsigned int width, height, border, depth ;
  /*XAEHandle h;*/
  int s ;

  /*h = XAESetIgnoreErrors(dpy);*/
  s = XGetGeometry
    ( disp, win, &root, &x, &y, &width, &height, &border, &depth ) ;
  /*XAEUnset(h);*/
  
  if( profilep != NULL && s ){
    profilep->width = width ;
    profilep->height = width ;
    profilep->root = root ;
  }
  return s ;
}

int IMValidateICAttributes( IMIC *icp, int checkonly )
{
  int error_occured ;
  int r ;

  /*
   * Be careful not to send multiple error messages.
   */

  error_occured = 0 ;

  r = validateCommonAttr( icp, checkonly ) ;
  if( r < 0 )
    error_occured = 1 ;

  r = validatePSAttr
    ( icp, NEST_PREEDIT, ( checkonly && error_occured ) ) ;
  if( r < 0 )
    error_occured = 1 ;

  r = validatePSAttr
    ( icp, NEST_STATUS, ( checkonly && error_occured ) ) ;
  if( r < 0 )
    error_occured = 1 ;

  return ( ( error_occured )? -1 : 0 ) ;
}

void IMFreeICAttributes( IMIC *icp )
{
  IMPSAttributes *pattr = &icp->preedit_attr ;
  IMPSAttributes *sattr = &icp->status_attr ;
  int i ;

  if( pattr->set_mask & ATTR_MASK_FONT_SET )
    free( pattr->font_set ) ;
  if( sattr->set_mask & ATTR_MASK_FONT_SET )
    free( sattr->font_set ) ;
  for( i = 0 ; i < NUMBER_OF_CHARSET ; i ++ ){
    fontMgr_FreeFont
      ( XtDisplay( icp->im->connection->protocol_widget ),
	icp->fontset[ i ] ) ;
    icp->fontset[ i ] = NULL ;
  }
  return ;
}

unsigned long IMMakeConvAttributes
( IMIC *icp, struct ConvAttrs *attr )
{
  IMCommonAttributes *cattr;
  IMPSAttributes *pattr, *sattr;
  unsigned long cmask;		/* changed attributes */
  unsigned long mask;			/* attributes to be set */

  mask = 0L;

  cattr = &icp->common_attr ;
  pattr = &icp->preedit_attr ;
  sattr = &icp->status_attr ;

  /*
   * Check changes of common attributes.
   */
  cmask = cattr->change_mask ;
  
  /* focus window */
  if( cmask & ATTR_MASK_FOCUS ){
    attr->focus_window = cattr->focus ;
    mask |= CAFocusWindow ;
  }

  /*
   * Check changes of preedit attributes.
   */
  cmask = pattr->change_mask ;
  
  /* client area */
  if( cmask & ATTR_MASK_AREA ){
    attr->client_area.x      = pattr->area.x ;
    attr->client_area.y      = pattr->area.y ;
    attr->client_area.width  = pattr->area.width ;
    attr->client_area.height = pattr->area.height ;
    mask |= CAClientArea ;
  }
  
  /* foreground/background */
  if( cmask & ATTR_MASK_FOREGROUND ){
    attr->foreground = pattr->foreground ;
    mask |= CAColor ;
  }
  if( cmask & ATTR_MASK_BACKGROUND ){
    attr->background = pattr->background ;
    mask |= CAColor ;
  }
  
  /* colormap */
  if( cmask & ATTR_MASK_COLORMAP ){
    attr->colormap = pattr->colormap ;
    mask |= CAColormap ;
  }
#if 0
  /* 鲼ϼƤʤ顢ѤǤޤ衣()*/
  /* background pixmap */
  if( cmask & ATTR_MASK_BG_PIXMAP ){
    attr->background_pixmap = pattr->bg_pixmap ;
    mask |= CABackgroundPixmap ;
  }
#endif
  /* line spacing */
  if( cmask & ATTR_MASK_LINESPACE ){
    attr->linespacing = pattr->line_space ;
    mask |= CALineSpacing ;
  }
#if 0
  /* cursor */
  if( cmask & ATTR_MASK_CURSOR ){
    attr->cursor = pattr->cursor ;
    mask |= CACursor ;
  }
#endif
  if( cmask & ATTR_MASK_FONT_SET ){
    changeFonts( icp ) ;
    attr->fontset   = icp->fontset ;
    mask |= CAFonts ;
  }
  /* spot location */
  if( ( cmask & ATTR_MASK_SPOT_LOCATION ) &&
      icp->style == IMSTYLE_OVER_THE_SPOT ){
#if defined(DEBUG)
    fprintf( stderr, "XIM: Spot Localtion (%d, %d).\n",
	     pattr->spot_location.x, pattr->spot_location.y ) ;
#endif
    attr->spot_x = pattr->spot_location.x ;
    attr->spot_y = pattr->spot_location.y ;
    mask |= CASpotLocation ;
  }
  
  /*
   * Check changes of status attributes.
   */
  cmask = sattr->change_mask ;
  
  /* status area */
  if( cmask & ATTR_MASK_AREA ){
    attr->status_area.x      = sattr->area.x ;
    attr->status_area.y      = sattr->area.y ;
    attr->status_area.width  = sattr->area.width ;
    attr->status_area.height = sattr->area.height ;
    mask |= CAStatusArea ;
  }
  return mask ;
}

void IMMoveLocation( IMIC *icp, int x, int y )
{
  icp->preedit_attr.spot_location.x = x ;
  icp->preedit_attr.spot_location.y = y ;
  icp->preedit_attr.set_mask |= ATTR_MASK_SPOT_LOCATION ;
  icp->preedit_attr.change_mask = ATTR_MASK_SPOT_LOCATION ;

  if( icp->state & IC_CONVERTING ){
    changeConversionAttributes( icp ) ;
  }
}
