/***************************************************************************
                          qbrewcalc.cpp  -  description                              
                             -------------------                                         
    begin                : Sun Sep 26 1999                                           
    copyright            : (C) 1999 by David Johnson                         
    email                : arandir@meer.net                                     

    This software licensed under the Berkeley Software Distribution License
 ***************************************************************************/

#include "qbrewcalc.h"
#include "resource.h"
#include "store.h"

QBrewCalc::QBrewCalc(Settings *sttngs=0, QBrewDoc *doc=0)
{
	settings = sttngs;
	document = doc;
	
	og = srm = ibu = 0.0;
	utable.clear();
	
	// build hashes
	gmap[groupStyles] = gidStyles;
	gmap[groupCalcGrains] = gidCalcGrains;
	gmap[groupCalcHops] = gidCalcHops;
	gmap[groupCalcMisc] = gidCalcMisc;
	gmap[groupUtilTable] = gidUtilTable;
	
	if (!loadData()) {
		// failed to load data
		// what to do? use defaults?
		loadDefaults();
	}
	QObject::connect(document,SIGNAL(recalc()),this,SLOT(slotRecalc()));
}

QBrewCalc::~QBrewCalc() {}

///////////////////////////////////////////////////////////////////////////////
// database access

QStringList QBrewCalc::getStylesList()
{
	QStringList list;
    QMap<QString, Style>::Iterator it;
    for (it=styles.begin(); it != styles.end(); ++it) {
    	list += it.key();
    }
    list.sort();
    return list;
}

Style* QBrewCalc::getStyle(const QString &sname)
{
	return &styles[sname];
}

QStringList QBrewCalc::getGrainsList()
{
	QStringList list;
    GrainList::Iterator it;
    for (it=grains.begin(); it != grains.end(); ++it) {
    	list += (*it).getName();
    }
    list.sort();
    return list;
}

Grain* QBrewCalc::getGrain(const QString &gname)
{
    GrainList::Iterator it;
    for (it=grains.begin(); it != grains.end(); ++it) {
    	if (gname == (*it).getName())
    		return &(*it);
    }
	return NULL;
}

QStringList QBrewCalc::getHopsList()
{
	QStringList list;
    HopsList::Iterator it;
    for (it=hops.begin(); it != hops.end(); ++it) {
    	list += (*it).getName();
    }
    list.sort();
    return list;
}

QStringList QBrewCalc::getFormsList()
{
	QStringList list;
    HopsList::Iterator it;
    list += "pellet";
    list += "plug";
    list += "whole";
    for (it=hops.begin(); it != hops.end(); ++it) {
    	if (list.contains((*it).getForm()) == 0)
    		list += (*it).getForm();
    }
    list.sort();
    return list;
}

Hops* QBrewCalc::getHop(const QString &hname)
{
    HopsList::Iterator it;
    for (it=hops.begin(); it != hops.end(); ++it) {
    	if (hname == (*it).getName())
    		return &(*it);
    }
	return NULL;
}

QStringList QBrewCalc::getMiscList()
{
	QStringList list;
    MiscIngredientList::Iterator it;
    for (it=misc.begin(); it != misc.end(); ++it) {
    	list += (*it).getName();
    }
    list.sort();
    return list;
}

MiscIngredient* QBrewCalc::getMisc(const QString &mname)
{
    MiscIngredientList::Iterator it;
    for (it=misc.begin(); it != misc.end(); ++it) {
    	if (mname == (*it).getName())
    		return &(*it);
    }
	return NULL;
}

///////////////////////////////////////////////////////////////////////////////
// slots

void QBrewCalc::slotRecalc()
{
	// we've received a signal to recalculate stuff
	og = calcOG();
	ibu = calcIBU();
	srm = calcSRM();
	emit calcDone();
}

///////////////////////////////////////////////////////////////////////////////
// calculations

double QBrewCalc::calcOG()
{
	GrainList *list = document->getGrainList();
    GrainList::Iterator it;
    double yield = 0.0, est = 0.0;
    for (it=list->begin(); it != list->end(); ++it) {
    	yield = (*it).getYield();
		switch ((*it).getUse()) {
			case GRAIN_MASHED:
				// adjust for mash efficiency
				yield *= settings->retrieveSetting(IDS_SETTINGS_EFFICIENCY).toDouble() / 100.0;
				break;
			case GRAIN_STEEPED:
				// steeped grains don't yield nearly as much as mashed grains
				yield *= 0.33; // TODO: should this be a user definable constant?
				break;
			case GRAIN_EXTRACT:
				// no modifications necessary
				break;
			default:
				break;
		}
    	est += yield;
    }
	est /= settings->retrieveSetting(IDS_SETTINGS_BATCH).toDouble() / 100.0;
	return est;
}

double QBrewCalc::calcIBU()
{
	HopsList *list = document->getHopsList();
    HopsList::Iterator it;
    double bitterness = 0.0;
    for (it=list->begin(); it != list->end(); ++it) {
    	bitterness += (*it).getHBU() * utilization((*it).getTime());
    	// we should also correct for hop form
    }
	bitterness /= settings->retrieveSetting(IDS_SETTINGS_BATCH).toDouble() / 100.0;
	// correct for boil gravity
	if (og > 50.0) {
		bitterness /= 1.0 + (0.005 * (og - 50.0));
	}
	return bitterness;
}

double QBrewCalc::utilization(const unsigned &time)
{
	QValueList<uentry>::Iterator it;
	for (it=utable.begin(); it != utable.end(); ++it) {
    	if (time >= (*it).time)
    		return (double((*it).utilization));
    }
    return 0.0;
}

double QBrewCalc::calcSRM()
{
	GrainList *list = document->getGrainList();
    GrainList::Iterator it;
    double est = 0.0;
    for (it=list->begin(); it != list->end(); ++it) {
    	est += (*it).getHCU();
    }
	est /= settings->retrieveSetting(IDS_SETTINGS_BATCH).toDouble() / 100.0;
	if (est > 8.0) {
		est *= 0.2;
		est += 8.4;
	}
	return est;
}

double QBrewCalc::getOG() { return og; }

double QBrewCalc::getFGEstimate() { return (og * 0.25); }

double QBrewCalc::getABV() { return (og * 0.09675); }

double QBrewCalc::getABW() { return (og * 0.07595); }

double QBrewCalc::getIBU() { return ibu; }

double QBrewCalc::getSRM() { return ( srm ); }

///////////////////////////////////////////////////////////////////////////////
// load databases

bool QBrewCalc::loadData()
{
	// TODO: should also load in user data to prevent corrupting calcdb
	
	unsigned ID;
	
	QString fname = settings->retrieveSetting(IDS_SETTINGS_QBREWDIR) + IDS_CALC_FILE;
    // open file for reading
	Store *rfile = new Store(IO_ReadOnly, fname, CALC_FORMAT, CALC_VERSION);
	if (!rfile->good()) {
		warning("QBrewCalc: Error opening " + fname + "\n");
		delete rfile;
		return false;
	} else {
		// file opened without error
		if (rfile->getVersion() < CALC_PREVIOUS) {
			warning("QBrewCalc: Unsupported data version " +  rfile->getVersion() + "\n");
			delete rfile;
			return false;
		}
		// read in file line by line
		while (rfile->getLine()) {
			ID = gmap[rfile->getGroup()];
			if (rfile->getGroup() != rfile->getName()) {
				switch (ID) {
					case gidStyles:
						processStyle(rfile->getName(), rfile->getValue());
						break;
					case gidCalcGrains:
						processGrain(rfile->getName(), rfile->getValue());
						break;
					case gidCalcHops:
						processHops(rfile->getName(), rfile->getValue());
						break;
					case gidCalcMisc:
						processMisc(rfile->getName(), rfile->getValue());
						break;
					case gidUtilTable:
						processUtilEntry(rfile->getName(), rfile->getValue());
						break;
					default:
						warning("QBrewCalc: " + rfile->getGroup() + " is not a valid group\n");
						break;
				}
			}
		}		
	}
	delete rfile;
	return true;
}

void QBrewCalc::loadDefaults()
{
	processStyle("Ale", "20, 120, 33, 0, 70, 1, 40");
	processGrain("Grain", "100, 1025, 10, FALSE");
	processHops("Hops", "100, pellet, 100, 60");
	processMisc("Misc", "100, ingredient");
	processUtilEntry("0", "20");
}

void QBrewCalc::processStyle(const QString &name, const QString &value)
{
	Style newstyle;
	newstyle.serializeIn(value);
	styles.replace(name, newstyle);
}

void QBrewCalc::processGrain(const QString &name, const QString &value)
{
	Grain newgrain;
	newgrain.serializeIn(name, value);
	grains.append(newgrain);	
}

void QBrewCalc::processHops(const QString &name, const QString &value)
{
	Hops newhop;
	newhop.serializeIn(name, value);
	hops.append(newhop);	
}

void QBrewCalc::processMisc(const QString &name, const QString &value)
{
	MiscIngredient newmisc;
	newmisc.serializeIn(name, value);
	misc.append(newmisc);	
}

void QBrewCalc::processUtilEntry(const QString &name, const QString &value)
{
	uentry entry;
	entry.time = name.toUInt();
	entry.utilization = value.toUInt(); // TODO: assumes that there's nothing else on the line
	// keep the list sorted from highest time to lowest
    QValueList<uentry>::Iterator it;
    if (utable.isEmpty()) {
    	utable.append(entry);
    } else {
    	for (it=utable.begin(); it != utable.end(); ++it) {
    		if ((*it).time < entry.time)
    			break;
    	}
    	utable.insert(it, entry);
    }
}





















