/* this file is a part of Ami software, (C) Hwang chi-deok 1999 */

#include "config.h"

#include <stdio.h>
#include <ctype.h>
#include <sys/stat.h>
#include <unistd.h>
#include "ami.h"
#include "conf.h"
#include "keyparse.h"

#ifndef ENGLISH_KEYBOARD
#define ENGLISH_KEYBOARD "1"
#endif
#ifndef HANGUL_KEYBOARD
#define HANGUL_KEYBOARD "3"
#endif

static char * config_readline(FILE *fp);
static gboolean config_add_item(guchar *desc, guchar *key, guchar *val_string);
static int save_default_config_file(void);
static ConfigItem *ami_config = NULL;


static ConfigDesc ami_config_type[] = 
{
    {"ѱ ", INT, &ami_keyboard},
    {" ", INT, &ami_ekeyboard},
    {"Է ", INT, &ami_editing_mode},
    {"⺻ ۲", STRING, &default_fontname},
    {"ѿ ȯŰ", KEYSYM, &trigger_keys},
    {" ȯŰ", KEYSYM, &hanja_keys},
    {"Ư ԷŰ", KEYSYM, &special_char_keys},
    {"ѱۻ ", STRING, &hangul_mode_label},
    {" ", STRING, &english_mode_label},
    {" ѿ", BOOLEAN, &unique_han},
    {"flush key", KEYSYM, &flush_keys},
    {"last line scroll", BOOLEAN, &ami_line_wraping_mode},
    {"escape hangul toggle", BOOLEAN, &esc_han_toggle},
    {"â ", BOOLEAN, &support_status},
    {" ġȯ ", STRING, &ami_hanja_subst_format},
    {" ġȯ ", INT, &ami_hanja_def_subst_mode},
    {" ", BOOLEAN, &ami_use_underline},
    {"ڻ", STRING, &hanja_dic_filename},
    {NULL, 0, NULL}
};

gboolean
ami_config_init(void)
{
    FILE *fp;
    gboolean ret;
    int i;
    char *conf_file = g_strconcat(g_get_home_dir(), "/.ami/ami.conf", NULL);
    fp = fopen(conf_file, "r");
    if (fp == NULL) {
	if (!save_default_config_file())
	    g_error("%s  ϴ", conf_file);
    } else 
	fclose(fp);
    ret = ami_config_load_file(conf_file, ami_config_type);
    on_keys = g_new(XIMTriggerKeys, 1);
    if (trigger_keys == NULL) {
	g_error("ѿ ȯŰ Ǿ ʽϴ.");
    }
    for(i=0;trigger_keys[i].keysym;i++);
    on_keys->count_keys = i;
    on_keys->keylist = trigger_keys;
    g_free(conf_file);
    ami_hanja_subst_mode = ami_hanja_def_subst_mode;
    if (!hangul_mode_label) 
       hangul_mode_label = g_strdup(" [ѱ] ");
    if (!english_mode_label) 
       english_mode_label = g_strdup(" [] ");
    if (!ami_hanja_subst_format) 
       ami_hanja_subst_format = g_strdup(":():()");
    if (!hanja_dic_filename) hanja_dic_filename = g_strdup(HANJA_DIC);
    return ret;
}

static guchar *
line_skip_space(guchar *s)
{
    while(isspace(*s)) s++;
    return s;
}

static guchar *
line_rskip_space(guchar *s)
{
    while(isspace(*s)) s--;
    return s;
}

static gboolean
line_is_empty(guchar *s)
{
    while(*s) {
	if (!isspace(*s)) return FALSE;
	s++;
    }
    return TRUE;
}

static int
save_default_config_file(void)
{
    char *conf_dir = g_strdup_printf("%s/%s", g_get_home_dir(), ".ami");
    char *conf;
    extern char *default_config;
    struct stat statbuf;
    FILE *fp;
    if (stat(conf_dir, &statbuf) != 0) {
	mkdir(conf_dir,0750);
    }
    conf = g_strdup_printf("%s/%s", conf_dir, "ami.conf");
    g_free(conf_dir);
    fp = fopen(conf, "w");
    g_free(conf);
    if (fp == NULL) return FALSE;
    fprintf(fp, "%s", default_config);
    fclose(fp);
    return TRUE;
}

static char *
config_readline(FILE *fp)
{
    char buf[1024];
    if (fgets(buf, sizeof(buf), fp)) {
	return g_strdup(buf);
    }
    return NULL;
}

static ConfigItem *
config_find_config_item(guchar *key)
{
    ConfigItem *config = ami_config;
    while (config->key) {
	if (strcmp(config->key, key) == 0) return config;
	config++;
    }
    return NULL;
}

static gboolean
config_add_item(guchar *desc, guchar *key, guchar *val_string)
{
    ConfigItem *config_item;
    config_item = config_find_config_item(key);
    if (config_item == NULL) return FALSE;
    config_item->desc = g_strdup(desc);
    switch(config_item->type) {
	case INT:
	    *config_item->val.integer = atoi(val_string);
	    break;
	case BOOLEAN:
	    if (*val_string == 't' 
	    	|| strcmp(val_string, "on") == 0 
		|| *val_string == '0') *config_item->val.boolean = TRUE;
	    else if(*val_string == 'f'
	    	|| strcmp(val_string, "off") == 0
		|| *val_string == '1') *config_item->val.boolean = FALSE;
	    else {
		g_warning("%s: valid value is true or false", key);
	    }
	    break;
	case KEYSYM:
	    make_trigger_keys(config_item->val.keys, val_string);
	    break;
	case STRING:
	    if (*config_item->val.string) g_free(*config_item->val.string);
	    if (val_string[0] == '"' && 
	        val_string[strlen(val_string)-1] == '"') {
		val_string[strlen(val_string)-1] = 0;
		val_string++;
	    }
	    *config_item->val.string = g_strdup(val_string);
	    break;
    }
    return TRUE;
}

gboolean
ami_config_load_file(char *file, ConfigDesc *ami_config_type)
{
    int config_size;
    int i;
    GSList *list, *comment = NULL;
    FILE *fp;
    char *line;

    for(i=0;ami_config_type[i].key;i++);
    config_size = i;
    ami_config = g_new0(ConfigItem, config_size + 1);
    for(i=0;i<config_size;i++) {
	ami_config[i].key = ami_config_type[i].key;
	ami_config[i].type = ami_config_type[i].type;
	switch(ami_config[i].type) {
	    case STRING: 
		ami_config[i].val.string = ami_config_type[i].val_pointer; 
		break;
	    case INT: 
		ami_config[i].val.integer = ami_config_type[i].val_pointer; 
		break;
	    case KEYSYM: 
		ami_config[i].val.keys = ami_config_type[i].val_pointer; 
		break;
	    case BOOLEAN: 
		ami_config[i].val.boolean = ami_config_type[i].val_pointer; 
		break;
	}
    }
    fp = fopen(file, "r");
    if (fp == NULL) return FALSE;
    while ((line = config_readline(fp))) {
	if (line[0] == '#' || line_is_empty(line)) {
	    comment = g_slist_append(comment, line);
	} else {
	    char *sep = strchr(line, '=');
	    guchar * start, *end, *key, *val;
	    GString *desc_string;
	    if (sep == NULL) {
		g_warning("invalid config line %s", line);
		continue;
	    }
	    *sep = 0;
	    start = line_skip_space(line);
	    end = line_rskip_space(sep - 1); 
	    if (isspace(*end)) *end = 0;
	    else *(end+1) = 0;
	    key = start;
	    sep += 1;
	    start = line_skip_space(sep);
	    end = line_rskip_space(sep + strlen(sep) - 1);
	    if (isspace(*end)) *end = 0;
	    else *(end+1) = 0;
	    val = start;
	    desc_string = g_string_new("");
	    list = comment;
	    while(list) {
		g_string_append(desc_string , list->data);
		g_free(list->data);
		list = list->next;
	    }
	    g_slist_free(comment);
	    comment = NULL;
	    if (!config_add_item(desc_string->str, key, val)) {
		g_warning("%s is not valid key", key);
	    };
	    g_string_free(desc_string, TRUE);
	    g_free(line);
	}
    }
    return TRUE;
}

gboolean
ami_config_save_file(char *file)
{
    FILE *fp;
    ConfigItem *config;
    char *key_desc;
    if (file) {
	fp = fopen(file, "w");
    } else {
	file =g_strconcat(g_get_home_dir(), "/.ami/ami.conf", NULL);
	fp = fopen(file, "w");
	g_free(file);
    }
    if (fp == NULL) return FALSE;
    config = ami_config;
    while(config->key) {
	if (config->desc) fprintf (fp, "%s", config->desc);
	else fprintf (fp, "# %s\n", config->key);
	switch(config->type) {
	    case INT:
		fprintf (fp, "%s=%d\n", config->key, *config->val.integer);
		break;
	    case BOOLEAN:
		if(*config->val.boolean) 
		    fprintf (fp, "%s=true\n", config->key);
		else
		    fprintf (fp, "%s=false\n", config->key);
		break;
	    case STRING:
		fprintf (fp, "%s=\"%s\"\n", config->key, *config->val.string);
		break;
	    case KEYSYM:
		key_desc = trigger_keys_to_str(*config->val.keys);
		fprintf (fp, "%s=%s\n", config->key, key_desc);
		g_free(key_desc);
		break;
	}
	config++;
    }
    fclose(fp);
    return TRUE;
}


char *default_config = 
"
# ƹ 
# :  Ű  ٲʽÿ.  \"ѱ \"̶ Ǿ
#   ؼ Ȯ ̸̾մϴ.

# # ѱ Ű 
# 2 = ι, 3 =  390, 4 =  

ѱ  = " HANGUL_KEYBOARD "

# #  Ű 
# 1 = qwerty, 2 = Dvorak 

  = " ENGLISH_KEYBOARD "

# Է½  Ƽ   ݴϴ.
# 1=ܾ ϸ Է  ȭǥŰ ̿Ͽ ̵ մϴ.
# 2=ܾ ϸ ȭǥŰ ̿ ̵ α׷  
#   ʿ ó ϵ մϴ. ϴ Դϴ.
# 3=ڴ  մϴ.

Է  = 1

#  α׷  ѿ¸  Ұ 
# false: ѿ° â ˴ϴ. XIM ǥؿ  յǴ Դϴ.
#  httͰ α׷ ޸ program ִ  ʰ
#   â ؼ gtk+Ͱ α׷   ϵ
#  մϴ.
# true:  α׷  ѿ¸ Ե˴ϴ. â ʽϴ.
#   ٸ     մϴ.

 ѿ = false


# ѿȯ  Ǵ Ű ݴϴ

ѿ ȯŰ = Shift<Key>space, <Key>Hangul

#  Է Ű

 ȯŰ = <Key>F9, <Key>Hangul_Hanja

# Ư  ԷŰ 

Ư ԷŰ = <Key>F3

#  Է Էâ  α׷   Ű

flush key = Ctrl<Key>Return

# â ȭ  Ʒκп Ÿ (Ǵ gtk entryͰ ¥ 
# Էâ) Էâ  ʱ ѱԷ  Ǿ   Է
# ȭ  κп ̷ ƴϸ  Էµǰִ  
# ũ ų Ÿ մϴ.
# true̸ ׳ ũ ŵϴ.

last line scroll = true

# ⺻ ۲ 
# ƹ̿ ϴ α׷ ۲ ʰų ߸ ̸ 쿡
#  ۲ ֽø ˴ϴ.  ۲,ѱ۱۲

⺻ ۲ = *,*

# ѱ  ϶ â ǥõǴ ڿ ݴϴ.

ѱۻ  = \" [ѱ] \"

#  Է ϶ â ǥõǴ ڿ ݴϴ.

  = \" [] \"

# ѱԷ ¿ Escape  · ڵ 
# ݴϴ. true/false

escape hangul toggle = true

# â  ݴϴ.
# â Ʒʿ   â 彺 е  ɼ 
# false Ͻʽÿ. 
#  ѱۻ¸ Ͻø  ɼǰ  â ʽϴ.

â  = true

# ڸ ġȯϴ ¸ ݴϴ.
#  Ǿִ κ   ڷ ġȯǰ \"\"
#  κ  ڿ شϴ ѱ ־Ե˴ϴ.
# µ  \":\" մϴ.

 ġȯ  = :():()

# ڸ ġȯ¿  °  ⺻ Ұ
# մϴ.
# 1 

 ġȯ  = 1

#  Էâ  ִ ѱ ǥ  Ʒʿ   
# ƴϸ   â  ٸ  

  = true

#   ġ 

ڻ = " HANJA_DIC "
";
