/*  Screem:  main.c,
 *  This contains the main() function for Screem, along with handling the
 *  Gnome session management features
 * 
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gmodule.h>
#include <gnome.h>
#include <unistd.h>

#include <glade/glade.h>
#include <libgnorba/gnorba.h>
#include <sys/stat.h>

#ifdef HAVE_GUILE
#include <libguile.h>
#include <guile/gh.h>
#endif

#include "editor.h"
#include "interface.h"
#include "site.h"
#include "siteUI.h"
#include "page.h"
#include "pageUI.h"
#include "plugin.h"
#include "preferences.h"
#include "preview.h"
#include "siteTreeUI.h"

GtkWidget *app;
GnomeClient *client = NULL;
Preferences *cfg;

Site *current_site = NULL;
Page *current_page = NULL;

GList *loaded_sites = NULL;
GList *loaded_pages = NULL;

/* CORBA stuff */
CORBA_ORB my_orb;
CORBA_Environment ev;
CORBA_Object ns;

static gint no_splash;

/* passed parameters stuff */
static poptContext ctx;
static const struct poptOption options[] = {
	{ "no-splash", '\0', POPT_ARG_NONE, &no_splash, 1,
	  N_( "Disables splash screen on start up" ), NULL },
        { NULL, '\0', 0, NULL, 0 }
};

static char* start_file = NULL;   /* we only allow 1 file, the first one
                                     in the above array */

static GtkWidget *text;
static GtkWidget *splash;

#define update_splash(message) 	gtk_text_insert(GTK_TEXT( text ), 0, 0, 0, message, -1);\
while( gtk_events_pending() )\
gtk_main_iteration();

static void main_real( int argc, char *argv[] );
static gint
save_yourself( GnomeClient *client, gint phase, GnomeSaveStyle save_style,
               gint is_shutdown, GnomeInteractStyle interact_style,
               gint is_fast, gpointer client_data );
static void parse_args( int argc, char *argv[] );


int main( int argc, char *argv[] )
{
	/* initialise i18n stuff */
        bindtextdomain( PACKAGE, GNOMELOCALEDIR );
        textdomain( PACKAGE );
	
	/* initialise gnome, gtk corba etc */
        CORBA_exception_init( &ev );
	my_orb = gnome_CORBA_init_with_popt_table( PACKAGE, VERSION, &argc,
						   argv, options, 0, &ctx,
						   0, &ev );
        if( ev._major != CORBA_NO_EXCEPTION )
                exit( 1 ); /* error initializing ORB */

	/* init libglade */
	glade_gnome_init();
	
        ns = gnome_name_service_get();
	
        /* parse arguments */
        parse_args( argc, argv );

	/* show splash */
	splash = create_splash();
	text = gtk_object_get_data( GTK_OBJECT( splash ), "text1" );
	gtk_window_position( GTK_WINDOW( splash ), GTK_WIN_POS_CENTER );

	if( ! no_splash )
		gtk_widget_show_all( splash );
	else
		gtk_widget_realize( text );

#ifdef HAVE_GUILE
	/* init guile */
	update_splash( _("Initialising Guile\n") );

	gh_enter(argc, argv, main_real);
#else
	main_real( argc, argv );
#endif

	return 0;
}

static void main_real( int argc, char *argv[] )
{
	struct stat s;
	const gchar *mime_type;
	gchar cwd[ 16384 ];

	update_splash( _("Creating Interface\n") );
	
	/* create interface */
	app = create_app( FALSE, PACKAGE, "Screem:" );
	
	/* load modules */
	if( g_module_supported() ) {
		update_splash( _("Loading Modules...\n") );
		scan_plugins( text );
		update_splash( _("Modules loaded\n") );
	} else {
		update_splash( _("Modules no supported\n") );
	}
	gtk_widget_show_all( app );

	getcwd( cwd, 16385 );
	screem_browse_list_show_dir( cwd );

	resource_list_display();

	build_recent_sites_menu();
	build_recent_pages_menu();

	/* destroy splash */
	gtk_widget_destroy( splash );

	screem_preview_display( NULL );

	if( start_file ) {
		/* is it a project file or an html file? */
		mime_type = gnome_mime_type( start_file );
		if( ! strcmp( mime_type, "text/html" ) ||
		    ! strcmp( mime_type, "application/x-asp" ) ||
		    ! strcmp( mime_type, "application/x-php" ) ) {
			screem_page_open_with_filename( start_file );
		} else if( ! strcmp( mime_type, "application/x-screem" ) ) {
			/* does it even exist? */
			if( ! stat( start_file, &s ) ) {
				screem_site_open_with_filename( start_file );
			}
		}
	}

	gtk_main();

	save_preferences();
}

static void parse_args( int argc, char *argv[] )
{
	char** start_files = NULL;

        /* passed arguements */
        start_files = (char**)poptGetArgs( ctx );

        if( start_files )
                start_file = g_strdup( start_files[ 0 ] );
        poptFreeContext( ctx );

        /* Session Managment stuff */
        client = gnome_master_client ();
        gtk_signal_connect (GTK_OBJECT (client), "save_yourself",
                            GTK_SIGNAL_FUNC (save_yourself), NULL);
        gtk_signal_connect (GTK_OBJECT (client), "die",
                            GTK_SIGNAL_FUNC (gtk_main_quit), NULL);

        /* load the preferences */
        load_preferences();

        return;
}

static gint
save_yourself( GnomeClient *client, gint phase, GnomeSaveStyle save_style,
               gint is_shutdown, GnomeInteractStyle interact_style,
               gint is_fast, gpointer client_data )
{
        gchar* args[4] = { "rm", "-r", NULL, NULL };

	save_preferences();

	args[2] = gnome_config_get_real_path
                    (gnome_client_get_config_prefix (client));
        gnome_client_set_discard_command (client, 3, args);

        return TRUE;
}
