/*
 * html.c: code concerning the dump as an HTML base.
 *
 * Copyright (c) 1997 Daniel Veillard <veillard@apocalypse.org>
 * See COPYING for the status of this software.
 *
 * $Id: html.c,v 1.43 1998/02/26 21:02:52 veillard Exp $
 */

#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <time.h>
#include <errno.h>
#include <ctype.h>

#include "rpm2html.h"
#include "rpmdata.h"
#include "html.h"
#include "language.h"

/*
 * Global variables concerning the dump environment
 */

static char buf[500];

/*
 * The entities list for the extended ASCII charset.
 */
static char *entities[256] = {
NULL,		/* 000 */
NULL,		/* 001 */
NULL,		/* 002 */
NULL,		/* 003 */
NULL,		/* 004 */
NULL,		/* 005 */
NULL,		/* 006 */
NULL,		/* 007 */
NULL,		/* 008 */
NULL,		/* 009 */
NULL,		/* 010 */
NULL,		/* 011 */
NULL,		/* 012 */
NULL,		/* 013 */
NULL,		/* 014 */
NULL,		/* 015 */
NULL,		/* 016 */
NULL,		/* 017 */
NULL,		/* 018 */
NULL,		/* 019 */
NULL,		/* 020 */
NULL,		/* 021 */
NULL,		/* 022 */
NULL,		/* 023 */
NULL,		/* 024 */
NULL,		/* 025 */
NULL,		/* 026 */
NULL,		/* 027 */
NULL,		/* 028 */
NULL,		/* 029 */
NULL,		/* 030 */
NULL,		/* 031 */
NULL,		/* 032 */
NULL,		/* 033 */
"&quot;",	/* 034 */
NULL,		/* 035 */
NULL,		/* 036 */
NULL,		/* 037 */
"&amp;",	/* 038 */
NULL,		/* 039 */
NULL,		/* 040 */
NULL,		/* 041 */
NULL,		/* 042 */
NULL,		/* 043 */
NULL,		/* 044 */
NULL,		/* 045 */
NULL,		/* 046 */
NULL,		/* 047 */
NULL,		/* 048 */
NULL,		/* 049 */
NULL,		/* 050 */
NULL,		/* 051 */
NULL,		/* 052 */
NULL,		/* 053 */
NULL,		/* 054 */
NULL,		/* 055 */
NULL,		/* 056 */
NULL,		/* 057 */
NULL,		/* 058 */
NULL,		/* 059 */
"&lt;",		/* 060 */
NULL,		/* 061 */
"&gt;",		/* 062 */
NULL,		/* 063 */
NULL,		/* 064 */
NULL,		/* 065 */
NULL,		/* 066 */
NULL,		/* 067 */
NULL,		/* 068 */
NULL,		/* 069 */
NULL,		/* 070 */
NULL,		/* 071 */
NULL,		/* 072 */
NULL,		/* 073 */
NULL,		/* 074 */
NULL,		/* 075 */
NULL,		/* 076 */
NULL,		/* 077 */
NULL,		/* 078 */
NULL,		/* 079 */
NULL,		/* 080 */
NULL,		/* 081 */
NULL,		/* 082 */
NULL,		/* 083 */
NULL,		/* 084 */
NULL,		/* 085 */
NULL,		/* 086 */
NULL,		/* 087 */
NULL,		/* 088 */
NULL,		/* 089 */
NULL,		/* 090 */
NULL,		/* 091 */
NULL,		/* 092 */
NULL,		/* 093 */
NULL,		/* 094 */
NULL,		/* 095 */
NULL,		/* 096 */
NULL,		/* 097 */
NULL,		/* 098 */
NULL,		/* 099 */
NULL,		/* 100 */
NULL,		/* 101 */
NULL,		/* 102 */
NULL,		/* 103 */
NULL,		/* 104 */
NULL,		/* 105 */
NULL,		/* 106 */
NULL,		/* 107 */
NULL,		/* 108 */
NULL,		/* 109 */
NULL,		/* 110 */
NULL,		/* 111 */
NULL,		/* 112 */
NULL,		/* 113 */
NULL,		/* 114 */
NULL,		/* 115 */
NULL,		/* 116 */
NULL,		/* 117 */
NULL,		/* 118 */
NULL,		/* 119 */
NULL,		/* 120 */
NULL,		/* 121 */
NULL,		/* 122 */
NULL,		/* 123 */
NULL,		/* 124 */
NULL,		/* 125 */
NULL,		/* 126 */
NULL,		/* 127 */
NULL,		/* 128 */
NULL,		/* 129 */
NULL,		/* 130 */
NULL,		/* 131 */
NULL,		/* 132 */
NULL,		/* 133 */
NULL,		/* 134 */
NULL,		/* 135 */
NULL,		/* 136 */
NULL,		/* 137 */
NULL,		/* 138 */
NULL,		/* 139 */
NULL,		/* 140 */
NULL,		/* 141 */
NULL,		/* 142 */
NULL,		/* 143 */
NULL,		/* 144 */
NULL,		/* 145 */
NULL,		/* 146 */
NULL,		/* 147 */
NULL,		/* 148 */
NULL,		/* 149 */
NULL,		/* 150 */
NULL,		/* 151 */
NULL,		/* 152 */
NULL,		/* 153 */
NULL,		/* 154 */
NULL,		/* 155 */
NULL,		/* 156 */
NULL,		/* 157 */
NULL,		/* 158 */
NULL,		/* 159 */
"&nbsp;",	/* 160 */
"&iexcl;",	/* 161 */
"&cent;",	/* 162 */
"&pound;",	/* 163 */
"&curren;",	/* 164 */
"&yen;",	/* 165 */
"&brvbar;",	/* 166 */
"&sect;",	/* 167 */
"&uml;",	/* 168 */
"&copy;",	/* 169 */
"&ordf;",	/* 170 */
"&laquo;",	/* 171 */
"&not;",	/* 172 */
"&shy;",	/* 173 */
"&reg;",	/* 174 */
"&macr;",	/* 175 */
"&deg;",	/* 176 */
"&plusmn;",	/* 177 */
"&sup;",	/* 178 */
"&sup;",	/* 179 */
"&acute;",	/* 180 */
"&micro;",	/* 181 */
"&para;",	/* 182 */
"&middot;",	/* 183 */
"&cedil;",	/* 184 */
"&sup;",	/* 185 */
"&ordm;",	/* 186 */
"&raquo;",	/* 187 */
"&frac;",	/* 188 */
"&frac;",	/* 189 */
"&frac;",	/* 190 */
"&iquest;",	/* 191 */
"&Agrave;",	/* 192 */
"&Aacute;",	/* 193 */
"&Acirc;",	/* 194 */
"&Atilde;",	/* 195 */
"&Auml;",	/* 196 */
"&Aring;",	/* 197 */
"&AElig;",	/* 198 */
"&Ccedil;",	/* 199 */
"&Egrave;",	/* 200 */
"&Eacute;",	/* 201 */
"&Ecirc;",	/* 202 */
"&Euml;",	/* 203 */
"&Igrave;",	/* 204 */
"&Iacute;",	/* 205 */
"&Icirc;",	/* 206 */
"&Iuml;",	/* 207 */
"&ETH;",	/* 208 */
"&Ntilde;",	/* 209 */
"&Ograve;",	/* 210 */
"&Oacute;",	/* 211 */
"&Ocirc;",	/* 212 */
"&Otilde;",	/* 213 */
"&Ouml;",	/* 214 */
"&times;",	/* 215 */
"&Oslash;",	/* 216 */
"&Ugrave;",	/* 217 */
"&Uacute;",	/* 218 */
"&Ucirc;",	/* 219 */
"&Uuml;",	/* 220 */
"&Yacute;",	/* 221 */
"&THORN;",	/* 222 */
"&szlig;",	/* 223 */
"&agrave;",	/* 224 */
"&aacute;",	/* 225 */
"&acirc;",	/* 226 */
"&atilde;",	/* 227 */
"&auml;",	/* 228 */
"&aring;",	/* 229 */
"&aelig;",	/* 230 */
"&ccedil;",	/* 231 */
"&egrave;",	/* 232 */
"&eacute;",	/* 233 */
"&ecirc;",	/* 234 */
"&euml;",	/* 235 */
"&igrave;",	/* 236 */
"&iacute;",	/* 237 */
"&icirc;",	/* 238 */
"&iuml;",	/* 239 */
"&eth;",	/* 240 */
"&ntilde;",	/* 241 */
"&ograve;",	/* 242 */
"&oacute;",	/* 243 */
"&ocirc;",	/* 244 */
"&otilde;",	/* 245 */
"&ouml;",	/* 246 */
"&divide;",	/* 247 */
"&oslash;",	/* 248 */
"&ugrave;",	/* 249 */
"&uacute;",	/* 250 */
"&ucirc;",	/* 251 */
"&uuml;",	/* 252 */
"&yacute;",	/* 253 */
"&thorn;",	/* 254 */
"&yuml;"	/* 255 */
};

/*
 * Converting extended ASCII charset to valid HTML text.
 */

char *strdupHTML(const char *str) {
    static unsigned char buffer[5000];
    unsigned char *cur, *end;
    unsigned char c;

    cur = &buffer[0];
    end = &buffer[sizeof(buffer) - 20];

    while ((cur < end) && (*str != '\0')) {
        c = (unsigned char) *(str++);
	if (entities[(unsigned int) c] == NULL) {
	    *(cur++) = c;
	} else {
	    strcpy(cur, entities[(unsigned int) c]);
	    cur += strlen(entities[(unsigned int) c]);
	}
    }
    *cur = '\0';
    return(strdup(buffer));
}

/*
 * createDirectory : create a directory.
 */

void createDirectory(const char *dirname) {
    if (mkdir(dirname, 0777) != 0) {
        switch (errno) {
#ifdef EEXIST
            case EEXIST:
		if (verbose)
		    fprintf(stderr, "Directory \"%s\" already exists\n",
		            dirname);
		return;
#endif
	    case ENOENT: {
	        char *father = strdup(dirname);
		char *cur = &father[strlen(father)];

		while (cur > father) {
		    if (*cur == '/') {
		        *cur = '\0';
			break;
	            }
		    cur--;
		}
		if (cur > father) {
		    createDirectory(father);
		    if (mkdir(dirname, 0777) != 0) {
			fprintf(stderr, "createDirectory \"%s\" failed\n",
			        dirname);
			perror("mkdir failed:");
			return;
		    }
		}
		free(father);
		
	        break;
	    }
            default:
	        fprintf(stderr, "createDirectory \"%s\" failed\n", dirname);
	        perror("mkdir failed:");
		return;
	}
    } else if (verbose)
        fprintf(stderr, "Created directory \"%s\"\n", dirname);
}

/*
 * checkDate : check whether the last modification time of a file
 *             is anterior to a given time
 */

int checkDate(const char *filename, time_t stamp) {
    struct stat buf;

    if (force) return(0);
    if ((stat(filename, &buf)) != 0) {
        return(0);
    }
    return(buf.st_mtime > stamp);
}

/*
 * Transformation function from rpm to filename.
 */

char *rpmName(rpmDataPtr cur) {
    static char rbuf[500];

    if (cur->arch != NULL)
	sprintf(rbuf, "%s-%s-%s.%s", 
	    cur->name, cur->version, cur->release, cur->arch);
    else
	sprintf(rbuf, "%s-%s-%s", 
	    cur->name, cur->version, cur->release);
    return(rbuf);
}

/*
 * Transformation function from rpm to Software name.
 */

char *rpmSoftwareName(rpmDataPtr cur) {
    static char rbuf[500];

    sprintf(rbuf, "%s-%s-%s", 
	cur->name, cur->version, cur->release);
    return(rbuf);
}

/*
 * Transformation function from group to filename.
 */

char *groupName(char *group) {
    static char gbuf[500];
    char *cur = gbuf;

    while (*group != '\0') {
        if ((*group == '/') || (*group == ' ')) {
	    *cur++ = '_';
	    group ++;
	} else *cur++ = *group++;
    }
    strcpy(cur, localizedStrings[LANG_HTML_SUFFIX]);
    return(gbuf);
}

/*
 * Transformation function from group to filename.
 */

char *distribName(char *distrib) {
    static char dbuf[500];
    char *cur = dbuf;

    while (*distrib != '\0') {
        if ((*distrib == '/') || (*distrib == ' ')) {
	    *cur++ = '_';
	    distrib ++;
	} else *cur++ = *distrib++;
    }
    strcpy(cur, localizedStrings[LANG_HTML_SUFFIX]);
    return(dbuf);
}

/*
 * Transformation function from vendor to filename.
 */

char *vendorName(char *vendor) {
    static char vbuf[500];
    char *cur = vbuf;

    while (*vendor != '\0') {
        if ((*vendor == '/') || (*vendor == ' ')) {
	    *cur++ = '_';
	    vendor ++;
	} else *cur++ = *vendor++;
    }
    strcpy(cur, localizedStrings[LANG_HTML_SUFFIX]);
    return(vbuf);
}

/*
 * Transformation function from resource to filename.
 */

char *resourceName(char *resource) {
    static char rbuf[500];
    char *cur = rbuf;

    while (*resource != '\0') {
        if ((*resource == '/') || (*resource == ' ')) {
	    *cur++ = '_';
	    resource ++;
	} else *cur++ = *resource++;
    }
    strcpy(cur, localizedStrings[LANG_HTML_SUFFIX]);
    return(rbuf);
}

/*
 * Generate an HTML header
 */

void generateHtmlHeader(FILE *html, char *title, char *color) {
    fprintf(html, "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">\n");
    fprintf(html, "<html>\n<head>\n<title>%s</title>\n", title);
    fprintf(html, "<meta name=\"GENERATOR\" content=\"%s %s\">\n",
            rpm2html_rpm2html_name, rpm2html_rpm2html_ver);
    if (color == NULL)
        fprintf(html, "</head>\n<body bgcolor=\"#ffffff\">\n");
    else 
        fprintf(html, "</head>\n<body bgcolor=\"%s\">\n", color);
}

/*
 * Generate an HTML footer
 */

void generateHtmlFooter(FILE *html) {
    struct tm * tstruct;
    time_t current_time;

    current_time = time(NULL);
    tstruct = localtime(&current_time);

    fprintf(html, "<hr>\n");
    fprintf(html, "<p>%s <a href=\"%s\">%s %s</a>\n",
            localizedStrings[LANG_GENERATED],
	    rpm2html_rpm2html_url, rpm2html_rpm2html_name,
	    rpm2html_rpm2html_ver);
    fprintf(html, "<p><a href=\"mailto:%s\">%s</a>, %s\n",
            rpm2html_mail, rpm2html_maint, asctime(tstruct));
    fprintf(html, "</body>\n</html>\n");
}

/*
 * Generate a line in a table for a RPM.
 */

void generateHtmlRpmRow(FILE *html, rpmDataPtr cur) {
    static char buf[500];
    struct tm * tstruct;

    tstruct = localtime(&(cur->date));
#ifdef HAVE_STRFTIME
    strftime(buf, sizeof(buf) - 1, "%c", tstruct);
#else
#error "no strftime, please check !"
#endif

    fprintf(html, "<tr><td bgcolor=\"%s\" width=\"%d\">",
	    cur->dir->color, PACKAGE_FIELD_WIDTH);
    if ((cur->subdir != NULL) && (cur->subdir[0] != '\0')) {
	if (cur->dir->url)
	    fprintf(html, "<a href=\"%s/%s/%s.html\">",
	            cur->dir->url, cur->subdir, rpmName(cur));
	else
	    fprintf(html, "<a href=\"%s/%s.html\">",
	            cur->subdir, rpmName(cur));
    } else {
	if (cur->dir->url)
	    fprintf(html, "<a href=\"%s/%s.html\">", cur->dir->url, rpmName(cur));
	else
	    fprintf(html, "<a href=\"%s.html\">", rpmName(cur));
    }
    fprintf(html, "%s</a></td>\n", rpmName(cur));
    fprintf(html, "<td bgcolor=\"%s\" width=\"%d\">%s</td>\n",
            cur->dir->color, DESCRIPTION_FIELD_WIDTH, cur->summary);
    fprintf(html, "<td>%s</td></tr>\n", buf);
}

/*
 * Generate a line in a table for an RPM software and all it's architectures.
 */

void generateHtmlRpmArchRow(FILE *html, rpmDataPtr cur) {
    rpmDataPtr tmp;

    fprintf(html, "<tr><td width=\"%d\">", PACKAGE_FIELD_WIDTH);
    fprintf(html, "%s</td>\n", rpmSoftwareName(cur));
    fprintf(html, "<td width=\"%d\">%s</td>\n",
            DESCRIPTION_FIELD_WIDTH, cur->summary);
    /* dump the archs list */
    tmp = cur;
    while (tmp != NULL) {
        fprintf(html, "<td bgcolor=\"%s\" width=\"%d\">",
	        tmp->dir->color, SYSTEM_FIELD_WIDTH);
	if ((tmp->subdir != NULL) && (tmp->subdir[0] != '\0')) {
	    if (tmp->dir->url)
		fprintf(html, "<a href=\"%s/%s/%s.html\">",
			tmp->dir->url, tmp->subdir, rpmName(tmp));
	    else
		fprintf(html, "<a href=\"%s/%s.html\">", 
		        tmp->subdir, rpmName(tmp));
	} else {
	    if (tmp->dir->url)
		fprintf(html, "<a href=\"%s/%s.html\">",
			tmp->dir->url, rpmName(tmp));
	    else
		fprintf(html, "<a href=\"%s.html\">", rpmName(tmp));
	}
	fprintf(html, "%s/%s</a>", tmp->os, tmp->arch);
        fprintf(html, "</td>");
	tmp = tmp->nextArch;
    }
    fprintf(html, "\n</tr>\n");
}

/*
 * Generate the Links for the main pages
 */

void generateLinks(FILE *html, int installed) {
    int i;

    fprintf(html, "<table border=5 cellspacing=5 cellpadding=5>\n");
    fprintf(html, "<tbody>\n<tr>\n");

    fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
            rpm2html_url, localizedStrings[LANG_INDEX_HTML],
	    localizedStrings[LANG_INDEX]);
    fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
            rpm2html_url, localizedStrings[LANG_GROUP_HTML],
	    localizedStrings[LANG_SORTED_BY_GROUP]);
    fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
            rpm2html_url, localizedStrings[LANG_DISTRIB_HTML],
	    localizedStrings[LANG_SORTED_BY_DISTRIB]);
    fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
            rpm2html_url, localizedStrings[LANG_VENDOR_HTML],
	    localizedStrings[LANG_SORTED_BY_VENDOR]);
    if (installed)
	fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
		rpm2html_url, localizedStrings[LANG_BYDATE_HTML],
		localizedStrings[LANG_SORTED_BY_IDATE]);
    else
	fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
		rpm2html_url, localizedStrings[LANG_BYDATE_HTML],
		localizedStrings[LANG_SORTED_BY_CDATE]);
    fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
            rpm2html_url, localizedStrings[LANG_BYNAME_HTML],
	    localizedStrings[LANG_SORTED_BY_NAME]);

    for (i = 0;i < rpm2html_nb_extra_headers;i++) {
        if ((*rpm2html_headers_url[i] == '/') ||
	    (!strncmp(rpm2html_headers_url[i], "http://", 7)) ||
	    (!strncmp(rpm2html_headers_url[i], "ftp://", 6)) ||
	    (!strncmp(rpm2html_headers_url[i], "mailto", 6)))
	    fprintf(html, "<td><a href=\"%s\">%s</a></td>\n",
	            rpm2html_headers_url[i], rpm2html_headers_name[i]);
	else
	    fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
	            rpm2html_url, rpm2html_headers_url[i],
		    rpm2html_headers_name[i]);
    }

    fprintf(html, "</tr>\n</tbody></table>\n");
}

/*
 * Generate a color indicator
 */

void generateColorIndicator(FILE *html) {
    int nb = 0;
    rpmDirPtr dir = dirList;

    fprintf(html, "<table align=\"center\"><tbody>\n<tr>\n");
    while (dir != NULL) {
	if (strcasecmp(dir->color, "#ffffff")) {
	    if ((nb > 0) && ((nb % MAX_COLOR_PER_LINE) == 0))
		fprintf(html, "</tr><tr>\n");
	    fprintf(html, "<td bgcolor=\"%s\">", dir->color);
	    if (dir->name != NULL)
		fprintf(html, "%s</td>", dir->name);
	    else if (dir->mirrors[0] != NULL)
		fprintf(html, "%s</td>", dir->mirrors[0]);
	    else
		fprintf(html, "%s</td>", dir->ftp);
	    nb++;
	}
	dir = dir->next;
    }
    fprintf(html, "</tr>\n</tbody></table>\n");
}

void dumpIndex(time_t start_time, int installed) {
    int i;
    FILE *html;
    rpmDirPtr cur = dirList;

    if (verbose > 1) {
    printf("Dumping %s/%s\n", rpm2html_dir, localizedStrings[LANG_INDEX_HTML]);
    }
    sprintf(buf, "%s/%s", rpm2html_dir, localizedStrings[LANG_INDEX_HTML]);

    html = fopen(buf, "w");
    if (html == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }


    if (installed) {
	sprintf(buf, "%s %s", localizedStrings[LANG_WELCOME_INSTALL], rpm2html_host);
	generateHtmlHeader(html, buf, NULL);
	generateLinks(html, installed);
	fprintf(html, "<h1 align=center>%s %s</h1>\n",
		localizedStrings[LANG_WELCOME_INSTALL], rpm2html_host);
    } else {
	sprintf(buf, "%s %s", localizedStrings[LANG_WELCOME_REPOSITORY], rpm2html_host);
	generateHtmlHeader(html, buf, NULL);
	generateLinks(html, installed);
	fprintf(html, "<h1 align=center>%s %s</h1>\n",
		localizedStrings[LANG_WELCOME_REPOSITORY], rpm2html_host);
    }

    fprintf(html, "%s\n", localizedStrings[LANG_RPM2HTML_INTRO]);
    fprintf(html, "<h3>");
    if (installed)
	fprintf(html, localizedStrings[LANG_INSTALLED_STATS],
		rpm2html_install_files, rpm2html_install_size / 1024);
    else
	fprintf(html, localizedStrings[LANG_STATS],
		rpm2html_files, rpm2html_size / 1024);
    fprintf(html, "</h3>\n");
    fprintf(html, "<ul>\n");
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_GROUP_HTML],
	    localizedStrings[LANG_INDEX_GROUP]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_BYDATE_HTML],
	    localizedStrings[LANG_INDEX_CREATION]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_BYNAME_HTML],
	    localizedStrings[LANG_INDEX_NAME]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_VENDOR_HTML],
	    localizedStrings[LANG_INDEX_VENDOR]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_DISTRIB_HTML],
	    localizedStrings[LANG_INDEX_DISTRIB]);
    fprintf(html, "</ul>\n");

    while (cur != NULL) {
	fprintf(html, "<table bgcolor=\"%s\"><tbody>\n<tr>\n",
	        cur->color);
        if (cur->name == NULL)
	    fprintf(html, "<td bgcolor=\"%s\" colspan=2><a href=\"%s\">%s</a></td>\n",
	            cur->color, cur->ftp, cur->ftp);
	else
	    fprintf(html, "<td bgcolor=\"%s\" colspan=2><a href=\"%s\">%s</a></td>\n",
	            cur->color, cur->ftp, cur->name);
	fprintf(html, "</tr>\n");
	if ((cur->ftpsrc != NULL) && (strcmp(cur->ftp, cur->ftpsrc)))
	    fprintf(html, 
	        "<tr><td bgcolor=\"%s\">%s:</td><td bgcolor=\"%s\"><a href=\"%s\">%s</a></td></tr>\n",
	        cur->color, localizedStrings[LANG_SOURCES_REPOSITORY],
		cur->color, cur->ftpsrc, cur->ftpsrc);
	if (cur->nb_mirrors > 0)
	    fprintf(html,
		"<tr><td bgcolor=\"%s\">%s:</td><td bgcolor=\"%s\"><a href=\"%s\">%s</a></td></tr>\n",
		cur->color, localizedStrings[LANG_LOCAL_MIRROR],
		cur->color, cur->mirrors[0], cur->mirrors[0]);
	if (cur->nb_mirrors > 1) {
	    fprintf(html, "<tr><td bgcolor=\"%s\" colspan=2>%s</td></tr>\n",
	            cur->color, localizedStrings[LANG_MIRRORS]);
	    for (i = 1;i < cur->nb_mirrors;i++) {
		fprintf(html,
		    "<tr><td bgcolor=\"%s\" colspan=2><a href=\"%s\">%s</a></td></tr>\n",
		    cur->color, cur->mirrors[i], cur->mirrors[i]);
	    }
	}
	fprintf(html, "</tbody></table>\n");
	cur = cur->next;
	if (cur != NULL) fprintf(html, "<p></p>\n");
    }

    fprintf(html, "<p>%s %d %s</p>\n",
            localizedStrings[LANG_GENERATION_TIME],
            (int) (time(NULL) - start_time),
            localizedStrings[LANG_SECONDS]);
    generateHtmlFooter(html);
    fclose(html);
}

/*
 * Dump an RPM block as an HTML file.
 */

void dumpRpmHtml(rpmDataPtr rpm) {
    struct tm * tstruct;
    rpmDirPtr dir = rpm->dir;
    int installed = dir->installbase;
    FILE *html;
    int i;

    if ((rpm->subdir != NULL) && (rpm->subdir[0] != '\0'))
	sprintf(buf, "%s/%s/%s.html", dir->dir, rpm->subdir, rpmName(rpm));
    else
	sprintf(buf, "%s/%s.html", dir->dir, rpmName(rpm));

    if (checkDate(buf, rpm->stamp)) return;
    if (verbose > 1) {
        printf("Dumping %s\n", buf);
    }

    html = fopen(buf, "w");
    if (html == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    sprintf(buf, "%s RPM", rpmName(rpm));
    generateHtmlHeader(html, buf, NULL);
    generateLinks(html, installed);
    if (dir->mirrors[0] != NULL)
	fprintf(html, "<h1 align=center><a href=\"%s/%s\">\n",
	    dir->mirrors[0], rpm->filename);
    else
	fprintf(html, "<h1 align=center><a href=\"%s/%s\">\n",
	    dir->ftp, rpm->filename);
    if (rpm->arch) {
	fprintf(html, "%s-%s-%s RPM for %s</a></h1>\n",
            rpm->name, rpm->version, rpm->release, rpm->arch);
    } else {
	fprintf(html, "%s-%s-%s RPM</a></h1>\n",
            rpm->name, rpm->version, rpm->release);
    }
    if (dir->name) {
        fprintf(html, "<h3 align=center>%s <a href=\"%s\">%s</a></h3>\n",
	        localizedStrings[LANG_FROM],
	        dir->ftp, dir->name);
    }
    fprintf(html, "<table align=center border=5 cellspacing=5 cellpadding=5 bgcolor=\"%s\">", dir->color);
    fprintf(html, "<tbody>\n");
    fprintf(html, "<tr><td>%s: %s</td>\n",
	    localizedStrings[LANG_NAME],
            rpm->name);
    fprintf(html, "<td>%s: <a href=\"%s/%s\">%s</a></td></tr>\n",
	    localizedStrings[LANG_DISTRIBUTION], rpm2html_url,
            distribName(rpm->distribution), rpm->distribution);
    fprintf(html, "<tr><td>%s: %s</td>\n",
	    localizedStrings[LANG_VERSION],
            rpm->version);
    fprintf(html, "<td>%s: <a href=\"%s/%s\">%s</a></td></tr>\n",
	    localizedStrings[LANG_VENDOR], rpm2html_url,
            vendorName(rpm->vendor), rpm->vendor);
    tstruct = localtime(&(rpm->date));
#ifdef HAVE_STRFTIME
    strftime(buf, sizeof(buf) - 1, "%c", tstruct);
#else
#error "no strftime, please check !"
#endif
    if (installed) {
	fprintf(html, "<tr><td>%s: %s</td>\n<td>%s: %s</td></tr>\n",
		localizedStrings[LANG_RELEASE],
		rpm->release,
		localizedStrings[LANG_INSTALL_DATE],
		buf);
    } else {
	fprintf(html, "<tr><td>%s: %s</td>\n<td>%s: %s</td></tr>\n",
		localizedStrings[LANG_RELEASE],
		rpm->release,
		localizedStrings[LANG_BUILD_DATE],
		buf);
    }
    if (rpm2html_url != NULL)
	fprintf(html, "<tr><td>%s: <a href=\"%s/%s\">%s</a></td>\n",
		localizedStrings[LANG_GROUP], rpm2html_url,
		groupName(rpm->group), rpm->group);
    else
	fprintf(html, "<tr><td>%s: <a href=\"%s\">%s</a></td>\n",
		localizedStrings[LANG_GROUP],
		groupName(rpm->group), rpm->group);
    fprintf(html, "<td>%s: %s</td></tr>\n",
	    localizedStrings[LANG_BUILD_HOST], rpm->host);
    fprintf(html, "<tr><td>%s: %d</td>\n",
	    localizedStrings[LANG_SIZE],
            rpm->size);
    if (dir->ftpsrc) {
	fprintf(html, "<td>%s: <a href=\"%s/%s\">%s</a></td></tr>\n",
	        localizedStrings[LANG_RPM_SRC],
		dir->ftpsrc, rpm->srcrpm, rpm->srcrpm);
    } else {
	fprintf(html, "<td>%s: %s</td></tr>\n",
	        localizedStrings[LANG_RPM_SRC],
		rpm->srcrpm);
    }
    if (rpm->packager) {
        char *email = extractEMail(rpm->packager);
	if (email == NULL)
	    fprintf(html, "<tr><td colspan=\"2\">%s: %s</td></tr>\n",
		    localizedStrings[LANG_PACKAGER],
		    rpm->packager);
        else
	    fprintf(html, "<tr><td colspan=\"2\">%s: <a href=\"mailto:%s\">%s</a></td></tr>\n",
		    localizedStrings[LANG_PACKAGER],
		    email, rpm->packager);
    }
    if (rpm->url)
	fprintf(html, "<tr><td colspan=\"2\">%s: <a href=\"%s\">%s</a></td></tr>\n",
		localizedStrings[LANG_URL],
		rpm->url, rpm->url);
    fprintf(html, "<tr><td colspan=\"2\">%s: %s</td></tr>\n",
	    localizedStrings[LANG_SUMMARY],
            rpm->summary);
    fprintf(html, "</tbody>\n</table>\n");
    fprintf(html, "<pre>%s\n</pre>\n", rpm->description);
    if ((rpm->nb_resources + rpm->nb_requires < 2) &&
        (!strstr(rpm->name, "lib"))) {
       if (verbose > 1)
	   fprintf(stderr, "Resource lists problem : %s\n", rpmName(rpm));
       fprintf(html, "<h2 align=center style=\"color : #ff0000\">%s</h2>\n",
               localizedStrings[LANG_WARNING_RESOURCES]);
       fprintf(html, "<h2 align=center><a href=\"%s/%s\">%s</a></h2>\n",
               rpm2html_url, resourceName(rpm->name),
               localizedStrings[LANG_CHOOSE_ANOTHER]);
    }
    if (rpm->nb_resources > 0) {
       fprintf(html, "<h3>%s</h3>\n",
	    localizedStrings[LANG_PROVIDE]);
       fprintf(html, "<ul>\n");
       for (i = 0;i < rpm->nb_resources;i++) {
	   if (rpm2html_url != NULL)
	       fprintf(html, "<li><a href=\"%s/%s\">%s</a>\n",
		       rpm2html_url, resourceName(rpm->resources[i]->name),
		       rpm->resources[i]->name);
	   else
	       fprintf(html, "<li><a href=\"%s\">%s</a>\n",
		       resourceName(rpm->resources[i]->name),
		       rpm->resources[i]->name);
       }
       if (i >= MAX_RES)
           fprintf(html, "<li>%s</a>\n", localizedStrings[LANG_MORE]);
       fprintf(html, "</ul>\n");
    }
    if (rpm->nb_requires > 0) {
       fprintf(html, "<h3>%s</h3>\n",
	    localizedStrings[LANG_REQUIRE]);
       fprintf(html, "<ul>\n");
       for (i = 0;i < rpm->nb_requires;i++) {
	   if (rpm2html_url != NULL)
	       fprintf(html, "<li><a href=\"%s/%s\">%s</a>\n",
		       rpm2html_url, resourceName(rpm->requires[i]->name),
		       rpm->requires[i]->name);
	   else
	       fprintf(html, "<li><a href=\"%s\">%s</a>\n",
		       resourceName(rpm->requires[i]->name),
		       rpm->requires[i]->name);
       }
       if (i >= MAX_REQU)
           fprintf(html, "<li>...</a>\n");
       fprintf(html, "</ul>\n");
    }
    if (rpm->copyright) {
       fprintf(html, "<h3>%s</h3>\n",
	    localizedStrings[LANG_COPYRIGHT]);
	fprintf(html, "<pre>%s\n</pre>\n", rpm->copyright);
    }
    if (rpm->changelog) {
        fprintf(html, "<h3>%s</h3>\n",
	    localizedStrings[LANG_CHANGELOG]);
	fprintf(html, "<pre>%s\n</pre>\n", rpm->changelog);
    }
    fprintf(html, "<h3>%s</h3>\n",
	localizedStrings[LANG_FILES]);
    if (rpm->filelist == NULL) 
	fprintf(html, "<bold>%s</bold>\n",
	     localizedStrings[LANG_NO_FILES]);
    else
	fprintf(html, "<pre>%s\n</pre>\n", rpm->filelist);
    
    generateHtmlFooter(html);
    fclose(html);
}

/*
 * Dump a resource block as an HTML file.
 */

void dumpRessHtml(rpmRessPtr ress, int installed) {
    rpmDataPtr rpm;
    FILE *html;
    int i;

    sprintf(buf, "%s/%s", rpm2html_dir,
           resourceName(ress->name));
    if (checkDate(buf, ress->stamp)) return;
    if (verbose > 1) {
    printf("Dumping %s/%s\n", rpm2html_dir,
           resourceName(ress->name));
    }

    html = fopen(buf, "w");
    if (html == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    sprintf(buf, "%s %s",
	 localizedStrings[LANG_RPM_RESOURCE], ress->name);
    generateHtmlHeader(html, buf, NULL);
    generateLinks(html, installed);
    generateColorIndicator(html);
    fprintf(html, "<h1 align=center>%s %s</h1>\n",
	 localizedStrings[LANG_RPM_RESOURCE], ress->name);
    fprintf(html, "<h3>%s</h3>\n",
	 localizedStrings[LANG_PROVIDED_BY]);
    fprintf(html, "<table><tbody>\n");
    for (i = 0;i < ress->nb_provider;i++) {
        rpm = ress->provider[i];
	generateHtmlRpmRow(html, rpm);
    }
    fprintf(html, "</tbody></table>\n");
    
    generateHtmlFooter(html);
    fclose(html);
}

/*
 * Dump all RPM blocks in the HTML files.
 * Not used anymore there are dumped are they are parsed
 * to minimize memory requirements
void dumpAllRpmHtml(int installed) {
    rpmDataPtr cur;

    if (installed) cur = rpmInstalledList;
    else cur = rpmList;

    while (cur != NULL) {
        dumpRpmHtml(cur);
	cur = cur->next;
    }
}
 */

/*
 * Dump all RPM blocks in the HTML files.
 */
void dumpAllRessHtml(int installed) {
    rpmRessPtr cur;

    if (installed) cur = ressInstalledList;
    else cur = ressList;

    while (cur != NULL) {
        dumpRessHtml(cur, installed);
	cur = cur->next;
    }
}

/*
 * Dump the Groups HTML files.
 * One expect that the RPM files have been sorted by group.
 */
void dumpRpmGroups(int installed) {
    FILE *Groups;
    FILE *currentGroup = NULL;
    rpmDataPtr cur, prev = NULL;
    int count = 0;
    int pcount = 0;

    cur = rpmSoftwareList;
    
    if (verbose > 1) {
    printf("Dumping %s/%s\n", rpm2html_dir,
           localizedStrings[LANG_GROUP_HTML]);
    }
    sprintf(buf, "%s/%s", rpm2html_dir,
           localizedStrings[LANG_GROUP_HTML]);

    Groups = fopen(buf, "w");
    if (Groups == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    generateHtmlHeader(Groups, localizedStrings[LANG_SORTED_BY_GROUP], NULL);
    generateLinks(Groups, installed);
    fprintf(Groups, "<h1 align=center>%s</h1>\n",
        localizedStrings[LANG_SORTED_BY_GROUP]);
    fprintf(Groups, "<ul>\n");

    while (cur != NULL) {
        if ((cur->group != NULL) && (strlen(cur->group) > 0)) {
	    if ((prev == NULL) || (strcasecmp(prev->group, cur->group))) {
		if (pcount != 0)
		    fprintf(Groups, " (%d)</a></li>\n", pcount);
		pcount = 0;
	        if (currentGroup != NULL) {
		    /* one need to close the current group list */
		    fprintf(currentGroup,"</tbody></table>\n");
		    generateHtmlFooter(currentGroup);
		    fclose(currentGroup);
		}

		/* Add the current group in the Group list */
		fprintf(Groups, "<li><a href=\"%s\">%s",
		        groupName(cur->group), cur->group);

		/* open the new HTML group file */
    if (verbose > 1) {
		printf("Dumping %s/%s\n", rpm2html_dir, groupName(cur->group));
    }
		sprintf(buf, "%s/%s", rpm2html_dir, groupName(cur->group));

		currentGroup = fopen(buf, "w");
		if (currentGroup == NULL) {
		    fprintf(stderr, "Couldn't save to file %s: %s\n",
			    buf, strerror(errno));
		    return;
		}
                sprintf(buf, "%s %s", localizedStrings[LANG_OF_GROUP],
		        cur->group);
                generateHtmlHeader(currentGroup, buf, NULL);
		generateLinks(currentGroup, installed);
		fprintf(currentGroup,
		  "<h1 align=center>%s %s</h1>\n",
		     localizedStrings[LANG_OF_GROUP], cur->group);
		if (!installed) generateColorIndicator(currentGroup);
		fprintf(currentGroup,"<table><tbody>\n");
		count = 0;
	    }
	    generateHtmlRpmArchRow(currentGroup, cur);
	    count++;
	    pcount++;
	    if ((count % MAX_TABLE_LENGHT) == 0)
		fprintf(currentGroup, "</tbody></table>\n<table><tbody>\n");
	}
	prev = cur;
	cur = cur->nextSoft;
    }

    /*
     * finish the last group line.
     */
    if (pcount != 0)
	fprintf(Groups, " (%d)</a></li>\n", pcount);

    if (currentGroup != NULL) {
	/* one need to close the current group list */
	fprintf(currentGroup,"</tbody></table>\n");
	generateHtmlFooter(currentGroup);
	fclose(currentGroup);
    }
    fprintf(Groups, "</ul>\n");
    generateHtmlFooter(Groups);
    fclose(Groups);
}

/*
 * Dump the Distribs HTML files.
 * One expect that the RPM files have been sorted by distribution.
 */
void dumpRpmDistribs(int installed) {
    FILE *Distribs;
    FILE *currentDistrib = NULL;
    rpmDataPtr cur, prev = NULL;
    int count = 0;
    int pcount = 0;

    if (installed) cur = rpmInstalledList;
    else cur = rpmList;

    if (verbose > 1) {
    printf("Dumping %s/%s\n", rpm2html_dir, localizedStrings[LANG_DISTRIB_HTML]);
    }
    sprintf(buf, "%s/%s", rpm2html_dir, localizedStrings[LANG_DISTRIB_HTML]);

    Distribs = fopen(buf, "w");
    if (Distribs == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    generateHtmlHeader(Distribs,
        localizedStrings[LANG_SORTED_BY_DISTRIB], NULL);
    generateLinks(Distribs, installed);
    fprintf(Distribs, "<h1 align=center>%s</h1>\n",
        localizedStrings[LANG_SORTED_BY_DISTRIB]);
    fprintf(Distribs, "<ul>\n");

    while (cur != NULL) {
        if ((cur->distribution != NULL) && (strlen(cur->distribution) > 0)) {
	    if ((prev == NULL) || (strcasecmp(prev->distribution, cur->distribution))) {
		if (pcount != 0)
		    fprintf(Distribs, " (%d)</a></li>\n", pcount);
		pcount = 0;

	        if (currentDistrib != NULL) {
		    /* one need to close the current distribution list */
		    fprintf(currentDistrib,"</tbody></table>\n");
		    generateHtmlFooter(currentDistrib);
		    fclose(currentDistrib);
		}

		/* Add the current distribution in the Distrib list */
		fprintf(Distribs, "<li><a href=\"%s\">%s",
		        distribName(cur->distribution), cur->distribution);

		/* open the new HTML distribution file */
    if (verbose > 1) {
		printf("Dumping %s/%s\n", rpm2html_dir, distribName(cur->distribution));
    }
		sprintf(buf, "%s/%s", rpm2html_dir, distribName(cur->distribution));

		currentDistrib = fopen(buf, "w");
		if (currentDistrib == NULL) {
		    fprintf(stderr, "Couldn't save to file %s: %s\n",
			    buf, strerror(errno));
		    return;
		}
                sprintf(buf, "%s %s",
		    localizedStrings[LANG_OF_DISTRIB], cur->distribution);
                generateHtmlHeader(currentDistrib, buf, NULL);
		generateLinks(currentDistrib, installed);
		fprintf(currentDistrib,
		  "<h1 align=center>%s %s</h1>\n",
		  localizedStrings[LANG_OF_DISTRIB],
		  cur->distribution);
		if (!installed) generateColorIndicator(currentDistrib);
		fprintf(currentDistrib,"<table><tbody>\n");
		count = 0;
	    }
	    generateHtmlRpmRow(currentDistrib, cur);
	    count++;
	    pcount++;
	    if ((count % MAX_TABLE_LENGHT) == 0)
		fprintf(currentDistrib, "</tbody></table>\n<table><tbody>\n");
	}
	prev = cur;
	cur = cur->next;
    }
    /*
     * finish the last group line.
     */
    if (pcount != 0)
	fprintf(Distribs, " (%d)</a></li>\n", pcount);

    if (currentDistrib != NULL) {
	/* one need to close the current distribution list */
	fprintf(currentDistrib,"</tbody></table>\n");
	generateHtmlFooter(currentDistrib);
	fclose(currentDistrib);
    }
    fprintf(Distribs, "</ul>\n");
    generateHtmlFooter(Distribs);
    fclose(Distribs);
}

/*
 * Dump the Vendors HTML files.
 * One expect that the RPM files have been sorted by vendors.
 */
void dumpRpmVendors(int installed) {
    FILE *Vendors;
    FILE *currentVendor = NULL;
    rpmDataPtr cur, prev = NULL;
    int count = 0;
    int pcount = 0;

    if (installed) cur = rpmInstalledList;
    else cur = rpmList;

    if (verbose > 1) {
    printf("Dumping %s/%s\n", rpm2html_dir, localizedStrings[LANG_VENDOR_HTML]);
    }
    sprintf(buf, "%s/%s", rpm2html_dir, localizedStrings[LANG_VENDOR_HTML]);

    Vendors = fopen(buf, "w");
    if (Vendors == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    generateHtmlHeader(Vendors, localizedStrings[LANG_SORTED_BY_VENDOR], NULL);
    generateLinks(Vendors, installed);
    fprintf(Vendors, "<h1 align=center>%s</h1>\n",
        localizedStrings[LANG_SORTED_BY_VENDOR]);
    fprintf(Vendors, "<ul>\n");

    while (cur != NULL) {
        if ((cur->vendor != NULL) && (strlen(cur->vendor) > 0)) {
	    if ((prev == NULL) || (strcasecmp(prev->vendor, cur->vendor))) {
		if (pcount != 0)
		    fprintf(Vendors, " (%d)</a></li>\n", pcount);
		pcount = 0;

	        if (currentVendor != NULL) {
		    /* one need to close the current vendor list */
		    fprintf(currentVendor,"</tbody></table>\n");
		    generateHtmlFooter(currentVendor);
		    fclose(currentVendor);
		}

		/* Add the current vendor in the Vendor list */
		fprintf(Vendors, "<li><a href=\"%s\">%s",
		        vendorName(cur->vendor), cur->vendor);

		/* open the new HTML vendor file */
    if (verbose > 1) {
		printf("Dumping %s/%s\n", rpm2html_dir, vendorName(cur->vendor));
    }
		sprintf(buf, "%s/%s", rpm2html_dir, vendorName(cur->vendor));

		currentVendor = fopen(buf, "w");
		if (currentVendor == NULL) {
		    fprintf(stderr, "Couldn't save to file %s: %s\n",
			    buf, strerror(errno));
		    return;
		}
                sprintf(buf, "%s %s", localizedStrings[LANG_OF_VENDOR],
		    cur->vendor);
                generateHtmlHeader(currentVendor, buf, NULL);
		generateLinks(currentVendor, installed);
		fprintf(currentVendor,
		  "<h1 align=center>%s %s</h1>\n",
		  localizedStrings[LANG_OF_VENDOR], cur->vendor);
		if (!installed) generateColorIndicator(currentVendor);
		fprintf(currentVendor,"<table><tbody>\n");
		count = 0;
	    }
	    generateHtmlRpmRow(currentVendor, cur);
	    count++;
	    pcount++;
	    if ((count % MAX_TABLE_LENGHT) == 0)
		fprintf(currentVendor, "</tbody></table>\n<table><tbody>\n");
	}
	prev = cur;
	cur = cur->next;
    }

    /*
     * finish the last group line.
     */
    if (pcount != 0)
	fprintf(Vendors, " (%d)</a></li>\n", pcount);

    if (currentVendor != NULL) {
	/* one need to close the current vendor list */
	fprintf(currentVendor,"</tbody></table>\n");
	generateHtmlFooter(currentVendor);
	fclose(currentVendor);
    }
    fprintf(Vendors, "</ul>\n");
    generateHtmlFooter(Vendors);
    fclose(Vendors);
}

/*
 * Dump the RPM by Date HTML file.
 * One expect that the RPM files have been sorted by date.
 */
void dumpRpmByDate(int installed) {
    FILE *ByDate;
    rpmDataPtr cur;
    time_t current_time = time(NULL);
    int count = 0;
    int page = 1;

    if (installed) cur = rpmInstalledList;
    else cur = rpmList;

    sprintf(buf, "%s/%s", rpm2html_dir, localizedStrings[LANG_BYDATE_HTML]);
    if (verbose > 1) {
	printf("Dumping %s\n", buf);
    }

    ByDate = fopen(buf, "w");
    if (ByDate == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    if (installed) {
	generateHtmlHeader(ByDate, localizedStrings[LANG_SORTED_BY_IDATE], NULL);
	generateLinks(ByDate, installed);
	fprintf(ByDate, "<h1 align=center>%s</h1>\n",
	    localizedStrings[LANG_SORTED_BY_IDATE]);
    } else {
	generateHtmlHeader(ByDate, localizedStrings[LANG_SORTED_BY_CDATE], NULL);
	generateLinks(ByDate, installed);
	fprintf(ByDate, "<h1 align=center>%s</h1>\n",
	    localizedStrings[LANG_SORTED_BY_CDATE]);
    }

    /*
     * Skip all the RPMs with date in the futur :-(
     */
    while ((cur != NULL) && ((current_time < cur->date))) {
	cur = cur->next;
    }

    /*
     * First dump RPMs less than 3 days old.
     */
    if (!installed) generateColorIndicator(ByDate);
    if (installed)
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_3D_OLD]);
    else
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_3D_OLD]);
    fprintf(ByDate, "<table><tbody>\n");
    while ((cur != NULL) && ((current_time - cur->date) < (3 * 24 * 60 * 60))) {
	generateHtmlRpmRow(ByDate, cur);
	cur = cur->next;
	count++;
	if ((count % MAX_TABLE_LENGHT) == 0)
	    fprintf(ByDate, "</tbody></table>\n<table><tbody>\n");
	if (count > MAX_PAGE_LENGHT) {
	    count = 0;
	    fprintf(ByDate, "</tbody></table>\n");
	    fprintf(ByDate, "<a href=\"%d%s\">...</a>\n",
	            page, localizedStrings[LANG_BYDATE_HTML]);
	    generateHtmlFooter(ByDate);
	    fclose(ByDate);
	    sprintf(buf, "%s/%d%s", rpm2html_dir,
	            page++, localizedStrings[LANG_BYDATE_HTML]);
	    if (verbose > 1) {
		printf("Dumping %s\n", buf);
	    }
	    ByDate = fopen(buf, "w");
	    if (ByDate == NULL) {
		fprintf(stderr, "Couldn't save to file %s: %s\n",
			buf, strerror(errno));
		return;
	    }
	    if (installed) {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_IDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_IDATE]);
	    } else {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_CDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_CDATE]);
	    }
	    if (installed)
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_3D_OLD]);
	    else
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_3D_OLD]);
	    fprintf(ByDate, "<table><tbody>\n");
	}
    }
    fprintf(ByDate, "</tbody></table>\n");

    /*
     * Then dump RPMs less than one week old.
     */
    if (installed)
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_1W_OLD]);
    else
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_1W_OLD]);
    fprintf(ByDate, "<table><tbody>\n");
    while ((cur != NULL) && ((current_time - cur->date) < (7 * 24 * 60 * 60))) {
	generateHtmlRpmRow(ByDate, cur);
	cur = cur->next;
	count++;
	if ((count % MAX_TABLE_LENGHT) == 0)
	    fprintf(ByDate, "</tbody></table>\n<table><tbody>\n");
	if (count > MAX_PAGE_LENGHT) {
	    count = 0;
	    fprintf(ByDate, "</tbody></table>\n");
	    fprintf(ByDate, "<a href=\"%d%s\">...</a>\n",
	            page, localizedStrings[LANG_BYDATE_HTML]);
	    generateHtmlFooter(ByDate);
	    fclose(ByDate);
	    sprintf(buf, "%s/%d%s", rpm2html_dir,
	            page++, localizedStrings[LANG_BYDATE_HTML]);
	    if (verbose > 1) {
		printf("Dumping %s\n", buf);
	    }
	    ByDate = fopen(buf, "w");
	    if (ByDate == NULL) {
		fprintf(stderr, "Couldn't save to file %s: %s\n",
			buf, strerror(errno));
		return;
	    }
	    if (installed) {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_IDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_IDATE]);
	    } else {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_CDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_CDATE]);
	    }
	    if (installed)
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_1W_OLD]);
	    else
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_1W_OLD]);
	    fprintf(ByDate, "<table><tbody>\n");
	}
    }
    fprintf(ByDate, "</tbody></table>\n");

    /*
     * Then dump RPMs less than two weeks old.
     */
    if (installed)
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_2W_OLD]);
    else
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_2W_OLD]);
    fprintf(ByDate, "<table><tbody>\n");
    while ((cur != NULL) && ((current_time - cur->date) < (14 * 24 * 60 * 60))) {
	generateHtmlRpmRow(ByDate, cur);
	cur = cur->next;
	count++;
	if ((count % MAX_TABLE_LENGHT) == 0)
	    fprintf(ByDate, "</tbody></table>\n<table><tbody>\n");
	if (count > MAX_PAGE_LENGHT) {
	    count = 0;
	    fprintf(ByDate, "</tbody></table>\n");
	    fprintf(ByDate, "<a href=\"%d%s\">...</a>\n",
	            page, localizedStrings[LANG_BYDATE_HTML]);
	    generateHtmlFooter(ByDate);
	    fclose(ByDate);
	    sprintf(buf, "%s/%d%s", rpm2html_dir,
	            page++, localizedStrings[LANG_BYDATE_HTML]);
	    if (verbose > 1) {
		printf("Dumping %s\n", buf);
	    }
	    ByDate = fopen(buf, "w");
	    if (ByDate == NULL) {
		fprintf(stderr, "Couldn't save to file %s: %s\n",
			buf, strerror(errno));
		return;
	    }
	    if (installed) {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_IDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_IDATE]);
	    } else {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_CDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_CDATE]);
	    }
	    if (installed)
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_2W_OLD]);
	    else
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_2W_OLD]);
	    fprintf(ByDate, "<table><tbody>\n");
	}
    }
    fprintf(ByDate, "</tbody></table>\n");

    /*
     * Then dump RPMs less than one month old.
     */
    if (installed)
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_1M_OLD]);
    else
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_1M_OLD]);
    fprintf(ByDate, "<table><tbody>\n");
    while ((cur != NULL) && ((current_time - cur->date) < (30 * 24 * 60 * 60))) {
	generateHtmlRpmRow(ByDate, cur);
	cur = cur->next;
	count++;
	if ((count % MAX_TABLE_LENGHT) == 0)
	    fprintf(ByDate, "</tbody></table>\n<table><tbody>\n");
	if (count > MAX_PAGE_LENGHT) {
	    count = 0;
	    fprintf(ByDate, "</tbody></table>\n");
	    fprintf(ByDate, "<a href=\"%d%s\">...</a>\n",
	            page, localizedStrings[LANG_BYDATE_HTML]);
	    generateHtmlFooter(ByDate);
	    fclose(ByDate);
	    sprintf(buf, "%s/%d%s", rpm2html_dir,
	            page++, localizedStrings[LANG_BYDATE_HTML]);
	    if (verbose > 1) {
		printf("Dumping %s\n", buf);
	    }
	    ByDate = fopen(buf, "w");
	    if (ByDate == NULL) {
		fprintf(stderr, "Couldn't save to file %s: %s\n",
			buf, strerror(errno));
		return;
	    }
	    if (installed) {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_IDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_IDATE]);
	    } else {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_CDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_CDATE]);
	    }
	    if (installed)
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_1M_OLD]);
	    else
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_1M_OLD]);
	    fprintf(ByDate, "<table><tbody>\n");
	}
    }
    fprintf(ByDate, "</tbody></table>\n");

    /*
     * Then dump RPMs more than one month old.
     */
    if (installed)
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_MORE_1M_OLD]);
    else
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_MORE_1M_OLD]);
    fprintf(ByDate, "<table><tbody>\n");
    while (cur != NULL) {
	generateHtmlRpmRow(ByDate, cur);
	cur = cur->next;
	count++;
	if ((count % MAX_TABLE_LENGHT) == 0)
	    fprintf(ByDate, "</tbody></table>\n<table><tbody>\n");
	if (count > MAX_PAGE_LENGHT) {
	    count = 0;
	    fprintf(ByDate, "</tbody></table>\n");
	    fprintf(ByDate, "<a href=\"%d%s\">...</a>\n",
	            page, localizedStrings[LANG_BYDATE_HTML]);
	    generateHtmlFooter(ByDate);
	    fclose(ByDate);
	    sprintf(buf, "%s/%d%s", rpm2html_dir,
	            page++, localizedStrings[LANG_BYDATE_HTML]);
	    if (verbose > 1) {
		printf("Dumping %s\n", buf);
	    }
	    ByDate = fopen(buf, "w");
	    if (ByDate == NULL) {
		fprintf(stderr, "Couldn't save to file %s: %s\n",
			buf, strerror(errno));
		return;
	    }
	    if (installed) {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_IDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_IDATE]);
	    } else {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_CDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_CDATE]);
	    }
	    if (installed)
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_MORE_1M_OLD]);
	    else
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_MORE_1M_OLD]);
	    fprintf(ByDate, "<table><tbody>\n");
	}
    }
    fprintf(ByDate, "</tbody></table>\n");

    generateHtmlFooter(ByDate);
    fclose(ByDate);
}

/*
 * Dump the RPM by Name HTML file.
 * One expect that the RPM files have been sorted by name.
 */
void dumpRpmByName(int installed) {
    FILE *ByName;
    FILE *CurrName = NULL;
    rpmDataPtr cur;
    int i = 0;
    char last_letter = '\0', letter;

    cur = rpmSoftwareList;

    sprintf(buf, "%s/%s", rpm2html_dir, localizedStrings[LANG_BYNAME_HTML]);
    if (verbose > 1) {
        printf("Dumping %s\n", buf);
    }

    ByName = fopen(buf, "w");
    if (ByName == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    generateHtmlHeader(ByName, localizedStrings[LANG_SORTED_BY_NAME], NULL);
    generateLinks(ByName, installed);
    fprintf(ByName, "<h1 align=center>%s</h1>\n",
        localizedStrings[LANG_SORTED_BY_NAME]);

    while (cur != NULL) {
	letter = toupper(cur->name[0]);
	if (letter != last_letter) {
	    if (CurrName != NULL) {
		/*
		 * Finish the previous file.
		 */
		fprintf(CurrName, "</tbody></table>\n");
		generateHtmlFooter(CurrName);
		fclose(CurrName);
	    }
	    if (i != 0)
		fprintf(ByName, "<p><a href=\"%c%s\">%d %s %c</a></p>\n",
		        last_letter, localizedStrings[LANG_BYNAME_HTML],
			i, localizedStrings[LANG_BEGINNING_LETTER],
			last_letter);
	    sprintf(buf, "%s/%c%s", rpm2html_dir, letter,
	            localizedStrings[LANG_BYNAME_HTML]);
	    if (verbose > 1) {
		printf("Dumping %s\n", buf);
	    }

	    CurrName = fopen(buf, "w");
	    if (CurrName == NULL) {
		fprintf(stderr, "Couldn't save to file %s: %s\n",
			buf, strerror(errno));
		return;
	    }
	    sprintf(buf, "%s %c", localizedStrings[LANG_BEGINNING_LETTER],
	            letter);
	    generateHtmlHeader(CurrName, buf, NULL);
	    generateLinks(CurrName, installed);
	    fprintf(CurrName, "<h1 align=center>%s</h1>\n", buf);
	    if (!installed) generateColorIndicator(CurrName);
	    fprintf(CurrName, "<table><tbody>\n");
	    i = 0;
	}
	i++;
	if ((i % MAX_TABLE_LENGHT) == 0)
	    fprintf(CurrName, "</tbody></table>\n<table><tbody>\n");
	last_letter = letter;
	generateHtmlRpmArchRow(CurrName, cur);
	cur = cur->nextSoft;
    }

    if (i != 0)
	fprintf(ByName, "<p><a href=\"%c%s\">%d %s %c</a></p>\n",
		last_letter, localizedStrings[LANG_BYNAME_HTML],
		i, localizedStrings[LANG_BEGINNING_LETTER],
		last_letter);
    if (CurrName != NULL) {
	/*
	 * Finish the previous file.
	 */
	fprintf(CurrName, "</tbody></table>\n");
	generateHtmlFooter(CurrName);
	fclose(CurrName);
    }
    generateHtmlFooter(ByName);
    fclose(ByName);
}
