/*
 * config.c : handle the configuration file.
 *
 * Copyright (c) 1997 Daniel Veillard <veillard@apocalypse.org>
 * See COPYING for the status of this software.
 *
 * $Id: config.c,v 1.15 1998/02/27 20:30:12 veillard Exp $
 */

#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "rpm2html.h"
#include "rpmdata.h"

/*
 * configuration variables for rpm2html
 */

char *rpm2html_rpm2html_name	= RPM2HTML_NAME;
char *rpm2html_rpm2html_ver	= RPM2HTML_VER;
char *rpm2html_rpm2html_url	= RPM2HTML_URL;
char *rpm2html_rpm2html_thishost= NULL;
char *rpm2html_maint		= RPM2HTML_MAINT;
char *rpm2html_mail		= RPM2HTML_MAIL;
char *rpm2html_dir		= NULL;
char *rpm2html_name		= NULL;
char *rpm2html_url		= NULL;
char *rpm2html_ftp		= NULL;
char *rpm2html_ftpsrc		= NULL;
char *rpm2html_host		= NULL;

int  rpm2html_files = 0;
int  rpm2html_size = 0;

int  rpm2html_install_files = 0;
int  rpm2html_install_size = 0;

char *rpm2html_headers_name[MAX_EXTRA_HEADERS];
char *rpm2html_headers_url[MAX_EXTRA_HEADERS];
int   rpm2html_nb_extra_headers = 0; 

void addHeader(char *value);

#ifdef HAVE_STRNDUP
extern char *strndup (const char *source, size_t len);
#else /* ! HAVE_STRNDUP */
/*
 * 
 */
char *strndup (const char *source, size_t len) {
    char* tmp = NULL;

    if ((source == NULL) || (len < 0)) return(NULL);
    if (len <= strlen(source)) return strdup(source);

    tmp = malloc(len+1);
    strncpy(tmp, source, len);
    tmp[len] = '\0';

    return(tmp);
}
#endif /* HAVE_STRNDUP */

/*
 * free a directory structure.
 */

void rpmDirFree(rpmDirPtr dir) {
    int i;

    if (dir->color) free(dir->color);
    if (dir->dir != NULL) free(dir->dir);
    if (dir->ftp != NULL) free(dir->ftp);
    if (dir->ftpsrc != NULL) free(dir->ftpsrc);
    if (dir->host != NULL) free(dir->host);
    if (dir->mail != NULL) free(dir->mail);
    if (dir->maint != NULL) free(dir->maint);
    if (dir->name != NULL) free(dir->name);
    if (dir->url != NULL) free(dir->url);
    for (i = 0;i < dir->nb_mirrors ; i++)
        if (dir->mirrors[i] != NULL) free(dir->mirrors[i]);
    if (dir->trust != NULL) free(dir->trust);
    if (dir->rpmdir != NULL) free(dir->rpmdir);
    dir->next = NULL;
    free(dir);
}

/*
 * free a directory list.
 */

void rpmDirListFree(rpmDirPtr *base) {
    rpmDirPtr dir = *base, next;

    while (dir != NULL) {
        next = dir->next;
	rpmDirFree(dir);
	dir = next;
    }
    *base = NULL;
}

/*
 * free an RPM structure.
 */

void rpmDataFree(rpmDataPtr rpm) {
    if (rpm->filename) free(rpm->filename);
    if (rpm->name != NULL) free(rpm->name);
    if (rpm->version != NULL) free(rpm->version);
    if (rpm->release != NULL) free(rpm->release);
    if (rpm->url != NULL) free(rpm->url);
    if (rpm->arch != NULL) free(rpm->arch);
    if (rpm->os != NULL) free(rpm->os);
    if (rpm->distribution != NULL) free(rpm->distribution);
    if (rpm->vendor != NULL) free(rpm->vendor);
    if (rpm->packager != NULL) free(rpm->packager);
    if (rpm->group != NULL) free(rpm->group);
    if (rpm->summary != NULL) free(rpm->summary);
    if (rpm->description != NULL) free(rpm->description);
    if (rpm->copyright != NULL) free(rpm->copyright);
    if (rpm->srcrpm != NULL) free(rpm->srcrpm);
    if (rpm->host != NULL) free(rpm->host);
    if (rpm->filelist != NULL) free(rpm->filelist);
    if (rpm->subdir != NULL) free(rpm->subdir);
    rpm->next = NULL;
    free(rpm);
}

/*
 * free an RPM list.
 */

void rpmDataListFree(rpmDataPtr *base) {
    rpmDataPtr rpm = *base, next;

    while (rpm != NULL) {
        next = rpm->next;
	rpmDataFree(rpm);
	rpm = next;
    }
    *base = NULL;
}

/*
 * free a resource structure.
 */

void rpmRessFree(rpmRessPtr res) {
    if (res->name != NULL) free(res->name);
    res->next = NULL;
    free(res);
}

/*
 * free a resource list.
 */

void rpmRessListFree(rpmRessPtr *base) {
    rpmRessPtr res = *base, next;

    while (res != NULL) {
        next = res->next;
	rpmRessFree(res);
	res = next;
    }
    *base = NULL;
}

/*
 * Search a directory in the list. If not found, create a new one.
 */

rpmDirPtr rpmDirSearch(char *dirname) {
    rpmDirPtr cur = dirList;

    while (cur != NULL) {
        if (!strcmp(dirname, cur->rpmdir)) return(cur);
	cur = cur->next;
    }
    cur = (rpmDirPtr) malloc(sizeof(rpmDir));
    memset(cur, 0, sizeof(rpmDir));
    cur->color = strdup("#ffffff");
    cur->dir = NULL;
    cur->files = 0;
    cur->ftp = NULL;
    cur->ftpsrc = NULL;
    cur->host = NULL;
    cur->mail = NULL;
    cur->maint = NULL;
    cur->name = NULL;
    cur->nb_mirrors = 0;
    cur->mirrors[0] = NULL;
    cur->rpmdir = strdup(dirname);
    cur->size = 0;
    cur->trust = "1.0";
    cur->trust = NULL;
    cur->url = NULL;

    if (strcmp(dirname, "localbase"))
        cur->installbase = 0;
    else
        cur->installbase = 1;

    cur->next = dirList;
    dirList = cur;
    return(cur);
}

/*
 * addConfigEntry : an entry in the config file has just been read.
 */
void addConfigEntry(char *rpmdir, char *name, char *value) {
    rpmDirPtr cur;

    if (verbose > 1)
	printf("addConfigEntry(\"%s\", \"%s\", \"%s\")\n", rpmdir, name, value);

    /*
     * case of global option for rpm2html.
     */
    if (!strcasecmp(rpmdir, RPM2HTML_NAME)) {
        if (!strcasecmp(name, "url")) {
	    rpm2html_url = strdup(value);
	} else if (!strcasecmp(name, "maint")) {
	    rpm2html_maint = strdup(value);
	} else if (!strcasecmp(name, "mail")) {
	    rpm2html_mail = strdup(value);
	} else if (!strcasecmp(name, "dir")) {
	    rpm2html_dir = strdup(value);
	} else if (!strcasecmp(name, "ftp")) {
	    rpm2html_ftp = strdup(value);
	} else if (!strcasecmp(name, "ftpsrc")) {
	    rpm2html_ftpsrc = strdup(value);
	} else if (!strcasecmp(name, "name")) {
	    rpm2html_name = strdup(value);
	} else if (!strcasecmp(name, "host")) {
	    rpm2html_host = strdup(value);
	} else if (!strcasecmp(name, "header")) {
	    addHeader(value);
        } else {
	    printf("Config file : %s global entry ignored\n", name);
	}
	return;
    }

    /*
     * option for a directory.
     */
    cur = rpmDirSearch(rpmdir);
    if (!strcasecmp(name, "name")) {
	cur->name = strdup(value);
    } else if (!strcasecmp(name, "dir")) {
	cur->dir = strdup(value);
    } else if (!strcasecmp(name, "url")) {
	cur->url = strdup(value);
    } else if (!strcasecmp(name, "ftp")) {
	cur->ftp = strdup(value);
    } else if (!strcasecmp(name, "ftpsrc")) {
	cur->ftpsrc = strdup(value);
    } else if (!strcasecmp(name, "color")) {
	if (cur->color != NULL) free(cur->color);
	cur->color = strdup(value);
    } else if (!strcasecmp(name, "trust")) {
	if (cur->trust != NULL) free(cur->color);
	cur->trust = strdup(value);
    } else if (!strcasecmp(name, "host")) {
	rpm2html_host = strdup(value);
    } else if (!strcasecmp(name, "mirror")) {
        /*
	 * all "mirrors" values are collected in the mirrors array.
	 */
	if (cur->nb_mirrors >= MAX_URLS) {
	    fprintf(stderr, "increase MAX_URLS %d overflow\n", MAX_URLS);
	} else {
	    cur->mirrors[cur->nb_mirrors] = strdup(value);
	    cur->nb_mirrors++;
	}
    } else {
	printf("Config file : %s entry for %s ignored\n", name, rpmdir);
    }
}

/****************************************************************
 *								*
 *		The configuration file parser			*
 *								*
 ****************************************************************/

/*
 * A few macro needed to help building the parser
 */

#define IS_BLANK(ptr) \
     (((*(ptr)) == ' ') || ((*(ptr)) == '\b') || \
      ((*(ptr)) == '\n') || ((*(ptr)) == '\r'))
#define SKIP_BLANK(ptr) \
     { while (((*(ptr)) == ' ') || ((*(ptr)) == '\b') || \
              ((*(ptr)) == '\n') || ((*(ptr)) == '\r')) ptr++; }
#define GOTO_EQL(ptr) \
     { while (((*(ptr)) != '\0') && ((*(ptr)) != '=') && \
              ((*(ptr)) != '\n') && ((*(ptr)) != '\r')) ptr++; }
#define GOTO_EOL(ptr) \
     { while (((*(ptr)) != '\0') && \
              ((*(ptr)) != '\n') && ((*(ptr)) != '\r')) ptr++; }

/*
 * addHeader : parse an Header entry, we expect the first
 *             part to be an URL and the end of the line is the name.
 *
 * e.g: "help.html Get Help"
 */
void addHeader(char *value) {
    char *url;
    char *name;

    /*
     * Check for more room in the header arrays.
     * Yes I'm lazy ...
     */
    if (rpm2html_nb_extra_headers >= MAX_EXTRA_HEADERS) {
        fprintf(stderr, "Too many headers, increase MAX_EXTRA_HEADERS\n");
	return;
    }

    /*
     * Check the url, parse until finding a blank.
     */
    name = value;
    while (!IS_BLANK(name)) name++;
    url = strndup(value, name - value);
    if (url == NULL) {
        fprintf(stderr, "strndup \"%s\" failed\n", value);
	exit(1);
    }
    SKIP_BLANK(name);
    if (*name == '\0') {
        fprintf(stderr, "Config file : expecting \"header URL description\"\n");
        fprintf(stderr, "\tfound \"header: %s\", ignored\n", value);
	free(url);
	return;
    }

    /*
     * Store the values in global variables.
     */
    rpm2html_headers_name[rpm2html_nb_extra_headers] = strdup(name);
    rpm2html_headers_url[rpm2html_nb_extra_headers] = url;
    rpm2html_nb_extra_headers++;
}


/*
 * read config file: parse a configuration file.
 */
int readConfigFile(char *filename)
{
   FILE *input;
   char *str, *base;
   char string[1000];
   char rpmdir[1000] = "rpm2html";
   char *name;
   char *value;
   int errors = 0;

   rpm2html_host = rpm2html_rpm2html_thishost;
   input = fopen (filename, "r");
   if (input == NULL)
     {
	fprintf (stderr, "Cannot read config from %s :\n", filename);
	perror ("fopen failed");
	return -1;
     }

   while (1)
     {
	/*
	 * read one line in string buffer.
	 */
	if (fgets (&string[0], sizeof (string) - 1, input) == NULL)
	   break;

	str = &string[0];
	SKIP_BLANK (str)
	string[sizeof (string) - 1] = '\0';

	/*
	 * Comment starts with a semicolumn.
	 */
	if (*str == ';')
	   continue;
	if (*str == '\0')
	   continue;

	/*
	 * sections are indicated between brackets, e.g. [amaya]
	 */
	if (*str == '[')
	  {
	     str++;
	     SKIP_BLANK (str)
	     base = str;
	     while ((*str != '\0') && (*str != ']'))
		str++;
	     if (*str == '\0')
	       {
		  fprintf (stderr, "config file %s corrupted :\n\t\"%s\"\n",
			   filename, string);
		  break;
	       }
	     *str = '\0';
	     strcpy (&rpmdir[0], base);
	     rpmDirSearch(rpmdir);

	     if (verbose > 1)
		 fprintf (stderr, "readConfigFile section [%s]\n", rpmdir);

	     continue;
	  }

	/*
	 * entries have the following form :
	 *    name=value
	 */
	name = str;
	GOTO_EQL (str)
	if (*str != '=') {
	   errors++;
	   if (errors >= 30) {
	       fprintf (stderr, "config file %s seems invalid\n", filename);
	       break;
	   }
	   continue;
	}
	*str++ = '\0';
	SKIP_BLANK (str)
	value = str;
	GOTO_EOL (str)
	*str = '\0';
	addConfigEntry(rpmdir, name, value);
     }

   fclose (input);
   return(0);
}

/*
 * reinitialize the base setup.
 */
void reinitialize(void) {
    if (rpm2html_dir) free(rpm2html_dir);
    if (rpm2html_name) free(rpm2html_name);
    if (rpm2html_url) free(rpm2html_url);
    if (rpm2html_ftp) free(rpm2html_ftp);
    if (rpm2html_ftpsrc) free(rpm2html_ftpsrc);
    if ((rpm2html_rpm2html_name != RPM2HTML_NAME) &&
        (rpm2html_rpm2html_name != NULL))
        free(rpm2html_rpm2html_name);
    rpm2html_rpm2html_name = RPM2HTML_NAME;
    if ((rpm2html_rpm2html_ver != RPM2HTML_VER) &&
        (rpm2html_rpm2html_ver != NULL))
        free(rpm2html_rpm2html_ver);
    rpm2html_rpm2html_ver = RPM2HTML_VER;
    if ((rpm2html_rpm2html_url != RPM2HTML_URL) &&
        (rpm2html_rpm2html_url != NULL))
        free(rpm2html_rpm2html_url);
    rpm2html_rpm2html_url = RPM2HTML_URL;
    if ((rpm2html_maint != RPM2HTML_MAINT) && (rpm2html_maint != NULL))
        free(rpm2html_maint);
    rpm2html_maint = RPM2HTML_MAINT;
    if ((rpm2html_mail != RPM2HTML_MAIL) && (rpm2html_mail != NULL))
        free(rpm2html_mail);
    rpm2html_mail = RPM2HTML_MAIL;
    rpm2html_dir = NULL;
    rpm2html_name = NULL;
    rpm2html_url = NULL;
    rpm2html_ftp = NULL;
    rpm2html_ftpsrc = NULL;
    rpm2html_host = rpm2html_rpm2html_thishost;

    for (;rpm2html_nb_extra_headers > 0;) {
	rpm2html_nb_extra_headers--;
        free(rpm2html_headers_name[rpm2html_nb_extra_headers]);
	rpm2html_headers_name[rpm2html_nb_extra_headers] = NULL;
	free(rpm2html_headers_url[rpm2html_nb_extra_headers]);
	rpm2html_headers_url[rpm2html_nb_extra_headers] = NULL;
    }
    rpm2html_nb_extra_headers = 0;

    rpmDirListFree(&dirList);
    rpmDataListFree(&rpmList);
    rpmDataListFree(&rpmInstalledList);
    rpmRessListFree(&ressList);
    rpmRessListFree(&ressInstalledList);
    dirList = NULL;
}

