/*
   parsehtml.c - display code for html

 ParseHTML() parses the HTML elements and generates a stream of commands for
 displaying the document in the Paint buffer. The paint commands specify the
 appearence of the document as a sequence of text, lines, and images. Each
 command includes the position as a pixel offset from the start of the
 document. This makes it easy to scroll efficiently in either direction.
 The paint buffer must be freed and recreated if the window is resized.

 The model needs to switch to relative offsets to enable the transition
 to an wysiwyg editor for html+. Relative values for pixel offsets and
 pointers to the html+ source would make it much easier to edit documents
 as it would limit revisions to the paint stream to the region changed.

*/

/*

 Janne Saarela
 janne.saarela@hut.fi
 28.7.1995

 PrintSeqText() and PutText() now use Put*() to place
 the emph attribute to the PaintStream. The function prototypes
 now have emph added as unsigned int.

*/


#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "arena.h"
#include "defs.h"
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
#  include "debug.h"
#endif
#include "bridge.h"
#include "display.h"
#include "entities.h"
#include "error.h"
#include "forms.h"
#include "html.h"
#include "image.h"
#include "main.h"
#include "math_arena.h"
#include "parsehtml.h"
#include "parsehtml_i.h"
#include "status.h"
#include "style.h"
#include "util.h"
#include "x11.h"


#define LBUFSIZE 1024


/* The current top line is displayed at the top of the window,
 * the pixel offset is the number of pixels from the start of the document.
 */

char *bufptr;             /* parse position in the HTML buffer */
char *lastbufptr;         /* keep track of last position to store delta's */

Byte* paint = NULL;              /* holds the sequence of paint commands */
static int paintbufsize = 8192;  /* size of buffer, not its contents */
static int paintlen = 0;         /* where to add next entry */

int paintStartLine;       /* where line starts in the paint stream */
int above;                /* above baseline */
int below;                /* below baseline */
int voffset;              /* if positive then above baseline */
int IdAttributeFlag;      /* signals attribute is ID value */

int ParseHTMLerror = 0;   /* set by parser (initialize to force into .data) */
Bool prepass;             /* true during table prepass */
Bool damn_table = False;  /* to debug table formatter */
int html_width;           /* tracks maximum width */
int min_width, max_width; /* table cell width */
int list_indent;

int preformatted;

static int EndTag, TagLen;
static int TokenClass, TokenValue, Token;

int baseline;       /* from top of line */
long TermTop, TermBottom;

long PixOffset  = 0;  /* current offset from start of document */
long PrevOffset = 0;  /* keep track for saving delta's */
long LastLIoffset;    /* kludge for <LI><LI> line spacing */

int Here, TextLineWidth;
int HTMLInit = 0;
Image *start_figure, *figure;
long figEnd;
Form* form;
Bool prealign = False;		/* Global indicator for <CENTER> </CENTER> */
int isindex_state = 0;          /* <ISINDEX> 0=no where, 1=header, 2=body  */
Bool DocIsIndex = False;

int LeftFlowMargin, RightFlowMargin;
int LeftMarginIndent = 0, RightMarginIndent = 0;
long FigureEnd = 0;

int class_len = 0;
char *class = NULL;
/* Bool initial_cap = False;*/

char *LastBufPtr, *StartOfLine, *StartOfWord; /* in HTML document */
static int LineLen, LineWidth, WordStart, WordWidth;
static char LineBuf[LBUFSIZE]; /* line buffer */

static char *Ones[] = {"i", "ii", "iii", "iv", "v", "vi", "vii", "viii", "ix"};
static char *Tens[] = {"x", "xx", "xxx", "xl", "l", "lx", "lxx", "lxxx", "xc"};
static char *Hundreds[] = {"c", "cc", "ccc", "cd", "d", "dc", "dcc", "dccc", "cm"};

char small_caps[256] = {   0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 
                          16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
                          32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
                          48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
                          64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
                          80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95,
                          96, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
                          80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90,123,124,125,126,127,
                         128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
                         144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
                         160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
                         176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
                         192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,
                         208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,
                         192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,
                         208,209,210,211,212,213,214,247,216,217,218,219,220,221,222,255 }; 


int  LeftMargin(Frame* frame, int  left, int voffset);
int RightMargin(Frame* frame, int right, int voffset);


/*
 * Allocate pain stream buffer for the first time, or reset it.
 */
void InitPaintStream(void)
{
 if (paint)
   memset(paint, 0, paintlen);
  else
   paint = (Byte*)Arena_CAlloc(paintbufsize, sizeof(char), True);

 paintlen = 0;
}


/*
 * Expand paint stream to fit `length' bytes.
 * If length is zero, then just return current position in paint stream.
 */
Byte* MakeRoom(int length)
{
 if (length > 0)
   {
    Byte* p;

    if (paintlen > paintbufsize - length)
      {
       paintbufsize *= 2;
       paint = (Byte*)realloc(paint, paintbufsize);
      }

    p = paint + paintlen;
    paintlen += length;
    return p;
   }
  else
   return (paint + paintlen);
}


/*
 * Get current position in paint stream.
 */
Byte* PaintStreamCurrentPosition(void)
{
 return MakeRoom(0);
}


/*
 * Fill in begin-frame marker fields.
 * The marker position in global paint stream is computed from
 * frame->info value.
 */
void SetBeginFrameParameters(Frame* frame)
{
 if (frame)
   {
    Byte* paint_p = paint + frame->info;
#ifdef ARENA_DEBUG
    char Iam[] = "SetBeginFrameParameters";
#endif

    if (*paint_p == BEGIN_FRAME)
      {
       Byte* p = paint_p + 1;

       PutLong(p, frame->offset);
       PutInt( p, frame->indent);
       PutInt( p, frame->width);
       PutLong(p, frame->height);
       *p++ = frame->style;   /* frame's background style */
       *p++ = frame->border;  /* frame's border style */
#ifdef STYLE_COLOUR_BORDER
       *p++ = frame->cb_ix;    /* frame's foreground colour index */
#endif
#ifdef STYLE_BACKGROUND
       PutPointer(p, (void*)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
#endif
       PutInt(p, 0);        /* subsequently filled in with length */

#ifdef ARENA_DEBUG
       if (PARSEHTML_TRACE && VERBOSE_TRACE)
	 {
	  Arena_TracePrint(Iam,
			   " p = "POINTER_FORMAT", info = %d,\n"
			   "\tp_end = "POINTER_FORMAT", offset = %d.\n",
			   paint_p, frame->info, p, frame->offset);
	 }
#endif
      }
#ifdef ARENA_DEBUG
     else
      {
       Arena_TracePrint(Iam,
			" ERROR! "POINTER_FORMAT" (of "POINTER_FORMAT")"
			" does NOT begin frame.\n",
			paint_p, frame);
      }
#endif
   }
}


#if 1	/* QingLong.17-10-97 */
/*
 * Fill in frame height field of begin-frame marker.
 * The marker position in global paint stream is computed from
 * frame->info value.
 */
void SetBeginFrameHeight(Frame* frame)
{
 if (frame)
   {
    Byte* paint_p = paint + frame->info;
#ifdef ARENA_DEBUG
    char Iam[] = "SetBeginFrameHeight";
#endif

    if (*paint_p == BEGIN_FRAME)
      {
       Byte* p = paint_p +
            /* offset to (long)height in BEGIN_FRAME */
           (ARENA_PAINTSTREAM_TAG_LEN +
            ARENA_LONG_INT_SIZE + ARENA_INT_SIZE + ARENA_INT_SIZE);
       PutLong(p, frame->height);

#ifdef ARENA_DEBUG
       if (PARSEHTML_TRACE && VERBOSE_TRACE)
	 Arena_TracePrint(Iam,
			  " p = "POINTER_FORMAT", info = %d, height = %d.\n",
			  paint_p, frame->info, frame->height);
#endif
      }
#ifdef ARENA_DEBUG
     else
      {
       Arena_TracePrint(Iam,
			" ERROR! "POINTER_FORMAT" (of "POINTER_FORMAT")"
			" does NOT begin frame.\n",
			paint_p, frame);
      }
#endif
   }
}
#endif


unsigned int PrintBeginFrame(Frame* theFrame)
{
 unsigned int info = 0;

 if (!prepass)
   {
    Byte* p = MakeRoom(FRAMESTLEN);
#ifdef ARENA_DEBUG
    char Iam[] = "PrintBeginFrame";
#endif

    info = p - paint;
    *p = BEGIN_FRAME;
#ifdef ARENA_DEBUG
    if (PARSEHTML_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam, " "POINTER_FORMAT".\n", p);
#endif
    if (theFrame)
      {
       theFrame->info = info;
       SetBeginFrameParameters(theFrame);
      }
   }

 return info;
}


/*
 * The frame content length field.
 */
void PrintFrameLength(Frame* frame, Byte* paintstream_cur_pos)
{
 if (!prepass && (frame))
   {
    Byte* p0 = paint + frame->info;
                          /* offset to `length' field in BEGIN_FRAME */
    Byte* plength = p0 + (FRAMESTLEN - ARENA_INT_SIZE);
#ifdef ARENA_DEBUG
    char Iam[] = "PrintFrameLength";
#endif

    /* calculate the frame `length' field */
    frame->length = paintstream_cur_pos - p0 - FRAMESTLEN;

    /* write the frame `length' field in the frame's header */
    PutInt(plength, frame->length);

#ifdef ARENA_DEBUG
    if (PARSEHTML_TRACE && VERBOSE_TRACE)
      {
       Arena_TracePrint(Iam,
			" p = "POINTER_FORMAT", info = %d,\n"
			"\tlength = %d,   offset = %d.\n",
			p0, frame->info, frame->length,	frame->offset);
      }
#endif
   }
}


/*
 * The size field after end of frame contents.
 */
void PrintFrameSize(Frame* frame)
{
 if (!prepass && (frame))
   {
    Byte* p0 = paint + frame->info;
    Byte* psize = MakeRoom(ARENA_INT_SIZE);
    unsigned int size;
#ifdef ARENA_DEBUG
    char Iam[] = "PrintFrameSize";
#endif

    /* calculate the frame `size' field */
    size = psize - p0;

    /* write the frame `length' field in the frame's header */
    PrintFrameLength(frame, psize);

    /* write the size field after frame's contents */
    PutInt(psize, size);

#ifdef ARENA_DEBUG
    if (PARSEHTML_TRACE && VERBOSE_TRACE)
      {
       Arena_TracePrint(Iam,
			" p = "POINTER_FORMAT", info = %d,\n"
			"\tp_end = "POINTER_FORMAT", size = %d.\n",
			p0, frame->info, psize, size);
      }
#endif
   }
}


/*
 * marker for pixel offset to end of frame
 */
void PrintEndFrame(Frame* parent, Frame* frame)
{
 if (!prepass && frame)
   {
#ifdef ARENA_DEBUG
    char Iam[] = "PrintEndFrame";
#endif

    {
     Byte* p;
     unsigned int frame_start_offset; /* backward offset to the correspondent
					 BEGIN_FRAME */


     p = MakeRoom(FRAMENDLEN + ARENA_INT_SIZE);
     frame_start_offset = p - (paint + frame->info);
#ifdef ARENA_DEBUG
   /* QingLong.16-10-97: Sanity check. */
     if (PARSEHTML_TRACE && VERBOSE_TRACE)
       Arena_TracePrint(Iam,
			" "POINTER_FORMAT": "POINTER_FORMAT
			"->info = 0x%x, back offset to begin = %d.\n",
			p, frame, frame->info, frame_start_offset);
#endif
     *p++ = END_FRAME;
     PutInt(p, frame_start_offset);
     PutInt(p, FRAMENDLEN);
    }

    /* increase width of parent frame if necessary */
    if (parent)
      {
       unsigned int length;

       length = frame->indent + frame->width + 2;

       if (parent->width < length) parent->width = length;

       /* and restore parent frame margins if this frame
	  was involved in a text flow indent and is not
	  obstructed by a subsequent text flow frame */

       /* howcome 7/10/94: prolonged the "if parent" block */

       if (frame->flow == ALIGN_LEFT)
	 {
	  if (frame->pushcount == parent->leftcount)
	    {
	     parent->leftcount -= 1;
	     parent->leftmargin = frame->oldmargin;
	    }
	 }
        else
	 {
	  if (frame->flow == ALIGN_RIGHT)
	    {
	     if (frame->pushcount == parent->rightcount)
	       {
	        parent->rightcount -= 1;
	        parent->rightmargin = frame->oldmargin;
	       }
	    }
	   else
	    {
	     if (frame->flow == ALIGN_NOTE)
	       if (frame->pushcount == parent->leftcount)
	         parent->leftcount -= 1;
	    }
	 }
      }
   }
}


/* Write end markers for all peer frames and children.
 * Note that the lists are flushed in the same order
 * that items were pushed onto the list
 */
void FlushAllFrames(Frame* parent, Frame* frame)
{
 if (frame)
   {
    FlushAllFrames(parent, frame->next);
    frame->next = NULL;
    FlushAllFrames(frame, frame->child);
    frame->child = NULL;
    PrintEndFrame(parent, frame);
    FreeFrames(frame);
    frame = NULL;
   }
}


/*
 * Write end markers for any frames in peer list which have a
 * finishing offset <= PixOffset. For any such frames, all
 * descendant frames are flushed first. The process frees
 * frames and removes them from the list of peers.
 *
 * If frame is the current frame then this procedure
 * should be invoked as:
 *
 *   frame->child = FlushFrames(frame, frame->child);
 *
 * This procedure assumes that BeginFrame() pushes
 * new frames onto the front of the list of children,
 * and guarantees that frames with the same offset are
 * flushed to the paint stream in the order they were
 * created. This property is needed for display purposes.
 */
Frame* FlushFrames(Frame* parent, Frame* frame)
{
 Frame* next;
#ifdef ARENA_DEBUG
 char Iam[] = "FlushFrames";
#endif


 if (frame)
   {
    next = FlushFrames(parent, frame->next);

    if (frame->offset <= PixOffset)
      {
      /* first flush frame's children */
       FlushAllFrames(frame, frame->child);
       frame->child = NULL;

       /* and now the frame itself */
       PrintEndFrame(parent, frame);
       Free(frame);
       frame = NULL;
       return next;
      }

#ifdef ARENA_DEBUG
    if (next == frame)
      {
       Arena_TracePrint(Iam,
			" ERROR! Frames deadloop at "POINTER_FORMAT".\n",
			frame);
       next = NULL;
      }
#endif

    frame->next = next;
   }

 return frame;
}


void FlushPending(Frame* frame)
{
 if (frame && frame->child) frame->child = FlushFrames(frame, frame->child);
}


/* 
 * The frame is created here, the new frame is returned so that
 * the parser can later call EndFrame() at the end of the frame.
 * Any frames which end before PixOffset are first flushed.
 */
Frame* BeginFrame(Frame* parent, int style, int border, int left, int right,
		  BG_Style* bg)
{
 Frame* frame;


 FlushPending(parent);

 /* create frame and write begin frame marker */
 frame = (Frame*)Arena_CAlloc(1, sizeof(Frame), False);
 frame->offset = PixOffset;
 frame->indent = (parent ?  parent->leftmargin + left : left);
 frame->width  = right - left;
 frame->style  = style;
 frame->border = border;
#ifdef STYLE_COLOUR_BORDER
 frame->cb_ix  = (int)bg;
# else
 frame->cb_ix  = 0;
#endif
 frame->flow   = ALIGN_CENTER; /* implies this is a figure frame */
 frame->next   = frame->child = NULL;
 frame->box_list = NULL;
 PrintBeginFrame(frame);
 return frame;
}


/*
 * This writes the frame's height in the frame's header.
 * Here we need to push the frame onto the front of the
 * list of the parent frame's children so that FlushFrames()
 * can write the end frame marker to the paint queue
 */
void EndFrame(Frame* parent, Frame* frame)
{
#ifdef ARENA_DEBUG
 char Iam[] = "EndFrame";
#endif


 /* update background.height if needed */
 if ((PixOffset - background.offset) > background.height)
   (background.height = PixOffset) ;

 /* write height into paint struct for frame */

 frame->height = PixOffset - frame->offset;
 SetBeginFrameHeight(frame);

 /* change frame->offset to end of frame */
 frame->offset = PixOffset;

#ifdef ARENA_DEBUG
 if (parent->child == frame)
   {
    Arena_TracePrint(Iam,
		     " ERROR! Frames deadloop at "POINTER_FORMAT".\n",
		     frame);
    parent->child = NULL;
   }
#endif

 /* and now push onto list of children */
 frame->next = parent->child;
 parent->child = frame;
}


int ListCells(Frame* cells)
{
 int n;

 for (n = 0; cells != NULL; ++n)
   {
    printf("address = "POINTER_FORMAT","
           "indent = %d, width = %d,height = %ld\n",
           cells, cells->indent, cells->width, cells->height);
    cells = cells ->next;
   }

 return n;
}


/*
 * Insert cell at end of list of cells
 */
void InsertCell(Frame** cells, Frame* cell)
{
 Frame *frame, *next;

 frame = *cells;
 cell->next = NULL;

 if (frame == NULL)
   *cells = cell;
  else
   {
    for (frame = *cells; ; )
      {
       next = frame->next;

       if (next == NULL)
	 {
	  frame->next = cell;
	  break;
	 }

       frame = next;
      }
   }
}


/*
 * This routine adjusts height of all table cells which end
 * on this row and then calls EndFrame() to move them to
 * the list of frames awaiting PrintEndFrame()
 */
void FlushCells(Frame* parent, int row, Frame** cells)
{
 Frame *prev, *frame, *next;

 prev = NULL;
 frame = *cells;

 while (frame)
   {
    if (frame->lastrow <= row)
      {
       next = frame->next;

       if (prev)
	 prev->next = next;
        else
	 *cells = next;

       frame->height = PixOffset - frame->offset;
       frame->next = NULL;
       EndFrame(parent, frame);
       frame = next;
       continue;
      }

    prev = frame;
    frame = frame->next;
   }
}


/*
 * insert TEXTLINE container
 */
void TextLineFrame(Frame* frame)
{
 if (prepass)
   {
    paintStartLine = 0;
    TextLineWidth = 0;
   }
  else
   {
    Byte* p;
#ifdef ARENA_DEBUG
    char Iam[] ="TextLineFrame";
#endif

    FlushPending(frame);
    TextLineWidth = 0;
    p = MakeRoom(TXTLINLEN);
#ifdef ARENA_DEBUG
    if (PixOffset < 0)
      if (PARSEHTML_TRACE && VERBOSE_TRACE)
	Arena_TracePrint(Iam,
			 " WEIRD? PixOffset = %d at "POINTER_FORMAT".\n",
			 PixOffset, p);
#endif
    paintStartLine = p - paint;
    *p++ = TEXTLINE;

    PutLong(p, PixOffset);

    /* baseline, indent, and height are set at end of line by EndOfLine() */
    PutInt(p, 0);
    PutInt(p, 0);
    PutInt(p, 0);
   }
}


/*
 * This procedure writes the end of text line element and the baseline,
 * line height and indent as appropriate for the frame's
 * current margin settings.
 * This handles horizontal alignment in the face of changing margins.
 */
void EndOfLine(Frame* frame, int align)
{
 unsigned int n, height;
 int indent, delta = 0, len;
 Byte* p;

 if (!prepass) StyleSetFlag(CurrentDoc, S_LEADING_FLAG, False);

 if (paintStartLine >= 0)
   {
   /* fill in baseline for current line */

    if (!prepass)
      {
       if (frame == &background)
	 {
	  int w;   /* Although `w' is ``width'' let it be `signed'
		    * to avoid errors in calcualtions...
		    */

	  w = (int)background.width - background.rightmargin -
	            (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_RIGHT);  /* howcome 26/2/95 */

	  {
	   unsigned int docdispwin_width = GetDocDispWinWidth() - 4;

	   if (w > docdispwin_width) w = docdispwin_width;
	  }

	  w -= background.leftmargin;

	  if (align == ALIGN_LEFT)
	    delta = 0;
	   else
	    if (align == ALIGN_CENTER)
	      delta = (w - TextLineWidth) / 2;
	     else
	      if (align == ALIGN_RIGHT)
		delta = w - TextLineWidth;
	       else
		if (align == ALIGN_JUSTIFY)
		  delta = 0;
	 }
        else
	 {
	  if (align == ALIGN_LEFT)
	    delta = 0;
	   else
	    if (align == ALIGN_CENTER)
	      delta = ((int)frame->width - TextLineWidth
		       /* - (frame->leftmargin + frame->rightmargin) */) / 2;
	     else
	      if (align == ALIGN_RIGHT)
		delta = (int)frame->width - TextLineWidth
		        /* - (frame->leftmargin + frame->rightmargin) */;
	       else
		if (align == ALIGN_JUSTIFY)
		  delta = 0;
	 }

       indent = (delta > 0 ? frame->leftmargin + delta : frame->leftmargin);
       height = above + below;

       p = paint + paintStartLine +
	   /* offset to baseline in TEXTLINE */
	   (ARENA_PAINTSTREAM_TAG_LEN + ARENA_LONG_INT_SIZE);

       PutInt(p, above);
       PutInt(p, indent);
       PutInt(p, height);

       p = MakeRoom(1 + ARENA_INT_SIZE);
       *p++ = '\0';  /* push end of elements marker */

       /* and write text line frame length */
       n = p - paint - paintStartLine;
       PutInt(p, n);
      }

    if (frame->flow == ALIGN_JUSTIFY) 
      len = LineWidth + frame->leftmargin;
     else
      len = TextLineWidth + frame->leftmargin; 

    if (len > frame->width) frame->width = len;

    PixOffset += above + below;
    paintStartLine = -1;
    above = 0;
    below = 0;
    TextLineWidth = 0;
#if 0
    EndFigure();
#endif
   }

#if 0
 if (start_figure) PrintFigure(BEGIN_FIG);
#endif
}


/*
 * push horizontal rule onto paint stream
 */
void PrintRule(Frame* frame, int type, int left, int right, int dy)
{
 Byte* p;

 if (paintStartLine < 0) TextLineFrame(frame);

 if (type == HLINE)
   {
    if (prepass) return;

    above = max(above, 2);
    below = max(below, 2);
   }
  else
   {
    above = max(above, (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_TOP));
    below = max(below, (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_BOTTOM));

    if (prepass) return;
   }

 p = MakeRoom(RULEFLEN);

 *p++ = (RULE | type);
 PutInt(p,  left);
 PutInt(p, right);
 PutInt(p, dy);
#ifdef STYLE_COLOUR
 *p++ = (Byte)StyleGet(CurrentDoc, ArenaSI_COLOUR);
#endif
#ifdef STYLE_BACKGROUND
 /* must be set to the structures defined by the style
  * --Spif 18-Oct-95
  */
 PutPointer(p, (void *)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
#endif
}


     /* staalesc 13/12/95 */
void PrintLine(Frame* frame, int left, int right, int top, int bottom)
{
 Byte* p;

 if (paintStartLine < 0) TextLineFrame(frame);

 if (prepass) return;

 p = MakeRoom(LINEFLEN);

 *p++ = (LINE);
 PutInt(p,   left);
 PutInt(p,  right);
 PutInt(p,    top);
 PutInt(p, bottom);
#ifdef STYLE_COLOUR
 *p++ = (Byte) StyleGet(CurrentDoc, ArenaSI_COLOUR);
#endif
#ifdef STYLE_BACKGROUND
 PutPointer(p, (void *)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
#endif

 above = max(above, 2); /* staalesc: Are these correct??? */
 below = max(below, 2);
}


/*
 * push bullet onto paint stream
 */
void PrintBullet(Frame* frame, int depth, int font, int size)
{
 Byte* p;

 if (paintStartLine < 0) TextLineFrame(frame);

 if (Here + BULLET_WIDTH > TextLineWidth) TextLineWidth = Here + BULLET_WIDTH;

 above = max(above,  ASCENT(font));
 below = max(below, DESCENT(font));

 if (!prepass)
   {
    p = MakeRoom(BULLETFLEN);
    *p++ = BULLET;
    PutInt(p, Here);
    PutInt(p, depth);

    /* howcome 25/4/95: adding vertical position of bullet item */
    *p++ = font; /* StyleGet(CurrentDoc, ArenaSI_FONT); */
    *p++ = size; /* StyleGet(CurrentDoc, ArenaSI_FONT_SIZE); */ /* spif 8/1/96: size... */

#ifdef STYLE_COLOUR
    *p++ = (Byte) StyleGet(CurrentDoc, ArenaSI_COLOUR);
#endif
#ifdef STYLE_BACKGROUND
    /* must be set to the structures defined by the style
     * --Spif 18-Oct-95
     */
    PutPointer(p, (void *)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
#endif
   }
}


void WrapImageIfNeeded(Frame* frame, int align,
                       int left, int right, int width, int up, int down)
{
 if (preformatted)
   {
    if (  up > above) above = up;
    if (down > below) below = down;
   }
  else
   {
    if (Here > left &&
	Here + width + 2 > frame->width - frame->rightmargin - right)
      {
       EndOfLine(frame, align);

       Here = left;
       above = up;
       below = down;
      }
     else
      {
       if (  up > above) above = up;
       if (down > below) below = down;
      }
   }
}


Bool WrapFieldIfNeeded(Field* field, Frame* frame,
		       int align, int left, int right)
{
#ifdef ARENA_DEBUG
 char Iam[] = "WrapFieldIfNeeded";
#endif

 if (preformatted)
   {
#ifdef ARENA_DEBUG
    if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam, " Flag ``preformatted'' is set!\n");
#endif

    if (above < field->above) above = field->above;

    if (below < field->height - field->above)
      below = field->height - field->above;
   }
  else
   {
    if ((Here > left) &&
	(field->x + field->width > RightMargin(frame, right, frame->offset)))
      {
       EndOfLine(frame, align);
       field->x = Here = left;
       above = field->above;
       below = field->height - above;

       return True;
      }
     else
      {
       if (above < field->above) above = field->above;

       if (below < field->height - field->above)
	 below = field->height - above;
      }
   }

 return False;
}


/*
 * push text input field
 */
void PrintInputField(Frame* frame, Field* field)
{
 Byte* p;
#ifdef ARENA_DEBUG
 char Iam[] = "PrintInputField";
#endif


 if (paintStartLine < 0) TextLineFrame(frame);

 if (field->x + field->width > TextLineWidth)
   TextLineWidth = field->x + field->width;

 if (!prepass)
   {
    p = MakeRoom(INPUTFLEN);
    field->object = p - paint;
#ifdef ARENA_DEBUG
    if (PAINTSTREAM_TRACE)
      Arena_TracePrint(Iam,
		       "\n\tPut INPUT field "POINTER_FORMAT
		       " frame ("POINTER_FORMAT") to paintstream"
		       " at "POINTER_FORMAT".\n",
		       field, frame, p);
#endif
    *p++ = INPUT;
    PutPointer(p, (void*)field);
#ifdef STYLE_BACKGROUND
    /* must be set to the structures defined by the style --Spif 18-Oct-95 */
    PutPointer(p, (void*)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
#endif
   }
}


/*
 * push normal or preformatted string.
 */                                /* emph contains font and emphasis */
void RealPrintString(Frame* frame, unsigned int emph, int font,
		     int delta, char *ref, char *buf, int len, int width,
		     Bool tight_font, Byte text_colour,
		     BG_Style *text_background)
{
 unsigned char *p = NULL;
#if 0
 int indent;

 indent = Here + LeftMarginIndent;   /* figures on LHS */
 indent = Here + style->margin_left; /* howcome 21/2/95 */
#endif

 if (paintStartLine < 0) TextLineFrame(frame);

 if (!prepass) p = MakeRoom(STRINGFLEN);

 if (Here + width > TextLineWidth) TextLineWidth = Here + width;


 /* In the case of a big initial, the descent should be calculated
    specifically for the character in question */
 if (tight_font)
   {
    above = max(above, Fonts[font]->per_char[(unsigned char)(*buf)].ascent);
    /* if (above == 0) above = max(above, ASCENT(font)); */
    below = max(below, Fonts[font]->per_char[(unsigned char)(*buf)].descent);
    /* if (below == 0) below = max(below, DESCENT(font)); */
    }
  else
   {
    if (!StyleGetFlag(CurrentDoc, S_LEADING_FLAG))
#if 1
      above = max(above, ASCENT(font) + (int)StyleGet(CurrentDoc, ArenaSI_LINE_HEIGHT));
# else
      above = (int)StyleGet(CurrentDoc, ArenaSI_LINE_HEIGHT);
#endif
     else
      above = max(above, ASCENT(font));

    /* howcome 28/8/95: is this the right place to add leading? */
    below = max(below, DESCENT(font));
   } /* --Spif 13-Nov-95 this was not the right place ;) */

 if (!prepass)
   {
    *p++ = (preformatted ? (STRING | PRE_TEXT) : STRING);
    PutInt(p, emph);
    *p++ = font;

    if (StyleGetFlag(CurrentDoc, S_INDENT_FLAG))
      {
       /* Here += StyleGet(CurrentDoc, S_INDENT); */

       /* flag carrying a value... hum... --Spif 14-Nov-95 */
       Here += StyleGetFlag(CurrentDoc, S_INDENT_FLAG);
       StyleSetFlag(CurrentDoc, S_INDENT_FLAG,False);
      };
#if 0
    /* this is the right place for this, but it doesn't work... yet ;)
       --Spif 16-Jan-96 */
    if (StyleGetFlag(CurrentDoc, S_MARGIN_TOP_FLAG))
      {
       PixOffset += StyleGetFlag(CurrentDoc, S_MARGIN_TOP_FLAG);
       StyleSetFlag(CurrentDoc, S_MARGIN_TOP_FLAG, False);
      };
#endif
#ifdef STYLE_COLOUR
    *p++ = text_colour;
#endif
#ifdef STYLE_BACKGROUND
    /* must be set to the structures defined by the style
       --Spif 18-Oct-95 */
    PutPointer(p, (void*)text_background);
#endif
    /* baseline offset 128 is zero, 255 is +127, 0 is -128 */
    *p++ = delta + 128;

    PutInt(p, Here);
    PutInt(p, len);
    PutInt(p, width);
    PutPointer(p, (void *)buf);
    PutPointer(p, (void *)ref);
   }
}


void PrintString(Frame* frame, unsigned int emph, int font,
		 int delta, char *buf, int len, int width, int wrapped)
{
 int i, j, k, fix, index, icw = 0;
 int sc_width, dumpint; /* --Spif 5/10/95 small caps fix */
 int OldTextLineWidth, OldHere; 
 Bool emph_set = False;
 Bool ok;
 char dump_char[2];


 fix = (int)StyleGet(CurrentDoc, ArenaSI_FONT);

 if ((int)StyleGet(CurrentDoc, ArenaSI_FONT_STYLE_SMALL_CAPS))
   {
    int sc_fix = /*(int)StyleGet(CurrentDoc, S_SMALL_CAPS_FONT);*/
                 GetFont((char *)StyleGet(CurrentDoc, ArenaSI_FONT_FAMILY),
			 ((int)StyleGet(CurrentDoc, ArenaSI_FONT_SIZE) * 4) / 5,
			 (int)StyleGet(CurrentDoc, ArenaSI_FONT_WEIGHT),
			 (int)StyleGet(CurrentDoc, ArenaSI_FONT_STYLE_SLANT),
			 True);

    for (i = k = 0; k < len; i++, k++)
      {
       sc_width = CompoundTextWidth(buf + i, 0, 1,&dumpint) +
	                                 (int)StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING);
       /* --Spif 6-Oct-95
	  keep this to update TextLineWidth with the right width */     
       OldTextLineWidth = TextLineWidth ;
       OldHere = Here;
       /* we must give the entire width if it is emphasized,
	* to tell the paint stream to display a beautiful box around the text,
	* but the real TextLineWidth must be calculated with the width of
	* only ONE character at a time
	*/
       if (buf[i] == '&')
	 {
	  for (ok = False, j = i; !ok && j < len; j++)
	    ok = ((buf[j] == ';') || (buf[j] == ' ')); /* ' ' is here for
						      dummy html writers ;) */
	  index = *dump_char = (char)entity(buf + i + 1, &j);
	  if (index < 0) index += 256; 
	  dump_char[1] = 0;
	  RealPrintString(frame, (emph_set ? 0 : emph), sc_fix, delta,
			  buf, &small_caps[index], 1, width, False,
			  /* how can we capitalize entities? */
			  (Byte) StyleGet(CurrentDoc, ArenaSI_COLOUR),
			  (BG_Style *)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
	  emph_set = True;
	  if (((index + small_caps[index]) % 256))
	    Here += XTextWidth(Fonts[sc_fix], dump_char,1) +
	                                 (int)StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING);
	   else
	    Here += XTextWidth(Fonts[fix], dump_char,1) +
	                                 (int)StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING);
	  i += j -1;
	 }
        else
	 {
	  int the_fix = ((islower(buf[i]) || isdigit(buf[i])) ? sc_fix : fix);

	  RealPrintString(frame, (emph_set ? 0 : emph),
			  the_fix,
			  delta, buf, &small_caps[(unsigned char)buf[i]], 1,
			  width, False,
			  (Byte) StyleGet(CurrentDoc, ArenaSI_COLOUR),
			  (BG_Style *)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
	  emph_set = True;
	  Here += XTextWidth(Fonts[the_fix],
			     &small_caps[(unsigned char)buf[i]], 1) +
	                                 (int)StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING);
	 }

       /* -- Spif: 6-Oct-95 small caps fix */ 
       if ( OldHere + sc_width > OldTextLineWidth)
	 TextLineWidth = OldHere + sc_width;
      };

    TextLineWidth -= (int)StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING);
    Here -= (int)StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING);
   }
  else
   if (StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING))
     {
      for(i = k = 0; k < len; i++, k++)
	{
	 /* a hack... rename this stuff */
	 sc_width = CompoundTextWidth(buf + i, 0, 1,&dumpint) +
	                                 (int)StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING);
	 OldTextLineWidth = TextLineWidth ;  
	 OldHere = Here;
	 /* we must give the entire width if it is emphasized,
	  * to tell the paint stream to display a beautiful box around
	  * the text, but the real TextLineWidth must be calculated with
	  * the width of only ONE character at a time.
	  */

	 if (buf[i] == '&')
	   {
	    for (ok = False, j = i; !ok && j < len; j++)
	      ok = ((buf[j] == ';') || (buf[j] == ' ')); /* ' ' is here for
						       dummy html writers ;) */

	    RealPrintString(frame, (emph_set ? 0 : emph), fix, delta,
			    buf, buf + i, 1, width, False,
			    /* how can we capitalize entities? */
			    (Byte) StyleGet(CurrentDoc, ArenaSI_COLOUR),
			    (BG_Style *)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
	    emph_set = True;
	    *dump_char = entity(buf + i + 1, &j);
	    dump_char[1] = 0;
	    Here += XTextWidth(Fonts[fix], dump_char,1) +
	                                 (int)StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING);
	    i += j -1;
	   }
	  else
	   {
	    RealPrintString(frame, (emph_set ? 0 : emph), fix, delta,
			    buf, buf + i, 1, width, False,
			    (Byte) StyleGet(CurrentDoc, ArenaSI_COLOUR),
			    (BG_Style *)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
	    emph_set = True;
	    Here += XTextWidth(Fonts[fix], buf + i, 1) +
	                                 (int)StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING);
	   }

	 if ( OldHere + sc_width > OldTextLineWidth)
	   TextLineWidth = OldHere + sc_width;
        };

      TextLineWidth -= (int)StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING);
      Here -= (int)StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING);
     }
    else
     {
      if (frame->flow == ALIGN_JUSTIFY)
        {
	 int nb_space = 0;
	 int remaining_space;


	 for (k = 0; k < len; k++)
	   nb_space += (buf[k] == ' ') || (buf[k] == '\n');

	 if (!nb_space || !wrapped)
	   {
	    RealPrintString(frame, (emph_set ? 0 : emph), fix, delta,
			    buf, buf, len, width, False,
			    (Byte)StyleGet(CurrentDoc, ArenaSI_COLOUR),
			    (BG_Style *)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
	   }
	  else
	   {
	    remaining_space = frame->width - width - Here -
	                                           StyleGetFlag(CurrentDoc, S_INDENT_FLAG);

	    for (i = k = 0; k < len; i++, k++)
	      {
	       if (buf[i] == '\n')
		 sc_width = XTextWidth(Fonts[fix], " ", 1) +
		                    (remaining_space/(nb_space ? nb_space :1));
	        else
		 sc_width = CompoundTextWidth(buf + i, 0, 1,&dumpint) +
		  (*(buf+i) == ' ')*(remaining_space/(nb_space ? nb_space :1));

	       OldTextLineWidth = TextLineWidth ;  
	       OldHere = Here;
	       /* we must give the entire width if it is emphasized,
		* to tell the paint stream to display a beautiful box around
		* the text, but the real TextLineWidth must be calculated with
		* the width of only ONE character at a time
		*/
	       if (buf[i] == '&')
		 {
		  for (ok = False, j = i; !ok && j < len; j++)
		    ok = ((buf[j] == ';') || (buf[j] == ' '));
		  /* ' ' is here for dummy html writers ;) */

		  RealPrintString(frame, (emph_set ? 0 : emph), fix, delta,
				  buf, buf + i, 1, width,
				  /* how can we capitalize entities? */
				  False,
				  (Byte)StyleGet(CurrentDoc, ArenaSI_COLOUR),
				  (BG_Style *)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
		  emph_set = True;
		  *dump_char = entity(buf+i+1, &j);
		  dump_char[1] = 0;
		  Here += XTextWidth(Fonts[fix], dump_char,1);
		  i += j -1;
		 } 
	        else
		 {
		  if (buf[i] == '\n')
		    {
		     emph_set = True;
		     Here += XTextWidth(Fonts[fix], " ", 1) +
		                    (remaining_space/(nb_space ? nb_space :1));
		     remaining_space -= (remaining_space/(nb_space ?
							  nb_space : 1));
		     nb_space--;
		    }
		   else
		    {
		     RealPrintString(frame, (emph_set ? 0 : emph), fix, delta,
				     buf, buf + i, 1, width, False,
				     (Byte)StyleGet(CurrentDoc, ArenaSI_COLOUR),
				     (BG_Style *)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
		     emph_set = True;
		     Here += XTextWidth(Fonts[fix], buf + i, 1) +
		       (*(buf+i) == ' ')*(remaining_space/(nb_space ?
							   nb_space : 1)); 
		     if (*(buf+i) == ' ')
		       {
			remaining_space -= (remaining_space/(nb_space ?
							     nb_space : 1));
			nb_space--;
		       }
		    }
		 }
	       if ( OldHere + sc_width > OldTextLineWidth)
		 TextLineWidth = OldHere + sc_width;
	      }
#if 0
	    RealPrintString(frame, (emph_set ? 0 : emph), fix, delta,
			    buf, buf, len, width, False,
			    (Byte)StyleGet(CurrentDoc, ArenaSI_COLOUR),
			    (BG_Style *)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
#endif
	   }
        }
       else
	RealPrintString(frame, (emph_set ? 0 : emph), fix, delta,
			buf, buf, len, width, False,
			(Byte)StyleGet(CurrentDoc, ArenaSI_COLOUR),
			(BG_Style *)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));

      emph_set = True;
      Here += (width - icw);
     }
}



int CompoundTextWidth(char *s, int start, int len, int *space_p)
{
 int width = 0, i, k, j, ok;
 int fix = (int)StyleGet(CurrentDoc, ArenaSI_FONT);
 char dump_char[2];


#if 0
 if (((int)StyleGet(CurrentDoc, S_TEXT_EFFECT) == TEXT_EFFECT_INITIAL_CAP) &&
     (StyleGetFlag(CurrentDoc, S_INITIAL_FLAG)))
   {
    int alt_fix = (int)StyleGet(CurrentDoc, S_ALT_FONT);

    width += XTextWidth(Fonts[alt_fix], s, 1);
    start++; len--;
   }
#endif

 if ((int)StyleGet(CurrentDoc, ArenaSI_FONT_STYLE_SMALL_CAPS))
   {
    int sc_fix = /*(int)StyleGet(CurrentDoc, S_SMALL_CAPS_FONT);*/
                 GetFont((char *)StyleGet(CurrentDoc, ArenaSI_FONT_FAMILY),
			 ((int)StyleGet(CurrentDoc, ArenaSI_FONT_SIZE) * 4) / 5,
			 (int)StyleGet(CurrentDoc, ArenaSI_FONT_WEIGHT),
			 (int)StyleGet(CurrentDoc, ArenaSI_FONT_STYLE_SLANT),
			 True);

    for (i = k = start; k < (len + start); i++, k++)
      {
       width += (int)StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING);
       if (islower(s[i]))
	 width += XTextWidth(Fonts[sc_fix],
			     &small_caps[(unsigned char)(s[i])], 1);
        else
	 if (s[i] == '&')
	   {
	    for (ok = False, j = i; !ok && j < (len+start); j++)
	      ok = ((s[j] == ';') || (s[j] == ' ')); 

	    *dump_char = entity(s+i+1, &j);
	    dump_char[1] = 0;
	    i += j-1;
	    width += XTextWidth(Fonts[sc_fix], dump_char, 1);
	   }
	  else
	   if (isdigit(s[i]))
	     {
	      width += XTextWidth(Fonts[sc_fix], s + i, 1);
	     }
	   else
	     width += XTextWidth(Fonts[fix], s + i, 1);
      }
    *space_p = XTextWidth(Fonts[sc_fix], " ", 1) +
                                         (int)StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING);
    return width;
   }
 else
   if (StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING))
     {
      for(i = k = start; k < (len + start); i++, k++)
	{
	 width += (int)StyleGet(CurrentDoc, ArenaSI_LETTER_SPACING);
	 if (s[i] =='&')
	   {
	    for (ok = False, j = i; !ok && j < (len+start); j++)
	      ok = ((s[j] == ';') || (s[j] == ' ')); 

	    *dump_char = entity(s+i+1, &j);
	    dump_char[1] = 0;
	    i += j-1;
	    width += XTextWidth(Fonts[fix], dump_char, 1);
	   }
	 else
	   width += XTextWidth(Fonts[fix], s + i, 1);
        }

      *space_p = XTextWidth(Fonts[fix], " ", 1) +
	        (int)StyleGet(CurrentDoc, ArenaSI_WORD_SPACING); /* WORD_SPACING instead */
      return width;
     }
    else
     {
      *space_p = XTextWidth(Fonts[fix], " ", 1);
      return (width + XTextWidth(Fonts[fix], s + start, len));
     }
}


/*
 * Push explicit text onto paint stream
 */
void PrintSeqText(Frame* frame, unsigned int emph, int font, char *s,
		  int width)
{
 Byte* p;
 int len;

 if (paintStartLine < 0) TextLineFrame(frame);

 if (!prepass)
   {
    len = Arena_StrLen(s);
    p = MakeRoom(SEQTEXTFLEN(len));

    if (Here + width > TextLineWidth) TextLineWidth = Here + width;

    above = max(above, ASCENT(font));
    below = max(below, DESCENT(font));

    *p++ = SEQTEXT;
    PutInt (p, emph);
    *p++ = font;
#ifdef STYLE_COLOUR
    *p++ = (Byte)StyleGet(CurrentDoc, ArenaSI_COLOUR);
#endif
#ifdef STYLE_BACKGROUND
    /* must be set to the structures defined by the style
     * --Spif 18-Oct-95
     */
    PutPointer(p, (void *)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
#endif
    PutInt(p, Here);
    PutInt(p, 0);
    *p++ = len; /* seqtext block length is stored in the first byte */
    memcpy(p, s, len);
   }
}


/*
 * Declare for PrintPhrase!
 */
void WrapIfNeeded(Frame* frame, int align, unsigned int emph, int font,
		  int left, int right);


/* Quick, and very dirty routine to print "directly" to the paint stream
 * a phrase (tokenized) without first knowing its length or width.
 * You can specify the length if it is not zString format.
 * This way you can get auto wrapping done when resizing is performed.
 */
void PrintPhrase(Frame* frame, int align, unsigned int emph, int font,
		 int left, int right,  char *s, int mlen)
{
 int nlen, width, spc_p;
 char *c, *e, *b, *o, *t;

 if (!s) return;
 if (mlen == -1) mlen = Arena_StrLen(s);
 if (!mlen) return;

 t = (char*)Arena_CAlloc(mlen + 1, sizeof(char), False);
 if (!t) return;

 b = s;
 e = s + mlen;
 while (*b != 0 && b < e)
   {
    c = b;
    o = t;
    while (*c != 0 && c < e && !isspace(*c))
      *o++ = *c++;                                /* blow though word!     */
      
    if (isspace(*c))
      while (*c != 0 && c < e && isspace(*c))
	*o++ = *c++;                              /* and following " "     */

    nlen = c - b;                                 /* put in paint stream   */
    *o++ = 0;
    width = CompoundTextWidth(t, 0, nlen, &spc_p);
    WrapIfNeeded(frame, align, emph, font, left, right);
    PrintSeqText(frame, emph, font, t, width);
    Here += width;
    b = c;
   }

 Free(t);
 return;
}


/*
 * for use by html-math parser
 */
int TextWidth(int font, char *str, int len, int *up, int *down)
{
 *up = Fonts[font]->max_bounds.ascent;
 *down = Fonts[font]->max_bounds.descent;
 return XTextWidth(Fonts[font], str, len);
}


/*
 * for use by html-math parser
 */
void FontSize(int font, int *ascent, int *descent)
{
 *ascent = Fonts[font]->max_bounds.ascent;
 *descent = Fonts[font]->max_bounds.descent;
}


box* CreateBox(int x, int y, int width, int height)
{
 box* new_box;


 if ((new_box = (box*)Arena_MAlloc(sizeof(box), False)))
   {
    new_box->x      = x;
    new_box->y      = y;
    new_box->width  = width;
    new_box->height = height;
   }
  else
   {
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
    Arena_TracePrint("CreateBox", " Ran Out of memory, exiting\n");
# else
    Arena_PrintError(_("Ran Out of memory, exiting\n"));
#endif
    Exit(1);
   }

 return new_box;
}


void AddBox(Frame* frame, int x, int y, int width, int height)
{
 box_link* new_link = NULL;

 if (frame == NULL) return;

#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
 if (VERBOSE_TRACE)
   Arena_TracePrint("AddBox", " (%d, %d) -> (%d, %d)\n", x, y, width, height); 
#endif

 new_link = (box_link*)Arena_MAlloc(sizeof(box_link), False);

 if (new_link)
   {
    new_link->box  = CreateBox(x, y, width, height);
    new_link->next = frame->box_list; /* we add the box in first position */
    frame->box_list = new_link;
   }
  else
   {
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
    Arena_TracePrint("AddBox", "Ran Out of memory, exiting\n");
# else
    Arena_PrintError(_("Ran Out of memory, exiting\n"));
#endif
    Exit(1);
   }
}


box* IsInBox(Frame* frame, int x, int y)
{
 if (frame)
   {
    box_link* curr_box;
    box_link* is_in;

    for (is_in = NULL, curr_box = frame->box_list;
	 !is_in && curr_box;
	 curr_box = curr_box->next)
      if ((x >= curr_box->box->x) &&
	  (x <= curr_box->box->x + curr_box->box->width) &&
	  (y >= curr_box->box->y) &&
	  (y <= curr_box->box->y + curr_box->box->height))
	is_in = curr_box;

    return (is_in) ? is_in->box : (box*)NULL;
   }
 else
   return NULL;
}


void CopyBoxList(Frame* dest_frame, Frame* frame)
{
 box_link* curr_box;

 if (!(dest_frame && frame)) return;

 for (curr_box = frame->box_list;curr_box; curr_box = curr_box->next)
   AddBox(dest_frame,
	  curr_box->box->x, curr_box->box->y,
	  curr_box->box->width, curr_box->box->height);
}


int LeftMargin(Frame* frame, int left, int voffset)
{
 int current_left;
 box* thebox;

 if (frame)
   {
    if (frame == &background)
      {
       current_left = left + background.indent + background.leftmargin;
       if (current_left < 3) current_left = 3;
      }
     else
      {
       current_left = left + frame->leftmargin;

       while ((thebox = IsInBox(frame, current_left, voffset)) != NULL)
	 {
	  current_left = thebox->x + thebox->width + 1;
	 }
      }
   }
  else
   {
    current_left = left;
   }

#ifdef ARENA_DEBUG
 if (PARSEHTML_TRACE && VERBOSE_TRACE)
   Arena_TracePrint("LeftMargin",
		    " %d"
		    "\n\t(frame = "POINTER_FORMAT","
		    " left = %d, voffset = %d)\n",
		    current_left, frame, left, voffset);
#endif

 return current_left;
}


int RightMargin(Frame* frame, int right, int voffset)
{
 int current_right;
 box* thebox;

 if (frame)
   {
    if ((voffset < 0) && (frame->offset >= 0)) voffset = frame->offset;

    if (frame == &background)
      {
       current_right = background.width - background.rightmargin - right;

       {
	unsigned int docdispwin_width = GetDocDispWinWidth() - 4;

	if (current_right > docdispwin_width) current_right = docdispwin_width;
       }

       current_right -= frame->leftmargin;
      }
     else
      {
       current_right = frame->width - right;
      }

    while ((thebox = IsInBox(frame, current_right, voffset)) != NULL)
      {
       current_right = thebox->x - 1;
      }
   }
  else
   {
    current_right = right;
   }

#ifdef ARENA_DEBUG
 if (PARSEHTML_TRACE && VERBOSE_TRACE)
   Arena_TracePrint("RightMargin",
		    " %d"
		    "\n\t(frame = "POINTER_FORMAT","
		    " right = %d, voffset = %d)\n",
		    current_right, frame, right, voffset);
#endif

 return current_right;
}


int Width(Frame* frame, int voffset)
{
 return frame->width;
}


void PutText(Frame* frame, unsigned int emph, int font, char *s, int len,
	     int x, int y)
{
 Byte* p;

 if (paintStartLine < 0) TextLineFrame(frame);

 if (!prepass)
   {
    p = MakeRoom(SEQTEXTFLEN(len));
    *p++ = SEQTEXT;
    PutInt (p, emph);
    *p++ = font;
#ifdef STYLE_COLOUR
    *p++ = (Byte)StyleGet(CurrentDoc, ArenaSI_COLOUR);
#endif
#ifdef STYLE_BACKGROUND
    /* must be set to the structures defined by the style
       --Spif 18-Oct-95 */
    PutPointer(p, (void*)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
#endif
    PutInt(p, x);
    PutInt(p, y);
    *p++ = len;
    memcpy(p, s, len);
   }
}


/*
 * buf points to start of element in html source iff ismap is present
 */
void PrintImage(Frame* frame, int delta, unsigned int emph,
                char *buf, Image* theImage,
		unsigned int width, unsigned int height)
{
 Byte* p;
 Pixmap thePixmap = default_pixmap, theMask = default_pixmap;

 if (theImage)
   {
    thePixmap = theImage->pixmap;
    theMask   = theImage->mask;
   }

 if (paintStartLine < 0) TextLineFrame(frame);

 if (Here + width > TextLineWidth) TextLineWidth = Here + width;

 if (prepass)  /* just update min/max widths */
   {
    if (width > min_width) min_width = width;
   }
  else
   {
    p = MakeRoom(IMAGEFLEN);

    *p++ = (IMAGE & 0xF) | (emph & (ISMAP | EMPH_ANCHOR | EMPH_INPUT));
    PutInt(p, delta);
    PutInt(p, Here);
    PutInt(p, width);
    PutInt(p, height);
    PutPixmap(p, thePixmap);
    PutPixmap(p, theMask);
    PutPointer(p, (void*)buf);
#ifdef STYLE_BACKGROUND
    /* must be set to the structures defined by the style
       --Spif 18-Oct-95 */
    PutPointer(p, (void*)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
#endif
   }
}


/*
 * check if current word forces word wrap and flush line as needed
 */
void WrapIfNeeded(Frame* frame, int align, unsigned int emph, int font,
		  int left, int right)
{
 int WordLen, space, rightMargin;
 long line;

     /* alters the flow margins, if a figure just ended */
 if (paintStartLine < 0) TextLineFrame(frame);

 rightMargin = RightMargin(frame, right, PixOffset);

 if (StyleGetFlag(CurrentDoc, S_INDENT_FLAG)) rightMargin -= StyleGetFlag(CurrentDoc, S_INDENT_FLAG);

 LineBuf[LineLen] = '\0';  /* debug*/
 WordLen = LineLen - WordStart;
/*    WordWidth = XTextWidth(Fonts[font], LineBuf+WordStart, WordLen);*/
 WordWidth = CompoundTextWidth(LineBuf, WordStart, WordLen, &space);

               /* width of a space char */
/*    space = XTextWidth(Fonts[StyleGet(CurrentDoc, S_FONT)], " ", 1); */
               /* width of a space char */
/*    space = XTextWidth(Fonts[font], " ", 1); */

 line = LineSpacing[font];                   /* height of a line */

     /* for tables */
 if (WordWidth > min_width) min_width = WordWidth;

 if (prepass)
   {
    TextLineWidth += WordWidth;
    LineWidth = LineLen = WordStart = 0;
    StartOfLine = bufptr;
   }
  else
   if (WordStart == 0 && Here + WordWidth > rightMargin)
     {
     /* word wider than window */
      if (left + WordWidth > rightMargin)
        {
	 if (emph & EMPH_ANCHOR) WordWidth += 2;

	 PrintString(frame, emph, font, voffset, StartOfLine,
		     WordLen, WordWidth, False);
	 EndOfLine(frame, align);
	 LineWidth = LineLen = WordStart = 0;
	 StartOfLine = bufptr;
        }
      else /* wrap to next line */
        {
	 EndOfLine(frame, align);
	 LineWidth = WordWidth;
	 LineLen = WordLen;
	 WordStart = LineLen;
	 StartOfLine = StartOfWord;
        }

      Here = LeftMargin(frame, left, PixOffset);
     }
    else
     if (WordStart > 0 && Here + LineWidth + space + WordWidth > rightMargin)
       {
        if (emph & EMPH_ANCHOR) LineWidth += 2;

        PrintString(frame, emph, font, voffset, StartOfLine,
		    WordStart-1, LineWidth, True);

        EndOfLine(frame, align);
        Here = left;

     /* was memmove(LineBuf, LineBuf+WordStart, WordLen); 
        but memmove not available for SUNs and
        memcpy screws up for overlapping copies */

        {
	 int n;
	 char *p, *q;

	 n = WordLen;
	 p = LineBuf;
	 q = LineBuf+WordStart;
	 while (n-- > 0) *p++ = *q++;
        }

        LineWidth = WordWidth;
        LineLen = WordLen;
        WordStart = LineLen;
        StartOfLine = StartOfWord;
       }
     else /* word will fit on end of current line */
       {
        if (WordStart > 0) LineWidth += space;

        if (WordWidth > 0) LineWidth += WordWidth;

        WordStart = LineLen;
       }
}


void FlushLine(Bool linebreak, Frame* frame, int align,
	       unsigned int emph, int font, int left, int right)
{
 int space, WordLen = LineLen - WordStart;


 if (preformatted)
   {
    LineWidth = CompoundTextWidth(LineBuf, WordStart, WordLen, &space);
   }
  else
   if (LineLen > 0)
     {
      if ((LineLen == 1) && (LineBuf[0] == ' '))
	LineWidth = XTextWidth(Fonts[font], " ", 1);
       else
	WrapIfNeeded(frame, align, emph, font, left, right);
     }

 if (LineLen > 0)
   {
    if (emph & EMPH_ANCHOR) LineWidth += 2;

    /* Should skip squeezed leading and trailing spaces in anchors */
    if (!((emph & EMPH_ANCHOR) && (LineLen == 1) && (*LineBuf == ' ')))
      PrintString(frame, emph, font, voffset, StartOfLine, LineLen, LineWidth,
		  False);

    if (linebreak)
      {
       EndOfLine(frame, align);
       Here = left;
      }
    /*
     else
      Here += LineWidth;
     */

    LineWidth = LineLen = WordStart = 0;
   }
  else
   if (linebreak)
     {
     /* watch out for empty preformatted lines */
      if (preformatted && TextLineWidth == 0)
	PixOffset += ASCENT(font) + DESCENT(font);

      if (paintStartLine >= 0)  /* was if (Here > left) */
        {
	 EndOfLine(frame, align);
	 Here = left;
        }
     }

 StartOfLine = StartOfWord = bufptr;
}



/*
 * needs to cope with > in quoted text for '' and ""
 */
void SwallowAttributes(void)
{
 int c;

 while ((c = *bufptr) && c != '>') ++bufptr;

 if (c == '>') ++bufptr;
}


/*
 * Handle comments correctly. (?)
 * Original code by <mlvanbie@valeyard.uwaterloo.ca>. Hacked, QL.15-06-97.
 *
 * Get poinetr to the '<' in the beginning of the comment.
 * Return a pointer to the '>' in an end comment
 * (or to '\0' if there isn't one).
 * Example:
 *         "huya <!-- <foo>there</foo> -- --whatever-- > there"
 *     get this -^ ^- pass this           return this -^
 */
char *FindEndComment(char* ptr)
{
 char c, *s;


 ptr += 2; /* Eat "<!" */

 /*
  * Handle comments of 2 types: ``<!--...-->'' and ``<!...>'' separately.
  */
 if ((*ptr == '-') && (*(ptr + 1) == '-'))
   {
    for (ptr += 2;                /* eat "--" from the "<!--" */
	 (s = strstr(ptr, "--")); /* seek for the first occurence of "--" */
	 ptr = s)
      {
      /*
       * Jump to the first character after the found "--",
       * then squeeze possible sequence of "-",
       * i.e. jump to the first character after "-...-" sequence
       * and skip past the optional white space (<= ' ' from GetToken).
       */
       for (s += 2; *s == '-' ; s++);
       for (c = *s; c && c <= ' '; c = *(++s));

       if (c == '>') return s; /* comment terminated correctly */
      }
   }
  else
   {
    if ((s =  ARENA_Index(ptr, '>'))) return s;
   }

 /* Document ends in unfinished comment -- seek for '\0' */

 if (*ptr) while(*(++ptr));

 return ptr;
}

/* char *tag points to start of tag string which is terminated by whitespace
 * (including EOF) or a '>' character.  Return tag code or 0 if unknown.
 * Valid tags are in parsehtml_i.h.  A second array of pointers is used to
 * find the beginng of the 'a's, 'b's... etc.  This setup is done here 1 time!
 */
int RecognizeTag(void)
{
 int c, len, tagret;
 char *s;
 VALIDTokens *vt = NULL;

 /* skip the opening <      */
 s = bufptr + 1;

 /* skip any comments!      */
 if (*s == '!')
   {
    EndTag = 0;
    TagLen = 0;  /* we will be manipulating bufptr directly */
    TokenClass = EN_UNKNOWN;
    bufptr = FindEndComment(bufptr);
    return UNKNOWN; /* unknown tag */
   }

 /* check for "end of" tag... e.g. </head>    */
 if (*s == '/')
   {
    EndTag = 1;
    ++s;
   }
  else
   EndTag = 0;

 if ((c = *s) == '!' || c == '?')
   ++s;
 else if (c != '>' && !isalpha(c))
   return PCDATA;

 /* find end of token tag to allow use of strncasecmp */
 while ((c = *s, isalpha(c)) || isdigit(c))
   ++s;

 TagLen = s - bufptr;        /* how far to next char after tag name */
 len = TagLen - EndTag - 1;  /* number of chars in tag name itself */
 s -= len;
 c = TOLOWER(*s);

 /* One time only! Setup the ABCs to the ARENA_ValidTokens array.
  * I suppose this could be moved to an init routine,
  * but 1 if here will skip it forever on!
  * The ABC table is [0; 255] !
  */
 if (ARENA_ValidTokensABC[(unsigned int)'a'] == NULL)
   {
    int i;

    vt = ARENA_ValidTokens;
    while (vt->Txt != NULL)
      {
       i = *vt->Txt;
       if (ARENA_ValidTokensABC[i] == NULL)  ARENA_ValidTokensABC[i] = vt;
       vt->Len = Arena_StrLen(vt->Txt);
       vt++;
      }
    /* End ``while'' */
   }

 /* NOW, find the token in the input buffer in the table of valid tokens.
  * Get a "beginning" pointer from the ABC table and roar ahead quickly until
  * either the end of the section containing the first letter of the word,
  * or the end of the table is reached... wecf.
  */
 TokenClass = EN_UNKNOWN;
 tagret = UNKNOWN;

 vt = ARENA_ValidTokensABC[(unsigned int)c];  /* stay out of lookup if NULL */
 if (vt == NULL) return tagret;               /* when 1st letter is invalid */

 while (vt->Txt != NULL)
   {
    if (c != *(vt->Txt)) break;
    if (len == vt->Len && strncasecmp(s, vt->Txt, len) == 0)
      {
       TokenClass = vt->Class;
       tagret = vt->Name;
       break;
      }
    vt++;
   }

 if (tagret == UNKNOWN || vt->SpecialCase == 0) return tagret;

 /* something Special about this tag?    */
 switch (vt->SpecialCase)
   {
   case TAG_HEAD:
     isindex_state = EndTag ? 0 : TAG_HEAD;
     break;
   case TAG_BODY:
     isindex_state = EndTag ? 0 : TAG_BODY;
     break;
   case TAG_ISINDEX:
     if (isindex_state == TAG_HEAD)
       {
	TokenClass = EN_SETUP;       /* within <head>...</head>    */
	tagret = TAG_ISINDEX;
       }
     else
       {
	TokenClass = EN_TEXT;        /* in limbo (0) or body (2)   */
	tagret = TAG_ISINDEX_DE;     /* setup to do input field    */
       }
     break;
   default:
     break;
   }

 /* get out of here!   */
 return tagret;
}


void UnGetToken(void)
{
 bufptr = LastBufPtr;
}


/*
 * The token type is returned in the global token.
 * Characters are returned in TokenValue while TokenClass
 * is used to return a class value e.g. EN_SETUP or EN_BLOCK.
 * Entity definitions are pointed to by EntityValue.
 *
 * The bufptr is moved past the token, except at the end
 * of the buffer - as a safety precaution.
 */
int GetToken(void)
{
 int c, k, n;


 LastBufPtr = bufptr;
 TokenValue = c = *(unsigned char *)bufptr;

 if (targetptr)
   if (bufptr <= targetptr)
     ViewOffset = PixOffset;

 if (c == '<' && (Token = RecognizeTag()) != PCDATA)
   {
    bufptr += TagLen;   /* to first char after tag name */
    return Token;
   }

 TokenClass = EN_TEXT;
 EndTag = 0;

 if (c == '&' && (isalpha(bufptr[1]) || bufptr[1] == '#'))
   {
    n = entity(bufptr + 1, &k);

    if (n)
      {
       bufptr += k;
       TokenValue = n;
       Token = PCDATA;
       return Token;
      }
   }

 if (c <= ' ')
   {
    if (c == '\0')
      {
       Token = ENDDATA;
       TokenClass = EN_UNKNOWN;
       return Token;
      }

    bufptr++;
    Token = WHITESPACE;
    return Token;
   }

 bufptr++;
 Token = PCDATA;
 return Token;
}


/*
 * Condense subsequent white space into a single one.
 */
Bool SqueezeWhiteSpace(void)
{
 if (GetToken() == WHITESPACE)
   {
    while (GetToken() == WHITESPACE);
    UnGetToken();
    bufptr--;
    return True;
   }
  else
   {
    UnGetToken();
    return False;
   }
}


/*
 * Skip subsequent white space.
 */
void SkipWhiteSpace(void)
{
 while (GetToken() == WHITESPACE);
 UnGetToken();
}


/*
 * get token, skipping white space and unknown tokens
 */
int SkipSpace(void)
{
 for (;;)
   {
    while (GetToken() == WHITESPACE);

    if (Token == UNKNOWN)
      SwallowAttributes();
     else
      break;
   }

 return Token;
}



/*
 * assumes bufptr points to start of attribute
 */
char *ParseAttribute(int *len)
{
 int c;
 char *attr;

 *len = 0;
 attr = bufptr;
 IdAttributeFlag = 0;

 for (;;)
   {
    c = *bufptr;

    if (c == '>' || c == '\0') return attr;

    if (c == '=' || isspace(c)) break;

    ++(*len);
    ++bufptr;
   }

 if (*len == 2 && strncasecmp(attr, "id", 2) == 0) IdAttributeFlag = 1;

 return attr;
}


/*
 * values start with "=" or " = " etc.
 */
char *ParseValue(int *len)
{
 int c, delim;
 char *value;

 *len = 0;

 while (c = *bufptr, isspace(c)) ++bufptr;

 if (c != '=') return 0;

 ++bufptr;   /* past the = sign */

 while (c = *bufptr, isspace(c)) ++bufptr;

 if (c == '"' || c == '\'')
   {
    delim = c;
    ++bufptr;
   }
  else
   delim = 0;

 value = bufptr;

 for (;;)
   {
    c = *bufptr;

    if (c == '\0')
      {
       if (delim)
	 return 0;  /* a big and dirty hack --Spif 27-Oct-95 */
        else
	 break;
      }

    if (delim)
      {
       if (c == delim)
	 {
	  ++bufptr;
	  break;
	 }
      }
    else
      if (c == '>' || isspace(c)) break;

    ++(*len);
    ++bufptr;
   }

 if (IdAttributeFlag && value && targetId)
   {
    if (Arena_StrLen(targetId) == *len &&
	strncasecmp(value, targetId, *len) == 0)
      {
       IdOffset = PixOffset;
      }
   }

 IdAttributeFlag = 0;
 return value;
}


/*
 * Parse </...> (end tag).
 */
Bool ParseEndTag(Frame* theFrame)
{
 extern int Token;
#ifdef ARENA_DEBUG
 char Iam[] = "ParseEndTag";
#endif


 if (EndTag)
   {
    switch (Token)
      {
       case TAG_CENTER:
	 if (prealign)
	   {
#ifdef ARENA_DEBUG
	    if (TAG_TRACE && VERBOSE_TRACE)
	      Arena_TracePrint(Iam, " Clearing `prealign'.\n");
#endif
	    if (theFrame)
	      {
	       FlushLine(True, theFrame,
			 ALIGN_CENTER, EMPH_NORMAL,
			 (int)StyleGet(CurrentDoc, ArenaSI_FONT),
			 (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT),
			 (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_RIGHT));
	      }

	    prealign = False;
	   }
	 break;

       default:
	 break;
      }
    /* End ``switch (Token)'' */

    SwallowAttributes();
    return True;
   }
  else
   return False;
}


/*
 * HREF attribute defines original document URL and is added by
 * the browser when making a local copy of a document so that
 * relative references can be deferenced to their original links
 */
void ParseBase(Frame* frame)
{
 char *href, *name;
 int hreflen, namelen;

 ParseAnchorAttrs(&href, &hreflen, &name, &namelen, &class, &class_len);
 if (href) CurrentDoc->base = strndup(href, hreflen);
}


void ParseLink(Frame* frame)
{
 char *href = NULL, *rel = NULL;
 int hreflen = 0, rellen = 0;
 int c, n, m;
 char *attr, *value;

 for (;;)
   {
    c = *bufptr++;

    if (c == '\0' || c == '>') break;

    if (isspace(c)) continue;

    --bufptr;
    attr = ParseAttribute(&n);
    value = ParseValue(&m);

    if (n == 4 && strncasecmp(attr, "href", n) == 0)
      {
       href = value;
       hreflen = m;
       continue;
      }

    if (n == 3 && strncasecmp(attr, "rel", n) == 0)
      {
       rel = value;
       rellen = m;
       continue;
      }
   }

 if (href && rel)
   {
    if (!NoStyle &&
	!CurrentDoc->link_style &&
	(strncasecmp(rel, "style", 5) == 0))
      CurrentDoc->link_style = StyleLoad(href, hreflen,
					 CurrentDoc->pending_reload);
   }
}


void ParseTitle(Bool implied, Frame* frame)
{
 if (ParseEndTag(frame)) return;

 if (!implied) SwallowAttributes();

 /* Skip leading white space ---
  * subsequently contigous white space is compressed to a single space.
  */
 SqueezeWhiteSpace();

 LineLen = 0;

 for (;;)
   {
    GetToken();

    if (Token == TAG_TITLE)
      if (ParseEndTag(frame))
	break;

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == WHITESPACE)
      {
       SkipWhiteSpace();
       if (LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = ' ';

       continue;
      }

    if (Token != PCDATA)
      {
       UnGetToken();
       break;
      }

    if (LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = TokenValue;
   }

 LineBuf[LineLen] = '\0';
 SetBanner(LineBuf);
}


/* howcome 26/2/95 */
void ParseStyle(Frame* frame)
{
 if (ParseEndTag(frame)) return;

 SwallowAttributes();

 /* Skip leading white space ---
  * subsequently contigous white space is compressed to a single space.
  */
 SqueezeWhiteSpace();

 LineLen = 0;

 for (;;)
   {
    GetToken();

    if (Token == TAG_STYLE)
      if (ParseEndTag(frame))
	break;

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = TokenValue;
    }

 LineBuf[LineLen] = '\0';
 if (!NoStyle && !CurrentDoc->head_style)
   CurrentDoc->head_style = strdup (LineBuf);
}



void ParseSetUp(Bool implied, Frame* frame)
{
 if (ParseEndTag(frame)) return;

 if (!implied) SwallowAttributes();

 for (;;)
   {
    while (GetToken() == WHITESPACE);

    if (Token == TAG_HEAD)
      if (ParseEndTag(frame))
	break;

    if (Token == TAG_TITLE)
      {
       UnGetToken();
       ParseTitle(0, frame);
       continue;
      }

    if (Token == TAG_ISINDEX)
      {
       SwallowAttributes();
       DocIsIndex = True;
       ClearStatus();
       continue;
      }

    if (Token == TAG_BASE)
      {
       UnGetToken();
       ParseBase(frame);
       continue;
      }

    /* howcome 26/2/95 */
    if (Token == TAG_STYLE)
      {
       UnGetToken();
       ParseStyle(frame);
       continue;
      }

    /* howcome 25/4/95 */

    if (Token == TAG_LINK)
      {
       UnGetToken();
       ParseLink(frame);
       continue;
      }

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == PCDATA || Token == ENTITY)
      {
       UnGetToken();
       break;
      }

    if (Token == ENDDATA || TokenClass != EN_SETUP)
      {
       UnGetToken();
       break;
      }
   }
}


void ParseAnchorAttrs(char **href, int *hreflen, char **name, int *namelen,
		      char **class_p, int *class_len_p)
{
 int c, n, m;
 char *attr, *value;
#ifdef ARENA_DEBUG
 char Iam[] = "ParseAnchorAttrs";
#endif


 *href = NULL;
 *name = NULL;
#if 1	/* QingLong.04-10-97 */
 *hreflen = 0;
 *namelen = 0;
#endif

 for (;;)
   {
    c = *bufptr++;

    if (c == '\0') break;

    if (c == '>') break;

    if (isspace(c)) continue;

    --bufptr;
    attr = ParseAttribute(&n);
    value = ParseValue(&m);

    if (n == 4 && strncasecmp(attr, "href", n) == 0)
      {
       *href = value;
       *hreflen = m;
       continue;
      }

    if (n == 4 && strncasecmp(attr, "name", n) == 0)
      {
       *name = value;
       *namelen = m;
       continue;
      }

    if (n == 5 && strncasecmp(attr, "class", n) == 0)
      {
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
       if (STYLE_TRACE)
	 {
	  char *s = strndup(attr, n);
	  Arena_TracePrint(Iam, " style class \"%s\".\n", s);
	  Free(s);
	 }
#endif

       *class_p = value;
       *class_len_p = m;
      }
   }

#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
 if (*href == NULL)
   if (TAG_TRACE)
     Arena_TracePrint(Iam, " *href == NULL, hreflen = %d.\n", *hreflen);
#endif
}


/* howcome30/5/95: addded support for widht and height */
void ParseImageAttrs(char **href, int *hreflen, int *align, int *ismap,
		     int *width, int *height)
{
 int c, n, m;
 char *attr, *value;

 *href = NULL;               /* howcome: NULL !! */
 *align = ALIGN_BOTTOM;
 if (prealign) *align = ALIGN_CENTER;
 *ismap = 0;
 *width = 0; *height = 0;

 for (;;)
   {
    c = *bufptr++;

    if (c == '\0') break;
    if (c == '>') break;

    if (isspace(c)) continue;

    --bufptr;
    attr = ParseAttribute(&n);
    value = ParseValue(&m);

    if (n == 3 && strncasecmp(attr, "src", n) == 0)
      {
       *href = value;
       *hreflen = m;
       continue;
      }


    if (n == 5 && strncasecmp(attr, "width", n) == 0)
      {
       *width = atoi(value);
       continue;
      }

    if (n == 6 && strncasecmp(attr, "height", n) == 0)
      {
       *height = atoi(value);
       continue;
      }

    if (n == 5 && strncasecmp(attr, "align", n) == 0)
      {
       if (m == 3 && strncasecmp(value, "top", m) == 0)
	 *align = ALIGN_TOP;
       else if (m == 6 && strncasecmp(value, "middle", m) == 0)
	 *align = ALIGN_MIDDLE;
       else if (m == 6 && strncasecmp(value, "bottom", m) == 0)
	 *align = ALIGN_BOTTOM;
       else if (m == 4 && strncasecmp(value, "left", m) == 0)
         *align = ALIGN_LEFT;
       else if (m == 5 && strncasecmp(value, "right", m) == 0)
         *align = ALIGN_RIGHT;
       else if (m == 6 && strncasecmp(value, "center", m) == 0)
         *align = ALIGN_CENTER;

       continue;
      }

    if (n == 5 && strncasecmp(attr, "ismap", n) == 0) *ismap = 1;
   }
}


void ParseFigureAttrs(char **href, int *hreflen, int *align,
		      int *width, int *height)
{
 int c, n, m;
 char *attr, *value;

 *href = 0;
 *align = ALIGN_CENTER;
 *width = 0; *height = 0;

 for (;;)
   {
    c = *bufptr++;

    if (c == '\0') break;

    if (c == '>') break;

    if (isspace(c)) continue;

    --bufptr;
    attr = ParseAttribute(&n);
    value = ParseValue(&m);

    if (n == 3 && strncasecmp(attr, "src", n) == 0)
      {
       *href = value;
       *hreflen = m;
       continue;
      }

    if (n == 5 && strncasecmp(attr, "width", n) == 0)
      {
       *width = atoi(value);
       continue;
      }

    if (n == 6 && strncasecmp(attr, "height", n) == 0)
      {
       *height = atoi(value);
       continue;
      }

    if (n == 5 && strncasecmp(attr, "align", n) == 0)
      {
       if (m == 4 && strncasecmp(value, "left", m) == 0)
	 *align = ALIGN_LEFT;
        else
	 if (m == 6 && strncasecmp(value, "center", m) == 0)
	   *align = ALIGN_CENTER;
	  else
	   if (m == 5 && strncasecmp(value, "right", m) == 0)
	     *align = ALIGN_RIGHT;
	    else
	     if (m == 9 && strncasecmp(value, "bleedleft", m) == 0)
	       *align = ALIGN_BLEEDLEFT;
	      else
	       if (m == 10 && strncasecmp(value, "bleedright", m) == 0)
		 *align = ALIGN_BLEEDRIGHT;

       continue;
      }
   }
}


void ParseFormAttrs(FormSubmitMethod* method, char **action, int *alen)
{
 int c, n, m;
 char *attr, *attrval;

 *action = NULL;
 *method = GET; /* implied must add encoded-type.. */
 for (;;)
   {
    c= *bufptr++;

    if (c == '\0') break;
    if (c == '>') break;

    if (isspace(c)) continue;

    --bufptr;
    attr = ParseAttribute(&n);
    attrval = ParseValue(&m);
    if (n == 6 && strncasecmp(attr, "method", n) == 0)
      {
       if ( m == 3 && strncasecmp(attrval, "get", m) == 0)
	 *method = GET;
        else
	 if ( m == 4 && strncasecmp(attrval, "post", m) == 0)
	   *method = POST;
      }
    if ( n == 6 && strncasecmp(attr, "action", n) == 0)
      {
       *action = attrval;
       *alen = m;
       continue;
      };
   };
}


/* wm 19.Jan.95 */
void ParseTextAreaAttrs(char **name, int *nlen,
			int *rows, int *cols, int *flags)
{
 int c, n, m; /* janet: not used: checked */
 char *attr, *attrval;

 *rows = 3;
 *cols = 33;
 *flags = 0;
 *name = "";
 *nlen = 0;

 for (;;)
   {
    c = *bufptr++;

    if (c == '\0') break;
    if (c == '>') break;

    if (isspace(c)) continue;

    --bufptr;
    attr = ParseAttribute(&n);
    attrval = ParseValue(&m);

    if (n == 4)
      {
       if (strncasecmp(attr, "name", n) == 0)
	 {
	  *name = attrval;
	  *nlen = m;
	  continue;
	 }
        else
	 if (strncasecmp(attr, "rows", n) == 0)
	   {
	    sscanf(attrval, "%d", rows);
	    continue;
	   }
	  else
	   if (strncasecmp(attr, "cols", n) == 0)
	     {
	      sscanf(attrval, "%d", cols);
	      continue;
	     }

       continue;
      }

    if (n == 7 && strncasecmp(attr, "checked", n) == 0)
      {
       *flags |= CHECKED;
       continue;
      }

    if (n == 8 && strncasecmp(attr, "disabled", n) == 0)
      {
       *flags |= DISABLED;
       continue;
      }
   }
}


/* Parse <ISINDEX> attributes. (<ISINDEX> is handled as a form of special type)
 * The only allowed by HTML3.2 standard attribute is PROMPT. :(
 */
void ParseIsindexAttrs(char** thePrompt, int* thePromptLength,
		       char** theAction, int* theActionLength,
		       char** theValue,  int* theValueLength)
{
 int c, n, m;
 char *attr, *attrval;

 (*thePrompt)       = (*theAction)       = (*theValue)       = NULL;
 (*thePromptLength) = (*theActionLength) = (*theValueLength) = 0;

 for ( ; (c = *bufptr++); )
   {
    if (c == '>') break;

    if (isspace(c)) continue;

    --bufptr;
    attr    = ParseAttribute(&n);
    attrval = ParseValue(&m);

    switch (n)
      {
       case 5:
	 if (strncasecmp(attr, "value", n) == 0)
	   {
	    *theValue = attrval;
	    *theValueLength = m;
	   }
	 break;

       case 6:
	 if (strncasecmp(attr, "prompt", n) == 0)
	   {
	    *thePrompt = attrval;
	    *thePromptLength = m;
	   }
	  else
	   if (strncasecmp(attr, "action", n) == 0)
	     {
	      *theAction = attrval;
	      *theActionLength = m;
	     }
	 break;

       default:
	 break;
      }
    /* End ``switch (n)'' */
   }
}


void ParseInputAttrs(int* type,
		     char** name, int* nlen, char** value, int* vlen,
		     int* maxlength, int* size, int* flags, Image** image)
{
 int c, n, m; /* janet: not used: checked */
 char *attr, *attrval, *href = NULL;
 int hreflen = 0; /* --SPif */
#ifdef ARENA_DEBUG
 char Iam[] = "ParseInputAttrs";
#endif


 *type = TEXTFIELD;
 *size = 20;
 *flags = 0;
 *name = *value = "";
 *nlen = *vlen = 0;
 hreflen = 0;
 *image = NULL;

 for (;;)
   {
    c = *bufptr++;

    if (c == '\0') break;
    if (c == '>') break;

    if (isspace(c)) continue;

    --bufptr;
    attr = ParseAttribute(&n);
    attrval = ParseValue(&m);

    switch (n)
      {
       case 3:
	 /* --Spif 9-Oct-1995 */ 
	 if (strncasecmp(attr, "src", n) == 0)
	   {
	    href = attrval;
	    hreflen = m;
	   }

	 break;

       case 4:
	 if (strncasecmp(attr, "type", n) == 0)
	   {
	    switch (m)
	      {
	       case 4:
		 if (strncasecmp(attrval, "text", m) == 0) *type = TEXTFIELD;
		 break;

	       case 5:
		 if (strncasecmp(attrval, "radio", m) == 0)
		   *type = RADIOBUTTON;
		  else
		   if (strncasecmp(attrval, "reset", m) == 0)
		     *type = RESETBUTTON;
		    else
		     if (strncasecmp(attrval, "image", m) == 0)
		       *type = SUBMITBUTTON; /* --Spif:
						image in html2 = Submit */
		 break;

	       case 6:
		 if (strncasecmp(attrval, "submit", m) == 0)
		   *type = SUBMITBUTTON;
		  else
		   if (strncasecmp(attrval, "hidden", m) == 0)
		     *type = HIDDEN;
		    else
		     if (strncasecmp(attrval, "passwd", m) == 0)
		       *type = PASSWD;
		 break;

	       case 8:
		 if (strncasecmp(attrval, "checkbox", m) == 0)
		   *type = CHECKBOX;
		 break;

	       default:
#ifdef ARENA_DEBUG
		 if (PARSEHTML_TRACE && VERBOSE_TRACE)
		   Arena_TracePrint(Iam,
				    " Strange <INPUT> type length %d.\n", m);
#endif
		 break;
	      } /* End ``switch (m)'' */
	    continue;
	   }
	  else
	   if (strncasecmp(attr, "name", n) == 0)
	     {
	      *name = attrval;
	      *nlen = m;
	      continue;
	     }
	    else
	     if (strncasecmp(attr, "size", n) == 0)
	       {
		sscanf(attrval, "%d", size);
		continue;
	       }
	   
	 break;

       case 5:
	 if (strncasecmp(attr, "value", n) == 0)
	   {
	    *value = attrval;
	    *vlen = m;
	    continue;
	   }

	 break;

       case 7:
	 if (strncasecmp(attr, "checked", n) == 0) *flags |= CHECKED;
	 break;

       case 8:
	 if (strncasecmp(attr, "disabled", n) == 0)
	   {
	    *flags |= DISABLED;
	    continue;
	   }
	 else
	   if (strncasecmp(attr, "multiple", n) == 0) *flags |= MULTIPLE;

	 break;

       case 9:
	 if (strncasecmp(attr, "maxlength", n) == 0)
	   {
	    if (maxlength != NULL)
	      {
	       sscanf(attrval, "%d", maxlength);
	       continue;
	      }
	   }
	 break;

       default:
#ifdef ARENA_DEBUG
	 if (PARSEHTML_TRACE && VERBOSE_TRACE)
	   Arena_TracePrint(Iam,
			    " Strange <INPUT> attribute length %d.\n", m);
#endif
	 break;
      } /* End ``switch (n)''*/
   } /* End ``for (;;)'' */

 /* --Spif 9-Oct-1995 */
 if (hreflen > 0) *image = GetImage(href, hreflen, CurrentDoc->pending_reload);
}


/*
 * QingLong.10-04-97
 */
void ParseOptionAttrs(char** value, int* flags)
{
 char c;
 int n, m;
 char* attr;
 char* attrval;

 for (;;)
   {
    c = *bufptr++;

    if (c == '\0') break;
    if (c == '>') break;

    if (isspace(c)) continue;

    --bufptr;
    attr = ParseAttribute(&n);
    attrval = ParseValue(&m);

    if (n == 5 && strncasecmp(attr, "value", n) == 0)
      {
       *value = strndup(attrval, m);
       continue;
      }

    if (n == 8)
      {
       if (strncasecmp(attr, "selected", n) == 0)
	 *flags |= OPTION_SELECTED;
        else
	 if (strncasecmp(attr, "disabled", n) == 0)
	   *flags |= OPTION_DISABLED;
      }
   }
}


void ParseSelectAttrs(char** name, int* nlen, int* size, int* flags)
{
 int c, n, m;
 char *attr, *attrval;

 *size = 20;
 *flags = 0;
 *name = "";
 *nlen = 0;

 for (;;)
   {
    c = *bufptr++;

    if (c == '\0') break;
    if (c == '>') break;

    if (isspace(c)) continue;

    --bufptr;
    attr = ParseAttribute(&n);
    attrval = ParseValue(&m);

    if (n == 4 && strncasecmp(attr, "name", n) == 0)
      {
       *name = attrval;
       *nlen = m;
       continue;
      }

    if (n == 4 && strncasecmp(attr, "size", n) == 0)
      {
       sscanf(attrval, "%d", size);
       continue;
      }

    if (n == 8 && strncasecmp(attr, "multiple", n) == 0) *flags |= MULTIPLE;
   }
}


void ParseCellAttrs(int *rowspan, int *colspan, int *align, int *nowrap)
{
 int c, n, m;
 char *attr, *attrval;

 *rowspan = *colspan = 1;
 *align = ALIGN_CENTER;
 *nowrap = 0;

 for (;;)
   {
    c = *bufptr++;

    if (c == '\0') break;
    if (c == '>') break;

    if (isspace(c)) continue;

    --bufptr;
    attr = ParseAttribute(&n);
    attrval = ParseValue(&m);

    if (n == 7 && strncasecmp(attr, "rowspan", n) == 0)
      {
       sscanf(attrval, "%d", rowspan);
       continue;
      }

    if (n == 7 && strncasecmp(attr, "colspan", n) == 0)
      {
       sscanf(attrval, "%d", colspan);
       continue;
      }

    if (n == 5 && strncasecmp(attr, "align", n) == 0)
      {
       if (m == 6 && strncasecmp(attrval, "center", m) == 0)
         *align = ALIGN_CENTER;
        else
         if (m == 5 && strncasecmp(attrval, "right", m) == 0)
           *align = ALIGN_RIGHT;
          else
           if (m == 4 && strncasecmp(attrval, "left", m) == 0)
             *align = ALIGN_LEFT;

       continue;
      }

    if (n == 6 && strncasecmp(attr, "nowrap", n) == 0)
      *nowrap = 1;
   }
 /* End for */
}


void ParseTableAttrs(int *border)
{
 int c, n, m;
 char *attr, *attrval;

 *border = 0;

 for (;;)
   {
    c = *bufptr++;

    if (c == '\0') break;
    if (c == '>') break;

    if (isspace(c)) continue;

    --bufptr;
    attr = ParseAttribute(&n);
    attrval = ParseValue(&m);

    if (n == 6 && strncasecmp(attr, "border", n) == 0)
      {
       int bd = 1;   /* DJB 17-Jan-96 */

       if (attrval) sscanf(attrval, "%d", &bd);

       *border = bd ? 1: 0; /* Restrict to 1 and 0 ? */
       continue;
      }
   }
}


void ParseClassAttrs(char **class_p, int *class_len_p)
{
 int c, n, m;
 char *attr, *attrval;

 *class_p = NULL;
 *class_len_p = 0;

 for (;;)
   {
    c = *bufptr++;

    if (c == '\0') break;
    if (c == '>')  break;

    if (isspace(c)) continue;

    --bufptr;
    attr = ParseAttribute(&n);
    attrval = ParseValue(&m);

    if (n == 5 && strncasecmp(attr, "class", n) == 0)
      {
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
       if (STYLE_TRACE)
	 {
	  char *s = strndup(attrval, m);

	  Arena_TracePrint("ParseClassAttrs", " style_class %s\n", s);
	  Free(s);
	 }
#endif
       *class_p = attrval;
       *class_len_p = m;
      }
   }
}



void ParseParaAttrs(int *align, char **class_p, int *class_len_p)
{
 int c, n, m;
 char *attr, *attrval;

 *class_p = NULL;
 *class_len_p = 0;

 if (prealign) *align = ALIGN_CENTER;

 for (;;)
   {
    c = *bufptr++;

    if (c == '\0') break;
    if (c == '>') break;

    if (isspace(c)) continue;

    --bufptr;
    attr = ParseAttribute(&n);
    attrval = ParseValue(&m);

    if (n == 5 && strncasecmp(attr, "align", n) == 0)
      {
       if (m == 6 && strncasecmp(attrval, "center", m) == 0)
	 *align = ALIGN_CENTER;
       else if (m == 5 && strncasecmp(attrval, "right", m) == 0)
	 *align = ALIGN_RIGHT;
       else if (m == 4 && strncasecmp(attrval, "left", m) == 0)
	 *align = ALIGN_LEFT;
       else if (m == 7 && strncasecmp(attrval, "justify", m) == 0)
	 *align = ALIGN_JUSTIFY;
      }

    if (n == 5 && strncasecmp(attr, "class", n) == 0)
      {
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
       if (STYLE_TRACE)
	 {
	  char *s = strndup(attrval, m);
	  Arena_TracePrint("ParseParaAttrs", " style_class %s\n", s);
	  Free(s);
	 }
#endif
       *class_p = attrval;
       *class_len_p = m;
      }
   }
}


#ifndef NO_BACKGROUND_IN_HTML
int Arena_strncpy(char** dest, char* s, int len, int* dest_capacity)
{
 int xlen;

 if (len == 0) return 0;
 if (s == NULL) return 0;
 if (dest == NULL) return 0;
 if ((*dest) == NULL) return 0;
 if (dest_capacity == NULL) return 0;

 if ((xlen = Arena_StrLen(s)) == 0) return 0;

 if (xlen > len) xlen = len;

 if (xlen <= (*dest_capacity))
   {
#ifdef HAVE_STRNCPY
    strncpy((*dest), s, xlen);
# else
    sprintf((*dest), "%.*s", xlen, s);
#endif
    (*dest_capacity) -= xlen;
    (*dest) += xlen;
    return xlen;
   }
  else
   {
    return 0;
   }
}


#  define ARENA_ParseBodyAttrs_BUFLENGTH	1024
void ParseBodyAttrs(char** thebufptr)
{
 int c, n, m;
 char* b_colour    = NULL; int b_colour_len    = 0;
 char* b_bg_img    = NULL; int b_bg_img_len    = 0;
 char* b_bg_colour = NULL; int b_bg_colour_len = 0;
 int quoted_name_rest = ARENA_ParseBodyAttrs_BUFLENGTH - 1;
 char *attr, *attrval;
 char *d;

 d = (*thebufptr);

 for (;;)
   {
    c = (unsigned char)(*(*thebufptr)++);

    if (c == '\0') break;
    if (c == '>') break;

    if (isspace(c)) continue;

    (*thebufptr)--;
    attr = ParseAttribute(&n);
    attrval = ParseValue(&m);

    if (n == 4 && strncasecmp(attr, "text",  n) == 0)
      {          
       b_colour = attrval;
       b_colour_len = m;
       continue;
      }

    if (n == 10 && strncasecmp(attr, "background", n) == 0)
      {
       b_bg_img = attrval;
       b_bg_img_len = m;
       continue;
      }

    if ((n == 7 && strncasecmp(attr, "bgcolor",  n) == 0) ||
	(n == 8 && strncasecmp(attr, "bgcolour", n) == 0))
      {          
       b_bg_colour = attrval;
       b_bg_colour_len = m;
       continue;
      }
   }

 (*thebufptr) = d;

 if (b_colour || b_bg_img || b_bg_colour)
   {
    char H[] = "HTML { }";
    int Hlen = sizeof(H) - 1;
    Bool NeedSeaparator = False;
    char i;
    char* xbuf = NULL;
    char *quoted_name, *quoted_name_p;

    quoted_name = (char *)Arena_CAlloc(ARENA_ParseBodyAttrs_BUFLENGTH,
				       sizeof(char),
				       False);
    quoted_name_p = quoted_name;

    /* This construction demonstrates structure programming ugliness */
    for (i = 1; i > 0; i--)
      {
       if (Arena_strncpy(&quoted_name_p, H, Hlen - 2, &quoted_name_rest) == 0)
	 break;

       if (b_colour)
	 {
	  char x[] = " color: ";
	  int xlen = sizeof(x) - 1;

	  if (NeedSeaparator)
	    {
	     if (Arena_strncpy(&quoted_name_p, ";", 1, &quoted_name_rest) == 0)
	       break;
	    }
	  if (Arena_strncpy(&quoted_name_p, x, xlen, &quoted_name_rest) == 0)
	    break;
	  if (Arena_strncpy(&quoted_name_p, b_colour, b_colour_len,
			    &quoted_name_rest)
	      == 0)
	    break;
	  NeedSeaparator = True;
	 }


       if (b_bg_img || b_bg_colour)
	 {
	  char x[] = " background:";
	  char y[] = " url()";
	  int xlen = sizeof(x) - 1;
	  int ylen = sizeof(y) - 1;
	  int len = 0;

	  len = xlen + (b_bg_img_len    ? (b_bg_img_len    + ylen) : 0)
	             + (b_bg_colour_len ? (b_bg_colour_len + 1   ) : 0);
	  xbuf = (char*)Arena_CAlloc(len + 1, sizeof(char), False);
	  {
	   char* xbuf_p = xbuf;
	   int xbuf_rest = len;

	   if (Arena_strncpy(&xbuf_p, x, xlen, &xbuf_rest) == 0) break;

	   if (b_bg_img_len)
	     {
	      if (Arena_strncpy(&xbuf_p, y, ylen - 1, &xbuf_rest) == 0) break;
	      if (Arena_strncpy(&xbuf_p, b_bg_img, b_bg_img_len, &xbuf_rest)
		  == 0)
		break;
	      if (Arena_strncpy(&xbuf_p, y + ylen - 1, 1, &xbuf_rest) == 0)
		break;
	     }

	   if (b_bg_colour_len)
	     {
	      if (Arena_strncpy(&xbuf_p, " ", 1, &xbuf_rest) == 0) break;
	      if (Arena_strncpy(&xbuf_p, b_bg_colour, b_bg_colour_len,
				&xbuf_rest)
		  == 0)
		break;
	     }
	  }

	  if (NeedSeaparator)
	    {
	     if (Arena_strncpy(&quoted_name_p, ";", 1, &quoted_name_rest) == 0)
	       break;
	    }
	  if (Arena_strncpy(&quoted_name_p, xbuf, len, &quoted_name_rest)
	      == 0)
	    break;
	  NeedSeaparator = True;
	 }
       if (Arena_strncpy(&quoted_name_p, H + Hlen - 2, 2, &quoted_name_rest)
	   == 0)
	 break;
      }
    /* End ``ugly for'' */

    Free(xbuf);

#ifdef ARENA_DEBUG
    if (TAG_TRACE && STYLE_TRACE)
      Arena_TracePrint("ParseBodyAttrs",
		       "\n\tFake style: \"%s\".\n", quoted_name);
#endif

    if (CurrentDoc->style == NULL)
      CurrentDoc->style = StyleCopy(context->style);

    StyleChew(CurrentDoc, CurrentDoc->style, quoted_name, S_AUTHOR);

    Free(quoted_name);
   }
}
#  undef ARENA_ParseBodyAttrs_BUFLENGTH
#endif /* NO_BACKGROUND_IN_HTML */


Image* ParseNoteAttrs(void)
{
 int c, n, m, hreflen;
 char *attr, *attrval, *href = NULL;
 Image* image;

 image = NULL;
 hreflen = 0;

 for (;;)
   {
    c = *bufptr++;

    if (c == '\0') break;
    if (c == '>') break;

    if (isspace(c)) continue;

    --bufptr;
    attr = ParseAttribute(&n);
    attrval = ParseValue(&m);

    if ((n == 4 && strncasecmp(attr, "role", n) == 0) ||
	(n == 5 && strncasecmp(attr, "class", n) == 0))
      {
       if (m == 4 && strncasecmp(attrval, "note", m) == 0)
	 image = note_image;
        else
	 if (m == 3 && strncasecmp(attrval, "tip", m) == 0)
	   image = note_image;
	  else
	   if (m == 7 && strncasecmp(attrval, "caution", m) == 0)
	     image = caution_image;
	    else
	     if (m == 7 && strncasecmp(attrval, "warning", m) == 0)
	       image = warning_image;
      }

    if (n == 3 && strncasecmp(attr, "src", n) == 0)
      {
       href = attrval;
       hreflen = m;
      }
   }

 if (hreflen > 0) image = GetImage(href, hreflen, CurrentDoc->pending_reload);

 return image;
}


void ParseOption(Bool implied, Frame* frame, Field* field, int font)
{
 int flags = 0;
 char* value = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "ParseOption";
#endif


 if (ParseEndTag(frame)) return;

 if (implied)
   {
    flags = 0;
   }
  else
   {
    ParseOptionAttrs(&value, &flags);
   }

 LineLen = 0;

 for (;;)
    {
     GetToken();

     if (Token == TAG_OPTION)
       if (ParseEndTag(frame))
	 break;

     if (Token == UNKNOWN)
       {
	SwallowAttributes();
	continue;
       }

     /* condense whitespace */
     if (Token == WHITESPACE)
       {
	SkipWhiteSpace();
	if (LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = ' ';

	continue;
       }

     if (Token == PCDATA)
       {
	if (LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = TokenValue;
	continue;
       }

     /* unexpected tag so terminate element */

     UnGetToken();
     break;
    }

 { /* Eat trailing spaces */
  char* p = LineBuf + LineLen - 1;

  while (*p == ' ' && LineLen) { p--; LineLen--; }
 }

 LineBuf[LineLen] = '\0';

 if ((flags & OPTION_DISABLED) && (flags & OPTION_SELECTED))
   {
#ifdef ARENA_DEBUG
    if (PARSEHTML_TRACE)
      Arena_TracePrint(Iam, " HTML ERROR: option is disabled & selected.\n");
#endif
    Arena_RecordError(CurrentDoc, bufptr, "ERR_OPTION");
   }

 AddOption(field, (flags|font), LineBuf, LineLen, value);

 Free(value);
}



void ParseSelect(Bool implied, Frame* frame, int font, int align,
		 int left, int right)
{
 int nlen, size, flags;
 char *name;
 Field* field;

#ifdef ARENA_DEBUG
 char Iam[] = "ParseSelect";
#endif


 if (ParseEndTag(frame)) return;

 if (implied)
   {
    name = bufptr; /* --Spif 16-Otc-95 click bug */
    nlen = flags = 0;
   }
  else
   {
    ParseSelectAttrs(&name, &nlen, &size, &flags);
   }

 if (form == NULL) form = DefaultForm();

 field = GetField(form, OPTIONLIST, Here,
		  (nlen ? name : bufptr), nlen, NULL, 0,
		  0, 1, 6, 0, (flags|font));

 for (;;)
   {
    while (GetToken() == WHITESPACE);

    if (Token == TAG_SELECT)
      if (ParseEndTag(frame))
	break;

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == TAG_OPTION)
      {
       ParseOption(0, frame, field, font);
       continue;
      }

    if (Token == PCDATA)
      {
       ParseOption(1, frame, field, font);
       continue;
      }

    /* unexpected tag so terminate element */

    ParseHTMLerror |= ERR_SELECT;
    Arena_RecordError(CurrentDoc, bufptr, "ERR_SELECT"); /* howcome 11/10/94 */
    UnGetToken();
    break;
   }    

#ifdef ARENA_DEBUG
 if (WrapFieldIfNeeded(field, frame, align, left, right))
   if (PARSEHTML_TRACE && VERBOSE_TRACE)
     Arena_TracePrint(Iam,
		      " Field "POINTER_FORMAT" (frame = "POINTER_FORMAT")"
		      " wrapped.\n",
		      field, frame);
# else
 WrapFieldIfNeeded(field, frame, align, left, right);
#endif

 PrintInputField(frame, field);

 if (!implied) Here += 2;
 Here += field->width;
}


/*
 * Parses <TEXTAREA>.
 */
Bool ParseTextArea(Frame* theFrame, int theAlign, int theFont,
		   int theLeft, int theRight)
{
#ifdef ARENA_DEBUG
 char Iam[] = "ParseTextArea";
#endif


 if (theFrame)
   {
    Field* theField;
    char* theName;
    char* theValue;
    int theNameL, theValueL, theFlags, theRowsN, theColsN;


    if (ParseEndTag(theFrame)) return True;

    ParseTextAreaAttrs(&theName, &theNameL, &theRowsN, &theColsN, &theFlags);

    /* Now parse <textarea> value (find <textarea></textarea> contents).
     * Strip all trailing white space, but not strip the leading one.
     */
    theValue  = bufptr;
    theValueL = 0;
    for (;;)
      {
       while (GetToken() == WHITESPACE);

       if (Token == ENDDATA)
	 {
	  UnGetToken();
	  break;
	 }

       if (Token == TAG_TEXTAREA)
	 if (ParseEndTag(theFrame))
	   break;

       if (Token == UNKNOWN)
	 {
	  SwallowAttributes();
	 }
        else
	 {
	  if (IsTag(Token))
	    {
	     ParseHTMLerror |= ERR_EMPH;
	     Arena_RecordError(CurrentDoc, bufptr, "ERR_TEXTAREA");
	    }
	 }

       theValueL = bufptr - theValue;
      }
    /* End ``for (;;)'' */

    if (theValueL == 0) theValue = NULL;

    /* what a hack! input outside a form is permitted ???? */
    if (form == NULL) form = DefaultForm();

    theField = GetField(form, TEXTAREA, Here + 1,
			theName, theNameL,
			theValue, theValueL,
			0, theRowsN, theColsN, 0, (theFlags|theFont));

    WrapFieldIfNeeded(theField, theFrame, theAlign, theLeft, theRight);
    PrintInputField(theFrame, theField);

    Here += (theField->width + 2);
    StartOfLine = StartOfWord = bufptr;
    LineLen = LineWidth = WordStart = 0;

    return True;
   }
  else
   {
#ifdef ARENA_DEBUG
    if (PARSEHTML_TRACE)
      Arena_TracePrint(Iam, " ERROR? No frame given! (theFrame is NULL)\n");
#endif
    return False;
   }
}


void ParseEmph(Frame* frame, int align, unsigned int emph, int font,
	       int left, int right)
{
 int ThisToken, WordLen, hreflen, namelen, nlen, vlen, maxlength = 0, size,
     valign, up, down, suboffset, type, delta, ismap,
     width = 0, height = 0, flags, indent = 0; /* janet: not used: h */
 char *href, *name, *value, *p;
 Image* image;
 Field* field;
 Bool element_started = False;
#ifdef ARENA_DEBUG
 char Iam[] ="ParseEmph";
#endif


 if (ParseEndTag(frame)) return;

 suboffset = 0;
 ThisToken = Token;

#if 0
 /* we only want to mark anchors that are hrefs, so we'll have to
  * wait until we parse the anchor attributes to determine this
  */
  if (ThisToken != TAG_ANCHOR)
    style = FormatElementStart(CurrentDoc, ThisToken, class, class_len)
#endif

 switch (ThisToken)
   {
   case TAG_ANCHOR:
     ParseAnchorAttrs(&href, &hreflen, &name, &namelen, &class, &class_len);
     if (href)
       {
	emph |= EMPH_ANCHOR;
	FormatElementStart(CurrentDoc, Token, class, class_len);
	element_started = True;
       }

     /* I'm relatively certain that this test (and code attached) NOT used */
     if (name && targetId)           /* FIXME */
       {
	if (Arena_StrLen(targetId) == namelen &&
	    strncasecmp(name, targetId, namelen) == 0)
	  {
	   IdOffset = PixOffset;
	  }
       }

     /* Keep track of named anchors! (e.g. <a NAME="#Info"...
      * Keep a linked list of all the name= anchors in current document.
      */
     if (name)
       {
	FragmentAnchor* fa = Arena_MAlloc(sizeof(FragmentAnchor), True);
	if (CurrentDoc == NULL)
	  {
#ifdef ARENA_DEBUG
	   Arena_TracePrint(Iam, " NULL CurrentDoc for NAME=\"%s\".\n", name);
# else
	   Arena_PrintError(_("NULL Current Doc for NAME=\"%s\".\n"), name);
#endif
	   Exit(1);
	  }
	fa->Name = name;
	fa->NameLen = namelen;
	fa->NamePixelsFromTop = PixOffset;

	/* If the fragment pointer is not already an HTList*  MAKE IT!       */
	if (CurrentDoc->FragmentPtr == NULL)
	  CurrentDoc->FragmentPtr = HTList_new();

	/* Add our magic name and location to the current doc info           */
	HTList_addObject(CurrentDoc->FragmentPtr, (void *)fa);
	CurrentDoc->FragmentCounter++;
       } /* end of if (name)   */

     /* Subsequently contigous white space is compressed to a single space */
     if (!preformatted) SqueezeWhiteSpace();

     break;
     /* end of `case TAG_ANCHOR' */

   case TAG_MARGIN:
     SwallowAttributes();
     font = IDX_H2FONT;
     indent = XTextWidth(Fonts[IDX_NORMALFONT], "mmm", 3);
     left += indent;
     right -= indent;
     EndOfLine(frame, align);
     Here = left;
     PixOffset += LineSpacing[font];
     break;
     
   case TAG_TT:
   case TAG_CODE:
   case TAG_SAMP:
   case TAG_KBD:
     ParseClassAttrs(&class, &class_len);
     FormatElementStart(CurrentDoc, ThisToken, class, class_len);
     element_started = True;
     font = StyleGet(CurrentDoc, ArenaSI_FONT);
     /*
       if (font == IDX_BNORMALFONT)
       font = IDX_BIFIXEDFONT;
       else
       font = IDX_FIXEDFONT;
       */
     break;

   case TAG_SMALL:
     ParseClassAttrs(&class, &class_len);
     FormatElementStart(CurrentDoc, ThisToken, class, class_len);
     element_started = True;
     font = (int)StyleGet(CurrentDoc, ArenaSI_FONT);
     break;

   case TAG_SUP:
     ParseClassAttrs(&class, &class_len);
     FormatElementStart(CurrentDoc, ThisToken, class, class_len);
     element_started = True;
     font = (int)StyleGet(CurrentDoc, ArenaSI_FONT);
     
     suboffset = ASCENT(font)/2;
     voffset += suboffset;
     above = max(above, voffset + ASCENT(font));
     break;
     
   case TAG_SUB:
     ParseClassAttrs(&class, &class_len);
     FormatElementStart(CurrentDoc, ThisToken, class, class_len);
     element_started = True;
     font = (int)StyleGet(CurrentDoc, ArenaSI_FONT);
     
     suboffset = -ASCENT(font)/2;
     voffset += suboffset;
     below = max(below, DESCENT(font) - voffset);
     break;
     
   case TAG_MATH:
     ParseClassAttrs(&class, &class_len);
     FormatElementStart(CurrentDoc, ThisToken, class, class_len);
     element_started = True;
     font = (int)StyleGet(CurrentDoc, ArenaSI_FONT);
     /*	    left += style->indent;*/
     
     FlushLine(False, frame, align, emph, font, left, right);
     /*            font = IDX_IFIXEDFONT;*/
     
     /*	    StylePush(style);*/
     ParseMath(frame, &up, &down);
     /*	    style = StylePop();*/
     
     above = max(above, up);
     below = max(below, down);
     
     if (Here > TextLineWidth)
       TextLineWidth = Here;
     
     if (element_started)
       FormatElementEnd(CurrentDoc);
     return;
     
   case TAG_ITALIC:
   case TAG_EM:
   case TAG_DFN:
   case TAG_CITE:
   case TAG_VAR:
   case TAG_Q:
   case TAG_BOLD:
   case TAG_STRONG:
     ParseClassAttrs(&class, &class_len);
     FormatElementStart(CurrentDoc, ThisToken, class, class_len);
     element_started = True;
     font = (int)StyleGet(CurrentDoc, ArenaSI_FONT);
     break;
     
   case TAG_UNDERLINE:
     SwallowAttributes();
     emph |= EMPH_UNDERLINE;
     break;
     
   case TAG_STRIKE:
   case TAG_REMOVED:  /* <removed> doesn't work across <P> etc. */
     SwallowAttributes();
     emph |= EMPH_STRIKE;
     break;
     
   case TAG_ADDED:  /* doesn't work across <P> etc! */
     SwallowAttributes();
     emph |= EMPH_HIGHLIGHT;

     ParseClassAttrs(&class, &class_len);
     FormatElementStart(CurrentDoc, ThisToken, class, class_len);
     element_started = True;
     font = (int)StyleGet(CurrentDoc, ArenaSI_FONT);
     /*
       if (preformatted)
       {
       if (font == IDX_BFIXEDFONT)
       font = IDX_BIFIXEDFONT;
       else
       font = IDX_IFIXEDFONT;
       }
       else
       {
       if (font == IDX_BNORMALFONT)
       font = IDX_BINORMALFONT;
       else
       font = IDX_INORMALFONT;
       }
       */
     break;

   case TAG_FORM: /* --Spif */
     name = NULL;
     {
      FormSubmitMethod theMethod = GET;
      
      ParseFormAttrs(&theMethod, &name, &nlen);
      if ((form = FindFormByAction(name)) == NULL)
	{
	 form = GetForm(theMethod, name, nlen);
	}
     }
     break;

   case TAG_CENTER: /* --rhw */
     prealign = True;
     align = ALIGN_CENTER;
     ++bufptr;
     break;

   case TAG_TEXTAREA:  /* a short term bodge */
     element_started = True;
     FormatElementStart(CurrentDoc, ThisToken, NULL, 0);
     ParseTextArea(frame, align, font, left, right);
     if (element_started) FormatElementEnd(CurrentDoc);
     return;

   case TAG_ISINDEX_DE:
     /* OK!  <ISINDEX> within the <body> is a single text input field
      * that will resubmit the current document with the field value as
      * the single argument.  Seems to be used only by cgi programmers.
      * At this time, the Attrs should only allow Lang=  Dir=  Action=
      * and Prompt=     For now, I'm cheating... just assume FORM attributes
      * can be entered!  All we're doing here is creating a form anyway!
      * Most of this section was duplicated from TAG_TEXTAREA and TAG_FORM
      * PLEASE NOTE: The string "<ISINDEX>" with a length of 9 is used as the
      * name of the field in our very simple form. But that is form submit
      * method (our homebrewed private ISINDEX) which is used in forms.c
      * to handle argument formatting differently...
      */
     FormatElementStart(CurrentDoc, ThisToken, NULL, 0); /* field */
     element_started = True;
      {
       char* thePrompt = NULL;
       char* theAction = NULL;
       char* theValue  = NULL;
       int thePromptL = 0, theActionL = 0, theValueL = 0;
       char* prompt = NULL;
       int prompt_l = 0;


       ParseIsindexAttrs(&thePrompt, &thePromptL,
			 &theAction, &theActionL,
			 &theValue,  &theValueL);
       if (thePrompt)
	 {
	  if (thePromptL)
	    {
	     prompt   = thePrompt;
	     prompt_l = thePromptL;
	    }
	 }
        else
	 {
	  prompt   = "This is a searchable index. Enter search keywords: ";
	  prompt_l = -1;
	 }

       if (prompt)
	 PrintPhrase(frame, align, emph, font, left, right, prompt, prompt_l);

       if ((form = FindForm(ISINDEX, theAction)) == NULL)
	 {
         /* see NOTE: */
	  form = DefaultForm();
	  form->method = ISINDEX;
	  if (theAction && theActionL)
	    {
	     form->action = theAction;
	     form->alen   = theActionL;
	    }
	 }

      field = GetField(form, TEXTFIELD, Here + 1,
		       "<ISINDEX>", 9,
		       theValue, theValueL,
		       0, 1, 32, 64, (0|font));
      }
      WrapFieldIfNeeded(field, frame, align, left, right);
      PrintInputField(frame, field);
      Here += field->width + 2;
      if (element_started) FormatElementEnd(CurrentDoc);
      return;

    case TAG_INPUT:   /* --Spif src=img ? */
      ParseInputAttrs(&type, &name, &nlen, &value, &vlen, &maxlength, &size,
		      &flags, &image);

      if (form == NULL) form = DefaultForm();

      if (type == HIDDEN) /* --Spif don't show hidden fields */
        { /* add hidden input field */
	 field = GetField(form, type, Here + 1, name, nlen, value, vlen,
			  0, 1, size, maxlength, (flags|font));
        }
       else
        {
	 FormatElementStart(CurrentDoc, ThisToken, NULL, 0);
	 element_started = True;

	 if (type == SUBMITBUTTON || type == RESETBUTTON)
	   /* font = IDX_H3FONT; */
	   font = (int)StyleGet(CurrentDoc, ArenaSI_FONT);

	 /* --Spif image only on submit and reset buttons... */    
	 if ((type == SUBMITBUTTON || type == RESETBUTTON) && (image != NULL)) 
	   {
	    int above1, below1;

	    p = bufptr;
	    while (*p != '<') p--;
	    width  = image->width  + 8; /* +8 because of the emph_whatever. */
	    height = image->height + 8;
	    if (! (width && height))
	      {
	       width  = IMG_WIDTH;
	       height = IMG_HEIGHT;
	      }

	    emph |= EMPH_INPUT; /* --Spif hack */
	    delta = height;
	    above1 = delta + 2;
	    below1 = 0;
	    field = GetField(form, type, Here + 1, name, nlen, value, vlen,
			     0, 1, size, maxlength, (flags|font));
	    WrapImageIfNeeded(frame, align, left, right, width,
			      above1, below1);
	    PrintImage(frame, delta, emph, p, image, width, height);
	    Here += width + 2; /* check that with emph... */ 
	   }
	  else
	   {
	        /* test if no name and/or value tags --SPif 12-Oct-95 */
	    if (type == SUBMITBUTTON || type == RESETBUTTON)
	      {
	       if (nlen)
		 field = GetField(form, type, Here + 1,
				  name, nlen, value, vlen,
				  0, 1, size, maxlength, (flags|font));
	        else
		 if (vlen)
		   field = GetField(form, type, Here + 1,
				    value, vlen, value, vlen,
				    0, 1, size, maxlength,
				    (flags|font)); /* value as id */
		  else  /* do this because name is not necessary in
			   submit & reset buttons */
		   if (type == SUBMITBUTTON)
		     field = GetField(form, type, Here + 1,
				      bufptr, 14 ,
				      " Submit Query ", 14,
				      0, 1, size, maxlength, (flags|font));
		    else  /* --Spif 11-Oct-95 it's ugly but it works */
		     field = GetField(form, type, Here + 1,
				      bufptr, 7, " Reset ", 7,
				      0, 1, size, maxlength, (flags|font)); 
	      }
	     else
	      field = GetField(form, type, Here + 1, name, nlen,
			       value, vlen, 0, 1, size, maxlength,
			       (flags|font));

	    WrapFieldIfNeeded(field, frame, align, left, right);
	    PrintInputField(frame, field);
	    Here += field->width + 2;
	   };

	 StartOfLine = StartOfWord = bufptr;
	 LineLen = LineWidth = WordStart = 0;
	 if (element_started) FormatElementEnd(CurrentDoc);
        }

      return;

    case TAG_SELECT:
        ParseSelect(0, frame, font, align, left, right);
        StartOfLine = StartOfWord = bufptr;
        LineLen = LineWidth = WordStart = 0;
        if (element_started) FormatElementEnd(CurrentDoc);
        return;

    case TAG_OPTION:
        ParseSelect(1, frame, font, align, left, right);
        StartOfLine = StartOfWord = bufptr;
        LineLen = LineWidth = WordStart = 0;
        if (element_started) FormatElementEnd(CurrentDoc);
        return;

    case TAG_BR:
        ParseParaAttrs(&align, &class, &class_len);
        if (paintStartLine >= 0)
	  EndOfLine(frame, align);
         else
	  PixOffset += ASCENT((int)StyleGet(CurrentDoc, ArenaSI_FONT)) +
	              DESCENT((int)StyleGet(CurrentDoc, ArenaSI_FONT));

        Here = left;
        StartOfLine = StartOfWord = bufptr;
        LineLen = LineWidth = WordStart = 0;
        if (element_started) FormatElementEnd(CurrentDoc);
        return;

    case TAG_IMG:
      p = bufptr - 4;
      ParseImageAttrs(&href, &hreflen, &valign, &ismap, &width, &height);

      if ((image = GetImage(href, hreflen, CurrentDoc->pending_reload)))
	{
	 if ( width == 0)  width = image->width;
	 if (height == 0) height = image->height;
        }
       else
	{
	 if ( width == 0)  width = IMG_WIDTH;
	 if (height == 0) height = IMG_HEIGHT;
	}

      if (ismap | (emph & EMPH_ANCHOR))
        {
	 if (ismap) emph |= ISMAP;

	 width  += 8;
	 height += 8;
        }
       else
	p = NULL;

      {
       int above1, below1;

       if (valign == ALIGN_BOTTOM)
	 {
	  delta = height;
	  above1 = delta + 2;
	  below1 = 0;
	  WrapImageIfNeeded(frame, align, left, right, width, above1, below1);
	  PrintImage(frame, delta, emph, p, image, width, height);
	 }
        else
	 if (valign == ALIGN_MIDDLE)
	   {
	    delta = height/2;
	    above1 = delta + 2;
	    below1 = (above1 < height ? height - above1 : 0);
	    WrapImageIfNeeded(frame, align, left, right, width, above1, below1);
	    PrintImage(frame, delta, emph, p, image, width, height);
	   }
	  else  /* ALIGN_TOP */
	   {
	    delta = ASCENT(font) - 2;
	    below1 = 2 + height - delta;

	    if (below1 < 0) below1 = 0;

	    above1 = 2 + (height > below1 ? height - below1 : 0);

	    WrapImageIfNeeded(frame, align, left, right, width, above1, below1);
	    PrintImage(frame, delta, emph, p, image, width, height);
	   }

       Here += width + 2;
      }

      StartOfLine = StartOfWord = bufptr;
      LineLen = LineWidth = WordStart = 0;

      if (element_started) FormatElementEnd(CurrentDoc);
      return;

    default:
      SwallowAttributes();
      break;
   }

 StartOfLine = StartOfWord = bufptr;
 LineLen = LineWidth = WordStart = 0;

 if (ThisToken == TAG_Q)
   {
   /* open quote char  */
   /* width = XTextWidth(Fonts[IDX_NORMALFONT], "``", 2);*/
   /* PrintSeqText(frame, EMPH_NORMAL, IDX_NORMALFONT, "``", width);*/
    width = XTextWidth(Fonts[(int)StyleGet(CurrentDoc, ArenaSI_FONT)], "``", 2);
    PrintSeqText(frame, EMPH_NORMAL, (int)StyleGet(CurrentDoc, ArenaSI_FONT), "``", width);
    Here += width;
   }

 for (;;)
   {
    GetToken();

    if (Token == ThisToken)
      if (ParseEndTag(frame))
	break;

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (TokenClass != EN_TEXT)
      {
       if (Token != TAG_HR)
	 {
	  ParseHTMLerror |= ERR_EMPH;
	  Arena_RecordError(CurrentDoc, bufptr, "ERR_EMPH"); /* howcome 11/10/94 */
	 }

       UnGetToken();
       break;
      }

    if (Token == WHITESPACE)
      {
       if (preformatted)
	 {
	  if (TokenValue == '\n')
	    FlushLine(True, frame, align, emph, font, left, right);
	   else
	    if (LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = ' ';
	 }
        else
	 {
	  SkipWhiteSpace();
	  if (Here == left && LineLen == 0)
	    {
	     StartOfLine = StartOfWord = bufptr;
	     continue;
	    }

	  /* check that we have a word */

	  if ((WordLen = LineLen - WordStart) > 0)
	    WrapIfNeeded(frame, align, emph, font, left, right);

	  if (LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = ' ';

	  WordStart = LineLen;
	  StartOfWord = bufptr;
	 }

       continue;
      }

    if (IsTag(Token))
      {
       LineBuf[LineLen] = '\0';
       FlushLine(False, frame, align, emph, font, left, right);

       ParseEmph(frame, align, emph, font, left, right);
       StartOfLine = StartOfWord = bufptr;
       LineLen = LineWidth = WordStart = 0;
       continue;
      }

    /* must be PCDATA */
    if (LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = TokenValue;
   }

 LineBuf[LineLen] = '\0';
 FlushLine(False, frame, align, emph, font, left, right);

 voffset -= suboffset;

 if (ThisToken == TAG_Q)
   {
   /* close quote char  */
   /* width = XTextWidth(Fonts[IDX_NORMALFONT], "''", 2);*/
   /* PrintSeqText(frame, IDX_NORMALFONT, "''", width);*/
    width = XTextWidth(Fonts[(int)StyleGet(CurrentDoc, ArenaSI_FONT)], "''", 2);
    PrintSeqText(frame, EMPH_NORMAL, (int)StyleGet(CurrentDoc, ArenaSI_FONT), "''", width);
    Here += width;
   }

 if (ThisToken == TAG_MARGIN)
   {
    EndOfLine(frame, align);
    Here = left -  indent;
    PixOffset += LineSpacing[font];
    StartOfLine = StartOfWord = bufptr;
    LineLen = LineWidth = WordStart = 0;
   }

/*    style = StylePop();*/

/*    style = FormatElementStart(CurrentDoc, revert_tag, NULL, 0); *//* reset to P style after emph */

 if (element_started) FormatElementEnd(CurrentDoc);
}


void ParseHeader(Frame* frame, int align, int left, int right, int outdent)
{
 int HeaderTag, WordLen, emph;


 if (EndTag)
   {
    SwallowAttributes();
    return;
   }

 align = -1;
 ParseParaAttrs(&align, &class, &class_len);

 FormatElementStart(CurrentDoc, Token, class, class_len);
 font = (int)StyleGet(CurrentDoc, ArenaSI_FONT); 
 if (align < 0) align = StyleGet(CurrentDoc, ArenaSI_TEXT_ALIGN);

  left = (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT);
 right = (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_RIGHT);

 /* howcome: applied patch from dsr 18/11/94 */
 if (frame->leftmargin <= MININDENT) left -= outdent;

 HeaderTag  = Token;
 Here = left;

/*
 switch (HeaderTag)
   {
    case TAG_H1:
      font = IDX_H1FONT;
      break;

    case TAG_H2:
      font = IDX_H2FONT;
      break;

    case TAG_H3:
    case TAG_H4:
      font = IDX_H3FONT;
      break;

    default:
      font = IDX_H4FONT;
      break;
   }
 */

 PixOffset += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_TOP);

 emph = EMPH_NORMAL;

 StartOfLine = StartOfWord = bufptr;
 LineLen = LineWidth = WordStart = 0;

 for (;;)
   {
    GetToken();

    if (Token == HeaderTag)
      if (ParseEndTag(frame))
        break;

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (TokenClass != EN_TEXT)
      {
       ParseHTMLerror |= ERR_HEADER;
       /* howcome 11/10/94 */
       Arena_RecordError(CurrentDoc, bufptr, "ERR_HEADER");
       UnGetToken();
       break;
      }

    if (Token == WHITESPACE)
      {
       SkipWhiteSpace();
       if (Here == left && LineLen == 0)
	 {
	  StartOfLine = StartOfWord = bufptr;
	  continue;
	 }

       /* check that we have a word */

       if ((WordLen = LineLen - WordStart) > 0)
	 WrapIfNeeded(frame, align, emph,
		      (int)StyleGet(CurrentDoc, ArenaSI_FONT),
		      left, right);

       if (LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = ' ';

       WordStart = LineLen;
       StartOfWord = bufptr;
       continue;
      }

    if (IsTag(Token))
      {
       if (ParseEndTag(frame)) continue;

       FlushLine(False, frame, align, emph,
		 (int)StyleGet(CurrentDoc, ArenaSI_FONT),
		 left, right);
       ParseEmph(frame, align, emph, font /* StyleGet(CurrentDoc, ArenaSI_FONT) */,
		 left, right);
#if 0
       /* howcome 30/2/95: revert style after emph section */
       if (Token != TAG_BR && Token !=TAG_HR)
         FormatElementStart(CurrentDoc, HeaderTag, class, class_len);
#endif
       continue;
      }

    /* must be PCDATA */
    if (LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = TokenValue;
   }

 LineBuf[LineLen] = '\0';
 FlushLine(True, frame, align, emph, (int)StyleGet(CurrentDoc, ArenaSI_FONT),
	   left, right);
 /* PixOffset += LineSpacing[StyleGet(CurrentDoc, SF_ONT)]/2;*/
 PixOffset += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_BOTTOM);

#if 0
 FormatElementStart(CurrentDoc, TAG_P, NULL, 0); /* reset to P style after headline */
 align = StyleGet(CurrentDoc, ArenaSI_TEXT_ALIGN); /* dummy call to force
				      the initialization of last_value array */
#endif

 FormatElementEnd(CurrentDoc);
}


void ParsePara(Bool implied, Bool initial, Frame* frame, int palign, int tag,
	       int left, int right)
{
 int WordLen, emph;
 int font;
 int align = -1;
 Frame* new_frame;
 int hack = 0;


 if (ParseEndTag(frame)) return;

 hack = (paintStartLine > 0) || damn_table;

 if (!implied) ParseParaAttrs(&align, &class, &class_len);

 FormatElementStart(CurrentDoc, tag, class, class_len);

 if (align < 0) align = (int)StyleGet(CurrentDoc, ArenaSI_TEXT_ALIGN);
 if (initial) left = (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT);

 font = (int)StyleGet(CurrentDoc, ArenaSI_FONT);
 right = (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_RIGHT);

 if (Here < left) Here = left;

 if (!implied && !hack)
   {
    new_frame = (Frame*)Arena_CAlloc(1, sizeof(Frame), False);
    PixOffset += (int)StyleGet(CurrentDoc, ArenaSI_PADDING_TOP) +
                 (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_TOP); /* must be S_PADDING_TOP,
						       righT! */
    new_frame->offset = PixOffset;
    new_frame->leftmargin  = StyleGet(CurrentDoc, ArenaSI_PADDING_LEFT);
    new_frame->rightmargin = StyleGet(CurrentDoc, ArenaSI_PADDING_RIGHT);
#if 0
    right + frame->rightmargin/* + StyleGet(CurrentDoc, ArenaSI_MARGIN_RIGHT)*/;
#endif
    new_frame->indent = frame->indent + left; /* S_PADDING_LEFT */
    new_frame->width = frame->width - left - right;
    new_frame->style = 0;
    new_frame->border = 0;
#ifdef STYLE_COLOUR_BORDER
    new_frame->cb_ix = 0;
#endif
    new_frame->flow = align;
    new_frame->next = new_frame->child = NULL;
    new_frame->box_list = NULL;
    CopyBoxList(new_frame, frame);
    new_frame->leftcount = frame->leftcount;
    new_frame->pushcount = frame->pushcount;
    new_frame->oldmargin = frame->oldmargin;
    PrintBeginFrame(new_frame); 
    left  = (int)StyleGet(CurrentDoc, ArenaSI_PADDING_LEFT);  /* S_PADDING_LEFT */
    right = (int)StyleGet(CurrentDoc, ArenaSI_PADDING_RIGHT); /* S_PADDING_RIGHT */
    Here = left;
    Push(new_frame);
   }
  else
   new_frame = frame;

 emph = EMPH_NORMAL;

 /* squeeze leading white space ---
  * subsequently contigous white space is compressed to a single space
  */
 SqueezeWhiteSpace();

 StartOfLine = StartOfWord = bufptr;
 LineLen = LineWidth = WordStart = 0;

 for (;;)
   {
    GetToken();

    if (Token == TAG_P)
      if (ParseEndTag(frame))
	break;

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (TokenClass != EN_TEXT)
      {
       UnGetToken();
       break;
      }

    if (Token == WHITESPACE)
      {
       SkipWhiteSpace();

       if (Here == left && LineLen == 0)
	 {
	  StartOfLine = StartOfWord = bufptr;
	  continue;
	 }

       /* check that we have a word */

       if ((WordLen = LineLen - WordStart) > 0)
	 WrapIfNeeded(new_frame, align, emph, font, left, right);

       if (LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = ' ';

       WordStart = LineLen;
       StartOfWord = bufptr;
       continue;
      }

    if (IsTag(Token))
      {
       if (ParseEndTag(frame))
	 {
	  ParseHTMLerror |= ERR_EMPH;
	  /* howcome 24/11/94 */
	  Arena_RecordError(CurrentDoc, bufptr, "ERR_EMPH");
	 }
        else
	 {
	  LineBuf[LineLen] = '\0';
	  FlushLine(False, frame, align, emph, font, left, right);
	  ParseEmph(new_frame, align, emph, font, left, right);

	  /* howcome 28/10/95:
	   * added to reset buffer after math -- does it break anything?
	   */
	  StartOfLine = StartOfWord = bufptr;
#if 0
	  /* howcome 30/2/95: revert style after emph section */
	  if (Token != TAG_BR && Token !=TAG_HR)
	    FormatElementStart(CurrentDoc, revert_tag, class, class_len);
#endif
	 }

       continue;
      }

    /* must be PCDATA */
    if (LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = TokenValue;
    }

 LineBuf[LineLen] = '\0';
 FlushLine(True, new_frame, align, emph, font, left, right);

 if (!implied && !hack)
   {
    unsigned char *p;

    Pop();
    new_frame->height = PixOffset - new_frame->offset +
                        StyleGet(CurrentDoc, ArenaSI_PADDING_TOP) +
                        StyleGet(CurrentDoc, ArenaSI_PADDING_BOTTOM);
    new_frame->offset -= (int)StyleGet(CurrentDoc, ArenaSI_PADDING_TOP); /* ?? */
    PixOffset += (int)StyleGet(CurrentDoc, ArenaSI_PADDING_TOP);  /* ?? */
    new_frame->indent -= (int)StyleGet(CurrentDoc, ArenaSI_PADDING_LEFT);  /* ?? */
    p = paint + new_frame->info + ARENA_PAINTSTREAM_TAG_LEN;
    PutLong(p, new_frame->offset);
    PutInt(p, new_frame->indent);
    PutInt(p, new_frame->width);
    PutLong(p, new_frame->height);
    PrintFrameSize(new_frame);
    PrintEndFrame(frame, new_frame); 
    FreeFrames(new_frame);
    new_frame = NULL;
   }

 PixOffset += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_BOTTOM);
 FormatElementEnd(CurrentDoc);
}


void ItemNumber(char *buf, int depth, int n)
{
 int w, ones, tens, hundreds, thousands;
 char *p, *q;

 w = depth % 3;

 if (w == 0)
   sprintf(buf, "%d.", n);
  else
   if (w == 1)
     {
      thousands = n/1000;
      n = n % 1000;
      hundreds = n/100;
      n = n % 100;
      tens = n/10;
      ones = n % 10;

      p = buf;

      while (thousands-- > 0) *p++ = 'm';

      if (hundreds)
        {
	 q = Hundreds[hundreds-1];
	 while ((*p++ = *q++));
	 --p;
        }

      if (tens)
        {
	 q = Tens[tens-1];
	 while ((*p++ = *q++));
	 --p;
        }

      if (ones)
        {
	 q = Ones[ones-1];
	 while ((*p++ = *q++));
	 --p;
        }

      *p++ = ')';
      *p = '\0';
     }
   else
     sprintf(buf, "%c)", 'a' + (n-1)%26);
}


void ParsePRE(Bool implied, Bool initial, Frame* frame, int left, int right)
{
 int ch, n, emph, font = 0, lastTokenValue;
 int pre_align = (int)StyleGet(CurrentDoc, ArenaSI_TEXT_ALIGN);

 if (ParseEndTag(frame)) return;

 FormatElementStart(CurrentDoc, Token, class, class_len);

 if (!implied)
   {
  /* SwallowAttributes();*/

    ParseClassAttrs(&class, &class_len);
  /*
     StyleRegisterSequential(Token, class, class_len);
     StyleRegisterHierarchical(Token, class, class_len);
   */
  /* FormatElementStart(CurrentDoc, Token, class, class_len);*/
    font = (int)StyleGet(CurrentDoc, ArenaSI_FONT);
   }


 if (initial) left = (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT);

 Here = left;

 preformatted = 1;
 emph = EMPH_NORMAL;
/* font = IDX_FIXEDFONT;*/
 ch = CHWIDTH(font);

/* skip leading newline */

 if (*bufptr == '\r') ++bufptr;
 if (*bufptr == '\n') ++bufptr;

/* howcome 12/10/94:
   moved the two lines below from before the ++bufptr statements */

 StartOfLine = StartOfWord = bufptr;
 LineLen = LineWidth = WordStart = 0;

 for (;;)
    {
     lastTokenValue = TokenValue;
     GetToken();

     if (Token == TAG_PRE)
       if (ParseEndTag(frame))
	 break;

     if (Token == UNKNOWN)
       {
	SwallowAttributes();
	continue;
       }

     /* HR isn't really allowed in PRE, but ... */
     if (Token == TAG_HR)
       {
	LineBuf[LineLen] = '\0';
	FlushLine(True, frame, pre_align, emph, font, left, right);
	/* SwallowAttributes();*/

	if (EndTag) continue;

	ParseClassAttrs(&class, &class_len);
	FormatElementStart(CurrentDoc, Token, class, class_len);

	PrintRule(frame, GROOVE, left, right, 0);
	EndOfLine(frame, pre_align);

	FormatElementEnd(CurrentDoc);
	continue;
       }

     /* P isn't really allowed in PRE, but ... */
     if (Token == TAG_P)
       {
	LineBuf[LineLen] = '\0';
	FlushLine(True, frame, pre_align, emph, font, left, right);
	/* there were two lines here... suspect a bug */
	SwallowAttributes();
	continue;
       }

     /* Ignore header tags in preformatted text */

     if (TokenClass == EN_HEADER)
       {
	ParseHTMLerror |= ERR_PRE;
	Arena_RecordError(CurrentDoc, bufptr, "ERR_PRE"); /* howcome 11/10/94 */
	SwallowAttributes();
	continue;
       }

     if (TokenClass != EN_TEXT)
       {
	ParseHTMLerror |= ERR_PRE;
	Arena_RecordError(CurrentDoc, bufptr, "ERR_PRE"); /* howcome 11/10/94 */
	UnGetToken();
	break;
       }

     if (Token == WHITESPACE && TokenValue == '\n')
       {
	LineBuf[LineLen] = '\0';
	FlushLine(True, frame, pre_align, emph, font, left, right);
	continue;
       }

     if (IsTag(Token))
       {
	if (ParseEndTag(frame)) continue;

	LineBuf[LineLen] = '\0';
	FlushLine(False, frame, pre_align, emph, font, left, right);
	ParseEmph(frame, ALIGN_LEFT, emph, font, left, right);

	/* howcome 15/3/95: try to align characters in pre mode */
	if (Token == TAG_ANCHOR)  /* emph & EMPH_ANCHOR)*/
	  Here -= 2;
/*
        if (Token != TAG_BR && Token !=TAG_HR)
          style = FormatElementStart(CurrentDoc, s, class, class_len);
*/
	continue;
       }

     /* must be PCDATA */

     /* CopyLine can't work out how many spaces to use! */
     if (TokenValue == '\t')
       {
	n = LineLen;  /* + (Here - left)/ch; */
	n = 8 - n % 8;

	while (n-- > 0 && LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = ' ';
	continue;
       }

     if (LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = TokenValue;
    }

 LineBuf[LineLen] = '\0';
 FlushLine((lastTokenValue == '\n'), frame, pre_align, emph, font,
	   left, right);

 if (paintStartLine >= 0) EndOfLine(frame, pre_align);

 PixOffset += LineSpacing[font]/3; /* fix whitespace bug dsr 16-Nov-94 */ /* howcome applied patch 23/1/94 */
 preformatted = 0;

/* FormatElementStart(CurrentDoc, TAG_P, NULL, 0);*/ /* reset to P style after PRE */

 FormatElementEnd(CurrentDoc);
}



/* advance declaration due for nested lists */
void ParseUL(Bool implied, Bool initial, Frame* frame, int depth, int align,
	     int left, int right);
void ParseOL(Bool implied, Bool initial, Frame* frame, int depth, int align,
	     int left, int right);
void ParseDL(Bool implied, Bool initial, Frame* frame, int left, int align,
	     int right);


void ParseLI(Bool implied, Frame* frame, int depth, int seq, int align,
	     int left, int right)
{
 int indent, w;
 long y;
 char buf[16];


 if (ParseEndTag(frame)) return;

 y = PixOffset;

 FormatElementStart(CurrentDoc, TAG_LI, class, class_len);

 if (!implied)
   {
    ParseClassAttrs(&class, &class_len);
    /* FormatElementStart(CurrentDoc, Token, class, class_len);*/
   }

 indent = (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT);


 w = left + indent/3;

 if (w < Here + 20) EndOfLine(frame, ALIGN_LEFT);

 Here = w;

 if (seq > 0)
   {
    ItemNumber(buf, depth++, seq);
    w = XTextWidth(Fonts[(int)StyleGet(CurrentDoc, ArenaSI_FONT)], buf, Arena_StrLen(buf));
    PrintSeqText(frame, EMPH_NORMAL, (int)StyleGet(CurrentDoc, ArenaSI_FONT), buf, w);

    if (w + indent/3 > indent - 4) indent = 4 + w + indent/3;
   }
  else
   PrintBullet(frame, depth++,
	       (int)StyleGet(CurrentDoc, ArenaSI_FONT),
	       (int)StyleGet(CurrentDoc, ArenaSI_FONT_SIZE));

 if ((w = left + indent) > list_indent) list_indent = w;    /* for tables */

 for (;;)
   {
    while (GetToken() == WHITESPACE);
    /* style = FormatElementStart(CurrentDoc, Token, class, class_len); *//* howcome 15/3/94 */

    if (Token == TAG_LI)
      if (ParseEndTag(frame))
	break;

    if (Token == TAG_UL)
      {
       UnGetToken();

       if (EndTag) break;

       ParseUL(0, False, frame, depth, align, left + indent, right);
       continue;
      }

    if (Token == TAG_OL)
      {
       UnGetToken();

       if (EndTag) break;

       ParseOL(0, False, frame, depth, align, left + indent, right);
       continue;
      }

    if (Token == TAG_DL)
      {
       UnGetToken();

       if (EndTag) break;

       ParseDL(0, False, frame, align, left + indent, right);
       continue;
      }

    if (Token == TAG_DT || Token == TAG_DD)
      {
       UnGetToken();
       ParseDL(1, False, frame, align, left + indent, right);
       continue;
      }

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == TAG_P)
      {
       Here = left + indent;
       /* StyleSetFlag(CurrentDoc, S_INDENT_FLAG,True);
	  StyleSetFlag(CurrentDoc, S_LEADING_FLAG,True);*/
       /* PixOffset += style->margin_top; *//* howcome 15/3/95:
	                                       not a good idea */
       ParsePara(0, False, frame, align, TAG_P, left + indent, right);
       continue;
      }

    if (Token == TAG_HR)
      {
       /* SwallowAttributes();*/
       if (EndTag) continue;

       ParseClassAttrs(&class, &class_len);
/*
       StyleRegisterSequential(Token, class, class_len);
*/
       FormatElementStart(CurrentDoc, Token, class, class_len);

       PrintRule(frame, GROOVE,
#if 1	/* QingLong.18-02-97 */
		  LeftMargin(frame,  left + indent, PixOffset),
# else
		 left + indent,
#endif
		 RightMargin(frame,          right, PixOffset),
		 0);

       EndOfLine(frame, (int)StyleGet(CurrentDoc, ArenaSI_TEXT_ALIGN));
       FormatElementEnd(CurrentDoc);
       continue;
      }

    /* howcome 11/8/95: support for PRE added */

    if (Token == TAG_PRE)
      {
       ParsePRE(0, False, frame, left + indent, right);
       continue;
      }

    if (TokenClass == EN_TEXT)
      {
       UnGetToken();
       ParsePara(1, False, frame, align, TAG_P, left + indent, right);
       continue;
      }

    /* unexpected tag so terminate element */

    UnGetToken();
    break;
   }    

 /* kludge to cope with an <LI> element with no content */

 if (y == PixOffset) PixOffset += (4 * LineSpacing[IDX_NORMALFONT])/3;

 FormatElementEnd(CurrentDoc);
}


void ParseUL(Bool implied, Bool initial, Frame* frame, int depth, int align,
	     int left, int right)
{
 if (ParseEndTag(frame)) return;

 FormatElementStart(CurrentDoc, Token, class, class_len);

 if (!implied)
   {
    /* SwallowAttributes();*/
    ParseClassAttrs(&class, &class_len);
/*
    StyleRegisterSequential(Token, class, class_len);
    StyleRegisterHierarchical(Token, class, class_len);
*/
    /* FormatElementStart(CurrentDoc, Token, class, class_len);*/
    left += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT);
   }

 if (initial) left += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT);

 for (;;)
   {
    while (GetToken() == WHITESPACE);
    /* style = FormatElementStart(CurrentDoc, Token, class, class_len); */ /* howcome 15/3/95 */

    if (Token == TAG_UL)
      if (ParseEndTag(frame))
	break;

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == TAG_LI)
      {
       ParseLI(0, frame, depth, 0, align, left, right);
       continue;
      }

    if (Token == TAG_P || Token == TAG_HR || TokenClass == EN_TEXT)
      {
       if (ParseEndTag(frame)) continue;

       UnGetToken();
       ParseLI(1, frame, depth, 0, align, left, right);
       continue;
      }

    if (Token == TAG_UL || Token == TAG_OL ||
	Token == TAG_DL || Token == TAG_DT || Token == TAG_DD)
      {
       if (ParseEndTag(frame)) continue;

       ParseHTMLerror |= ERR_UL;
       Arena_RecordError(CurrentDoc, bufptr, "ERR_UL");

       UnGetToken();
       ParseLI(1, frame, depth, 0, align, left, right);
       continue;
      }

    /* unexpected tag so terminate element */

    ParseHTMLerror |= ERR_UL;
    Arena_RecordError(CurrentDoc, bufptr, "ERR_UL"); /* howcome 11/10/94 */
    UnGetToken();
    break;
   }    

 Here = left;

 FormatElementEnd(CurrentDoc);
}


void ParseOL(Bool implied, Bool initial, Frame* frame, int depth, int align,
	     int left, int right)
{
 int seq;


 if (ParseEndTag(frame)) return;

 FormatElementStart(CurrentDoc, Token, class, class_len);

 if (!implied)
   {
    ParseClassAttrs(&class, &class_len);
/*
    StyleRegisterSequential(Token, class, class_len);
    StyleRegisterHierarchical(Token, class, class_len);
*/
    /* FormatElementStart(CurrentDoc, Token, class, class_len);*/
   }

 if (initial) left += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT);

 seq = 0;

 for (;;)
   {
    ++seq;

    while (GetToken() == WHITESPACE);
    /* style = FormatElementStart(CurrentDoc, Token, class, class_len); *//* howcome 15/3/95 */

    if (Token == TAG_OL)
      if (ParseEndTag(frame))
	break;

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == TAG_LI)
      {
       ParseLI(0, frame, depth, seq, align, left, right);
       continue;
      }

    if (Token == TAG_P || Token == TAG_HR || TokenClass == EN_TEXT)
      {
       if (ParseEndTag(frame)) continue;

       UnGetToken();
       ParseLI(1, frame, depth, seq, align, left, right);
       continue;
      }

    if (Token == TAG_UL || Token == TAG_OL ||
	Token == TAG_DL || Token == TAG_DT || Token == TAG_DD)
      {
       if (ParseEndTag(frame)) continue;

       UnGetToken();
       ParseLI(1, frame, depth, seq, align, left, right);
       continue;
      }

    /* unexpected tag so terminate element */

    ParseHTMLerror |= ERR_OL;
    Arena_RecordError(CurrentDoc, bufptr, "ERR_OL"); /* howcome 11/10/94 */
    UnGetToken();
    break;
   }    

 Here = left;

 FormatElementEnd(CurrentDoc);
}


void ParseDT(Bool implied, Frame* frame, int align,
	     int left, int right)
{
 int WordLen, emph, font;


 if (ParseEndTag(frame)) return;

 FormatElementStart(CurrentDoc, Token, class, class_len);

 if (!implied)
   {
    /* SwallowAttributes();*/
    ParseClassAttrs(&class, &class_len);
   }
/*
  StyleRegisterSequential(Token, class, class_len);
*/

 Here = left;

 emph = EMPH_NORMAL;

 font = (int)StyleGet(CurrentDoc, ArenaSI_FONT);
 /* font = IDX_BNORMALFONT;*/

 /* skip leading white space - subsequently contigous
  * white space is compressed to a single space
  */
 SqueezeWhiteSpace();

 StartOfLine = StartOfWord = bufptr;
 LineLen = LineWidth = WordStart = 0;

 for (;;)
   {
    GetToken();
    /* style = FormatElementStart(CurrentDoc, Token, class, class_len); *//* howcome 15/3/95 */

    if (Token == TAG_DT)
     if (ParseEndTag(frame))
       break;

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (TokenClass != EN_TEXT)
      {
       UnGetToken();
       break;
      }

    if (Token == WHITESPACE)
      {
       SkipWhiteSpace();

       /* check that we have a word */

       if ((WordLen = LineLen - WordStart) > 0)
	 WrapIfNeeded(frame, align, emph, font, left, right);

       if (LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = ' ';

       WordStart = LineLen;
       StartOfWord = bufptr;
       continue;
      }

    if (IsTag(Token))
      {
       if (ParseEndTag(frame)) continue;

       FlushLine(False, frame, align, emph, font, left, right);
       ParseEmph(frame, align, emph, font, left, right);
       /* if (Token != TAG_BR && Token !=TAG_HR)
	  FormatElementStart(CurrentDoc, TAG_P, class, class_len); *//* howcome 6/3/95: should there be a revert_tag here ? */
       continue;
      }

    /* must be PCDATA */

    if (LineLen < LBUFSIZE - 1) LineBuf[LineLen++] = TokenValue;
   }

 LineBuf[LineLen] = '\0';
 FlushLine(False, frame, align, emph, font, left, right);
 Here += 5;

 FormatElementEnd(CurrentDoc);
}


void ParseDD(Bool implied, Frame* frame, int align, int left, int right)
{
 int indent;
 long y;


 if (ParseEndTag(frame)) return;

 FormatElementStart(CurrentDoc, Token, class, class_len);

 if (!implied)
   {
    ParseClassAttrs(&class, &class_len);
   }

 y = PixOffset;
 indent = (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT);

 for (;;)
   {
    while (GetToken() == WHITESPACE);
    /* style = FormatElementStart(CurrentDoc, Token, class, class_len); *//* howcome 15/3/95 */

    if (Token == TAG_DD)
      if (ParseEndTag(frame))
	break;

    if (Token == TAG_DL && EndTag)
      {
       UnGetToken();
       break;
      }

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == TAG_UL)
      {
       ParseUL(0, False, frame, 0, align, left, right);
       continue;
      }

    if (Token == TAG_OL)
      {
       ParseOL(0, False, frame, 0, align, left, right);
       continue;
      }

    if (Token == TAG_LI)
      {
       if (ParseEndTag(frame)) continue;

       UnGetToken();
       ParseUL(1, False, frame, 0, align, left, right);
       continue;
      }

    if (Token == TAG_DL)
      {
       if (PixOffset == y) /* force a line break */
	 {
	  EndOfLine(frame, align);
	  Here = left;
	 }

       ParseDL(0, False, frame, align, left, right);
       Here = left;
       continue;
      }

    if (Token == TAG_HR)
      {
       if (EndTag) continue;

       ParseClassAttrs(&class, &class_len);
       FormatElementStart(CurrentDoc, Token, class, class_len);

       PrintRule(frame, GROOVE,
#if 0	/* QingLong.18-02-97 */
		  LeftMargin(frame,  left + indent, PixOffset),
# else
		 left + indent,
#endif
		 RightMargin(frame,          right, PixOffset),
		 0);

       EndOfLine(frame, (int)StyleGet(CurrentDoc, ArenaSI_TEXT_ALIGN));
       FormatElementEnd(CurrentDoc);
       continue;
      }

    /* howcome 25/9/95: support for PRE added */

    if (Token == TAG_PRE)
      {
       ParsePRE(0, False, frame,
		left + (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT), right);
       continue;
      }

    if (Token == TAG_P)
      {
       ParsePara(0, False, frame, align, TAG_P, left, right);
       continue;
      }

    if (TokenClass == EN_TEXT || Token == ENTITY)
      {
       UnGetToken();
       ParsePara(1, False, frame, align, TAG_P, left, right);
       continue;
      }

    /* unexpected tag so terminate element */

    if (Token != TAG_DT)
      {
       ParseHTMLerror |= ERR_DL;
       Arena_RecordError(CurrentDoc, bufptr, "ERR_DL"); /* howcome 11/10/94 */
      }
    UnGetToken();
    break;
   }

 /* kludge to cope with an <DD> element with no content */

 if (y == PixOffset)
   {
    PixOffset += LineSpacing[(int)StyleGet(CurrentDoc, ArenaSI_FONT)]/3;
    EndOfLine(frame, (int)StyleGet(CurrentDoc, ArenaSI_TEXT_ALIGN));
   }

 FormatElementEnd(CurrentDoc);
}


void ParseDL(Bool implied, Bool initial, Frame* frame, int align,
	     int left, int right)
{
 int indent, LastToken;
 Bool ForceDTend = False;  /* If got unbalanced <DT> tag (without </DT>)
			    * at the end of list (or even document).
			    */

 if (ParseEndTag(frame)) return;

 FormatElementStart(CurrentDoc, Token, class, class_len);

 if (!implied)
   {
    /* SwallowAttributes();*/
    ParseClassAttrs(&class, &class_len);
/*
    StyleRegisterSequential(Token, class, class_len);
    StyleRegisterHierarchical(Token, class, class_len);
*/
    /* FormatElementStart(CurrentDoc, Token, class, class_len);*/
    font = (int)StyleGet(CurrentDoc, ArenaSI_FONT);
    left += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT);
   }

 if (initial) left += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT);

 LastToken = TAG_DL;
/* indent = XTextWidth(Fonts[IDX_NORMALFONT], "mm", 2);*/
 indent = (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT);


 for (;;)
   {
    while (GetToken() == WHITESPACE);
    /* style = FormatElementStart(CurrentDoc, Token, class, class_len); *//* howcome 15/3/95 */

    if (Token == TAG_DL && EndTag)
      {
       SwallowAttributes();
       /* StyleDeregisterHierarchical();*/

       if (LastToken == TAG_DT) EndOfLine(frame, align);
       break;
      }

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == TAG_DT )
      {
       if (ParseEndTag(frame)) continue;

       if (LastToken == TAG_DT) EndOfLine(frame, align);

       ParseDT(0, frame, align, left, right);
       LastToken = TAG_DT;
       continue;
      }

    if (Token == TAG_DD )
      {
       if (EndTag)
	 SwallowAttributes();
        else
	 {
	  if (LastToken != TAG_DT)
	    {
	     ParseHTMLerror |= ERR_DL;
	     Arena_RecordError(CurrentDoc, bufptr, "ERR_DL");
	    }

	  ParseDD(0, frame, align, left + indent, right);
	  LastToken = TAG_DD;
	 }

       continue;
      }

    if (Token == TAG_UL || Token == TAG_OL || Token == TAG_DL ||
	Token == TAG_LI)
      {
       ParseHTMLerror |= ERR_DL;
       Arena_RecordError(CurrentDoc, bufptr, "ERR_DL");

       if (EndTag)
	 {
	  SwallowAttributes();
	  if (LastToken == TAG_DT) ForceDTend = True;
	  break;
	 }
        else
	 {
	  UnGetToken();
	  ParseDD(1, frame, align, left + indent, right);
	  LastToken = TAG_DD;
	  continue;
	 }
      }

    if (TokenClass == EN_TEXT ||
	Token == TAG_P  || Token == TAG_HR)
      {
       if (EndTag)
	 SwallowAttributes();
        else
	 {
	  ParseHTMLerror |= ERR_DL;
	  Arena_RecordError(CurrentDoc, bufptr, "ERR_DL");

	  UnGetToken();
	  ParseDD(1, frame, align, left + indent, right);
	  LastToken = TAG_DD;
	 }

       continue;
      }

    /* unexpected tag so terminate element */

    ParseHTMLerror |= ERR_DL;
    Arena_RecordError(CurrentDoc, bufptr, "ERR_DL"); /* howcome 11/10/94 */

    if (LastToken == TAG_DT) ForceDTend = True;

    UnGetToken();
    break;
   }
 /* End ``for (;;)'' */

 if (ForceDTend) EndOfLine(frame, align);

 Here = left;

 FormatElementEnd(CurrentDoc);
}


void ParseCaption(Bool implied, Frame* frame, int figure, int left, int right)
{
 int align;


 if (ParseEndTag(frame)) return;

 if (!implied) SwallowAttributes();

 align = (figure ? ALIGN_LEFT : ALIGN_CENTER);

 for (;;)
   {
    while (GetToken() == WHITESPACE);

    if (Token == TAG_CAPTION)
      if (ParseEndTag(frame))
	break;

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == TAG_P)
      {
       Here = left;
       /*  PixOffset += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_TOP); */ 
       ParsePara(0, False, frame, align, TAG_P, left, right);
       continue;
      }

    if (TokenClass == EN_TEXT)
      {
       if (ParseEndTag(frame)) continue;

       UnGetToken();
       Here = left;
       PixOffset += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_TOP);
       ParsePara(1, False, frame, align, TAG_P, left, right);
       continue;
      }

    /* unexpected tag so terminate element */

    UnGetToken();
    break;
   }    
}


/*
 * skip alt text for non-graphical displays
 */
void ParseAlt(Bool implied)
{
 if (!implied) SwallowAttributes();

 for (;;)
   {
    GetToken();

    if (Token == TAG_ALT)
      if (ParseEndTag(NULL))
	break;

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == TAG_P)
      {
       SwallowAttributes();
       continue;
      }

    if (TokenClass == EN_TEXT)
      {
       if (IsTag(Token)) SwallowAttributes();

       continue;
      }

    /* unexpected tag so terminate element */

    UnGetToken();
    break;
   }    
}


void ParseFigure(Bool implied, Frame* frame, int left, int right)
{
 int align, font, hreflen, indent;
 char *href;
 Frame* figframe;
 Image* image;
 long FigureStart;
 int width, height;

/*
  frame = Pop();
  if (frame)
    Push(frame);
   else
    frame = eframe;
 */


 if (ParseEndTag(frame)) return;

 align = ALIGN_CENTER;
 image = NULL;
 font = IDX_NORMALFONT;
 Here = left;
 indent = 0;

 if (!implied)
   {
    ParseFigureAttrs(&href, &hreflen, &align, &width, &height);
    if ((image = GetImage(href, hreflen, CurrentDoc->pending_reload)))
      {
       width = image->width;
       height = image->height;
      }

    if (! (width && height))
      {
       width = IMG_WIDTH;
       height = IMG_HEIGHT;
      }
   }

 /* if (image)*/
 {
  if (align == ALIGN_RIGHT)
    {
     indent = frame->width - width - right
                                    - (frame->leftmargin + frame->rightmargin);
    }
   else
    if (align == ALIGN_CENTER)
      {
       indent = (frame->width - width - right
		 - (frame->rightmargin + frame->leftmargin))/2;
      }
     else
      if (align == ALIGN_BLEEDLEFT)
	indent = 0;
       else
	if (align == ALIGN_BLEEDRIGHT)
	  {
           indent = frame->width - width
	                            - (frame->rightmargin + frame->leftmargin);
	  }
         else
	  /* indent = left; */
	  indent = (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT);

  if (indent > 0 && indent < left) indent = left;

  FigureStart = PixOffset;
  figframe = BeginFrame(frame, 0, 0, indent, indent+width,NULL);
  Push(figframe);

  if (align == ALIGN_LEFT || align == ALIGN_BLEEDLEFT)
    {
     frame->leftcount += 1;
     figframe->pushcount = frame->leftcount;
     figframe->oldmargin = frame->leftmargin;
     figframe->flow = ALIGN_LEFT;
    }
   else
    if (align == ALIGN_RIGHT || align == ALIGN_BLEEDRIGHT)
      {
       frame->rightcount += 1;
       figframe->pushcount = frame->rightcount;
       figframe->oldmargin = frame->rightmargin;
       figframe->flow = ALIGN_RIGHT;
      }

  Here = 0; /* position image flush left in figframe */
  PrintImage(figframe, 0, 0, 0, image, width, height);
  above = 0;
  below = height;
  EndOfLine(figframe, ALIGN_LEFT);
#if 0
  if (align == ALIGN_LEFT || align == ALIGN_BLEEDLEFT)
    AddBox(frame, frame->leftmargin, FigureStart,
	   figframe->width, figframe->height);
   else
    AddBox(frame,
	   frame->width - frame->rightmargin - figframe->width, PixOffset,
	   figframe->width, figframe->height);
#endif
  above = 0;
  Pop();
 }

 Here = 0;

 for (;;)
   {
    while (GetToken() == WHITESPACE);

    if (Token == TAG_FIG)
      if (ParseEndTag(frame))
	break;

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == TAG_CAPTION)
      {
       ParseCaption(0, figframe, 1, 0, 0);
       continue;
      }

    if (Token == TAG_ALT)
      {
       ParseAlt(0);
       continue;
      }

    if (Token == TAG_P)
      {
       UnGetToken();
       ParseAlt(1);
       continue;
      }

    if (TokenClass == EN_TEXT)
      {
       UnGetToken();
       ParseAlt(1);
       continue;
      }

    /* unexpected tag so terminate element */

    ParseHTMLerror |= ERR_FIG;
    Arena_RecordError(CurrentDoc, bufptr, "ERR_FIG"); /* howcome 11/10/94 */
    UnGetToken();
    break;
   }    

 /* if (image)*/
 {
  figframe->height = PixOffset - FigureStart;
  PrintFrameSize(figframe);
  PrintEndFrame(frame, figframe); 

  /* FigureEnd = PixOffset -= LineSpacing[font]/3; */

  if (align == ALIGN_LEFT || align == ALIGN_BLEEDLEFT)
    {
     frame->leftmargin += width + 5;
     PixOffset = FigureStart;
    }
   else
    if (align == ALIGN_RIGHT || align == ALIGN_BLEEDRIGHT)
      {
       frame->rightmargin += /*width*/ + 5;
       PixOffset = FigureStart;
      }

  if (align == ALIGN_LEFT || align == ALIGN_BLEEDLEFT)
    AddBox(frame,
	   frame->leftmargin - 3, FigureStart - 3,
	   figframe->width + 6, figframe->height + 6);
   else
    AddBox(frame,
	   frame->width - frame->rightmargin - figframe->width-3, PixOffset-3,
	   figframe->width + 6, figframe->height + 6);
 }
}


void ParseBlock(Bool implied, Frame* frame,
		int tag, int align, int font, int left, int right)
{
 int indent;
 long offset = 0;
 Frame* blockframe;
 Image* image = NULL;
 

 if (ParseEndTag(frame)) return;

 FormatElementStart(CurrentDoc, Token, class, class_len);

 align = -1;

 if (!implied)
   {
    if (tag == TAG_NOTE)
      image = ParseNoteAttrs();
     else
      {
       /* SwallowAttributes();*/
       ParseParaAttrs(&align, &class, &class_len);
      }
   }

 left += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT);
 right += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_RIGHT);
 PixOffset += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_TOP);
 font = (int)StyleGet(CurrentDoc, ArenaSI_FONT);
 if (align < 0) align = (int)StyleGet(CurrentDoc, ArenaSI_TEXT_ALIGN);

 Here = left;
 indent = 0;

 if (tag == TAG_NOTE)
   {
#if 0
    PrintRule(frame, GROOVE, left, right, 0);
    EndOfLine(frame, StyleGet(CurrentDoc, ArenaSI_TEXT_ALIGN));
#endif
    offset = PixOffset;

    if (image)
      {
       indent = 20 + image->width;
       blockframe = BeginFrame(frame, 0, 0, left, left + indent, NULL);
       Here = 0;
       above = 0;
       below = image->height;    
       PrintImage(blockframe, 0, 0, 0, image, image->width, image->height);
       EndOfLine(blockframe, (int)StyleGet(CurrentDoc, ArenaSI_TEXT_ALIGN));
       blockframe->height = image->height;
       PrintFrameSize(blockframe);
       PrintEndFrame(frame, blockframe); 
       /* frame->leftcount += 1;
	  blockframe->pushcount = frame->leftcount;
        */     
       blockframe->flow = ALIGN_NOTE;
       AddBox(frame, Here, PixOffset, image->width, image->height);
       /* left += indent;*/
       Here = LeftMargin(frame, left, PixOffset);
       PixOffset = offset;
      }
   }

 for (;;)
   {
    while (GetToken() == WHITESPACE);

    if (Token == tag)
      if (ParseEndTag(frame))
	break;

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == TAG_P)
      {
       Here = left;
       /* PixOffset += StyleGet(CurrentDoc, ArenaSI_MARGIN_TOP);*/
       ParsePara(0, False, frame, align, TAG_P, left, right);
       continue;
      }

    if (TokenClass == EN_TEXT)
      {
       if (ParseEndTag(frame)) continue;

       UnGetToken();
       /* PixOffset += StyleGet(CurrentDoc, ArenaSI_MARGIN_TOP); */
       ParsePara(1, False, frame, align, tag, left, right);
       continue;
      }

    /* unexpected tag so terminate element */

    ParseHTMLerror |= ERR_BLOCK;
    Arena_RecordError(CurrentDoc, bufptr, "ERR_BLOCK"); /* howcome 11/10/94 */

    UnGetToken();
    break;
   }    

 if (tag == TAG_NOTE)
   {
    /* ensure image ends before closing rule */
    if (image && PixOffset < offset + image->height)
      PixOffset = offset + image->height;
#if 0
    PrintRule(frame, GROOVE, left-indent, right, 0);
    EndOfLine(frame, StyleGet(CurrentDoc, ArenaSI_TEXT_ALIGN));
#endif
   }

 PixOffset += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_BOTTOM);
 FormatElementEnd(CurrentDoc);
}


/* tag is TAG_TH or TAG_TD, 
 * col is column number starting from 1 upwards,
 * returns cell height
 */
long ParseTableCell(Bool implied, Frame* frame, int row,
                    Frame** cells, int border,
                    ColumnWidth** pwidths,
                    int *pcol, int tag, int left, int right,
                    BG_Style* bg)
{
 int align, nowrap, col, rowspan, colspan, m, font;
 long cellTop = 0, cellHeight = 0;
 Frame *cellframe = NULL, *newframes, preframe;
 ColumnWidth *widths;
#ifdef ARENA_DEBUG
 char Iam[] = "ParseTableCell";
#endif


 if (ParseEndTag(frame)) return 0;

 newframes = NULL;
 col = *pcol;
 widths = *pwidths;

 align = ALIGN_CENTER;
 nowrap = 0;

 if (prepass)
   {
    preframe.next = preframe.child = NULL;
    preframe.box_list = NULL;
    preframe.indent = 0;
#if 1	/* Should the offset be initialized? Probably should. */
#  if 1	/* 03-11-97 */
    preframe.offset = 0;       /* Steffen Zahn.03-11-97 */
#   else
    preframe.offset = PixOffset;   /* QingLong.03-11-97 */
#  endif
#endif
    preframe.height = 0;
    preframe.width = 0;
    preframe.leftmargin = 0;
    preframe.rightmargin = 0;
    preframe.flow = ALIGN_LEFT;
    cellframe = &preframe;
   }

 list_indent = min_width = max_width = 0;
 rowspan = colspan = 1;

 if (!implied) ParseCellAttrs(&rowspan, &colspan, &align, &nowrap);

 *pcol += colspan;

 if (!prepass)
   {
       font = (tag == TAG_TH ? IDX_H3FONT : IDX_NORMALFONT);
    cellTop = PixOffset;
       left = widths[col].left;
      right = widths[col + colspan - 1].right;
    cellframe = BeginFrame(frame, 0, border, left - 3, right + 4, bg);
    cellframe->lastrow = row + rowspan - 1;

   /* left and right now adjusted to indents from frame margins */
     left = 3;  /* style */
    right = 4;

        /* try to make TH and TD baselines match for first line */
   /*
    if (tag == TAG_TH)
      PixOffset += ASCENT(IDX_NORMALFONT) - ASCENT(IDX_H3FONT);

      PixOffset += LineSpacing[IDX_NORMALFONT]/3
    */

     PixOffset += LineSpacing[(int)StyleGet(CurrentDoc, ArenaSI_FONT)]/3;

   }

 for (;;)
   {
    while (GetToken() == WHITESPACE);
   /* style = FormatElementStart(CurrentDoc, Token, class, class_len); */

   /*
    if (Token == TAG_TABLE && !EndTag)
      {
       Push(bufptr); 
       if (prepass) 
         ParseTable(0, cellframe, left, right);
        else
         {
          UnGetToken();
          ParseTable(1, cellframe, left, right);
         }
       bufptr = (char *)Pop();
       printf(", after : %X\n", bufptr);
       while(strncasecmp(bufptr,"/TABLE",5))
         {
          bufptr++;
         }
       bufptr += 6;
       printf("and then... %X\n", bufptr); 
       break;
      }
    */

    if (Token == TAG_DD)
      if (ParseEndTag(frame))
	break;

    if (Token == TAG_DL && EndTag)
      {
       UnGetToken();
       break;
      }

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == TAG_UL)
      {
       ParseUL(0, False, cellframe, 0, align, left, right);
       continue;
      }

    if (Token == TAG_OL)
      {
       ParseOL(0, False, cellframe, 0, align, left, right);
       continue;
      }

    if (Token == TAG_LI)
      {
       if (ParseEndTag(frame)) continue;

       UnGetToken();
       ParseUL(1, False, cellframe, 0, align, left, right);
       continue;
      }

    if (Token == TAG_DL)
      {
       ParseDL(0, False, cellframe, align, left, right);
       Here = left;
       continue;
      }

    if (Token == TAG_HR)
      {
      /* SwallowAttributes(); */

       if (EndTag) continue;

       ParseClassAttrs(&class, &class_len);
      /*
       StyleRegisterSequential(Token, class, class_len);
       */
       FormatElementStart(CurrentDoc, Token, class, class_len);

       PrintRule(cellframe, GROOVE,
#if 1	/* QingLong.18-02-97 */
		  LeftMargin(cellframe,  left, PixOffset),
		 RightMargin(cellframe, right, PixOffset),
# else
		 left, right,
#endif
		 0);

       EndOfLine(cellframe, (int)StyleGet(CurrentDoc, ArenaSI_TEXT_ALIGN));
       FormatElementEnd(CurrentDoc);
       continue;
      }

    if (Token == TAG_P)
      {
       Here = left;
       PixOffset += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_TOP);
      /* ParsePara(0, False, cellframe, align,
                   (tag == TAG_TH ? IDX_H3FONT : IDX_NORMALFONT),
                    left, right);
       */
       ParsePara(0, False, cellframe, align, tag, left, right);
       continue;
      }

    if (TokenClass == EN_HEADER)
      {
       UnGetToken();
       ParseHeader(cellframe, align, left, right, 0);
       continue;
      }

    if (TokenClass == EN_TEXT || Token == ENTITY)
      {
       UnGetToken();
       Here = left;
       PixOffset += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_TOP);
       ParsePara(1, False, cellframe, align, tag, left, right);
       continue;
      }

    if (Token == TAG_PRE)
      {
       UnGetToken();
       ParsePRE(0, False, cellframe, left, right);
       continue;
      }

    if (TokenClass == EN_BLOCK)
      {
       ParseBlock(0, cellframe, Token, ALIGN_LEFT, IDX_NORMALFONT,
                  left, right);
       Here = left;
       continue;
      }

    /* unexpected tag so terminate element */

    UnGetToken();
    break;
   }

 if (col > COLS(widths)) COLS(widths) = col;   /* update table column count */

 m = MAXCOLS(widths);

 if (col + colspan > m)   /* double array size as needed */
   {
    while (col + colspan> m) m = m << 1;

#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
    if (TABLE_TRACE)
      Arena_TracePrint(Iam,
		       " realloc, size = %d\n", (m + 1) * sizeof(ColumnWidth));
#endif

    widths = *pwidths = (ColumnWidth *)realloc(widths,
                                               (m + 1)*sizeof(ColumnWidth));
    MAXCOLS(widths) = m;

    while (m >= col)   /* zero widths for newly allocated elements */
      {
       widths[m].Cmin = 0;
       widths[m].Cmax = 0;
       widths[m].rows = 0;
       --m;
      }
   }

 max_width = cellframe->width + left + right;
 min_width += list_indent;

 if (nowrap && max_width > min_width)
   min_width = max_width;

 if (colspan > 1)  /* apportion widths evenly */
   {
    min_width /= colspan;
    max_width /= colspan;

    for (m = 0; m < colspan; ++m)
      {
       if (min_width > widths[col + m].Cmin)
         widths[col + m].Cmin = min_width;

       if (max_width > widths[col + m].Cmax)
         widths[col + m].Cmax = max_width;
      }
   }
  else
   {
    if (min_width > widths[col].Cmin)
      widths[col].Cmin = min_width;

    if (max_width > widths[col].Cmax)
      widths[col].Cmax = max_width;
   }

 widths[col].rows = rowspan - 1;

 if (!prepass)
   {
    cellHeight = PixOffset - cellTop ;
    if (rowspan != 0) { cellHeight /= rowspan; }
    cellframe->height = cellHeight;  /* for debug only */
    widths[col].Cmin = cellHeight;
    FlushPending(frame);
    PrintFrameSize(cellframe);
    InsertCell(cells, cellframe);
    return cellHeight;
   }
  else
    return 0;
 /* End if */
}


void DummyCell(Frame* frame, int row, Frame** cells,
               int border, ColumnWidth *widths, int col)
{
 int left, right;
 Frame* cellframe;

 left = widths[col].left;
 right = widths[COLS(widths)].right;
 cellframe = BeginFrame(frame, 0, border, left - 3, right + 4, NULL);
 cellframe->lastrow = row;
 PrintFrameSize(cellframe);
 InsertCell(cells, cellframe);
}


void ParseTableRow(Bool implied, Frame* frame, int row, Frame** cells,
		   int border, ColumnWidth** pwidths,
		   int left, int right, BG_Style *bg)
{
 int cols = 1;
 long rowTop, rowHeight, cellHeight;
 ColumnWidth* widths;


 if (ParseEndTag(frame)) return;

 if (!implied) SwallowAttributes();

 Here = left;
 rowTop = PixOffset;
 rowHeight = 0;

 for (;;)
   {
    PixOffset = rowTop;
    widths = *pwidths;

    /* if this cell spans more than one row */
    if (widths[cols].rows > 0)
      {
       widths[cols].rows -= 1;  /* decrement span count */

       if (!prepass) /* does spanned cell effect rowBottom? */
	 {
	  if (widths[cols].Cmin > rowHeight)
	    rowHeight = widths[cols].Cmin;
	 }

       ++cols;
       continue;
      }

    while (GetToken() == WHITESPACE);

    if (Token == TAG_TR && EndTag)
      {
       if (ParseEndTag(frame)) break;

       UnGetToken();
       break;
      }

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == TAG_TH)
      {
       cellHeight = ParseTableCell(0, frame, row, cells, border,
				   pwidths, &cols, TAG_TH, left, right, bg);

       if (cellHeight > rowHeight) rowHeight = cellHeight;
       continue;
      }

    if (Token == TAG_TD)
      {
       cellHeight = ParseTableCell(0, frame, row, cells, border,
				   pwidths, &cols, TAG_TD, left, right, bg);

       if (cellHeight > rowHeight) rowHeight = cellHeight;
       continue;
      }

    if (Token == TAG_TABLE)
      {
       UnGetToken();
       break;
      }

    if (TokenClass == EN_LIST || TokenClass == EN_TEXT ||
	TokenClass == EN_HEADER || TokenClass == EN_BLOCK)
      {
       if (ParseEndTag(frame)) continue;

       UnGetToken();
       cellHeight = ParseTableCell(1, frame, row, cells, border,
				   pwidths, &cols, TAG_TD,
				   left, right, bg);

       if (cellHeight > rowHeight) rowHeight = cellHeight;
       continue;
      }

    /* unexpected tag so terminate element */

    UnGetToken();
    break;
   }    

 widths = *pwidths; /* Just in case it changes again */
 if (!prepass && cols <= COLS(widths))
   DummyCell(frame, row, cells, border, widths, cols);

 PixOffset = rowTop + rowHeight;

 if (!prepass)
   FlushCells(frame, row, cells);  /* calls EndFrame() for cells ending this row */
}


void ParseTable(Bool implied, Frame* frame, int left, int right)
{
 int row, border, i, w, W, x, ptmin, ptmax, spare, max_width;
 long table_offset;
 char *table_bufptr;
 Frame* cells;
 Frame* new_frame;
 ColumnWidth* widths;


 if (ParseEndTag(frame)) return;

 if (!implied) ParseTableAttrs(&border);

 FormatElementStart(CurrentDoc, Token, NULL, 0);

 new_frame = (Frame*)Pop();
 Push(new_frame);

 damn_table = True; /* debug table format */

 Here = left;
 ProcessTableForm(1);   /* what the HECK is THIS ??!! */

 widths = (ColumnWidth *)Arena_MAlloc((NCOLS + 1) * sizeof(ColumnWidth),
				      False);

 COLS(widths) = 0;               /* current number of columns */
 MAXCOLS(widths) = NCOLS;        /* space currently allocated */
 widths[0].rows = 0;

 for (i = NCOLS; i > 0; --i)     /* zero widths for allocated elements */
   {
    widths[i].Cmin = 0;
    widths[i].Cmax = 0;
    widths[i].rows = 0;
    widths[i].left = 0;
    widths[i].right = 0;
   }

 prepass = True;
 table_bufptr = bufptr;          /* note parse position for second pass */
 table_offset = PixOffset;

draw_table:

 row = 0;
 cells = NULL;

 for (; row < 1000 ;) 
   {
    while (GetToken() == WHITESPACE);

    if (Token == TAG_TABLE && !EndTag)
      {
       while(strncasecmp(bufptr,"/TABLE",5)) bufptr++;
       bufptr += 6;
       break;
      }

    if (Token == TAG_TABLE && EndTag)
      {
       if (ParseEndTag(frame)) break;

       UnGetToken();
       break;
      }

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == TAG_CAPTION)
      {
       ParseCaption(0, frame, 0, 0, 0);
       continue;
      }

    if (Token == TAG_TR)
      {
       ++row;
       ParseTableRow(0, new_frame, row, &cells, border, &widths,
		     left, right,
		     (BG_Style*)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
       continue;
      }

    if (TokenClass == EN_LIST || TokenClass == EN_TEXT ||
	TokenClass == EN_TABLE ||
	TokenClass == EN_HEADER || TokenClass == EN_BLOCK)
      {
       if (ParseEndTag(frame)) continue;

       UnGetToken();
       ++row;
       ParseTableRow(1, new_frame, row, &cells, border, &widths,
		     left, right,
		     (BG_Style *)StyleGet(CurrentDoc, ArenaSI_BACKGROUND));
       continue;
      }

    /* unexpected tag so terminate element */

    UnGetToken();
    break;
   }    

 if (prepass)   /* assign actual column widths */
   {
    for (i = 1, ptmin = 3, ptmax = 3, W = 0; i <= COLS(widths); ++i)
      {
       ptmin += 7 + widths[i].Cmin;
       ptmax += 7 + widths[i].Cmax;
       W += widths[i].Cmax - widths[i].Cmin;
      }

    /* if max fits in window then use max widths */

    max_width = new_frame->width - right - left;

    if (ptmax <= max_width)
      {
       x = left + (max_width - ptmax)/2;

       for (i = 1; i <= COLS(widths); ++i)
	 {
	  widths[i].left = x + 3;
	  x += 7 + widths[i].Cmax;
	  widths[i].right = x - 4;
	 }
      }
    else
      if (ptmin < max_width)
        {
	 x = left;
	 spare = max_width - ptmin;

	 for (i = 1; i <= COLS(widths); ++i)
	   {
	    w = widths[i].Cmax - widths[i].Cmin;
	    widths[i].left = x + 3;
	    x += 7 + widths[i].Cmin + (spare * w)/W;
	    widths[i].right = x - 4;
	   }
        }
       else /* assign minimum column widths */
        {
	 x = left;

	 for (i = 1; i <= COLS(widths); ++i)
	   {
	    widths[i].left = x + 3;
	    x += 7 + widths[i].Cmin;
	    widths[i].right = x - 4;
	   }
        }

    /* and do second pass to draw table */

    bufptr = table_bufptr;
    PixOffset = table_offset;
    prepass = False;
    ProcessTableForm(2);
    goto draw_table;
   }

 Free(widths);   /* free column widths */
 Here = left;
 PixOffset += (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_BOTTOM);
 FormatElementEnd(CurrentDoc);
 damn_table = False;
}


/*
 * left and right are the indents from the current LHS, RHS respectively
 */
void ParseBody(Bool implied, Frame* frame, int left, int right, int margin)
{
 int indent;
#ifdef ARENA_DEBUG
 char Iam[] = "ParseBody";
#endif


 if (ParseEndTag(frame)) return;

 indent = margin;

 if (!implied) SwallowAttributes();

  /* howcome 3/4/95: first call to FormatElementStart */
 FormatElementStart(CurrentDoc, TAG_HTML, NULL, 0);

 for (;;)
   {
    while (GetToken() == WHITESPACE);

    if (Token == TAG_BODY)
      if (ParseEndTag(frame))
	break;

    if (IsTag(Token) && TokenClass == EN_HEADER)
      {
       if (ParseEndTag(frame)) continue;

       ParseHeader(frame, ALIGN_LEFT, left, right, 0);
       continue;
      }

    if (Token == UNKNOWN)
      {
       SwallowAttributes();
       continue;
      }

    if (Token == TAG_UL)
      {
       ParseUL(0, True, frame, 0, ALIGN_LEFT, left, right);
       continue;
      }

    if (Token == TAG_OL)
      {
       ParseOL(0, True, frame, 0, ALIGN_LEFT, left, right);
       continue;
      }

    if (Token == TAG_LI)
      {
       if (ParseEndTag(frame)) continue;

       ParseHTMLerror |= ERR_LI;
       Arena_RecordError(CurrentDoc, bufptr, "ERR_LI"); /* howcome 11/10/94 */
       UnGetToken();
        /* howcome 11/8/95: ALIGN_CENTER -> ALIGN_LEFT */
       ParseUL(1, True, frame, 0, ALIGN_LEFT, left, right);
       continue;
      }

    if (Token == TAG_DL)
      {
      /* ParseDL(0, frame, StyleGet(CurrentDoc, ArenaSI_TEXT_ALIGN), StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT),
	 right+margin);*/
       ParseDL(0, True, frame, ALIGN_LEFT, left, right + margin);
       continue;
      }

    if (TokenClass == EN_TABLE)
      {
       if (ParseEndTag(frame)) continue;

       ParseHTMLerror |= ERR_TABLE;
       Arena_RecordError(CurrentDoc, bufptr, "ERR_TABLE"); /* howcome 11/10/94 */
       UnGetToken();
       ParseTable(1, frame, left, right);
       continue;
      }

    if (Token == TAG_TABLE)
      {
       ParseTable(0, frame, left, right + margin);
       continue;
      }

    if (Token == TAG_DT || Token == TAG_DD)
      {
       if (ParseEndTag(frame)) continue;

       ParseHTMLerror |= ERR_DL;
       Arena_RecordError(CurrentDoc, bufptr, "ERR_DL"); /* howcome 11/10/94 */
       UnGetToken();
       ParseDL(1, True, frame, ALIGN_LEFT, left, right);
       continue;
      }

    if (Token == TAG_HR)
      {
       if (ParseEndTag(frame)) continue;

      /* SwallowAttributes();*/

       ParseClassAttrs(&class, &class_len);
/*
       StyleRegisterSequential(Token, class, class_len);
*/
       FormatElementStart(CurrentDoc, Token, class, class_len);

       PrintRule(frame, GROOVE,
		  LeftMargin(frame,  left, PixOffset),
		 RightMargin(frame, right, PixOffset),
		 0);

       EndOfLine(frame, (int)StyleGet(CurrentDoc, ArenaSI_TEXT_ALIGN));
       FormatElementEnd(CurrentDoc);
       continue;
      }

    if (Token == TAG_PRE)
      {
       ParsePRE(0, True, frame, left, right);
       continue;
      }

    if (Token == TAG_P)
      {
       Here = left + margin;
       /* PixOffset += top;*/
       ParsePara(0, True, frame, ALIGN_LEFT, TAG_P, left, right);
       continue;
      }

    if (TokenClass == EN_TEXT || Token == ENTITY)
      {
       if (ParseEndTag(frame)) continue;

       UnGetToken();
       Here = left + margin;
       /* style = FormatElementStart(CurrentDoc, Token, class, class_len); *//* howcome 6/3/94: need to reset */

#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
       if (REGISTER_TRACE) Arena_TracePrint(Iam, " implied: ");
#endif

#if 0
       StyleRegisterSequential(TAG_P, NULL, 0); /* reason to live?? */
       FormatElementStart(CurrentDoc, TAG_P, NULL, 0);
#endif
       ParsePara(1, True, frame, ALIGN_LEFT, TAG_P, left, right);
       continue;
      }

    if (Token == TAG_FIG)
      {
       ParseFigure(0, frame, left, right);
       continue;
      }

    if (Token == TAG_ADDRESS)
      {
      /* style = FormatElementStart(CurrentDoc, Token, class, class_len);*/
       ParseBlock(0, frame, Token, ALIGN_LEFT, IDX_NORMALFONT, left, right);
       Here = left;
       continue;
      }

    if (Token == TAG_ABSTRACT)
      {
       ParseBlock(0, frame, Token, ALIGN_LEFT, IDX_NORMALFONT, left, right);
       Here = left;
       continue;
      }

    if (Token == TAG_QUOTE)
      {
       ParseBlock(0, frame, Token, ALIGN_CENTER, IDX_NORMALFONT, left, right);
       Here = left;
       continue;
      }

    if (Token == TAG_NOTE)
      {
       ParseBlock(0, frame, Token, ALIGN_LEFT, IDX_NORMALFONT, left, right);
       Here = left;
       continue;
      }

    if (Token == ENDDATA)
      {
       UnGetToken();
       break;
      }

    if (Token == TAG_BODY || TokenClass == EN_SETUP)
      {
       ParseHTMLerror |= ERR_SETUP;
       /* howcome 11/10/94 */
       Arena_RecordError(CurrentDoc, bufptr, "ERR_SETUP");
       SwallowAttributes();
       continue;
      }
    /* must add form */

    /* unexpected but known tag name */
    ParseHTMLerror |= ERR_BODY;
    Arena_RecordError(CurrentDoc, bufptr, "ERR_BODY");
    SwallowAttributes();
   }
 FormatElementEnd(CurrentDoc);
}


/*
 * Initialise background frame.
 */
void InitBackgroundFrame()
{
 background.next = NULL;
 background.child = NULL;
 background.box_list = NULL;
 background.top = paint + FRAMESTLEN;

#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
 if (TAG_TRACE)
   Arena_TracePrint("InitBackgroundFrame",
		    " paint = "POINTER_FORMAT","
		    " background.top = "POINTER_FORMAT".\n",
		    paint, background.top);
#endif

 background.info   = 0;
 background.length = 3;
 background.indent = 0;
 background.offset = 0;
 background.width  = GetDocDispWinWidth();
 background.height = 0;  /* zero height might cause problems? */
 background.style  = 0;
 background.border = 0;
 background.leftmargin  = MININDENT;
 background.rightmargin = MININDENT;
}


long ParseHTML(int *width, Bool style_parse)
{
 int margin = 0;
 Bool implied_BODY_tag = True, theDoc_has_some_body;
#ifdef ARENA_DEBUG
 char Iam[] = "ParseHTML";
#endif


 PixOffset  = 0;
 LastBufPtr = bufptr = buffer + hdrlen;
 ParseHTMLerror = prepass = preformatted = prealign = False;
 DocIsIndex = False;
 start_figure = figure = 0;
 font = paintStartLine = -1;
 FigureEnd = LeftMarginIndent = RightMarginIndent = 0;
 form = NULL;

 /* Here we are parsing a document.
  * HOW MANY TIMES have we been here for THIS document?!!
  * Oh well, clear the fragment info for the reparse.
  * MUST also clear the bad_flags setup: as we may be reparsing,
  * and the errors can be in different locations this time!
  */
 if (CurrentDoc != NULL)
   {
    CurrentDoc->FragmentCounter = 0;
    HTList_delete(CurrentDoc->FragmentPtr);
    CurrentDoc->FragmentPtr = HTList_new();

    if (delete_Document_bad_flags(CurrentDoc))
      CurrentDoc->bad_flags = NULL;
   }

 InitPaintStream();
 InitBackgroundFrame();
 Push(&background); 

#ifdef ARENA_DEBUG
 if (TAG_TRACE)
   Arena_TracePrint(Iam, " starting buffer at "POINTER_FORMAT".\n", paint);
#endif

 /* Reserve space for background's begin-frame object,
  * which is needed to simplify display and scrolling routines.
  */
 background.info = PrintBeginFrame(NULL);

 SkipSpace();

 if (Token == TAG_BASE)
   {
    ParseBase(&background);
    SkipSpace();
   }

 if (Token == TAG_HEAD)
   ParseSetUp(0, &background);
  else
   {
    UnGetToken();
    ParseSetUp(1, &background);
   }

 if (style_parse) StyleParse(CurrentDoc);

 if ((theDoc_has_some_body = (SkipSpace() != ENDDATA)))
   {
    if ((implied_BODY_tag = (Token != TAG_BODY)))
      {
       UnGetToken();
      }
#ifndef NO_BACKGROUND_IN_HTML
     else
      {
       ParseBodyAttrs(&bufptr);
      }
#endif /* NO_BACKGROUND_IN_HTML */
   }

 FormatElementStart(CurrentDoc, TAG_HTML, NULL, 0);
 background.leftmargin  = (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_LEFT);
 background.rightmargin = (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_RIGHT);
 background.offset = PixOffset = (int)StyleGet(CurrentDoc, ArenaSI_MARGIN_TOP);

 if (theDoc_has_some_body)
   ParseBody(implied_BODY_tag, &background, MININDENT, MININDENT, margin);

 FlushAllFrames(NULL, background.child); /* flush remaining end of frames */

 background.child = NULL;
 background.top = paint + background.info + FRAMESTLEN;

 /*
  * update background frame height.
  */
 {
  int h;

   h = PixOffset - background.offset;
   if (background.height < h) background.height = h;

   h = GetDocDispWinHeight() - background.offset;
   if (background.height < h) background.height = h;
 }

 /* And fill in begin-frame object fields. */
 SetBeginFrameParameters(&background);

 PrintFrameSize(&background);
 PrintEndFrame(NULL, &background);
 FormatElementEnd(CurrentDoc);

 *width = html_width = background.width;
 Pop();

 return background.height;
}


/* ParseImage doesn't really parse the image.
 * It does, however, for the equivalent for an image that
 * ParseHTML does for HTML: create a paintstream.
 * In the case of a single image, the paint stream is very simple.
 * Alternativley, one could have wrapped the image in dummy html
 * and formatted the document as normal.
 * This would have given greater flexibiliy (e.g. to put it into a table),
 * but would have been more expensive. howcome 13/2/95.
 */
long ParseImage(Doc* doc, int* width)
{
 int margin;


 FormatElementStart(doc, TAG_HTML, NULL, 0);
 FormatElementStart(doc, TAG_IMG, NULL, 0);

 LastBufPtr = bufptr = buffer + hdrlen;
 ParseHTMLerror = prepass = False;
 preformatted = True;
 DocIsIndex = False;
 start_figure = figure = 0;
 font = paintStartLine = -1;
 FigureEnd = LeftMarginIndent = RightMarginIndent = 0;
 form = NULL;
 margin = XTextWidth(Fonts[IDX_NORMALFONT], "mmm", 2);


 InitPaintStream();
 InitBackgroundFrame();
 Here = background.leftmargin;
 PixOffset = 5;


 /* Reserve space for image's begin-frame object. */
 background.info = PrintBeginFrame(NULL);

 WrapImageIfNeeded(&background, ALIGN_LEFT,
		   background.leftmargin, background.rightmargin,
		   doc->image->width, 0, doc->image->height);
 PrintImage(&background, 0, 0, NULL,
	    doc->image, doc->image->width, doc->image->height);
 EndOfLine(&background, ALIGN_LEFT);

 background.top = paint + background.info + FRAMESTLEN;
 background.width  = background.leftmargin + doc->image->width
                   + background.rightmargin;
 background.height = 5 + doc->image->height + 5;

 /* And fill in begin-frame object fields. */
 SetBeginFrameParameters(&background);

 PrintFrameSize(&background);
 PrintEndFrame(NULL, &background);
 FormatElementEnd(doc);
 FormatElementEnd(doc);

 *width = html_width = background.width;

 return background.height;
}
