/*


   Almost everything about editor operations
   (c) Nov 1995 By Yves Lafon for INRIA / W3C


*/

#include <stdio.h>
#include <string.h>

#include "types.h"
#include "neweditor_i.h"
#ifdef ARENA_DEBUG
#  include "debug.h"
# else
#  include "error.h"
#endif
#include "util.h"


EditorBuffer* CreateBuffer(void)
{
 EditorBuffer* new_buffer;

 new_buffer = (EditorBuffer*)Arena_MAlloc(sizeof(EditorBuffer), False);
 if(new_buffer)
   {
    new_buffer->cell = NULL;
    new_buffer->pos = 0;
    new_buffer->size = 0;
    return(new_buffer);
   };

#ifdef ARENA_DEBUG
 Arena_TracePrint("CreateBuffer", " Ran out of memory.\n");
#else
 Arena_PrintError("Ran out of memory.\n");
#endif

 exit(0);
}

cell *CreateCell(int capacity)
{
 cell *new_cell;

 new_cell = (cell*)Arena_MAlloc(sizeof(cell), False);
 if (new_cell) /* did this to avoid gcc warning */
   {
    new_cell->bank = (char*)Arena_CAlloc(capacity + 1, sizeof(char), False);
    if (new_cell->bank)
      {
       new_cell->next = NULL;
       new_cell->prev = NULL;
       new_cell->size = 0;
       new_cell->capacity = capacity;
       return(new_cell);
      };
   }

#ifdef ARENA_DEBUG
 Arena_TracePrint("CreateCell", " Ran out of memory.\n");
#else
 Arena_PrintError("Ran out of memory.\n");
#endif

 exit(0);    
}

void FreeCell(cell *TheCell)
{
 if (TheCell)
   {
    if (TheCell->bank) free(TheCell->bank);
    free(TheCell);
   };
}

void FreeBuffer(EditorBuffer *buffer)
{
 cell *wiped_cell;

 if (buffer)
   {
    wiped_cell = buffer->cell;
    if (wiped_cell)
      {
       if (wiped_cell->next)
	 {
	  wiped_cell = wiped_cell->next;
	  while (wiped_cell->next)
	    {
	     FreeCell(wiped_cell->prev);
	     wiped_cell = wiped_cell->next;
	    };
	  FreeCell(wiped_cell->prev);
	 };
       FreeCell(wiped_cell);
      };
    free(buffer);
   };
}

Bool SplitCell(cell *buffer_cell, int split_size)
{
 cell *new_cell;
 int i;
 char *oldbank, *newbank;

 if (split_size <= 0) return False;
 if (buffer_cell == NULL) return False;

 /* Check if splitting is actually required */
 if (buffer_cell->size < buffer_cell->capacity) return False;

 if (split_size > BANK_SIZE)
   i = split_size + (BANK_SIZE - SPLIT_SIZE);
  else
   i = BANK_SIZE;

 new_cell = CreateCell(i);
 new_cell->prev = buffer_cell;
 new_cell->next = buffer_cell->next;
 buffer_cell->next = new_cell;
 if (new_cell->next) new_cell->next->prev = new_cell;
 new_cell->size = new_cell->capacity - split_size;
 buffer_cell->size = split_size;
 oldbank = buffer_cell->bank + split_size;
 newbank  = new_cell->bank;
 for (i = new_cell->size; i > 0; i--) *newbank++ = *oldbank++;	/* strncpy? */
 *newbank = '\0';	/* Ensure zero ending */

 return True;
}


Bool MergeCell(cell *origin)
{
 int i,size;
 char *o_bank,*w_bank;
 cell *wiped;

 if (origin == NULL) return False;
 if (origin->next == NULL) return False;

 if (origin->size + origin->next->size < origin->capacity)
   {
    wiped = origin->next;
    size = wiped->size;
    o_bank = origin->bank + origin->size;
    w_bank = wiped->bank;
    for (i = 0; i < size; i++) *o_bank++ = *w_bank++;
    origin->size += wiped->size;
    origin->next = wiped->next;
    if (origin->next) origin->next->prev = origin;
    FreeCell(wiped);
    return True;
   }
  else
   return False;

}

int DeleteChar(EditorBuffer *buffer, int pos)
{
 cell *buffer_cell;
 int i,j;
 char *position;

 buffer_cell = buffer->cell;

 if (buffer_cell)
   {
    for (i = pos; (i >= buffer_cell->size) && buffer_cell; )
      {
       i -= buffer_cell->size;
       if (i)
	 buffer_cell = buffer_cell->next;
        else
	 {
	  if (buffer->size) --buffer->size;
	  if (buffer->pos > buffer->size) buffer->pos = buffer->size;
	  return(1);
	 };
      };

    if (!buffer_cell)
      {
#ifdef ARENA_DEBUG
       Arena_TracePrint("DeleteChar", " Ran out of editor.\n");
#endif
       return(0);  /* error, ran out of editor */
      }

    if (buffer_cell->size > 1)
      {
       if (i < (buffer_cell->capacity - 1)) /* if it is not at the end,
					       we update */
	 {
	  j = buffer_cell->size;
	  for (position = buffer_cell->bank + i + 1; i <= j; i++, position++)
	    *(position-1) = *position;
	 }
        else
	 {
	  if (buffer->pos) --buffer->pos;	
	 };

       --buffer_cell->size;  /* update bank size */
       MergeCell(buffer_cell);  /* optimize bank filling if needed */
       if (buffer_cell->prev) MergeCell(buffer_cell->prev);
      }
     else /* size 1 -> we remove the bank */
      {
#ifdef ARENA_USE_STRICT_ANSI_C
       if (buffer_cell->prev)
	 buffer_cell->prev->next = buffer_cell->next;
        else
	 buffer->cell = buffer_cell->next;
# else
       ((buffer_cell->prev) ? buffer_cell->prev->next : buffer->cell) = buffer_cell->next;
#endif /* ARENA_USE_STRICT_ANSI_C */
       if (buffer_cell->next)
	 buffer_cell->next->prev = buffer_cell->prev;
       FreeCell(buffer_cell); 
      };

    if (buffer->size) --buffer->size; /* update buffer size */
    if (buffer->pos > buffer->size) buffer->pos = buffer->size;
    return(1);
   }

 return(0);
}

int InsertChar(EditorBuffer *buffer, int pos, char c)
{
 cell *buffer_cell;
 int i, j, b_size;
 char *position;

 if (!buffer) buffer = CreateBuffer();

 buffer_cell = buffer->cell;

 if (!buffer_cell)
   {
    buffer->cell = CreateCell(BANK_SIZE);
    buffer_cell  = buffer->cell;
   };

 /* Find the cell where the position `pos' resides... */
 for (i = pos;
      buffer_cell && (i > buffer_cell->size);
      buffer_cell = buffer_cell->next)
   i -= buffer_cell->size;

 /* Now `i' is believed to contain relative position of the `pos'
  * within the cell where it resides (buffer_cell).
  */

 if (!buffer_cell)
   {
#ifdef ARENA_DEBUG
    Arena_TracePrint("InsertChar", " Ran out of editor.\n");
#endif
    return(0);  /* error, ran out of editor! */
   }

 b_size = buffer_cell->size;

 SplitCell(buffer_cell, SPLIT_SIZE);  /* if max size -> split */

 position = buffer_cell->bank + buffer_cell->size;
 /* Shift all the trailing stuff by 1 */
 for (j = buffer_cell->size; j >= i; j--, position--)
   *(position + 1) = *position;

 *(buffer_cell->bank + i) = c;
 ++buffer_cell->size; /* update bank size */
 ++buffer->size;     /* update buffer size */

 return(1);
}

int InsertnChar(EditorBuffer *buffer, int pos, char *s, int size)
{
 int ok,i;
 char *s_pos;

 s_pos = s;

 for (ok = 1, i = 0; ok && (i < size); i++)
   ok = InsertChar(buffer, pos + i, *s_pos++);

 return(ok);
}

int InsertString(EditorBuffer *buffer, int pos, char *s)
{
 int ok,size,i;
 char *s_pos;

 size = Arena_StrLen(s);
 s_pos = s;

 for (ok = 1, i = 0; ok && (i < size); i++)
   ok = InsertChar(buffer, pos + i, *s_pos++);

 return(ok);
}

int AppendChar(EditorBuffer *buffer, char c)
{
 return(InsertChar(buffer, buffer->size, c));
}

int AppendnChar(EditorBuffer *buffer, char *c, int size)
{
 return(InsertnChar(buffer, buffer->size, c, size));
}

int AppendString(EditorBuffer *buffer, char *s)
{
 return(InsertString(buffer, buffer->size, s));
}

char *Buffer2Str(EditorBuffer *buffer)
{
 char *string,*pos;
 cell *cell_link;

 if (buffer)
   {
    string = (char*)Arena_CAlloc(buffer->size + 2, sizeof(char), False);

    if (string == NULL)
      {
#ifdef ARENA_DEBUG
       Arena_TracePrint("Buffer2Str", " Ran out of memory.\n");
#else
       Arena_PrintError("Ran out of memory.\n");
#endif
       exit(0);
      };

    pos = string;
    for (cell_link = buffer->cell; cell_link; cell_link = cell_link->next)
      {
       strncpy(pos, cell_link->bank, cell_link->size);
       pos += cell_link->size;
      };

    *pos = 0; /* end of string */
    return(string);
   }
 return(NULL);
}

EditorBuffer *Str2Buffer(char *string)
{
 int size;
 EditorBuffer *buffer;

 buffer = CreateBuffer();
 if (string == NULL)
   {
    buffer->size = 0;
    return buffer;
   }

 buffer->size = size = Arena_StrLen(string);

 if (size)
   {
    int capacity = size/2;

    if (size < BANK_SIZE)
      capacity = BANK_SIZE;
     else
      if (capacity > BANK_SIZE)
	capacity = size + BANK_SIZE;
       else
	capacity += size;

    buffer->cell = CreateCell(capacity);
    strncpy(buffer->cell->bank, string, size);
    buffer->cell->size = size;
    *(buffer->cell->bank + size) = '\0';	/* Ensure zero ending */
   };
 return(buffer);
}

int LineNumber(EditorBuffer *buffer)
{
    int numline,i,j;
    cell *buffer_cell;

    numline = 0;
    buffer_cell = (buffer) ? buffer->cell : NULL;
    for(i = 0, j = 0; (i<buffer->pos) && buffer_cell; i++, j++)
    {
        if(j >= buffer_cell->size)
        {
            buffer_cell = buffer_cell->next;
            j = 0;
        }
        if(buffer_cell)
            numline += (*(buffer_cell->bank + j) == '\n');
    };
    return(numline);
}

int ColNumber(EditorBuffer *buffer)
{
    cell *buffer_cell;
    int i, j;
    int numcol;

    buffer_cell = buffer->cell;

    numcol = 0;
    buffer_cell = (buffer) ? buffer->cell : NULL;
    for(i = 0, j= 0 ; (i<buffer->pos) && buffer_cell; i++, j++)
      {
        if (j >= buffer_cell->size)
	  {
            buffer_cell = buffer_cell->next;
            j = 0;
	  }

        if (buffer_cell)
	  {
           if (*(buffer_cell->bank + j) == '\n')
	     numcol = 0;
	    else
	     numcol++;
	  }
      }

    return(numcol);
}

void NextLine(EditorBuffer* buffer)
{
 cell* buffer_cell;
 int i,j,done;
 int numcol,k;


 buffer_cell = buffer->cell;
 numcol = ColNumber(buffer);
 for (i = 0, j = 0; (i < buffer->pos) && buffer_cell; i++, j++)
   {
    if (j >= buffer_cell->size)
      {
       buffer_cell = buffer_cell->next;
       j = 0;
      }
   }

 for (done = 0; (i < buffer->size) && buffer_cell && !done; i++, j++)
   {
    if (j >= buffer_cell->size)
      {
       buffer_cell = buffer_cell->next;
       j = 0;
      }

    if (buffer_cell)
      {
       if (*(buffer_cell->bank + j) == '\n')
	 done = 1;
        else
	 ++buffer->pos;
      }
   }

 if (done)
   {
    for (done = 0, k= 0 ;
	 (i < buffer->size) && buffer_cell && !done;
	 i++, j++, k++)
      {
       if(j >= buffer_cell->size)
	 {
	  buffer_cell = buffer_cell->next;
	  j = 0;
	 }

       if (buffer_cell)
	 {
	  if ((*(buffer_cell->bank + j) == '\n')||(k == numcol)) done = 1;
	  ++buffer->pos;
	 }
      }
   }
}

void PrevLine(EditorBuffer *buffer)
{
    cell *buffer_cell;
    int i,j,done;
    int numcol,numrow,k;

    numcol = ColNumber(buffer);
    numrow = LineNumber(buffer);
    buffer_cell = buffer->cell;
    if(numrow > 1)
    {
        for(done = 0, k = 0, i = 0, j = 0;
	    (i<buffer->size) && buffer_cell && !done;
	    i++, j++)
        {
            if (j >= buffer_cell->size)
            {
                buffer_cell = buffer_cell->next;
                j = 0;
            }
            if(buffer_cell)
                if(*(buffer_cell->bank + j) == '\n')
                {
                    k++;
                    done = (k == (numrow-1));
                };
        };
    }
    else
    {
        i = 0;
        j = 0;
        done = 1;
    };
    buffer->pos = i;
    if(done)
    {
        for(done = 0, k = 0;
	    (i<buffer->size) && buffer_cell && !done;
	    i++, j++, k++)
        {
            if(j >= buffer_cell->size)
            {
                buffer_cell = buffer_cell->next;
                j = 0;
            };
            if(buffer_cell)
            {
                if((*(buffer_cell->bank + j) == '\n') || (k == numcol))
                    done = 1;
                else
                    ++buffer->pos;
            };
        };
    };   
}
