/*
    libparted
    Copyright (C) 1998-2000  Andrew Clausen  <clausen@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>

#include "fat.h"

/* Note: this deals with filesystem start and end sectors, even if the physical
 * devices are different (eg for fat_copy())  Perhaps this is a hack, but it
 * works ;-)
 */
static int
calc_deltas (FatOpContext* ctx)
{
	PedFileSystem*	old_fs = ctx->old_fs;
	PedFileSystem*	new_fs = ctx->new_fs;
	FatSpecific*	old_fs_info = FAT_SPECIFIC (old_fs);
	FatSpecific*	new_fs_info = FAT_SPECIFIC (new_fs);
	PedSector	old_cluster_ofs;
	PedSector	new_cluster_ofs;
	PedSector	sector_delta;

	old_cluster_ofs = old_fs->geom->start + old_fs_info->cluster_offset;
	new_cluster_ofs = new_fs->geom->start + new_fs_info->cluster_offset;

	if (new_cluster_ofs > old_cluster_ofs) {
		ctx->start_move_dir = FAT_DIR_FORWARD;
		sector_delta = new_cluster_ofs - old_cluster_ofs;
	} else {
		ctx->start_move_dir = FAT_DIR_BACKWARD;
		sector_delta = old_cluster_ofs - new_cluster_ofs;
	}

	if (sector_delta % new_fs_info->cluster_sectors) {
		ped_exception_throw (
			PED_EXCEPTION_BUG, PED_EXCEPTION_CANCEL,
			_("Cluster start delta = %d, which is not a multiple of "
			  "the cluster size %d."),
			(int) sector_delta,
			(int) new_fs_info->cluster_sectors);
		return 0;
	}
	
	ctx->start_move_delta = sector_delta / new_fs_info->cluster_sectors;

#ifdef VERBOSE
	printf ("First cluster delta is: %d %s.\n",
		ctx->start_move_delta,
		(ctx->start_move_dir == FAT_DIR_FORWARD)?
			"forwards" : "backwards");
#endif

	return 1;
}

FatOpContext*
fat_op_context_new (PedFileSystem* new_fs, PedFileSystem* old_fs)
{
	FatSpecific*	old_fs_info = FAT_SPECIFIC (old_fs);
	FatOpContext*	ctx;

	ctx = (FatOpContext*) ped_malloc (sizeof (FatOpContext));
	if (!ctx)
		goto error;
	
	ctx->remap = (FatCluster*) ped_malloc (sizeof (FatCluster)
					* (old_fs_info->cluster_count + 2));
	if (!ctx->remap)
		goto error_free_ctx;

	ctx->new_fs = new_fs;
	ctx->old_fs = old_fs;
	if (!calc_deltas (ctx))
		goto error_free_ctx;

	return ctx;

error_free_ctx:
	ped_free (ctx);
error:
	return NULL;
}

void
fat_op_context_destroy (FatOpContext* ctx)
{
	free (ctx->remap);
	free (ctx);
}

FatCluster
fat_op_context_map_static_cluster (FatOpContext* ctx, FatCluster clst)
{
	FatSpecific*	new_fs_info = FAT_SPECIFIC (ctx->new_fs);
	FatCluster	result;

	if (ctx->new_fs->geom->disk != ctx->old_fs->geom->disk)
		return 0;

	if (ctx->start_move_dir == FAT_DIR_FORWARD) {
		if (clst < ctx->start_move_delta + 2)
			return 0;
		result = clst - ctx->start_move_delta;
	} else {
		result = clst + ctx->start_move_delta;
	}

	if (result >= new_fs_info->cluster_count + 2)
		return 0;

	return result;
}

static FatCluster
sector_to_cluster (PedFileSystem* fs, PedSector sector)
{
	FatSpecific*	fs_info = FAT_SPECIFIC (fs);
	FatCluster	clst;

	if (sector < fs_info->cluster_offset)
		return 0;

	clst = (sector - fs_info->cluster_offset) / fs_info->cluster_sectors;

	if (clst < 2 || clst >= fs_info->cluster_count + 2)
		return 0;

	return clst;
}

/* This function sets the initial fat for the new resized file system.
   This is in *NO WAY* a proper FAT table - all it does is:
	a) mark bad clusters as bad.
	b) mark used clusters (that is, clusters from the original FS that are
	   reachable from the resized one).  Marks as EOF (i.e. used, end of
	   file chain).
	c) mark original filesystem metadata as EOF (i.e. used), to prevent
	   it from being clobbered.  This will leave the original file system
	   intact, until the partition table is modified, if the start of
	   the partition is moved.

   The FATs are rebuilt *properly* after cluster relocation.  This here is
   only to mark clusters as used, so when cluster relocation occurs, clusters
   aren't relocated on top of ones marked in a, b or c.
*/
int
fat_op_context_create_initial_fat (FatOpContext* ctx)
{
	FatSpecific*	old_fs_info = FAT_SPECIFIC (ctx->old_fs);
	FatSpecific*	new_fs_info = FAT_SPECIFIC (ctx->new_fs);
	FatCluster	clst;
	FatCluster	new_clst;
	PedSector	sect;
	PedSector	new_sect;

	new_fs_info->fat = fat_table_new (
		new_fs_info->fat_type,
		new_fs_info->fat_sectors * 512
			/ fat_table_entry_size (new_fs_info->fat_type));
	if (!new_fs_info->fat)
		return 0;

	new_fs_info->fat->cluster_count = new_fs_info->cluster_count;

/* mark bad and used clusters */
	for (clst = 2; clst < old_fs_info->cluster_count + 2; clst++) {
		if (fat_table_is_available (old_fs_info->fat, clst))
			continue;

		new_clst = fat_op_context_map_static_cluster (ctx, clst);
		if (!new_clst)
			continue;

		if (fat_table_is_bad (old_fs_info->fat, clst))
			fat_table_set_bad (new_fs_info->fat, new_clst);
		else
			fat_table_set_eof (new_fs_info->fat, new_clst);
	}

/* mark metadata regions that map to clusters on the new FS */
	for (sect = 0;
	     sect < old_fs_info->cluster_offset
		+ 2 * old_fs_info->cluster_sectors;
	     sect++) {
		new_sect = ped_geometry_map (ctx->new_fs->geom,
					     ctx->old_fs->geom, sect);
		if (new_sect == -1)
			continue;
			
		clst = sector_to_cluster (ctx->new_fs, new_sect);
		if (!clst)
			continue;

		fat_table_set_eof (new_fs_info->fat, clst);
	}

	return 1;
}
