#include <stdio.h>
#include <string.h>
#include "fm.h"

#ifdef JP_CHARSET
#include "Str.h"

#ifdef DEBUG
#include <malloc.h>
#endif				/* DEBUG */

#define	uchar		unsigned char

#ifdef TRUE
#undef TRUE
#endif
#ifdef FALSE
#undef FALSE
#endif
#define	TRUE		1
#define	FALSE		0
#ifdef ESC_CODE
#undef ESC_CODE
#endif
#define ESC_CODE	'\033'

#define CODE_NORMAL	0x00
#define CODE_OK		0x01
#define CODE_BROKEN	0x02
#define CODE_ERROR	0x04
#define EUC_NOSTATE	0x00
#define EUC_MBYTE1	0x10
#define EUC_SS2		0x20
#define EUC_SS3		0x40
#define SJIS_NOSTATE	0x00
#define SJIS_SHIFT_L	0x10
#define SJIS_SHIFT_H	0x20
#define ISO_NOSTATE	0x00
#define ISO_ESC		0x10
#define ISO_CS94	0x20
#define ISO_MBCS	0x40
#define ISO_MBYTE1	0x80
#define CODE_STATE(c)	((c) & 0x0f)
#define EUC_STATE(c)	((c) & 0xf0)
#define SJIS_STATE(c)	((c) & 0xf0)
#define ISO_STATE(c)	((c) & 0xf0)

#define CSET_ASCII	0
#define CSET_X0208	1
#define CSET_X0201K	2
#define CSET_UNKNOWN	3

#define	CODES		7	/* Number of supported Kanji code */

#define	JSIcode  "\033$@"
#define	JSOcode  "\033(H"
#define	J2SIcode "\033$@"
#define	J2SOcode "\033(J"
#define	NSIcode  "\033$B"
#define	NSOcode  "\033(J"
#define	N2SIcode  "\033$B"
#define	N2SOcode  "\033(B"
#define	N3SIcode "\033$@"
#define	N3SOcode "\033(B"
#define	USIcode  "\033$"
#define	USOcode  "\033+"

static char *SIcode, *SOcode;

static Str cConvEE(char *is);
static Str cConvEJ(char *is);
static Str cConvES(char *is);
static Str cConvSE(char *is);
static Str cConvJE(char *is);
char checkShiftCode(char *buf, unsigned char);

static char *han2zen_tab[] =
{
    "!!", "!#", "!V", "!W", "!\"", "!&", "%r", "%!",
    "%#", "%%", "%'", "%)", "%c", "%e", "%g", "%C",
    "!<", "%\"", "%$", "%&", "%(", "%*", "%+", "%-",
    "%/", "%1", "%3", "%5", "%7", "%9", "%;", "%=",
    "%?", "%A", "%D", "%F", "%H", "%J", "%K", "%L",
    "%M", "%N", "%O", "%R", "%U", "%X", "%[", "%^",
    "%_", "%`", "%a", "%b", "%d", "%f", "%h", "%i",
    "%j", "%k", "%l", "%m", "%o", "%s", "!+", "!,",
};

typedef struct _ConvRoutine {
    char key;
     Str(*routine) ();
    char *ShiftIn, *ShiftOut;
} ConvRoutine;

static ConvRoutine FromEJ[] =
{
    {'J', cConvEJ, JSIcode, JSOcode},
    {'N', cConvEJ, NSIcode, NSOcode},
    {'n', cConvEJ, N2SIcode, N2SOcode},
    {'m', cConvEJ, N3SIcode, N3SOcode},
    {'U', cConvEJ, USIcode, USOcode},
    {'j', cConvEJ, J2SIcode, J2SOcode},
    {'S', cConvES, "", ""},
    {'E', cConvEE, "", ""}
};

static ConvRoutine ToEJ[] =
{
    {'J', cConvJE, JSIcode, JSOcode},
    {'N', cConvJE, NSIcode, NSOcode},
    {'n', cConvJE, N2SIcode, N2SOcode},
    {'m', cConvJE, N3SIcode, N3SOcode},
    {'U', cConvJE, USIcode, USOcode},
    {'j', cConvJE, J2SIcode, J2SOcode},
    {'S', cConvSE, "", ""},
    {'E', cConvEE, "", ""}
};

char *
GetSICode(char key)
{
    int i;
    for (i = 0; i < CODES; i++)
	if (FromEJ[i].key == key)
	    return FromEJ[i].ShiftIn;
    return "";
}

char *
GetSOCode(char key)
{
    int i;
    for (i = 0; i < CODES; i++)
	if (FromEJ[i].key == key)
	    return FromEJ[i].ShiftOut;
    return "";
}

static void
n_impr(char s)
{
    fprintf(stderr, "conv: option %c(0x%02x) is not implemented yet... sorry\n", s, s);
    exit(1);
}

Str
conv(char *is, char fc, char tc)
{
    int i;
    Str os;

    if (fc == 'E' && tc == 'E')
	return cConvEE(is);
    if (fc == tc || fc == '\0' || tc == '\0')
	return Strnew_charp(is);

    if (fc == 'E')
	os = Strnew_charp(is);
    else {
	for (i = 0; i < CODES; i++) {
	    if (ToEJ[i].key == fc) {
		os = (*ToEJ[i].routine) (is);
		goto next;
	    }
	}
	n_impr(fc);
	return NULL;
    }
  next:
    if (tc == 'E')
	return os;
    else {
	for (i = 0; i < CODES; i++) {
	    if (FromEJ[i].key == tc) {
		SIcode = FromEJ[i].ShiftIn;
		SOcode = FromEJ[i].ShiftOut;
		return (*FromEJ[i].routine) (os->ptr);
	    }
	}
	n_impr(tc);
	return NULL;
    }
}

static uchar
getSLb(uchar * ptr, uchar * ub)
{				/* Get Shift-JIS Lower byte */
    uchar c = *ptr;

    *ub <<= 1;
    if (c < 0x9f) {
	if (c > 0x7e)
	    c--;
	*ub -= 1;
	c -= 0x3f;
    }
    else {
	c -= 0x9e;
    }
    return c;
}

static Str
cConvSE(char *is)
{				/* Convert Shift-JIS to EUC-JP */
    uchar *p, ub, lb;
    int state = SJIS_NOSTATE;
    Str os = Strnew_size(strlen(is));

    for (p = (uchar *) is; *p != '\0'; p++) {
	switch (state) {
	case SJIS_NOSTATE:
	    if (!(*p & 0x80))	/* ASCII */
		Strcat_char(os, (char) (*p));
	    else if (0x80 <= *p && *p <= 0x9f) {	/* JIS X 0208,
							 * 0213 */
		ub = *p & 0x7f;
		state = SJIS_SHIFT_L;
	    }
	    else if (0xe0 <= *p && *p <= 0xef) {	/* JIS X 0208 */
		/* } else if (0xe0 <= *p && *p <= 0xfc) { *//* JIS X 0213 */
		ub = (*p & 0x7f) - 0x40;
		state = SJIS_SHIFT_H;
	    }
	    else if (0xa0 <= *p && *p <= 0xdf) {	/* JIS X 0201-Kana 
							 */
		Strcat_char(os, (char) (han2zen_tab[*p - 0xa0][0] | 0x80));
		Strcat_char(os, (char) (han2zen_tab[*p - 0xa0][1] | 0x80));
	    }
	    break;
	case SJIS_SHIFT_L:
	case SJIS_SHIFT_H:
	    if ((0x40 <= *p && *p <= 0x7e) ||
		(0x80 <= *p && *p <= 0xfc)) {	/* JIS X 0208, 0213 */
		lb = getSLb(p, &ub);
		ub += 0x20;
		lb += 0x20;
		Strcat_char(os, (char) (ub | 0x80));
		Strcat_char(os, (char) (lb | 0x80));
	    }
	    else if (!(*p & 0x80))	/* broken ? */
		Strcat_char(os, (char) (*p));
	    state = SJIS_NOSTATE;
	    break;
	}
    }
    return os;
}

static Str
cConvJE(char *is)
{				/* Convert ISO-2022-JP to EUC-JP */
    uchar *p, ub;
    char cset = CSET_ASCII;
    int state = ISO_NOSTATE;
    Str os = Strnew_size(strlen(is));

    for (p = (uchar *) is; *p != '\0'; p++) {
	switch (state) {
	case ISO_NOSTATE:
	    if (*p == ESC_CODE)	/* ESC sequence */
		state = ISO_ESC;
	    else if (cset == CSET_ASCII || *p < 0x21)
		Strcat_char(os, (char) (*p));
	    else if (cset == CSET_X0208 && *p <= 0x7e) {
		/* JIS X 0208 */
		ub = *p;
		state = ISO_MBYTE1;
	    }
	    else if (cset == CSET_X0201K && *p <= 0x5f) {
		/* JIS X 0201-Kana */
		Strcat_char(os, (char) (han2zen_tab[*p - 0x20][0] | 0x80));
		Strcat_char(os, (char) (han2zen_tab[*p - 0x20][1] | 0x80));
	    }
	    break;
	case ISO_MBYTE1:
	    if (*p == ESC_CODE)	/* ESC sequence */
		state = ISO_ESC;
	    else if (0x21 <= *p && *p <= 0x7e) {	/* JIS X 0208 */
		Strcat_char(os, (char) (ub | 0x80));
		Strcat_char(os, (char) (*p | 0x80));
		state = ISO_NOSTATE;
	    }
	    else {
		Strcat_char(os, (char) (*p));
		state = ISO_NOSTATE;
	    }
	    break;
	case ISO_ESC:
	    if (*p == '(')	/* ESC ( F */
		state = ISO_CS94;
	    else if (*p == '$')	/* ESC $ F, ESC $ ( F */
		state = ISO_MBCS;
	    else {
		Strcat_char(os, ESC_CODE);
		Strcat_char(os, (char) (*p));
		state = ISO_NOSTATE;
	    }
	    break;
	case ISO_CS94:
	    if (*p == 'B' || *p == 'J' || *p == 'H')
		cset = CSET_ASCII;
	    else if (*p == 'I')
		cset = CSET_X0201K;
	    else {
		Strcat_char(os, ESC_CODE);
		Strcat_char(os, '(');
		Strcat_char(os, (char) (*p));
	    }
	    state = ISO_NOSTATE;
	    break;
	case ISO_MBCS:
	    if (*p == '(') {	/* ESC $ ( F */
		state = ISO_MBCS | ISO_CS94;
		break;
	    }
	case ISO_MBCS | ISO_CS94:
	    if (*p == 'B' || *p == '@')
		cset = CSET_X0208;
	    else {
		Strcat_char(os, ESC_CODE);
		Strcat_char(os, '$');
		if (state == (ISO_MBCS | ISO_CS94))
		    Strcat_char(os, '(');
		Strcat_char(os, (char) (*p));
	    }
	    state = ISO_NOSTATE;
	    break;
	}
    }
    return os;
}

static Str
_cConvEE(char *is, char is_euc)
{				/* Convert EUC-JP to EUC-JP / ISO-2022-JP
				 * (no JIS X 0201-Kana, 0212, 0213-2) */
    uchar *p, ub, euc = 0;
    int state = EUC_NOSTATE;
    char cset = CSET_ASCII;
    Str os;

    if (is_euc) {
	os = Strnew_size(strlen(is));
	euc = 0x80;
    }
    else
	os = Strnew_size(strlen(is) * 3 / 2);

    for (p = (uchar *) is; *p != '\0'; p++) {
	switch (state) {
	case EUC_NOSTATE:
	    if (!(*p & 0x80)) {	/* ASCII */
		if (!is_euc && cset != CSET_ASCII) {
		    Strcat_charp(os, SOcode);
		    cset = CSET_ASCII;
		}
		Strcat_char(os, (char) (*p));
	    }
	    else if (0xa1 <= *p && *p <= 0xfe) {	/* JIS X 0208,
							 * 0213-1 */
		ub = *p;
		state = EUC_MBYTE1;
	    }
	    else if (*p == 0x8e)	/* SS2 + JIS X 0201-Kana */
		state = EUC_SS2;
	    else if (*p == 0x8f)	/* SS3 + JIS X 0212, 0213-2 */
		state = EUC_SS3;
	    break;
	case EUC_MBYTE1:
	    if (0xa1 <= *p && *p <= 0xfe) {	/* JIS X 0208, 0213-1 */
		if (!is_euc && cset != CSET_X0208) {
		    Strcat_charp(os, SIcode);
		    cset = CSET_X0208;
		}
		Strcat_char(os, (char) ((ub & 0x7f) | euc));
		Strcat_char(os, (char) ((*p & 0x7f) | euc));
	    }
	    else if (!(*p & 0x80)) {	/* broken ? */
		if (!is_euc && cset != CSET_ASCII) {
		    Strcat_charp(os, SOcode);
		    cset = CSET_ASCII;
		}
		Strcat_char(os, (char) (*p));
	    }
	    state = EUC_NOSTATE;
	    break;
	case EUC_SS2:
	    if (0xa0 <= *p && *p <= 0xdf) {	/* JIS X 0201-Kana */
		if (!is_euc && cset != CSET_X0208) {
		    Strcat_charp(os, SIcode);
		    cset = CSET_X0208;
		}
		Strcat_char(os, (char) (han2zen_tab[*p - 0xa0][0] | euc));
		Strcat_char(os, (char) (han2zen_tab[*p - 0xa0][1] | euc));
	    }
	    state = EUC_NOSTATE;
	    break;
	case EUC_SS3:
	    state = (EUC_SS3 | EUC_MBYTE1);
	    break;
	case EUC_SS3 | EUC_MBYTE1:
	    state = EUC_NOSTATE;
	    break;
	}
    }
    if (!is_euc && cset != CSET_ASCII)
	Strcat_charp(os, SOcode);
    return os;
}

static Str
cConvEE(char *is)
{
    return _cConvEE(is, TRUE);
}

static Str
cConvEJ(char *is)
{
    return _cConvEE(is, FALSE);
}

static void
put_js(Str os, uchar ub, uchar lb)
{
    ub -= 0x20;
    lb -= 0x20;
    if ((ub & 1) == 0)
	lb += 94;
    ub = ((ub - 1) >> 1) + 0x81;
    lb += 0x3f;
    if (ub > 0x9f)
	ub += 0x40;
    if (lb > 0x7e)
	lb++;

    Strcat_char(os, (char) (ub));
    Strcat_char(os, (char) (lb));
}

static Str
cConvES(char *is)
{				/* Convert EUC-JP to Shift-JIS */
    uchar *p, ub;
    int state = EUC_NOSTATE;
    Str os = Strnew_size(strlen(is));

    for (p = (uchar *) is; *p != '\0'; p++) {
	switch (state) {
	case EUC_NOSTATE:
	    if (!(*p & 0x80))	/* ASCII */
		Strcat_char(os, (char) (*p));
	    else if (0xa1 <= *p && *p <= 0xfe) {	/* JIS X 0208,
							 * 0213-1 */
		ub = *p;
		state = EUC_MBYTE1;
	    }
	    else if (*p == 0x8e)	/* SS2 + JIS X 0201-Kana */
		state = EUC_SS2;
	    else if (*p == 0x8f)	/* SS3 + JIS X 0212, 0213-2 */
		state = EUC_SS3;
	    break;
	case EUC_MBYTE1:
	    if (0xa1 <= *p && *p <= 0xfe)	/* JIS X 0208, 0213-1 */
		put_js(os, ub & 0x7f, *p & 0x7f);
	    else if (!(*p & 0x80))	/* broken ? */
		Strcat_char(os, (char) (*p));
	    state = EUC_NOSTATE;
	    break;
	case EUC_SS2:
	    if (0xa0 <= *p && *p <= 0xdf)	/* JIS X 0201-Kana */
		put_js(os, han2zen_tab[*p - 0xa0][0],
		       han2zen_tab[*p - 0xa0][1]);
	    state = EUC_NOSTATE;
	    break;
	case EUC_SS3:
	    state = (EUC_SS3 | EUC_MBYTE1);
	    break;
	case EUC_SS3 | EUC_MBYTE1:
	    state = EUC_NOSTATE;
	    break;
	}
    }
    return os;
}

/* 
 * static unsigned short sjis_shift[8] = { 0x7fff, 0xffff, 0x0, 0x0, 0x0,
 * 0x0, 0xffff, 0x0 }; static unsigned short sjis_second[16] = { 0x0, 0x0, 
 * 0x0, 0x0, 0xffff, 0xffff, 0xffff, 0xfffe, 0xffff, 0xffff, 0xffff,
 * 0xffff, 0xffff, 0xffff, 0xffff, 0xfff8 }; */

char
checkShiftCode(char *buf, unsigned char hint)
{
    uchar *p, si = '\0', so = '\0';
    int euc = (CODE_NORMAL | EUC_NOSTATE), sjis = (CODE_NORMAL | SJIS_NOSTATE),
     iso = (CODE_NORMAL | ISO_NOSTATE), iso_kana = CODE_NORMAL;

    p = (uchar *) buf;
    while (1) {
	if (iso != CODE_ERROR && (si == '\0' || so == '\0')) {
	    switch (ISO_STATE(iso)) {
	    case ISO_NOSTATE:
		if (*p == ESC_CODE)	/* ESC sequence */
		    iso = (CODE_STATE(iso) | ISO_ESC);
		break;
	    case ISO_ESC:
		if (*p == '(')	/* ESC ( F */
		    iso = (CODE_STATE(iso) | ISO_CS94);
		else if (*p == '$')	/* ESC $ F, ESC $ ( F */
		    iso = (CODE_STATE(iso) | ISO_MBCS);
		else
		    iso = (CODE_STATE(iso) | ISO_NOSTATE);
		break;
	    case ISO_CS94:
		if (*p == 'B' || *p == 'J' || *p == 'H')
		    so = *p;
		else if (*p == 'I')
		    iso_kana = CODE_OK;
		iso = (CODE_STATE(iso) | ISO_NOSTATE);
		break;
	    case ISO_MBCS:
		if (*p == '(') {	/* ESC $ ( F */
		    iso = (CODE_STATE(iso) | ISO_MBCS | ISO_CS94);
		    break;
		}
	    case ISO_MBCS | ISO_CS94:
		if (*p == 'B' || *p == '@')
		    si = *p;
		iso = (CODE_STATE(iso) | ISO_NOSTATE);
		break;
	    }
	    if (*p & 0x80)
		iso = CODE_ERROR;
	}
	if (euc != CODE_ERROR) {
	    switch (EUC_STATE(euc)) {
	    case EUC_NOSTATE:
		if (!(*p & 0x80))	/* ASCII */
		    ;
		else if (0xa1 <= *p && *p <= 0xfe)	/* JIS X 0208,
							 * 0213-1 */
		    euc = (CODE_STATE(euc) | EUC_MBYTE1);
		else if (*p == 0x8e)	/* SS2 + JIS X 0201-Kana */
		    euc = (CODE_STATE(euc) | EUC_SS2);
		else if (*p == 0x8f)	/* SS3 + JIS X 0212, 0213-2 */
		    euc = (CODE_STATE(euc) | EUC_SS3);
		else
		    euc = CODE_ERROR;
		break;
	    case EUC_MBYTE1:
		if (CODE_STATE(euc) == CODE_NORMAL)
		    euc = CODE_OK;
	    case EUC_SS3 | EUC_MBYTE1:
		if (0xa1 <= *p && *p <= 0xfe)	/* JIS X 0208, 0213-1 */
		    euc = (CODE_STATE(euc) | EUC_NOSTATE);
		else if (euc & CODE_BROKEN)
		    euc = CODE_ERROR;
		else
		    euc = (CODE_BROKEN | EUC_NOSTATE);
		break;
	    case EUC_SS2:
		if (0xa0 <= *p && *p <= 0xdf)	/* JIS X 0201-Kana */
		    euc = (CODE_STATE(euc) | EUC_NOSTATE);
		else
		    euc = CODE_ERROR;
		break;
	    case EUC_SS3:
		if (0xa1 <= *p && *p <= 0xfe)	/* JIS X 0212, 0213-2 */
		    euc = (CODE_STATE(euc) | EUC_SS3 | EUC_MBYTE1);
		else
		    euc = CODE_ERROR;
		break;
	    }
	}
	if (sjis != CODE_ERROR) {
	    switch (SJIS_STATE(sjis)) {
	    case SJIS_NOSTATE:
		if (!(*p & 0x80))	/* ASCII */
		    ;
		else if (0x81 <= *p && *p <= 0x9f)
		    sjis = (CODE_STATE(sjis) | SJIS_SHIFT_L);
		else if (0xe0 <= *p && *p <= 0xef)	/* JIS X 0208 */
		    /* else if (0xe0 <= *p && *p <= 0xfc) */
		    /* JIS X 0213 */
		    sjis = (CODE_STATE(sjis) | SJIS_SHIFT_H);
		else if (0xa0 <= *p && *p <= 0xdf)	/* JIS X 0201-Kana 
							 */
		    ;
		else
		    sjis = CODE_ERROR;
		break;
	    case SJIS_SHIFT_L:
	    case SJIS_SHIFT_H:
		if (CODE_STATE(sjis) == CODE_NORMAL)
		    sjis = CODE_OK;
		if ((0x40 <= *p && *p <= 0x7e) ||
		    (0x80 <= *p && *p <= 0xfc))		/* JIS X 0208,
							 * 0213 */
		    sjis = (CODE_STATE(sjis) | SJIS_NOSTATE);
		else if (sjis & CODE_BROKEN)
		    sjis = CODE_ERROR;
		else
		    sjis = (CODE_BROKEN | SJIS_NOSTATE);
		break;
	    }
	}
	if (euc == CODE_ERROR || sjis == CODE_ERROR)
	    break;
	if (*p == '\0')
	    break;
	p++;
    }
    if (iso != CODE_ERROR) {
	if (si == '\0' && so == '\0' && iso_kana != CODE_OK)
	    return '\0';
	switch (si) {
	case '@':
	    switch (so) {
	    case 'H':
		return 'J';
	    case 'J':
		return 'j';
	    case 'B':
		return 'm';
	    default:
		return 'm';
	    }
	case 'B':
	    switch (so) {
	    case 'J':
		return 'N';
	    case 'B':
		return 'n';
	    default:
		return 'n';
	    }
	default:
	    switch (so) {
	    case 'H':
		return 'J';
	    case 'J':
		return 'N';
	    case 'B':
		return 'n';
	    default:
		return 'n';
	    }
	}
    }
    if (hint == 'E') {
	if (euc != CODE_ERROR)
	    return 'E';
	if (sjis != CODE_ERROR)
	    return 'S';
	return 'E';
    }
    if (hint == 'S') {
	if (sjis != CODE_ERROR)
	    return 'S';
	if (euc != CODE_ERROR)
	    return 'E';
	return 'S';
    }
    if (CODE_STATE(euc) == CODE_OK)
	return 'E';
    if (CODE_STATE(sjis) == CODE_OK)
	return 'S';
    if (CODE_STATE(euc) == CODE_NORMAL)
	return 'E';
    if (CODE_STATE(sjis) == CODE_NORMAL)
	return 'S';
    return 'E';
}
#endif				/* JP_CHARSET */
