/*
 * Copyright (c) 1997, 1998, 1999  Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * This program requires the following Autoconf macros:
 *   AC_C_CONST
 *   AC_HEADER_STDC
 *   AC_CHECK_HEADERS(string.h, memory.h, unistd.h)
 *   AC_CHECK_FUNCS(strchr)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <syslog.h>

#if defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#else /* not STDC_HEADERS and not HAVE_STRING_H */
#include <strings.h>
#endif /* not STDC_HEADERS and not HAVE_STRING_H */

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifndef HAVE_STRCHR
#define strchr index
#define strrchr rindex
#endif /* HAVE_STRCHR */

#include "readconf.h"

#ifdef __STDC__
static Configuration *find_directive(Configuration *, const char *,
    const char *);
static void reset_configuration(Configuration *, const char *);
static Configuration *check_configuration(Configuration *, const char *);
#else /* not __STDC__ */
static Configuration *find_directive();
static void reset_configuration();
static Configuration *check_configuration();
#endif /* not __STDC__ */

#ifdef USE_FAKELOG
#include "fakelog.h"
#endif

#define READCONF_IS_REDEFINABLE(t) \
((t) == READCONF_ZERO_OR_MORE || (t) == READCONF_ONCE_OR_MORE)

#define READCONF_IS_REQUIRED(t) \
((t) == READCONF_ONCE || (t) == READCONF_ONCE_OR_MORE)


/*
 * Read configurations from the file `filename'.
 * `table' is a table which lists all recognized directives.
 * If no error occurrs, 0 is returned.  Otherwise -1 is returned.
 */
int
read_configuration(filename, table)
    const char *filename;
    Configuration *table;
{
    Configuration *cfg, *lackcfg;
    FILE *file;
    char linebuf[READCONF_SIZE_LINE + 1];
    char block[READCONF_SIZE_BLKNAME + 1];
    char *p, *wsp;
    char *name, *arg;
    int lineno = 0;


    /*
     * Reset counters in the directive table.
     */
    block[0] = '\0';
    reset_configuration(table, block);

    /*
     * Call a pre-process hook.
     */
    cfg = find_directive(table, block, "begin");
    if (cfg != NULL && cfg->function(NULL, filename, lineno) < 0)
	return -1;

    /*
     * Open the configuration file.
     */
    file = fopen(filename, "r");
    if (file == NULL) {
	syslog(LOG_ERR, "%s: cannot open the file, %m", filename);
	goto failed;
    }
    syslog(LOG_DEBUG, "%s: debug: open the file", filename);

    /*
     * Read each line in the configuration file.
     */
    while (fgets(linebuf, READCONF_SIZE_LINE + 1, file) != NULL) {
	lineno++;

	/* 
	 * Check for the line length.
	 * An error occurs if it exceeds a limit.
	 */
	p = strchr(linebuf, '\n');
	if (p != NULL)
	    *p = '\0';
	else if (strlen(linebuf) == READCONF_SIZE_LINE) {
	    *(linebuf + READCONF_SIZE_LINE) = '\0';
	    syslog(LOG_ERR, "%s:%d: too long line: %s...", filename,
		lineno, linebuf);
	    goto failed;
	}

	/*
	 * Get a directive name.
	 */
	for (name = linebuf; *name == ' ' || *name == '\t'; name++)
	    ;

	/*
	 * Ignore this line if the line is empty or starts with '#'.
	 */
	if (*name == '#') {
	    syslog(LOG_DEBUG, "%s:%d: debug: comment line", filename,
		lineno);
	    continue;
	}
	if (*name == '\0') {
	    syslog(LOG_DEBUG, "%s:%d: debug: empty line", filename,
		lineno);
	    continue;
	}

	/*
	 * Get an argument to the directive name.
	 */
	for (p = name; *p != ' ' && *p != '\t' && *p != '\0'; p++)
	    ;
	if (*p != '\0') {
	    *p++ = '\0';
	    while (*p == ' ' || *p == '\t')
		p++;
	}
	if (*p == '\0')
	    arg = NULL;
	else
	    arg = p;

	/*
	 * Delete spaces and tabls in the tail of the line.
	 */
	if (arg != NULL) {
	    for (p = arg, wsp = NULL; *p != '\0'; p++) {
		if (*p == ' ' || *p == '\t') {
		    if (wsp == NULL)
			wsp = p;
		} else
		    wsp = NULL;
	    }
	    if (wsp != NULL)
		*wsp = '\0';
	}

	/*
	 * Check for the directive name.
	 */
	if (strcmp(name, "begin") == 0) {
	    char arg_begin[(READCONF_SIZE_BLKNAME * 2) + 1];

	    /*
	     * `begin' keyword.  Beginning of the block.
	     * Check for the argument.
	     */
	    if (arg == NULL) {
		syslog(LOG_ERR, "%s:%d: missing argument: begin", filename,
		    lineno);
		goto failed;
	    }
	    if (READCONF_SIZE_BLKNAME < strlen(block) + 1 + strlen(arg)) {
		syslog(LOG_ERR, "%s:%d: unkonwn directive name: begin %s",
		    filename, lineno, arg);
		goto failed;
	    }

	    /*
	     * Find a directive entry in the table.
	     */
	    strcpy(arg_begin, "begin ");
	    strcat(arg_begin, arg);
	    cfg = find_directive(table, block, arg_begin);
	    if (cfg == NULL) {
		syslog(LOG_ERR, "%s:%d: unknown directive name: begin %s",
		    filename, lineno, arg);
		goto failed;
	    }
	    cfg->count++;
	    if (1 < cfg->count
		&& !READCONF_IS_REDEFINABLE(cfg->deftype)) {
		syslog(LOG_ERR, "%s:%d: directive redefined: begin %s",
		    filename, lineno, arg);
		goto failed;
	    }

	    /*
	     * Update the current block name.
	     */
	    if (block[0] != '\0')
		strcat(block, " ");
	    strcat(block, arg);

	    /*
	     * Reset counters for the block.
	     */
	    reset_configuration(table, block);

	    /*
	     * Dispatch.
	     */
	    if (cfg->function != NULL
		&& cfg->function(NULL, filename, lineno) < 0)
		goto failed;

	} else if (strcmp(name, "end") == 0) {
	    char arg_end[(READCONF_SIZE_BLKNAME * 2) + 1];

	    /*
	     * `end' keyword.  End of the book definition.
	     * Check for the argument.
	     */
	    if (arg != NULL) {
		syslog(LOG_ERR, "%s:%d: unwanted argument: %s", filename,
		    lineno, arg);
		goto failed;
	    }

	    /*
	     * If there is no correspondig `begin' keyword, an error is
	     * caused.
	     */
	    if (block[0] == '\0') {
		syslog(LOG_ERR, "%s:%d: unexpected: end", filename, lineno);
		goto failed;
	    }

	    /*
	     * Checks for counters.
	     * If required directive is not defined, an error occurs.
	     */
	    lackcfg = check_configuration(table, block);
	    if (lackcfg != NULL) {
		syslog(LOG_ERR, "%s:%d: missing directive: %s", filename,
		    lineno, lackcfg->name);
		goto failed;
	    }

	    /*
	     * Update the current block name, and get a directive
	     * name to be found by `find_directive()'.
	     */
	    p = strrchr(block, ' ');
	    strcpy(arg_end, "end ");
	    if (p != NULL) {
		strcat(arg_end, p + 1);
		*p = '\0';
	    } else {
		strcat(arg_end, block);
		block[0] = '\0';
	    }

	    /*
	     * Dispatch.
	     */
	    cfg = find_directive(table, block, arg_end);
	    if (cfg == NULL) {
		syslog(LOG_ERR, "%s:%d: unexpected: end", filename, lineno);
		goto failed;
	    }
	    if (cfg->function != NULL
		&& cfg->function(NULL, filename, lineno) < 0)
		goto failed;

        } else {
	    /*
	     * It is a directive.
	     * Look for a directive entry in the table.
	     */
	    if (arg == NULL) {
		syslog(LOG_ERR, "%s:%d: missing argument: %s", filename,
		    lineno, name);
		goto failed;
	    }
	    cfg = find_directive(table, block, name);
	    if (cfg == NULL) {
		syslog(LOG_ERR, "%s:%d: unknown directive: %s", filename,
		    lineno, name);
		goto failed;
	    }
	    cfg->count++;
	    if (1 < cfg->count
		&& !READCONF_IS_REDEFINABLE(cfg->deftype)) {
		syslog(LOG_ERR, "%s:%d: directive redefined: %s", filename,
		    lineno, name);
		goto failed;
	    }

	    /*
	     * Dispatch.
	     */
	    if (cfg->function != NULL
		&& cfg->function(arg, filename, lineno) < 0)
		goto failed;
	}
    }

    /*
     * End of the configuration file.
     * Check for balance of "begin" and "end".
     */
    if (block[0] != '\0') {
	syslog(LOG_ERR, "%s: missing: end", filename, lineno);
	goto failed;
    }

    /*
     * Checks for counters in the directive table.
     * If required directive is not defined, an error occurs.
     */
    lackcfg = check_configuration(table, block);
    if (lackcfg != NULL) {
	syslog(LOG_ERR, "%s: missing directive: %s", filename, lackcfg->name);
	    goto failed;
    }

    /*
     * Call a post-process hook.
     */
    cfg = find_directive(table, block, "end");
    if (cfg != NULL && cfg->function(NULL, filename, lineno) < 0)
	goto failed;

    /*
     * Close the configuration file.
     */
    if (fclose(file) == EOF) {
	syslog(LOG_ERR, "%s: cannot close the file, %m", filename);
	file = NULL;
	goto failed;
    }
    syslog(LOG_DEBUG, "%s: debug: close the file", filename);

    return 0;
    
    /*
     * An error occurs...
     */
  failed:
    if (file != NULL && fclose(file) == EOF)
	syslog(LOG_ERR, "%s: cannot close the file, %m", filename);
    return -1;
}


/*
 * Find the directive which has `block' and `name' pair.
 *
 * If found, the pointer to the entry is returned.  Otherwise NULL is
 * returned.
 */
static Configuration *
find_directive(table, block, name)
    Configuration *table;
    const char *block;
    const char *name;
{
    Configuration *cfg;

    for (cfg = table; cfg->name != NULL; cfg++) {
	if (strcmp(cfg->name, name) == 0 && strcmp(cfg->block, block) == 0)
	    return cfg;
    }
    return NULL;
}


/*
 * Reset counters which have `block' name in the table.
 */
static void
reset_configuration(table, block)
    Configuration *table;
    const char *block;
{
    Configuration *cfg;

    for (cfg = table; cfg->name != NULL; cfg++) {
	if (cfg->block != NULL && strcmp(cfg->block, block) == 0)
	    cfg->count = 0;
    }
}


/*
 * Check for counters in the configuration table.
 *
 * If required directive is not defined, its name is returned.
 * When more than one required table are undefined, the ealiest one in
 * the table is returned.
 *
 * If all required table are defined, NULL is returned.
 */
static Configuration *
check_configuration(table, block)
    Configuration *table;
    const char *block;
{
    Configuration *cfg;

    for (cfg = table; cfg->name != NULL; cfg++) {
	if (cfg->block != NULL && strcmp(cfg->block, block) == 0
	    && cfg->count == 0 && READCONF_IS_REQUIRED(cfg->deftype))
	    return cfg;
    }
    return NULL;
}


