/*
 * Copyright (c) 1997, 1998  Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * This program requires the following Autoconf macros:
 *   AC_C_CONST
 *   AC_TYPE_SIZE_T
 *   AC_HEADER_STDC
 *   AC_CHECK_HEADERS(string.h, memory.h, stdlib.h, unistd.h)
 *   AC_CHECK_FUNCS(memcpy, strtol, sigprocmask, sigsetjmp)
 *   AC_TYPE_SIGNAL
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <signal.h>
#include <syslog.h>
#include <setjmp.h>
#include <errno.h>

#if defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#else /* not STDC_HEADERS and not HAVE_STRING_H */
#include <strings.h>
#endif /* not STDC_HEADERS and not HAVE_STRING_H */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "ident.h"

#ifdef USE_FAKELOG
#include "fakelog.h"
#endif

#ifndef HAVE_MEMCPY
#define memcpy(d, s, n) bcopy((s), (d), (n))
#ifdef __STDC__
void *memchr(const void *, int, size_t);
int memcmp(const void *, const void *, size_t);
void *memmove(void *, const void *, size_t);
void *memset(void *, int, size_t);
#else /* not __STDC__ */
char *memchr();
int memcmp();
char *memmove();
char *memset();
#endif /* not __STDC__ */
#endif /* not HAVE_MEMCPY */

#ifndef HAVE_STRTOL
#ifdef __STDC__
long strtol(const char *, char **, int);
#else /* not __STDC__ */
long strtol();
#endif /* not __STDC__ */
#endif /* not HAVE_STRTOL */

/*
 * Unexported functions.
 */
#ifdef __STDC__
static RETSIGTYPE ident_timeout(int);
#else
static RETSIGTYPE ident_timeout();
#endif

/*
 * The port number of Identification protocol.
 */
#ifndef RFC1413_PORT
#define RFC1413_PORT		113
#endif

/*
 * The maximum length of request and response lines in Identification
 * protocol.  (default: 1023)
 */
#ifndef MAXLEN_RFC1413_LINE
#define MAXLEN_RFC1413_LINE	511
#endif

/*
 * The maximum number of arguments in a response line in Identification
 * protocol.  The get_ideint() function assumes that arguments in a
 * response line consist of following 5 fields.
 *
 *    <port-on-server>, <port-on-client> : <status> : <opsys> : <user-id>
 */
#define MAX_RFC1413_ARGS		5

/*
 * Argument separators in a response line in Identification protocol.
 */
static char separators[MAX_RFC1413_ARGS] = {',', ':', ':', ':', '\0'};

/*
 * The buffer for setjmp() or sigsetjmp().
 */
#ifdef HAVE_SIGSETJMP
sigjmp_buf jump_buffer;
#else
jmp_buf jump_buffer;
#endif


/*
 * Get a remote user-id of the socket `file' by Identification protocol
 * described in RFC1413.
 *
 * The user-id is put into `user'.
 * `Timeout' specifies seconds until the server gives up to get an user-id.
 *
 * If succeed, 0 is returned.  Otherwise -1 is returned and `user' is
 * set to UNKNOWN_USER.
 */
int
identify_user(file, user, timeout)
    int file;
    char *user;
    int timeout;
{
#ifdef HAVE_SIGPROCMASK
    struct sigaction act;
#endif
    static int idfile;
    struct sockaddr_in myaddr, hisaddr;
    char linebuf[MAXLEN_RFC1413_LINE + 1];
    char *args[MAX_RFC1413_ARGS];
    char *bufp, *wsp;
    int dstport, srcport;
    int size;
    int opt;
    int len;
    int i;

    /*
     * Initialize static variables.
     */
    idfile = -1;

    /*
     * Get an address of mine and remote.
     */
    size = sizeof(myaddr);
    if (getsockname(file, (struct sockaddr *)&myaddr, &size) < 0) {
	syslog(LOG_ERR, "RFC1413 getsockname() failed, %m");
	goto failed;
    }
    srcport = ntohs(myaddr.sin_port);
    myaddr.sin_port = htons(0);

    size = sizeof(hisaddr);
    if (getpeername(file, (struct sockaddr *)&hisaddr, &size) < 0) {
	syslog(LOG_ERR, "RFC1413 getpeername() failed, %m");
	goto failed;
    }
    dstport = ntohs(hisaddr.sin_port);
    hisaddr.sin_port = htons(RFC1413_PORT);

    /*
     * Set an alarm for timeout.
     */
#ifdef HAVE_SIGSETJMP
    if (sigsetjmp(jump_buffer, 1) != 0)
	goto failed;
#else
    if (setjmp(jump_buffer) != 0)
	goto failed;
#endif

#ifdef HAVE_SIGPROCMASK
    act.sa_handler = ident_timeout;
#ifdef SA_INTERRUPT
    act.sa_flags = SA_INTERRUPT;
#else /* not SA_INTERRUPT */
    act.sa_flags = 0;
#endif /* not SA_INTERRUPT */
    sigemptyset(&act.sa_mask);
    sigaction(SIGALRM, &act, NULL);
#else /* not HAVE_SIGPROCMASK */
    signal(SIGALRM, ident_timeout);
#endif /* not HAVE_SIGPROCMASK */

    alarm(timeout);

    /*
     * Establish a connection to the RFC1413_PORT on the remote host.
     */
    idfile = socket(AF_INET, SOCK_STREAM, 0);
    if (idfile < 0) {
	syslog(LOG_ERR, "RFC1413 socket() failed, %m");
	goto failed;
    }

#ifdef SO_REUSEADDR
    opt = 1;
    if (setsockopt(idfile, SOL_SOCKET, SO_REUSEADDR, (char *)&opt,
	sizeof(opt)) < 0) {
        syslog(LOG_ERR, "RFC1413 setsockopt() failed, %m");
	goto failed;
    }
#endif

    if (bind(idfile, (struct sockaddr *)&myaddr, sizeof(myaddr)) < 0) {
	syslog(LOG_ERR, "RFC1413 bind() failed, %m");
	goto failed;
    }
    if (connect(idfile, (struct sockaddr *)&hisaddr, sizeof(hisaddr)) < 0) {
	syslog(LOG_INFO, "RFC1413 connect() failed, %m");
	goto failed;
    }

    /*
     * Send a request to the remote host.
     */
    sprintf(linebuf, "%d, %d\r\n", dstport, srcport);
    write(idfile, linebuf, strlen(linebuf));

    /*
     * Receive an answer from the remote host.
     */
    len = 0;
    bufp = linebuf;
    for (;;) {
	int n;
	char *nl;

        /*
         * Read a response line.
         */
        n = read(idfile, bufp, MAXLEN_RFC1413_LINE - len);
        if (n < 0) {
	    if (errno == EINTR)
		continue;
            syslog(LOG_INFO, "RFC1413 read() failed, %m");
	    goto failed;
        } else if (n == 0) {
            syslog(LOG_INFO, "RFC1413 receives unexpected EOF");
	    goto failed;
	}

        /*
         * If `<CR>' or `<CR><LF>' is found in the line, remove it and
	 * break the loop.
         */
        nl = (char *)memchr(bufp, '\n', n);
        if (nl != NULL) {
	    *nl = '\0';
	    if (linebuf[0] != '\0' && *(nl - 1) == '\r')
		*(nl - 1) = '\0';
	    break;
	}
        len += n;
        bufp += n;
        if (MAXLEN_RFC1413_LINE < len) {
	    *(linebuf + MAXLEN_RFC1413_LINE) = '\0';
            syslog(LOG_INFO, "RFC1413 receives too long line: %s", linebuf);
	    goto failed;
	}
    }
    syslog(LOG_DEBUG, "debug: RFC1413 receives the line: %s", linebuf);

    /*
     * Close the RFC1413 socket.
     */
    if (shutdown(idfile, 2) < 0) {
	idfile = -1;
	goto failed;
    }

    /*
     * Split the response line to fields.
     */
    i = 0;
    bufp = linebuf;
    for (;;) {
	/*
	 * Skip preceeding spaces and tabs.
	 */
	while (*bufp == ' ' || *bufp == '\t')
	    bufp++;

	args[i] = bufp;

	/*
	 * Skip following spaces and tabs.
	 */
	wsp = NULL;
	while (*bufp != separators[i] && *bufp != '\0') {
	    if (*bufp == ' ' || *bufp == '\t')
		wsp = bufp;
	    else
		wsp = NULL;
	    bufp++;
	}
	if (wsp != NULL)
	    *wsp = '\0';

	i++;
	if (*bufp == '\0')
	    break;
	*bufp++ = '\0';
    }

    /*
     * Check for the fields.
     */
    if (i < 4) {
	syslog(LOG_INFO, "RFC1413 responce has too few argument");
	goto failed;
    }
    if ((int)strtol(args[0], &bufp, 10) != dstport || *bufp != '\0') {
	syslog(LOG_INFO, "RFC1413 responce has unmatched port-on-server");
	goto failed;
    }
    if ((int)strtol(args[1], &bufp, 10) != srcport || *bufp != '\0') {
	syslog(LOG_INFO, "RFC1413 response has unmatched port-on-client");
	goto failed;
    }
    if (strcmp(args[2], "ERROR") == 0) {
	syslog(LOG_INFO, "RFC1413 responce indicates error status: %s",
	    args[3]);
	goto failed;
    } else if (strcmp(args[2], "USERID") != 0) {
	syslog(LOG_INFO, "RFC1413 response has unknown status: %s", args[3]);
	goto failed;
    }
    if (i < 5) {
	syslog(LOG_INFO, "RFC1413 response has too many arguments");
	goto failed;
    }
    if (MAXLEN_USER_ID < strlen(args[4])) {
	*(args[4] + MAXLEN_USER_ID) = '\0';
	syslog(LOG_INFO, "RFC1413 response has too long user-id: %s...",
	    args[4]);
	goto failed;
    }

    /*
     * Put the user-id to `user'.
     */
    strcpy(user, args[4]);
    syslog(LOG_INFO, "RFC1413 get user-id: %s", user);
    return 0;

    /*
     * An error occurs...
     */
  failed:

    /*
     * Close the file for identification.
     */
    if (0 <= idfile) {
	if (shutdown(idfile, 2) < 0)
	    syslog(LOG_INFO, "RFC1413 shutdown() failed, %m");
    }

    /*
     * Set `user' to unknown.
     */
    strncpy(user, UNKNOWN_USER, MAXLEN_USER_ID);
    *(user + MAXLEN_USER_ID) = '\0';

    return 0;
}


/*
 * The signal handler for SIGALRM.
 */
static RETSIGTYPE
ident_timeout(sig)
    int sig;
{
    alarm(0);
    syslog(LOG_INFO, "RFC1413 timeout");

#ifdef HAVE_SIGSETJMP
    siglongjmp(jump_buffer, 1);
#else
    longjmp(jump_buffer, 1);
#endif

#ifdef RETSIGTYPE_VOID
    return;
#else
    return 0;
#endif
}


#ifdef TEST
int
main()
{
    char user[512];
    openlog("test", LOG_PID|LOG_NDELAY, LOG_DAEMON);
    get_ident(0, user, 511, 10);
    write(1, user, strlen(user));
    write(1, "\r\n", 2);
    closelog();
}

#endif /* TEST */
