/*
 * Copyright (c) 1997, 1998  Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * This program requires the following Autoconf macros:
 *   AC_C_CONST
 *   AC_TYPE_SIZE_T
 *   AC_HEADER_STDC
 *   AC_CHECK_HEADERS(string.h, memory.h, unistd.h, limits.h)
 *   AC_CHECK_FUNCS(strchr, memcpy, getcwd)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <sys/types.h>
#include <syslog.h>

#if defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#else /* not STDC_HEADERS and not HAVE_STRING_H */
#include <strings.h>
#endif /* not STDC_HEADERS and not HAVE_STRING_H */

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

#ifndef HAVE_STRCHR
#define strchr index
#define strrchr rindex
#endif /* HAVE_STRCHR */

#ifndef HAVE_MEMCPY
#define memcpy(d, s, n) bcopy((s), (d), (n))
#ifdef __STDC__
void *memchr(const void *, int, size_t);
int memcmp(const void *, const void *, size_t);
void *memmove(void *, const void *, size_t);
void *memset(void *, int, size_t);
#else /* not __STDC__ */
char *memchr();
int memcmp();
char *memmove();
char *memset();
#endif /* not __STDC__ */
#endif

#ifndef HAVE_GETCWD
#define getcwd(d,n) getwd(d)
#endif

/*
 * The maximum length of a filename.
 */
#ifndef PATH_MAX
#ifdef MAXPATHLEN
#define PATH_MAX        MAXPATHLEN
#else /* not MAXPATHLEN */
#define PATH_MAX        1024
#endif /* not MAXPATHLEN */
#endif /* not PATH_MAX */

#ifdef USE_FAKELOG
#include "fakelog.h"
#endif


/*
 * Canonicalize `filename'.
 * Replace `/./' and `/../' in `filename' to equivalent straight
 * forms.
 */
int
canonicalize_filename(filename)
    char *filename;
{
    char curdir[PATH_MAX + 1];
    char *src;
    char *dst;
    char *rslash;
    size_t namelen, curlen;

    /*
     * If `filename' is a relative path, it is converted to an absolute
     * path.
     */
    if (*filename != '/') {
	if (getcwd(curdir, PATH_MAX + 1) == NULL) {
	    syslog(LOG_ERR, "cannot get the current directory, %m");
	    return -1;
	}

	curlen = strlen(curdir);
	namelen = strlen(filename);
	if (PATH_MAX < curlen + 1 + namelen) {
	    syslog(LOG_ERR, "too long filename: %s/...", curdir);
	    return -1;
	}

	memmove(filename + 1 + curlen, filename, namelen + 1);
	*(filename + curlen) = '/';
	memcpy(filename, curdir, curlen);
    }

    /*
     * Replace "." and ".." segments in the filename to equivalent
     * straight forms.
     */
    src = filename;
    dst = filename;
    while (*src != '\0') {
	if (*src != '/') {
	    *dst++ = *src++;
	    continue;
	}

	if (*(src + 1) == '/' || *(src + 1) == '\0') {
	    /*
	     * `//' -- Ignore 2nd slash (`/').
	     */
	    src++;
	    continue;
	} else if (*(src + 1) == '.'
	    && (*(src + 2) == '/' || *(src + 2) == '\0')) {
	    /*
	     * `/.' -- The current segment itself.  Removed.
	     */
	    src += 2;
	} else if (*(src + 1) == '.' && *(src + 2) == '.'
	    && (*(src + 3) == '/' || *(src + 3) == '\0')) {
	    /*
	     * `/..' -- Back to the parent segment.
	     */
	    src += 3;
	    *dst = '\0';
	    rslash = strrchr(filename, '/');
	    if (rslash == NULL)
		dst = filename;
	    else
		dst = rslash;
	} else
	    *dst++ = *src++;
    }
    *dst = '\0';

    /*
     * If the filename becomes empty, set the path to `/'.
     */
    if (*(filename) == '\0') {
	*(filename) = '/';
	*(filename + 1) = '\0';
    }

    return 0;
}

