
#include <cstdio>
#include <iomanip>

#include "config_data.hh"
#include "file_exceps.hh"
#include "file_util.hh"
#include "bad_value.hh"
#include "itemize.hh"
#include "clone_ptr-t.hh"

namespace autil {

  const ConfigData empty_config_data("Empty", 0,0);

  UnknownKey::UnknownKey(const string & k) 
    : key(k)
  {
    message += "The key \"";
    message += key;
    message += "\" is not known";
  }
  
  static const ConfigData::Module a_module = ConfigData::Module();

  ConfigData::ConfigData(const string & n,
			 const KeyInfo  * mainbegin, const KeyInfo * mainend)
    : name_(n),
      main_begin(mainbegin), main_end(mainend),
      extra_begin(0), extra_end(0),
      modules_begin(&a_module), modules_end(&a_module)
  {}

  ConfigData::ConfigData(const string & n,
			 const KeyInfo  * mainbegin, const KeyInfo * mainend, 
			 const Module * modbegin, const Module * modend)
    : name_(n),
      main_begin(mainbegin), main_end(mainend),
      extra_begin(0), extra_end(0),
      modules_begin(modbegin), modules_end(modend)
  {}

  void ConfigData::set_extra(const KeyInfo * begin, const KeyInfo * end) {
    extra_begin = begin;
    extra_end   = end;
  }

  bool ConfigData::retrieve_bool(const string & key) const {
    return (retrieve(key)[0] == 't');
  }

  int ConfigData::retrieve_int(const string & key) const {
    int i;
    sscanf(retrieve(key).c_str(), "%i", &i);
    return i;
  }

  string ConfigData::retrieve(const string & key) const {
    const string & value = StringMap::lookup(key);
    if (value.size()) 
      return value;
    else 
      return get_default(key);
  }

  string ConfigData::retrieve_list(const string & key) const {
    return get_default(key) + ',' + StringMap::lookup(key);
  }

  void ConfigData::retrieve_list(const string & key, 
				 MutableContainer & m) const 
  {
    string value = get_default(key) + ',' + StringMap::lookup(key);
    itemize(value, m);
  }

  template <class T>
  static const T * find(const string & key, const T * i, const T * end) {
    while (i != end) {
      if (key == i->name) 
	return i;
      ++i;
    }
    return i;
  }

  const char * ConfigData::base_name(const char * name) {
    const char * c = strchr(name, '-');
    unsigned int p = c ? c - name : -1;
    if ((p == 3 && (strncmp(name, "add",p) == 0 
		    || strncmp(name, "rem",p) == 0))
	|| (p == 4 && strncmp(name, "dont",p) == 0)) 
      return name + p + 1;
    else
      return name;
  }

  const KeyInfo * ConfigData::keyinfo(const string & key) const {
    const KeyInfo * i;

    i = autil::find(key, main_begin, main_end);
    if (i != main_end) return i;

    i = autil::find(key, extra_begin, extra_end);
    if (i != extra_end) return i;

    string::size_type p = key.find('-');
    if (p == string::npos) throw UnknownKey(key);
    string k(key,0,p);
    const Module * j = autil::find(k, modules_begin, modules_end);
    if (j == modules_end) throw UnknownKey(key);

    i = autil::find(key, j->begin, j->end);
    if (i != j->end) return i;

    throw UnknownKey(key);
  }

  string ConfigData::get_default(const string & key) const {
    const KeyInfo * ki = keyinfo(key);
    string value;
    bool   in_replace = false;
    string replace;
    for(const char * i = ki->def; *i; ++i) {

      if (!in_replace) {

	if (*i == '<') {
	  in_replace = true;
	} else {
	  value += *i;
	}

      } else {

	if (*i == '/' || *i == '|') {

	  char sep = *i;
	  string second;
	  ++i;
	  while (*i != '\0' && *i != '>') second += *i++;
	  if (sep == '/') {
	    value += add_possible_dir(retrieve(replace), retrieve(second));
	  } else { // sep == '|'
	    assert(replace[0] == '$');
	    const char * env = getenv(replace.c_str()+1);
	    value += env ? env : second;
	  }
	  replace = "";
	  in_replace = false;

	} else if (*i == '>') {

	  value += retrieve(replace);
	  replace = "";
	  in_replace = false;

	} else {

	  replace += *i;

	}

      }
      
    }
    return value;
  }


  void ConfigData::simple_add(const string & k, const string & v, AddAction a) {
    if (a == Insert) 
      StringMap::insert(k,v);
    else
      StringMap::replace(k,v);
  }

  void ConfigData::fancy_add(const string & k, const string & value, AddAction action) {
    string::size_type p = k.find('-');
    string key;
    if ((p == 3 && (strncmp(k.c_str(), "add",p) == 0 
		    || strncmp(k.c_str(), "rem",p) == 0))
	|| (p == 4 && strncmp(k.c_str(), "dont",p) == 0)) 
      {
	key.assign(k, p+1);
      } else {
	key.assign(k);
	p = 0;
      }
    const KeyInfo * ki = keyinfo(key);
    assert(ki->def != 0); // if null this key should never have values
                          // directly added to it
    switch (ki->type) {
      
    case KeyInfo::Bool:
      
      if (p == 4 || (p == 0 && value == "false"))
	simple_add(key, "false", action);
      else if (p != 0)
	throw UnknownKey(k);
      else if (value.empty() || value == "true")
	simple_add(key, "true", action);
      else
	throw BadValue(key, value, "either \"true\" or \"false\"");
      break;
      
    case KeyInfo::String:
      
      if (p == 0) 
	simple_add(key,value, action);
      else 
	throw UnknownKey(key);
      break;
      
    case KeyInfo::Int:

      int i;
      if (p == 0 && sscanf(value.c_str(), "%i", &i) == 1 && i >= 0)
	simple_add(key,value, action);
      else if (p != 0)
	throw UnknownKey(key);
      else
	throw BadValue(key, value, "a positive integer");
      break;

    case KeyInfo::List:

      char a;
      if (p == 0)
	a = ' ';
      else if (strncmp(k.c_str(),"rem-all-",8) == 0) {
	if (value.size())
	  throw BadValue(k, value, "nothing");
	a = '!';
      }
      else if (k[0] == 'a')
	a = '+';
      else
	a = '-';
      
      string & s = find(key);
      if (action == Insert) {
	s.insert(0, a + value + ',');
      } else {
	s += ',';
	s += a;
	s += value;
      }
      {
	int i = s.rfind('!');
	if (i != string::npos) {
	  s.erase(0,i);
	}
      }
      break;
    }
  }

  void ConfigData::insert(const string & key, const string & value) {
    fancy_add(key, value, Insert);
  }

  void ConfigData::replace(const string & key, const string & value) {
    fancy_add(key, value, Replace);
  }

  class ConfigData::PossibleElementsEmul : public VirEmulation<const KeyInfo *> 
  {
  private:
    bool include_extra;
    const ConfigData * cd;
    const KeyInfo * i;
    const Module  * m;
  public:
    PossibleElementsEmul(const ConfigData * d, bool ic)
      : include_extra(ic), cd(d), i(d->main_begin), m(0) {}
    PossibleElementsEmul * clone() const {
      return new PossibleElementsEmul(*this);
    }
    void assign(const VirEmulation<const KeyInfo *> * other) {
      *this = *static_cast<const PossibleElementsEmul *>(other);
    }

    Value next() {
      if (i == cd->main_end) {
	if (include_extra)
	  i = cd->extra_begin;
	else
	  i = cd->extra_end;
      }
      
      if (i == cd->extra_end) {
	m = cd->modules_begin;
	if (m == cd->modules_end) return 0;
	else i = m->begin;
      }

      if (m == 0)
	return i++;

      if (m == cd->modules_end)
	return 0;

      if (i == m->end) {
	++m;
	if (m == cd->modules_end) return 0;
	else i = m->begin;
      }

      return i++;
    }

    bool at_end() const {
      return (m == cd->modules_end);
    }
  };

  VirEmulation<const KeyInfo *> * 
  ConfigData::possible_elements(bool include_extra) const 
  {
    return new PossibleElementsEmul(this, include_extra);
  }

  void ConfigData::write_to_stream(ostream & out, bool include_extra) const {
    ios::fmtflags f = out.flags();
    out.setf(ios::left);
    out << "## " << name() << "\n\n";
    Emulation<const KeyInfo *> els = possible_elements(include_extra);
    const KeyInfo * i;
    while (i = els.next(), i != 0) {
      if (i->desc == 0) continue;
      out << "# " 
	  << (i->type ==  KeyInfo::List ? "add|rem-" : "")
	  << i->name << " descrip: " 
	  << (i->def == 0 ? "(action option) " : "")
	  << i->desc << "\n";
      if (i->def != 0) {
	out << "# " << i->name << " default: " << i->def << "\n";
	const string & value = lookup(i->name);
	if (i->type != KeyInfo::List) {
	  out << "# " << i->name << " current: " << retrieve(i->name) << "\n";
	  if (value.size()) 
	    out << setw(15) << i->name << " " << value << "\n";
	} else {
	  Emulation<ListItem> els = split_list(value);
	  ListItem li;
	  while (li = els.next(), li.name != 0) {
	    cout << (li.action == '+' ? "add-" : li.action == '-' ? "rem-" : "rem-all-")
		 << i->name << " " << li.name << "\n";
	  }
	}
      }
      out << "\n\n";
    }
    out.flags(f);
  }
  
}
