#ifndef __aspell_language__
#define __aspell_language__

#include <string>
#include "phonetic.hh"
#include "clone_ptr.hh"

namespace aspell {

  using namespace autil;

  class Config;
  class Manager;

  class Language {
  public:
    enum CasePattern {all_lower, first_upper, all_upper};
    enum CharType {letter, space, other};

    struct SpecialChar {
      bool begin;
      bool middle;
      bool end;
      SpecialChar() : begin(false), middle(false), end(false) {}
      SpecialChar(bool b, bool m, bool e) : begin(b), middle(m), end(e) {}
    };

  private:
    string   name_;
    string   charset_;

    unsigned char to_uchar(char c) const {return static_cast<unsigned char>(c);}

    SpecialChar special_[256];
    char        to_lower_[256];
    char        to_upper_[256];
    char        to_title_[256];
    char        to_sl_[256];
    int         to_uni_[256];
    CharType    char_type_[256];

    string      soundslike_chars_;

    ClonePtr<const Soundslike> soundslike_;

    void setup(string lang, const Config *);

  public:
    Language() {setup("",0);}
    Language(string lang) {setup(lang,0);}
    Language(string lang, const Config & config) {setup(lang,&config);}
    Language(const Config & config) {setup("",&config);}

    const char * name() const {return name_.c_str();}

    CasePattern case_pattern (const char * word) const;
    string fix_case (CasePattern, const char *) const;

    bool trim_n_try (const Manager &sc, const char * word) const;

    bool trim_n_try (const Manager &sc, const string &word) const
    {return trim_n_try(sc, word.c_str());}

    char to_upper(char c) const {return to_upper_[to_uchar(c)];}
    bool is_upper(char c) const {return to_upper(c) == c;}
    string to_upper(const char *word) const {
      string new_word; 
      for (; *word; ++word) 
	new_word += to_upper(*word); 
      return new_word;
    }
    string to_upper(const string &word) const {return to_upper(word.c_str());}

    char to_lower(char c) const {return to_lower_[to_uchar(c)];}
    bool is_lower(char c) const {return to_lower(c) == c;}
    string to_lower(const char *word) const {
      string new_word; 
      for (; *word; ++word) 
	new_word += to_lower(*word); 
      return new_word;
    }

    string to_lower(const string &word) const {return to_lower(word.c_str());}

    char to_title(char c) const {return to_title_[to_uchar(c)];}
    bool is_title(char c) const {return to_title(c) == c;}
    string to_title(const char *word) const {
      string new_word; 
      if (*word)
	new_word += to_title(*word);
      ++word;
      for (; *word; ++word) 
	new_word += to_lower(*word); 
      return new_word;
    }

    string to_title(const string &word) const {return to_title(word.c_str());}
  
    char to_sl(char c) const {return to_sl_[to_uchar(c)];}
  
    int to_uni(char c) const {return to_uni_[to_uchar(c)];}
  
    SpecialChar special(char c) const {return special_[to_uchar(c)];}
  
    CharType char_type(char c) const {return char_type_[to_uchar(c)];}
    bool is_alpha(char c) const {return char_type(c) == letter;}
  
    string to_soundslike(const char * word) const 
    {
      return soundslike_->to_soundslike(word);
    }

    string to_soundslike(const string &word) const {
      return soundslike_->to_soundslike(word);
    }

    const char * soundslike_version() const {
      return soundslike_->version();
    }
    
    bool have_phoneme() const {return false;}
    string to_phoneme(const char * word) const 
    {
      return "";
    }
    string to_phoneme(const string &word) const 
    {
      return "";
    }

    const char * soundslike_chars() const {return soundslike_chars_.c_str();}
  };
  
}

#endif
