/*
 *   cddb - CD Database Management Library
 *
 *   Copyright (C) 1993-1999  Ti Kan
 *   E-mail: ti@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef LINT
static char *_hist_c_ident_ = "@(#)hist.c	6.16 99/03/27";
#endif

#define _CDDB_INTERN	/* Expose internal function protos in cddb.h */

#include "common_d/appenv.h"
#include "common_d/patchlevel.h"
#include "common_d/util.h"
#include "cddb_d/cddb.h"

#define HIST_HYST	10
#define HIST_BANNER	\
"# xmcd %s History file\n# %s\n#\n\
# device disc time type discid categ disc_title\n#\n"

extern appdata_t	app_data;
extern FILE		*errfp;
extern cddb_client_t	*cddb_clinfo;
extern bool_t		ischild;                

STATIC cddb_dlist_t	*cddb_hist_head,	/* History list head */
			*cddb_hist_tail;	/* History list tail */
STATIC cddb_dlist_t	*cddb_chgr_array;	/* CD changer list array */
STATIC int		cddb_hist_mcnt,		/* History list mem cnt */
			cddb_hist_fcnt;		/* History list file cnt */


/***********************
 *  internal routines  *
 ***********************/

/*
 * cddb_hist_writefile
 *	Save the history file as history.bak, and then create a new
 *	history file based on the old history file's contents.
 *
 * Args:
 *	hp1 - Pointer to a cddb_dlist_t structure whose entry is
 *	      to be deleted from the history file.  If NULL, then
 *	      no designated entry will be deleted.
 *	hp2 - Pointer to a cddb_dlist_t structure whose entry is
 *	      to be appended to the history file.  If NULL, then
 *	      no designated entry will be appended.
 *	skip - Truncate the history file by deleting this number of
 *	       old entries.
 *
 * Return:
 *	Error code, or 0 for success.
 */
STATIC int
cddb_hist_writefile(cddb_dlist_t *hp1, cddb_dlist_t *hp2, int skip)
{
	int		i,
			ret;
	FILE		*ifp,
			*ofp;
	char		*homep,
			*histbuf,
			device[FILE_PATH_SZ],
			category[FILE_BASE_SZ],
			dtitle[DLIST_BUF_SZ],
			path[FILE_PATH_SZ + 32],
			bakpath[FILE_PATH_SZ + 32];
	cddb_dlist_t	h;

	homep = util_homedir(util_get_ouid());
	if ((int) strlen(homep) >= FILE_PATH_SZ) {
		/* Path name too long */
		return OPEN_ERR;
	}

	(void) sprintf(path, USR_HIST_PATH, homep);
	(void) sprintf(bakpath, "%s.bak", path);

	/* Allocate temporary buffer */
	if ((histbuf = (char *) MEM_ALLOC("histbuf", DLIST_BUF_SZ)) == NULL)
		return MEM_ERR;

	DBGPRN(errfp, "Writing history file %s\n", path);

	/* Rename old history file */
	(void) UNLINK(bakpath);

#if defined(USE_RENAME)

	if (rename(path, bakpath) < 0) {
		DBGPRN(errfp, "Cannot rename %s -> %s: (errno=%d)\n",
			path, bakpath, errno);
		MEM_FREE(histbuf);
		return LINK_ERR;
	}

#else	/* USE_RENAME */

	if (LINK(path, bakpath) < 0) {
		DBGPRN(errfp, "Cannot link %s -> %s: (errno=%d)\n",
			path, bakpath, errno);
		MEM_FREE(histbuf);
		return LINK_ERR;
	}

	(void) UNLINK(path);

#endif	/* USE_RENAME */

	/* Get history file handles */
	if ((ifp = fopen(bakpath, "r")) == NULL) {
		DBGPRN(errfp, "File open error: %s (errno=%d)\n",
			bakpath, errno);
		MEM_FREE(histbuf);
		return OPEN_ERR;
	}
	if ((ofp = fopen(path, "w")) == NULL) {
		DBGPRN(errfp, "File open error: %s (errno=%d)\n",
			path, errno);
		MEM_FREE(histbuf);
		return OPEN_ERR;
	}

	/* Set file permissions */
	cddb_setperm(path, app_data.hist_filemode);

	/* Write history file header */
	(void) fprintf(ofp, HIST_BANNER, VERSION, COPYRIGHT);

	/* Read from old history file */
	for (i = 1; fgets(histbuf, DLIST_BUF_SZ, ifp) != NULL; i++) {
		/* Skip comments and blank lines */
		if (histbuf[0] == '#' || histbuf[0] == '!' ||
		    histbuf[0] == '\n')
			continue;

		/* Truncate old entries if necessary */
		if (--skip > 0)
			continue;

		if (sscanf(histbuf, "%255s %x %lx %x %x %63s %127[^\n]\n",
			   device,
			   &h.discno,
			   &h.time,
			   &h.type,
			   &h.discid,
			   category,
			   dtitle) != 7) {
			DBGPRN(errfp,
				"Old history file syntax error: line %d\n",
				i);
			continue;
		}

		/* Skip deleted entry */
		if (hp1 != NULL &&
		    h.time == hp1->time && h.discid == hp1->discid)
			continue;

		/* Write to history file */
		if (fprintf(ofp, "%s", histbuf) < 0) {
			ret = errno;
			(void) fclose(ifp);
			(void) fclose(ofp);
			errno = ret;

			DBGPRN(errfp, "File write error: %s (errno=%d)\n",
				path, errno);
			MEM_FREE(histbuf);
			return WRITE_ERR;
		}
	}

	/* Append new entry */
	if (hp2 != NULL) {
		ret = fprintf(
			ofp,
			"%.255s %x %lx %x %08x %.63s %.127s\n",
			(hp2->device == NULL || hp2->device[0] == '\0') ?
				"-" : hp2->device,
			hp2->discno,
			hp2->time,
			hp2->type,
			(int) hp2->discid,
			(hp2->category == NULL || hp2->category[0] == '\0') ?
				"-" : hp2->category,
			(hp2->dtitle == NULL || hp2->dtitle[0] == '\0') ?
				"-" : hp2->dtitle
		);
		if (ret < 0) {
			ret = errno;
			(void) fclose(ifp);
			(void) fclose(ofp);
			errno = ret;

			DBGPRN(errfp, "File write error: %s (errno=%d)\n",
				path, errno);
			MEM_FREE(histbuf);
			return WRITE_ERR;
		}
	}

	/* Close file */
	if (fclose(ifp) != 0) {
		DBGPRN(errfp, "File close error: %s (errno=%d)\n",
			bakpath, errno);
		MEM_FREE(histbuf);
		return CLOSE_ERR;
	}
	if (fclose(ofp) != 0) {
		DBGPRN(errfp, "File close error: %s (errno=%d)\n",
			path, errno);
		MEM_FREE(histbuf);
		return CLOSE_ERR;
	}

	MEM_FREE(histbuf);
	return 0;
}


/***********************
 *   public routines   *
 ***********************/


/* cddb_hist_init
 *	Initialize history mechanism.
 *
 * Args:
 *	None.
 *
 * Return:
 *	Nothing.
 */
void
cddb_hist_init(void)
{
	int		i;
	FILE		*fp;
	char		*histbuf,
			*homep,
			device[FILE_PATH_SZ],
			category[FILE_BASE_SZ],
			dtitle[DLIST_BUF_SZ],
			path[FILE_PATH_SZ + 32];
	cddb_dlist_t	h,
			*hp,
			*nhp;
#ifndef __VMS
	pid_t		cpid;
	waitret_t	stat_val;
	int		ret,
			pfd[2];
	FILE		*wfp;

	if (app_data.histfile_dsbl)
		return;

	if (PIPE(pfd) < 0) {
		DBGPRN(errfp, "cddb_hist_init: pipe failed (errno=%d)\n",
			errno);
		return;
	}

	homep = util_homedir(util_get_ouid());
	if ((int) strlen(homep) >= FILE_PATH_SZ)
		/* Path name too long */
		return;

	(void) sprintf(path, USR_HIST_PATH, homep);

	switch (cpid = FORK()) {
	case 0:
		/* Child */
		ischild = TRUE;

		/* Close un-needed pipe descriptor */
		(void) close(pfd[0]);

		/* Force uid and gid to original setting */
		if (!util_set_ougid()) {
			(void) close(pfd[1]);
			exit(1);
		}

		/* Allocate temporary buffer */
		if ((histbuf = (char *) MEM_ALLOC("histbuf",
					      DLIST_BUF_SZ)) == NULL) {
			(void) close(pfd[1]);
			exit(2);
		}

		DBGPRN(errfp, "Loading history: %s\n", path);
		if ((fp = fopen(path, "r")) == NULL) {
			DBGPRN(errfp, "Cannot open %s\n", path);
			(void) close(pfd[1]);
			exit(0);
		}
		
		if ((wfp = fdopen(pfd[1], "w")) == NULL) {
			DBGPRN(errfp,
			    "cddb_hist_init: write pipe fdopen failed\n");
			(void) close(pfd[1]);
			exit(4);
		}

		while (fgets(histbuf, DLIST_BUF_SZ, fp) != NULL)
			(void) fprintf(wfp, "%s", histbuf);

		(void) fclose(fp);
		(void) fclose(wfp);

		exit(0);
		/*NOTREACHED*/

	case -1:
		DBGPRN(errfp, "cddb_hist_init: fork failed (errno=%d)\n",
			errno);
		(void) close(pfd[0]);
		(void) close(pfd[1]);
		return;

	default:
		/* Parent */

		/* Close un-needed pipe descriptor */
		(void) close(pfd[1]);

		if ((fp = fdopen(pfd[0], "r")) == NULL) {
			DBGPRN(errfp,
			    "cddb_hist_init: read pipe fdopen failed\n");
			return;
		}
		break;
	}
#else
	homep = util_homedir(util_get_ouid());
	if (strlen(homep) >= FILE_PATH_SZ)
		/* Path name too long */
		return;

	(void) sprintf(path, USR_HIST_PATH, homep);
	DBGPRN(errfp, "Loading history: %s\n", path);
	if ((fp = fopen(path, "r")) == NULL) {
		DBGPRN(errfp, "Cannot open %s\n", path);
		return;
	}
#endif	/* __VMS */

	/* Allocate temporary buffer */
	if ((histbuf = (char *) MEM_ALLOC("histbuf", DLIST_BUF_SZ)) == NULL) {
		CDDB_FATAL(app_data.str_nomemory);
		return;
	}

	/* Free old list, if any */
	for (hp = nhp = cddb_hist_head; hp != NULL; hp = nhp) {
		if (hp->device != NULL)
			MEM_FREE(hp->device);
		if (hp->category != NULL)
			MEM_FREE(hp->category);
		if (hp->dtitle != NULL)
			MEM_FREE(hp->dtitle);
		nhp = hp->next;
		MEM_FREE(hp);
	}
	cddb_hist_head = NULL;
	cddb_hist_mcnt = cddb_hist_fcnt = 0;

	h.device = device;
	h.category = category;
	h.dtitle = dtitle;

	/* Read in history file */
	for (i = 1; fgets(histbuf, DLIST_BUF_SZ, fp) != NULL; i++) {
		/* Skip comments and blank lines */
		if (histbuf[0] == '#' || histbuf[0] == '!' ||
		    histbuf[0] == '\n')
			continue;

		if (sscanf(histbuf, "%255s %x %lx %x %x %63s %127[^\n]\n",
			   h.device,
			   &h.discno,
			   &h.time,
			   &h.type,
			   &h.discid,
			   h.category,
			   h.dtitle) != 7) {
			DBGPRN(errfp, "History file syntax error: line %d\n",
				i);
			continue;
		}
		h.device[FILE_PATH_SZ - 1] = '\0';
		h.category[FILE_BASE_SZ - 1] = '\0';
		h.dtitle[DLIST_BUF_SZ - 1] = '\0';

		cddb_hist_fcnt++;

		if (++cddb_hist_mcnt > app_data.cddb_maxhist) {
			if (cddb_hist_tail == NULL)
				break;	/* Empty list: Cannot continue */

			/* Max entries exceeded: circulate around and
			 * re-use an existing structure.
			 */
			hp = cddb_hist_tail;
			cddb_hist_tail = hp->prev;
			cddb_hist_tail->next = NULL;

			if (hp->device != NULL)
				MEM_FREE(hp->device);
			if (hp->category != NULL)
				MEM_FREE(hp->category);
			if (hp->dtitle != NULL)
				MEM_FREE(hp->dtitle);

			if (cddb_hist_mcnt > 0)
				cddb_hist_mcnt--;
		}
		else {
			/* Allocate new entry */
			hp = (cddb_dlist_t *) MEM_ALLOC("cddb_hist",
							sizeof(cddb_dlist_t));
			if (hp == NULL) {
				CDDB_FATAL(app_data.str_nomemory);
				return;
			}
		}

		*hp = h;	/* Structure copy */

		if (strcmp(h.device, "-") != 0) {
			hp->device = (char *) MEM_ALLOC(
				"hp->device", strlen(h.device) + 1
			);
			if (hp->device == NULL) {
				CDDB_FATAL(app_data.str_nomemory);
				return;
			}
			(void) strcpy(hp->device, h.device);
		}
		else
			hp->device = NULL;

		if (strcmp(h.category, "-") != 0) {
			hp->category = (char *) MEM_ALLOC(
				"hp->category", strlen(h.category) + 1
			);
			if (hp->category == NULL) {
				CDDB_FATAL(app_data.str_nomemory);
				return;
			}
			(void) strcpy(hp->category, h.category);
		}
		else
			hp->category = NULL;

		if (strcmp(h.dtitle, "-") != 0) {
			hp->dtitle = (char *) MEM_ALLOC(
				"hp->dtitle", strlen(h.dtitle) + 1
			);
			if (hp->dtitle == NULL) {
				CDDB_FATAL(app_data.str_nomemory);
				return;
			}
			(void) strcpy(hp->dtitle, h.dtitle);
		}
		else
			hp->dtitle = NULL;

		hp->next = cddb_hist_head;
		hp->prev = NULL;
		if (cddb_hist_head != NULL)
			cddb_hist_head->prev = hp;
		else
			cddb_hist_tail = hp;
		cddb_hist_head = hp;
	}

	(void) fclose(fp);
	MEM_FREE(histbuf);

#ifndef __VMS
	while ((ret = WAITPID(cpid, &stat_val, 0)) != cpid) {
		if (ret < 0)
			break;
	}
	if (ret == cpid) {
	    if (WIFEXITED(stat_val)) {
		if (WEXITSTATUS(stat_val) != 0) {
			DBGPRN(errfp,
				"cddb_hist_init: child exited (status=%d)\n",
				WEXITSTATUS(stat_val));
		}
	    }
	    else if (WIFSIGNALED(stat_val)) {
		DBGPRN(errfp, "cddb_hist_init: child killed.\n");
	    }
	}
#endif
}


/*
 * cddb_hist_addent
 *	Add one entry to the in-core history list and to the user's
 *	history file.
 *
 * Args:
 *	hp - Pointer to the filled cddb_dlist_t structure containing
 *	     information to save.
 *	updfile - Whether to update the history file
 *
 * Return:
 *	return code as defined by cddb_ret_t
 */
cddb_ret_t
cddb_hist_addent(cddb_dlist_t *hp, bool_t updfile)
{
	int		ret,
			skip;
	cddb_dlist_t	*hp2;
	FILE		*fp;
	char		*homep,
			path[FILE_PATH_SZ + 32];
	bool_t		newfile,
			wholefile;
	struct stat	stbuf;
#ifndef __VMS
	cddb_ret_t	retcode;
	pid_t		cpid;
	waitret_t	stat_val;
#endif

	if (cddb_hist_head != NULL && cddb_hist_head->discid == hp->discid) {
		/* Same disc ID as the most recent item on history list:
		 * no need to add.
		 */
		return 0;
	}

	/*
	 * Add to in-core list
	 */

	if (++cddb_hist_mcnt > app_data.cddb_maxhist) {
		if (cddb_hist_tail == NULL)
			return 0;	/* Empty list: Cannot continue */

		/* Max entries exceeded: circulate around and
		 * re-use an existing structure.
		 */
		hp2 = cddb_hist_tail;
		cddb_hist_tail = hp2->prev;
		cddb_hist_tail->next = NULL;

		if (hp2->device != NULL)
			MEM_FREE(hp2->device);
		if (hp2->category != NULL)
			MEM_FREE(hp2->category);
		if (hp2->dtitle != NULL)
			MEM_FREE(hp2->dtitle);

		if (cddb_hist_mcnt > 0)
			cddb_hist_mcnt--;
	}
	else {
		/* Allocate new entry */
		hp2 = (cddb_dlist_t *) MEM_ALLOC(
			"cddb_hist",
			sizeof(cddb_dlist_t)
		);
		if (hp2 == NULL) {
			CDDB_FATAL(app_data.str_nomemory);
			return 0;
		}
	}

	*hp2 = *hp;	/* Structure copy */

	if (hp->device != NULL && hp->device[0] != '\0') {
		hp2->device = (char *) MEM_ALLOC(
			"hp2->device", strlen(hp->device) + 1
		);
		if (hp2->device == NULL) {
			CDDB_FATAL(app_data.str_nomemory);
			return 0;
		}
		(void) strcpy(hp2->device, hp->device);
	}
	else
		hp2->device = NULL;

	if (hp->category != NULL && hp->category[0] != '\0') {
		hp2->category = (char *) MEM_ALLOC(
			"hp2->category", strlen(hp->category) + 1
		);
		if (hp2->category == NULL) {
			CDDB_FATAL(app_data.str_nomemory);
			return 0;
		}
		(void) strcpy(hp2->category, hp->category);
	}
	else
		hp2->category = NULL;

	if (hp->dtitle != NULL && hp->dtitle[0] != '\0') {
		hp2->dtitle = (char *) MEM_ALLOC(
			"hp2->dtitle", strlen(hp->dtitle) + 1
		);
		if (hp2->dtitle == NULL) {
			CDDB_FATAL(app_data.str_nomemory);
			return 0;
		}
		(void) strcpy(hp2->dtitle, hp->dtitle);
	}
	else
		hp2->dtitle = NULL;

	hp2->next = cddb_hist_head;
	hp2->prev = NULL;
	if (cddb_hist_head != NULL)
		cddb_hist_head->prev = hp2;
	else
		cddb_hist_tail = hp2;
	cddb_hist_head = hp2;

	if (app_data.histfile_dsbl || !updfile)
		return 0;

	cddb_hist_fcnt++;

	skip = 0;
	if ((cddb_hist_fcnt - cddb_hist_mcnt) > HIST_HYST) {
		/* Write out the whole file */
		skip = cddb_hist_fcnt - cddb_hist_mcnt + 1;
		cddb_hist_fcnt = cddb_hist_mcnt;
		wholefile = TRUE;
	}
	else
		wholefile = FALSE;

	/*
	 * Write out one entry to file
	 */

#ifndef __VMS
	/* Fork child to perform actual I/O */
	switch (cpid = FORK()) {
	case 0:
		/* Child process */
		ischild = TRUE;
		break;

	case -1:
		return CDDB_SET_CODE(FORK_ERR, errno);

	default:
		/* Parent process: wait for child to exit */
		while ((ret = WAITPID(cpid, &stat_val, 0)) != cpid) {
			if (ret < 0)
				return 0;

			/* Do some work */
			if (cddb_clinfo->workproc != NULL)
				cddb_clinfo->workproc(cddb_clinfo->arg);
		}

		retcode = 0;
		if (WIFEXITED(stat_val))
			retcode = CDDB_SET_CODE(WEXITSTATUS(stat_val), 0);
		else if (WIFSIGNALED(stat_val))
			retcode = CDDB_SET_CODE(KILLED_ERR, WTERMSIG(stat_val));

		return (retcode);
	}

	/* Force uid and gid to original setting */
	if (!util_set_ougid())
		exit(SETUID_ERR);
#endif	/* __VMS */

	if (wholefile) {
		/* Write out the whole file */
		ret = cddb_hist_writefile(NULL, hp, skip);
		CH_RET(ret);
	}

	homep = util_homedir(util_get_ouid());
	if ((int) strlen(homep) >= FILE_PATH_SZ)
		/* Path name too long */
		CH_RET(OPEN_ERR);

	(void) sprintf(path, USR_HIST_PATH, homep);

	DBGPRN(errfp, "Writing history file %s\n", path);

	/* Check the path */
	newfile = FALSE;
	if (stat(path, &stbuf) < 0) {
		if (errno == ENOENT)
			newfile = TRUE;
		else {
			DBGPRN(errfp, "Cannot stat %s (errno=%d)\n",
				path, errno);
			CH_RET(OPEN_ERR);
		}
	}
	else {
		if (!S_ISREG(stbuf.st_mode)) {
			DBGPRN(errfp,
				"Not a regular file error: %s\n", path);
			CH_RET(OPEN_ERR);
		}

		if (stbuf.st_size == 0)
			newfile = TRUE;
	}

	/* Get a history file handle */
	if ((fp = fopen(path, "a")) == NULL) {
		DBGPRN(errfp, "Cannot open file: %s\n", path);
		CH_RET(OPEN_ERR);
	}

	/* Set file permissions */
	cddb_setperm(path, app_data.hist_filemode);

	if (newfile)
		/* Write history file header */
		(void) fprintf(fp, HIST_BANNER, VERSION, COPYRIGHT);

	/* Write to history file */
	ret = fprintf(
		fp,
		"%.255s %x %lx %x %08x %.63s %.127s\n",
		(hp->device == NULL || hp->device[0] == '\0') ?
			"-" : hp->device,
		hp->discno,
		hp->time,
		hp->type,
		(int) hp->discid,
		(hp->category == NULL || hp->category[0] == '\0') ?
			"-" : hp->category,
		(hp->dtitle == NULL || hp->dtitle[0] == '\0') ?
			"-" : hp->dtitle
	);

	if (ret < 0) {
		ret = errno;
		(void) fclose(fp);
		errno = ret;

		DBGPRN(errfp, "File write error: %s (errno=%d)\n",
			path, errno);
		CH_RET(WRITE_ERR);
	}

	/* Close file */
	if (fclose(fp) != 0) {
		DBGPRN(errfp, "File close error: %s (errno=%d)\n",
			path, errno);
		CH_RET(CLOSE_ERR);
	}

	/* Child exits here */
	CH_RET(0);
	/*NOTREACHED*/
}


/*
 * cddb_hist_delent
 *	Delete one entry from the in-core history list and from the user's
 *	history file.
 *
 * Args:
 *	hp - Pointer to the filled cddb_dlist_t structure containing
 *	     information to delete.
 *	updfile - Whether to update the history file.
 *
 * Return:
 *	return code as defined by cddb_ret_t
 */
cddb_ret_t
cddb_hist_delent(cddb_dlist_t *hp, bool_t updfile)
{
	int		ret,
			skip;
	cddb_dlist_t	*hp2,
			h;
#ifndef __VMS
	cddb_ret_t	retcode;
	pid_t		cpid;
	waitret_t	stat_val;
#endif

	h = *hp;	/* Structure copy */

	/*
	 * Delete from in-core list
	 */
	for (hp2 = cddb_hist_head; hp2 != NULL; hp2 = hp2->next) {
		if (hp2->time == hp->time && hp2->discid == hp->discid) {
			if (hp2 == cddb_hist_head)
				cddb_hist_head = hp2->next;
			else
				hp2->prev->next = hp2->next;

			if (hp2->next != NULL)
				hp2->next->prev = hp2->prev;

			if (hp2->device != NULL)
				MEM_FREE(hp2->device);
			if (hp2->category != NULL)
				MEM_FREE(hp2->category);
			if (hp2->dtitle != NULL)
				MEM_FREE(hp2->dtitle);
			MEM_FREE(hp2);

			if (cddb_hist_mcnt > 0)
				cddb_hist_mcnt--;
			break;
		}
	}

	if (app_data.histfile_dsbl || !updfile)
		return 0;

	/*
	 * Delete from history file
	 */

	skip = cddb_hist_fcnt - cddb_hist_mcnt;
	cddb_hist_fcnt = cddb_hist_mcnt;

#ifndef __VMS
	/* Fork child to perform actual I/O */
	switch (cpid = FORK()) {
	case 0:
		/* Child process */
		ischild = TRUE;
		break;

	case -1:
		return CDDB_SET_CODE(FORK_ERR, errno);

	default:
		/* Parent process: wait for child to exit */
		while ((ret = WAITPID(cpid, &stat_val, 0)) != cpid) {
			if (ret < 0)
				return 0;

			/* Do some work */
			if (cddb_clinfo->workproc != NULL)
				cddb_clinfo->workproc(cddb_clinfo->arg);
		}

		retcode = 0;
		if (WIFEXITED(stat_val))
			retcode = CDDB_SET_CODE(WEXITSTATUS(stat_val), 0);
		else if (WIFSIGNALED(stat_val))
			retcode = CDDB_SET_CODE(KILLED_ERR, WTERMSIG(stat_val));

		return (retcode);
	}

	/* Force uid and gid to original setting */
	if (!util_set_ougid())
		exit(SETUID_ERR);
#endif	/* __VMS */

	/* Write the history file */
	ret = cddb_hist_writefile(&h, NULL, skip);

	/* Child exits here */
	CH_RET(ret);
	/*NOTREACHED*/
}


/*
 * cddb_hist_delall
 *	Delete the user's entire xmcd history from the in-core history list
 *	and the user's history file.
 *
 * Args:
 *	updfile - Whether to update the history file.
 *
 * Return:
 *	return code as defined by cddb_ret_t
 */
cddb_ret_t
cddb_hist_delall(bool_t updfile)
{
	cddb_dlist_t	*hp,
			*nhp;
	char		*homep,
			path[FILE_PATH_SZ + 32],
			bakpath[FILE_PATH_SZ + 32];
#ifndef __VMS
	int		ret;
	cddb_ret_t	retcode;
	pid_t		cpid;
	waitret_t	stat_val;
#endif

	/* Delete in-core list */
	for (hp = nhp = cddb_hist_head; hp != NULL; hp = nhp) {
		nhp = hp->next;
		if (hp->device != NULL)
			MEM_FREE(hp->device);
		if (hp->category != NULL)
			MEM_FREE(hp->category);
		if (hp->dtitle != NULL)
			MEM_FREE(hp->dtitle);
		MEM_FREE(hp);
	}
	cddb_hist_head = NULL;
	cddb_hist_mcnt = cddb_hist_fcnt = 0;

	if (app_data.histfile_dsbl || !updfile)
		return 0;

	/* Clear history file */
	homep = util_homedir(util_get_ouid());
	if ((int) strlen(homep) >= FILE_PATH_SZ)
		/* Path name too long */
		return CDDB_SET_CODE(OPEN_ERR, ENAMETOOLONG);

	(void) sprintf(path, USR_HIST_PATH, homep);
	(void) sprintf(bakpath, "%s.bak", path);

#ifndef __VMS
	/* Fork child to perform actual I/O */
	switch (cpid = FORK()) {
	case 0:
		/* Child process */
		ischild = TRUE;
		break;

	case -1:
		return CDDB_SET_CODE(FORK_ERR, errno);

	default:
		/* Parent process: wait for child to exit */
		while ((ret = WAITPID(cpid, &stat_val, 0)) != cpid) {
			if (ret < 0)
				return 0;

			/* Do some work */
			if (cddb_clinfo->workproc != NULL)
				cddb_clinfo->workproc(cddb_clinfo->arg);
		}

		retcode = 0;
		if (WIFEXITED(stat_val))
			retcode = CDDB_SET_CODE(WEXITSTATUS(stat_val), 0);
		else if (WIFSIGNALED(stat_val))
			retcode = CDDB_SET_CODE(KILLED_ERR, WTERMSIG(stat_val));

		return (retcode);
	}

	/* Force uid and gid to original setting */
	if (!util_set_ougid())
		exit(SETUID_ERR);
#endif	/* __VMS */

	DBGPRN(errfp, "Deleting history file %s\n", path);

	/* Rename old history file */
	(void) UNLINK(bakpath);

#if defined(USE_RENAME)

	if (rename(path, bakpath) < 0) {
		DBGPRN(errfp, "Cannot rename file %s -> %s (errno=%d)\n",
			path, bakpath, errno);
		CH_RET(LINK_ERR);
	}

#else	/* USE_RENAME */

	if (LINK(path, bakpath) < 0) {
		DBGPRN(errfp, "Cannot link file %s -> %s (errno=%d)\n",
			path, bakpath, errno);
		CH_RET(LINK_ERR);
	}

	(void) UNLINK(path);

#endif	/* USE_RENAME */

	/* Child exits here */
	CH_RET(0);
	/*NOTREACHED*/
}


/*
 * cddb_hist_list
 *	Return the head of the xmcd history list.
 *
 * Args:
 *	None.
 *
 * Return:
 *	A pointer to the head of the xmcd history list, or NULL if the list
 *	is empty.
 */
cddb_dlist_t *
cddb_hist_list(void)
{
	return (cddb_hist_head);
}


/*
 * cddb_chgr_init
 */
void
cddb_chgr_init(void)
{
	int		i,
			lastdisc,
			allocsz;

	/* Allocate array of cddb_dlist_t pointers */
	if ((allocsz = sizeof(cddb_dlist_t) * app_data.numdiscs) <= 0)
		return;

	cddb_chgr_array = (cddb_dlist_t *) MEM_ALLOC(
		"cddb_chgr_array", allocsz
	);
	if (cddb_chgr_array == NULL) {
		CDDB_FATAL(app_data.str_nomemory);
		return;
	}

	(void) memset((byte_t *) cddb_chgr_array, 0, allocsz);

	lastdisc = app_data.numdiscs - 1;
	for (i = 0; i < app_data.numdiscs; i++) {
		cddb_chgr_array[i].discno = i+1;

		if (i == lastdisc)
			cddb_chgr_array[i].next = NULL;
		else
			cddb_chgr_array[i].next = &cddb_chgr_array[i+1];

		if (i == 0)
			cddb_chgr_array[i].prev = NULL;
		else
			cddb_chgr_array[i].prev = &cddb_chgr_array[i-1];
	}
}


/*
 * cddb_chgr_addent
 *	Add one entry to the CD changer list
 *
 * Args:
 *	hp - Pointer to the filled cddb_dlist_t structure containing
 *	     information to add.
 *
 * Return:
 *	return code as defined by cddb_ret_t
 */
cddb_ret_t
cddb_chgr_addent(cddb_dlist_t *cp)
{
	int	i;

	if (cddb_chgr_array == NULL)
		/* Not yet initialized */
		return 0;

	if (cp->discno <= 0 || cp->discno > app_data.numdiscs)
		return CDDB_SET_CODE(CMD_ERR, EINVAL);

	i = cp->discno - 1;

	/* Don't do a structure copy here, or we'd mess up the list links */
	cddb_chgr_array[i].discno = cp->discno;
	cddb_chgr_array[i].type = cp->type;
	cddb_chgr_array[i].discid = cp->discid;
	cddb_chgr_array[i].time = cp->time;

	if (cddb_chgr_array[i].device != NULL) {
		MEM_FREE(cddb_chgr_array[i].device);
		cddb_chgr_array[i].device = NULL;
	}
	if (cddb_chgr_array[i].category != NULL) {
		MEM_FREE(cddb_chgr_array[i].category);
		cddb_chgr_array[i].category = NULL;
	}
	if (cddb_chgr_array[i].dtitle != NULL) {
		MEM_FREE(cddb_chgr_array[i].dtitle);
		cddb_chgr_array[i].dtitle = NULL;
	}

	if (cp->device != NULL && cp->device[0] != '\0') {
		cddb_chgr_array[i].device = (char *) MEM_ALLOC(
			"cp->device", strlen(cp->device) + 1
		);
		if (cddb_chgr_array[i].device == NULL) {
			CDDB_FATAL(app_data.str_nomemory);
			return 0;
		}
		(void) strcpy(cddb_chgr_array[i].device, cp->device);
	}

	if (cp->category != NULL && cp->category[0] != '\0') {
		cddb_chgr_array[i].category = (char *) MEM_ALLOC(
			"cp->category", strlen(cp->category) + 1
		);
		if (cddb_chgr_array[i].category == NULL) {
			CDDB_FATAL(app_data.str_nomemory);
			return 0;
		}
		(void) strcpy(cddb_chgr_array[i].category, cp->category);
	}

	if (cp->dtitle != NULL && cp->dtitle[0] != '\0') {
		cddb_chgr_array[i].dtitle = (char *) MEM_ALLOC(
			"cp->dtitle", strlen(cp->dtitle) + 1
		);
		if (cddb_chgr_array[i].dtitle == NULL) {
			CDDB_FATAL(app_data.str_nomemory);
			return 0;
		}
		(void) strcpy(cddb_chgr_array[i].dtitle, cp->dtitle);
	}

	return 0;
}


/*
 * cddb_chgr_list
 *	Return the head of the CD changer list.
 *
 * Args:
 *	None.
 *
 * Return:
 *	A pointer to the head of the CD changer list, or NULL if the list
 *	is empty.
 */
cddb_dlist_t *
cddb_chgr_list(void)
{
	return (&cddb_chgr_array[0]);
}


