/* ---------------------------------------------------------------------
   cddbd interface for XfreeCD

   Copyright 1998 by Brian C. Lane
   nexus@tatoosh.com
   http://www.tatoosh.com/nexus

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

   =============================[ HISTORY ]=============================
   07/05/98	Removed the strcat of tmpstr from the track offset loop
   		when requesting CD info. This was causing the odd
   		behavior, depending on the state of memory.

   06/19/98     Changed offset in the track loop to be 0 reference
                instead of 1.

   06/16/98     NIN Broken is living up to its name. It is causing the
                server to return an error 500. Hmm, I somehow left out
		the length of the cd. This is fixed. It now downloads
		NIN correctly (woo hoo, ack, pthhhtttt). Anyone want
		a copy of NIN Broken? I'll mail it to you in little
		pieces...

   06/02/98     Sending the full line returned by the server to the
                caling process in cdinfo.line

   05/24/98     Adding support for inexact matches.
                Adding a return type of CDDBD_INEX_LINE

   05/11/98     Adding more information from main process. Passes a
                cdinfo structure.

   05/10/98     Added basic socket open/close/read/write functions.

   05/09/98     Started this part of the XfreeCD code. 


   ---------------------------------------------------------------------
   This implements the internet connection to a cddbd server. Local data
   is accessed using the functions in ... 

   Hmm, how to implemetn all this?
   State machine for error responses and link to next command?

   I also need to get more info on the CD than just the CD's discid

   --------------------------------------------------------------------- */
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <gtk/gtk.h>
#include "xfreecd.h"
#include "child_sync.h"
#include "cd_control.h"
#include "cddbd.h"
#include "cddb.h"


#undef DEBUG1
#undef DEBUG3

int cddbd( int );

/*
   Fire up the cddb process.
   Create a socket pair.
   Fork a child process
   Return the parent's file descriptor for the socket or -1 + errno
   fill in childpid with the child's PID
   Child calls cddbd 
*/
int start_cddbd( int *childpid )
{
  int  fd[2];

  if( socketpair( AF_UNIX, SOCK_STREAM, 0, fd ) < 0 )
    return(-1);

  if( ( *childpid = fork() ) < 0 )
  {
    perror("start_cddbd, cannot fork");
    return(-1);
  } else if( *childpid == 0 ) {
    close( fd[0] );

    /* Synchronize with the parent, exit if it fails */
    if( parent_sync( fd[1] ) == 0 )
      cddbd( fd[1] );
    
    close( fd[1] );
    exit(0);
  } 
  close( fd[1] );

  if( child_sync( fd[0] ) == 0 )
    return(fd[0]);  

  return(-1);
}

/* -----------------------------------------------------------------------
   Read a line from the socket
   ----------------------------------------------------------------------- */
int readn( register int fd, register char *ptr, register int nbytes )
{
  int   nleft,
        nread;

  nleft = nbytes;
  while( nleft > 0 )
    {
      nread = read( fd, ptr, nleft );
      if( nread < 0 )
	return( nread );
      else if( nread == 0 )
	break;

      nleft -= nread;
      ptr += nread;
    }
  return (nbytes - nleft);
}


/* -----------------------------------------------------------------------
   Write n bytes of data to the socket

   Loop until all the data is written. 
   ----------------------------------------------------------------------- */
int writen( register int fd, register char *ptr, register int nbytes )
{
  int   nleft,
        nwritten;

  nleft = nbytes;
  while( nleft > 0 )
    {
      nwritten = write( fd, ptr, nleft );
      if( nwritten <= 0 )
	return( nwritten );

      nleft -= nwritten;
      ptr += nwritten;
    }

  return (nbytes - nleft );
}


/* -----------------------------------------------------------------------
   A VERY inefficent read routine. I need to rewrite this to read as much
   as possible and scan the buffer read...
   
   But then how do you put back characters after the CR that you want to
   read the next time it is called? Static holding previous spares maybe?
   ----------------------------------------------------------------------- */
int readline( register int fd, register char *ptr, register int maxlen )
{
  int  n,
       rc;
  char c;

  for( n = 1; n < maxlen; n++ )
    {
      if( ( rc = read( fd, &c, 1 ) ) == 1 )
	{
	  *ptr++ = c;
	  if( c == '\n' )
	    break;
	} else if( rc == 0 ) {
	  if( n == 1 )
	    return(0);
	  else
	    break;
	} else {
	  return(-1);
	}
    }

  *ptr = 0;
  return(n);
}


/* -----------------------------------------------------------------------
   Open a cddbd connection, return the fd

   This handles the connection and initial login to the server,
   returns error codes on failure.

   Pass the server name and port
   ----------------------------------------------------------------------- */
int open_cddbd( char *server, int port )
{
  struct sockaddr_in   tcp_srv_addr;    /* Server's Internet socket addr. */
  struct hostent       tcp_host_info;   /* from gethostbyname */
  struct hostent       *hp;
  unsigned long        inaddr;
  int                  fd;


  bzero( &tcp_srv_addr, sizeof( tcp_srv_addr ) );
  tcp_srv_addr.sin_family = AF_INET;


  /* Setup the port, exit if illegal passed */
  if( port < 1 )
    return(-2);

  tcp_srv_addr.sin_port = htons( port );

  /* Try the hostname as a dotted decimal first */
  if( ( inaddr = inet_addr( server ) ) != INADDR_NONE )
    {
      bcopy((char *)&inaddr,(char *)&tcp_srv_addr.sin_addr,sizeof(inaddr));
      tcp_host_info.h_name = NULL;
    } else {
      /* Not dotted decimal, try it as a name */
      if( ( hp = gethostbyname( server ) ) == NULL )
	{
	  return(-3);
	}
      tcp_host_info = *hp;    /* Copy the structure */
      bcopy( hp->h_addr, (char *) &tcp_srv_addr.sin_addr, hp->h_length );
    }

  /* Create the socket */
  if( ( fd = socket( AF_INET, SOCK_STREAM, 0 ) ) < 0 )
    {
      return(-4);
    }

  /* Connect to the server */
  if( connect(fd,(struct sockaddr *)&tcp_srv_addr,sizeof(tcp_srv_addr)) < 0 )
    {
      close( fd );
      return(-5);
    }

  return(fd);
}

/* -----------------------------------------------------------------------
   Close down the connection to the server.

   Send quit and close socket
   ----------------------------------------------------------------------- */
int close_cddbd( fd )
{
  char  line[255];

  writen( fd, "quit\n", strlen( "quit\n" ) );

  /* Read the closing banner from the server */
  if( readline( fd, line, 255 ) < 0 )
    {
      close( fd );
      return(-2);
    }

#ifdef DEBUG1
  g_print("%s", line );
#endif

  close( fd );
  return(0);
}


/* -----------------------------------------------------------------------
   Return the response code
   ----------------------------------------------------------------------- */
int cddbd_code( char *line )
{
  char *p;

  p = strtok( line, " \n" );
  return( atoi( p ) );
}



/* -----------------------------------------------------------------------
   Watch for commands from the main process and execute connections to
   the indicated cddbd internet database server.

   Commands ... DB_*
   Uses the cddbd_cmnd structure to pass the command, server and port.

   Also need to read the sites listing, and motd.

   Retrieve data for ID from server SERVER using PORT

   returns a CDINFO structure with a status byte? How to keep it updated
   without returning a full CDINFO every time?   

   As we progress thru the protocol, send back status updates to the parent
   process.

   ----------------------------------------------------------------------- */
int cddbd( int control_fd )
{
  struct CDINFO  cdinfo;
  FILE           *fp;
  int            cddbd_fd;
  char           line[255],
                 *tmpfile,
                 *p;
  int            x,
                 code;


  for(;;)
  {
    /* Get the command from the parent process */
    if( read( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
      {
	perror("cddbd read error");
	return(-1);
      }

    switch( cdinfo.cddbd_cmnd )
      {
      case DB_DIAG :
	puts("cddbd diagnostic\n");
	break;

	/* Get the info on a CD from the server */
      case DB_READ :

#ifdef DEBUG1
	g_print("Searching for 0x%08x at %s : %d\n", cdinfo.discid, cdinfo.server, cdinfo.port );
#endif

	if( (cddbd_fd = open_cddbd( cdinfo.server, cdinfo.port )) < 0 )
	  {
	    close_cddbd( cddbd_fd );

	    /* Tell the parent the connection failed */
	    cdinfo.cddbd_stat = CDDBD_OPEN_ERR;
	    sprintf( cdinfo.line, "err=%d", cddbd_fd );
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-1");
		break;
	      }
	    break;
	  }

	/* Tell the parent we are connected ok */
	cdinfo.cddbd_stat = CDDBD_OPEN_OK;
	if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-2");
		break;
	      }
	  	
	/* Read the opening banner from the server */
	if( readline( cddbd_fd, line, 255 ) < 0 )
	  {
	    close( cddbd_fd );
	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = CDDBD_READ_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-3");
		break;
	      }
	    break;
	  }  

#ifdef DEBUG1
	g_print( "%s", line );
#endif

	strncpy( cdinfo.line, line, 254 );
	code = cddbd_code( line );
	if( code > 201 )
	  {
	    close_cddbd( cddbd_fd );
	    /* Tell the parent the connect failed */
	    cdinfo.cddbd_stat = code;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-4");
		break;
	      }
	    break;
	  }

	/* Identify ourself to the server */
	sprintf( line, "cddb hello %s %s XfreeCD %s\n", getenv("USER"),
		                                        getenv("HOSTNAME"),
		                                        VERSION );

#ifdef DEBUG1
	g_print("%s", line );
#endif

	if( writen( cddbd_fd, line, strlen( line ) ) < 0 )
	  {
	    close( cddbd_fd );

	    /* Tell the parent the write failed */
	    cdinfo.cddbd_stat = CDDBD_WRITE_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-5");
		break;
	      }
	    break;
	  }

	/* Get the response from the server */
	if( readline( cddbd_fd, line, 255 ) < 0 )
	  {
	    close( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = CDDBD_READ_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-6");
		break;
	      }
	    break;
	  }  

#ifdef DEBUG1
	g_print( "%s", line );
#endif

	strncpy( cdinfo.line, line, 254 );
	code = cddbd_code( line );
	if( code > 200 )
	  {
	    close_cddbd( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = code;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-7");
		break;
	      }
	    break;
	  }

	/* Send a Query to the server */
	/* This can be quite! long (try NIN broken, 99 tracks) so break it up
	   into smaller chuncks
	*/
	sprintf( line, "cddb query %08lx %d ", cdinfo.discid, cdinfo.tochdr.cdth_trk1 );

#ifdef DEBUG1
	g_print("%s", line );
#endif

	/* Write the first part of the query to the server */
	if( writen( cddbd_fd, line, strlen( line ) ) < 0 )
	  {
	    close( cddbd_fd );

	    /* Tell the parent the write failed */
	    cdinfo.cddbd_stat = CDDBD_WRITE_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-8");
		break;
	      }
	    break;
	  }

	/* List of offsets, send it one offset at a time */
	for( x = 0; x < cdinfo.tochdr.cdth_trk1; x++ )
	  {
	    sprintf( line, "%ld ", cdinfo.track[x].frame_offset );

#ifdef DEBUG1
	    g_print("[%d]%s", x, line );
#endif

	    /* Write this next chunck of the request */
	    if( writen( cddbd_fd, line, strlen( line ) ) < 0 )
	      {
		close( cddbd_fd );

		/* Tell the parent the write failed */
		cdinfo.cddbd_stat = CDDBD_WRITE_ERR;
		if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
		  {
		    perror("cddbd write error-8");
		    break;
		  }
		break;
	      }
	  }

	/* End of the query with length in seconds */
	sprintf(line, "%d\n", cdinfo.cd_length );

#ifdef DEBUG1
	g_print("%s", line );
#endif

	if( writen( cddbd_fd, line, strlen( line ) ) < 0 )
	  {
	    close( cddbd_fd );

	    /* Tell the parent the write failed */
	    cdinfo.cddbd_stat = CDDBD_WRITE_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-8");
		break;
	      }
	    break;
	  }

        /* Get the response from the server, get the coded response,
	   and the rest up to the '.'
	*/
	/* Get the response from the server */
	if( readline( cddbd_fd, line, 255 ) < 0 )
	  {
	    close( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = CDDBD_READ_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-9");
		break;
	      }
	    break;
	  }  

#ifdef DEBUG1
	g_print( "%s", line );
#endif

	strncpy( cdinfo.line, line, 254 );
	code = cddbd_code( line );
	if( (code > 211) || (code==202) )
	  {
	    close_cddbd( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = code;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-10");
		break;
	      }
	    break;
	  }

	/* Inexact match, get the list */
	if( code == 211 )
	  {
	    /* Return the inexact match lines to the sparent process */
	    while( line[0] != '.' )
	      {
		if( readline( cddbd_fd, line, 255 ) < 0 )
		  {
		    close( cddbd_fd );

		    /* Tell the parent the read failed */
		    cdinfo.cddbd_stat = CDDBD_READ_ERR;
		    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
		      {
			perror("cddbd write error-11");
			break;
		      }
		    break;
		  }  

#ifdef DEBUG1
		g_print( "%s", line );
#endif
		/* Send the line to the parent */
		cdinfo.cddbd_stat = CDDBD_INEX_LINE;
		strncpy( cdinfo.line, line, 254 );
		if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
		  {
		    perror("cddbd write error-11a");
		    break;
		  } 
	      }

	    /* We are finished with the server */
	    close_cddbd( cddbd_fd );
	    break;
	  }

	/* Exact match, get the category ?? and discid ?? */
	if( code == 200 )
	  {
	    /* Tell the parent that we got a match */
	    cdinfo.cddbd_stat = CDDBD_MATCH_OK;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-11a");
		break;
	      }
	    
	    /* Get the category (code has already placed a null in line) */
	    p = strtok( NULL, " \n" );
	    if( cdinfo.category == NULL )
	      cdinfo.category = g_string_new( p );
	    else
	      cdinfo.category = g_string_assign( cdinfo.category, p );

#ifdef DEBUG3
	    g_print("category = %s\n", cdinfo.category->str );
#endif
	    /* We could also get the discid and title here... */
	  }


	/* Send a read command to the server */
	sprintf( line, "cddb read %s %08lx\n", cdinfo.category->str, cdinfo.discid );

#ifdef DEBUG1
		g_print( "%s", line );
#endif

	if( writen( cddbd_fd, line, strlen( line ) ) < 0 )
	  {
	    close( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = CDDBD_WRITE_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-12");
		break;
	      }
	    break;
	  }

	/* Get the response from the server */
	if( readline( cddbd_fd, line, 255 ) < 0 )
	  {
	    close( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = CDDBD_READ_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-13");
		break;
	      }
	    break;
	  }  

#ifdef DEBUG1
	g_print( "%s", line );
#endif

	strncpy( cdinfo.line, line, 254 );
	code = cddbd_code( line );
	if( code > 210 )
	  {
	    close_cddbd( cddbd_fd );
	    
	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = code;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-14");
		break;
	      }
	    break;
	  }

	/* 
	   Get the database file from the server.
	   Write the file into a temporary file to be read by cddb_read
	*/
	if( ( tmpfile = tmpnam( NULL ) ) == NULL )
	  {
	    close_cddbd( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = CDDBD_TMPF_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-15");
		break;
	      }
	    break;	    
	  }

	if( ( fp = fopen( tmpfile, "wb" ) ) == NULL )
	  {
	    close_cddbd( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = CDDBD_FOPEN_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-16");
		break;
	      }
	    break;
	  }

	/* Tell the parent we are reading the database entry */
	cdinfo.cddbd_stat = CDDBD_ENTRY_OK;
	if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	  {
	    perror("cddbd write error-14a");
	    break;
	  }

	while( line[0] != '.' )
	  {
	    /* Get the response from the server */
	    if( readline( cddbd_fd, line, 255 ) < 0 )
	      {
		close( cddbd_fd );

		/* Tell the parent the read failed */
		cdinfo.cddbd_stat = CDDBD_READ_ERR;
		if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
		  {
		    perror("cddbd write error-17");
		    break;
		  }
		break;
	      }  
      
#ifdef DEBUG1
	    g_print( "%s", line );
#endif
	    /* Strip trailing CR -- kludge, may not be there... */
	    line[strlen(line)-2] = 0;
	    fprintf(fp, "%s\n", line);
	  }

	fclose( fp );

	/* We are finished with the server */
	close_cddbd( cddbd_fd );


	/* Read the file into the cdinfo structure, and write it to its
	   final location in the local database.
	*/
	if( read_cddb_file( tmpfile, &cdinfo ) == 0 )
	  {
	    /* Make sure it is saved with our discid! */
	    cdinfo.discid = cddb_discid( &cdinfo );

	    if( write_cddb( &cdinfo, 1 ) == 0 )
	      cdinfo.cddbd_stat = CDDBD_DONE_OK;
	    else
	    cdinfo.cddbd_stat = CDDBD_DONE_ERR;
	  } else {
	    cdinfo.cddbd_stat = CDDBD_DONE_ERR;
	  }

#ifdef DEBUG3
	    g_print("category = %s\n", cdinfo.category->str );
#endif

	/* Send the new structure back to the parent process */
	if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	  {
	    perror("cddbd write error-18");
	    break;
	  }

	/* Delete the temporary file */
#ifdef DEBUG1
	g_print( "tmpfile=%s\n", tmpfile );
#else
	unlink( tmpfile );
#endif
	
	break;

      /* Get the MOTD from the current server */
      case DB_MOTD :
	if( (cddbd_fd = open_cddbd( cdinfo.server, cdinfo.port )) < 0 )
	  {
	    close_cddbd( cddbd_fd );

	    /* Tell the parent the connection failed */
	    cdinfo.cddbd_stat = CDDBD_OPEN_ERR;
	    sprintf( cdinfo.line, "err=%d", cddbd_fd );
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-1");
		break;
	      }
	    break;
	  }

	/* Read the opening banner from the server */
	if( readline( cddbd_fd, line, 255 ) < 0 )
	  {
	    close( cddbd_fd );
	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = CDDBD_READ_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-3");
		break;
	      }
	    break;
	  }  

#ifdef DEBUG1
	g_print( "%s", line );
#endif

	strncpy( cdinfo.line, line, 254 );
	code = cddbd_code( line );
	if( code > 201 )
	  {
	    close_cddbd( cddbd_fd );
	    /* Tell the parent the connect failed */
	    cdinfo.cddbd_stat = code;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-4");
		break;
	      }
	    break;
	  }

	/* Identify ourself to the server */
	sprintf( line, "cddb hello %s %s XfreeCD %s\n", getenv("USER"),
		                                        getenv("HOSTNAME"),
		                                        VERSION );

#ifdef DEBUG1
	g_print("%s", line );
#endif

	if( writen( cddbd_fd, line, strlen( line ) ) < 0 )
	  {
	    close( cddbd_fd );

	    /* Tell the parent the write failed */
	    cdinfo.cddbd_stat = CDDBD_WRITE_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-5");
		break;
	      }
	    break;
	  }

	/* Get the response from the server */
	if( readline( cddbd_fd, line, 255 ) < 0 )
	  {
	    close( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = CDDBD_READ_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-6");
		break;
	      }
	    break;
	  }  

#ifdef DEBUG1
	g_print( "%s", line );
#endif

	strncpy( cdinfo.line, line, 254 );
	code = cddbd_code( line );
	if( code > 200 )
	  {
	    close_cddbd( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = code;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-7");
		break;
	      }
	    break;
	  }

	/* Tell the server we want the MOTD */
	strcpy( line, "motd\n" );
	if( writen( cddbd_fd, line, strlen( line ) ) < 0 )
	  {
	    close( cddbd_fd );

	    /* Tell the parent the write failed */
	    cdinfo.cddbd_stat = CDDBD_WRITE_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-7");
		break;
	      }
	    break;
	  }

	/* Get the response from the server */
	if( readline( cddbd_fd, line, 255 ) < 0 )
	  {
	    close( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = CDDBD_READ_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-8");
		break;
	      }
	    break;
	  }

#ifdef DEBUG1
	g_print( "%s", line );
#endif

	strncpy( cdinfo.line, line, 254 );
	code = cddbd_code( line );
	if( code > 210 )
	  {
	    close_cddbd( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = code;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-7");
		break;
	      }
	    break;
	  }

	/* Tell the parent about the MOTD date and time */
	cdinfo.cddbd_stat = CDDBD_MOTD_LINE;
	strncpy( cdinfo.line, line, 254 );
	if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	  {
	    perror("cddbd write error-8");
	    break;
	  }

	/* Read the MOTD one line at a time */
	while( line[0] != '.' )
	  {
	    /* Get the response from the server */
	    if( readline( cddbd_fd, line, 255 ) < 0 )
	      {
		close( cddbd_fd );

		/* Tell the parent the read failed */
		cdinfo.cddbd_stat = CDDBD_READ_ERR;
		if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
		  {
		    perror("cddbd write error-8");
		    break;
		  }
		break;
	      }

#ifdef DEBUG10
	    g_print( "%s", line );
#endif

	    /* Send the line to the parent */
	    cdinfo.cddbd_stat = CDDBD_MOTD_LINE;
	    strncpy( cdinfo.line, line, 254 );
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-8");
		break;
	      } 
	  }

	close_cddbd( cddbd_fd );

	break;

	/* 
	   Read the list of sites from the server.
	   Only return sites that we can talk to (cddb, not html)
	*/
      case DB_SITES :
	if( (cddbd_fd = open_cddbd( cdinfo.server, cdinfo.port )) < 0 )
	  {
	    close_cddbd( cddbd_fd );

	    /* Tell the parent the connection failed */
	    cdinfo.cddbd_stat = CDDBD_OPEN_ERR;
	    sprintf( cdinfo.line, "err=%d", cddbd_fd );
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-1");
		break;
	      }
	    break;
	  }

	/* Read the opening banner from the server */
	if( readline( cddbd_fd, line, 255 ) < 0 )
	  {
	    close( cddbd_fd );
	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = CDDBD_READ_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-3");
		break;
	      }
	    break;
	  }  

#ifdef DEBUG1
	g_print( "%s", line );
#endif

	strncpy( cdinfo.line, line, 254 );
	code = cddbd_code( line );
	if( code > 201 )
	  {
	    close_cddbd( cddbd_fd );
	    /* Tell the parent the connect failed */
	    cdinfo.cddbd_stat = code;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-4");
		break;
	      }
	    break;
	  }

	/* Identify ourself to the server */
	sprintf( line, "cddb hello %s %s XfreeCD %s\n", getenv("USER"),
		                                        getenv("HOSTNAME"),
		                                        VERSION );

#ifdef DEBUG1
	g_print("%s", line );
#endif

	if( writen( cddbd_fd, line, strlen( line ) ) < 0 )
	  {
	    close( cddbd_fd );

	    /* Tell the parent the write failed */
	    cdinfo.cddbd_stat = CDDBD_WRITE_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-5");
		break;
	      }
	    break;
	  }

	/* Get the response from the server */
	if( readline( cddbd_fd, line, 255 ) < 0 )
	  {
	    close( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = CDDBD_READ_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-6");
		break;
	      }
	    break;
	  }  

#ifdef DEBUG1
	g_print( "%s", line );
#endif

	strncpy( cdinfo.line, line, 254 );
	code = cddbd_code( line );
	if( code > 200 )
	  {
	    close_cddbd( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = code;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-7");
		break;
	      }
	    break;
	  }

	/* Tell the server we want the MOTD */
	strcpy( line, "sites\n" );
	if( writen( cddbd_fd, line, strlen( line ) ) < 0 )
	  {
	    close( cddbd_fd );

	    /* Tell the parent the write failed */
	    cdinfo.cddbd_stat = CDDBD_WRITE_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-7");
		break;
	      }
	    break;
	  }

	/* Get the response from the server */
	if( readline( cddbd_fd, line, 255 ) < 0 )
	  {
	    close( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = CDDBD_READ_ERR;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-8");
		break;
	      }
	    break;
	  }

#ifdef DEBUG1
	g_print( "%s", line );
#endif

	strncpy( cdinfo.line, line, 254 );
	code = cddbd_code( line );
	if( code > 210 )
	  {
	    close_cddbd( cddbd_fd );

	    /* Tell the parent the read failed */
	    cdinfo.cddbd_stat = code;
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-7");
		break;
	      }
	    break;
	  }

	/* Read the MOTD one line at a time */
	while( line[0] != '.' )
	  {
	    /* Get the response from the server */
	    if( readline( cddbd_fd, line, 255 ) < 0 )
	      {
		close( cddbd_fd );

		/* Tell the parent the read failed */
		cdinfo.cddbd_stat = CDDBD_READ_ERR;
		if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
		  {
		    perror("cddbd write error-8");
		    break;
		  }
		break;
	      }

#ifdef DEBUG10
	    g_print( "%s", line );
#endif

	    /*
	       Here we should check the protocol level and parse the
	       response and only return sites that we can talk to.
	    
	       For the moment, just return the lines to the parent
	    */

	    /* Send the line to the parent */
	    cdinfo.cddbd_stat = CDDBD_SITE_LINE;
	    strncpy( cdinfo.line, line, 254 );
	    if( write( control_fd, &cdinfo, sizeof( struct CDINFO ) ) != sizeof( struct CDINFO ) )
	      {
		perror("cddbd write error-8");
		break;
	      } 
	  }

	close_cddbd( cddbd_fd );
	break;


      case DB_QUIT :
	return(0);
	break;
      }
  }
  return(0);
}
