#include "snd.h"

#include <X11/IntrinsicP.h>

axis_context *free_axis_context(axis_context *ax)
{
  if (ax) FREE(ax);
  return(NULL);
}

void draw_line (axis_context *ax,int x0,int y0,int x1,int y1) 
{
  XDrawLine(ax->dp,ax->wn,ax->gc,x0,y0,x1,y1);
}

void fill_rectangle (axis_context *ax,int x0, int y0, int width, int height)
{
  XFillRectangle(ax->dp,ax->wn,ax->gc,x0,y0,width,height);
}

static void erase_rectangle (chan_info *cp, axis_context *ax,int x0, int y0, int width, int height)
{
  XFillRectangle(ax->dp,ax->wn,erase_GC(cp),x0,y0,width,height);
}

void draw_string (axis_context *ax, int x0, int y0, char *str, int len)
{
  XDrawString(ax->dp,ax->wn,ax->gc,x0,y0,str,len);
}

void fill_polygon(axis_context *ax,int points, ...)
{
  int i;
  XPoint *pts;
  va_list ap;
  if (points == 0) return;
  pts = (XPoint *)CALLOC(points,sizeof(XPoint));
  va_start(ap,points);
  for (i=0;i<points;i++)
    {
      pts[i].x = va_arg(ap,int);
      pts[i].y = va_arg(ap,int);
    }
  va_end(ap);
  XFillPolygon(ax->dp,ax->wn,ax->gc,pts,points,Convex,CoordModeOrigin);
  FREE(pts);
}

void draw_polygon(axis_context *ax,int points, ...)
{
  int i;
  XPoint *pts;
  va_list ap;
  if (points == 0) return;
  pts = (XPoint *)CALLOC(points,sizeof(XPoint));
  va_start(ap,points);
  for (i=0;i<points;i++)
    {
      pts[i].x = va_arg(ap,int);
      pts[i].y = va_arg(ap,int);
    }
  va_end(ap);
  XDrawLines(ax->dp,ax->wn,ax->gc,pts,points,CoordModeOrigin);
  FREE(pts);
}


void set_number_font(axis_context *ax)
{
  XSetFont(ax->dp,ax->gc,((XFontStruct *)(ax->numbers_font))->fid);
}
   
void set_button_font(axis_context *ax, snd_state *ss)
{
  state_context *sgx;
  sgx = ss->sgx;
  XSetFont(ax->dp,ax->gc,(sgx->button_fontstruct)->fid);
}

void set_label_font(axis_context *ax)
{
  XSetFont(ax->dp,ax->gc,((XFontStruct *)(ax->label_font))->fid);
}

int label_width(axis_context *ax, char *txt)
{
  if (txt)
    return(XTextWidth(ax->label_font,txt,strlen(txt)));
  else return(0);
}

int mark_name_width(snd_state *ss, char *txt)
{
  state_context *sgx;
  if (txt)
    {
      sgx = ss->sgx;
      return(XTextWidth(sgx->button_fontstruct,txt,strlen(txt)));
    }
  return(0);
}

int number_width(axis_context *ax, char *num)
{
  if (num)
    return(XTextWidth(ax->numbers_font,num,strlen(num)));
  return(0);
}

int number_height(axis_context *ax)
{
  XFontStruct *numbers_font;
  numbers_font = ax->numbers_font;
  return(numbers_font->ascent+numbers_font->descent);
}

int label_height(axis_context *ax)
{
  XFontStruct *label_font;
  label_font = ax->label_font;
  return(label_font->ascent+label_font->descent);
}

void draw_lines (axis_context *ax, XPoint *points,int num)
{
  if (num == 0) return;
  XDrawLines(ax->dp,ax->wn,ax->gc,points,num,CoordModeOrigin);
}

void draw_points (axis_context *ax,XPoint *points,int num, int size)
{
  XArc *rs;
  int i,size2;
  if (num == 0) return;
  if (size == 1)
    XDrawPoints(ax->dp,ax->wn,ax->gc,points,num,CoordModeOrigin);
  else
    {
      /* create squares or whatever centered on each point */
      size2 = size/2;
      rs = (XArc *)CALLOC(num,sizeof(XArc));
      for (i=0;i<num;i++)
	{
	  rs[i].x = points[i].x - size2;
	  rs[i].y = points[i].y - size2;
	  rs[i].angle1 = 0;
	  rs[i].angle2 = 360*64;
	  rs[i].width = size;
	  rs[i].height = size;
	}
      XFillArcs(ax->dp,ax->wn,ax->gc,rs,num);
      FREE(rs);
    }
}

void draw_point (Display *dp, Drawable wn, GC gc, XPoint point, int size)
{
  if (size == 1)
    XDrawPoint(dp,wn,gc,point.x,point.y);
  else
    XFillArc(dp,wn,gc,point.x - size/2,point.y - size/2,size,size,0,360*64);
}

static XPoint polypts[4];

void fill_polygons (axis_context *ax,XPoint *points,int num,axis_info *ap, int y0)
{
  int i;
  for (i=1;i<num;i++)
    {
      polypts[0].x = points[i-1].x;
      polypts[0].y = points[i-1].y;
      polypts[1].x = points[i].x;
      polypts[1].y = points[i].y;
      polypts[2].x = polypts[1].x;
      polypts[2].y = y0;
      polypts[3].x = points[i-1].x;
      polypts[3].y = y0;
      XFillPolygon(ax->dp,ax->wn,ax->gc,polypts,4,Convex,CoordModeOrigin);
    }
}

void fill_two_sided_polygons(axis_context *ax,XPoint *points,XPoint *points1,int num)
{
  int i;
  for (i=1;i<num;i++)
    {
      polypts[0].x = points[i-1].x;
      polypts[0].y = points[i-1].y;
      polypts[1].x = points[i].x;
      polypts[1].y = points[i].y;
      polypts[2].x = points1[i].x;
      polypts[2].y = points1[i].y;
      polypts[3].x = points1[i-1].x;
      polypts[3].y = points1[i-1].y;
      XFillPolygon(ax->dp,ax->wn,ax->gc,polypts,4,Convex,CoordModeOrigin);
    }
}

void clear_window(axis_context *ax)
{
  if (ax) XClearWindow(ax->dp,ax->wn);
}

int map_over_children (Widget w, int (*func)(Widget,void *), void *userptr)
{
  /* apply func to each child in entire tree beneath top widget */
  /* taken from Douglas Young, "Motif Debugging and Performance Tuning" Prentice-Hall 1995 */
  /* used mostly to get colors right in non-scheme environments with "convenience" widgets */
  /* (also to make mix consoles handle key press correctly despite non-traversable widgets) */
  unsigned int i;
  int res;
  res = 0;
  if (w)
    {
      (*func)(w,userptr);
      if (XtIsComposite(w))
	{
	  CompositeWidget cw = (CompositeWidget)w;
	  for (i=0;i<cw->composite.num_children;i++)
	    {
	      res = map_over_children(cw->composite.children[i],func,userptr);
	      if (res) return(res);
	    }
	}
      if (XtIsWidget(w))
	{
	  for (i=0;i<w->core.num_popups;i++)
	    {
	      Widget child = w->core.popup_list[i];
	      res = map_over_children(child,func,userptr);
	      if (res) return(res);}}}
  return(res);
}

void raise_dialog(Widget w)
{
  /* since we're using non-transient message dialogs, the dialog window can become completely
   * hidden behind other windows, with no easy way to raise it back to the top, so...
   */
  Widget parent;
  if ((w) && (XtIsManaged(w)))
    {
      parent = XtParent(w);
      if ((parent) && (XtIsSubclass(parent,xmDialogShellWidgetClass)))
	XtPopup(parent,XtGrabNone);
      /* XtGrabNone means don't lock out events to rest of App (i.e. modeless dialog) */
    }
}

void raise_widget(Widget w)
{
  /* try to change stacking order (mix consoles in graph; form widgets in drawingarea; overlap covers desired console) */
  XtWidgetGeometry *request;
  request = (XtWidgetGeometry *)CALLOC(1,sizeof(XtWidgetGeometry));
  request->request_mode = CWStackMode; /* (1<<6) */
  request->stack_mode = 0; /* Above */
  XtMakeGeometryRequest(w,request,NULL);
}

int set_main_color_of_widget (Widget w,void *userptr)
{
  if (XtIsWidget(w))
    {
      if (XmIsScrollBar(w)) 
	XmChangeColor(w,(Pixel)(((snd_state *)userptr)->sgx)->position_color);
      else XmChangeColor(w,(Pixel)(((snd_state *)userptr)->sgx)->basic_color);
    }
  return(0);
}

void highlight_color(snd_state *ss, Widget w) {XmChangeColor(w,(ss->sgx)->highlight_color);}
void white_color(snd_state *ss, Widget w) {XmChangeColor(w,(ss->sgx)->white);}

void make_button_label(Widget button,char *str)
{
  XmString s1;
  s1=XmStringCreate(str,"button_font");
  XtVaSetValues(button,XmNlabelString,s1,NULL);
  XmStringFree(s1);
}

void make_bold_button_label(Widget button,char *str)
{
  XmString s1;
  s1=XmStringCreate(str,"bold_button_font");
  XtVaSetValues(button,XmNlabelString,s1,NULL);
  XmStringFree(s1);
}

void make_name_label(Widget label,char *str)
{
  XmString s1;
  s1=XmStringCreate(str,XmFONTLIST_DEFAULT_TAG);
  XtVaSetValues(label,XmNlabelString,s1,NULL);
  XmStringFree(s1);
}


int background_zoom_color(Arg *args, int n, snd_state *ss) {XtSetArg(args[n],XmNbackground,(ss->sgx)->zoom_color); return(n+1);}
int background_position_color(Arg *args, int n, snd_state *ss) {XtSetArg(args[n],XmNbackground,(ss->sgx)->position_color); return(n+1);}
int background_basic_color(Arg *args, int n, snd_state *ss) {XtSetArg(args[n],XmNbackground,(ss->sgx)->basic_color); return(n+1);}
int background_highlight_color(Arg *args, int n, snd_state *ss) {XtSetArg(args[n],XmNbackground,(ss->sgx)->highlight_color); return(n+1);}
int background_white_color(Arg *args, int n, snd_state *ss) {XtSetArg(args[n],XmNbackground,(ss->sgx)->white); return(n+1);}
int background_mix_color(Arg *args, int n, snd_state *ss) {XtSetArg(args[n],XmNbackground,(ss->sgx)->mix_color); return(n+1);}

/* error handlers -- these include the error dialog (in case no sound is active) and an error history list */

static char *snd_error_buffer = NULL;
static Widget snd_error_dialog = NULL;
static Widget snd_error_history = NULL;

static void create_snd_error_dialog(snd_state *ss)
{
  Arg args[32];
  int n;
  XmString titlestr;
  titlestr = XmStringCreate(STR_Error,XmFONTLIST_DEFAULT_TAG);
  n=0;
  if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
  XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;
#if RESIZE_DIALOG
  XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
  XtSetArg(args[n],XmNdialogTitle,titlestr); n++;
  snd_error_dialog = XmCreateErrorDialog(main_PANE(ss),STR_error,args,n);
  XtUnmanageChild(XmMessageBoxGetChild(snd_error_dialog,XmDIALOG_SYMBOL_LABEL));
  XtUnmanageChild(XmMessageBoxGetChild(snd_error_dialog,XmDIALOG_CANCEL_BUTTON));
  XtUnmanageChild(XmMessageBoxGetChild(snd_error_dialog,XmDIALOG_HELP_BUTTON));

  n=0;
#ifdef LESSTIF_VERSION
  if (!(ss->using_schemes)) n = background_white_color(args,n,ss);
#endif
  XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
  XtSetArg(args[n],XmNeditMode,XmMULTI_LINE_EDIT); n++;
  XtSetArg(args[n],XmNscrollBarDisplayPolicy,XmAS_NEEDED); n++;
  XtSetArg(args[n],XmNeditable,FALSE); n++;
  XtSetArg(args[n],XmNautoShowCursorPosition,FALSE); n++;
  XtSetArg(args[n],XmNcursorPositionVisible,FALSE); n++;
  XtSetArg(args[n],XmNscrollingPolicy,XmAUTOMATIC); n++;
  snd_error_history = XmCreateScrolledText(snd_error_dialog,STR_Error_History,args,n);
  XtManageChild(snd_error_history);

#if MANAGE_DIALOG
  XtManageChild(snd_error_dialog);
#endif

  if (!(ss->using_schemes)) map_over_children(snd_error_dialog,set_main_color_of_widget,(void *)ss);
  XmStringFree(titlestr);
  if (!(ss->using_schemes))
    {
#ifndef LESSTIF_VERSION
      XtVaSetValues(XtNameToWidget(snd_error_dialog,"OK"),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
#else
      XtVaSetValues(XmMessageBoxGetChild(snd_error_dialog,XmDIALOG_OK_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
#endif
      XtVaSetValues(snd_error_history,XmNbackground,(ss->sgx)->white,NULL);
    }
}

static void add_to_error_history(snd_state *ss, char *msg)
{
#if (!defined(HAVE_CONFIG_H)) || defined(HAVE_STRFTIME)
  char *tim,*buf;
  time_t ts;
  int pos;
#endif
  if (!snd_error_dialog) 
    create_snd_error_dialog(ss);
  else
    if (!(XtIsManaged(snd_error_dialog)))
      XtManageChild(snd_error_dialog);
#if (!defined(HAVE_CONFIG_H)) || defined(HAVE_STRFTIME)
  tim = (char *)CALLOC(32,sizeof(char));
  buf = (char *)CALLOC(32,sizeof(char));
  time(&ts);
  strftime(tim,32,"%H:%M:%S",localtime(&ts));
  sprintf(buf,"\n[%s] ",tim);
  pos = XmTextGetLastPosition(snd_error_history);
  if (pos == 0) 
    XmTextSetString(snd_error_history,buf);
  else XmTextInsert(snd_error_history,pos,buf);
  FREE(buf);
  FREE(tim);
#endif
  pos = XmTextGetLastPosition(snd_error_history);
  if (pos == 0) 
    XmTextSetString(snd_error_history,msg);
  else 
    {
      XmTextInsert(snd_error_history,pos,msg);
      if (XmGetVisibility(snd_error_history) != XmVISIBILITY_FULLY_OBSCURED)
	{
	  pos = XmTextGetLastPosition(snd_error_history);
	  XmTextShowPosition(snd_error_history,pos-1); /* if pos here, stupid thing segfaults! */
	}
    }
}

static void post_error_dialog(snd_state *ss, char *msg)
{
  XmString error_msg;
  if (!snd_error_dialog) create_snd_error_dialog(ss);
  error_msg = XmStringCreateLtoR(msg,XmFONTLIST_DEFAULT_TAG);
  XtVaSetValues(snd_error_dialog,XmNmessageString,error_msg,NULL);
  if (!(XtIsManaged(snd_error_dialog))) XtManageChild(snd_error_dialog);
  XmStringFree(error_msg);
}

void snd_error(char *format, ...)
{
  va_list ap;
  snd_info *sp;
  snd_state *ss;
  if (snd_error_buffer == NULL) snd_error_buffer = (char *)CALLOC(1024,sizeof(char));
  if (snd_error_buffer == NULL) {fprintf(stderr,"serious memory trouble!!"); abort();}
  va_start(ap,format);
  vsprintf(snd_error_buffer,format,ap);
  va_end(ap);
  ss = get_global_state();
  if (ss)
    {
#ifdef DEBUGGING
      fprintf(stderr,snd_error_buffer);
#endif
      add_to_error_history(ss,snd_error_buffer);
      sp = selected_sound(ss);
      if (sp)
	report_in_minibuffer(sp,snd_error_buffer);
      else post_error_dialog(ss,snd_error_buffer);
    }
  else fprintf(stderr,snd_error_buffer);
}

void show_snd_errors(snd_state *ss)
{
  if (snd_error_dialog)
    {
      if (!(XtIsManaged(snd_error_dialog))) 
	XtManageChild(snd_error_dialog);
      else raise_dialog(snd_error_dialog);
    }
  else post_error_dialog(ss,"no errors yet");
}

static int yes_or_no = 0;

static void YesCallback(Widget w,XtPointer clientData,XtPointer callData) {yes_or_no = 1;}
static void NoCallback(Widget w,XtPointer clientData,XtPointer callData) {yes_or_no = 0;}

int snd_yes_or_no_p(snd_state *ss,char *question)
{
  /* bad form, but in this case we want a modal dialog that stops Snd cold and gets a response -- used only for serious trouble */
  static Widget yes_or_no_dialog = NULL;
  Arg args[20];
  int n;
  XmString titlestr,error_msg;
  yes_or_no = 0;
  if (!yes_or_no_dialog)
    {
      titlestr = XmStringCreate(STR_Big_Trouble,XmFONTLIST_DEFAULT_TAG);
      n=0;
      if (!(ss->using_schemes)) n = background_basic_color(args,n,ss);
      XtSetArg(args[n],XmNresizePolicy,XmRESIZE_GROW); n++;
#if RESIZE_DIALOG
      XtSetArg(args[n],XmNnoResize,FALSE); n++;
#endif
      XtSetArg(args[n],XmNdialogTitle,titlestr); n++;
      yes_or_no_dialog = XmCreateQuestionDialog(main_PANE(ss),"yow!",args,n);
      error_msg = XmStringCreateLtoR(question,XmFONTLIST_DEFAULT_TAG);
#if MANAGE_DIALOG
      XtManageChild(yes_or_no_dialog);
#endif
      XtUnmanageChild(XmMessageBoxGetChild(yes_or_no_dialog,XmDIALOG_SYMBOL_LABEL)); /* idiotic */
      XtUnmanageChild(XmMessageBoxGetChild(yes_or_no_dialog,XmDIALOG_HELP_BUTTON));
      if (!(ss->using_schemes)) map_over_children(yes_or_no_dialog,set_main_color_of_widget,(void *)ss);
      XtVaSetValues(yes_or_no_dialog,XmNdialogStyle,XmDIALOG_FULL_APPLICATION_MODAL,XmNmessageString,error_msg,NULL);
      XtAddCallback(yes_or_no_dialog,XmNokCallback,YesCallback,NULL);
      XtAddCallback(yes_or_no_dialog,XmNcancelCallback,NoCallback,NULL);

      if (!(ss->using_schemes))
	{
	  XtVaSetValues(XmMessageBoxGetChild(yes_or_no_dialog,XmDIALOG_OK_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	  XtVaSetValues(XmMessageBoxGetChild(yes_or_no_dialog,XmDIALOG_CANCEL_BUTTON),XmNarmColor,(ss->sgx)->pushed_button_color,NULL);
	}
      XmStringFree(error_msg);
      XmStringFree(titlestr);
    }
  if (!(XtIsManaged(yes_or_no_dialog))) XtManageChild(yes_or_no_dialog);
  while (XtIsManaged(yes_or_no_dialog))
    {
      XEvent event;
      XtAppNextEvent(XtWidgetToApplicationContext(yes_or_no_dialog),&event);
      XtDispatchEvent(&event);
    }
  return(yes_or_no);
}
  

void set_title(snd_state *ss, char *title)
{
  XtVaSetValues(main_SHELL(ss),XmNtitle,title,NULL);
}

void make_axes(chan_info *cp, axis_info *ap, int x_style)
{
  Widget w;
  snd_info *sp;
  axis_context *ax;
  chan_info *cp0;
  if (!(ap->ax))
    {
      ax = (axis_context *)CALLOC(1,sizeof(axis_context));
      ap->ax = ax;
      ax->label_font = axis_label_FONTSTRUCT(ap);
      ax->numbers_font = axis_numbers_FONTSTRUCT(ap);
    }
  else ax = ap->ax;
  sp = cp->sound;
  if (cp->tcgx) 
    w = channel_graph(sp->chans[0]);
  else w = channel_graph(cp);
  ax->dp = XtDisplay(w);
  ax->gc = copy_GC(cp);
  ax->wn = XtWindow(w);
  if (cp->clear) 
    {
      switch (sp->combining)
	{
	case CHANNELS_SUPERIMPOSED:     /* clear our portion and mark so others don't clear */
	  cp0 = sp->chans[0];
	  if (cp0->clear)
	    {
	      clear_window(ap->ax);     /* clear entire channel window (once) */
	      cp0->clear = 0;           /* channel graphs can occur in any order (background procs) */
	    }
	  break;
	case CHANNELS_COMBINED:         /* clear only our (full width) portion of the window */
	  erase_rectangle(cp,ap->ax,0,ap->y_offset,ap->window_width,ap->height); 
	  break; 
	default: 
	  clear_window(ap->ax);         /* clear entire channel window */
	  break;
	}
      cp->clear = 0;
    }
  make_axes_1(cp,ap,x_style,snd_SRATE(cp));
}

static int complain_about_focus_policy = 1;

void goto_window(Widget text)
{
  int err;
  if ((XmIsTraversable(text)) && (XmGetVisibility(text) != XmVISIBILITY_FULLY_OBSCURED))
    {
      if (!(XmProcessTraversal(text,XmTRAVERSE_CURRENT)))
	{
	  if (complain_about_focus_policy)
	    {
	      XtVaGetValues(text,XmNkeyboardFocusPolicy,&err,NULL);
	      if (err == XmEXPLICIT)
		snd_error("%s[%d] %s: traverse to %s failed!",__FILE__,__LINE__,__FUNCTION__,XtName(text));
	      else 
		{
		  snd_error("%s[%d] %s: keyboard focus policy is not explicit!",__FILE__,__LINE__,__FUNCTION__);
		  complain_about_focus_policy = 0;
		}
	    }
	}
    }
}

void goto_graph(chan_info *cp)
{
  snd_info *sp;
  if (cp)
    {
      sp = cp->sound;
      if ((cp->chan == 0) || (sp->combining == CHANNELS_SEPARATE))
	goto_window(channel_graph(cp));
      else goto_window(channel_graph(sp->chans[0]));
    }
}

void goto_minibuffer(snd_info *sp)
{
  if (sp) goto_window(w_snd_info(sp));
}

void text_set_string(Widget txt, char *str) 
{
  XmTextSetString(txt,str);
  XmUpdateDisplay(txt);
}

void text_set_cursor_position(Widget txt, int pos)
{
  XmTextSetCursorPosition(txt,pos);
}

char *text_get_string(Widget txt) {return(XmTextGetString(txt));}

int get_window_height(Widget w)
{
  Dimension height;
  XtVaGetValues(w,XmNheight,&height,NULL);
  return(height);
}

int get_window_width(Widget w)
{
  Dimension width;
  XtVaGetValues(w,XmNwidth,&width,NULL);
  return(width);
}

int get_raw_value(Widget w) {int val; XtVaGetValues(w,XmNvalue,&val,NULL); return(val);}
int get_raw_size(Widget w) {int val; XtVaGetValues(w,XmNsliderSize,&val,NULL); return(val);}
int set_raw_value(Widget w, int val) {XtVaSetValues(w,XmNvalue,val,NULL); return(val);}

XtCallbackList make_callback_list(XtCallbackProc callback, XtPointer closure)
{
  XtCallbackList nlist;
  nlist = (XtCallbackList)CALLOC(2,sizeof(XtCallbackRec));
  nlist[0].callback = callback;
  nlist[0].closure = closure;
  nlist[1].callback = NULL;
  nlist[1].closure = NULL;
  return(nlist);
}

#include <Xm/SashP.h>
int color_sashes(Widget w, void *ptr)
{
  if ((XtIsWidget(w)) && (XtIsManaged(w)) && (XtIsSubclass(w,xmSashWidgetClass)))
    XmChangeColor(w,(Pixel)(((snd_state *)ptr)->sgx)->sash_color);
  return(0);
}

void check_for_event(snd_state *ss)
{
  /* this is needed to force label updates and provide interrupts for long computations */
  XEvent event;
  XtInputMask msk = 0;
  XtAppContext app;
  ss->checking_explicitly = 1;
  app = main_APP(ss);
  while (1)
    {
      msk = XtAppPending(app);
      if (msk & (XtIMXEvent | XtIMAlternateInput))
	{
	  XtAppNextEvent(app,&event);
	  XtDispatchEvent(&event);
	}
      else break;
    }
  ss->checking_explicitly = 0;
}

static Dimension app_x,app_y;

void save_window_size(snd_state *ss)
{
  XtVaGetValues(main_SHELL(ss),XmNwidth,&app_x,XmNheight,&app_y,NULL);
}

void restore_window_size(snd_state *ss)
{
  XtVaSetValues(main_SHELL(ss),XmNwidth,app_x,XmNheight,app_y,NULL);
}


/* ---------------- text widget specializations ---------------- */

void textfield_focus_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_state *ss = (snd_state *)clientData;
  if (!(ss->using_schemes)) XtVaSetValues(w,XmNbackground,(ss->sgx)->text_focus_color,NULL);
}

void textfield_unfocus_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  snd_state *ss = (snd_state *)clientData;
  if (!(ss->using_schemes)) XtVaSetValues(w,XmNbackground,(ss->sgx)->basic_color,NULL);
}


/* -------- specialized action procs -------- */

static snd_state *action_ss; /* need some way to get Snd global state into action proc */
static int actions_loaded = 0;
#define CONTROL_KEY 4

static void No_op (Widget w, XEvent *ev, char **str, Cardinal *num) 
{
  /* return does not cause widget activation in many textfield cases -- it is a true no-op */
}

#define snd_K_u XK_u 
#define snd_K_x XK_x 

static void Activate_keyboard (Widget w, XEvent *ev, char **str, Cardinal *num) 
{
  /* make the current channel active preloading kbd cmd with str[0]+ctrl bit */
  chan_info *cp;
  cp = current_channel(action_ss);
  if (cp) 
    {
      goto_graph(cp);
      keyboard_command(cp,(str[0][0] == 'u') ? snd_K_u : snd_K_x,CONTROL_KEY);
    }
}

static void Activate_channel (Widget w, XEvent *ev, char **str, Cardinal *num) 
{
  /* make the current channel active and abort if anything in progress */
  chan_info *cp;
  if (action_ss->checking_explicitly) action_ss->stopped_explicitly = 1; 
  cp = current_channel(action_ss);
  if (cp) goto_graph(cp);
}

static char *listener_selection = NULL;

static void Kill_line(Widget w, XEvent *ev, char **str, Cardinal *num) 
{
  /* C-k with storage of killed text */
  XmTextPosition curpos,loc;
  Boolean found;
  curpos = XmTextGetCursorPosition(w);
  found = XmTextFindString(w,curpos,"\n",XmTEXT_FORWARD,&loc);
  if (!found) loc = XmTextGetLastPosition(w);
  if (loc > curpos)
    {
      if (listener_selection) FREE(listener_selection);
      XmTextSetSelection(w,curpos,loc,CurrentTime);
      listener_selection = XmTextGetSelection(w); /* xm manual p329 sez storage is allocated here */
      XmTextCut(w,CurrentTime);
    }
}

static void Yank(Widget w, XEvent *ev, char **str, Cardinal *num) 
{
  /* copy current selection at current cursor position */
  XmTextPosition curpos;
  if (listener_selection) 
    {
      curpos = XmTextGetCursorPosition(w);
      XmTextInsert(w,curpos,listener_selection);
      curpos+=strlen(listener_selection);
      XmTextShowPosition(w,curpos);
      XmTextSetCursorPosition(w,curpos);
      XmTextClearSelection(w,ev->xkey.time); /* so C-y + edit doesn't forbid the edit */
    }
}

static int last_prompt;

static void Begin_of_line(Widget w, XEvent *ev, char **str, Cardinal *num) 
{
  /* don't back up before listener prompt */
  XmTextPosition curpos,loc;
  Boolean found;
  curpos = XmTextGetCursorPosition(w) - 1;
  found = XmTextFindString(w,curpos,"\n",XmTEXT_BACKWARD,&loc);
  XmTextSetCursorPosition(w,((found) && (loc>last_prompt)) ? (loc+1) : (last_prompt+1));
}

static void Delete_region(Widget w, XEvent *ev, char **str, Cardinal *num) 
{
  XmTextCut(w,CurrentTime);
}

static XmTextPosition down_pos, last_pos;
static Widget lisp_listener = NULL;

static void B1_press(Widget w, XEvent *event, char **str, Cardinal *num) 
{
  XmTextPosition pos;
  XButtonEvent *ev = (XButtonEvent *)event;
  XmProcessTraversal(w,XmTRAVERSE_CURRENT);
  /* we're replacing the built-in take_focus action here, so do it by hand, but leave listener blue, so to speak */
  if ((!(action_ss->using_schemes)) && (w != lisp_listener))
    XtVaSetValues(w,XmNbackground,(action_ss->sgx)->white,NULL);
  if (w == lisp_listener) XmTextClearSelection(lisp_listener,CurrentTime); /* should this happen in other windows as well? */
  pos = XmTextXYToPos(w,ev->x,ev->y);
  XmTextSetCursorPosition(w,pos);
  down_pos = pos;
  last_pos = pos;
}

static void B1_move(Widget w, XEvent *event, char **str, Cardinal *num) 
{
  XmTextPosition pos;
  XButtonEvent *ev = (XButtonEvent *)event;
  pos = XmTextXYToPos(w,ev->x,ev->y);
  if (last_pos > pos) /* must have backed up the cursor */
    XmTextSetHighlight(w,pos,last_pos,XmHIGHLIGHT_NORMAL);
  if (down_pos != pos)
    XmTextSetHighlight(w,down_pos,pos,XmHIGHLIGHT_SELECTED);
  last_pos = pos;
}

static void B1_release(Widget w, XEvent *event, char **str, Cardinal *num) 
{
  XmTextPosition pos;
  XButtonEvent *ev = (XButtonEvent *)event;
  pos = XmTextXYToPos(w,ev->x,ev->y);
  XmTextSetCursorPosition(w,pos);
  if (down_pos != pos)
    {
      XmTextSetHighlight(w,down_pos,pos,XmHIGHLIGHT_SELECTED);
      if (listener_selection) FREE(listener_selection);
      XmTextSetSelection(w,down_pos,pos,CurrentTime);
      listener_selection = XmTextGetSelection(w);
    }
}

static void Text_transpose(Widget w, XEvent *event, char **str, Cardinal *num) 
{
  XmTextPosition curpos;
  char buf[3]; /* needs room for null */
  char tmp;
  curpos = XmTextGetCursorPosition(w);
  if (curpos > 1)
    {
      XmTextGetSubstring(w,(XmTextPosition)(curpos-1),2,3,buf);
      tmp = buf[0];
      buf[0]=buf[1];
      buf[1]=tmp;
      XmTextReplace(w,curpos-1,curpos+1,buf);
      XmTextSetCursorPosition(w,curpos+1);
    }
}

static void Word_upper(Widget w, XEvent *event, char **str, Cardinal *num) 
{
  int i,j,length,wstart,wend,up,cap;
  XmTextPosition curpos,endpos;
  char *buf;
  up = (str[0][0] == 'u');
  cap = (str[0][0] == 'c');
  curpos = XmTextGetCursorPosition(w);
  endpos = XmTextGetLastPosition(w);
  if (curpos < endpos)
    {
      length = endpos - curpos;
      buf = (char *)CALLOC(length+1,sizeof(char));
      XmTextGetSubstring(w,curpos,length,length+1,buf);
      wstart = 0;
      wend = length;
      for (i=0;i<length;i++)
	if (!isspace((int)(buf[i])))
	  {
	    wstart=i;
	    break;
	  }
      for (i=wstart+1;i<length;i++)
	if (isspace((int)(buf[i])))
	  {
	    wend = i;
	    break;
	  }
      if (cap)
	{
	  buf[0] = toupper(buf[wstart]);
	  buf[1]='\0';
	  XmTextReplace(w,curpos+wstart,curpos+wstart+1,buf);
	}
      else
	{
	  for (i=wstart,j=0;i<wend;i++,j++)
	    if (up) 
	      buf[j] = toupper(buf[i]);
	    else buf[j] = tolower(buf[i]);
	  buf[j]='\0';
	  XmTextReplace(w,curpos+wstart,curpos+wend,buf);
	}
      XmTextSetCursorPosition(w,curpos+wend);
    }
}

static Widget *cmpwids = NULL;
static int cmpwids_size = 0;

static void add_completer_widget(Widget w, int row)
{
  int i;
  if (row >= 0)
    {
      if (cmpwids_size <= row)
	{
	  i = cmpwids_size;
	  cmpwids_size = row+8;
	  if (cmpwids == NULL)
	    cmpwids = (Widget *)CALLOC(cmpwids_size,sizeof(Widget));
	  else 
	    {
	      cmpwids = (Widget *)REALLOC(cmpwids,cmpwids_size * sizeof(Widget));
	      for (;i<cmpwids_size;i++) cmpwids[i] = NULL;
	    }
	}
      cmpwids[row] = w;
    }
}

static void Name_completion(Widget w, XEvent *event, char **str, Cardinal *num) 
{
  int data,i,matches,need_position;
  Position wx,wy;
  int xoff,yoff; 
  Window wn;
  Pixel old_color;
  char *old_text,*new_text,*search_text;
  data = -1;
  for (i=0;i<cmpwids_size;i++)
    if (w == cmpwids[i])
      {
	data = i;
	break;
      }
  if (data >= 0)
    {
      old_text = XmTextGetString(w);
      new_text = complete_text(old_text,data);
      matches = get_completion_matches();
      XmTextSetString(w,new_text);
      XmTextSetCursorPosition(w,XmTextGetLastPosition(w));
      if ((strcmp(old_text,new_text) == 0) && (matches != -1))
	{
	  XtVaGetValues(w,XmNforeground,&old_color,NULL);
	  if (matches > 1)
	    XtVaSetValues(w,XmNforeground,(action_ss->sgx)->green,NULL);
	  else if (matches == 0) XtVaSetValues(w,XmNforeground,(action_ss->sgx)->red,NULL);
	  XmUpdateDisplay(w);
#ifndef _MSC_VER
	  sleep(1);
#endif
	  XtVaSetValues(w,XmNforeground,old_color,NULL);
	  XmUpdateDisplay(w);
	  if (matches > 1) 
	    {
	      set_save_completions(1);
	      search_text = complete_text(old_text,data);
	      if (search_text) FREE(search_text);
	      need_position = (!(help_shell(action_ss)));
	      display_completions(action_ss);
	      set_save_completions(0);
	      if (need_position)
		{
		  /* try to position the newly popped up help window below the text field */
		  XtVaGetValues(w,XmNx,&wx,XmNy,&wy,NULL);
		  XTranslateCoordinates(XtDisplay(w),XtWindow(w),DefaultRootWindow(XtDisplay(w)),0,0,&xoff,&yoff,&wn);
		  wx+=xoff; 
		  wy+=yoff;
		  XtVaSetValues(help_shell(action_ss),XmNx,wx,XmNy,wy+40,NULL);
		}
	    }
	  else
	    {
	      /* (if matches == 0) here we could back up the text looking for the nearest completion */
	    }
	}
      if (old_text) FREE(old_text);
      if (new_text) FREE(new_text);
    }
}

static int find_indentation(char *str,int loc)
{
  int line_beg = 0,open_paren = -1,parens,i;
  parens = 0;
  for (i=loc-1;i>=0;i--)
    {
      if (str[i] == ')') parens--;
      if (str[i] == '(') parens++;
      if (parens == 1) {open_paren = i; break;}
    }
  if (open_paren == -1) return(1);
  if (open_paren == 0) return(3);
  for (i=open_paren-1;i>0;i--)
    {
      if (str[i] == '\n') {line_beg = i; break;}
    }
  if (line_beg == 0) return(1);
  return(open_paren - line_beg + 2);
}

static void Listener_completion(Widget w, XEvent *event, char **str, Cardinal *num) 
{
  /* used only by the listener widget -- needs to be smart about text since overall string can be enormous 
   *   and we don't want to back up past the last prompt
   *   also if at start of line (or all white-space to previous \n, indent
   */
  int beg,end,len,i,k,matches = 0,need_position,spaces,text_pos = 0,cr_pos = 0;
  char *old_text,*new_text = NULL,*file_text = NULL,*new_file = NULL;
  Position wx,wy;
  int xoff,yoff; 
  Window wn;
  beg = last_prompt+1;
  end = XmTextGetLastPosition(w);
  if (end <= beg) return;
  len = end-beg+1;
  old_text = (char *)CALLOC(len+1,sizeof(char));
  XmTextGetSubstring(w,beg,len,len+1,old_text);
  /* now old_text is the stuff typed since the last prompt */
  if (old_text)
    {
      len = strlen(old_text);
      for (i=len-1;i>0;i--)
	{
	  if (old_text[i] == '\n')
	    {
	      /* tab as indentation */
	      /* look at previous line to decide */
	      spaces = find_indentation(old_text,i);
	      if (spaces > 0)
		{
		  file_text = (char *)CALLOC(spaces+1,sizeof(char));
		  for (k=0;k<spaces;k++) file_text[k] = ' ';
		  file_text[spaces] = 0;
		  XmTextInsert(w,end,file_text);
		  XmTextSetCursorPosition(w,XmTextGetLastPosition(w));
		  FREE(file_text);
		  file_text = NULL;
		}
	      FREE(old_text);
	      old_text = NULL;
	      return;
	    }
	  if (old_text[i] == ';')
	    {
	      /* this isn't quite right, but how much effort should we put in it? */
	      spaces = 20;
	      for (k=i-1;k>0;k--) 
		if (old_text[k] == '\n') 
		  {cr_pos = k; break;} 
		else 
		  if ((!(isspace((int)(old_text[k])))) && (text_pos == 0)) 
		    text_pos = k;
	      if (text_pos > 0)
		text_pos -= cr_pos;
	      if (cr_pos == 0) spaces--; 
	      if (text_pos < spaces)
		{
		  file_text = (char *)CALLOC(spaces+2,sizeof(char));
		  for (k=text_pos+1;k<spaces;k++) file_text[k-text_pos-1] = ' ';
		  file_text[spaces] = ';';
		  file_text[spaces+1] = 0;
		  XmTextInsert(w,end-1,file_text);
		  XmTextSetCursorPosition(w,XmTextGetLastPosition(w));
		  FREE(file_text);
		}
	      FREE(old_text);
	      return;
	    }
	  if (old_text[i] == '\"')
	    {
	      file_text = copy_string((char *)(old_text+i+1));
	      new_file = filename_completer(file_text);
	      len = i + 2 + snd_strlen(new_file);
	      new_text = (char *)CALLOC(len,sizeof(char));
	      strncpy(new_text,old_text,i+1);
	      strcat(new_text,new_file);
	      if (new_file) FREE(new_file);
	      break;
	    }
	  if (isspace((int)(old_text[i]))) break;
	}
      if (new_text == NULL) new_text = command_completer(old_text);
      if (strcmp(old_text,new_text) == 0) matches = get_completion_matches();
      XmTextReplace(w,beg,end,new_text);
      XmTextSetCursorPosition(w,XmTextGetLastPosition(w));
      if (new_text) {FREE(new_text); new_text = NULL;}
      if (matches > 1)
	{
	  clear_possible_completions();
	  set_save_completions(1);
	  if (file_text) new_text = filename_completer(file_text); else new_text = command_completer(old_text);
	  if (new_text) {FREE(new_text); new_text = NULL;}
	  need_position = (!(help_shell(action_ss)));
	  display_completions(action_ss);
	  set_save_completions(0);
	  if (need_position)
	    {
	      /* try to position the newly popped up help window below the text field */
	      XtVaGetValues(w,XmNx,&wx,XmNy,&wy,NULL);
	      XTranslateCoordinates(XtDisplay(w),XtWindow(w),DefaultRootWindow(XtDisplay(w)),0,0,&xoff,&yoff,&wn);
	      wx+=xoff; 
	      wy+=yoff;
	      XtVaSetValues(help_shell(action_ss),XmNx,wx,XmNy,wy+140,NULL);
	    }
	  if (file_text) FREE(file_text);
	}
      if (old_text) FREE(old_text);
    }
}

#define NUM_ACTS 14
static XtActionsRec acts[] = {
  {"no-op",No_op},
  {"activate-keyboard",Activate_keyboard},
  {"activate-channel",Activate_channel},
  {"yank",Yank},
  {"delete-region",Delete_region},
  {"kill-line",Kill_line},
  {"begin-of-line",Begin_of_line},
  {"b1-press",B1_press},
  {"b1-move",B1_move},
  {"b1-release",B1_release},
  {"text-transpose",Text_transpose},
  {"word-upper",Word_upper},
  {"name-completion",Name_completion},
  {"listener-completion",Listener_completion},
};

/* translation tables for emacs compatibility and better inter-widget communication */

/* for textfield (single-line) widgets */
static char TextTrans2[] =
       "Ctrl <Key>a:	    beginning-of-line()\n\
	Ctrl <Key>b:	    backward-character()\n\
	Mod1 <Key>b:	    backward-word()\n\
	Mod1 <Key>c:	    word-upper(c)\n\
	Ctrl <Key>d:	    delete-next-character()\n\
	Mod1 <Key>d:	    delete-next-word()\n\
	Ctrl <Key>e:	    end-of-line()\n\
	Ctrl <Key>f:	    forward-character()\n\
	Mod1 <Key>f:	    forward-word()\n\
	Ctrl <Key>g:	    activate()\n\
	Ctrl <Key>h:	    delete-previous-character()\n\
	Ctrl <Key>k:	    delete-to-end-of-line()\n\
	Mod1 <Key>l:	    word-upper(l)\n\
        Ctrl <Key>r:        activate()\n\
        Ctrl <Key>s:        activate()\n\
	Ctrl <Key>t:	    text-transpose()\n\
	Mod1 <Key>u:	    word-upper(u)\n\
	Mod1 <Key><:	    beginning-of-line()\n\
	Mod1 <Key>>:	    end-of-line()\n\
	<Key>Delete:	    delete-previous-character()\n\
	Mod1 <Key>Delete:   delete-to-start-of-line()\n\
	<Key>Tab:	    name-completion()\n\
	<Key>Return:	    activate()\n";
static XtTranslations transTable2 = NULL;

/* same but try to avoid causing the currently active pushbutton widget to appear to be activated by <cr> in the text widget */
static char TextTrans6[] =
       "Ctrl <Key>a:	    beginning-of-line()\n\
	Ctrl <Key>b:	    backward-character()\n\
	Mod1 <Key>b:	    backward-word()\n\
	Mod1 <Key>c:	    word-upper(c)\n\
	Ctrl <Key>d:	    delete-next-character()\n\
	Mod1 <Key>d:	    delete-next-word()\n\
	Ctrl <Key>e:	    end-of-line()\n\
	Ctrl <Key>f:	    forward-character()\n\
	Mod1 <Key>f:	    forward-word()\n\
	Ctrl <Key>g:	    no-op()\n\
	Ctrl <Key>h:	    delete-previous-character()\n\
	Ctrl <Key>k:	    delete-to-end-of-line()\n\
	Mod1 <Key>l:	    word-upper(l)\n\
	Ctrl <Key>t:	    text-transpose()\n\
	Mod1 <Key>u:	    word-upper(u)\n\
	Mod1 <Key><:	    beginning-of-line()\n\
	Mod1 <Key>>:	    end-of-line()\n\
	<Key>Delete:	    delete-previous-character()\n\
	Mod1 <Key>Delete:   delete-to-start-of-line()\n\
	<Key>Tab:	    name-completion()\n\
	<Key>Return:	    no-op()\n";
static XtTranslations transTable6 = NULL;

/* for text (multi-line) widgets */
static char TextTrans3[] =
       "Ctrl <Key>a:	    beginning-of-line()\n\
	Ctrl <Key>b:	    backward-character()\n\
	Mod1 <Key>b:	    backward-word()\n\
	Mod1 <Key>c:	    word-upper(c)\n\
	Ctrl <Key>d:	    delete-next-character()\n\
	Mod1 <Key>d:	    delete-next-word()\n\
	Ctrl <Key>e:	    end-of-line()\n\
	Ctrl <Key>f:	    forward-character()\n\
	Mod1 <Key>f:	    forward-word()\n\
	Ctrl <Key>g:	    activate()\n\
	Ctrl <Key>h:	    delete-previous-character()\n\
	Ctrl <Key>j:	    newline-and-indent()\n\
	Ctrl <Key>k:	    kill-line()\n\
	Mod1 <Key>l:	    word-upper(l)\n\
	Ctrl <Key>n:	    next-line()\n\
	Ctrl <Key>o:	    newline-and-backup()\n\
	Ctrl <Key>p:	    previous-line()\n\
	Ctrl <Key>t:	    text-transpose()\n\
	Mod1 <Key>u:	    word-upper(u)\n\
	Ctrl <Key>v:	    next-page()\n\
	Mod1 <Key>v:	    previous-page()\n\
	Ctrl <Key>w:	    delete-region()\n\
	Ctrl <Key>y:	    yank()\n\
	Ctrl <Key>z:	    activate()\n\
	Mod1 <Key>[:	    backward-paragraph()\n\
	Mod1 <Key>]:	    forward-paragraph()\n\
	Mod1 <Key><:	    beginning-of-file()\n\
	Mod1 <Key>>:	    end-of-file()\n\
	<Key>Delete:	    delete-previous-character()\n\
	Mod1 <Key>Delete:   delete-to-start-of-line()\n\
	Ctrl <Key>osfLeft:  page-left()\n\
	Ctrl <Key>osfRight: page-right()\n\
	Ctrl <Key>osfDown:  next-page()\n\
	Ctrl <Key>osfUp:    previous-page()\n\
	Ctrl <Key>space:    set-anchor()\n\
	<Btn1Down>:	    b1-press()\n\
	<Btn1Up>:	    b1-release()\n\
	<Btn1Motion>:	    b1-move()\n\
	<Key>Return:	    newline()\n";
static XtTranslations transTable3 = NULL;

/* for lisp listener */
static char TextTrans4[] =
       "Ctrl <Key>a:	    begin-of-line()\n\
	Ctrl <Key>b:	    backward-character()\n\
	Mod1 <Key>b:	    backward-word()\n\
	Mod1 <Key>c:	    word-upper(c)\n\
	Ctrl <Key>d:	    delete-next-character()\n\
	Mod1 <Key>d:	    delete-next-word()\n\
	Ctrl <Key>e:	    end-of-line()\n\
	Ctrl <Key>f:	    forward-character()\n\
	Mod1 <Key>f:	    forward-word()\n\
	Ctrl <Key>g:	    activate-channel()\n\
	Ctrl <Key>h:	    delete-previous-character()\n\
	Ctrl <Key>j:	    newline-and-indent()\n\
	Ctrl <Key>k:	    kill-line()\n\
	Ctrl <Key>l:	    redraw-display()\n\
	Mod1 <Key>l:	    word-upper(l)\n\
	Ctrl <Key>n:	    next-line()\n\
	Ctrl <Key>o:	    newline-and-backup()\n\
	Ctrl <Key>p:	    previous-line()\n\
	Ctrl <Key>t:	    text-transpose()\n\
	Ctrl <Key>u:	    activate-keyboard(u)\n\
	Mod1 <Key>u:	    word-upper(u)\n\
	Ctrl <Key>v:	    next-page()\n\
	Mod1 <Key>v:	    previous-page()\n\
	Ctrl <Key>w:	    delete-region()\n\
	Ctrl <Key>x:	    activate-keyboard(x)\n\
	Ctrl <Key>y:	    yank()\n\
	Ctrl <Key>z:	    activate()\n\
	Mod1 <Key>[:	    backward-paragraph()\n\
	Mod1 <Key>]:	    forward-paragraph()\n\
	Mod1 <Key><:	    beginning-of-file()\n\
	Mod1 <Key>>:	    end-of-file()\n\
	<Key>Delete:	    delete-previous-character()\n\
	Mod1 <Key>Delete:   delete-to-start-of-line()\n\
	Ctrl <Key>osfLeft:  page-left()\n\
	Ctrl <Key>osfRight: page-right()\n\
	Ctrl <Key>osfDown:  next-page()\n\
	Ctrl <Key>osfUp:    previous-page()\n\
	Ctrl <Key>space:    set-anchor()\n\
	<Btn1Down>:	    b1-press()\n\
	<Btn1Up>:	    b1-release()\n\
	<Btn1Motion>:	    b1-move()\n\
	<Key>Tab:	    listener-completion()\n\
	<Key>Return:	    activate()\n";
static XtTranslations transTable4 = NULL;


/* -------- text related widgets -------- */

static void remember_event(Widget w,XtPointer clientData,XtPointer callData) 
{
  snd_state *ss = (snd_state *)clientData;
  XmAnyCallbackStruct *cb = (XmAnyCallbackStruct *)callData;
  (ss->sgx)->text_activate_event = cb->event;
  (ss->sgx)->text_widget = w;
}

Widget sndCreateTextFieldWidget(snd_state *ss, char *name, Widget parent, Arg *args, int n, int activatable, int completer)
{
  /* white background when active, emacs translations, text_activate_event in ss->sgx for subsequent activation check */
  Widget df;
  if (!actions_loaded) {XtAppAddActions(main_APP(ss),acts,NUM_ACTS); actions_loaded = 1; action_ss = ss;}
  XtSetArg(args[n],XmNactivateCallback,make_callback_list(remember_event,(XtPointer)ss)); n++;
  /* can't use XmNuserData here because it is in use elsewhere (snd-xmix.c) */
  df = XtCreateManagedWidget(name,xmTextFieldWidgetClass,parent,args,n);
  XtAddCallback(df,XmNfocusCallback,textfield_focus_Callback,ss);
  XtAddCallback(df,XmNlosingFocusCallback,textfield_unfocus_Callback,ss);
  if (activatable == ACTIVATABLE)
    {
      if (!transTable2) transTable2 = XtParseTranslationTable(TextTrans2);
      XtOverrideTranslations(df,transTable2);
    }
  else
    {
      if (!transTable6) transTable6 = XtParseTranslationTable(TextTrans6);
      XtOverrideTranslations(df,transTable6);
    }
  add_completer_widget(df,completer);
  return(df);
}

void add_completer_to_textfield(snd_state *ss, Widget w, int completer)
{
  /* used to make file selection dialog act like other text field widgets */
  if (!actions_loaded) {XtAppAddActions(main_APP(ss),acts,NUM_ACTS); actions_loaded = 1; action_ss = ss;}
  if (!transTable2) transTable2 = XtParseTranslationTable(TextTrans2);
  XtOverrideTranslations(w,transTable2);
  add_completer_widget(w,completer);
}

Widget sndCreateTextWidget(snd_state *ss, char *name, Widget parent, Arg *args, int n)
{
  /* white background when active, emacs translations, text_activate_event in ss->sgx for subsequent activation check */
  Widget df;
  if (!actions_loaded) {XtAppAddActions(main_APP(ss),acts,NUM_ACTS); actions_loaded = 1; action_ss = ss;}
  XtSetArg(args[n],XmNactivateCallback,make_callback_list(remember_event,(XtPointer)ss)); n++;
  XtSetArg(args[n],XmNeditMode,XmMULTI_LINE_EDIT); n++;
  df = XmCreateScrolledText(parent,name,args,n);
  XtManageChild(df);
  /* df = XtCreateManagedWidget(name,xmTextWidgetClass,parent,args,n); */
  /* XtAddCallback(df,XmNfocusCallback,textfield_focus_Callback,ss); */
  XtAddCallback(df,XmNlosingFocusCallback,textfield_unfocus_Callback,ss);
  /* b1_press action overrides focus */
  if (!transTable3) transTable3 = XtParseTranslationTable(TextTrans3);
  XtOverrideTranslations(df,transTable3);
  return(df);
}


/* ---------------- command widget replacement ---------------- */

static Widget lisp_window = NULL;

void snd_append_char(snd_state *ss, char *msg)
{
  if (lisp_listener)
    {
      XmTextInsert(lisp_listener,XmTextGetLastPosition(lisp_listener),msg);
    }
}
 
static Widget pane_frm = NULL; 

static char listener_prompt_buffer[4];
static char *listener_prompt_with_cr(snd_state *ss)
{
  sprintf(listener_prompt_buffer,"\n%s",listener_prompt(ss));
  return(listener_prompt_buffer);
}
 
void snd_append_command(snd_state *ss, char *msg)
{
  int cmd_eot;
  if (lisp_listener)
    {
      if (ss->result_printout != PLAIN_MESSAGE) 
	XmTextInsert(lisp_listener,XmTextGetLastPosition(lisp_listener),"\n");
      if (msg)
	XmTextInsert(lisp_listener,XmTextGetLastPosition(lisp_listener),msg);
      cmd_eot = XmTextGetLastPosition(lisp_listener);
      if (ss->result_printout == MESSAGE_WITH_CARET) XmTextInsert(lisp_listener,cmd_eot,listener_prompt_with_cr(ss));
      ss->result_printout = 0;
      cmd_eot = XmTextGetLastPosition(lisp_listener);
      last_prompt = cmd_eot-1;
      XmTextShowPosition(lisp_listener,cmd_eot-1);
      XmUpdateDisplay(lisp_listener);
    }
}

/* check_balance stolen from scmwm-0.9/utilities/scwmrepl/scwmrepl.c */

static int check_balance(char *expr, int start, int end) {
  /* If you think _this_ is hairy, try doing it for C statements. */
  int i;
  int non_whitespace_p=0;
  int paren_count=0;
  int prev_separator=1;
  int quote_wait=0;

  i=start;
  while (i<end) {
    switch(expr[i]) {
    case ';' :
      /* skip till newline. */
      do {
	i++;
      } while (expr[i]!='\n' && i<end);
      break;
    case ' ':
    case '\n':
    case '\t':
    case '\r':
      if (non_whitespace_p && paren_count==0 && !quote_wait) {
	return i;
      } else {
	prev_separator=1;
	i++;
      }
      break;
    case '\"' :
      if (non_whitespace_p && paren_count==0 &&
	  !quote_wait) {
	return i;
      } else {
	/* skip past ", ignoring \" */
	do {
 	  i++;
	  if (i < end && expr[i]=='\\') {
	    i++;
	  }
	} while (i < end && expr[i]!='\"');
	i++;
	if (paren_count==0) {
	  if (i < end) {
	    return i;
	  } else {
	    return 0;
	  }
	} else {
	  prev_separator=1;
	  non_whitespace_p=1;
	  quote_wait=0;
	}
      }
      break;
    case '#' :
      if (non_whitespace_p && paren_count==0 &&
	  !quote_wait) {
	return i;
      } else {
	if (prev_separator && i+1<end && expr[i+1]=='{') {
	  /* skip past }#, ignoring \} */
	  do {
	    i++;
	    if (i < end && expr[i]=='\\') {
	      i++;
	    }
	  } while (i < end && !(expr[i]=='}' && i+1<end
				&& expr[i+1]=='#'));
	  i+=2;
	  if (paren_count==0) {
	    if (i < end) {
	      return i;
	    } else {
	      return 0;
	    }
	  } else {
	    prev_separator=1;
	    non_whitespace_p=1;
	    quote_wait=0;
	  }
	  /* MS:FIXME:: Handle #\) properly! */
	} else {
	  prev_separator=0;
	  quote_wait=0;
	  non_whitespace_p=1;
	  i++;
	}
      }
      break;
    case '(' :
      if (non_whitespace_p && paren_count==0 &&!quote_wait) {
	return i;
      } else {
	i++;
	paren_count++;
	non_whitespace_p=1;
	prev_separator=1;
	quote_wait=0;
      }
      break;
    case ')' :
      paren_count--;
      if (non_whitespace_p && paren_count==0) {
	return i+1;
      } else {
	i++;
	non_whitespace_p=1;
	prev_separator=1;
	quote_wait=0;
      }
      break;
    case '\'' :
      if (prev_separator) {
	non_whitespace_p=1;
	quote_wait=1;
	prev_separator=1;
	i++;
      } else {
	non_whitespace_p=1;
	prev_separator=0;
	i++;
      }
      break;
    default :
      prev_separator=0;
      quote_wait=0;
      non_whitespace_p=1;
      i++;
      break;
    }
  }
  return 0;
}

#include <X11/cursorfont.h>
static Cursor wait_cursor;

static void Command_Return_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  /* try to find complete form either enclosing current cursor, or just before it */
  XmTextPosition new_eot,cmd_eot;
  char *str = NULL,*full_str = NULL,*prompt;
  int i,j,slen;
  snd_state *ss = (snd_state *)clientData;
  int end_of_text,start_of_text,last_position,current_position,parens;
  full_str = XmTextGetString(w);
  current_position = XmTextGetInsertionPosition(w);
  start_of_text = current_position;
  end_of_text = current_position;
  last_position = XmTextGetLastPosition(w);
  prompt = listener_prompt(ss);
  if (last_position > end_of_text)
    {
      for (i=current_position;i<last_position;i++)
	if ((full_str[i+1] == prompt[0]) && (full_str[i] == '\n'))
	  {
	    end_of_text = i-1;
	    break;
	  }
    }
  if (start_of_text > 0)
    {
      for (i=current_position;i>=0;i--)
	if ((full_str[i] == prompt[0]) && ((i == 0) || (full_str[i-1] == '\n')))
	  {
	    start_of_text = i+1;
	    break;
	  }
    }
  str = NULL;
  if (end_of_text > start_of_text)
    {
      parens = 0;
      slen = end_of_text - start_of_text + 2;
      str = (char *)CALLOC(slen,sizeof(char));
      for (i=start_of_text,j=0;i<=end_of_text;j++,i++) {str[j] = full_str[i]; if (str[j] == '(') parens++;}
      str[end_of_text-start_of_text+1] = 0;
      end_of_text = snd_strlen(str);
      if (parens)
	{
	  end_of_text = check_balance(str,0,end_of_text);
	  if ((end_of_text > 0) && (end_of_text < (slen-1)))
	    {
	      str[end_of_text+1] = 0;
	      if (str[end_of_text] == '\n') str[end_of_text]=0;
	    }
	  else
	    {
	      FREE(str);
	      str = NULL;
	      new_eot = XmTextGetLastPosition(w);
	      XmTextInsert(w,new_eot,"\n");
	      return;
	    }
	}
      if (str)
	{
	  if (current_position < (last_position-2))
	    {
	      XmTextInsert(lisp_listener,XmTextGetLastPosition(lisp_listener),str);
	    }
	  XDefineCursor(XtDisplay(lisp_listener),XtWindow(lisp_listener),wait_cursor);
	  XmUpdateDisplay(lisp_listener); /* not sure about this... */
	  clm_just_doit(ss,str);
	  XUndefineCursor(XtDisplay(lisp_listener),XtWindow(lisp_listener));
	  FREE(str);
	  str = NULL;
	}
      else
	{
	  new_eot = XmTextGetLastPosition(w);
	  XmTextInsert(w,new_eot,listener_prompt_with_cr(ss));
	}
      last_prompt = XmTextGetLastPosition(w) - 1;
    }
  else 
    {
      new_eot = XmTextGetLastPosition(w);
      XmTextInsert(w,new_eot,"\n");
    }
  cmd_eot = XmTextGetLastPosition(w);
  XmTextShowPosition(w,cmd_eot-1);
  XmTextSetInsertionPosition(w,cmd_eot+1);
  if (full_str) FREE(full_str);
}

static int last_highlight_position = -1;

static void Command_Motion_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  XmTextVerifyCallbackStruct *cbs = (XmTextVerifyCallbackStruct *)callData;
  snd_state *ss = (snd_state *)clientData;
  char *str=NULL,*prompt;
  int pos,i,parens;
  cbs->doit = TRUE; 
  if (last_highlight_position != -1)
    {
      XmTextSetHighlight(w,last_highlight_position,last_highlight_position+1,XmHIGHLIGHT_NORMAL);
      last_highlight_position = -1;
    }
  pos = cbs->newInsert - 1;
  if (pos > 0)
    {
      prompt = listener_prompt(ss);
      str = XmTextGetString(w);
      /* this can be confused by comments, quoted parens, and string constants */
      if (str[pos] == ')')
	{
	  parens = 1;
	  for (i=pos-1;i>0;i--)
	    {
	      if ((i>0) && (str[i] == prompt[0]) && (str[i-1] == '\n'))
		break;
	      if (str[i] == ')') parens++;
	      if (str[i] == '(') parens--;
	      if (parens == 0)
		{
		  XmTextSetHighlight(w,i,i+1,XmHIGHLIGHT_SECONDARY_SELECTED);
		  last_highlight_position = i;
		  break;
		}
	    }
	}
      if (str) FREE(str);
    }
}

static void Command_Modify_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
  XmTextVerifyCallbackStruct *cbs = (XmTextVerifyCallbackStruct *)callData;
  snd_state *ss = (snd_state *)clientData;
  char *str=NULL,*prompt;
  int len;
  if ((cbs->text)->length > 0)
    cbs->doit = TRUE;
  else
    { 
      if (cbs->currInsert < 2) 
	cbs->doit = FALSE;
      else
	{
	  prompt = listener_prompt(ss);
	  str = XmTextGetString(w);
	  len = XmTextGetLastPosition(w);
	  if ( ((str[cbs->currInsert-1] == prompt[0]) && (str[cbs->currInsert - 2] == '\n')) ||
	       ((cbs->currInsert < (len-1)) && (str[cbs->currInsert] == prompt[0]) && (str[cbs->currInsert-1] == '\n')))
	    cbs->doit = FALSE;
	  else cbs->doit = TRUE;
#if 0
	  snd_error("insert: %d (%c %c %c), len: %d\n",
		  cbs->currInsert,
		  (cbs->currInsert > 0) ? str[cbs->currInsert - 1] : '?',
		  str[cbs->currInsert],
		  (cbs->currInsert < len) ? str[cbs->currInsert + 1] : '?',
		  (cbs->text)->length);
#endif	  
	  if (str) FREE(str);
	}
    }
}

static void Command_Help_Callback(Widget w,XtPointer clientData,XtPointer callData)
{
#if HAVE_XmHTML
  snd_help((snd_state *)clientData,"Lisp Listener","#customization");
#else
  ssnd_help((snd_state *)clientData,
	    "Lisp Listener",
"This is the lisp listener pane; it is one way to\n\
access the Guile Scheme interpreter.\n\
\n",
	   get_init_file_help(),
	   NULL);
#endif
}

/* static Widget pane_frm = NULL; */

static void sndCreateCommandWidget(snd_state *ss, int height)
{
  Arg args[32];
  Widget wv,wh;
  int n;
  if (!lisp_listener)
    {
      if (!actions_loaded) {XtAppAddActions(main_APP(ss),acts,NUM_ACTS); actions_loaded = 1; action_ss = ss;}

      n=0;
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNheight,height); n++;
      pane_frm = sndCreateFormWidget("frm",sound_PANE(ss),args,n);
      /* this widget is not redundant at least in Metroworks Motif */

      n=0;
      if (!(ss->using_schemes)) 
	{
	  XtSetArg(args[n],XmNbackground,(ss->sgx)->listener_color); n++;
	}
      if ((ss->sgx)->listener_fontlist) {XtSetArg(args[n],XmNfontList,(ss->sgx)->listener_fontlist); n++;}
      XtSetArg(args[n],XmNtopAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNbottomAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNleftAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNrightAttachment,XmATTACH_FORM); n++;
      XtSetArg(args[n],XmNactivateCallback,make_callback_list(remember_event,(XtPointer)ss)); n++;
      XtSetArg(args[n],XmNeditMode,XmMULTI_LINE_EDIT); n++;
      XtSetArg(args[n],XmNskipAdjust,TRUE); n++;
      XtSetArg(args[n],XmNvalue,listener_prompt(ss)); n++;
      XtSetArg(args[n],XmNpendingDelete,FALSE); n++; /* don't cut selection upon paste */
      XtSetArg(args[n],XmNpositionIndex,XmLAST_POSITION); n++;
      XtSetArg(args[n],XmNpaneMinimum,height); n++;

      lisp_listener = XmCreateScrolledText(pane_frm,"lisp-listener",args,n);
#if 0
      {
	XtActionList lst;
	XtActionsRec *ls;
	Cardinal num,i;
	XtGetActionList(xmFormWidgetClass,&lst,&num);
	ls = lst;
	for (i=0;i<num;i++) snd_error("%s\n",ls[i].string);
      }
#endif

      XtManageChild(lisp_listener);
      XmTextSetCursorPosition(lisp_listener,1);
      if (!transTable4) transTable4 = XtParseTranslationTable(TextTrans4);
      XtOverrideTranslations(lisp_listener,transTable4);
      last_prompt = 0;
      XtAddCallback(lisp_listener,XmNactivateCallback,Command_Return_Callback,ss);
      XtAddCallback(lisp_listener,XmNmodifyVerifyCallback,Command_Modify_Callback,ss);
      XtAddCallback(lisp_listener,XmNmotionVerifyCallback,Command_Motion_Callback,ss);
      XtAddCallback(lisp_listener,XmNhelpCallback,Command_Help_Callback,ss);
      
      lisp_window = XtParent(lisp_listener);
      
      if (!(ss->using_schemes))
	{
	  XmChangeColor(lisp_window,(ss->sgx)->basic_color);
	  XtVaGetValues(lisp_window,XmNverticalScrollBar,&wv,XmNhorizontalScrollBar,&wh,NULL);
	  XmChangeColor(wv,(ss->sgx)->basic_color);
	  XmChangeColor(wh,(ss->sgx)->basic_color);
	  map_over_children(sound_PANE(ss),color_sashes,(void *)ss);
	}
      XtVaSetValues(lisp_window,XmNpaneMinimum,1,NULL);

#if (XmVERSION > 1)
      if (sound_style(ss) == SOUNDS_IN_NOTEBOOK)
	{
	  Widget tab;
	  n = 0;
	  if (!(ss->using_schemes)) {XtSetArg(args[n],XmNbackground,(ss->sgx)->lighter_blue); n++;}
	  tab = XtCreateManagedWidget("lisp",xmPushButtonWidgetClass,sound_PANE(ss),args,n);
	}
#endif

      wait_cursor = XCreateFontCursor(XtDisplay(lisp_listener),XC_watch);
    }
}

void goto_listener(void) 
{
  goto_window(lisp_listener);
  XmTextSetCursorPosition(lisp_listener,XmTextGetLastPosition(lisp_listener)+1);
  XmTextSetInsertionPosition(lisp_listener,XmTextGetLastPosition(lisp_listener)+1);
}

void color_listener(Pixel pix)
{
  snd_state *ss;
  ss = get_global_state();
  (ss->sgx)->listener_color = pix;
  if (lisp_listener)
    XmChangeColor(lisp_listener,pix);
}

void handle_listener(snd_state *ss, int new_state)
{
  if (!lisp_listener)
    {
      /* fire up listener at bottom of overall snd window */
      if (new_state == LISTENER_OPEN) 
	{
	  sndCreateCommandWidget(ss,100);
	  set_view_listener_label(STR_Hide_listener);
	  goto_window(lisp_listener);
	}
      else sndCreateCommandWidget(ss,1);
      ss->listening = new_state;
    }
  else
    {
      if (ss->listening == LISTENER_OPEN)
	{
	  /* close listener window but it remains active */
	  ss->listening = LISTENER_LISTENING;
	  set_view_listener_label(STR_Show_listener);
	  XtUnmanageChild(pane_frm);
	  XtVaSetValues(pane_frm,XmNpaneMaximum,1,XmNpaneMinimum,1,NULL);
	  XtManageChild(pane_frm);
	  XtVaSetValues(pane_frm,XmNpaneMaximum,1000,XmNpaneMinimum,1,NULL);
	}
      else
	{
	  /* reopen listener pane */
	  ss->listening = LISTENER_OPEN;
	  set_view_listener_label(STR_Hide_listener);
	  XtUnmanageChild(pane_frm);
	  XtVaSetValues(pane_frm,XmNpaneMinimum,100,NULL);
	  XtManageChild(pane_frm);
	  XtVaSetValues(pane_frm,XmNpaneMinimum,1,NULL);
	}
    }
}

int lisp_listener_height(void)
{
  Dimension hgt;
  if (lisp_listener)
    {
      XtVaGetValues(lisp_listener,XmNheight,&hgt,NULL);
      return(hgt);
    }
  return(0);
}

void set_listener_width(int width)
{
  if (lisp_listener)
    {
      XtUnmanageChild(pane_frm);
      XtVaSetValues(pane_frm,XmNwidth,width,NULL);
      XtManageChild(pane_frm);
    }
}

#if OVERRIDE_TOGGLE
/* Metrolink Motif defines control-button1 to be "take focus" and then segfaults if you use it!! */
static char ToggleTrans2[] =
       "c<Btn1Down>:   ArmAndActivate()\n";
static XtTranslations toggleTable2 = NULL;

static void override_toggle_translation(Widget w)
{
  if (!toggleTable2) toggleTable2 = XtParseTranslationTable(ToggleTrans2);
  XtOverrideTranslations(w,toggleTable2);
}

static char ToggleTrans3[] =
       "c<Btn1Down>:   DrawingAreaInput()\n";
static XtTranslations toggleTable3 = NULL;

static void override_drawing_translation(Widget w)
{
  if (!toggleTable3) toggleTable3 = XtParseTranslationTable(ToggleTrans3);
  XtOverrideTranslations(w,toggleTable3);
}

static char ToggleTrans4[] =
       "c<Btn1Down>:   Activate()\n";
static XtTranslations toggleTable4 = NULL;

static void override_manager_translation(Widget w)
{
  if (!toggleTable4) toggleTable4 = XtParseTranslationTable(ToggleTrans4);
  XtOverrideTranslations(w,toggleTable4);
}

static char ToggleTrans5[] =
       "c<Btn1Down>:   Return()\n";
static XtTranslations toggleTable5 = NULL;

void override_form_translation(Widget w)
{
  if (!toggleTable5) toggleTable5 = XtParseTranslationTable(ToggleTrans5);
  XtOverrideTranslations(w,toggleTable5);
}

/* push buttons also need the override, but they aren't causing Snd to crash for some reason */

#endif

Widget sndCreateFormWidget(char *name, Widget parent, Arg *args, int n)
{
  Widget w;
  w = XtCreateManagedWidget(name,xmFormWidgetClass,parent,args,n);
#if OVERRIDE_TOGGLE
  override_form_translation(w);
#endif
  return(w);
}

Widget sndCreateToggleButtonWidget(char *name, Widget parent, Arg *args, int n)
{
  Widget w;
  w = XtCreateManagedWidget(name,xmToggleButtonWidgetClass,parent,args,n);
#if OVERRIDE_TOGGLE
  override_toggle_translation(w);
#endif
  return(w);
}

Widget sndCreatePushButtonWidget(char *name, Widget parent, Arg *args, int n)
{
  Widget w;
  w = XtCreateManagedWidget(name,xmPushButtonWidgetClass,parent,args,n);
#if OVERRIDE_TOGGLE
  override_toggle_translation(w);
#endif
  return(w);
}

Widget sndCreateFrameWidget(char *name, Widget parent, Arg *args, int n)
{
  Widget w;
  w = XtCreateManagedWidget(name,xmFrameWidgetClass,parent,args,n);
#if OVERRIDE_TOGGLE
  override_manager_translation(w);
#endif
  return(w);
}

Widget sndCreateRowColumnWidget(char *name, Widget parent, Arg *args, int n)
{
  Widget w;
  w = XtCreateManagedWidget(name,xmRowColumnWidgetClass,parent,args,n);
#if OVERRIDE_TOGGLE
  override_manager_translation(w);
#endif
  return(w);
}

Widget sndCreateDrawingAreaWidget(char *name, Widget parent, Arg *args, int n)
{
  Widget w;
  w = XtCreateManagedWidget(name,xmDrawingAreaWidgetClass,parent,args,n);
#if OVERRIDE_TOGGLE
  override_drawing_translation(w);
#endif
  return(w);
}

Widget sndCreatePanedWindowWidget(char *name, Widget parent, Arg *args, int n)
{
  Widget w;
  w = XtCreateManagedWidget(name,xmPanedWindowWidgetClass,parent,args,n);
#if OVERRIDE_TOGGLE
  override_manager_translation(w);
#endif
  return(w);
}

char *key_to_name(int keysym) {if (keysym > 0) return(XKeysymToString(keysym)); else return("NUL");}
/* on the Sun, if keysym is 0, XKeysymToString segfaults! */

void color_cursor(snd_state *ss, Pixel color)
{
  state_context *sx;
  sx = ss->sgx;
  sx->cursor_color = color;
  XSetForeground(main_DISPLAY(ss),sx->cursor_gc,(Pixel)(XOR(color,sx->graph_color)));
  XSetForeground(main_DISPLAY(ss),sx->selected_cursor_gc,(Pixel)(XOR(color,sx->selected_graph_color)));
}

void color_marks(snd_state *ss, Pixel color)
{
  state_context *sx;
  sx = ss->sgx;
  sx->mark_color = color;
  XSetForeground(main_DISPLAY(ss),sx->mark_gc,(Pixel)(XOR(color,sx->graph_color)));
  XSetForeground(main_DISPLAY(ss),sx->selected_mark_gc,(Pixel)(XOR(color,sx->selected_graph_color)));
}

void color_selection(snd_state *ss, Pixel color)
{
  state_context *sx;
  sx = ss->sgx;
  sx->selection_color = color;
  XSetForeground(main_DISPLAY(ss),sx->selection_gc,(Pixel)(XOR(color,sx->graph_color)));
  XSetForeground(main_DISPLAY(ss),sx->selected_selection_gc,(Pixel)(XOR(color,sx->selected_graph_color)));
}

void color_graph(snd_state *ss, Pixel color)
{
  Display *dpy;
  state_context *sx;
  dpy = main_DISPLAY(ss);
  sx = ss->sgx;
  sx->graph_color = color;
  XSetBackground(dpy,sx->basic_gc,color);
  XSetForeground(dpy,sx->erase_gc,color);
  XSetForeground(dpy,sx->selection_gc,(Pixel)(XOR(sx->selection_color,color)));
  XSetForeground(dpy,sx->cursor_gc,(Pixel)(XOR(sx->cursor_color,color)));
  XSetForeground(dpy,sx->mark_gc,(Pixel)(XOR(sx->mark_color,color)));
}

void color_selected_graph(snd_state *ss, Pixel color)
{
  Display *dpy;
  state_context *sx;
  dpy = main_DISPLAY(ss);
  sx = ss->sgx;
  sx->selected_graph_color = color;
  XSetBackground(dpy,sx->selected_basic_gc,color);
  XSetForeground(dpy,sx->selected_erase_gc,color);
  XSetForeground(dpy,sx->selected_selection_gc,(Pixel)(XOR(sx->selection_color,color)));
  XSetForeground(dpy,sx->selected_cursor_gc,(Pixel)(XOR(sx->cursor_color,color)));
  XSetForeground(dpy,sx->selected_mark_gc,(Pixel)(XOR(sx->mark_color,color)));
}

void color_data(snd_state *ss, Pixel color)
{
  Display *dpy;
  state_context *sx;
  dpy = main_DISPLAY(ss);
  sx = ss->sgx;
  sx->data_color = color;
  XSetForeground(dpy,sx->basic_gc,color);
  XSetBackground(dpy,sx->erase_gc,color);
}

void color_mix_waveform(snd_state *ss, Pixel color)
{
  Display *dpy;
  state_context *sx;
  dpy = main_DISPLAY(ss);
  sx = ss->sgx;
  sx->mix_waveform_color = color;
  XSetForeground(dpy,sx->mix_gc,color);
}

void color_selected_data(snd_state *ss, Pixel color)
{
  Display *dpy;
  state_context *sx;
  dpy = main_DISPLAY(ss);
  sx = ss->sgx;
  sx->selected_data_color = color;
  XSetForeground(dpy,sx->selected_basic_gc,color);
  XSetBackground(dpy,sx->selected_erase_gc,color);
}

void recolor_graph(chan_info *cp, int selected)
{
  snd_state *ss;
  state_context *sx;
  ss = cp->state;
  sx = ss->sgx;
  XtVaSetValues(channel_graph(cp),XmNbackground,(selected) ? sx->selected_graph_color : sx->graph_color,NULL);
}
