// pvanalyzer.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "envelope.h"
#include "localdefs.h"
#include "query.h"
#include "request.h"
#include "pvanalyzer.h"
#include "pvocdata.h"
#include "pvocrequester.h"

class PVAnalysisRequester : public PvocRequester {
	friend PVAnalyzer;
protected:
	PVAnalysisRequester(const char* title, PhaseVocoder::Info &);
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	enum FFTSize {
		F_64 = 0x1, F_128 = 0x2, F_256 = 0x4, F_512 = 0x8, F_1024 = 0x10,
		F_2048 = 0x20, F_4096 = 0x40, F_8192 = 0x80, F_16384 = 0x100
	};
	enum OverLap { X4 = 0x1, X2 = 0x2, X1 = 0x4, Xp5 = 0x8 };
	double frate;
	ChoiceValue fftSize;
	ChoiceValue overlapFactor;
};

PVAnalysisRequester::PVAnalysisRequester(const char* title,
                                         PhaseVocoder::Info &info)
	: PvocRequester(title, info),
	  frate(0.0), fftSize(F_256), overlapFactor(X1) {}

void
PVAnalysisRequester::configureRequest(Request* request) {
	request->appendLabel("All \"0\" values will be set to defaults.");
	request->appendValue("Input frame size (samples):",
			     &pvocInfo.inputFrameSize,
			     PositiveIntegers);
	request->appendValue("Input frame offset (samples):",
			     &pvocInfo.inputFrameOffset,
			     PositiveIntegers);
	request->appendValue("Input frame rate (Hz):", &frate,
			     PositiveIntegers);
	request->appendValue("Time Scaling factor:",
			     &pvocInfo.timeScaleFactor,
			     PositiveNumbers);
	request->appendChoice("FFT Size:",
	        "|64|128|256|512|1024|2048|4096|8192|16384|", &fftSize, true);
	request->appendChoice("Filter Overlap Factor (x fftsize):",
	                      "|4x|2x|1x|x/2|", &overlapFactor, true);
	request->appendChoice("Window Type:", "|Hanning|Kaiser|",
	                      &windowType, true);
}

boolean
PVAnalysisRequester::confirmValues() {
	if(frate > 0.0)		// if user entered frame rate, reset frame offset
		pvocInfo.inputFrameOffset = (int(pvocInfo.samplingRate/frate));
	int fftpoints = fftSize << 6; // {1, 2, 4, ..} => {64, 128, 256, ...}
	int framesize = pvocInfo.inputFrameSize;
	if(framesize == 0)            // if nonzero, no need to use overlap factor
		switch(overlapFactor) {
			case X4:  framesize = fftpoints * 4;  break;
			case X2:  framesize = fftpoints * 2;  break;
			case X1:  framesize = fftpoints;  break;
			case Xp5: framesize = fftpoints/2;  break;
			default:  break;
		}
	pvocInfo.inputFrameSize = framesize;
	pvocInfo.fftSize = fftpoints;
	pvocInfo.K = (windowType == Kaiser);	// flag for Kaiser windowing
	return true;
}

//********

PVAnalyzer::PVAnalyzer(Data* data) : ArrayFunction(data),
	pvocInfo(
		data->sRate(),
		(data->dataType() == FloatData) ? 1.0 : 1/32767.0,
		PhaseVocoder::Analysis
	),
	pvoc(nil) {}

PVAnalyzer::PVAnalyzer(
		Data* data,
		int N, int F, int W, int M, int D, double T, double P,
		PhaseVocoder::Mode mode, boolean useKaiser)
			: ArrayFunction(data, M, D),
			  pvocInfo(data->sRate(),
		           (data->dataType()==FloatData) ? 1.0 : 1/32767.0,
		           mode,
		           N, F, W, M, D, T, P,
		           useKaiser
		 	 ),		
		 	 pvoc(nil) {
	initialize();
}

PVAnalyzer::~PVAnalyzer() {
	delete pvoc;
}

Requester *
PVAnalyzer::createRequester() {
	return new PVAnalysisRequester(
		"Phase Vocoder Analysis of Selected Region:",
		pvocInfo
	);	
}

void
PVAnalyzer::initialize() {		
	pvoc = new PhaseVocoder(pvocInfo);
	if(pvoc->isGood()) {
		setBaseValues(
			pvoc->roundedInputFrameSize(), 0.0, pvoc->getInputFrameOffset()
		);
		Super::initialize();
		setAnalysis(
			new PvocData(analysisLength(), pvoc->analysisChannels(),
			sampRate(), framerate())
		);
	}
}

int
PVAnalyzer::operator () (double* in, Data* frame) {
	int status = pvoc->runAnalysis(in, frame);
	setOffset(pvoc->getInputFrameOffset());		// update values
	return status;
}

int
PVAnalyzer::initialOffset() { return pvoc->getStartingOffset(); };

int
PVAnalyzer::analysisLength() {
	return pvoc->calculateAnalysisLength(target()->length());
}
