// graphplot.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <InterViews/color.h>
#include <InterViews/painter.h>
#include <InterViews/pattern.h>
#include <InterViews/perspective.h>
#include <InterViews/display.h>
#include <InterViews/world.h>
#include <IV-X11/xdisplay.h>
#include "data.h"
#include "graph.h"
#include "graphplot.h"
#include "localdefs.h"

SimplePlot::SimplePlot(Graph *gr, int points) : GraphPlot(gr, points),
		maxTraces(MaxTraces), nTraces(0), nPoints(0) {
	for(int i = 0; i < maxTraces; i++) {
		xlocs[i] = new Coord[maxPoints];
		ylocs[i] = new Coord[maxPoints];
	}
	maxVertLoc = minVertLoc = 0;
	plotdata = new Data(FloatData, defaultGraphWidth, maxTraces);
	plotdata->ref();
}

SimplePlot::~SimplePlot() {
	Resource::unref(plotdata);
	for(int i = 0; i < maxTraces; i++) {
		delete [] xlocs[i];
		delete [] ylocs[i];
	}
}

void
SimplePlot::load(Data *visible, EnvelopeType etype) {
	nTraces = visible->getEnvelope(plotdata, 0, etype);
}

void
SimplePlot::setPlotLength(int newlen) {
	if(newlen != plotdata->length())
		plotdata->changeLength(newlen);
}

void
SimplePlot::plot() {
	int width = graph->currentWidth();		// number of pixels in graph
	double plotGrain = max(graph->currentHGrain(), 1.0);
	int vlen = plotdata->length();
	// until Coords are floats, this factor must be figured in here
	double scalefactor = graph->vertScaleFactor();
	nPoints = 0;
	for(int i=0; i < vlen && i < maxPoints; i++) {
		int loc;
		if((loc = round(i*plotGrain)) > width)
			break;
		for(int j = 0; j < currentTraces(); j++) {
			int y = round(plotdata->get(i, j) * scalefactor);
			maxVertLoc = (y > maxVertLoc) ? y : maxVertLoc;
			minVertLoc = (y < minVertLoc) ? y : minVertLoc;
			ylocs[j][i] = y;
			xlocs[j][i] = loc;
		}
		nPoints++;
	}
}

boolean
SimplePlot::contains(Coord left, Coord bottom, Coord right, Coord top) {
	return top >= minVertLoc && bottom <= maxVertLoc;
}

void
LinePlot::draw(Canvas *cnv, Painter *out) {
	for(int i = 0; i < currentTraces(); i++)
		out->MultiLine(cnv, xlocs[i], ylocs[i], nPoints);
}

void
BarPlot::plot() {
	SimplePlot::plot();
	int horizOffset = xlocs[0][1] - xlocs[0][0];
	if(nPoints < maxPoints)	// add terminating value for last bar
		xlocs[0][nPoints] = xlocs[0][nPoints-1] + horizOffset;
}

inline boolean trueMax(int x, int y) {
	return (abs(x) > abs(y)) ? x : y;
}

inline boolean trueMin(int x, int y) {
	return (abs(x) < abs(y)) ? x : y;
}

void
BarPlot::draw(Canvas *cnv, Painter *out) {
	for(int i = 0; i < nPoints; i++) {
		Coord x0 = xlocs[0][i];
		Coord x1 = xlocs[0][i+1] - 1;
		Coord ytop = ylocs[0][i];
		Coord y0, y1;
		if(currentTraces() > 1) {
			Coord ybottom = ylocs[1][i];
			y0 = trueMax(ytop, ybottom);		// larger of two, ignoring sign
			y1 = (sign((long)ytop) != sign((long)ybottom)) ?
				((y0 == ytop) ? ybottom : ytop) : 0;	// the other one, or 0
		}
		else {
			y0 = ytop;
			y1 = 0;
		}
		out->FillRect(cnv, x0, y0, x1, y1);
	}
}

void
SolidPlot::plot() {
	SimplePlot::plot();
	for(int i = 0; i < currentTraces(); i++) {
		// add the end points for the base of the polygon
		xlocs[i][nPoints] = xlocs[i][nPoints-1];
		ylocs[i][nPoints] = 0;
		xlocs[i][nPoints+1] = xlocs[i][0];
		ylocs[i][nPoints+1] = 0;
	}	
	nPoints += 2;
}

// Draws a filled polygon and then borders it on top and sides with a line.
// To avoid drawing a line along the base, I leave out the final base segment

void
SolidPlot::draw(Canvas *cnv, Painter *out) {
	DisplayRep* rep = graph->GetWorld()->display()->rep();
	int serverDepth = XDefaultDepth(rep->display_, rep->screen_);
	int nSegments = nPoints - 1;
//	for(int i = 0; i < ntraces; i++) {
//  since max val trace hides min val trace, only draw former -- FIX ME
	int i = (currentTraces() > 1) ? 1 : 0;
		if(serverDepth > 1) {
			ColorIntensity gr = 0.5;
			out->SetColors(new Color(gr, gr, gr), nil);		// gray
		}
		else
			out->SetPattern(new Pattern(0x4141));
		out->FillPolygon(cnv, xlocs[i], ylocs[i], nPoints);		// interior
		if(serverDepth > 1) {
			ColorIntensity bl = 0;
			out->SetColors(new Color(bl, bl, bl), nil);			// black
		}
		else
			out->SetPattern(new Pattern(Pattern::solid));
		out->MultiLine(cnv, xlocs[i], ylocs[i], nSegments);		// border
		out->Line(cnv, xlocs[i][0], ylocs[i][0],
			xlocs[i][nSegments], ylocs[i][nSegments]);			// left edge
//	}
}
