// formantsynth.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <math.h>

#include "application.h"
#include "data.h"
#include "datamodifier.h"
#include "formantsynth.h"
#include "lpcdata.h"
#include "lpcfilter.h"
#include "formantrequester.h"
#include "synthfunction.h"

class FormantSynthRequester : public FormantRequester {
	friend FormantSynthesizer;
protected:
	FormantSynthRequester(const char* title, FormantSynthesizer *);
	redefined void configureRequest(Request *);
private:
	FormantSynthesizer* client;	
};

FormantSynthRequester::FormantSynthRequester(
			const char* title, FormantSynthesizer* fs)
	: FormantRequester(title, fs, fs->sampRate(), fs->gain,
	                   fs->warpFactor, fs->interpMode), client(fs) {}
		                 
void
FormantSynthRequester::configureRequest(Request* request) {
	request->appendValue("Unvoiced threshold:", &client->hiThreshold,
			    Range(0.0, 1.0), true);
	request->appendValue("Voiced threshold:", &client->lowThreshold,
	                    Range(0.0, 1.0), true);
	request->appendValue("Voiced/Unvoiced amp factor:",
	                     &client->voicedFactor,
			     Range(0.0, 10.0), true);
	FormantRequester::configureRequest(request);
}

//********

static unsigned seed = 3;

FormantSynthesizer::FormantSynthesizer(Data* output, Data* lpcdata,
	double freq, int nHarms, double amp)
		: LPC_Filter(lpcdata, output->length(), false),
		  Buzz(output, freq, nHarms, amp),
		  noise(seed), gain(1.0), voicedFactor(3.0),
		  hiThreshold(1.0), lowThreshold(0.0), warpFactor(0.0),
		  interpMode(Linear) {
    okIf(LPC_Filter::good());
}

FormantSynthesizer::FormantSynthesizer(Data* output, Data* lpcdata)
		: LPC_Filter(lpcdata, output->length(), false), Buzz(output),
		  noise(seed), gain(1.0), voicedFactor(3.0),
		  hiThreshold(1.0), lowThreshold(0.0), warpFactor(0.0),
		  interpMode(Linear) {
    okIf(LPC_Filter::good());
}

Requester *
FormantSynthesizer::createRequester() {
	return new FormantSynthRequester(
		"Resynthesize LPC Data into Selected Region:", this);
}

void
FormantSynthesizer::initialize() {
	if(warpFactor != 0.0) {
		Data* warpData = new Data(FloatData, getLPC()->length() + 1);
		warpData->ref();
		Offset o(warpData, warpFactor);	// set to single value
		o.apply();
		setWarp(warpData);
		Resource::unref(warpData);
	}
	hiThreshold = sqrt(hiThreshold);
	lowThreshold = sqrt(lowThreshold);
	setInterpMode(InterpMode(interpMode));
	LPC_Filter::initialize();
	Buzz::initialize();
	srOverTwo = sampRate()/2.0;
	restoreState();
}

void
FormantSynthesizer::restoreState() {
	LPC_Filter::reset();
	Buzz::restoreState();
	noise.setSeed(seed);
}

double
FormantSynthesizer::doSynthesis() {
	RunStatus status;
	float amp = 0.0;
	float error = 0.0;
	float pitch = 0.0;
	float* coeffs = getFrame(&amp, &error, &pitch);
	int nharm = int(srOverTwo/pitch) - 2;
	double sqerr = sqrt(error);
	double noiseAmp = (sqerr - lowThreshold) / (hiThreshold - lowThreshold);
	noiseAmp = max(0.0, min(1.0, noiseAmp));	// 0 <= noiseAmp <= 1.0
	double voicedAmp = 1.0 - noiseAmp;			// complement of that
	set(pitch, nharm, voicedAmp * voicedFactor);
	noise.setGain(noiseAmp);
	double signal = unvoiced() + voiced();
	return filter(coeffs, amp * signal * gain);
}
