// diskstruct.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author 	
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

// DiskStruct is the base class for all classes which read a "typed stream" of
// values from an on-disk file.  It is designed to allow reading and writing
// of all sizes of datatypes to and from files in such a way that the
// differences in in-memory struct layouts do not affect the on-disk layout
// Also, it takes into account cases of byte-swapped information.

#ifndef DISKSTRUCT_H
#ifdef __GNUG__
#pragma interface
#endif
#define DISKSTRUCT_H

#include "id.h"
#include "localdefs.h"

class DataFile;
class ValueSetterBase;

class DiskStruct {
public:
	DiskStruct() {}
	virtual ~DiskStruct() {}
	virtual int read(DataFile* file) { return doRead(file, valueSetters()); }
	virtual int write(DataFile* file) { return doWrite(file, valueSetters()); }
	virtual long readSize() { return sizeof(*this); }
protected:
	virtual int doRead(DataFile*, ValueSetterBase**);
	virtual int doWrite(DataFile*, ValueSetterBase**);
	virtual ValueSetterBase** valueSetters()=0;
};

// many standard file formats use an identifier chunk consisting of a 4-char
// identification string followed by a long indicating the size of what follows

class StandardChunk : public DiskStruct {
	typedef DiskStruct Super;
public:
	StandardChunk(const char* idString="", long len=0);
	virtual boolean isValid() { return true; }
	boolean isA(const char* idstr) { return (magic == idstr); }
	long size() { return chunkSize; }
	redefined long readSize() { return sizeof(*this) - Super::readSize(); }
protected:
	redefined ValueSetterBase** valueSetters();
	enum { NumElements = 2 };
private:
	Id magic;
	unsigned long chunkSize;		// size in bytes of information to follow
};

// an expansion of DiskStruct for handling variable-length on-disk data where
// the additional data is an undifferentiated char[] block

class VariableBlockDiskStruct : public DiskStruct {
	typedef DiskStruct Super;
public:
	redefined int read(DataFile *);
	redefined int write(DataFile *);
	redefined long readSize() { return sizeof(*this); }
protected:
	virtual void createVariableBlock()=0;
	ValueSetterBase** blockValueSetters();
private:
	virtual char* variableBlockAddress()=0;
	virtual long variableBlockLength()=0;
};

#endif
