// datamodifier.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// DataModifier is the child class of Modifier which functions as the base
// class for all classes which perform some type of modification or processing
// upon a Data object.  The public method Modifier::apply() is redefined here
// to call DataModifier::doApply(Data *), which is then redefined in all 
// subclasses of DataModifier.

#ifndef DATAMODIFIER_H
#ifdef __GNUG__
#pragma interface
#endif
#define DATAMODIFIER_H

#include "modifier.h"
#include "range.h"

class Data;
class DataEditor;

class DataModifier : public Modifier {
public:
	DataModifier(Data* data);
	DataModifier(Data* data, DataEditor* editor);
	virtual ~DataModifier();
	redefined int apply();
protected:
	virtual int doApply(Data *)=0;
	void setEditor(DataEditor* ed) { myEditor = ed; }
	void doPostModifyCommands();
	virtual void doEditorCommands(DataEditor *) {}
	Data* target() { return myData; }
	int sampRate();
private:
	Data* myData;			// pointer to the data to be modified
	DataEditor* myEditor;
};

class Reverser : public DataModifier {
	typedef DataModifier Super;
public:
	Reverser(Data* d) : DataModifier(d) { initialize(); }
	static Modifier* create(DataEditor *);
	redefined const char* message() { return "Reversing..."; }
	redefined Modifier* createUnDo();
protected:
	redefined int doApply(Data *);
};

class Normalizer : public DataModifier {
	typedef DataModifier Super;
	friend class NormalizeRequester;
public:
	Normalizer(Data* d) : DataModifier(d), reverseFactor(1.0) {}
	static Modifier* create(DataEditor *);
	redefined const char* message() { return "Normalizing..."; }
	redefined Modifier* createUnDo();
protected:
	redefined Requester* createRequester();
	redefined int doApply(Data *);
private:
	double reverseFactor;		// for undo
};

// the next 7 subclasses operate upon the entire data object, rather than
// on just a section of it.

class OutSplicer : public DataModifier {
	typedef DataModifier Super;
public:
	OutSplicer(Data *, const Range &);
	OutSplicer(Data* d) : DataModifier(d) {}
	static Modifier* create(DataEditor *);
	redefined Modifier* createUnDo();
protected:
	redefined int doApply(Data *);
private:
	Range region;		// for undo
};

class SpaceInsert : public DataModifier {
	typedef DataModifier Super;
public:
	SpaceInsert(Data* d, int nframes);
	SpaceInsert(Data* d, boolean asTime=false);
	static Modifier* create(DataEditor *);
protected:
	redefined Requester* createRequester();
	redefined int doApply(Data *);
protected:
	int nFrames;
	boolean insertTime;
	friend class SpaceInsertRequester;
	friend class TimeInsertRequester;
};

class LengthChanger : public DataModifier {
public:
	LengthChanger(Data* d, int newlen);
	LengthChanger(Data* d, boolean asDur=false);
	static Modifier* create(DataEditor *);
	redefined Modifier* createUnDo();
protected:
	redefined Requester* createRequester();
	redefined int doApply(Data *);
protected:
	int oldLength;
	int newLength;
	boolean asDuration;
	friend class LengthChangeRequester;
	friend class DurChangeRequester;
};

class SampRateChanger : public DataModifier {
	typedef DataModifier Super;
public:
	SampRateChanger(Data* d, int newsrate, boolean interp=true);
	SampRateChanger(Data* d);
	static Modifier* create(DataEditor *);
	redefined const char* message() { return "Changing sample rate..."; }
protected:
	redefined Requester* createRequester();
	redefined int doApply(Data *);
private:
	int newSrate;
	ChoiceValue doInterp;
	friend class SampRateRequester;
};

class FormatChanger : public DataModifier {
	typedef DataModifier Super;
public:
	FormatChanger(Data* d, DataType newtype);
	FormatChanger(Data* d);
	static Modifier* create(DataEditor *);
	redefined const char* message() { return "Changing sample format..."; }
	redefined Modifier* createUnDo();
protected:
	redefined Requester* createRequester();
	redefined int doApply(Data *);
private:
	ChoiceValue oldType;
	ChoiceValue newType;
	friend class FormatChangeRequester;
};

// specialized subclasses for working with LPC data file objects

class LPCData;

class LPCModifier : public DataModifier {
public:
	LPCModifier(Data* data) : DataModifier(data) {}
protected:
	redefined int doApply(Data *data) {
		return applyToLPCData((LPCData *) data);
	}
private:
	virtual int applyToLPCData(LPCData *) = 0;
};

class FrameStabilizer : public LPCModifier {
	typedef LPCModifier Super;
public:
	FrameStabilizer(Data* data) : LPCModifier(data) { initialize(); }
	static Modifier* create(DataEditor *);
private:
	redefined int applyToLPCData(LPCData *);
};
	
#endif

