/* "hardware.c" copyright 1994 thomas insel 
 *              copyrighy 1995 sven oliver moll */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <pwd.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/types.h>
#include <fcntl.h>
#include <string.h>
#include <mntent.h>
#include <errno.h>

#ifdef sun
#  include <sundev/srreg.h>
#elif defined linux
#  include <linux/cdrom.h>
#else
#  errror Please fix includes for your system in hardware.c
#endif

#include "config.h"
#include "database.h"
#include "hardware.h"

/* WWH hack to get rid of malloc/free's */
#ifdef CDCTRL
cdhw_t strHW;
#endif

extern int             errno;

/************************************************************************/
/* Procedure:  read_hw
 * Purpose:    to read hardware info
 * 
 * Inputs:     program name, CD file descriptor
 * Outputs:    to cdhw_t structure
 * Returns:    pointer to cdhw_t structure
 * Notes:  
 *   1.  The function free_hw_buf() must be called to free up the cdhw_t
 *       returned by this function.  This function uses malloc to get 
 *       a buffer (except when used with CDCTRL).
 */
/************************************************************************/
cdhw_t *
read_hw(char *progname, int cdfile, char *cdname)
{
  int ii;

#ifdef CDCTRL
  cdhw_t *hw = &strHW;
#else
  cdhw_t *hw = malloc(sizeof(cdhw_t));
#endif
#ifdef DEBUG
  fprintf (stderr, "read_hw:  buffer=0x%lx\n", (long)((void *)hw));
#endif


  /* WWH -- force all fields to 0 by default */
  memset (hw, 0, sizeof(cdhw_t));

  /* get status of CD-ROM */
  hw->iStatus = cd_status(progname, cdfile, cdname, hw);

#ifdef DEBUG
  fprintf (stderr, "read_hw:  CD status=%s\n", cd_status_text(hw->iStatus));
#endif
  
  /* if mounted or no CD, return extra error codes */
  if (hw->iStatus == CD_STATUS_ERROR) {
#ifdef CDCTRL
      hw->subchnl.cdsc_audiostatus = CDROM_AUDIO_ERROR;
      return (hw);
#else
      EXIT(1);
#endif
  } else if ((hw->iStatus == CD_STATUS_NO_CD) ||
             (hw->iStatus == CD_STATUS_REMOVED)) {
      hw->subchnl.cdsc_audiostatus=NO_CDROM;
      return hw;
  }
  else if (hw->iStatus == CD_STATUS_MOUNTED) {
      hw->subchnl.cdsc_audiostatus=CDROM_MOUNTED;
      return hw;
  }

  /* at this point, status must be IN or INSERTED */

/*   read header */
/*   if ( ioctl(cdfile, CDROMREADTOCHDR, &(hw->tochdr)) == -1 )  */
/*     { */
/*       fprintf(stderr, "%s: ioctl cdromreadtochdr\n", progname); */
/* #ifdef CDCTRL */
/*       hw->subchnl.cdsc_audiostatus = CDROM_AUDIO_ERROR; */
/*       return (hw); */
/* #else */
/*       EXIT(1); */
/* #endif */
/*     } */
/* printf ("Read TOCHDR\n"); */

  /* read individual tracks */
  hw->iData = hw->iAudio = 0;
  for (ii=hw->tochdr.cdth_trk0-1; ii<=hw->tochdr.cdth_trk1; ii++) {
      hw->tocentries[ii].cdte_track=
	  (ii==hw->tochdr.cdth_trk1) ? CDROM_LEADOUT : ii+1;
      
      hw->tocentries[ii].cdte_format = CDROM_MSF;
      if ( ioctl(cdfile, CDROMREADTOCENTRY, &(hw->tocentries[ii])) == -1 ) {
	  fprintf(stderr, "%s: ioctl cdromreadtocentry track %d\n", 
		  progname, ii);
#ifdef CDCTRL
	  hw->subchnl.cdsc_audiostatus = CDROM_AUDIO_ERROR;
	  return (hw);
#else
	  EXIT(1);
#endif
      }
      
      /* determine if data or audio track, skip CDROM_LEADOUT */
      if (ii!=hw->tochdr.cdth_trk1) {
	  if ((hw->tocentries[ii].cdte_ctrl&CDROM_DATA_TRACK) == 0)
	      hw->iAudio++;
	  else
	      hw->iData++;
      }
  }

  /* if no audio tracks, i.e., all-data disc, set status to DATA_DISC */
  if (hw->iAudio == 0) {
      hw->iStatus = CD_STATUS_DATA_DISC;
      hw->subchnl.cdsc_audiostatus = CDROM_DATA_DISC;
  } else {
      
      /* read subchannel info */
      hw->subchnl.cdsc_format = CDROM_MSF;
      if ( ioctl(cdfile, CDROMSUBCHNL, &(hw->subchnl)) == -1 ) {
	  fprintf(stderr, "%s: ioctl cdromsubchnl\n", progname);
#ifdef CDCTRL
	  hw->subchnl.cdsc_audiostatus = CDROM_AUDIO_ERROR;
	  return (hw);
#else
	  EXIT(1);
#endif
      }
  }
  
  return hw;

}

/************************************************************************/
/* Procedure:  checkmount 
 * Purpose:    to check if this CD is mounted
 * 
 * Inputs:     name of program, name of CD-ROM
 * Outputs:    debug
 * Returns:    T/F
 * Notes:  
 */
/************************************************************************/
int checkmount(char *progname, char *pszName) {
  FILE  *fp;
  struct mntent *mnt;
  int    ii;
  char   caB[100];
  char   *pszTest;
  
  /* check pszName -- if NULL, return -1 */
  if (pszName == NULL)
      return (-1);

  /* find out if the device is a link, resolve link name */
  ii = readlink(pszName, caB, sizeof(caB)-1);
  if (ii < 0) {
    pszTest = pszName;
#ifdef DEBUG
    fprintf (stderr, "chekmnt:  %s not a link\n", pszName);
#endif
  }else {
    pszTest = &caB[0];
    caB[ii] = '\0';
#ifdef DEBUG
    fprintf (stderr, "chekmnt:  link -> %s\n", caB);
#endif
  }

  /* check if drive is mounted (from Mark Buckaway's cdplayer code) */
  if ((fp = setmntent (MOUNTED, "r")) == NULL) {
    fprintf (stderr, "%s:  Couldn't open %s: %s\n", 
	     progname, MOUNTED, strerror (errno));
#ifdef CDCTRL
    return (-1);
#else
    EXIT(1);
#endif
  }
  while ((mnt = getmntent (fp)) != NULL) {
    if (strstr (mnt->mnt_fsname, pszTest) != NULL) {
      endmntent (fp);
      return (TRUE);
    }
  }
  endmntent (fp);
  return (FALSE);
}

/************************************************************************/
/* Procedure:  cd_status_text
 * Purpose:    to get status name in text
 * 
 * Inputs:     status code
 * Outputs:    to buffer
 * Returns:    pointer to status text string
 * Notes:  
 *   1. Input values:
 *
 *	0	No CD in drive.
 *	1	CD in drive.
 *	2	CD has just been inserted (TOC has been read)
 *	3	CD has just been removed
 *	4	CD is mounted
 *
 */
/************************************************************************/
char *cd_status_text (int iStatus)
{
  char *pszResult;
  switch (iStatus) 
    {
    case CD_STATUS_ERROR:
      pszResult = "error";
      break;
    case CD_STATUS_NO_CD:
      pszResult = "no_disc";
      break;
    case CD_STATUS_CD_IN:
      pszResult = "disc_in";
      break;
    case CD_STATUS_INSERTED:
      pszResult = "inserted";
      break;
    case CD_STATUS_REMOVED:
      pszResult = "removed";
      break;
    case CD_STATUS_MOUNTED:
      pszResult = "mounted";
      break;
    case CD_STATUS_DATA_DISC:
      pszResult = "data_disc";
      break;
    default:
      pszResult = "unknown";
      break;
    }
  return (pszResult);
}

/************************************************************************/
/* Procedure:  cd_status
 * Purpose:    to get status of the CD-ROM
 * 
 * Inputs:     name of program (text)
 *             CD file descriptor (int), -1 if not open
 *             cdname (text) name or symlink to CD device
 *             pHW (cdhw_t) pointer to hardware descriptor block
 * Outputs:    debug info only
 * Returns:    see below
 * Notes:  
 *   1. Stolen from XPlaycd code...
 *   2. Return values:
 *
 *     -N       error code
 *	0	No CD in drive.
 *	1	CD in drive.
 *	2	CD has just been inserted (TOC has been read)
 *	3	CD has just been removed
 *	4	CD is mounted
 *
 */
/************************************************************************/
int cd_status(char *progname, int cd_fd, char *cdname, cdhw_t *pHw) {
  /* struct cdrom_subchnl  sc;  */
  /* struct cdrom_tochdr   hdr; */
  int    iResult;

  /* check if the CD-ROM is mounted, if so, return MOUNTED */
  iResult = checkmount(progname, cdname);
  if (iResult < 0)
    return (iResult);
  else if (iResult == TRUE)
    return CD_STATUS_MOUNTED;


  /* try to open the CD-ROM.  If we fail, report error */
  if (cd_fd < 0) {
    if ((cd_fd = open(cdname, 0)) < 0) {
      if (errno == EACCES) {
	fprintf(stderr, "%s error opening %s\n", progname, cdname);
	fprintf(stderr,
		"As root, please run\n\nchmod 666 %s\n\n%s\n", "<yourname>",
		"to give yourself permission to access the CD-ROM device.");
      }
    }
    else if (errno==ENOENT) {
      fprintf(stderr,"%s: can't open %s: ", progname, cdname);
      perror("");
#ifdef CDCTRL
      return (-1);
#else
      EXIT(1);
#endif
    }
    else if (errno != ENXIO) {
      return CD_STATUS_NO_CD;
    }
  }
  
  /* get subchannel info, if invalid, no disc in drive */  
  pHw->subchnl.cdsc_format = CDROM_MSF;

  if (ioctl(cd_fd, CDROMSUBCHNL, &pHw->subchnl)) {
/*     cur_cdmode = 5; */
/*     cur_track = -1; */
/*     cur_cdlen = cur_tracklen = 1; */
/*     cur_pos_abs = cur_pos_rel = cur_frame = 0; */

/*     close(cd_fd); */
/*     cd_fd = -1; */
    
/* #ifdef DBG_EXIT */
/*     fprintf(stderr,"exit 2\n"); */
/* #endif */
    return CD_STATUS_NO_CD;
  }


  /* read the header */
  if (ioctl(cd_fd, CDROMREADTOCHDR, &pHw->tochdr)) 
    return CD_STATUS_NO_CD;

  switch (pHw->subchnl.cdsc_audiostatus) {
  case CDROM_AUDIO_PLAY:
  case CDROM_AUDIO_PAUSED:
  case CDROM_AUDIO_COMPLETED:
    return CD_STATUS_CD_IN;
    break;
  }   

  return CD_STATUS_CD_IN;

}

/************************************************************************/
/* Procedure:  free_hw_buf
 * Purpose:    to free up hardware buffer returned by read_hw
 * 
 * Inputs:     pointer to buffer
 * Outputs:    debug info only
 * Returns:    n/a
 * Notes:  
 *   1.
 */
/************************************************************************/
void free_hw_buf(cdhw_t *pHw)
{

#ifdef DEBUG
    fprintf (stderr,"free_hw:  bufp=0x%lx\n", (long)((char *)pHw));
#endif

#ifndef CDCTRL
    if ((void *)pHw != NULL)
	free ((void *)pHw);
#endif
}
