/*
** Copyright (C) 21 Sep 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>

#include "common_gtk.h"
#include "vector_commands.h"
#include "rip_audio.h"
#include "linebuffer.h"
#include "locks.h"
#include "contractions.h"
#include "readers.h"
#include "progressbars_gui.h"
#include "preferences.h"
#include "audio.h"
#include "status_text.h"
#include "mainwindow.h"
#include "globals.h"

char *cdda2wav_path = NULL, *cdda_device_path;
char *def_rip_path = NULL;
GtkWidget *rip_path_entry;

static GtkWidget *rip_win = NULL, *p_win;
static GtkWidget *autoscroll_cdda2wav_text, *cdda2wav_text, *cdda2wav_text_scrollbar;
static GtkWidget *rip_device_entry, *rip_speed_entry;
static linebuffer cdda2wav_stdout, cdda2wav_stderr;
static int n_rip_tracks;
static char *dump_path = NULL;

static cmd_v *make_rip_command(void) {
	cmd_v *rip_cmd;
	char *s, *tmp;

	if (!(strcmp((tmp = gtk_entry_get_text(GTK_ENTRY(rip_device_entry))), ""))) {
		alert_user_of_error(_("You need to supply a device name to dump audio tracks from."));
		return NULL;
	}

	rip_cmd = get_new_cmd();
	add_option_to_cmd(rip_cmd, cdda2wav_path);

	s = string_append("-D", NULL);
	s = string_append(s, tmp);
	add_option_to_cmd(rip_cmd, s);
	free(s);

	add_option_to_cmd(rip_cmd, "-B");
	/* Yes, this should be "-g". Please upgrade your cdda2wav! */
	add_option_to_cmd(rip_cmd, "-g");
	add_option_to_cmd(rip_cmd, "-H");

	if (strcmp((tmp = gtk_entry_get_text(GTK_ENTRY(rip_speed_entry))), "")) {
		s = string_append("-S", NULL);
		s = string_append(s, tmp);
		add_option_to_cmd(rip_cmd, s);
		free(s);
	}

	if (!(strcmp((tmp = gtk_entry_get_text(GTK_ENTRY(rip_path_entry))), ""))) {
		alert_user_of_error(_("You need to supply a path where to store the audio tracks."));
		destroy_cmd(rip_cmd);
		return NULL;
	}
/* TODO: if tmp is dir && tmp doens't end with / then append / */
	s = string_append(tmp, NULL);
	s = string_append(s, "gcombust");
	add_option_to_cmd(rip_cmd, s);

	/* save the path for later use when adding files to the audio clist */
	if (dump_path != NULL)
		free(dump_path);
	dump_path = s;

	return rip_cmd;
}

static int read_cdda2wav_stdout(int fd) {
	if (!read_into(&cdda2wav_stdout, fd))
		return FALSE;

	while	(extract_line(&cdda2wav_stdout, "\n") || extract_line(&cdda2wav_stdout, "\r")) {
		g_message("cdda2wav stdout (unexpected): %s\n", cdda2wav_stdout.newline);
	}
	return TRUE;
}

static void check_for_track_change(void) {
	char *tmp;

	if (NULL != (tmp = strstr(cdda2wav_stderr.newline, "  track "))) {
		int track;
		char *s;
		char p[] = "_00.wav";
		track = atol(tmp + strlen("  track "));
		if (0 == track)
			g_message("%s:%i detected track done is wrong",
					__FILE__, __LINE__);
		set_track_dumping_now(track + 1, n_rip_tracks);

		g_snprintf(p, strlen(p) + 1, "_%02d.wav", track);
		s = string_append(dump_path, NULL);
		s = string_append(s, p);
		add_track(NULL, s);
		free(s);
	} else
		g_warning("%s:%i no new track when expected: %s\n",
				__FILE__, __LINE__, cdda2wav_stderr.newline);
}

static int read_cdda2wav_stderr(int fd) {
	char *tmp;
	int changed = FALSE;

	if (!read_into(&cdda2wav_stderr, fd))
		return FALSE;

	while	(extract_line(&cdda2wav_stderr, "\n")
			|| extract_line(&cdda2wav_stderr, "\r")) {
		if (!(strncmp("Tracks:", cdda2wav_stderr.newline, strlen("Tracks:")))) {
			n_rip_tracks = atol(&cdda2wav_stderr.newline[strlen("Tracks:")]);
			set_track_dumping_now(1, n_rip_tracks);
		} else if (NULL != (tmp = strstr(cdda2wav_stderr.newline, "%"))) {
			float p;
			tmp--;
			if (isdigit(tmp[0]))
				tmp--;
			p = ((float)atoi(tmp))/100.0;
			if (p <= 1 && p >= 0) {
				estimate_finnish_time_update(p);
				gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), p);
			} else {
				g_warning("%s:%i percentage done is wrong", __FILE__, __LINE__);
			}
			if (NULL != strstr(cdda2wav_stderr.newline, "successfully recorded")) {
				check_for_track_change();
			}
		} else if (NULL != strstr(cdda2wav_stderr.newline, "successfully recorded")) {
			check_for_track_change();
		} else if (strstr(cdda2wav_stderr.newline, "cannot stat device ")) {
			alert_user_of_error(_("Cannot stat supplied device"));
		} else {
			if (!changed) {
				changed = TRUE;
				gtk_text_freeze(GTK_TEXT(cdda2wav_text));
			}
			gtk_text_insert(GTK_TEXT(cdda2wav_text), NULL, NULL, NULL,
					cdda2wav_stderr.newline, -1);
		}
	}

	if (changed) {
		if (GTK_TOGGLE_BUTTON(autoscroll_cdda2wav_text)->active)
			scroll_down_text(cdda2wav_text, cdda2wav_text_scrollbar);
		gtk_text_thaw(GTK_TEXT(cdda2wav_text));
	}

	return TRUE;
}

static void close_cdda2wav_reader(void) {
	destroy_buffer(&cdda2wav_stdout);
	destroy_buffer(&cdda2wav_stderr);
	gtk_widget_destroy(p_win);
	not_running();
}

static int wait_cdda2wav_output(gpointer data) {
	static int stdout_ok = TRUE, stderr_ok = TRUE;
	fd_set rset;
	struct timeval tv;
	int result, max;
	int *fd = data;

	g_assert((stdout_ok == TRUE) || (stderr_ok == TRUE));

	tv.tv_sec = 0;
	tv.tv_usec = 0;

	FD_ZERO(&rset);
	max = -1;

	if (stdout_ok) {
		FD_SET(fd[0], &rset);
		max = fd[0];
	}
	if (stderr_ok) {
		FD_SET(fd[1], &rset);
		if (-1 == max)
			max = fd[1];
		else
			max = fd[0]>fd[1] ? fd[0] : fd[1];
	}

	result = select(max + 1, &rset, NULL, NULL, &tv);

	if (result < 0)
		g_error("wait_cdda2wav_output: select returned negative!");
	else if (result == 0)
		return TRUE;

	if (stdout_ok && FD_ISSET(fd[0], &rset))
		stdout_ok = read_cdda2wav_stdout(fd[0]);
	if (stderr_ok && FD_ISSET(fd[1], &rset))
		stderr_ok = read_cdda2wav_stderr(fd[1]);

	if (!(stdout_ok || stderr_ok)) {
		stdout_ok = stderr_ok = TRUE;
		close_cdda2wav_reader();
		return FALSE;
	} else
		return TRUE;
}

static void start_ripping_clicked(GtkWidget *widget, gpointer data) {
	cmd_v *cmd;
	int *output;

	if (is_running())
		return;

	n_rip_tracks = -1;

	if (NULL == (cmd = make_rip_command())) {
		not_running();
		return;
	}

	if (NULL == (output = popen_re_unbuffered(cmd))) {
		destroy_cmd(cmd);
		g_warning("%s:%i failed to execute cdda2wav rip command", __FILE__, __LINE__);
		not_running();
		return;
	}

	destroy_cmd(cmd);

	init_buffer(&cdda2wav_stdout);
	init_buffer(&cdda2wav_stderr);

	bailed_out = FALSE;
	p_win = open_progress_win(_("Dumping CD to wav files"));
	if (misc_prefs.own_progresswin)
		gtk_window_set_title(GTK_WINDOW(p_win), _("Audio CD to wav dump"));
	estimate_finnish_time_start();
	gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), 0.0);

	gtk_timeout_add(TIMEOUT_GTK, wait_cdda2wav_output, output);
}

static void close_rip_win(GtkWidget *widget, gpointer data) {
	gtk_widget_destroy(rip_win);
	rip_win = NULL;
}

static void rip_dir_selected(GtkWidget *widget, gpointer data) {
	if (NULL != rip_win)
		gtk_entry_set_text(GTK_ENTRY(rip_path_entry),
				gtk_file_selection_get_filename(data));
	gtk_widget_destroy(GTK_WIDGET(data));
}
static void delete_cdda2wav_text(GtkWidget *widget, gpointer data) {
	del_text(cdda2wav_text);
}

static GtkWidget *add_rip_widget(GtkWidget *rip_win) {
	GtkWidget *vbox1, *hbox1;
	GtkWidget *select_rip_path;
	GtkWidget *hbox2, *hbox3, *hbox4, *but_box;
	GtkWidget *device_label;
	GtkWidget *rip_speed_label;
	GtkWidget *rip, *close, *sep, *clear_text;
	file_req_info *add_struct;

	vbox1 = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(rip_win), vbox1);

	add_struct = malloc(sizeof(file_req_info)); /* FIXME: free this somewhere */
	add_struct->title = g_strdup(_("Select dir where to dump audio files"));
	add_struct->func = (gpointer) &rip_dir_selected;

	hbox1 = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox1, FALSE, TRUE, 5);

	select_rip_path = gtk_button_new_with_label(_("Dump where:"));
	gtk_box_pack_start(GTK_BOX(hbox1), select_rip_path, FALSE, FALSE, 5);
	gtk_signal_connect(GTK_OBJECT(select_rip_path), "clicked",
			GTK_SIGNAL_FUNC(choose_path), (gpointer) add_struct);
	gtk_tooltips_set_tip(tooltips, select_rip_path, _("Open up a filerequester "
				"and select the path of where you want to dump the audio CD "
				"as .wav files (~700MB of free space required) "), NULL);

	rip_path_entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox1), rip_path_entry, TRUE, TRUE, 5);
	gtk_tooltips_set_tip(tooltips, rip_path_entry, _("This is the path to where "
				"the .wav files will be written"), NULL);
	if (NULL == def_rip_path)
		def_rip_path = g_strdup("");
	gtk_entry_set_text(GTK_ENTRY(rip_path_entry), def_rip_path);

	hbox2 = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox2, TRUE, TRUE, 5);

	hbox3 = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox2), hbox3, TRUE, TRUE, 5);

	device_label = gtk_label_new(_("device:"));
	gtk_box_pack_start(GTK_BOX(hbox3), device_label, TRUE, TRUE, 5);

	rip_device_entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox3), rip_device_entry, TRUE, TRUE, 5);
#ifdef __FreeBSD__
	gtk_tooltips_set_tip(tooltips, rip_device_entry,
			_("Path to device from which to copy (for example 0,6,0)"), NULL);
#else
	gtk_tooltips_set_tip(tooltips, rip_device_entry,
			_("Path to device from which to copy (for example for cooked style "
				"devices (plain IDE): /dev/hdb, or for SCSI devices in the "
				"\"cdrecord style\": 0,6,0)"), NULL);
#endif

	if (NULL != cdda_device_path)
		gtk_entry_set_text(GTK_ENTRY(rip_device_entry), cdda_device_path);

	rip_speed_label = gtk_label_new(_("speed:"));
	gtk_box_pack_start(GTK_BOX(hbox3), rip_speed_label, FALSE, FALSE, 5);

	rip_speed_entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox3), rip_speed_entry, FALSE, FALSE, 5);
	gtk_tooltips_set_tip(tooltips, rip_speed_entry,
			_("Speed to use on the cdrom when reading the audio tracks (leave empty "
				"to use the default setting of the cdrom drive)."), NULL);

	hbox4 = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox4, TRUE, TRUE, 5);

	close = gtk_button_new_with_label(_("Close"));
	gtk_box_pack_end(GTK_BOX(hbox4), close, FALSE, FALSE, 5);
	gtk_signal_connect(GTK_OBJECT(close), "clicked",
			GTK_SIGNAL_FUNC(close_rip_win), NULL);
	gtk_tooltips_set_tip(tooltips, close,
			_("Close this window"), NULL);

	rip = gtk_button_new_with_label(_("Rip"));
	gtk_box_pack_end(GTK_BOX(hbox4), rip, FALSE, FALSE, 5);
	gtk_signal_connect(GTK_OBJECT(rip), "clicked",
			GTK_SIGNAL_FUNC(start_ripping_clicked), NULL);
	gtk_tooltips_set_tip(tooltips, rip,
			_("Start converting the audio CD to .wav files"), NULL);

	sep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox1), sep, FALSE, FALSE, 0);

	/* buttons */
	but_box = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), but_box, FALSE, FALSE, 0);

	autoscroll_cdda2wav_text = gtk_check_button_new_with_label(_("Autoscroll text"));
      gtk_tooltips_set_tip(tooltips, autoscroll_cdda2wav_text, _("Scroll down to "
			"the end of the text whenever something new is added"),
			NULL);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(autoscroll_cdda2wav_text), TRUE);
	gtk_box_pack_start(GTK_BOX(but_box), autoscroll_cdda2wav_text, FALSE, FALSE, 0);

	clear_text = gtk_button_new_with_label(_("Clear"));
      gtk_tooltips_set_tip(tooltips, clear_text, _("Clears the text box"),
			NULL);
	gtk_box_pack_end(GTK_BOX(but_box), clear_text, FALSE, FALSE, 2);
	gtk_signal_connect(GTK_OBJECT(clear_text), "clicked",
			GTK_SIGNAL_FUNC(delete_cdda2wav_text), 0);

	cdda2wav_text_scrollbar	= create_text(vbox1, &cdda2wav_text, _("cdda2wav output"));

	return vbox1;
}

void open_rip_audio_req(void) {

	if (NULL != rip_win)
		return;

	rip_win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_signal_connect(GTK_OBJECT(rip_win), "destroy",
			GTK_SIGNAL_FUNC(close_rip_win), NULL);
	gtk_window_set_title(GTK_WINDOW(rip_win), _("Dump CD"));

	add_rip_widget(rip_win);

	gtk_widget_show_all(rip_win);
}
