/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <errno.h>

#include <sys/time.h>
#include <pthread.h>
#include <unistd.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "nntp.h"
#include "queue.h"
#include "queue-item-decode.h"
#include "debug.h"
#include "gui.h"
#include "log.h"
#include "util.h"

static const int CLOSE_IDLE_SOCKETS_TIMEOUT_SECS = 120;

gboolean queue_stop_all;

GSList *queue = NULL;

static pthread_mutex_t qlock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t qcond_m = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t qcond = PTHREAD_COND_INITIALIZER;

/**
***
**/

static void
queue_qcond_broadcast (void)
{
	pthread_mutex_lock (&qcond_m);
	pthread_cond_broadcast (&qcond);
	pthread_mutex_unlock (&qcond_m);
}

void
queue_add (QueueItem* item)
{
	g_return_if_fail (item!=NULL);

	pan_object_sink (PAN_OBJECT(item));

	/* add the item to the queue */
	pthread_mutex_lock (&qlock);
	if (queue_item_is_high_priority(item)) {
		debug (DEBUG_QUEUE,
			"queue_add: pushing %p to front of the queue", item);
		queue = g_slist_prepend (queue, item);
	}
	else {
		debug (DEBUG_QUEUE,
			"queue_add: pushing %p to back of the queue", item);
		queue = g_slist_append (queue, item);
	}
	gui_set_queue_size (g_slist_length(queue));
	pthread_mutex_unlock (&qlock);

	/* signal the addition */
	queue_qcond_broadcast();
}

/**************
***************  SOCKETS
**************/

typedef struct
{
	PanSocket *sock;
	QueueItem *item;
}
SocketsHashtableRecord;

static pthread_mutex_t server_pool_mutex = PTHREAD_MUTEX_INITIALIZER;
static GHashTable* sockets_hashtable = NULL;
static pthread_t queue_t;

static gint
queue_get_socket_for_item (QueueItem *item)
{
	GSList* list = NULL;
	GSList* l = NULL;
	SocketsHashtableRecord *rec = NULL;
	gint retval = -1;
	int i=0;
	const gboolean is_leech_item = QUEUE_ITEM(item)->is_leech;
	gboolean skip_first_idle = is_leech_item;
	gboolean nonleech_socket_exists = FALSE;
	server_data *sdata = item->sdata;

	debug (DEBUG_QUEUE, "queue_get_socket_for_item want server_pool_mutex");
	pthread_mutex_lock (&server_pool_mutex);
	debug (DEBUG_QUEUE, "queue_get_socket_for_item got server_pool_mutex");

	/* see if we have an idle socket */
	list = (GSList*) g_hash_table_lookup(sockets_hashtable, sdata);
	for (l=list, i=0; l!=NULL; l=l->next, ++i)
	{
		rec = (SocketsHashtableRecord*) l->data;
		if (rec->item != NULL) /* busy */
			continue;
		if (skip_first_idle) { /* save 1 sock for non-leech */
			nonleech_socket_exists = TRUE;
			skip_first_idle = FALSE;
			continue;
		}
		rec->item = item;
		queue_item_set_socket(item, rec->sock);
		retval = 0;
		debug (DEBUG_QUEUE,"queue_get_socket: reusing socket #%d", i);
		break;
	}

	/* if no idle sockets... */
	if (retval)
	{
		const int current_connections = g_slist_length(list);
		int slots_open = sdata->max_connections - current_connections;
		debug (DEBUG_QUEUE,"queue_get_socket: no idle sockets");
		if (!nonleech_socket_exists && is_leech_item && sdata->max_connections>1)
			slots_open--;
		if (slots_open<=0)
		{
			debug (DEBUG_QUEUE,
				"feh, we've already got %d connections "
				"and %d nonleech connections",
				current_connections,
				(nonleech_socket_exists?1:0) );
		}
		else
		{
			const char* errmsg = NULL;

			/* open a new socket.. */
			PanSocket *sock = NULL;
			debug (DEBUG_QUEUE,
				"queue_get_socket: make new connection to %s",
				sdata->address);
			sock = pan_socket_new (sdata->address, sdata->port);
			pan_object_sink (PAN_OBJECT(sock));

			debug (DEBUG_QUEUE,
				"queue_get_socket: new socket %p", sock);
			if (sock->sockfd == 0)
				errmsg = "Unable to connect to server";

			/* try to log in... */
			if (!errmsg)
				errmsg = nntp_login(sock, sdata);
			if (errmsg!=NULL)
			{
				log_add(errmsg);
				g_warning(errmsg);
				if (sock)
					pan_object_unref (PAN_OBJECT(sock));
				sock = NULL;
				debug (DEBUG_QUEUE,
					"queue_get_socket:%p can't login",sock);
				/*queue_item_set_paused (item, TRUE);*/
			}
			else
			{
				debug (DEBUG_QUEUE,
					"queue_get_socket: %p logged in", sock);

				/* looks like we got a new socket okay...
		 		 * (1) create a holding rec for the sock/item
		 		 * (2) let the item know what its socket is
		 		 * (3) update the hashtable
		 		*/
				rec = g_new0 (SocketsHashtableRecord, 1);
				rec->sock = sock;
				rec->item = item;
				queue_item_set_socket(item, sock);
				list = g_slist_prepend (list, rec);
				g_hash_table_insert(sockets_hashtable,
					sdata, list);
				retval = 0;
			}
		}
	}

	pthread_mutex_unlock (&server_pool_mutex);
	debug (DEBUG_QUEUE, "queue_get_socket_for_item releasing server_pool_mutex");

	debug (DEBUG_QUEUE,
		"queue_get_socket_for_item was %ssuccessful",
		(retval?"not ":""));

	return retval;
}

static void
queue_socket_checkin (PanSocket *sock, const server_data* sdata)
{
	GSList* l = NULL;
	GSList* list = NULL;
	SocketsHashtableRecord *rec = NULL;
	gboolean found = FALSE;
	int i=0;

	debug (DEBUG_QUEUE, "queue_socket_checkin trying for server_pool_mutex");
	pthread_mutex_lock (&server_pool_mutex);
	debug (DEBUG_QUEUE, "queue_socket_checkin got server_pool_mutex");

	list = (GSList*) g_hash_table_lookup(sockets_hashtable, sdata);
	for (l=list, i=0; l!=NULL; l=l->next, ++i)
	{
		rec = (SocketsHashtableRecord*) l->data;
		if (rec->sock!=sock)
			continue;

		rec->item = NULL;
		found = TRUE;
		if (sock->error)
		{
			/* I wish to complain about this parrot what I
        		 * purchased not half an hour ago from this
			 * very boutique.
			 */
			pan_object_unref (PAN_OBJECT(sock));
			list = g_slist_remove (list, rec);
			l = NULL;
			g_free (rec);
			g_hash_table_insert(
				sockets_hashtable, (gpointer)sdata, list);
			debug(DEBUG_QUEUE,"Burying dead socket #%d",i);
		}
		else debug(DEBUG_QUEUE,"Checking in socket #%d",i);
		break;
	}
	pan_warn_if_fail (found);

	pthread_mutex_unlock (&server_pool_mutex);
	debug (DEBUG_QUEUE, "queue_socket_checkin releasing server_pool_mutex");

	/* there may be a queued item waiting for this socket,
	   so wake up the queue manager. */
	queue_qcond_broadcast();
}


/**
 * Note that this must be called within a server_pool mutex lock.
 **/
static void
sockets_upkeep (gpointer key, gpointer value, gpointer user_data)
{
	GSList* l = NULL;
	GSList* new_list = NULL;
	GSList* old_list = (GSList*) value;
	const time_t current_time=time(0);

	debug (DEBUG_QUEUE,
		"in socket_upkeep for server %s -- %d sockets",
		((server_data*)key)->name,
		g_slist_length(old_list) );

	for ( l=old_list; l!=NULL; l=l->next )
	{
		SocketsHashtableRecord *rec = (SocketsHashtableRecord*)l->data;
		PanSocket *sock = rec->sock;
		const int age = current_time - sock->last_action_time;
		gboolean destroy = ((rec->item==NULL) &&
		                    (age>CLOSE_IDLE_SOCKETS_TIMEOUT_SECS));

		/* if it's idle and we're keeping it, send a keepalive msg.
		   Note that we don't let this affect the socket's
		   age, otherwise they would never grow old and close... */
		if (rec->item==NULL && !destroy) {
			const time_t last_action_time = sock->last_action_time;
			debug (DEBUG_QUEUE, "sending %p a keepalive", sock);
			destroy = nntp_send_noop(sock) != 0;
			sock->last_action_time = last_action_time;
		}

		/* either keep the socket or cull it from the herd */
		if (!destroy)
			new_list = g_slist_append(new_list, rec);
		else {
			debug (DEBUG_QUEUE,
				"socket %p idle for %d seconds...closing",
				sock, age);
			nntp_disconnect (sock);
			pan_object_unref (PAN_OBJECT(sock));
			g_free(rec);
		}
	}

	g_slist_free(old_list);
	g_hash_table_insert(sockets_hashtable, key, new_list);
}

/**************
***************  QUEUE MANAGEMENT
**************/

static void
queue_run (void* vp)
{
	int retval=0;
	QueueItem* item=QUEUE_ITEM(vp);
	gchar* desc = status_item_describe (STATUS_ITEM(item));
	const server_data* sdata = NULL;
	PanSocket *sock = NULL;
	gboolean sock_error;

	debug (DEBUG_QUEUE,"in queue_run with socket %p", item->sock);
	debug (DEBUG_QUEUE,"descrption of this queue item: [%s]", desc);

	gui_add_status_item(STATUS_ITEM(item));
	retval = queue_item_run (item);
	sock_error = item->sock ? item->sock->error : FALSE;
	debug (DEBUG_QUEUE,"queue item[%p] (%s) returned: %d",item,desc,retval);
	g_free(desc);

	/* clean up gui */
	gui_remove_status_item(STATUS_ITEM(item));

	/* check in the socket *after* disposing of the queue item, so that
	   if the task gets resubmitted to the front of the queue, it'll be
	   the first in line when the socket checkin code signals
	   run_what_we_can */
	sock = item->sock;
	sdata = item->sdata;

	if (!retval) /* if success, clean up the queue-item. */
	{
		pan_object_unref (PAN_OBJECT(item));
	}
	else if (sock_error) /* socket failed; retry */
	{
		gchar* description = status_item_describe (STATUS_ITEM(item));
		g_message (_("task `%s' failed: socket error; trying again."),
			description);
		g_free (description);
		item->high_priority = TRUE;
		queue_add (item);
	}
	else /* other conditions go here */
	{
		pan_object_unref (PAN_OBJECT(item));
	}

	/* check in the socket */
	if (sock!=NULL)
	{
		debug (DEBUG_QUEUE,
			"task %p done; checkin socket %p", item, sock);
		queue_socket_checkin(sock, sdata);
	}

	pthread_exit (NULL);
}


static void
queue_run_thread (QueueItem *item)
{
	debug (DEBUG_QUEUE,
		"starting a thread to run QueueItem %p, sock %p",
		item, item->sock);
       	pthread_create (&item->thread_id, NULL, (void*)queue_run, item);
       	pthread_detach (item->thread_id);
}


/**
 * This function tries to run any entries in the queue which have a free
 * socket at this time.
 **/
static void
queue_run_what_we_can (void)
{
	GSList* l = NULL;
	GSList* list = NULL;
	GSList* new_list = NULL;
	int len = 0;

	debug (DEBUG_QUEUE, "in run_what_we_can");

	pthread_mutex_lock (&qlock);
	list = queue;
	queue = NULL;
	pthread_mutex_unlock (&qlock);

	debug (DEBUG_QUEUE, "in run_what_we_can: got a qlock");
	for (l=list; l!=NULL; l=l->next)
	{
		QueueItem *item = QUEUE_ITEM(l->data);
		debug (DEBUG_QUEUE, "in run_what_we_can: QueueItem is %p", item);

		if (queue_item_is_paused(item))
		{
			debug (DEBUG_QUEUE,
				"QueueItem %p is paused, so we'll leave it in the queue...",
				item);
			new_list = g_slist_append (new_list, item);
			continue;
		}

		if (queue_item_needs_socket(item) && queue_get_socket_for_item(item))
		{
			debug (DEBUG_QUEUE,
				"QueueItem %p has no socket; wait in queue", item);
			new_list = g_slist_append (new_list, item);
			continue;
		}

		debug(DEBUG_QUEUE,"in run_what_we_can: %p is good to go",item);

		queue_run_thread (item);
	}

	g_slist_free (list);

	pthread_mutex_lock (&qlock);
	queue = g_slist_concat (new_list, queue);
	len = g_slist_length (queue);
	pthread_mutex_unlock (&qlock);

	gui_set_queue_size (len);

	debug(DEBUG_QUEUE,
		"in run_what_we_can: updating queue and leaving. "
		"queue is now address %p and has %d items",
		queue, len );
}


static void
queue_processing (void)
{
	/*time_t last_run = time(0);*/
	debug (DEBUG_QUEUE, "queue_processing");
	
	for ( ;; )
	{
		const int TIMEOUT_SECS = 20;
		struct timeval now;
		struct timespec timeout;
		int retcode = 0;
		pthread_mutex_lock (&qcond_m);
		gettimeofday (&now, NULL);

		debug (DEBUG_QUEUE,
			"queue_processing: sleeping %d secs", TIMEOUT_SECS);

		/* Wait TIMEOUT_SECS for something to hit the queue */
		timeout.tv_sec = now.tv_sec + TIMEOUT_SECS;
		timeout.tv_nsec = timeout.tv_sec;
		retcode = pthread_cond_timedwait (&qcond, &qcond_m, &timeout);
		pthread_mutex_unlock (&qcond_m);

		/* process any queue items */
		queue_run_what_we_can ();

		/* if timeout, do socket upkeep */
		if (retcode==ETIMEDOUT) {
			debug (DEBUG_QUEUE, "queue_processing getting server_pool_mutex");
			pthread_mutex_lock (&server_pool_mutex);
			debug (DEBUG_QUEUE, "queue_processing got server_pool_mutex");
			g_hash_table_foreach (
				sockets_hashtable,
				sockets_upkeep,
				NULL);
			pthread_mutex_unlock (&server_pool_mutex);
			debug (DEBUG_QUEUE, "queue_processing released server_pool_mutex");
		}

		pthread_mutex_unlock (&qcond_m);
	}
}


void
queue_init (void)
{
	debug(DEBUG_QUEUE,"queue_init");
	sockets_hashtable = g_hash_table_new (g_direct_hash, g_direct_equal);

        /* Fire up the queue thread */
        pthread_create (&queue_t, NULL, (void*)queue_processing, NULL);
        pthread_detach (queue_t);
}
