/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <errno.h>
#include <gnome.h>

#include "articlelist.h"
#include "fnmatch.h"
#include "globals.h"
#include "group-props.h"
#include "grouplist.h"
#include "gui.h"
#include "prefs.h"
#include "queue.h"
#include "queue-item-headers.h"
#include "queue-item-grouplist.h"
#include "util.h"

#include "xpm/disk.xpm"
#include "xpm/envelope.xpm"


static void group_dialog_clicked_cb (GnomeDialog*, int, server_data *);
static void grouplist_add_group (server_data *, group_data *);
static void grouplist_update_selected_rows (void);
static void grouplist_remove_row (const group_data * gdata);
static void grouplist_grep (GtkWidget *group_entry);

static GnomeUIInfo groups_menu_popup[] = {
	{
		GNOME_APP_UI_ITEM,
		N_("Subscribe to Selected"),
		N_("Subscribe to the selected group(s)"),
		grouplist_selected_subscribe
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Unsubscribe from Selected"),
		N_("Unsubscribe from the selected group(s)"),
		grouplist_selected_unsubscribe
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Get All Article Headers"),
		N_("Get all article headers"),
		grouplist_selected_download_all
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Get New Article Headers"),
		N_("Get new article headers"),
		grouplist_selected_download_new
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Empty Group"),
		N_("Remove all articles from the selected group(s)."),
		grouplist_selected_empty
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Remove `New' Status"),
		N_("Remove selected group(s) from `new groups' list."),
		grouplist_selected_unnew
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Properties"),
		N_("Set the properties for the selected group."),
		grouplist_selected_properties
	},
	GNOMEUIINFO_END
};

static GnomePixmap *disk_pixmap = NULL;
static GnomePixmap *subscribed_pixmap = NULL;

static int          group_mode = GROUP_ALL;
static GtkWidget   *group_clist_menu;
static server_data *my_server = NULL;
static GtkWidget   *grouplist_mode_menu;


GSList*
grouplist_get_selected_groups (void)
{
	GSList *groups = NULL;
	GList *l = NULL;
	for (l=GTK_CLIST(Pan.group_clist)->selection; l; l=l->next)
	{
		group_data *gdata = (group_data *)
			gtk_clist_get_row_data (
				GTK_CLIST (Pan.group_clist),
				GPOINTER_TO_INT (l->data));
		groups = g_slist_prepend (groups, gdata);
	}
	groups = g_slist_reverse (groups);
	return groups;
	
}

server_data *
grouplist_get_current_server (void)
{
	return (my_server);
}

int
grouplist_get_view_mode (void)
{
	return (group_mode);
}

void
grouplist_server_selected (server_data *sdata)
{
	gchar *path = g_strdup_printf ("/%s/%s_grouplist.db", data_dir, sdata->name);

	/* Do nothing if the server is already the selected one */
	if (my_server == sdata) return;

	/* Clear the ui and set the active list */
	gtk_clist_clear (GTK_CLIST(Pan.group_clist));
	my_server = sdata;

	/* Handle it */
	if (sdata->grouplist) {
		grouplist_update_ui (sdata);
	} else if (g_file_test (path, G_FILE_TEST_ISFILE)) {
		grouplist_load (sdata);
	} else {	
		GnomeDialog *dialog = (GnomeDialog *) gnome_dialog_new (_("No Group List"), GNOME_STOCK_BUTTON_YES, GNOME_STOCK_BUTTON_NO, NULL);
		gchar *p = g_strdup_printf (_("We don't have a group list for server '%s'\nShall we get one?\n"), sdata->name);
		GtkWidget *label = gtk_label_new (p);
		g_free (p);
		gtk_container_add (GTK_CONTAINER(dialog->vbox), label);
		gtk_signal_connect (GTK_OBJECT(dialog), "clicked",
				    GTK_SIGNAL_FUNC(group_dialog_clicked_cb), sdata);
		gui_popup_draw (GTK_WIDGET(dialog), Pan.window);
	}
	g_free (path);
}

#if 0
static void
click_groups_column_cb (GtkCList *clist, int n)
{
	if (n)
		return;

	switch (group_mode) {
		case GROUP_SUBSCRIBED:
			group_mode = GROUP_NEW;
			break;
		case GROUP_NEW:
			group_mode = GROUP_ALL;
			break;
		case GROUP_ALL:
			group_mode = GROUP_SUBSCRIBED;
			break;
	}

	grouplist_update_ui (NULL);
}
#endif


/* Process what happens when a user clicks on a group in the group list */
static void
grouplist_select_row (GtkWidget *widget, int r, int c,
		      GdkEvent *event)
{
	group_data *gdata;
	GdkEventButton *bevent = (GdkEventButton *)event;

	if (bevent)
	{
		if ((bevent->button) == 1 && (bevent->type == GDK_2BUTTON_PRESS))
		{
			gdata = (group_data *) gtk_clist_get_row_data(GTK_CLIST(widget), r);
			
			if (articlelist_get_current_group() == gdata) {
				gui_page_set (ARTICLELIST_PAGE, Pan.article_ctree);
				return;
			}

			/* if in notebook layout, flip to the article list */
			gui_page_set (ARTICLELIST_PAGE, Pan.article_ctree);
			articlelist_set_current_group ( grouplist_get_current_server(), gdata );//gui_set_title (gdata);
		}
	}
}

static void
grouplist_process_selected (int type)
{
	server_data* sdata = grouplist_get_current_server();
	GtkCList *clist = GTK_CLIST(Pan.group_clist);
	const gboolean has_selection = clist->selection != NULL;
	group_data *gdata = NULL;
	GList* list = NULL;
	GList* l;

	/* build the list of gdatas */
	if (has_selection) {
		list = g_list_copy (GTK_CLIST (Pan.group_clist)->selection);
		for (l=list; l!=NULL; l=l->next)
			l->data = gtk_clist_get_row_data (
				GTK_CLIST (Pan.group_clist),
				GPOINTER_TO_INT (l->data));
	}

	if (!list && (gdata=articlelist_get_current_group()))
	{
		switch (type)
		{
			case 3:
				queue_add (QUEUE_ITEM(
					queue_item_headers_new(
						sdata, TRUE, gdata,
						HEADERS_ALL)));
				break;
			case 4:
				queue_add (QUEUE_ITEM(
					queue_item_headers_new(
						sdata, TRUE, gdata,
						HEADERS_NEW)));
				break;
			default:
				break;
		}
	}

	for (l=list; l; l=l->next)
	{
		gdata = (group_data*) l->data;

		switch (type)
		{
			case 1: /* subscribe */
				gdata->flags |= GROUP_SUBSCRIBED;
				break;
			case 2: /* unsubscribe */
				gdata->flags &= ~GROUP_SUBSCRIBED;
				group_empty_dialog (gdata);
				break;
			case 3: /* download all */
				queue_add (QUEUE_ITEM(
					queue_item_headers_new(
						sdata, FALSE, gdata,
						HEADERS_ALL)));
				break;
			case 4: /* download new */
				queue_add (QUEUE_ITEM(
					queue_item_headers_new(
						sdata, FALSE, gdata,
						HEADERS_NEW)));
				break;
			case 5: /* clean/empty */
				group_empty_dialog (gdata);
				break;
			case 6: /* group properties */
				group_props_spawn (gdata);
				clist_unselect_all (Pan.group_clist);
				return;
				break;
			case 7: /* remove "new" mark */
				gdata->flags &= ~GROUP_NEW;
				break;
			default:
				break;
		}
	}

	/* if user made a permanent change, save the info to disk */
	if ((type == 1) || (type == 2) || (type == 7))
		grouplist_save_selected (grouplist_get_current_server());

	/* if unsubscribe and in subscribe mode, remove the groups */
	if (type==2 && group_mode==GROUP_SUBSCRIBED)
		for (l=list; l!=NULL; l=l->next)
			grouplist_remove_row ((const group_data*)l->data);

	/* refresh the modified entries */
	grouplist_update_selected_rows ();

	/* unselect */
	clist_unselect_all (Pan.group_clist);

	/* cleanup */
	g_list_free (list);
}

void grouplist_selected_subscribe (void) { grouplist_process_selected (1); }
void grouplist_selected_unsubscribe (void) { grouplist_process_selected (2); }
void grouplist_selected_download_all (void) { grouplist_process_selected (3); }
void grouplist_selected_download_new (void) { grouplist_process_selected (4); }
void grouplist_selected_empty (void) { grouplist_process_selected (5); }
void grouplist_selected_properties (void) { grouplist_process_selected (6); }
void grouplist_selected_unnew (void) { grouplist_process_selected (7); }

void
grouplist_get_all (void)
{
	if (my_server == NULL)
		pan_error_dialog ("No server selected.");
	else
		queue_add (
			QUEUE_ITEM(queue_item_grouplist_new (
					my_server, TRUE, GROUPLIST_ALL)));
}

void
grouplist_get_new (void)
{
	if (my_server == NULL)
		pan_error_dialog ("No server selected.");
	else
		queue_add (
			QUEUE_ITEM(queue_item_grouplist_new (
					my_server, TRUE, GROUPLIST_NEW)));
}

static void
grouplist_all_subscribed_gfunc (gpointer data, gpointer user_data)
{
	HeaderDownloadType dl_type;
	server_data *sdata = grouplist_get_current_server();
	group_data *group;

	g_return_if_fail (user_data != NULL);
	g_return_if_fail (sdata != NULL);
	g_return_if_fail (data != NULL);

	dl_type = GPOINTER_TO_INT (user_data);
	group = data;

	if (group->flags & GROUP_SUBSCRIBED)
		queue_add (QUEUE_ITEM(queue_item_headers_new(
					sdata, FALSE, group, dl_type)));
}

void
grouplist_all_subscribed_download_new (void)
{
	if (my_server == NULL)
		pan_error_dialog ("No server selected.");
	else
		g_slist_foreach (my_server->grouplist,
				 grouplist_all_subscribed_gfunc,
				 GINT_TO_POINTER (HEADERS_NEW));
}

void
grouplist_all_subscribed_download_all (void)
{
	if (my_server == NULL)
		pan_error_dialog ("No server selected.");
	else
		g_slist_foreach (my_server->grouplist,
				 grouplist_all_subscribed_gfunc,
				 GINT_TO_POINTER (HEADERS_ALL));
}

void
grouplist_select_all (void)
{
	pan_lock();
	gtk_clist_select_all (GTK_CLIST (Pan.group_clist));
	pan_unlock();
}


static void
grouplist_button_press (GtkWidget *widget, GdkEventButton *bevent)
{
	if (bevent->button != 1)
		gtk_signal_emit_stop_by_name (GTK_OBJECT(widget), "button_press_event");
	
	if (bevent->button == 3)
	{
		const int sel = GTK_CLIST(Pan.group_clist)->selection != NULL;

		GList* l = GTK_MENU_SHELL(group_clist_menu)->children; /* l is sub */
		gtk_widget_set_sensitive ( GTK_WIDGET(l->data), sel );
		l = l->next; /* l is unsub */
		gtk_widget_set_sensitive ( GTK_WIDGET(l->data), sel );
		l = l->next; /* l is separator */
		l = l->next; /* l is "get all headers" */
		gtk_widget_set_sensitive ( GTK_WIDGET(l->data), sel );
		l = l->next; /* l is "get new headers" */
		gtk_widget_set_sensitive ( GTK_WIDGET(l->data), sel );
		l = l->next; /* l is separator */
		l = l->next; /* l is "empty group" */
		gtk_widget_set_sensitive ( GTK_WIDGET(l->data), sel );
		l = l->next; /* l is separator */
		l = l->next; /* l is "unnew" */
		gtk_widget_set_sensitive ( GTK_WIDGET(l->data), sel );
		l = l->next; /* l is separator */
		l = l->next; /* l is "properties" */
		gtk_widget_set_sensitive ( GTK_WIDGET(l->data), sel );

		gtk_menu_popup (GTK_MENU(group_clist_menu), NULL, NULL, NULL,
				NULL, bevent->button, bevent->time);
	}
}

void
grouplist_set_view_mode (GtkWidget *widget, gpointer data)
{
	GtkWidget *menu = NULL;

	group_mode = GPOINTER_TO_INT (data);
	grouplist_update_ui (grouplist_get_current_server());

        pan_lock ();
        menu = gtk_option_menu_get_menu (GTK_OPTION_MENU(grouplist_mode_menu));
        gtk_object_ref (GTK_OBJECT(menu));
        gtk_option_menu_remove_menu (GTK_OPTION_MENU (grouplist_mode_menu));
	switch (group_mode) {
	case GROUP_ALL: gtk_menu_set_active(GTK_MENU (menu), 0); break;
	case GROUP_SUBSCRIBED: gtk_menu_set_active(GTK_MENU (menu), 1); break;
	case GROUP_NEW: gtk_menu_set_active(GTK_MENU (menu), 2);break;
	default: break;
	}
//        gtk_menu_set_active (GTK_MENU(menu), menu_item_index);
        gtk_option_menu_set_menu (GTK_OPTION_MENU(grouplist_mode_menu), menu);
        gtk_object_unref (GTK_OBJECT(menu));
        pan_unlock ();
}

typedef struct {
	gchar* name;
	int mode;
} GrouplistModeMenuStruct;

static GtkWidget*
grouplist_mode_menu_create (void)
{
	GtkWidget *option_menu = gtk_option_menu_new();
	GtkWidget *menu = gtk_menu_new ();
	int index = 0;
	int i;
	GrouplistModeMenuStruct foo[] =
	{
		{ _("All"), GROUP_ALL },
		{ _("Subscribed"), GROUP_SUBSCRIBED },
		{ _("New"), GROUP_NEW }
	};
	const int row_qty = sizeof(foo) / sizeof(foo[0]);

	for (i=0; i<row_qty; ++i) {
		GtkWidget *item = gtk_menu_item_new_with_label (foo[i].name);
		gtk_signal_connect (GTK_OBJECT(item), "activate",
				    (GtkSignalFunc)grouplist_set_view_mode,
				    GINT_TO_POINTER(foo[i].mode));
		gtk_menu_append (GTK_MENU (menu), item);
		if (group_mode == foo[i].mode)
			index = i;
	}

	gtk_menu_set_active (GTK_MENU(menu), index);
        gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), menu);
	gtk_widget_show_all (GTK_WIDGET(option_menu));

	return option_menu;
}

/*---[ grouplist_create ]---------------------------------------------
 *
 *--------------------------------------------------------------------*/
GtkWidget *
grouplist_create (void)
{
	GtkWidget *w;
        GtkWidget *vbox;
	GtkWidget *toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_TEXT);

	char *titles[] = {
		"",
		"",
		_("Groups"),
		_("Unread"),
		_("Total"),
		_("Description")
	};

	vbox = gtk_vbox_new (FALSE, 0);

	group_mode = gnome_config_get_int ("/Pan/State/group_mode=1");
	
	Pan.group_clist = gtk_clist_new_with_titles (6, titles);
	widget_set_font (GTK_WIDGET(Pan.group_clist), grouplist_font);

	w = server_menu_create ();
	
	gtk_widget_set (w, "width", 120, "height", 25, NULL); /* FIXME: If we don't do this the om doesn't show up*/
	gtk_toolbar_append_widget (GTK_TOOLBAR(toolbar), w, _("Select Usenet Server"), "");
	gtk_toolbar_append_space (GTK_TOOLBAR(toolbar));

	w = gtk_label_new (_("Groups"));
	gtk_toolbar_append_widget(GTK_TOOLBAR(toolbar), w, "", "");
	gtk_toolbar_append_space (GTK_TOOLBAR(toolbar));

	grouplist_mode_menu = grouplist_mode_menu_create();
	gtk_widget_set (grouplist_mode_menu, "width", 120, "height", 25, NULL); /* FIXME: If we don't do this the om doesn't show up*/
	gtk_toolbar_append_widget(GTK_TOOLBAR(toolbar), grouplist_mode_menu, _("Select Group Mode"), "");
	//gtk_toolbar_append_space (GTK_TOOLBAR(toolbar));

	//w = gtk_label_new (_("Find:"));
	//gtk_toolbar_append_widget(GTK_TOOLBAR(toolbar), w, "", "");
	//gtk_toolbar_append_space (GTK_TOOLBAR(toolbar));

	w = gtk_entry_new ();
	gtk_signal_connect (GTK_OBJECT (w), "activate",
			    GTK_SIGNAL_FUNC(grouplist_grep), w);
	gtk_toolbar_set_space_size(GTK_TOOLBAR(toolbar), 100);
	gtk_toolbar_append_widget(GTK_TOOLBAR(toolbar), w, 
				  _("Type in a group search string and press ENTER.  "
				    "Wildcards are allowed."), "");
	gtk_widget_show_all (toolbar);
	gnome_app_add_toolbar (GNOME_APP(Pan.window), GTK_TOOLBAR(toolbar), "group_toolbar", 0, GNOME_DOCK_TOP, 1, 0, 0);

	w = gtk_scrolled_window_new (NULL, NULL);

	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (w),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER(w), Pan.group_clist);
        gtk_box_pack_start (GTK_BOX (vbox), w, TRUE, TRUE, 0);

	gtk_signal_connect (GTK_OBJECT (Pan.group_clist),
			    "select_row",
			    GTK_SIGNAL_FUNC (grouplist_select_row), NULL);

	gtk_signal_connect (GTK_OBJECT (Pan.group_clist),
			    "unselect_row",
			    GTK_SIGNAL_FUNC (grouplist_select_row), NULL);

	gtk_signal_connect (GTK_OBJECT (Pan.group_clist),
			    "button_press_event",
			    GTK_SIGNAL_FUNC (grouplist_button_press), NULL);

	gtk_clist_set_column_width     (GTK_CLIST (Pan.group_clist), 0, 16);
	gtk_clist_set_column_min_width (GTK_CLIST (Pan.group_clist), 0, 16);
	gtk_clist_set_column_max_width (GTK_CLIST (Pan.group_clist), 0, 16);

	gtk_clist_set_column_width     (GTK_CLIST (Pan.group_clist), 1, 16);
	gtk_clist_set_column_min_width (GTK_CLIST (Pan.group_clist), 1, 16);
	gtk_clist_set_column_max_width (GTK_CLIST (Pan.group_clist), 1, 16);

	gtk_clist_set_column_width (GTK_CLIST (Pan.group_clist), 2, 170);
	gtk_clist_set_column_width (GTK_CLIST (Pan.group_clist), 3, 36);
	gtk_clist_set_column_width (GTK_CLIST (Pan.group_clist), 4, 33);
	gtk_clist_set_column_width (GTK_CLIST (Pan.group_clist), 5, 350);

//	gtk_clist_set_column_auto_resize (GTK_CLIST (group_clist), 1, TRUE);
//	gtk_clist_set_column_auto_resize (GTK_CLIST (group_clist), 2, TRUE);

/* JEL: this works well in paned mode, but not so well in notebook mode, we
   need to find a solution that will cover both ... maybe splitting the justification
   parts into the _construct funcs for both ...
	gtk_clist_set_column_justification ( GTK_CLIST(group_clist), 0, GTK_JUSTIFY_LEFT );
	gtk_clist_set_column_justification ( GTK_CLIST(group_clist), 1, GTK_JUSTIFY_RIGHT );
	gtk_clist_set_column_justification ( GTK_CLIST(group_clist), 2, GTK_JUSTIFY_RIGHT );
*/
	gtk_clist_set_selection_mode (GTK_CLIST (Pan.group_clist),
				      GTK_SELECTION_EXTENDED);
#if 0
	gtk_signal_connect (GTK_OBJECT (Pan.group_clist), "click_column",
			    GTK_SIGNAL_FUNC (click_groups_column_cb), NULL);
#endif

	group_mode = gnome_config_get_int ("/Pan/State/group_mode=1");
	if ( group_mode!=GROUP_ALL
	     && group_mode!=GROUP_SUBSCRIBED
	     && group_mode!=GROUP_NEW )
		group_mode = GROUP_SUBSCRIBED;

	/* Create the Popup Menu while we are here. */
	group_clist_menu = gnome_popup_menu_new (groups_menu_popup);	

	/* Lets have the Pixmap ready too */
	disk_pixmap = (GnomePixmap *) gnome_pixmap_new_from_xpm_d (disk_xpm);
	subscribed_pixmap = (GnomePixmap *) gnome_pixmap_new_from_xpm_d (envelope_xpm);

	return vbox;
}

/*---[ grouplist_free ]-----------------------------------------------
 * @list: the grouplist to be free'd.
 *
 * Free the allocated memory of an entire grouplist.  Called when
 * switching servers or refreshing the grouplist.
 *--------------------------------------------------------------------*/
void
grouplist_free (GSList *list)
{
	/* Warning:  Don't use this on the Pan.grouplist
	 * constructed by grouplist_load */
	if (list) {
		g_slist_foreach (list, (GFunc) group_free, NULL);
		g_slist_free (list);
		list = NULL;
	}
}

/* Grep the group listing for whatever the user typed into the Find box */
static void
grouplist_grep (GtkWidget *group_entry)
{
	GSList *pos = NULL;
	server_data *sdata = grouplist_get_current_server ();
	gchar *s = NULL;

	g_return_if_fail (Pan.group_clist);
	g_return_if_fail (sdata);

	pan_lock ();

	/* wrap the key in wildcards to make it a substring search...
	   unless there are already wildcards present, indicating the
	   user already knows what he wants */
	s = gtk_entry_get_text(GTK_ENTRY(group_entry));
	if (strchr(s,'*'))
		s = g_strdup (s);
	else
		s = g_strdup_printf ("*%s*", s);

	gtk_clist_freeze (GTK_CLIST(Pan.group_clist));
	gtk_clist_clear (GTK_CLIST(Pan.group_clist));
	for (pos=sdata->grouplist; pos!=NULL; pos=pos->next)
	{
		group_data* gdata = (group_data*) pos->data;

		if (group_mode!=GROUP_ALL && !(gdata->flags&group_mode))
			continue;
		if (fnmatch(s,gdata->name,0))
			continue;

		grouplist_add_group (sdata, gdata);
	}
	gtk_clist_thaw (GTK_CLIST(Pan.group_clist));
	pan_unlock();

	g_free (s);
}

static void
group_dialog_clicked_cb (GnomeDialog* dialog, int num, server_data *sdata)
{
	switch (num) {
		case 0: /* OK button */
			queue_add (QUEUE_ITEM(queue_item_grouplist_new (sdata, TRUE, GROUPLIST_ALL)));
		default:
			gtk_widget_destroy (GTK_WIDGET(dialog));
			break;
	}
}                                                                               

static void
grouplist_load_gdata (const char *key, const char *val, GSList **l)
{
	group_data *gdata = NULL;
	gchar **sd = g_strsplit (val, "\n", -1);
	gboolean end = FALSE;

	/* create the group */
	gdata = group_new ();
	gdata->name = g_strdup(key);
	if (!end) end=!sd[0]; gdata->flags = end ? 0 : atoi(sd[0]);
	if (!end) end=!sd[1]; gdata->description = g_strdup(end ? "" : sd[1]);
	if (!end) end=!sd[2]; gdata->state_filter = end ? ~STATE_FILTER_KILLFILE : atoi(sd[2]);

	/* cleanup */
	g_strfreev (sd);
	*l = g_slist_prepend (*l, gdata);
}

/*---[ grouplist_load ]-----------------------------------------------
 * @sdata: The server's grouplist to load.
 *
 * Load a grouplist database into memory.
 *--------------------------------------------------------------------*/
void
grouplist_load (server_data *sdata)
{
	gchar *path;
	pan_db db;
	GSList *l = NULL;

	path = g_strdup_printf ("/%s/%s_grouplist.db", data_dir, sdata->name);
	db = pan_db_open (path);
	g_free (path);

	if (db != NULL)
	{
		pan_db_foreach (db, (DBFunc)grouplist_load_gdata, &l);
		pan_db_close (db);
		sdata->grouplist = g_slist_reverse (l);
		grouplist_update_ui (sdata);
	}
}

void
grouplist_save_group (server_data* sdata, group_data* gdata)
{
	gchar *pch = NULL;
	pan_db db = 0;

	/* get the database */
	pch = g_strdup_printf ("/%s/%s_grouplist.db", data_dir, sdata->name);
	db = pan_db_open (pch);
	g_free (pch);
	if (!db)
		return;

	/* store the info */
	pch = g_strdup_printf ( "%d\n%s\n%u",
				gdata->flags,
				gdata->description?gdata->description:"",
				gdata->state_filter);
	pan_db_put_value_str (db, gdata->name, pch);
	g_free (pch);

	/* close the db */
	pan_db_close (db);
}

static void
grouplist_save_list (server_data* sdata, GSList *group_datas)
{
	gchar *path;
	pan_db db;
	GSList *l;

	path = g_strdup_printf ("/%s/%s_grouplist.db", data_dir, sdata->name);
	db = pan_db_open (path);
	g_free (path);
	
	if (!db)
		return;
	
	for (l=group_datas; l; l=l->next) {
		const group_data* gdata = (group_data*) l->data;
		gchar *pch = g_strdup_printf ( "%d\n%s\n%u",
					       gdata->flags,
					       gdata->description?gdata->description:"",
					       gdata->state_filter);
		pan_db_put_value_str (db, gdata->name, pch);
		g_free (pch);
	}

	pan_db_close (db);
}

void
grouplist_save_selected (server_data *sdata)
{
	GtkCList *c = GTK_CLIST(Pan.group_clist);
	GSList *groups = NULL;
	GList *l;

	for (l=c->selection; l; l=l->next)
		groups = g_slist_prepend (groups, 
			gtk_clist_get_row_data (c, GPOINTER_TO_INT(l->data)));
	grouplist_save_list (sdata, groups);

	g_slist_free (groups);
}

void
grouplist_save (server_data *sdata)
{
	grouplist_save_list (sdata, sdata->grouplist);
}

/*---[ grouplist_update ]---------------------------------------------
 * update the entire group CLIST, clearing and then filling it in
 * with the groups that are part of the current group mode (all, 
 * subscribed, etc.)
 *--------------------------------------------------------------------*/

static gchar*
grouplist_update_describe (const StatusItem *item)
{
	return g_strdup (_("Updating Group List"));
}

static void
grouplist_update_ui_thread (void* data)
{
	server_data* sdata = (server_data*)data;
	GtkCList *list = GTK_CLIST(Pan.group_clist);
	StatusItem *item = NULL;
	GSList *l = NULL;

        /* create a new status item to get sort/thread messages */
	item = STATUS_ITEM(status_item_new(grouplist_update_describe));
	pan_object_sink(PAN_OBJECT(item));
	gui_add_status_item (item);
	status_item_emit_init_steps (item, g_slist_length(sdata->grouplist));
	status_item_emit_status (item, _("Updating Group List"));

	/* clear the list */
	pan_lock();
	gtk_clist_freeze (list);
	gtk_clist_clear (list);

	/* add the groups */
	for (l=sdata->grouplist; l; l=l->next)
	{
		if ((group_mode==GROUP_ALL)
		    || (group_mode & ((group_data*)l->data)->flags))
		{
			grouplist_add_group (sdata, l->data);
			pan_unlock();
			status_item_emit_next_step (item);
			pan_lock();
		}
	}
	gtk_clist_thaw (list);
	pan_unlock();

	/* clean out the status item */
	gui_remove_status_item (item);
	pan_object_unref(PAN_OBJECT(item));
}

void
grouplist_update_ui (server_data *sdata)
{
	pthread_t thread;
	pthread_create (&thread, NULL, (void*)grouplist_update_ui_thread, sdata);
	pthread_detach (thread);
}

/* note that this is called inside a pan_lock() */
static void
grouplist_add_group (server_data *sdata, group_data *gdata)
{
	int row;
	char total_buf[16] = { '\0' };
	char unread_buf[16] = { '\0' };
	char *text[7];
	GtkCListRow *clistrow;
	const int total = group_get_attrib_i (sdata, gdata, "Total" );

	if ( total ) {
		int unread = total - group_get_attrib_i (sdata, gdata, "Read" );
		commatize_ulong (total, total_buf);
		commatize_ulong (unread, unread_buf);
	}

	text[0] = "";
	text[1] = "";
	text[2] = gdata->name;
	text[3] = unread_buf;
	text[4] = total_buf;
	text[5] = gdata->description ? gdata->description : "";
	text[6] = NULL;
	row = gtk_clist_append (GTK_CLIST(Pan.group_clist), text);

	if (gdata->flags & GROUP_SUBSCRIBED)
		gtk_clist_set_pixmap (
			GTK_CLIST(Pan.group_clist), row, 0,
			subscribed_pixmap->pixmap,
			subscribed_pixmap->mask);

	if (total)
		gtk_clist_set_pixmap (
			GTK_CLIST(Pan.group_clist), row, 1,
			disk_pixmap->pixmap,
			disk_pixmap->mask);

	clistrow = (GtkCListRow *) GTK_CLIST(Pan.group_clist)->row_list_end->data;
	clistrow->data = gdata;
}

/* Remove a single row from the group CList, useful in unsubscribing */
static void
grouplist_remove_row (const group_data *gdata)
{
	GtkCList *list = GTK_CLIST(Pan.group_clist);

	pan_lock();
	gtk_clist_remove (list, gtk_clist_find_row_from_data (list,(gpointer)gdata));
	pan_unlock();
}

/**
 * grouplist_update_row:
 * @gdata: Group who's row is to be updated.
 *
 * Update the CList row for a group to show it's message counts or
 * any other relative data (subscribed/on disk) for that group.
 **/
void
grouplist_update_row (const group_data *gdata)
{
	server_data *sdata = grouplist_get_current_server ();
	GtkCList* list = GTK_CLIST (Pan.group_clist);
	gint total = group_get_attrib_i (sdata, gdata, "Total" );
	gint row = 0;

	pan_lock();

	row = gtk_clist_find_row_from_data (list,(gpointer)gdata);

	if (gdata->flags & GROUP_SUBSCRIBED)
		gtk_clist_set_pixmap (
			GTK_CLIST(Pan.group_clist), row, 0,
			subscribed_pixmap->pixmap,
			subscribed_pixmap->mask);
	else
		gtk_clist_set_text (
			GTK_CLIST(Pan.group_clist), row, 0, "");

	if (total)
		gtk_clist_set_pixmap (
			GTK_CLIST(Pan.group_clist), row, 1,
			disk_pixmap->pixmap,
			disk_pixmap->mask);
	else
		gtk_clist_set_text (
			GTK_CLIST(Pan.group_clist), row, 1, "");

	if (total)
	{
		char buf[16];

		// total column
		commatize_ulong (total, buf);
		gtk_clist_set_text (list, row, 4, buf);

		// unread column
		commatize_ulong ( total-group_get_attrib_i(sdata,gdata,"Read"), buf );
		gtk_clist_set_text (list, row, 3, buf);
	}
	else
	{
		//gtk_clist_set_text ( list, row, 0, gdata->name);
		gtk_clist_set_text ( list, row, 3, "" );
		gtk_clist_set_text ( list, row, 4, "" );
	}

	pan_unlock();
}

static void
grouplist_update_selected_rows (void)
{
	GtkCList *list = GTK_CLIST(Pan.group_clist);
	GList *l = list->selection;
	for (l=list->selection; l!=NULL; l=l->next)
	{
		group_data *gdata = NULL;
		int index = GPOINTER_TO_INT(l->data);
		gdata = (group_data*) gtk_clist_get_row_data (list, index);
		grouplist_update_row (gdata);
	}
}

static void
grouplist_internal_select_row (int row)
{
        GtkCList* list = GTK_CLIST(Pan.group_clist);

	pan_lock();
	gtk_clist_unselect_all (list);
	gtk_clist_select_row (list,row,-1);
	pan_unlock();
}

void
grouplist_prev_group (void)
{
        GtkCList* list = GTK_CLIST(Pan.group_clist);
        const int row_qty = list->rows;
       	int row = 0;

	if (!row_qty)
		return;

	if ( list->selection ) {
		row = GPOINTER_TO_INT(list->selection->data) - 1;
		if (row<0)
			row = row_qty-1;
	}

	grouplist_internal_select_row (row);
	articlelist_set_current_group (my_server, (group_data*)gtk_clist_get_row_data(list,row) );
}

void
grouplist_next_group (void)
{
        GtkCList* list = GTK_CLIST(Pan.group_clist);
        const int row_qty = list->rows;
       	int row = 0;

	if (!row_qty)
		return;

	if ( list->selection )
		row = (GPOINTER_TO_INT(list->selection->data)+1) % row_qty;

	grouplist_internal_select_row (row);
	articlelist_set_current_group (
		my_server, (group_data*)gtk_clist_get_row_data(list,row) );
}

static gboolean
grouplist_select_row_if_unread (int row)
{
        GtkCList* list = GTK_CLIST(Pan.group_clist);
	group_data* g = gtk_clist_get_row_data(list,row);
	server_data *s = grouplist_get_current_server ();
	const int total = group_get_attrib_i (s, g, "Total");
	const int read = group_get_attrib_i (s, g, "Read");
	gboolean retval = FALSE;

	if ( read<total )
	{
		grouplist_internal_select_row (row);
		articlelist_set_current_group (my_server, g);
		retval = TRUE;
	}

	return retval;
}

void
grouplist_next_unread_group (void)
{
        GtkCList* list = GTK_CLIST(Pan.group_clist);
        const int row_qty = list->rows;
	int i;
       	int row = 0;

	if (!row_qty)
		return;

	/* get the first row */
	if ( list->selection )
		row = (GPOINTER_TO_INT(list->selection->data) + 1) % row_qty;
	else
		row = 0;

	/* walk through */
	for ( i=0; i<row_qty; row=++row%row_qty )
		if ( grouplist_select_row_if_unread (row) )
			return;
}

void
grouplist_prev_unread_group (void)
{
        GtkCList* list = GTK_CLIST(Pan.group_clist);
        const int row_qty = list->rows;
       	int row = 0;
	int i;

	if (!row_qty)
		return;

	/* get the first row */
	if ( list->selection ) {
		row = GPOINTER_TO_INT(list->selection->data);
		row = row ? row-1 : row_qty-1;
	}
	else
		row = row_qty-1;

	/* walk through */
	for ( i=0; i!=row_qty; ++i, row=row?--row:row_qty-1 )
		if ( grouplist_select_row_if_unread (row) )
			return;
}
