/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "date.h"

#ifndef isdigit
#define isdigit(x) (((x)>='0')&&((x)<='9'))
#endif
#ifndef isalpha
#define isalpha(x) ((((x)>='A')&&((x)<='Z')) || (((x)>='A')&&((x)<='Z')))
#endif

#define THE_YEAR_0		1970L
#define IS_LEAP_YEAR(y)		(((y) % 4) == 0)

typedef struct
{
	char name[5];
	int tz;
}
Timezone_Table_Type;


static Timezone_Table_Type Timezone_Table[] = {
	{"EDT", -500},
	{"PST", -800},
	{"CST", -600},
	{"MST", -700},
	{"MET", 100},				/* Middle European */
	{"MEZ", 100},				/* Middle European */
	{"MSK", 300},				/* Moscow */
	{"CET", 100},				/* Central European */
	{"HKT", 800},				/* Central European */
	{"JST", 900},				/* Central European */
	{"CAST", 930},				/* Central Autsralian */
	{"EAST", 1000},				/* Eastern Autsralian */
	{"NZST", 1200},				/* New Zealand Autsralian */
	{"EET", 200},				/* Eastern European */
	{"", 0}
};


static const gchar*
skip_whitespace (const gchar *b)
{
	register char ch;

	if (b == NULL)
		return NULL;

	while (((ch = *b) == ' ') || (ch == '\t') || (ch == '\n'))
		b++;
	return b;
}


static int
parse_timezone (const gchar *t)
{
	gchar ch;
	Timezone_Table_Type *table;

	table = Timezone_Table;

	while (0 != (ch = table->name[0])) {
		if ((*t == ch) && (0 == strncmp(t, table->name, sizeof(table->name))))
			return table->tz;
		table++;
	}

	return 0;
}

time_t
parse_date (const char *date)
{
	char ch;
	long day, month, year, minutes, seconds, hours, tz;
	int sign;

	date = skip_whitespace(date);
	if ( !date || !*date )
		return 0;

	/* Look for a weekday, if found skip it */
	while (isalpha((int)*date))
		date++;
	if (*date == ',')
		date++;

	date = skip_whitespace(date);

	/* expecting "03 Feb 1997 10:09:58 +0100" */
	day = 0;
	while (ch = *date, isdigit((int)ch)) {
		day = 10 * day + (long) (ch - '0');
		date++;
	}
	if ((day == 0) || (day > 31))
		return 0;
	day--;

	date = skip_whitespace(date);
	month = 0;
	if (!g_strncasecmp(date,"jan",3)) month = 0;
	else if (!g_strncasecmp(date,"feb",3)) month = 1;
	else if (!g_strncasecmp(date,"mar",3)) month = 2;
	else if (!g_strncasecmp(date,"apr",3)) month = 3;
	else if (!g_strncasecmp(date,"may",3)) month = 4;
	else if (!g_strncasecmp(date,"jun",3)) month = 5;
	else if (!g_strncasecmp(date,"jul",3)) month = 6;
	else if (!g_strncasecmp(date,"aug",3)) month = 7;
	else if (!g_strncasecmp(date,"sep",3)) month = 8;
	else if (!g_strncasecmp(date,"oct",3)) month = 9;
	else if (!g_strncasecmp(date,"nov",3)) month = 10;
	else if (!g_strncasecmp(date,"dec",3)) month = 11;
	else return 0;

	/* skip past month onto year. */
	while (isalpha((int)*date))
		date++;
	date = skip_whitespace(date);

	year = 0;
	if (*date == '9') /*  FIXME: this only works through the year 8999 :) */
		year = 19;

	while (ch = *date, isdigit((int)ch)) {
		year = year * 10 + (long) (ch - '0');
		date++;
	}
	date = skip_whitespace(date);

	/* Now parse hh:mm:ss */
	hours = 0;
	while (ch = *date, isdigit((int)ch)) {
		hours = hours * 10 + (long) (ch - '0');
		date++;
	}
	if ((ch != ':') || (hours >= 24))
		return 0;
	date++;

	minutes = 0;
	while (ch = *date, isdigit((int)ch)) {
		minutes = minutes * 10 + (long) (ch - '0');
		date++;
	}
	if (minutes >= 60)
		return 0;

	/* Seconds may not be present */
	seconds = 0;
	if (ch == ':') {
		date++;
		while (ch = *date, isdigit((int)ch)) {
			seconds = seconds * 10 + (long) (ch - '0');
			date++;
		}
		if (seconds >= 60)
			return 0;
	}
	/* Now timezone */
	date = skip_whitespace(date);

	sign = 1;
	if (*date == '+')
		date++;
	else if (*date == '-') {
		sign = -1;
		date++;
	}
	tz = 0;
	while (ch = *date, isdigit((int)ch)) {
		tz = tz * 10 + (long) (ch - '0');
		date++;
	}
	tz = sign * tz;

	date = skip_whitespace(date);
	if (isalpha((int)*date)) {
		sign = 1;
		if ((*date != 'G') && (*date != 'M') && (*date != 'T'))
			tz = parse_timezone(date);
	}
	/* Compute the number of days since beginning of year. */
	day = 31 * month + day;
	if (month > 1) {
		day -= (month * 4 + 27) / 10;
		if (IS_LEAP_YEAR(year))
			day++;
	}
	/* add that to number of days since beginning of time */
	year -= THE_YEAR_0;
	day += year * 365 + year / 4;
	if ((year % 4) == 1)
		day++;

	/* Adjust hours for timezone */
	hours -= tz / 100;
	minutes -= tz % 100;		/* ?? */


	/* Now convert to secs */
	seconds += 60L * (minutes + 60L * (hours + 24L * day));

	return seconds;
}


/*---[ generate_date ]------------------------------------------------
 * generate a properly formatted Date: for a message being sent to
 * an NNTP server.  This follows the format specified by _Son of 1036_
 * section 5.1.  --csk
 * http://www.chemie.fu-berlin.de/outerspace/netnews/son-of-1036.html#5.1
 *--------------------------------------------------------------------*/
gchar*
generate_date (void)
{
	static const gchar *const months[] = {
		"Jan","Feb","Mar","Apr","May","Jun",
		"Jul","Aug","Sep","Oct","Nov","Dec"
	};
	static const gchar *const days[] = {
		"Sun","Mon","Tue","Wed","Thu","Fri","Sat"
	};

	time_t secs = 0;
	struct tm local;
	gchar fmt[64] = { '\0' };
	gchar buf[64] = { '\0' };
	struct tm ut;
	int diff = 0;

	/* figure out the difference between this timezone and UT */
	time (&secs);
	localtime_r (&secs, &local);
	gmtime_r (&secs, &ut);
	diff = mktime(&local) - mktime(&ut);

	/* decide on the time string format.  see son of 1036 sec 5.1 for
	 * the rationale of the offset-from-ut notation.
         *
	 * Update: INN 1.7.2-4 (and others) don't accept the <TZ> string at
	 * the end of son-of-1036's date string.  Thanks to Valentini Jove
	 * for testing to see what worked & what didn't.
	 *
	 * Update 2: Big Thanks to tov are jacobsen <tovj@stud.ntnu.no> for
	 * investigating what format other programs use, for suggesting
	 * code, and for noting that we need to roll our own day-of-week and
	 * month abbreviations because son-of-1036 requires English text but
	 * strftime() localizes (Okt instead of Oct or Man instead of Mon).
	 */
	sprintf (fmt, "%s, %%d %s %%Y %%H:%%M:%%S %c%02d%02d",
		days[local.tm_wday],
		months[local.tm_mon],
		(diff<0?'-':'+'),
		(abs(diff)/3600),
		(abs(diff)%3600));

	/* generate the time */
	strftime (buf, sizeof(buf), fmt, &local);
	return g_strdup (buf);
}
