/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include "article.h"
#include "article-thread.h"
#include "util.h"

/*********************************************************************
 * The following stuff implements the "thread articles" functionality
 ********************************************************************/

static int 
mp_strcasecmp (
	register const char* mp,
	register const char* ph )
{
	register int val;

	while (*mp && *ph)
	{
		val = *mp - *ph;
		if (val) {
			val = toupper(*mp) - toupper(*ph);
			if ( val )
				return val;
		}

		if (*mp=='(' || *mp=='[')
		{
			++mp;
			++ph;
			if (isdigit((int)*mp) && isdigit((int)*ph))
			{
				while (isdigit((int)*mp)) ++mp;
				while (isdigit((int)*ph)) ++ph;
			}
		}
		else
		{
			++mp;
			++ph;
		}
	}

	return 0;
}


/**
 * Skip the "Re: " part of a subject header, if any
 * @param subject
 * @return the non-"Re:" portion of the subject header
 */
#define skip_reply_leader(a) ((a!=NULL && toupper(a[0])=='R' && toupper(a[1])=='E' && toupper(a[2])==':')?a+4:a)

static int
pp_adata_loc_comp (
	const void* va,
	const void* vb )
{
        const article_data* a = *(const article_data**)va;
        const article_data* b = *(const article_data**)vb;
	int a_loc;
	int b_loc;

	a_loc = a->linecount;
	if (a->parts && a->threads!=NULL) {
		GSList *l;
		for (l=a->threads; l; l=l->next)
			a_loc += ((article_data *)l->data)->linecount;
	}

	b_loc = b->linecount;
	if (b->parts && b->threads!=NULL) {
		GSList *l;
		for (l=b->threads; l; l=l->next)
			b_loc += ((article_data *)l->data)->linecount;
	}

	return a_loc - b_loc;
} 

static int
pp_adata_time_comp (
	const void* va,
	const void* vb )
{
        const article_data* a = *(const article_data**)va;
        const article_data* b = *(const article_data**)vb;
	return a->date - b->date;
}

static int
pp_adata_author_comp (
	const void* va,
	const void* vb )
{
        const article_data* a = *(const article_data**)va;
        const article_data* b = *(const article_data**)vb;
	const gchar *author_a = a->author;
	const gchar *author_b = b->author;

	/* try to skip over leading quotation marks, etc. */
	while (*author_a && !isalpha((int)*author_a)) ++author_a;
	if (!*author_a)
		author_a = a->author;
	while (*author_b && !isalpha((int)*author_b)) ++author_b;
	if (!*author_b)
		author_b = b->author;

	return g_strcasecmp (author_a, author_b);
}

/**
 * Sort this way:
 * (1) by subject, with replies coming after non-replies of the same subject, but before other subjects.
 * (2) by time
 */
static int
pp_adata_subject_comp (
	const void* va,
	const void* vb )
{
        const article_data* a = *(const article_data**)va;
        const char* subj_a = skip_reply_leader (a->subject);
        const gboolean a_is_re = a->subject != subj_a;

        const article_data* b = *(const article_data**)vb;
        const char* subj_b = skip_reply_leader (b->subject);
        const gboolean b_is_re = b->subject != subj_b;
	int value;

	/* try to skip non-alpha characters, if possible */
	while (*subj_a && !isalpha((int)*subj_a))
		++subj_a;
	if (!*subj_a)
		subj_a = skip_reply_leader (a->subject);
	while (*subj_b && !isalpha((int)*subj_b))
		++subj_b;
	if (!*subj_b)
		subj_b = skip_reply_leader (b->subject);

        /* if they're not related, a simple strcmp does the job */
        value = g_strcasecmp (subj_a, subj_b);
	if (value)
		return value;

	/* if one but not both is a reply, the reply goes second */
	if (!!a_is_re != !!b_is_re)
		return a_is_re ? 1 : -1;

	/* oldest goes first... */
	value = a->date - b->date;
	if ( value )
		return value;

	return 0;
}

/**
 * Sort by message id
 */
static int
pp_adata_unread_children_comp (
	const void* va,
	const void* vb )
{
	const article_data* a = *(const article_data**)va;
	const article_data* b = *(const article_data**)vb;
	return a->unread_children - b->unread_children;
}

/**
 * Sort by message id
 */
static int
pp_adata_message_id_comp (
	const void* va,
	const void* vb )
{
	const article_data* a = *(const article_data**)va;
	const article_data* b = *(const article_data**)vb;
	return strcmp (a->message_id, b->message_id);
}


/**
 * Works like article_subject_cmp, but handles multiparts better
 */
static int
p_adata_mp_subject_comp (
	const void* va,
	const void* vb )
{
	const article_data* a = (const article_data*) va;
	const char* subj_a = skip_reply_leader (a->subject);
	const gboolean a_is_re = a->subject != subj_a;

	const article_data* b = (const article_data*) vb;
	const char* subj_b = skip_reply_leader (b->subject);
	const gboolean b_is_re = b->subject != subj_b;

	/* get the easy stuff out of the way */
	int value = mp_strcasecmp (subj_a, subj_b);
	if (value)
		return value;

	/* check multipart */
	value = a->part - b->part;
	if (value)
		return value;

	/* if one but not both is a reply, the reply goes second */
	if (a_is_re != b_is_re)
		return a_is_re ? 1 : -1;

	/* oldest goes first... */
	value = a->date - b->date;
	if (value)
		return value;

	return 0;
}


/**
 * Sort thusly:
 * (1) in ascending number of part qty
 * (2) by subject via mp_strcasecmp
 * (3) by part number
 * This should lump all the multiparts together in ascending order.
 */
static int
pp_adata_multipart_comp (
	const void* va,
	const void* vb )
{
	register const article_data* a = *(const article_data**)va;
	register const article_data* b = *(const article_data**)vb;

	/* parts */
	int val = a->parts - b->parts;
	if (val)
		return val;

	/* subject */
	val = mp_strcasecmp (a->subject, b->subject);
	if (val)
		return val;

	/* part number */
	val = a->part - b->part;
	if (val)
		return val;

	/* did someone repost this? */
	val = a->date - b->date;
	if (val)
		return val;

	/* they are equal */
	return 0;
}

typedef int (*compare_func)(const void*, const void*);

void
sort_articles (
	article_data** buf,
	int article_qty,
	int sort_type,
	gboolean ascending )
{
	compare_func func = NULL;

	switch (sort_type)
	{
		case ARTICLE_SORT_AUTHOR:
			func = pp_adata_author_comp;
			break;
		case ARTICLE_SORT_LINES:
			func = pp_adata_loc_comp;
			break;
		case ARTICLE_SORT_DATE:
			func = pp_adata_time_comp;
			break;
		case ARTICLE_SORT_SUBJECT:
			func = pp_adata_subject_comp;
			break;
		case ARTICLE_SORT_UNREAD_CHILDREN:
			func = pp_adata_unread_children_comp;
			break;
		default:
			pan_warn_if_reached();
			func = pp_adata_subject_comp;
			break;
	}

	/* sort */
	msort (buf, article_qty, sizeof(article_data*), func);

	/* if not ascending, reverse the order */
	if (!ascending) {
		const int mid = article_qty/2;
		int i;
		article_data* tmp;
		for (i=0; i!=mid; ++i) { /* swap */
			tmp = buf[i];
			buf[i] = buf[article_qty-1-i];
			buf[article_qty-1-i] = tmp;
		}
	}
}

static int
count_unread_children (const article_data* adata)
{
	int unread = 0;
	GSList *l;

	if (!adata)
		return unread;

	if (!article_flag_on(adata, STATE_READ))
		++unread;

	for (l=adata->threads; l!=NULL; l=l->next) {
		const article_data* a = (const article_data*) l->data;
		g_assert (a!=NULL);
		unread += count_unread_children (a);
	}

	return unread;
}

static gboolean
is_child_of(const article_data* child, const article_data* parent)
{
	g_assert (child!=NULL);
	g_assert (parent!=NULL);

	for ( ;; )
	{
		if (!child)
			return FALSE;
		if (child == parent)
			return TRUE;
		child = child->parent;
	}
}

/**
 * Thread the articles specified in list
 */
void
thread_articles (
	GSList* list,
	int list_len,
	article_data*** setme_array,
	int* setme_qty,
	StatusItem *item)
{
	GSList* p;
	int i;
	article_data **refs, **subjs, **parts, **buf;
	article_data search_a, *p_search_a=&search_a;
	int article_qty = list_len;

	/* if nothing to do, do nothing */
	if (!list_len || !list) {
		*setme_array = 0;
		*setme_qty = 0;
		return;
	}

	/* make a reference copy of the articles... */
	buf = g_new (article_data*, article_qty);
	for (i=0, p=list; p!=NULL; p=p->next, ++i) {
		buf[i] = (article_data*) p->data;
		g_assert (buf[i] != NULL);
		buf[i]->unread_children = article_flag_on(buf[i], STATE_READ)?0:1;
	}
	g_slist_free (list);
	list = NULL;
	g_assert (i==article_qty);

	/* make various sorted copies of the articles for fast searching */
	i = sizeof(article_data*);
	refs = g_memdup (buf, i*article_qty);
	subjs = g_memdup (buf, i*article_qty);
	parts = g_memdup (buf, i*article_qty);
	*setme_array = g_memdup (buf, i*article_qty);
	*setme_qty = article_qty;
	qsort (refs, article_qty, i, pp_adata_message_id_comp);
	qsort (subjs, article_qty, i, pp_adata_subject_comp);
	qsort (parts, article_qty, i, pp_adata_multipart_comp);

	/* thread the articles */
	for (i=0; i!=article_qty; ++i)
	{
		article_data *parent = NULL;
		article_data *a = buf[i];
		int index = -1;

		status_item_emit_next_step (item);

		/* thread by reference
		   (except for multiparts, which are top-level: !a->parts ) */
		if (a->references && *a->references=='<' && !a->parts)
		{
			gboolean exact = FALSE;
			search_a.message_id = strrchr ( a->references, '<' );
			index = lower_bound (&p_search_a,
				refs, article_qty, sizeof(article_data*),
				pp_adata_message_id_comp, &exact);
			if (exact && !is_child_of(refs[index],a))
				parent = refs[index];
		}

		/* thread by multipart */
		if (!parent && a->parts>1 && a->part>1)
		{
			search_a.subject = a->subject;
			search_a.part = 1;
			search_a.parts = a->parts;
			search_a.date = 0; /* unlikely to get exact match :) */
			index = lower_bound (
				&p_search_a,
				parts, article_qty,
				sizeof(article_data*),
				pp_adata_multipart_comp,
				NULL);

			if (index<article_qty && parts[index]!=buf[i] && !is_child_of(parts[index],a)) {
				parent = parts[index];
			}
		}

		/* thread by subject */
		if (!parent && skip_reply_leader(a->subject)!=a->subject)
		{
			search_a.subject = (char*) skip_reply_leader (a->subject);
			search_a.date = 0; /* unlikely to get exact match :) */
			index = lower_bound (
				&p_search_a,
				subjs, article_qty,
				sizeof(article_data*),
				pp_adata_subject_comp,
				NULL);
			if (index<article_qty)
			{
				article_data* b = subjs[index];
				if (!g_strcasecmp (search_a.subject, b->subject) && !is_child_of(b,a)) /* simple case of source/reply */
				{
					parent = b;
				}
				else if (!g_strcasecmp (a->subject, b->subject) && b->date<a->date && !is_child_of(b,a)) /* 2 replies, no source: thread under oldest. */
				{
					parent = b;
				}
			}
		}

		if (parent != NULL) /* this article has a parent */
		{
			int unread_kids;
			article_data* tmp;

			g_assert (!is_child_of(parent,a));

			/* link the two articles */
			a->parent = parent;
			parent->threads = g_slist_insert_sorted (
				parent->threads, a, p_adata_mp_subject_comp);
			if (!article_flag_on (a, STATE_READ))
				parent->state |= STATE_UNREAD_CHILDREN;

			/* update unread child count */
			unread_kids = count_unread_children (a);
			for (tmp=parent; tmp!=NULL; tmp=tmp->parent)
				tmp->unread_children += unread_kids;
		}
	}

	/* cleanup */
	g_free (refs);
	g_free (parts);
	g_free (subjs);
	g_free (buf);
}


void
check_multipart_articles (
	article_data** buf,
	int article_qty )
{
	int i, j;

	/* set the multipart state (all/partial) */
	for (i=0; i!=article_qty; ++i)
	{
		GSList* p;

		article_data* a = buf[i];
		if (!a->parts)
			continue;

		/* not a multipart */
		if (a->part!=1 || a->parent!=NULL)  {
			a->state &= ~(STATE_MULTIPART_ALL & STATE_MULTIPART_SOME);
			continue;
		}

		/* handle the single-part attachment message */
		if (a->parts==1) {
			a->state |= STATE_MULTIPART_ALL;
			continue;
		}

		/* make sure we have each multipart. */
		for (j=a->part+1, p=a->threads;
		     j<=a->parts && p!=NULL;
		     p=p->next)
		{
			article_data* b = (article_data*)(p->data);
			if (b->part > j)
				break; /* some */
			else if (b->part == j)
				++j; /* okay so far */
			else
				; /* a repost of a multipart section? */
		}
		if (j==a->parts+1) {
			a->state |= STATE_MULTIPART_ALL;
		} else {
			a->state |= STATE_MULTIPART_SOME;
		}
	}
}
